<xsl:stylesheet version="1.0" xmlns:xsl="http://www.w3.org/1999/XSL/Transform">
  <xsl:variable name="parallel_info_node" 
    select="/report/sql_monitor_report/parallel_info"/>
  <xsl:variable name="target_node" 
    select="/report/sql_monitor_report/target"/>
  <xsl:variable name="query_duration" 
    select="/report/sql_monitor_report/target/duration"/>
  <xsl:variable name="queuing_duration" 
    select="round(/report/sql_monitor_report/
                  stats[@type='monitor']/stat[@name='queuing_time'] 
                  div 1000000)"/>
  <xsl:variable name="status" 
    select="/report/sql_monitor_report/target/status"/>
  <xsl:variable name="need_cell_filter" 
    select="/report/sql_monitor_report/stats[@type='monitor']/
           stat[@name='cell_offload_efficiency']"/>
  <xsl:variable name="need_read" 
    select="/report/sql_monitor_report/
            stats[@type='monitor']/stat[@name='read_reqs']"/>
  <xsl:variable name="need_write" 
    select="/report/sql_monitor_report/
            stats[@type='monitor']/stat[@name='write_reqs']"/>
  <xsl:variable name="need_fetch" 
    select="/report/sql_monitor_report/stats[@type='monitor']/
           stat[@name='user_fetch_count']"/>
  <xsl:variable name="need_progress" 
    select="$status = 'EXECUTING' and
            /report/sql_monitor_report/plan_monitor/*/
            stats[@type='plan_monitor']/stat[@name='time_left']"/>
  <xsl:variable name="total_activity_count" 
     select="/report/sql_monitor_report/activity_sampled/@count"/>
  <xsl:variable name="total_wait_count" 
     select="/report/sql_monitor_report/activity_sampled/@wait_count"/>
  <xsl:variable name="total_cpu_count" 
     select="/report/sql_monitor_report/activity_sampled/@cpu_count"/>
  <xsl:variable name="need_sqltext" 
    select="/report/sql_monitor_report/target/sql_fulltext"/>
  <xsl:variable name="need_user" 
    select="/report/sql_monitor_report/target/user"/>
  <xsl:variable name="need_module" 
    select="/report/sql_monitor_report/target/module"/>
  <xsl:variable name="need_action" 
    select="/report/sql_monitor_report/target/action"/>
  <xsl:variable name="need_service" 
    select="/report/sql_monitor_report/target/service"/>
  <xsl:variable name="need_program" 
    select="/report/sql_monitor_report/target/program"/>
  <xsl:variable name="need_plseoid"
    select="/report/sql_monitor_report/target/plsql_entry_object_id"/>
  <xsl:variable name="need_plsesid"
    select="/report/sql_monitor_report/target/plsql_entry_subprogram_id"/>
  <xsl:variable name="need_plsoid"
    select="/report/sql_monitor_report/target/plsql_object_id"/>
  <xsl:variable name="need_plssid"
    select="/report/sql_monitor_report/target/plsql_subprogram_id"/>
  <xsl:variable name="val_px_requested"
    select="/report/sql_monitor_report/target/servers_requested"/>
  <xsl:variable name="val_px_allocated"
    select="/report/sql_monitor_report/target/servers_allocated"/>
  <xsl:template match="/">
    <html>
     <head>
     <title> SQL Monitor Report </title>
       <style type="text/css"> 
         body, table, input, select, textarea
         {font:normal normal 8pt Verdana,Arial;text-decoration:none;
          color:#000000; empty-cells:show;}
         .s8 {font-size:8pt;color:#006699}
         .s9 {font-size:10pt;color:#006699}
         .s10 {font-size:14pt;color:#006699;}
         .s16 {border-width : 1px; border-color : #CCCC99;
              border-style: solid; color:#006699;font-size:8pt; 
              background-color:#CCCC99; empty-cells:show;}
        .s17 {border-width : 1px; border-color : #CCCC99;
              border-style: solid; font-size:8pt; 
              background-color:#E8E8E6; empty-cells: show }
        .s17a {border-width : 1px; border-color : #BDCCC3;
              border-style: solid; font-size:8pt; 
              background-color:#F5F5F5; empty-cells: show}
        .s17b {border-width : 1px; border-color : #BDCCC3;
               border-style: solid; font-size:8pt; 
               background-color:#F1F5DD; empty-cells: show}
        .s27 {border-width : 1px; border-color : #CCCC99; 
              border-style: solid;}
        .graph {
          border: solid 0px ; empty-cells: show
        }

        .bar {
          border-left: solid 0px;
          padding-right: 0.5em;
          font-size:1pt;
        }

        .sql {border-width : 1px; border-color : #CCCC99;
              border-style: solid; font-size:8pt; 
              background-color:#E8E8E6;
              empty-cells: show; text-align: left }

        .nodisp { font-size: 1% ; height: 10px; color:#F2F2F2 }
        
        .bar div { 
          text-align: center;
          float: left;
        }

        .executing { 
          background-color: #7CFF44;
        }      

        .activity { 
          border-right: solid 1px #FFFFFF
        }      

        .noright { 
          border-right: none;
          cursor: default;
        }      

        .noleft { 
          border-left: none;
          cursor: default;
        }      

        .div_Cpu { 
          border-top: solid 3px #1EC162;
          background-color: #19A352;
          color: #19A352;
          border-bottom: solid 3px #0E5D2F;
        }      

        .div_cpu_time { 
          border-top: solid 3px #1EC162;
          background-color: #19A352;
          color: #19A352;
          border-bottom: solid 3px #0E5D2F;
        }      

        .div_buffer_gets { 
          border-top: solid 3px #FFAB9A;
          background-color: #D89182;
          color: #D89182;
          border-bottom: solid 3px #BC7E72;
          cursor: default;
        }      

        .div_queuing_time {
          border-top: solid 3px #C5FFB7;
          background-color: #86FF86;
          color: #86FF86;
          border-bottom: solid 3px #74DD74;
          cursor: default;
        } 

        .div_Scheduler { 
          border-top: solid 3px #C5FFB7;
          background-color: #86FF86;
          color: #86FF86;
          border-bottom: solid 3px #74DD74;
          cursor: default;
        }      

        .div_application_wait_time { 
          border-top: solid 3px #C8C6FF;
          background-color: #B5B3E7;
          color: #B5B3E7;
          border-bottom: solid 3px #7C7B9E;
          cursor: default;
        }      

        .div_Application { 
          border-top: solid 3px #C8C6FF;
          background-color: #B5B3E7;
          color: #B5B3E7;
          border-bottom: solid 3px #7C7B9E;
          cursor: default;
        }      

        .div_sql_cpu_time { 
          border-top: solid 3px #1EC162;
          background-color: #19A352;
          color: #19A352;
          border-bottom: solid 3px #0E5D2F;
          cursor: default;
        }      

       .div_UserIO { 
          border-top: solid 3px #49A1FF;
          background-color: #3C85D2;
          color: #3C85D2;
          border-bottom: solid 3px #28588B;
          cursor: default;
        }      

       .div_user_io_wait_time { 
          border-top: solid 3px #FF8C6C;
          background-color: #E27C60;
          color: #E27C60;
          border-bottom: solid 3px #B1614B;
          cursor: default;
        }      

       .div_SystemIO { 
          border-top: solid 3px #49A1FF;
          background-color: #3C85D2;
          color: #3C85D2;
          border-bottom: solid 3px #28588B;
          cursor: default;
        }      

        .div_Concurrency { 
          border-top: solid 3px #E81B1F;
          background-color: #BE1619;
          color: #BE1619;
          border-bottom: solid 3px #760E10;
          cursor: default;
        }      

        .div_concurrency_wait_time { 
          border-top: solid 3px #E81B1F;
          background-color: #BE1619;
          color: #BE1619;
          border-bottom: solid 3px #760E10;
          cursor: default;
        }      

        .div_cluster_wait_time { 
          border-top: solid 3px #D43B61;
          background-color: #8B2740;
          color: #8B2740;
          border-bottom: solid 3px #5A1929;
          cursor: default;
        }      

        .div_Cluster { 
          border-top: solid 3px #D43B61;
          background-color: #8B2740;
          color: #8B2740;
          border-bottom: solid 3px #5A1929;
          cursor: default;
        }      

        .div_plsql_exec_time { 
          border-top: solid 3px #EDD957;
          background-color: #B1A241;
          color: #B1A241;
          border-bottom: solid 3px #8B7F33;
          cursor: default;
        }      

        .div_java_exec_time { 
          border-top: solid 3px #B40FEB;
          background-color: #8E0CB9;
          color: #8E0CB9;
          border-bottom: solid 3px #6A098B;
          cursor: default;
        }

        .div_other_wait_time { 
          border-top: solid 3px #4AFDEB;
          background-color: #41DECD;
          color: #41DECD;
          border-bottom: solid 3px #2B9389;
          cursor: default;
        }

        .div_read_reqs { 
          border-top: solid 3px #FFC846;
          background-color: #DDAD3D;
          color: #DDAD3D;
          border-bottom: solid 3px #B99133;
          cursor: default;
        }

        .div_write_reqs { 
          border-top: solid 3px #F76838;
          background-color: #BC4F2B;
          color: #BC4F2B;
          border-bottom: solid 3px #9C4223;
          cursor: default;
        }

        .div_read_bytes { 
          border-top: solid 3px #FFC846;
          background-color: #DDAD3D;
          color: #DDAD3D;
          border-bottom: solid 3px #B99133;
          cursor: default;
        }

        .div_write_bytes { 
          border-top: solid 3px #F76838;
          background-color: #BC4F2B;
          color: #BC4F2B;
          border-bottom: solid 3px #9C4223;
          cursor: default;
        }

        .div_user_fetch_count { 
          border-top: solid 3px #DFFF6C;
          background-color: #CAE762;
          color: #CAE762;
          border-bottom: solid 3px #9EB44C;
          cursor: default;
        }

        .div_Other { 
          border-top: solid 3px #4AFDEB;
          background-color: #41DECD;
          color: #41DECD;
          border-bottom: solid 3px #2B9389;
          cursor: default;
        }      

        .progress_Disp { 
          border-top: solid 3px #FFE4B5;
          background-color: #EAD1A6;
          text-align: center;
          border-bottom: solid 3px #D8C199;
          cursor: default;
          height: 13px;
          float:left;
        }      

        .progress_Nodisp { 
          border-top: solid 3px #FFE4B5;
          background-color: #EAD1A6;
          color: #E08877;
          border-bottom: solid 3px #D8C199;
          cursor: default;
          font-size: 1% ; color:#F2F2F2
          height: 13px; 
          float:left;
        }

        .pxtype_QC { 
          border-top: solid 2px #FAA589;
          background-color: #E2967C;
          border-bottom: solid 2px #BC7C67;
          cursor: default;
        }      

        .pxtype_S1 { 
          border-top: solid 2px #88F3B8;
          background-color: #79D8A4;
          border-bottom: solid 2px #60AC82;
          cursor: default;
        }      

        .pxtype_S2 { 
          border-top: solid 2px #9DF5FB;
          background-color: #87D2D7;
          border-bottom: solid 2px #70AEB2;
          cursor: default;
        }      

        .pxtype_Instance { 
          border: solid 1px #FFFFFF;
          cursor: default;
        }      

        .active_period { 
          border-top: solid 3px #A2C8E4;
          background-color: #8CADC5;
          color: #8CADC5;
          border-bottom: solid 3px #617888;
          cursor: default;
        }      

        .active_period_now { 
          border-top: solid 3px #EF5411;
          background-color: #DC4D10;
          color: #DC4D10;
          border-bottom: solid 3px #C5450E;
          cursor: default;
        }      

        .active_period_10s { 
          border-top: solid 3px #F09855;
          background-color: #DA8A4D;
          color: #DA8A4D;
          border-bottom: solid 3px #BF7943;
          cursor: default;
        }      

        .active_period_60s { 
          border-top: solid 3px #FFDC9A;
          background-color: #F0CF91;
          color: #F0CF91;
          border-bottom: solid 3px #CAAE7A;
          cursor: default;
        }      

       </style>
       <xsl:apply-templates
         select="/report/sql_monitor_report/report_parameters"/>
     </head>
     <body bgcolor="#FFFFFF">
       <h1 align="center">SQL Monitoring Report</h1>
      <xsl:apply-templates select="/report/sql_monitor_report"/>
     </body>
    </html>
  </xsl:template>

<!-- ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ -->
<!--                         begin function definitions                     -->
<!-- ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ -->
<!-- +++++++++++++++++++++++ lpad-loop template +++++++++++++++++++++++++++ -->
<xsl:template name="left-pad-loop">
<xsl:param name="n"/>
<xsl:if test="$n != 0">
  <xsl:text>.</xsl:text> 
  <xsl:call-template name="left-pad-loop">
    <xsl:with-param name="n" select="$n - 1"/>
  </xsl:call-template>
</xsl:if>
</xsl:template>

<!-- +++++++++++++++++++++++ greatest-power-two template ++++++++++++++++++ -->
<xsl:template name="greatest-power-two">
<xsl:param name="n"/>
<xsl:param name="limit">1</xsl:param>
<xsl:choose>
  <xsl:when test="$n &gt; $limit">
    <xsl:call-template name="greatest-power-two">
      <xsl:with-param name="n" select="$n"/>
      <xsl:with-param name="limit" select="$limit * 2"/>
    </xsl:call-template>
  </xsl:when>
  <xsl:otherwise>
    <xsl:value-of select="$limit"/>
  </xsl:otherwise>
</xsl:choose>
</xsl:template>

<!-- +++++++++++++++++++++++++++ lpad template +++++++++++++++++++++++++++ -->
<xsl:template name="left-pad">
<xsl:param name="col" select="'#E8E8E6'"/>
<xsl:param name="n"/>
  <xsl:if test="$n != 0">
    <font>
      <xsl:attribute name="color"><xsl:value-of 
                     select="$col"/></xsl:attribute>
      <xsl:call-template name="left-pad-loop">
        <xsl:with-param name="n" select="$n"/>
      </xsl:call-template>
    </font>
  </xsl:if>
</xsl:template>

<!-- ++++++++++++++++++++++++++ cell-value template ++++++++++++++++++++++ -->
<xsl:template name="cell-value">
  <xsl:param name="val" select="not_available"/>
  <xsl:param name="col" select="'#E8E8E6'"/>
  <xsl:choose>
    <xsl:when test="$val and $val != ''">
      <xsl:value-of select="$val"/>
    </xsl:when>
    <xsl:otherwise>
      <font>
        <xsl:attribute name="color"><xsl:value-of 
                       select="$col"/></xsl:attribute>
      .</font>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!-- ++++++++++++++++++++++++++++ normalize +++++++++++++++++++++++++++++++ -->
<xsl:template name="normalize">
<xsl:param name="n"/>
<xsl:choose>
  <xsl:when test="boolean($n) = false() or $n = ''">
    <P class="nodisp">.</P>
  </xsl:when>
  <xsl:when test="$n &lt; 100000">
    <xsl:value-of select="string($n)"/>
  </xsl:when>
  <xsl:when test="$n &lt; 1000000">
    <xsl:value-of select="concat(string(round($n div 1000)), 'K')"/>
  </xsl:when>
  <xsl:when test="$n &lt; 1000000000">
    <xsl:value-of select="concat(string(round($n div 1000000)), 'M')"/>
  </xsl:when>
  <xsl:when test="$n &lt; 1000000000000">
    <xsl:value-of select="concat(string(round($n div 1000000000)), 'G')"/>
  </xsl:when>
  <xsl:when test="$n &lt; 1000000000000000">
    <xsl:value-of select="concat(string(round($n div 1000000000000)), 'T')"/>
  </xsl:when>
  <xsl:otherwise>
   <xsl:value-of select="concat(string(round($n div 1000000000000000)), 'P')"/>
  </xsl:otherwise>
</xsl:choose>
</xsl:template>

<!-- ++++++++++++++++++++++++++++ normalize_bytes ++++++++++++++++++++++++ -->
<xsl:template name="normalize_bytes">
<xsl:param name="n"/>
<xsl:choose>
  <xsl:when test="boolean($n) = false() or $n = ''">
    <P class="nodisp">.</P>
  </xsl:when>
  <xsl:when test="$n &lt; 1024">
    <xsl:value-of select="concat(string($n), 'Bytes')"/>
  </xsl:when>
  <xsl:when test="$n &lt; 1024000">
    <xsl:value-of select="concat(format-number($n div 1024, '#0.0'), 'KB')"/>
  </xsl:when>
  <xsl:when test="$n &lt; 1048576000">
    <xsl:value-of select="concat(format-number($n div 1048576, '#0.0'), 
                                 'MB')"/>
  </xsl:when>
  <xsl:when test="$n &lt; 1073741824000">
    <xsl:value-of select="concat(format-number($n div 1073741824, '#0.0'), 
                                 'GB')"/>
  </xsl:when>
  <xsl:when test="$n &lt; 1099511627776000">
    <xsl:value-of select="concat(format-number($n div 1099511627776, '#0.0'), 
                                 'TB')"/>
  </xsl:when>
  <xsl:otherwise>
    <xsl:value-of 
        select="concat(format-number($n div 1125899906842624, '#0.0'), 'PB')"/>
  </xsl:otherwise>
</xsl:choose>
</xsl:template>

<!-- +++++++++++++++++++++++ break-text template +++++++++++++++++++++++++ -->
<xsl:template name="break-text">
  <xsl:param name="text" />
  <xsl:choose>
    <xsl:when test="contains($text, '&#10;')">
      <xsl:value-of select="substring-before($text, '&#10;')" />
      <br/>
      <xsl:call-template name="break-text">
        <xsl:with-param name="text" 
                        select="substring-after($text, '&#10;')" />
      </xsl:call-template>
    </xsl:when>
    <xsl:otherwise>
      <xsl:value-of select="$text" />
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!-- ++++++++++++++ translate monitor name template ++++++++++++++++++++++ -->
<xsl:template name="translate_monitor_name">
<xsl:param name="label"/>
<xsl:choose>
  <xsl:when test="$label = 'elapsed_time'">Elapsed</xsl:when>
  <xsl:when test="$label = 'queuing_time'">Queuing Time</xsl:when>
  <xsl:when test="$label = 'cpu_time'">Cpu</xsl:when>
  <xsl:when test="$label = 'sql_cpu_time'">Sql Cpu Time</xsl:when>
  <xsl:when test="$label = 'user_io_wait_time'">IO Time</xsl:when>
  <xsl:when
    test="$label = 'application_wait_time'">Application Wait Time</xsl:when>
  <xsl:when
    test="$label = 'concurrency_wait_time'">Concurrency Wait Time</xsl:when>
  <xsl:when test="$label = 'cluster_wait_time'">Cluster Wait Time</xsl:when>
  <xsl:when test="$label = 'other_wait_time'">Other Wait Time</xsl:when>
  <xsl:when test="$label = 'user_fetch_count'">Fetch Calls</xsl:when>
  <xsl:when test="$label = 'buffer_gets'">Buffer Gets</xsl:when>
  <xsl:when test="$label = 'plsql_exec_time'">PL/SQL Time</xsl:when>
  <xsl:when test="$label = 'java_exec_time'">Java Time</xsl:when>
  <xsl:when 
     test="$label = 'cell_offload_efficiency'">Cell Offload<br/>Efficiency
  </xsl:when>
  <xsl:when test="$label = 'read_reqs'">Read Requests</xsl:when>
  <xsl:when test="$label = 'read_bytes'">Read Bytes</xsl:when>
  <xsl:when test="$label = 'write_reqs'">Write Requests</xsl:when>
  <xsl:when test="$label = 'write_bytes'">Write Bytes</xsl:when>
  <xsl:otherwise><xsl:value-of select="$label"/></xsl:otherwise>
</xsl:choose>
</xsl:template>


<!-- ++++++++++++++++++ display-progress template ++++++++++++++++++++++++ -->
<xsl:template name="display-progress">
  <xsl:param name="percent_complete"/>
  <xsl:param name="time_left"/>

  <xsl:choose>
   <xsl:when test="$percent_complete and $percent_complete != ''">

    <!-- determine class for progress bar -->
    <xsl:variable name="class">
      <xsl:choose>
        <xsl:when test="$percent_complete &gt;= 50">
          progress_Disp
        </xsl:when>
        <xsl:otherwise>
          progress_Nodisp
        </xsl:otherwise>
      </xsl:choose>
    </xsl:variable>

    <!-- determine label -->
    <xsl:variable name="label">
      <xsl:choose>
        <xsl:when test="$time_left = 0">
          done
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of 
            select="concat(string($time_left), 's')"/>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:variable>

    <!-- determine label for tooltip -->
    <xsl:variable name="label_tooltip">
      <xsl:choose>
        <xsl:when test="$time_left = 0">
          <xsl:value-of 
            select="concat(string($percent_complete), '% (done)')"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of 
            select="concat(string($percent_complete), '%', 
                     ' (', string($time_left), 's)')"/>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:variable>

    <!-- make table -->
    <table cellspacing="0" cellpadding="0" width="100%">
    <tbody>
      <tr>
        <xsl:attribute name="title">
           <xsl:value-of select="$label_tooltip"/>
        </xsl:attribute>
       <td>
         <div>
           <xsl:attribute name="class">
             <xsl:value-of select="$class"/>
           </xsl:attribute>
           <xsl:attribute name="style">
             width:<xsl:value-of select="floor($percent_complete * 0.8)"/>%;
           </xsl:attribute>
           <xsl:choose>
             <xsl:when test="$percent_complete &gt;= 50">
               <xsl:value-of select="$label"/>
             </xsl:when>
             <xsl:otherwise>
               <P class="nodisp">.</P>
             </xsl:otherwise>
           </xsl:choose>
         </div>
         <xsl:if test="$percent_complete &lt;= 50">
           <div>
             <xsl:value-of select="concat(string($time_left), 's')"/>
           </div>
         </xsl:if>
       </td>
      </tr>
    </tbody>
    </table>
  </xsl:when>
  <xsl:otherwise>
    <P class="nodisp">.</P>
  </xsl:otherwise>
 </xsl:choose>
</xsl:template>


<!-- +++++++++++++++++++++++++ PX type Template +++++++++++++++++++++++++++ -->
<xsl:template name="px-type">
  <xsl:param name="server_set"/>
  <xsl:param name="qc_is_null"/>
  <td align="center">
    <xsl:attribute name="class">
      <xsl:choose>
        <xsl:when test="$server_set = 1">
          pxtype_S1
        </xsl:when>
        <xsl:when test="$server_set = 2">
          pxtype_S2
        </xsl:when>
        <xsl:when test="$parallel_info_node/sessions and 
                        ($server_set = 'QC' or $qc_is_null = 'yes')">
          pxtype_QC
        </xsl:when>
      </xsl:choose>
    </xsl:attribute>
    <xsl:choose>
      <xsl:when test="$server_set = 1">
        Set 1
      </xsl:when>
      <xsl:when test="$server_set = 2">
        Set 2
      </xsl:when>
      <xsl:when test="$parallel_info_node/sessions and
                      ($server_set = 'QC' or $qc_is_null = 'yes')">
        QC
      </xsl:when>
    </xsl:choose>
  </td>
</xsl:template>


<!-- ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ -->
<!--                         begin template definitions                     -->
<!-- ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ -->
<!-- ++++++++++++++++++++++++++++ main template +++++++++++++++++++++++++++ -->
<xsl:template match="sql_monitor_report">
  <xsl:apply-templates select="target"/>

  <!-- display table of binds -->
  <xsl:apply-templates select="binds"/>

  <!-- display parallel info if any -->
  <xsl:if test="$parallel_info_node/sessions">
    <xsl:apply-templates select="parallel_info"/>
  </xsl:if>

  <!-- display plan monitoring info -->
  <xsl:apply-templates select="plan_monitor"/>

  <!-- other_xml -->
  <xsl:apply-templates select="other"/>

</xsl:template>

<!-- +++++++++++++++++++++++++++++ error template +++++++++++++++++++++++++ -->
<xsl:template match="error">
    <font class="s9">
      <b>
        <xsl:value-of select="'Error: '"/>
      </b>
    </font>
    <font>
      <b>
        <xsl:value-of select="concat(@facility, '-', @number)"/>
      </b>
    </font>
    <hr size="3" align="left"/>
    <p>
      <xsl:choose>
        <xsl:when test="string(.)">
          <xsl:call-template name="break-text">    
            <xsl:with-param name="text" select="."/>
          </xsl:call-template>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="'No error message'"/>
        </xsl:otherwise>
      </xsl:choose>

    </p>
</xsl:template>

<!-- +++++++++++++++++++++++++++++ binds template +++++++++++++++++++++++++ -->
<xsl:template match="binds">
    <br/>   
    <font class="s9">
      <b>
        <xsl:value-of select="'Binds'"/>
      </b>
      <hr size="1" align="left"/>
    </font>
    <table border="1" cellspacing="0" cellpadding="0">
      <tr class="s16">
        <th align="center">Name</th>
        <th align="center">Position</th>
        <th align="center">Type</th>
        <th align="center">Value</th>
      </tr>
      <xsl:apply-templates select="bind"/>
    </table>
</xsl:template>

<!-- +++++++++++++++++++++++++++++ binds template +++++++++++++++++++++++++ -->
<xsl:template match="bind">
     <xsl:variable name="class_name">
       <xsl:choose>
         <xsl:when test="((position() + 1)mod 2) = 0">s17a</xsl:when>
         <xsl:otherwise>s17b</xsl:otherwise>
       </xsl:choose>
     </xsl:variable>

      <tr>

        <!-- define general class -->
        <xsl:attribute name="class">
          <xsl:value-of select="$class_name"/>
        </xsl:attribute>

        <td align="left"><xsl:value-of select="@name"/></td>
        <td align="center"><xsl:value-of select="@pos"/></td>
        <td align="left"><xsl:value-of select="@dtystr"/></td>
        <td align="left"><xsl:value-of select="."/></td>
      </tr>
</xsl:template>

<!-- +++++++++++++++++++++++++++++ other template +++++++++++++++++++++++++ -->
<xsl:template match="other">
  <br/>   
  <font class="s9">
    <b>
      <xsl:value-of select="'Other XML'"/>
    </b>
    <hr size="1" align="left"/>
  </font>
  <p>
    <xsl:copy-of select="."/>
  </p>
</xsl:template>


<!-- ++++++++++++++++++++++++++++ target template +++++++++++++++++++++++++ -->
<xsl:template match="target">
<xsl:if test="$target_node"> 
  <br/>
     <xsl:if test="$need_sqltext">
       <font class="s9">
         <b> SQL Text</b>
         <hr size="3" align="left"/>
       </font>
       <p><xsl:value-of select="sql_fulltext"/></p>
     </xsl:if>
     <xsl:apply-templates select="../error"/>        
     <font class="s9">
       <b> Global Information: </b>
     </font>
     <font><b><xsl:value-of select="status"/></b>
       <xsl:if test="status = 'EXECUTING'">
         <xsl:attribute name="class">
           s9 executing
         </xsl:attribute>      
       </xsl:if>
     </font>
     <hr size="3" align="left"/>
     <table border="0" cellspacing="0" cellpadding="0" width="100%">
       <tr> 
         <th width="10%" align="left">Instance ID</th>
         <th align="center" width="20">:</th>
         <td><xsl:value-of select="./@instance_id"/></td>
         <td rowspan="5" align="left" width="75%">
           <xsl:call-template name="global-stats"/>
         </td>
       </tr>
       <xsl:choose>
         <xsl:when test="$need_user and ./@session_id and ./@session_serial">
           <tr>
             <th align="left">Session</th><th>:</th>
             <td><xsl:value-of select="concat(user, ' (', ./@session_id, 
             ':', ./@session_serial, ')')"/></td>
           </tr>
         </xsl:when>
         <xsl:when test="$need_user and not(string(./@session_id))">
           <tr>
             <th align="left">Session</th><th>:</th>
             <td><xsl:value-of select="user"/></td>
           </tr>
         </xsl:when>
         <xsl:when test="not(string($need_user)) and ./@session_id">
           <tr>
           <th align="left">Session</th><th>:</th>
           <td><xsl:value-of select="concat('(', ./@session_id, ':', 
           ./@session_serial, ')')"/></td>
         </tr>
         </xsl:when>
       </xsl:choose>
       <tr> 
         <th align="left">SQL ID </th><th >:</th>
         <td><xsl:value-of select="./@sql_id"/></td>
       </tr>
       <tr> 
         <th align="left">SQL Execution ID</th><th >:</th>
         <td><xsl:value-of select="./@sql_exec_id"/></td>
       </tr>
       <tr> 
         <th align="left">Execution Started</th><th >:</th>
         <td><xsl:value-of select="./@sql_exec_start"/></td>
       </tr>
       <tr> 
         <th align="left">First Refresh Time</th><th>:</th>
         <td><xsl:value-of select="first_refresh_time"/></td>
       </tr>
       <tr> 
         <th align="left">Last Refresh Time</th><th>:</th>
         <td><xsl:value-of select="last_refresh_time"/></td>
       </tr>
       <xsl:choose>
         <xsl:when test="(string(number($queuing_duration)) = 'NaN') or
                         not($queuing_duration) or 
                         ($queuing_duration = 0)">
           <tr> 
             <th align="left">Duration</th><th>:</th>
             <td><xsl:value-of select="concat(duration, 's')"/></td>
           </tr>
         </xsl:when>
         <xsl:otherwise>
           <tr> 
             <th align="left">Duration (Queued) </th><th>:</th>
             <td><xsl:value-of select="concat(duration, 
                                              's (', 
                                              $queuing_duration,
                                              's)')"/></td>
           </tr>
         </xsl:otherwise>
       </xsl:choose>
       <xsl:choose>
         <xsl:when test="$need_module and $need_action">
           <tr>
             <th align="left">Module/Action</th><th>:</th>
             <td><xsl:value-of select="concat(module, '/', action)"/></td>
           </tr>
         </xsl:when>
         <xsl:when test="$need_module and not(string($need_action))">
           <tr>
             <th align="left">Module/Action</th><th>:</th>
             <td><xsl:value-of select="concat(module, '/-')"/></td>
           </tr>
         </xsl:when>
         <xsl:when test="not(string($need_module)) and $need_action">
           <tr>
             <th align="left">Module/Action</th><th>:</th>
             <td><xsl:value-of select="concat('-/', action)"/></td>
           </tr>
         </xsl:when>
       </xsl:choose>

       <xsl:if test="$need_service">  
         <tr> 
           <th align="left">Service</th><th>:</th> 
           <td><xsl:value-of select="service"/></td>
         </tr>
       </xsl:if>

       <xsl:if test="$need_program">  
         <tr> 
           <th align="left">Program</th><th>:</th> 
           <td><xsl:value-of select="program"/></td>
         </tr>
       </xsl:if>

       <xsl:if test="$need_plseoid or $need_plsesid">
         <tr>
           <th align="left">PL/SQL Entry Ids</th><th>:</th>
           <xsl:choose>
             <xsl:when test="$need_plseoid and $need_plsesid">
               <td><xsl:value-of select="concat(plsql_entry_object_id, ',', 
               plsql_entry_subprogram_id)"/></td>
             </xsl:when>
             <xsl:when test="$need_plseoid and not(string($need_plsesid))">
               <td><xsl:value-of select="concat(plsql_entry_object_id, '/-')"/>
               </td>
             </xsl:when>
             <xsl:when test="$not(string(need_plseoid)) and $need_plsesid">
               <td><xsl:value-of select="concat('-/', 
                                         plsql_entry_subprogram_id)"/>
               </td>
             </xsl:when>
           </xsl:choose>
         </tr>
       </xsl:if>
       <xsl:if test="$need_plsoid or $need_plssid">
         <tr>
           <th align="left">PL/SQL Ids (Obj/Sub)</th><th>:</th>
           <xsl:choose>
             <xsl:when test="$need_plsoid and $need_plssid">
               <td><xsl:value-of select="concat(plsql_object_id, ',', 
               plsql_subprogram_id)"/></td>
             </xsl:when>
             <xsl:when test="$need_plsoid and not(string($need_plssid))">
               <td><xsl:value-of select="concat(plsql_object_id, '/-')"/>
               </td>
             </xsl:when>
             <xsl:when test="$not(string(need_plsoid)) and $need_plssid">
               <td><xsl:value-of select="concat('-/', plsql_subprogram_id)"/>
               </td>
             </xsl:when>
           </xsl:choose>
         </tr>
       </xsl:if>  
       <xsl:if test="$val_px_requested &gt; $val_px_allocated"> 
         <tr>
           <th align="left">DOP Downgrade</th><th>:</th> 
           <td><xsl:value-of select="concat(round((($val_px_requested - 
                $val_px_allocated) div $val_px_requested)*100), '%')"/>
           </td>
         </tr>
       </xsl:if>
       <xsl:if test="$need_fetch">
         <tr> 
           <th align="left">Fetch Calls</th><th>:</th>
           <td><xsl:value-of
                   select="../stats[@type='monitor']/
                           stat[@name = 'user_fetch_count']"/>
           </td>
         </tr>
       </xsl:if>
     </table>
   </xsl:if> 
</xsl:template>


<!-- +++++++++++++++++++++ global stats template ++++++++++++++++++++++++ -->
<xsl:template name="global-stats">
 <table border="1" ora_borderstyle="headeronly" cellspacing="0"
        cellpadding="0" align="left" width="100%">

   <!-- call appropriate template -->
       <tr class="s16">
         <th width="19%">Buffer Gets</th>
         <xsl:if test="$need_read or $need_write">
           <th width="19%">IO Requests</th>
         </xsl:if>
         <xsl:if test="$need_cell_filter">
           <th width="4%" align="center">Cell Offload<br/>Efficiency</th>
         </xsl:if>
         <th width="19%">Database Time</th>
         <th width="19%">Wait Activity</th>
       </tr>
       <tr class="s17">

         <td>
           <xsl:call-template name="build-linear-pie">
             <xsl:with-param 
               name="node_list" 
               select="../stats[@type='monitor']/
                       stat[@name = 'buffer_gets']"/>
           </xsl:call-template>
         </td>
    
         <xsl:if test="$need_read or $need_write">
         <td>
           <xsl:call-template name="build-linear-pie">
             <xsl:with-param 
               name="node_list" 
               select="../stats[@type='monitor']/
                       stat[@name = 'read_reqs' or @name = 'write_reqs']"/>
             <xsl:with-param 
               name="node_tooltip_list" 
               select="../stats[@type='monitor']/
                       stat[@name = 'read_bytes' or @name = 'write_bytes']"/>
           </xsl:call-template>
        </td>
        </xsl:if>

        <!-- Cell offload efficiency -->
        <xsl:if test="$need_cell_filter">
          <xsl:call-template name="cell-offload-efficiency">
             <xsl:with-param 
               name="disk_bytes" 
               select="sum(../stats[@type='monitor']/
                       stat[@name = 'read_bytes' or @name = 'write_bytes'])"/>
             <xsl:with-param 
               name="offload_efficiency" 
               select="../stats[@type='monitor']/
                       stat[@name = 'cell_offload_efficiency']"/>
           </xsl:call-template>
        </xsl:if>

        <td>
           <xsl:call-template name="build-linear-pie">
             <xsl:with-param 
               name="node_list" 
                 select="../stats[@type='monitor']/
                       stat[@name!='elapsed_time' and 
                       substring(@name, 1 + string-length(@name)-5)='_time']"/>
             <xsl:with-param name="unit" select="'s'"/>
           </xsl:call-template>
         </td>
    
         <td>
           <xsl:apply-templates select="../activity_sampled">
             <xsl:with-param name="max_count" select="@wait_count"/>
             <xsl:with-param name="sample_count" select="@wait_count"/>
             <xsl:with-param name="filter_out" select="'Cpu'"/>
             <xsl:with-param name="total_count" select="$total_wait_count"/>
           </xsl:apply-templates>
         </td>
       </tr>
 </table>
</xsl:template>


<!-- +++++++++++++++++++++ parallel info template ++++++++++++++++++++++++ -->
<xsl:template match="parallel_info">
 <br/>   
 <br/>   
 <font class="s9">
   <!-- bookmark -->
   <a name="pq_stats"/>

   <b> Parallel Execution Details 
   <xsl:if test="@dop"> 
   (DOP=<xsl:value-of select="@dop"/>
   <xsl:if test="$val_px_requested &gt; $val_px_allocated"> 
      , Servers Requested=<xsl:value-of select=
           "/report/sql_monitor_report/target/servers_requested"/>
   </xsl:if>
   <xsl:if test="$val_px_allocated">
      , Servers Allocated=<xsl:value-of select=
           "/report/sql_monitor_report/target/servers_allocated"/>
   </xsl:if>) 
   </xsl:if>
  </b>
   <hr size="2" align="left"/>
 </font>
 <table border="0" cellspacing="0" cellpadding="0">
   <xsl:if test="@inst_count &gt; 1">
     <tr> 
       <th align="left">Instances</th>
       <td>: <xsl:value-of select="@inst_count"/></td>
     </tr>
   </xsl:if>
 </table>

 <xsl:apply-templates select="./sessions"/>

 <xsl:if test="./instances/instance">
   <xsl:apply-templates select="./instances"/>
 </xsl:if>

 
</xsl:template>


<!-- ++++++++++++++++++ parallel info/sessions template +++++++++++++ -->
<xsl:template match="sessions">
 <br/>
 <table border="1" ora_borderstyle="headeronly" 
   cellspacing="0" cellpadding="0" width="100%">
   <caption align="top"><b>PX Processes Drill-Down</b></caption>
   <tr class="s16">
     <xsl:if test="../@inst_count &gt; 1">
       <th width="3%">Instance</th>
     </xsl:if>
     <th width="7%">Name</th>
     <th width="3%">Type</th>
     <xsl:if test="../@server_group_count &gt; 1">
       <th width="2%">Group#</th>
     </xsl:if>
     <th width="2%">Server#</th>
     <th width="20%">Buffer Gets</th>
     <xsl:if test="$need_read or $need_write">
       <th width="20%">IO Requests</th>
     </xsl:if>
     <xsl:if test="$need_cell_filter">
        <th width="4%" align="center">Cell Offload<br/>Efficiency</th>
     </xsl:if>
     <th width="20%">Database Time</th>
     <th width="20%">Wait Events</th>
   </tr>

   <xsl:apply-templates select="./session"/>

 </table>
</xsl:template>


<!-- ++++++++++ parallel info/sessions/session template +++++++++++ -->
<xsl:template match="session">
  <!-- display PX session row -->
     <tr class="s17">

       <!-- Instance ID (when count > 1) -->
       <xsl:if test="../../@inst_count &gt; 1">
         <td style="text-align: center;">
           <b><xsl:value-of select="@inst_id"/></b>
         </td>
       </xsl:if>

       <!-- Process Name -->
       <td align="left">
         <b><xsl:value-of select="@process_name"/></b>
       </td>

       <!-- Process Type -->
       <xsl:call-template name="px-type">
         <xsl:with-param name="server_set" select="@server_set"/>
         <xsl:with-param name="qc_is_null" select="'yes'"/>
       </xsl:call-template>

       <!-- Group # if group count > 1 -->
       <xsl:if test="../../@server_group_count &gt; 1">
          <td align="right" style="text-align: center;">
            <b><xsl:value-of select="@server_group"/></b>
          </td>
       </xsl:if>

       <!-- Slave # -->
       <td align="right" style="text-align: center;">
         <b><xsl:value-of select="@server_num"/></b>
       </td>

       <!-- Buffer Gets -->
       <td>
         <xsl:call-template name="build-linear-pie">
           <xsl:with-param 
             name="node_list" 
             select="./stats[@type='monitor']/
                     stat[@name = 'buffer_gets']"/>
           <xsl:with-param 
             name="max_value" 
             select="../../sessions/@max_buffer_gets"/>
           <xsl:with-param 
             name="total_value"
             select="sum(/report/sql_monitor_report/stats[@type='monitor']/
                         stat[@name='buffer_gets'])"/>
         </xsl:call-template>
       </td>

       <!-- IO Requests (and bytes as a tooltip) -->
       <xsl:if test="$need_read or $need_write">
       <td>
         <xsl:call-template name="build-linear-pie">
           <xsl:with-param 
             name="node_list" 
             select="./stats[@type='monitor']/
                     stat[@name = 'read_reqs' or @name = 'write_reqs']"/>
           <xsl:with-param 
             name="node_tooltip_list" 
             select="./stats[@type='monitor']/
                     stat[@name = 'read_bytes' or @name = 'write_bytes']"/>
           <xsl:with-param 
             name="max_value" 
             select="../../sessions/@max_io_reqs"/>
           <xsl:with-param 
             name="total_value"
             select="sum(/report/sql_monitor_report/stats[@type='monitor']/
                         stat[@name='read_reqs' or @name='write_reqs'])"/>
         </xsl:call-template>
       </td>
       </xsl:if>

       <!-- Cell offload efficiency -->
       <xsl:if test="$need_cell_filter">

          <xsl:call-template name="cell-offload-efficiency">
             <xsl:with-param 
               name="disk_bytes" 
               select="sum(./stats[@type='monitor']/
                       stat[@name = 'read_bytes' or @name = 'write_bytes'])"/>
             <xsl:with-param 
               name="offload_efficiency" 
               select="./stats[@type='monitor']/
                       stat[@name = 'cell_offload_efficiency']"/>
           </xsl:call-template>
       </xsl:if>

       <!-- DB Time -->
       <td>
         <xsl:call-template name="build-linear-pie">
           <xsl:with-param 
             name="node_list" 
             select="./stats[@type='monitor']/stat[@name!='elapsed_time' and 
                     substring(@name, 1 + string-length(@name)-5)='_time']"/>
           <xsl:with-param 
             name="max_value" 
             select="../../sessions/@max_elapsed_time"/>
           <xsl:with-param 
             name="total_value" 
             select="/report/sql_monitor_report/stats[@type='monitor']/
                     stat[@name='elapsed_time']"/>
           <xsl:with-param name="unit" select="'s'"/>
         </xsl:call-template>
       </td>

       <!-- Wait Activity -->
       <td>
         <xsl:apply-templates 
             select="./activity_sampled">
           <xsl:with-param name="sample_count" select="@wait_count"/>
           <xsl:with-param name="max_count" 
             select="../../../sessions/@max_wait_count"/>
           <xsl:with-param name="filter_out" select="'Cpu'"/>
           <xsl:with-param name="total_count" select="$total_wait_count"/>
         </xsl:apply-templates>
       </td>
     </tr>
</xsl:template>

<!-- ++++++++++++++++++++ parallel info/instances template +++++++++++++ -->
<xsl:template match="instances">

 <br/>
 <br/>
 <table border="1" ora_borderstyle="headeronly" 
   cellspacing="0" cellpadding="0" width="100%">
   <caption align="top"><b>Instance Drill-Down</b></caption>
   <tr class="s16">
     <th instance="5%">Instance</th>
     <th width="35%">Process Names</th>
     <th width="14%">Buffer Gets</th>
     <xsl:if test="$need_read or $need_write">
       <th width="14%">IO Requests</th>
     </xsl:if>
     <xsl:if test="$need_cell_filter">
        <th width="4%" align="center">Cell Offload<br/>Efficiency</th>
     </xsl:if>
     <th width="14%">Database Time</th>
     <th width="14%">Wait Events</th>
   </tr>

   <xsl:apply-templates select="instance"/>

 </table>
</xsl:template>


<!-- ++++++++++ parallel info/instances/instance template +++++++++++ -->
<xsl:template match="instance">
  <!-- display PX session row -->
     <tr class="s17">
       <xsl:variable name="inst_id" select="@inst_id"/>
       <td align="center">
         <b><xsl:value-of select="$inst_id"/></b>
       </td>
       <td>
         <xsl:for-each select="../../sessions/session[@inst_id = $inst_id]">

           <xsl:choose>
               <xsl:when test="@process_name = 'PX Coordinator'">
                 <div class="pxtype_QC pxtype_Instance" style="float:left">
                   <xsl:value-of select="'QC'"/>
                 </div>
               </xsl:when>
               <xsl:when test="@server_set = '1'">
                 <div class="pxtype_S1 pxtype_Instance" style="float:left">
                   <xsl:value-of select="@process_name"/>
                 </div>
               </xsl:when>
               <xsl:when test="@server_set = '2'">
                 <div class="pxtype_S2 pxtype_Instance" style="float:left">
                   <xsl:value-of select="@process_name"/>
                 </div>
               </xsl:when>
           </xsl:choose>
         </xsl:for-each>
       </td>
       <td>
         <xsl:call-template name="build-linear-pie">
           <xsl:with-param 
             name="node_list" 
             select="./stats[@type='monitor']/
                     stat[@name = 'buffer_gets']"/>
           <xsl:with-param 
             name="max_value" 
             select="../../instances/@max_buffer_gets"/>
           <xsl:with-param 
             name="total_value"
             select="sum(/report/sql_monitor_report/stats[@type='monitor']/
                         stat[@name='buffer_gets'])"/>
         </xsl:call-template>
       </td>

       <xsl:if test="$need_read or $need_write">
       <td>
         <xsl:call-template name="build-linear-pie">
           <xsl:with-param 
             name="node_list" 
             select="./stats[@type='monitor']/
                     stat[@name = 'read_reqs' or @name = 'write_reqs']"/>
           <xsl:with-param 
             name="node_tooltip_list" 
             select="./stats[@type='monitor']/
                     stat[@name = 'read_bytes' or @name = 'write_bytes']"/>
           <xsl:with-param 
             name="max_value" 
             select="../../instances/@max_io_reqs"/>
           <xsl:with-param 
             name="total_value"
             select="sum(/report/sql_monitor_report/stats[@type='monitor']/
                         stat[@name='read_reqs' or @name='write_reqs'])"/>
         </xsl:call-template>
       </td>
       </xsl:if>

       <!-- Cell offload efficiency -->
       <xsl:if test="$need_cell_filter">
          <xsl:call-template name="cell-offload-efficiency">
             <xsl:with-param 
               name="disk_bytes" 
               select="sum(./stats[@type='monitor']/
                       stat[@name = 'read_bytes' or @name = 'write_bytes'])"/>
             <xsl:with-param 
               name="offload_efficiency" 
               select="./stats[@type='monitor']/
                       stat[@name = 'cell_offload_efficiency']"/>
           </xsl:call-template>
       </xsl:if>

       <td>
         <xsl:call-template name="build-linear-pie">
           <xsl:with-param 
             name="node_list" 
             select="./stats[@type='monitor']/stat[@name!='elapsed_time' and 
                     substring(@name, 1 + string-length(@name)-5)='_time']"/>
           <xsl:with-param 
             name="max_value" 
             select="../../instances/@max_elapsed_time"/>
           <xsl:with-param 
             name="total_value" 
             select="/report/sql_monitor_report/stats[@type='monitor']/
                     stat[@name='elapsed_time']"/>
           <xsl:with-param name="unit" select="'s'"/>
         </xsl:call-template>
       </td>
       <td>
         <xsl:apply-templates 
             select="./activity_sampled">
           <xsl:with-param name="sample_count" select="@wait_count"/>
           <xsl:with-param name="max_count" select="../../../instances/@max_wait_count"/>
           <xsl:with-param name="filter_out" select="'Cpu'"/>
           <xsl:with-param name="total_count" select="$total_activity_count"/>
         </xsl:apply-templates>
       </td>
     </tr>
</xsl:template>


<!-- +++++++++++++++ activity sampled template (graphical) ++++++++++++++++ -->
<xsl:template match="activity_sampled">
  <xsl:param name="sample_count"/>
  <xsl:param name="max_count"/>
  <xsl:param name="total_count" select="0"/>
  <xsl:param name="filter_out" select="'Nothing'"/>
  <xsl:param name="cpu_only" select="'no'"/>

  <!-- few local variables -->
  <xsl:variable name="activity_perc">
    <xsl:choose>
      <xsl:when test="$sample_count">
        <xsl:value-of
            select="($sample_count * 100) div $total_count"/>
      </xsl:when>
      <xsl:otherwise>
        0
      </xsl:otherwise>
    </xsl:choose>      
  </xsl:variable>

  <!-- width of that bar -->
  <xsl:variable name="total_activity_width" 
    select="ceiling(($sample_count * 100) div $max_count)"/>

  <!--  now display activity -->
  <table border="0" cellspacing="0" cellpadding="0" width="100%">
  <tbody>
  <xsl:if test="$activity_perc &gt; 0 and
                $total_activity_width &gt; 1">
  <td>
    <xsl:attribute name="style">
      width:<xsl:value-of select="$total_activity_width"/>%
    </xsl:attribute>
    <xsl:variable name="factor">
      <xsl:choose>
        <xsl:when test="$total_activity_width &gt; 20">
          0.9
        </xsl:when>
        <xsl:when test="$total_activity_width &gt; 10">
          0.8
        </xsl:when>
        <xsl:otherwise>
          0.6
        </xsl:otherwise>
      </xsl:choose>
    </xsl:variable>
    <table class="graph" cellspacing="0" cellpadding="0" width="100%">
     <tbody>
       <tr>
          <td class="bar">
            <xsl:for-each select="./activity">
              <xsl:variable name="class_activity_width" 
                select="floor(((. * 100) div $sample_count)*$factor)"/>
              <xsl:variable name="absolute_class_activity_perc" >
                <xsl:choose>
                  <xsl:when test="(. * 100) div $total_count &lt; 10">
                    <xsl:value-of
                       select="format-number((. * 100) div $total_count,
                                             '#.0')"/>
                  </xsl:when>
                  <xsl:otherwise>
                    <xsl:value-of
                       select="format-number((. * 100) div $total_count,
                                             '#.0')"/>
                  </xsl:otherwise>
                </xsl:choose>
              </xsl:variable>
              <xsl:if test="@class != $filter_out and
                            (@class = 'Cpu' or $cpu_only='no') and
                            $class_activity_width &gt; 1">
                <div>
                  <xsl:attribute name="class">
                    activity div_<xsl:value-of 
                      select="translate(translate(@class, ' ', ''), '/', '')"/>
                  </xsl:attribute> 
                  <xsl:attribute name="style">
                    width:<xsl:value-of select="$class_activity_width"/>%
                  </xsl:attribute>
                  <xsl:variable name="event">
                    <xsl:if test="@event">
                      <xsl:value-of select="concat(':', @event)"/>
                    </xsl:if>
                  </xsl:variable>
                  <xsl:attribute name="title">
                    <xsl:value-of 
                     select="concat(@class, $event, ' (',
                             string(.), '->',
                             string($absolute_class_activity_perc),
                             '%)')"/>
                  </xsl:attribute>
                  <P class="nodisp">.</P>
                </div>
              </xsl:if>
            </xsl:for-each>
          </td>
        </tr>
      </tbody>
    </table>
    </td>
    </xsl:if>
    <td align="left">
      <xsl:if test="$activity_perc &gt; 0">
        <xsl:choose>
          <xsl:when test="$activity_perc = 0">
            0%
          </xsl:when>
          <xsl:when test="$activity_perc &lt; 1">
            <xsl:value-of select="format-number($activity_perc, '#.00')"/>%
          </xsl:when>
          <xsl:when test="$activity_perc &lt; 10">
            <xsl:value-of select="format-number($activity_perc, '#.0')"/>%
          </xsl:when>
        <xsl:otherwise>
            <xsl:value-of select="format-number($activity_perc, '#')"/>%
        </xsl:otherwise>
        </xsl:choose>
      </xsl:if>
    </td>
  </tbody>
  </table>
</xsl:template>


<!-- ++++++++++++++++++++++++++ cell-offload-efficiency +++++++++++++++++ -->
<xsl:template name="cell-offload-efficiency">

  <!-- offload efficiency factor -->
  <xsl:param name="offload_efficiency"/>

  <!-- bytes exchanged with disk -->
  <xsl:param name="disk_bytes"/>

  <td align="right">

  <xsl:if test="$offload_efficiency and $offload_efficiency != 0">

    <!-- offload bytes -->
    <xsl:variable name="offload_bytes" 
      select="round($disk_bytes div $offload_efficiency)"/>

    <!-- offload bytes normalized -->
    <xsl:variable name="offload_bytes_norm">
      <xsl:call-template name="normalize_bytes">
        <xsl:with-param name="n"
            select="$offload_bytes"/>
      </xsl:call-template>
    </xsl:variable>

    <!-- disk bytes normalized -->
    <xsl:variable name="disk_bytes_norm">
      <xsl:call-template name="normalize_bytes">
        <xsl:with-param name="n"
            select="$disk_bytes"/>
      </xsl:call-template>
    </xsl:variable>

    <!-- offload efficiency factor percentage -->
    <xsl:variable name="offload_efficiency_perc"
      select="concat(format-number((1 - (1 div $offload_efficiency)) * 100, 
              '#0.00'), '%')"/>

      <xsl:value-of select="$offload_efficiency_perc"/>
      <xsl:attribute name="title">
        <xsl:value-of select="concat($disk_bytes_norm, ' -> ',
                                     $offload_bytes_norm)"/>
      </xsl:attribute>
  </xsl:if>
  
  </td>

</xsl:template>



<!-- +++++++++++++++++++++++++++++ build-linear-pie +++++++++++++++++++++ -->
<xsl:template name="build-linear-pie">

  <!-- list of stat nodes for which to build the pie chart -->
  <xsl:param name="node_list"/>

  <!-- list of additional tooltip info -->
  <xsl:param name="node_tooltip_list" select="null"/>

  <!-- optional: when building a table of pie-charts, value for the -->
  <!--           biggest one                                        -->
  <xsl:param name="max_value" select="null"/>

  <!-- optional: when building a table of pie-charts, value for the -->
  <!--           overall total (over all pie charts) -->
  <xsl:param name="total_value" select="null"/>

  <!-- unit. Use 's' for time, nothing for plain counts  -->
  <xsl:param name="unit" select="null"/>

  <!-- sum of java and pl/sql exec time  -->
  <xsl:variable name="sum_java_plsql"
                select="sum($node_list[@name='java_exec_time' or
                                       @name='plsql_exec_time'])"/>

  <!-- sum of all values -->
  <xsl:variable name="sum_value" select="sum($node_list) - $sum_java_plsql"/>


  <!-- total value is the sum of all values when not specified -->
  <xsl:variable name="total_valuem">
    <xsl:choose>
      <xsl:when test="$total_value">
        <xsl:value-of select="$total_value"/>
      </xsl:when>
      <xsl:otherwise>
        <xsl:value-of select="$sum_value"/>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:variable>

  <!-- percentage of this pie-chart over the grand total  -->
  <xsl:variable name="perc_value" 
    select="($sum_value * 100) div $total_valuem"/>

  <!-- percentage of this pie-chart compare to the other pie-charts  -->
  <xsl:variable name="total_value_width" >
    <xsl:choose>
      <xsl:when test="$max_value">
        <xsl:value-of 
          select="ceiling((($sum_value * 100) div $max_value)*0.6)"/>
      </xsl:when>
      <xsl:otherwise>
        <xsl:value-of select="80"/>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:variable>

  <!-- class prefix, e.g. 'div_'. This allows to build a unique name  -->
  <!-- for the class associated to each pie-chart                    -->
  <xsl:variable name="class-prefix" select="'div_'"/>

  <!--  now display the bars -->
   <table cellspacing="0" cellpadding="0" width="100%">
   <tbody>
   <xsl:if test="$perc_value &gt; 0 and
                 $total_value_width &gt; 1">
     <td>
       <xsl:attribute name="style">
         width:<xsl:value-of select="$total_value_width"/>%
       </xsl:attribute>
     <table class="graph" cellspacing="0" cellpadding="0" width="100%">
     <tbody>
       <tr>
          <td class="bar">
            <xsl:for-each select="$node_list">

              <!--  determine node name and value -->
              <xsl:variable name="node_name">
                <xsl:choose>
                  <xsl:when test="@name != 'cpu_time' or
                                  $sum_java_plsql = 0">
                    <xsl:value-of select="@name"/>
                  </xsl:when>
                  <xsl:otherwise>
                    <xsl:value-of select="'sql_cpu_time'"/>
                  </xsl:otherwise>
                </xsl:choose>
              </xsl:variable>
              <xsl:variable name="node_value">
                <xsl:choose>
                  <xsl:when test="@name != 'cpu_time' or
                                  $sum_java_plsql = 0">
                    <xsl:value-of select="."/>
                  </xsl:when>
                  <xsl:otherwise>
                    <xsl:value-of select=". - $sum_java_plsql"/>
                  </xsl:otherwise>
                </xsl:choose>
              </xsl:variable>

              <!--  compute relative and absolute percentatge -->
              <xsl:variable name="relative_value_perc" 
                  select="floor(($node_value * 100) div $sum_value)"/>
              <xsl:variable name="raw_value_perc" 
                  select="($node_value * 100) div $total_valuem"/>
              <xsl:variable name="absolute_value_perc">
                <xsl:choose>
                  <xsl:when test="$raw_value_perc &lt; 10">
                    <xsl:value-of
                      select="format-number($raw_value_perc, '#.0')"/>
                  </xsl:when>
                  <xsl:otherwise>
                    <xsl:value-of
                      select="format-number($raw_value_perc, '#')"/>
                  </xsl:otherwise>
                </xsl:choose>
              </xsl:variable>

              <!--  determine tool-tip of value -->
              <xsl:variable name="label_name">
                <xsl:call-template name="translate_monitor_name">
                  <xsl:with-param name="label" select="$node_name"/>
                </xsl:call-template>
              </xsl:variable>

              <xsl:variable name="label_value">
                <xsl:choose>
                  <xsl:when test="$unit = 's'">
                    <xsl:value-of 
                      select="concat($label_name, ' - ',
                                     format-number(($node_value div 1000000), 
                                                   '#.00'), 's')"/>
                  </xsl:when>
                  <xsl:otherwise>
                    <xsl:variable name="norm_value">
                      <xsl:call-template name="normalize">
                        <xsl:with-param name="n" select="$node_value"/>
                      </xsl:call-template>
                    </xsl:variable>
                    <xsl:choose>
                      <xsl:when test="$node_tooltip_list">
                        <xsl:variable name="node_pos" select="position()"/>
                        <xsl:variable name="norm_value_bytes">
                          <xsl:call-template name="normalize_bytes">
                            <xsl:with-param name="n"
                            select="$node_tooltip_list[position()=$node_pos]"/>
                          </xsl:call-template>
                        </xsl:variable>
                        <xsl:value-of select="concat($label_name, ': ',
                                           string($norm_value), $unit, ' = ',
                                           $norm_value_bytes)"/>
                      </xsl:when>
                      <xsl:otherwise>
                        <xsl:value-of select="concat($label_name, ': ',
                                           string($norm_value), $unit)"/>
                      </xsl:otherwise>
                    </xsl:choose>
                  </xsl:otherwise>
                </xsl:choose>
              </xsl:variable>
              <xsl:variable name="label_perc">
                <xsl:if test="$absolute_value_perc != 100">
                  <xsl:value-of 
                      select="concat(' (', $absolute_value_perc, '%)')"/>
                </xsl:if>
              </xsl:variable>

              <xsl:variable name="label_final"
                select="concat($label_value, $label_perc)"/>

              <!--  build div if not 0 -->
              <xsl:if test="$relative_value_perc &gt; 0">
                <div>
                  <xsl:attribute name="class">
                    <xsl:value-of select="$class-prefix"/>
                    <xsl:value-of 
                      select="translate(translate(@name, ' ', ''), '/', '')"/>
                  </xsl:attribute> 
                  <xsl:attribute name="style">
                    width:<xsl:value-of select="$relative_value_perc"/>%
                  </xsl:attribute>
                  <xsl:attribute name="title">
                    <xsl:value-of select="$label_final"/>
                  </xsl:attribute>
                  <P class="nodisp">.</P>
                </div>
              </xsl:if>
            </xsl:for-each>
          </td>
       </tr>
     </tbody>
     </table>
     </td> 
  </xsl:if>

  <td align="left">
    <xsl:if test="$perc_value &gt; 0">

       <!--  compute value for total lable -->
       <xsl:variable name="label_value_total">
         <xsl:choose>
           <xsl:when test="$unit = 's'">
             <xsl:value-of 
               select="concat(string(round($sum_value div 1000000)), 's')"/>
           </xsl:when>
           <xsl:otherwise>
             <xsl:variable name="norm_value">
               <xsl:call-template name="normalize">
                 <xsl:with-param name="n" select="$sum_value"/>
               </xsl:call-template>
             </xsl:variable>
             <xsl:value-of 
               select="concat(string($norm_value), $unit)"/>
           </xsl:otherwise>
         </xsl:choose>
       </xsl:variable>

       <!-- add percentage of total if there are multiple values -->
       <xsl:variable name="label_name_total">
         <xsl:choose>
           <xsl:when test="$total_value and $perc_value &lt; 0.1">
             <xsl:value-of
               select="concat($label_value_total, ' (&lt;0.1%)')"/>
           </xsl:when>
           <xsl:when test="$total_value and $perc_value &lt; 10">
             <xsl:value-of
               select="concat($label_value_total, ' (',
                              format-number($perc_value, '#.0'), '%)')"/>
           </xsl:when>
           <xsl:when test="$perc_value != 100">
             <xsl:value-of
               select="concat($label_value_total, ' (',
                              format-number($perc_value, '#'), '%)')"/>
           </xsl:when>
           <xsl:otherwise>
             <xsl:value-of select="$label_value_total"/>
           </xsl:otherwise>
         </xsl:choose>
       </xsl:variable>

       <!-- display result -->
       <xsl:value-of select="$label_name_total"/>

    </xsl:if>
  </td>
  </tbody>
  </table>
</xsl:template>

<!-- ++++++++++++++++++++++++ plan-timeline template +++++++++++++++++++++ -->
<xsl:template name="plan-timeline">

 <xsl:choose>
  <xsl:when test="stats[@type = 'plan_monitor']/stat[@name='duration']">

    <!-- first compute absolute number -->
    <xsl:variable
      name="line_start"
      select="stats[@type='plan_monitor']/stat[@name='from_sql_exec_start']"/>
    <xsl:variable
      name="line_duration"
      select="stats[@type = 'plan_monitor']/stat[@name='duration']"/>
    <xsl:variable
      name="line_end"
      select="stats[@type = 'plan_monitor']/stat[@name='from_most_recent']"/>
    <xsl:variable
      name="line_total_duration"
      select="$line_start + $line_duration + $line_end"/>

    <!-- express all of those in % -->
    <xsl:variable name="line_start_percent"
      select="round(($line_start * 100) div $line_total_duration)"/>
    <xsl:variable name="line_duration_percent">
      <xsl:choose>
        <xsl:when test="round(($line_duration * 100)
                                     div $line_total_duration) &gt; 0">
          <xsl:value-of select="round(($line_duration * 100)
                                        div $line_total_duration)"/>
        </xsl:when>
        <xsl:otherwise>
          1
        </xsl:otherwise>
      </xsl:choose>
    </xsl:variable>
    <xsl:variable name="line_end_percent"
      select="round(($line_end * 100) div $line_total_duration)"/>

    <!-- get active date -->
    <xsl:variable name="factive" 
      select="stats[@type='plan_monitor']/stat[@name='first_active']"/>
    <xsl:variable name="lactive" 
      select="stats[@type='plan_monitor']/stat[@name='last_active']"/>

    <!-- label title -->
    <xsl:variable name="label_title_first">
      <xsl:choose>
        <xsl:when test="substring($factive, 1, 11) = 
                        substring($lactive, 1, 11)">
          <xsl:value-of select="concat(substring($factive, 11),
                                       ' -> ',
                                       substring($lactive, 11))"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="concat($factive, ' -> ', $lactive)"/>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:variable>

    <!-- final label title -->
    <xsl:variable name="label_title">
      <xsl:choose>
        <xsl:when test="$status = 'EXECUTING'">
          <xsl:choose>
            <xsl:when test="$line_end = 0">
              <xsl:value-of select="concat($label_title_first,
                            ' (most recent)')"/>
            </xsl:when>
            <xsl:otherwise>
              <xsl:value-of
                select="concat($label_title_first, ' (', string($line_end),
                               's from most recent)')"/>
            </xsl:otherwise>
          </xsl:choose>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="$label_title_first"/>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:variable>

    <!-- build the table with percentages for each time -->
    <table class="graph" cellspacing="0" cellpadding="0" width="100%">
     <tbody>
       <tr>
         <xsl:attribute name="title">
           <xsl:value-of select="$label_title"/>
         </xsl:attribute> 

          <td class="bar">
              <!--  build div if not 0 -->
              <xsl:if test="$line_start_percent &gt; 0">
                <div>
                  <xsl:attribute name="style">
                    <xsl:choose>
                      <xsl:when test="$line_start_percent +
                                      $line_duration_percent &gt; 100">
                        width:<xsl:value-of
                                select="100 - $line_duration_percent"/>%
                      </xsl:when>
                      <xsl:otherwise>
                        width:<xsl:value-of select="$line_start_percent"/>%
                      </xsl:otherwise>
                    </xsl:choose>
                  </xsl:attribute>
                  <P class="nodisp">.</P>
                </div>
              </xsl:if>
              <div class="active_period">
                <xsl:attribute name="class">
                  <xsl:choose>
                    <xsl:when test="$status != 'EXECUTING'">
                      active_period
                    </xsl:when>
                    <xsl:when test="$line_end = 0">
                      active_period_now
                    </xsl:when>
                    <xsl:when test="$line_end &lt;= 10">
                      active_period_10s
                    </xsl:when>
                    <xsl:when test="$line_end &lt;= 60">
                      active_period_60s
                    </xsl:when>
                    <xsl:otherwise>
                      active_period
                    </xsl:otherwise>
                  </xsl:choose>
                  <xsl:value-of select="$line_duration_percent"/>%
                </xsl:attribute>
                <xsl:attribute name="style">
                  width:<xsl:value-of select="$line_duration_percent"/>%
                </xsl:attribute>
                <P class="nodisp">.</P>
              </div>
              <xsl:if test="$line_end_percent &gt; 0">
                <div>
                  <xsl:attribute name="style">
                    <xsl:choose>
                      <xsl:when test="$line_end_percent +
                                      $line_start_percent +
                                      $line_duration_percent > 100">
                        width:<xsl:value-of
                                 select="100 - $line_start_percent
                                             - $line_duration_percent"/>%
                      </xsl:when>
                      <xsl:otherwise>
                        width:<xsl:value-of select="$line_end_percent"/>%
                      </xsl:otherwise>
                    </xsl:choose>
                  </xsl:attribute>
                  <P class="nodisp">.</P>
                </div>
              </xsl:if>
          </td>
       </tr>
     </tbody>
    </table>
  </xsl:when>
  <xsl:otherwise>
    <P class="nodisp">.</P>
  </xsl:otherwise>
 </xsl:choose>
</xsl:template>


<!-- ++++++++++++++++++++++++ plan monitor template +++++++++++++++++++++++ -->
<xsl:template match="plan_monitor">
<xsl:if test="operation">
   <br/>
   <br/>
   <font class="s9">
     <!-- bookmark -->
     <a name="xplan"/>

     <b> SQL Plan Monitoring Details (Plan Hash Value=<xsl:value-of select="/report/sql_monitor_report/target/@sql_plan_hash"/>)</b>
     <hr size="1" align="left"/>
   </font>

   <!-- display plan lines -->
   <table width="100%" border="1" cellspacing="0" cellpadding="0" 
                       ora_borderstyle="headeronly">
     <tr class="s16">
       <!-- optimizer stats -->
       <th colspan="2">Id</th>

       <th>Operation</th>

       <th>Name</th>
       <th>Estimated<br/>Rows</th>
       <th>Cost</th>

       <!-- plan monitor stats -->
       <th width="120">Active Period
       <br/>(<xsl:value-of select="$query_duration"/>s)</th>
       <th>Execs</th>
       <th>Rows</th>

       <!-- Memory/Max Memory (real) -->
       <th>
         <xsl:choose>
           <xsl:when 
             test="/report/sql_monitor_report/target/status = 'EXECUTING'">
             Memory
           </xsl:when>
           <xsl:otherwise>
             Memory<br/>(Max)
           </xsl:otherwise>
         </xsl:choose>
       </th>

       <!-- Temp/Max Temp (real) -->
       <th>
         <xsl:choose>
           <xsl:when test="/report/sql_monitor_report/target/
                           status = 'EXECUTING'">
             Temp
           </xsl:when>
           <xsl:otherwise>
             Temp<br/>(Max)
           </xsl:otherwise>
         </xsl:choose>
       </th>

       <!-- IOs -->
       <th width="120">IO Requests</th>

       <!-- Cell Filtering -->
       <xsl:if test="$need_cell_filter">
        <th align="center">Cell Offload<br/>Efficiency</th>
       </xsl:if>

       <!-- CPU Activity -->
       <th width="120">CPU Activity</th>

       <!-- Wait Activity -->
       <th width="120">Wait Activity</th>

       <!-- Progress Info -->
       <xsl:if test="$need_progress">
         <th width="100">Progress</th>
       </xsl:if>
     </tr>

     <xsl:for-each select="operation">
       <xsl:apply-templates select="."/>   
     </xsl:for-each>
   </table>
  </xsl:if>
</xsl:template>


<!-- +++++++++++++++++++++++ plan operation template ++++++++++++++++++++++ -->
<xsl:template match="operation">
  <tr class="s17a" align="right">
    <xsl:variable name="class_name">
      <xsl:choose>
        <xsl:when test="(./@id mod 2) = 0">s17a</xsl:when>
        <xsl:otherwise>s17b</xsl:otherwise>
      </xsl:choose>
    </xsl:variable>

    <xsl:variable name="row_color">
      <xsl:choose>
        <xsl:when test="(./@id mod 2) = 0">#F5F5F5</xsl:when>
        <xsl:otherwise>#F1F5DD</xsl:otherwise>
      </xsl:choose>
    </xsl:variable>

    <!-- define general class -->
    <xsl:attribute name="class">
      <xsl:value-of select="$class_name"/>
    </xsl:attribute>

    <xsl:variable name="most_recent"
      select="stats[@type='plan_monitor']/stat[@name='from_most_recent']"/>
  
    <!-- show -> if execution is active here -->
    <td align="center">
      <xsl:attribute name="class">
        <xsl:choose>
          <xsl:when test="not($most_recent)">
            noright
          </xsl:when>
          <xsl:when test="@px_type = 'QC'">
            noright pxtype_QC
          </xsl:when>
          <xsl:when test="@px_type = 1">
            noright pxtype_S1
          </xsl:when>
          <xsl:when test="@px_type = 2">
            noright pxtype_S2
          </xsl:when>
          <xsl:otherwise>
            noright pxtype_QC
          </xsl:otherwise>
        </xsl:choose>
      </xsl:attribute> 

      <xsl:choose>
        <xsl:when test="$status != 'EXECUTING' or
                          not($most_recent) or $most_recent != 0">
          <P class="nodisp">.</P>
        </xsl:when>
        <xsl:otherwise>
          <b>
            <xsl:value-of select="'->'"/>
          </b>
        </xsl:otherwise>
      </xsl:choose>
    </td>

    <!-- display id -->
    <td align="right">
      <xsl:value-of select="./@id"/>
      <xsl:attribute name="class">
        <xsl:choose>
          <xsl:when test="not($most_recent)">
            noleft
          </xsl:when>
          <xsl:when test="@px_type = 'QC'">
            noleft pxtype_QC
          </xsl:when>
          <xsl:when test="@px_type = 1">
            noleft pxtype_S1
          </xsl:when>
          <xsl:when test="@px_type = 2">
            noleft pxtype_S2
          </xsl:when>
          <xsl:otherwise>
            noleft pxtype_QC
          </xsl:otherwise>
        </xsl:choose>
      </xsl:attribute> 
      <xsl:attribute name="title">
        <xsl:choose>
          <xsl:when
            test="not($most_recent) and
                  status = 'EXECUTING'">Not yet executed</xsl:when>
          <xsl:when test="not($most_recent)">Never executed</xsl:when>
          <xsl:when test="@px_type = 'QC'">PX Coordinator</xsl:when>
          <xsl:when test="@px_type = 1">PX Set 1</xsl:when>
          <xsl:when test="@px_type = 2">PX Set 2</xsl:when>
          <xsl:otherwise>Serial Execution</xsl:otherwise>
        </xsl:choose>
      </xsl:attribute> 
    </td>

    <!-- display operation -->
    <td align="left">
       <!-- indent operation using depth if needed -->
       <xsl:call-template name="left-pad">
         <xsl:with-param name="n" select="./@depth"/>
         <xsl:with-param name="col" select="$row_color"/>
       </xsl:call-template>

       <!-- display operation name and options -->
       <xsl:value-of select="concat(@name, ' ', @options)"/>
    </td>

    <!-- display object name -->
    <td align="left">
      <xsl:value-of select="object/name"/>
      <xsl:if test="not(object/name)">
        <P class="nodisp">.</P>
      </xsl:if>
    </td>

    <!-- display optimizer stats -->
    <!-- cardinality (estimated) -->
    <td>
      <xsl:call-template name="normalize">
        <xsl:with-param name="n" select="optimizer/cardinality"/>
      </xsl:call-template>
    </td>

    <!-- cost -->
    <td>
      <xsl:call-template name="normalize">
        <xsl:with-param name="n" select="optimizer/cost"/>
      </xsl:call-template>
    </td>

    <!-- activive period -->
    <td align="left">
      <xsl:call-template name="plan-timeline"/>
    </td>

    <!-- executions -->
    <td>
      <xsl:call-template name="normalize">
        <xsl:with-param name="n" 
              select="stats[@type='plan_monitor']/stat[@name='starts']"/>
        <xsl:with-param name="col" select="$row_color"/>
      </xsl:call-template>
    </td>

    <!-- cardinality (real) -->
    <td>
      <xsl:call-template name="normalize">
        <xsl:with-param name="n" 
              select="stats[@type='plan_monitor']/stat[@name='cardinality']"/>
      </xsl:call-template>
    </td>

    <!-- Memory/Max Memory (real) -->
    <td>
       <xsl:call-template name="normalize_bytes">
         <xsl:with-param name="n">
           <xsl:choose>
             <xsl:when test="/report/sql_monitor_report/target/
                             status = 'EXECUTING'">
               <xsl:value-of select="stats[@type='plan_monitor']/
                                     stat[@name='memory']"/>
             </xsl:when>
             <xsl:otherwise>
               <xsl:value-of select="stats[@type='plan_monitor']/
                                     stat[@name='max_memory']"/>
             </xsl:otherwise>
           </xsl:choose>     
         </xsl:with-param>         
       </xsl:call-template>
    </td>

    <!-- Temp/Max Temp (real) -->
    <td>
       <xsl:call-template name="normalize_bytes">
        <xsl:with-param name="n">
           <xsl:choose>
             <xsl:when test="/report/sql_monitor_report/target/
                             status = 'EXECUTING'">
               <xsl:value-of select="stats[@type='plan_monitor']/
                                     stat[@name='temp']"/>
             </xsl:when>
             <xsl:otherwise>
               <xsl:value-of select="stats[@type='plan_monitor']/
                                     stat[@name='max_temp']"/>
             </xsl:otherwise>
           </xsl:choose>              
         </xsl:with-param>
       </xsl:call-template>
    </td>

    <!-- IOs for this row -->
    <td align="left">
         <xsl:call-template name="build-linear-pie">
           <xsl:with-param 
             name="node_list" 
             select="./stats[@type='plan_monitor']/
                     stat[@name = 'read_reqs' or @name = 'write_reqs']"/>
           <xsl:with-param 
             name="node_tooltip_list" 
             select="./stats[@type='plan_monitor']/
                     stat[@name = 'read_bytes' or @name = 'write_bytes']"/>
           <xsl:with-param 
             name="max_value" 
             select="../../plan_monitor/@max_io_reqs"/>
           <xsl:with-param 
             name="total_value"
             select="sum(/report/sql_monitor_report/stats[@type='monitor']/
                         stat[@name='read_reqs' or @name='write_reqs'])"/>
         </xsl:call-template>
    </td>

    <!-- Cell Offload Efficiency -->
    <xsl:if test="$need_cell_filter">

      <xsl:call-template name="cell-offload-efficiency">
        <xsl:with-param 
               name="disk_bytes" 
               select="sum(./stats[@type='plan_monitor']/
                       stat[@name = 'read_bytes' or @name = 'write_bytes'])"/>
        <xsl:with-param 
               name="offload_efficiency" 
               select="./stats[@type='plan_monitor']/
                       stat[@name = 'cell_offload_efficiency']"/>
      </xsl:call-template>

    </xsl:if>

    <!-- CPU activity for this row -->
    <td align="left">
      <xsl:choose>
        <xsl:when test="./activity_sampled">
          <xsl:apply-templates select="./activity_sampled">
            <xsl:with-param name="sample_count" select="@cpu_count"/>
            <xsl:with-param name="max_count" 
              select="/report/sql_monitor_report/plan_monitor/
                      @max_cpu_count"/>
            <xsl:with-param name="cpu_only" select="'yes'"/>
            <xsl:with-param name="total_count" select="$total_cpu_count"/>
          </xsl:apply-templates>
        </xsl:when>
        <xsl:otherwise>
          <P class="nodisp">.</P>
        </xsl:otherwise>
      </xsl:choose>
    </td>

    <!-- WAIT activity for this row -->
    <td align="left">
      <xsl:choose>
        <xsl:when test="./activity_sampled">
          <xsl:apply-templates select="./activity_sampled">
            <xsl:with-param name="sample_count" select="@wait_count"/>
            <xsl:with-param name="max_count" 
              select="/report/sql_monitor_report/plan_monitor/
                      @max_wait_count"/>
            <xsl:with-param name="filter_out" select="'Cpu'"/>
            <xsl:with-param name="total_count" select="$total_wait_count"/>
          </xsl:apply-templates>
        </xsl:when>
        <xsl:otherwise>
          <P class="nodisp">.</P>
        </xsl:otherwise>
      </xsl:choose>
    </td>

    <!-- Progress Info -->
    <xsl:if test="$need_progress">
      <td align="left">
        <xsl:call-template name="display-progress">
          <xsl:with-param name="time_left"
            select="stats[@type='plan_monitor']/stat[@name='time_left']"/>
          <xsl:with-param 
            name="percent_complete"
            select="stats[@type='plan_monitor']/
                    stat[@name='percent_complete']"/>
        </xsl:call-template>
      </td>
    </xsl:if>
  </tr>
</xsl:template>

<!-- +++++++++++++++++++++++ auto-refresh template +++++++++++++++++++++++ -->
<xsl:template match="report_parameters">
  <xsl:if test="./auto_refresh and 
              /report/sql_monitor_report/target/status = 'EXECUTING'">
  <meta http-equiv="refresh">
    <xsl:attribute name="content">
      <xsl:value-of select="auto_refresh"/>
    </xsl:attribute>
  </meta>
</xsl:if>
</xsl:template>

</xsl:stylesheet>
