<xsl:stylesheet version="1.0" xmlns:xsl="http://www.w3.org/1999/XSL/Transform">
  <xsl:variable name="parallel_info_node" 
    select="/report/sql_monitor_report/parallel_info"/>
  <xsl:variable name="target_node" 
    select="/report/sql_monitor_report/target"/>
  <xsl:variable name="query_duration" 
    select="/report/sql_monitor_report/target/duration"/>
  <xsl:variable name="queuing_duration" 
    select="round(/report/sql_monitor_report/
                  stats[@type='monitor']/stat[@name='queuing_time'] 
                  div 1000000)"/>
  <xsl:variable name="status" 
    select="/report/sql_monitor_report/target/status"/>
  <xsl:variable name="need_progress" 
    select="$status = 'EXECUTING' and
            /report/sql_monitor_report/plan_monitor/*/
            stats[@type='plan_monitor']/stat[@name='time_left']"/>
  <xsl:variable name="need_temp" 
    select="/report/sql_monitor_report/plan_monitor/*/
            stats[@type='plan_monitor']/stat[@name='max_temp']"/>
  <xsl:variable name="need_read" 
    select="/report/sql_monitor_report/plan_monitor/*/
            stats[@type='plan_monitor']/stat[@name='read_reqs']"/>
  <xsl:variable name="need_write" 
    select="/report/sql_monitor_report/plan_monitor/*/
            stats[@type='plan_monitor']/stat[@name='write_reqs']"/>
  <xsl:variable name="need_fetch" 
    select="/report/sql_monitor_report/stats[@type='monitor']/
           stat[@name='user_fetch_count']"/>
  <xsl:variable name="need_cell_filter" 
    select="/report/sql_monitor_report/plan_monitor/*/
          stats[@type='plan_monitor']/stat[@name='cell_offload_efficiency']"/>
  <xsl:variable name="need_memory" 
    select="/report/sql_monitor_report/plan_monitor/*/
            stats[@type='plan_monitor']/stat[@name='max_memory']"/>
  <xsl:variable name="total_activity_count" 
     select="/report/sql_monitor_report/activity_sampled/@count"/>
  <xsl:variable name="total_wait_count" 
     select="/report/sql_monitor_report/activity_sampled/@wait_count"/>
  <xsl:variable name="need_sqltext" 
    select="/report/sql_monitor_report/target/sql_fulltext"/>
  <xsl:variable name="need_user" 
    select="/report/sql_monitor_report/target/user"/>
  <xsl:variable name="need_module" 
    select="/report/sql_monitor_report/target/module"/>
  <xsl:variable name="need_action" 
    select="/report/sql_monitor_report/target/action"/>
  <xsl:variable name="need_service" 
    select="/report/sql_monitor_report/target/service"/>
  <xsl:variable name="need_program" 
    select="/report/sql_monitor_report/target/program"/>
  <xsl:variable name="need_plseoid"
    select="/report/sql_monitor_report/target/plsql_entry_object_id"/>
  <xsl:variable name="need_plsesid"
    select="/report/sql_monitor_report/target/plsql_entry_subprogram_id"/>
  <xsl:variable name="need_plsoid"
    select="/report/sql_monitor_report/target/plsql_object_id"/>
  <xsl:variable name="need_plssid"
    select="/report/sql_monitor_report/target/plsql_subprogram_id"/>
  <xsl:variable name="val_px_requested"
    select="/report/sql_monitor_report/target/servers_requested"/>
  <xsl:variable name="val_px_allocated"
    select="/report/sql_monitor_report/target/servers_allocated"/>
  <xsl:template match="/">
    <html>
     <head>
     <title> SQL Monitor Report </title>
      <style type="text/css">
         body, table, input, select, textarea
         {font:normal normal 8pt Verdana,Arial;
          text-decoration:none;
          color:#000000; empty-cells:show;
          cellspacing:5; cellpadding:5;
          border-collapse: collapse; 
         }
         .thead
           { font:bold 8pt Arial,Helvetica,Geneva,sans-serif;
             color:White;
             background:#0066CC;
             padding-left:3px;
             padding-right:3px;
             padding-bottom:2px;
             padding-top:2px;
           }
         .tcell { padding-left:3px;
              padding-right:3px;
              padding-bottom:2px;
              padding-top:2px}
         .s9 {font-size:10pt;color:#006699}
         .row0 { background-color:#ffffff}
         .row1 { background-color:#ffffcc}
      </style>

     </head>
     <body bgcolor="#FFFFFF">
       <h1 align="center">SQL Monitoring Report</h1>
      <xsl:apply-templates select="/report/sql_monitor_report"/>
     </body>
    </html>
  </xsl:template>

<!-- ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ -->
<!--                         begin function definitions                     -->
<!-- ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ -->
<!-- +++++++++++++++++++++++ lpad-loop template +++++++++++++++++++++++++++ -->
<xsl:template name="left-pad-loop">
<xsl:param name="n"/>
<xsl:if test="$n != 0">
  <xsl:text>.</xsl:text> 
  <xsl:call-template name="left-pad-loop">
    <xsl:with-param name="n" select="$n - 1"/>
  </xsl:call-template>
</xsl:if>
</xsl:template>

<!-- +++++++++++++++++++++++ greatest-power-two template ++++++++++++++++++ -->
<xsl:template name="greatest-power-two">
<xsl:param name="n"/>
<xsl:param name="limit">1</xsl:param>
<xsl:choose>
  <xsl:when test="$n &gt; $limit">
    <xsl:call-template name="greatest-power-two">
      <xsl:with-param name="n" select="$n"/>
      <xsl:with-param name="limit" select="$limit * 2"/>
    </xsl:call-template>
  </xsl:when>
  <xsl:otherwise>
    <xsl:value-of select="$limit"/>
  </xsl:otherwise>
</xsl:choose>
</xsl:template>

<!-- +++++++++++++++++++++++++++ lpad template +++++++++++++++++++++++++++ -->
<xsl:template name="left-pad">
<xsl:param name="n"/>
  <xsl:if test="$n != 0">
    <font color="#E8E8E6">
    <xsl:call-template name="left-pad-loop">
      <xsl:with-param name="n" select="$n"/>
    </xsl:call-template>
    </font>
  </xsl:if>
</xsl:template>

<!-- ++++++++++++++++++++++++++ cell-value template ++++++++++++++++++++++ -->
<xsl:template name="cell-value">
  <xsl:param name="val" select="not_available"/>
  <xsl:param name="col" select="'#E8E8E6'"/>
  <xsl:choose>
    <xsl:when test="$val and $val != ''">
      <xsl:value-of select="$val"/>
    </xsl:when>
    <xsl:otherwise>
      <font>
        <xsl:attribute name="color"><xsl:value-of 
                       select="$col"/></xsl:attribute>
      .</font>
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!-- ++++++++++++++++++++++++++++ normalize +++++++++++++++++++++++++++++++ -->
<xsl:template name="normalize">
<xsl:param name="n"/>
<xsl:param name="def" select="./not_available"/>
<xsl:choose>
  <xsl:when test="boolean($n) = false() or $n = ''">
    <xsl:value-of select="$def"/>        
  </xsl:when>
  <xsl:when test="$n &lt; 100000">
    <xsl:value-of select="concat(string($n), ' ')"/>
  </xsl:when>
  <xsl:when test="$n &lt; 1000000">
    <xsl:value-of select="concat(string(round($n div 1000)), 'K')"/>
  </xsl:when>
  <xsl:when test="$n &lt; 1000000000">
    <xsl:value-of select="concat(string(round($n div 1000000)), 'M')"/>
  </xsl:when>
  <xsl:when test="$n &lt; 1000000000000">
    <xsl:value-of select="concat(string(round($n div 1000000000)), 'G')"/>
  </xsl:when>
  <xsl:when test="$n &lt; 1000000000000000">
    <xsl:value-of select="concat(string(round($n div 1000000000000)), 'T')"/>
  </xsl:when>
  <xsl:otherwise>
    <xsl:value-of select="concat(string(round($n div 1000000000000000)), 
                          'P')"/>
  </xsl:otherwise>
</xsl:choose>
</xsl:template>

<!-- ++++++++++++++++++++++++++++ normalize_bytes ++++++++++++++++++++++++ -->
<xsl:template name="normalize_bytes">
<xsl:param name="n"/>
<xsl:choose>
  <xsl:when test="boolean($n) = false() or $n = ''">
    <P class="nodisp">.</P>
  </xsl:when>
  <xsl:when test="$n &lt; 100000">
    <xsl:value-of select="string($n)"/>
  </xsl:when>
  <xsl:when test="$n &lt; 1000000">
    <xsl:value-of select="concat(string(round($n div 1024)), 'KB')"/>
  </xsl:when>
  <xsl:when test="$n &lt; 1000000000">
    <xsl:value-of select="concat(string(round($n div 1048576)), 'MB')"/>
  </xsl:when>
  <xsl:when test="$n &lt; 1000000000000">
    <xsl:value-of select="concat(string(round($n div 1073741824)), 'GB')"/>
  </xsl:when>
  <xsl:when test="$n &lt; 1000000000000000">
    <xsl:value-of select="concat(string(round($n div 1099511627776)), 'TB')"/>
  </xsl:when>
  <xsl:otherwise>
    <xsl:value-of select="concat(string(round($n div 1125899906842624)),
                                 'PB')"/>
  </xsl:otherwise>
</xsl:choose>
</xsl:template>

<!-- +++++++++++++++++++++++ break-text template +++++++++++++++++++++++++ -->
<xsl:template name="break-text">
  <xsl:param name="text" />
  <xsl:choose>
    <xsl:when test="contains($text, '&#10;')">
      <xsl:value-of select="substring-before($text, '&#10;')" />
      <br/>
      <xsl:call-template name="break-text">
        <xsl:with-param name="text" 
                        select="substring-after($text, '&#10;')" />
      </xsl:call-template>
    </xsl:when>
    <xsl:otherwise>
      <xsl:value-of select="$text" />
    </xsl:otherwise>
  </xsl:choose>
</xsl:template>

<!-- ++++++++++++++ translate monitor name template ++++++++++++++++++++++ -->
<xsl:template name="translate_monitor_name">
<xsl:param name="label"/>
<xsl:choose>
  <xsl:when test="$label = 'elapsed_time'">Elapsed<br/>Time(s)</xsl:when>
  <xsl:when test="$label = 'queuing_time'">Queuing<br/>Time(s)</xsl:when>
  <xsl:when test="$label = 'cpu_time'">Cpu<br/>Time(s)</xsl:when>
  <xsl:when test="$label = 'sql_cpu_time'">Sql Cpu Time</xsl:when>
  <xsl:when test="$label = 'user_io_wait_time'">IO<br/>Waits(s)</xsl:when>
  <xsl:when
    test="$label = 'application_wait_time'">Application<br/>Waits(s)</xsl:when>
  <xsl:when
    test="$label = 'concurrency_wait_time'">Concurrency<br/>Waits(s)</xsl:when>
  <xsl:when test="$label = 'cluster_wait_time'">Cluster<br/>Waits(s)</xsl:when>
  <xsl:when test="$label = 'other_wait_time'">Other<br/>Waits(s)</xsl:when>
  <xsl:when test="$label = 'user_fetch_count'">Fetch<br/>Calls</xsl:when>
  <xsl:when test="$label = 'buffer_gets'">Buffer<br/>Gets</xsl:when>
  <xsl:when test="$label = 'disk_reads'">Reads</xsl:when>
  <xsl:when test="$label = 'direct_writes'">Writes</xsl:when>
  <xsl:when test="$label = 'plsql_exec_time'">PL/SQL<br/>Time(s)</xsl:when>
  <xsl:when test="$label = 'java_exec_time'">Java<br/>Time(s)</xsl:when>
  <xsl:when test="$label = 'cell_offload_efficiency'">Cell<br/>Offload
  </xsl:when>
  <xsl:when test="$label = 'read_reqs'">Read<br/>Reqs</xsl:when>
  <xsl:when test="$label = 'read_bytes'">Read<br/>Bytes</xsl:when>
  <xsl:when test="$label = 'write_reqs'">Write<br/>Reqs</xsl:when>
  <xsl:when test="$label = 'write_bytes'">Write<br/>Bytes</xsl:when>
  <xsl:otherwise><xsl:value-of select="$label"/></xsl:otherwise>
</xsl:choose>
</xsl:template>


<!-- ++++++++++++++++++ display-progress template ++++++++++++++++++++++++ -->
<xsl:template name="display-progress">
  <xsl:param name="percent_complete"/>
  <xsl:param name="time_left"/>

  <xsl:if test="$percent_complete and $percent_complete != ''">
    <!-- determine label for tooltip -->
    <xsl:value-of select="concat(string($percent_complete), '%')"/>
  </xsl:if>

</xsl:template>


<!-- +++++++++++++++++++++++++ PX type Template +++++++++++++++++++++++++++ -->
<xsl:template name="px-type">
  <xsl:param name="server_set"/>
  <xsl:param name="qc_is_null"/>
  <td align="left">
    <xsl:choose>
      <xsl:when test="$server_set = 1">
        Set 1
      </xsl:when>
      <xsl:when test="$server_set = 2">
        Set 2
      </xsl:when>
      <xsl:when test="$parallel_info_node/sessions and
                      ($server_set = 'QC' or $qc_is_null = 'yes')">
        QC
      </xsl:when>
    </xsl:choose>
  </td>
</xsl:template>


<!-- ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ -->
<!--                         begin template definitions                     -->
<!-- ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ -->
<!-- ++++++++++++++++++++++++++++ main template +++++++++++++++++++++++++++ -->
<xsl:template match="sql_monitor_report">

  <!-- display target info -->
  <xsl:apply-templates select="target"/>

  <!-- display table of binds -->
  <xsl:apply-templates select="binds"/>

  <!-- display table of global stats -->
  <xsl:call-template name="global-stats"/>

  <!-- display parallel info if any -->
  <xsl:if test="$parallel_info_node/sessions">
    <xsl:apply-templates select="parallel_info"/>
  </xsl:if>

  <!-- display plan monitoring info -->
  <xsl:apply-templates select="plan_monitor"/>

  <!-- other_xml -->
  <xsl:apply-templates select="other"/>

</xsl:template>

<!-- +++++++++++++++++++++++++++++ error template +++++++++++++++++++++++++ -->
<xsl:template match="error">
    <font class="s9">
      <b>
        <xsl:value-of select="'Error: '"/>
      </b>
    </font>
    <font>
      <b>
        <xsl:value-of select="concat(@facility, '-', @number)"/>
      </b>
    </font>
    <hr size="3" align="left" ora_widthc="30"/>
      <xsl:choose>
        <xsl:when test="string(.)">
          <xsl:call-template name="break-text">    
            <xsl:with-param name="text" select="."/>
          </xsl:call-template>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="'No error message'"/>
        </xsl:otherwise>
      </xsl:choose>
      <br/>
      <br/>      
</xsl:template>

<!-- +++++++++++++++++++++++++++++ binds template +++++++++++++++++++++++++ -->
<xsl:template match="binds">
    <br/>   
    <font class="s9">
      <b>
        <xsl:value-of select="'Binds'"/>
      </b>
    </font>
    <table border="1" ora_borderstyle="headeronly"
           ora_headercharhorz="=" ora_headercharvert="|"
           ora_footercharhorz="="
           cellspacing="0" cellpadding="0"
           ora_widthc="120">
      <tr class="thead">
        <th align="center">Name</th>
        <th align="center">Position</th>
        <th align="center">Type</th>
        <th align="center">Value</th>
      </tr>
      <xsl:apply-templates select="bind"/>
    </table>
</xsl:template>

<!-- +++++++++++++++++++++++++++++ binds template +++++++++++++++++++++++++ -->
<xsl:template match="bind">
      <tr>
        <xsl:attribute name="class">
          row<xsl:value-of select="position() mod 2"/> tcell
        </xsl:attribute>
        <td class="tcell" align="left"><xsl:value-of select="@name"/></td>
        <td class="tcell" align="right"><xsl:value-of select="@pos"/></td>
        <td class="tcell" align="left"><xsl:value-of select="@dtystr"/></td>
        <td class="tcell" align="left"><xsl:value-of select="."/></td>
      </tr>
</xsl:template>

<!-- +++++++++++++++++++++++++++++ other template +++++++++++++++++++++++++ -->
<xsl:template match="other">
  <br/>   
  <font class="s9">
    <b>
      <xsl:value-of select="'Other XML'"/>
    </b>
    <hr size="1" align="left" ora_widthc="30"/>
  </font>
  <p>
    <xsl:copy-of select="."/>
  </p>
</xsl:template>

<!-- ++++++++++++++++++++++++++++ target template +++++++++++++++++++++++++ -->
<xsl:template match="target">
<xsl:if test="$target_node">
   <br/>   
   <br/>   
   <xsl:if test="$need_sqltext">
   <font class="s9">
     <b>SQL Text</b>
   </font>
   <hr size="1" align="left" ora_widthc="30"/>
       <xsl:value-of select="sql_fulltext"/>
   <br/>   
   <br/>   
 </xsl:if>
   <xsl:apply-templates select="../error"/>
   <font class="s9">
     <b> Global Information</b>
     <hr size="1" align="left" ora_widthc="30"/>
   </font>
   <table border="0" cellspacing="0" cellpadding="0">
     <tr> 
       <th align="left">Status</th>
       <th align="center">:</th>
       <td><xsl:value-of select="status"/></td>
     </tr>
     <tr> 
       <th align="left">Instance ID</th>
       <th align="center">:</th>
       <td><xsl:value-of select="./@instance_id"/></td>
     </tr>
       <xsl:choose>
         <xsl:when test="$need_user and ./@session_id and ./@session_serial">
           <tr>
             <th align="left">Session</th><th>:</th>
             <td><xsl:value-of select="concat(user, ' (', ./@session_id, 
             ':', ./@session_serial, ')')"/></td>
           </tr>
         </xsl:when>
         <xsl:when test="$need_user and not(string(./@session_id))">
           <tr>
             <th align="left">Session</th><th>:</th>
             <td><xsl:value-of select="user"/></td>
           </tr>
         </xsl:when>
         <xsl:when test="not(string($need_user)) and ./@session_id">
           <tr>
           <th align="left">Session</th><th>:</th>
           <td><xsl:value-of select="concat('(', ./@session_id, ':', 
           ./@session_serial, ')')"/></td>
         </tr>
         </xsl:when>
       </xsl:choose>
     <tr> 
       <th align="left">SQL ID </th><th >:</th>
       <td><xsl:value-of select="./@sql_id"/></td>
     </tr>
     <tr> 
       <th align="left">SQL Execution ID</th><th >:</th>
       <td><xsl:value-of select="./@sql_exec_id"/></td>
     </tr>
     <tr> 
       <th align="left">Execution Started</th><th >:</th>
       <td><xsl:value-of select="./@sql_exec_start"/></td>
     </tr>
     <tr> 
       <th align="left">First Refresh Time</th><th>:</th>
       <td><xsl:value-of select="first_refresh_time"/></td>
     </tr>
     <tr> 
       <th align="left">Last Refresh Time</th><th>:</th>
       <td><xsl:value-of select="last_refresh_time"/></td>
     </tr>
     <xsl:choose>
       <xsl:when test="(string(number($queuing_duration)) = 'NaN') or
                       not($queuing_duration) or 
                       ($queuing_duration = 0)">
         <tr> 
           <th align="left">Duration</th><th>:</th>
           <td><xsl:value-of select="concat(duration, 's')"/></td>
         </tr>
       </xsl:when>
       <xsl:otherwise>
         <tr> 
           <th align="left">Duration (Queued) </th><th>:</th>
           <td><xsl:value-of select="concat(duration, 
                                     's (' , $queuing_duration, 's)')"/></td>
         </tr>
       </xsl:otherwise>
     </xsl:choose>
       <xsl:choose>
         <xsl:when test="$need_module and $need_action">
           <tr>
             <th align="left">Module/Action</th><th>:</th>
             <td><xsl:value-of select="concat(module, '/', action)"/></td>
           </tr>
         </xsl:when>
         <xsl:when test="$need_module and not(string($need_action))">
           <tr>
             <th align="left">Module/Action</th><th>:</th>
             <td><xsl:value-of select="concat(module, '/-')"/></td>
           </tr>
         </xsl:when>
         <xsl:when test="not(string($need_module)) and $need_action">
           <tr>
             <th align="left">Module/Action</th><th>:</th>
             <td><xsl:value-of select="concat('-/', action)"/></td>
           </tr>
         </xsl:when>
       </xsl:choose>
     <xsl:if test="$need_service">  
       <tr> 
         <th align="left">Service</th><th>:</th> 
         <td><xsl:value-of select="service"/></td>
       </tr>
     </xsl:if>
     <xsl:if test="$need_program">  
       <tr> 
         <th align="left">Program</th><th>:</th> 
         <td><xsl:value-of select="program"/></td>
       </tr>
     </xsl:if>
     <xsl:if test="$need_plseoid or $need_plsesid">
         <tr>
           <th align="left">PLSQL Entry Ids (Object/Subprogram)</th><th>:</th>
           <xsl:choose>
             <xsl:when test="$need_plseoid and $need_plsesid">
               <td><xsl:value-of select="concat(plsql_entry_object_id, ',', 
               plsql_entry_subprogram_id)"/></td>
             </xsl:when>
             <xsl:when test="$need_plseoid and not(string($need_plsesid))">
               <td><xsl:value-of select="concat(plsql_entry_object_id, '/-')"/>
               </td>
             </xsl:when>
             <xsl:when test="not(string(need_plseoid)) and $need_plsesid">
               <td><xsl:value-of select="concat('-/',
                                                plsql_entry_subprogram_id)"/>
               </td>
             </xsl:when>
           </xsl:choose>
         </tr>
       </xsl:if>
       <xsl:if test="$need_plsoid or $need_plssid">
         <tr>
           <th align="left">PLSQL Current Ids (Object/Subprogram)</th>
           <th>:</th>
           <xsl:choose>
             <xsl:when test="$need_plsoid and $need_plssid">
               <td><xsl:value-of select="concat(plsql_object_id, ',', 
               plsql_subprogram_id)"/></td>
             </xsl:when>
             <xsl:when test="$need_plsoid and not(string($need_plssid))">
               <td><xsl:value-of select="concat(plsql_object_id, '/-')"/>
               </td>
             </xsl:when>
             <xsl:when test="not(string(need_plsoid)) and $need_plssid">
               <td><xsl:value-of select="concat('-/', plsql_subprogram_id)"/>
               </td>
             </xsl:when>
           </xsl:choose>
         </tr>
       </xsl:if>  

     <xsl:if test="$val_px_requested &gt; $val_px_allocated"> 
       <tr>
         <th align="left">DOP Downgrade</th><th>:</th> 
         <td><xsl:value-of select="concat(round((($val_px_requested - 
              $val_px_allocated) div $val_px_requested)*100), '%')"/>
         </td>
       </tr>
     </xsl:if>
     <xsl:if test="$need_fetch">
       <tr> 
         <th align="left">Fetch Calls</th><th>:</th>
         <td><xsl:value-of
                 select="../stats[@type='monitor']/
                         stat[@name = 'user_fetch_count']"/>
         </td>
       </tr>
     </xsl:if>
   </table>
</xsl:if>   
</xsl:template>


<!-- +++++++++++++++++++++ global stats template ++++++++++++++++++++++++ -->
<xsl:template name="global-stats">

  <br/>
  <font class="s9">
    <b>
      <xsl:value-of select="'Global Stats'"/>
    </b>
  </font>
  <table border="1" ora_borderstyle="headeronly"
           ora_headercharhorz="=" ora_headercharvert="|"
           ora_footercharhorz="="
           cellspacing="0" cellpadding="0">
   <!-- call text template -->
   <tr class="thead">
     <xsl:call-template name="build-linear-pie-header">
       <xsl:with-param name="node_list" 
             select="stats[@type='monitor']/stat"/>
       <xsl:with-param name="filter_out" select="'bla'"/>
     </xsl:call-template>
   </tr>
   <tr>
     <xsl:attribute name="class">
        row<xsl:value-of select="position() mod 2"/> tcell
     </xsl:attribute>
     <xsl:call-template name="build-linear-pie-text">
       <xsl:with-param name="node_list" 
             select="stats[@type='monitor']/stat"/>
       <xsl:with-param name="filter_out" select="'bla'"/>
     </xsl:call-template>
   </tr>
 </table>
</xsl:template>


<!-- +++++++++++++++++++++ parallel info template ++++++++++++++++++++++++ -->
<xsl:template match="parallel_info">
 <br/>   
 <font class="s9">
   <!-- bookmark -->
   <a name="pq_stats"/>

   <b> Parallel Execution Details 
   <xsl:if test="@dop"> 
   (DOP=<xsl:value-of select="@dop"/>
   <xsl:if test="$val_px_requested &gt; $val_px_allocated"> 
      , Servers Requested=<xsl:value-of select=
           "/report/sql_monitor_report/target/servers_requested"/>
   </xsl:if>
   <xsl:if test="$val_px_allocated">
      , Servers Allocated=<xsl:value-of select=
           "/report/sql_monitor_report/target/servers_allocated"/> 
   </xsl:if>)  
   </xsl:if>
   </b>
 </font>
 <xsl:if test="@inst_count &gt; 1">
   <table border="0" cellspacing="0" cellpadding="0">
     <tr> 
       <th align="left">Instances</th>
       <td>: <xsl:value-of select="@inst_count"/></td>
     </tr>
   </table>
 </xsl:if>

 <xsl:apply-templates select="./sessions"/>

 <xsl:if test="./instances/instance">
   <xsl:apply-templates select="./instances"/>
 </xsl:if>

 
</xsl:template>


<!-- ++++++++++++++++++ parallel info/sessions template +++++++++++++ -->
<xsl:template match="sessions">
 <br/>
 <table border="1" ora_borderstyle="headeronly"
   ora_headercharhorz="=" ora_headercharvert="|" ora_footercharhorz="="
   cellspacing="0" cellpadding="0">
   <tr class="thead">
     <xsl:if test="../@inst_count &gt; 1">
       <th align="center">Instance</th>
     </xsl:if>
     <th align="center">Name</th>
     <th align="center">Type</th>
     <xsl:if test="../@server_group_count &gt; 1">
       <th width="2%" align="center">Group#</th>
     </xsl:if>
     <th align="center">Server#</th>

     <!-- Monitor Stats -->
     <xsl:call-template name="build-linear-pie-header">
       <xsl:with-param
           name="node_list" select="./stats[@type='monitor']/stat"/>
       <xsl:with-param name="filter_out" select="'user_fetch_count'"/>
     </xsl:call-template>
       
     <th align="center">Wait Events<br/>(sample #)</th>
   </tr>

   <xsl:apply-templates select="./session"/>

 </table>
</xsl:template>


<!-- ++++++++++ parallel info/sessions/session template +++++++++++ -->
<xsl:template match="session">
  <!-- display PX session row -->
     <tr>

       <xsl:attribute name="class">
          row<xsl:value-of select="position() mod 2"/> tcell
        </xsl:attribute>

       <!-- Instance ID (when count > 1) -->
       <xsl:if test="../../@inst_count &gt; 1">
         <td class="tcell" style="text-align: center;">
           <b><xsl:value-of select="@inst_id"/></b>
         </td>
       </xsl:if>

       <!-- Process Name -->
       <td class="tcell" align="left">
         <b><xsl:value-of select="@process_name"/></b>
       </td>

       <!-- Process Type -->
       <xsl:call-template name="px-type">
         <xsl:with-param name="server_set" select="@server_set"/>
         <xsl:with-param name="qc_is_null" select="'yes'"/>
       </xsl:call-template>

       <!-- Group # if group count > 1 -->
       <xsl:if test="../../@server_group_count &gt; 1">
          <td class="tcell" align="right" style="text-align: center;">
            <b><xsl:value-of select="@server_group"/></b>
          </td>
       </xsl:if>

       <!-- Slave # -->
       <td class="tcell" align="right" style="text-align: center;">
         <b><xsl:value-of select="@server_num"/></b>
       </td>

       <!-- Monitor Stats -->
       <xsl:call-template name="build-linear-pie-text">
         <xsl:with-param
             name="node_list" select="./stats[@type='monitor']/stat"/>
         <xsl:with-param name="filter_out" select="'user_fetch_count'"/>
       </xsl:call-template>

       <!-- Activity detail -->
       <td class="tcell" >
         <xsl:apply-templates select="./activity_sampled" mode="detail">
           <xsl:with-param name="sample_count" 
             select="@wait_count"/>
           <xsl:with-param name="max_count" 
             select="../../../sessions/@max_wait_count"/>
           <xsl:with-param name="filter_out" select="'Cpu'"/>
           <xsl:with-param name="total_count" select="$total_wait_count"/>
         </xsl:apply-templates>
       </td>
     </tr>
</xsl:template>


<!-- ++++++++++++++++++++ parallel info/instances template +++++++++++++ -->
<xsl:template match="instances">

 <br/>
 <b>Instance Drill-Down</b>
 <table border="1" ora_borderstyle="headeronly"
   ora_headercharhorz="=" ora_headercharvert="|" ora_footercharhorz="="
   cellspacing="0" cellpadding="0">
   <tr class="thead">
     <th>Instance</th>
     <th>Process Names</th>

     <!-- display monitoring stats -->
     <xsl:call-template name="build-linear-pie-header">
        <xsl:with-param
          name="node_list" select="./stats[@type='monitor']/stat"/>
       <xsl:with-param name="filter_out" select="'user_fetch_count'"/>
     </xsl:call-template>

     <th>Wait Events</th>
   </tr>

   <xsl:apply-templates select="instance"/>

 </table>
</xsl:template>


<!-- ++++++++++ parallel info/instances/instance template +++++++++++ -->
<xsl:template match="instance">
  <!-- display PX session row -->
     <tr>

       <xsl:attribute name="class">
         row<xsl:value-of select="position() mod 2"/> tcell
       </xsl:attribute>

       <xsl:variable name="inst_id" select="@inst_id"/>
       <td class="tcell" align="center">
         <b><xsl:value-of select="$inst_id"/></b>
       </td>
       <td class="tcell">
         <xsl:for-each select="../../sessions/session[@inst_id = $inst_id]">

           <xsl:choose>
               <xsl:when test="@process_name = 'PX Coordinator'">
                 <div class="pxtype_QC pxtype_Instance" style="float:left">
                   <xsl:value-of select="'QC'"/>
                 </div>
               </xsl:when>
               <xsl:when test="@server_set = '1'">
                 <div class="pxtype_S1 pxtype_Instance" style="float:left">
                   <xsl:value-of select="@process_name"/>
                 </div>
               </xsl:when>
               <xsl:when test="@server_set = '2'">
                 <div class="pxtype_S2 pxtype_Instance" style="float:left">
                   <xsl:value-of select="@process_name"/>
                 </div>
               </xsl:when>
           </xsl:choose>
         </xsl:for-each>
       </td>

       <!-- display monitoring stats -->
       <xsl:call-template name="build-linear-pie-text">
          <xsl:with-param
            name="node_list" select="./stats[@type='monitor']/stat"/>
         <xsl:with-param name="filter_out" select="'user_fetch_count'"/>
       </xsl:call-template>

       <!-- Activity detail -->
       <td class="tcell">
         <xsl:apply-templates select="./activity_sampled" mode="detail">
           <xsl:with-param name="sample_count" 
             select="@wait_count"/>
           <xsl:with-param name="max_count" 
             select="../../../instances/@max_wait_count"/>
           <xsl:with-param name="filter_out" select="'Cpu'"/>
           <xsl:with-param name="total_count" select="$total_wait_count"/>
         </xsl:apply-templates>
       </td>
     </tr>
</xsl:template>


<!-- +++++++++++++++ activity sampled template (summary) +++++++++++++++ -->
<xsl:template match="activity_sampled" mode="summary">
  <xsl:param name="sample_count"/>
  <xsl:param name="max_count"/>
  <xsl:param name="total_count" select="0"/>
  <xsl:param name="filter_out" select="'Nothing'"/>

  <!-- few local variables -->
  <xsl:variable name="activity_perc">
    <xsl:choose>
      <xsl:when test="$sample_count">
        <xsl:value-of
            select="($sample_count * 100) div $total_count"/>
      </xsl:when>
      <xsl:otherwise>
        0
      </xsl:otherwise>
    </xsl:choose>      
  </xsl:variable>

  <!--  now display activity -->
  <xsl:if test="$activity_perc &gt; 0">
    <xsl:value-of select="format-number($activity_perc, '#0.00')"/>
  </xsl:if>
</xsl:template>


<!-- +++++++++++++++ activity sampled template (detailed) +++++++++++++++ -->
<xsl:template match="activity_sampled" mode="detail">
  <xsl:param name="sample_count"/>
  <xsl:param name="max_count"/>
  <xsl:param name="total_count" select="0"/>
  <xsl:param name="filter_out" select="'Nothing'"/>

  <!-- few local variables -->
  <xsl:variable name="activity_perc">
    <xsl:choose>
      <xsl:when test="$sample_count">
        <xsl:value-of
            select="($sample_count * 100) div $total_count"/>
      </xsl:when>
      <xsl:otherwise>
        0
      </xsl:otherwise>
    </xsl:choose>      
  </xsl:variable>

  <!--  now display activity -->
  <xsl:if test="$activity_perc &gt; 0">

      <!-- Display breakdown  -->
      <xsl:for-each select="./activity">

        <xsl:variable name="absolute_class_activity_perc" >
          <xsl:choose>
            <xsl:when test="(. * 100) div $total_count &lt; 10">
              <xsl:value-of
                 select="format-number((. * 100) div $total_count,
                                       '#.0')"/>
            </xsl:when>
            <xsl:otherwise>
              <xsl:value-of
                 select="format-number((. * 100) div $total_count,
                                       '#.0')"/>
            </xsl:otherwise>
          </xsl:choose>
        </xsl:variable>
        <xsl:if test="@class != $filter_out">
          <xsl:variable name="event">
            <xsl:choose>
              <xsl:when test="@event">
                <xsl:value-of select="@event"/>
              </xsl:when>
              <xsl:otherwise>
                <xsl:value-of select="@class"/>
              </xsl:otherwise>
            </xsl:choose>
          </xsl:variable>
          <xsl:value-of select="$event"/>
          <xsl:choose>
            <xsl:when test="count(./activity[@class != $filter_out]) > 1">
              <xsl:value-of select="concat(' (',
                                string(.), '->',
                                string($absolute_class_activity_perc), '%)')"/>
            </xsl:when>
            <xsl:otherwise>
              <xsl:value-of select="concat(' (', string(.), ')')"/>
            </xsl:otherwise>
            
          </xsl:choose>
          <xsl:if test="position() != last()">
            <br/>
          </xsl:if>
        </xsl:if>
      </xsl:for-each>
  </xsl:if>
</xsl:template>


<!-- ++++++++++++++++++ build-linear-pie-header (textual)++++++++++++++++ -->
<xsl:template name="build-linear-pie-header">

  <xsl:param name="filter_out" select="'bla'"/>

  <!-- iterate over all the values which are not null  -->
  <xsl:for-each
    select="/report/sql_monitor_report/stats[@type='monitor']/
            stat[($filter_out = 'bla' or @name != $filter_out) and 
                 @name != 'disk_reads' and 
                 @name != 'direct_writes' ]">
    <xsl:variable name="node_name"  select="@name"/>
    <xsl:variable name="node_value" select="."/>
    <xsl:if test="$node_value and $node_value != 0">
      <th align="center">
        <!--  determine text value -->
        <xsl:call-template name="translate_monitor_name">
          <xsl:with-param name="label" select="$node_name"/>
        </xsl:call-template>
      </th>
    </xsl:if>
  </xsl:for-each>
</xsl:template>


<!-- ++++++++++++++++++ build-linear-pie-text (textual)++++++++++++++++ -->
<xsl:template name="build-linear-pie-text">

  <!-- list of stat nodes for which to build the pie chart -->
  <xsl:param name="node_list"/>

  <!-- list of nodes to exclude -->
  <xsl:param name="filter_out" select="bla"/>

  <!-- total number of bytes read from disk -->
  <xsl:variable name="total_disk_read_bytes"
    select="$node_list[@name='read_bytes']"/>

  <!-- total number of disk bytes -->
  <xsl:variable name="total_disk_bytes"
                select="$node_list[@name='read_bytes'] +
                        $node_list[@name='write_bytes']"/>

  <!-- iterate over all the values which are not null  -->
  <xsl:for-each
    select="/report/sql_monitor_report/stats[@type='monitor']/
            stat[($filter_out = 'bla' or @name != $filter_out) and 
                 @name != 'disk_reads' and 
                 @name != 'direct_writes' ]">
    <xsl:variable name="node_name"  select="@name"/>
    <xsl:variable name="node_agg_value"  select="."/>
    <xsl:variable name="node_value">
      <xsl:choose>
        <xsl:when test="$node_name='cell_offload_efficiency'">
          <xsl:value-of 
              select="(1 - (1 div $node_list[@name=$node_name]))*100"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="$node_list[@name=$node_name]"/>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:variable>

    <xsl:variable name="unit">
      <xsl:choose>
        <xsl:when test="substring($node_name, 1 + 
                                  string-length($node_name)-5)='_time'">
          <xsl:value-of select="'s'"/>
        </xsl:when>
        <xsl:when test="substring($node_name, 1 + 
                                  string-length($node_name)-6)='_bytes'">
          <xsl:value-of select="'b'"/>
        </xsl:when>
        <xsl:when test="$node_name='cell_offload_efficiency'">
          <xsl:value-of select="'%'"/>
        </xsl:when>
      </xsl:choose>
    </xsl:variable>

    <xsl:choose>
      <xsl:when test="$node_value and $node_value != 0">
        <td class="tcell" align="right">
          <!--  compute percentage of total -->
          <xsl:variable name="raw_value_perc" 
              select="($node_value * 100) div $node_agg_value"/>
          <xsl:variable name="absolute_value_perc">
            <xsl:choose>
              <xsl:when test="$raw_value_perc &lt; 10">
                <xsl:value-of
                  select="format-number($raw_value_perc, '#.0')"/>
              </xsl:when>
              <xsl:otherwise>
                <xsl:value-of
                  select="format-number($raw_value_perc, '#')"/>
              </xsl:otherwise>
            </xsl:choose>
          </xsl:variable>

          <!--  determine text value -->
          <xsl:variable name="label_value">
            <xsl:choose>
              <xsl:when test="$unit = 's' and
                              round($node_value div 1000000) &lt; 10">
                <xsl:value-of 
                 select="format-number($node_value div 1000000, '#0.00')"/>
              </xsl:when>
              <xsl:when test="$unit = 's'">
                <xsl:value-of 
                 select="round($node_value div 1000000)"/>
              </xsl:when>
              <xsl:when test="$unit = '%'">
                <xsl:value-of 
                 select="concat(format-number($node_value, '#0.00'), '%')"/>
              </xsl:when>
              <xsl:when test="$unit = 'b'">
                <xsl:variable name="norm_value">
                  <xsl:call-template name="normalize_bytes">
                    <xsl:with-param name="n" select="$node_value"/>
                  </xsl:call-template>
                </xsl:variable>
                <xsl:value-of select="$norm_value"/>
              </xsl:when>
              <xsl:otherwise>
                <xsl:variable name="norm_value">
                  <xsl:call-template name="normalize">
                    <xsl:with-param name="n" select="$node_value"/>
                  </xsl:call-template>
                </xsl:variable>
                <xsl:value-of select="$norm_value"/>
              </xsl:otherwise>
            </xsl:choose>
          </xsl:variable>
          <xsl:if test="$label_value != 'NaN'">
            <xsl:value-of select="$label_value"/>            
          </xsl:if>
        </td>
      </xsl:when>
      <xsl:when test="$node_agg_value and $node_agg_value != 0">
        <td class="tcell"></td>
      </xsl:when>
    </xsl:choose>
  </xsl:for-each>
</xsl:template>


<!-- ++++++++++++++++++++++++ plan-timeline template +++++++++++++++++++++ -->
<xsl:template name="plan-timeline">

  <xsl:variable 
        name='most_recent'
        select="stats[@type='plan_monitor']/stat[@name='from_most_recent']"/>

  <xsl:choose>
    <xsl:when test="$most_recent">

      <!-- first compute absolute number -->
      <xsl:variable
        name="line_start"
        select="stats[@type='plan_monitor']/
                                        stat[@name='from_sql_exec_start']"/>
      <xsl:variable
        name="line_duration"
        select="stats[@type = 'plan_monitor']/stat[@name='duration']"/>
      <xsl:variable
        name="line_end"
        select="stats[@type = 'plan_monitor']/stat[@name='from_most_recent']"/>
      <xsl:variable
        name="line_total_duration"
        select="$line_start + $line_duration + $line_end"/>

      <!-- final label title -->
      <td class="tcell" align="right">
        <xsl:value-of select="$line_duration"/>
      </td>
      <td class="tcell" align="right">
        <xsl:value-of select="concat('+', $line_start)"/>
      </td>     
    </xsl:when>
    <xsl:otherwise>    
      <td class="tcell" align="right">
      </td>
      <td class="tcell" align="right">
      </td>     
    </xsl:otherwise>    
  </xsl:choose>
</xsl:template>


<!-- ++++++++++++++++++++++++ plan monitor template +++++++++++++++++++++++ -->
<xsl:template match="plan_monitor">
<xsl:if test="operation">
   <br/>
   <font class="s9">
     <!-- bookmark -->
     <a name="xplan"/>

     <b>
     SQL Plan Monitoring Details (Plan Hash Value=<xsl:value-of select="/report/sql_monitor_report/target/@sql_plan_hash"/>)
     </b>
   </font>

   <!-- display plan lines -->
   <table border="1" cellspacing="0" cellpadding="0"
      ora_headercharhorz="=" ora_headercharvert="|" ora_footercharhorz="="
      ora_borderstyle="headeronly">
     <tr class="thead tcell">
       <!-- optimizer stats -->
       <th>Id</th>

       <th align="center">Operation</th>

       <th align="center">Name</th>
       <th align="center">Rows<br/>(Estim)</th>
       <th align="center">Cost</th>

       <!-- plan monitor stats -->
       <th align="center">Time<br/>Active(s)</th>
       <th align="center">Start<br/>Active</th>
       <th align="center">Execs</th>
       <th align="center">Rows<br/>(Actual)</th>

       <!-- Read reqs/bytes (real) -->
       <xsl:if test="$need_read">
         <th align="center">Read<br/>Reqs</th>
         <th align="center">Read<br/>Bytes</th>
       </xsl:if>

       <!-- Write reqs/bytes (real) -->
       <xsl:if test="$need_write">
         <th align="center">Write<br/>Reqs</th>
         <th align="center">Write<br/>Bytes</th>
       </xsl:if>

       <!-- Cell Offload Efficiencying (real) -->
       <xsl:if test="$need_cell_filter">
        <th align="center">Cell<br/>Offload</th>
       </xsl:if>

       <!-- Memory/Max Memory (real) -->
       <xsl:if test="$need_memory">
        <th align="center">
         <xsl:choose>
           <xsl:when 
             test="/report/sql_monitor_report/target/status = 'EXECUTING'">
             Mem
           </xsl:when>
           <xsl:otherwise>
             Mem<br/>(Max)
           </xsl:otherwise>
         </xsl:choose>
        </th>
       </xsl:if>

       <!-- Temp/Max Temp (real) -->
       <xsl:if test="$need_temp">
        <th align="center">
         <xsl:choose>
           <xsl:when test="/report/sql_monitor_report/target/
                           status = 'EXECUTING'">
             Temp
           </xsl:when>
           <xsl:otherwise>
             Temp<br/>(Max)
           </xsl:otherwise>
         </xsl:choose>
        </th>
       </xsl:if>

       <!-- Activity -->
       <th align="center">Activity<br/>(%)</th>

       <!-- Activity Detail -->
       <th align="center">Activity Detail<br/>(# samples)</th>

       <!-- Progress Info -->
       <xsl:if test="$need_progress">
         <th>Progress</th>
       </xsl:if>
     </tr>

     <xsl:for-each select="operation">

       <tr align="right">

       <xsl:attribute name="class">
         row<xsl:value-of select="position() mod 2"/>
       </xsl:attribute>

       <xsl:apply-templates select="."/>
       </tr>
     </xsl:for-each>

   </table>
  </xsl:if>
</xsl:template>


<!-- +++++++++++++++++++++++ plan operation template ++++++++++++++++++++++ -->
<xsl:template match="operation">

    <td class="tcell" align="right">
      <xsl:choose>
        <xsl:when test="$status = 'EXECUTING' and
                        stats[@type = 'plan_monitor']/
                               stat[@name='from_most_recent'] = 0">
          <xsl:value-of select="concat('-> ', ./@id)"/>
        </xsl:when>
        <xsl:otherwise>
          <xsl:value-of select="./@id"/>
        </xsl:otherwise>
      </xsl:choose>
    </td>

    <td class="tcell" align="left" ora_space_char=".">
       <!-- indent operation using depth if needed -->
       <xsl:call-template name="left-pad">
         <xsl:with-param name="n" select="./@depth"/>
       </xsl:call-template>

       <!-- display operation name and options -->
       <xsl:value-of select="concat(@name, ' ', @options)"/>
    </td>

    <td class="tcell" align="left">
      <xsl:value-of select="object/name"/>
    </td>

    <!-- display optimizer stats -->
    <!-- cardinality (estimated) -->
    <td class="tcell" align="right">
      <xsl:call-template name="normalize">
        <xsl:with-param name="n" select="optimizer/cardinality"/>
      </xsl:call-template>
    </td>

    <!-- cost -->
    <td class="tcell" align="right">
      <xsl:call-template name="normalize">
        <xsl:with-param name="n" select="optimizer/cost"/>
      </xsl:call-template>
    </td>

    <!-- activive period -->
    <xsl:call-template name="plan-timeline"/>

    <!-- Execs -->
    <td class="tcell" align="right">
      <xsl:call-template name="normalize">
        <xsl:with-param name="n" 
              select="stats[@type='plan_monitor']/stat[@name='starts']"/>
      </xsl:call-template>
    </td>

    <!-- cardinality (real) -->
    <td class="tcell" align="right">
      <xsl:if 
         test="stats[@type='plan_monitor']/stat[@name='from_most_recent']">
        <xsl:call-template name="normalize">
        <xsl:with-param name="n" 
              select="stats[@type='plan_monitor']/stat[@name='cardinality']"/>
        </xsl:call-template>
      </xsl:if>
    </td>

    <!-- Read reqs/bytes (real) -->
    <xsl:if test="$need_read">
      <td class="tcell" align="right">
      <xsl:if 
         test="stats[@type='plan_monitor']/stat[@name='read_reqs']">
         <xsl:call-template name="normalize">
             <xsl:with-param name="n" 
               select="stats[@type='plan_monitor']/stat[@name='read_reqs']"/>
         </xsl:call-template>
      </xsl:if>
      </td>
      <td class="tcell" align="right">
      <xsl:if 
         test="stats[@type='plan_monitor']/stat[@name='read_bytes']">
         <xsl:call-template name="normalize_bytes">
             <xsl:with-param name="n" 
               select="stats[@type='plan_monitor']/stat[@name='read_bytes']"/>
         </xsl:call-template>
      </xsl:if>
      </td>
    </xsl:if>

    <!-- Write reqs/bytes (real) -->
    <xsl:if test="$need_write">
      <td class="tcell" align="right">
      <xsl:if 
         test="stats[@type='plan_monitor']/stat[@name='write_reqs']">
         <xsl:call-template name="normalize">
             <xsl:with-param name="n" 
               select="stats[@type='plan_monitor']/stat[@name='write_reqs']"/>
         </xsl:call-template>
      </xsl:if>
      </td>
      <td class="tcell" align="right">
      <xsl:if 
         test="stats[@type='plan_monitor']/stat[@name='write_bytes']">
         <xsl:call-template name="normalize_bytes">
             <xsl:with-param name="n" 
               select="stats[@type='plan_monitor']/stat[@name='write_bytes']"/>
         </xsl:call-template>
      </xsl:if>
      </td>
    </xsl:if>

    <!-- Cell offload efficiency -->
    <xsl:if test="$need_cell_filter">

      <xsl:call-template name="cell-offload-efficiency">
        <xsl:with-param 
               name="offload_efficiency" 
               select="stats[@type='plan_monitor']/
                       stat[@name = 'cell_offload_efficiency']"/>
      </xsl:call-template>
    </xsl:if>

    <!-- Memory/Max Memory (real) -->
    <xsl:if test="$need_memory">
     <td class="tcell" align="right">
       <xsl:call-template name="normalize">
         <xsl:with-param name="n">
           <xsl:choose>
             <xsl:when test="/report/sql_monitor_report/target/
                             status = 'EXECUTING'">
               <xsl:value-of select="stats[@type='plan_monitor']/
                                     stat[@name='memory']"/>
             </xsl:when>
             <xsl:otherwise>
               <xsl:value-of select="stats[@type='plan_monitor']/
                                     stat[@name='max_memory']"/>
             </xsl:otherwise>
           </xsl:choose>     
         </xsl:with-param>         
       </xsl:call-template>
     </td>
    </xsl:if>

    <!-- Temp/Max Temp (real) -->
    <xsl:if test="$need_temp">
     <td class="tcell" align="right">
       <xsl:call-template name="normalize">
         <xsl:with-param name="n">
           <xsl:choose>
             <xsl:when test="/report/sql_monitor_report/target/
                             status = 'EXECUTING'">
               <xsl:value-of select="stats[@type='plan_monitor']/
                                     stat[@name='temp']"/>
             </xsl:when>
             <xsl:otherwise>
               <xsl:value-of select="stats[@type='plan_monitor']/
                                     stat[@name='max_temp']"/>
             </xsl:otherwise>
           </xsl:choose>              
         </xsl:with-param>
       </xsl:call-template>
     </td>
    </xsl:if>

    <!-- activity for this row -->
    <td class="tcell" align="right">
        <xsl:if test="./activity_sampled">
          <xsl:apply-templates select="./activity_sampled" mode="summary">
            <xsl:with-param name="sample_count" select="@count"/>
            <xsl:with-param name="max_count" 
              select="/report/sql_monitor_report/plan_monitor/
                      @max_activity_count"/>
            <xsl:with-param name="total_count" select="$total_activity_count"/>
          </xsl:apply-templates>
        </xsl:if>
    </td>

    <!-- activity detail for this row -->
    <td class="tcell" align="left">
        <xsl:if test="./activity_sampled">
          <xsl:apply-templates select="./activity_sampled" mode="detail">
            <xsl:with-param name="sample_count" select="@count"/>
            <xsl:with-param name="max_count" 
              select="/report/sql_monitor_report/plan_monitor/
                      @max_activity_count"/>
            <xsl:with-param name="total_count" select="$total_activity_count"/>
          </xsl:apply-templates>
        </xsl:if>
    </td>

    <!-- Progress Info -->
    <xsl:if test="$need_progress">
      <td class="tcell" align="right">
        <xsl:call-template name="display-progress">
          <xsl:with-param name="time_left"
            select="stats[@type='plan_monitor']/stat[@name='time_left']"/>
          <xsl:with-param 
            name="percent_complete"
            select="stats[@type='plan_monitor']/
                    stat[@name='percent_complete']"/>
        </xsl:call-template>
      </td>
    </xsl:if>
</xsl:template>


<!-- ++++++++++++++++++++++++++ cell-offload-efficiency +++++++++++++++++ -->
<xsl:template name="cell-offload-efficiency">

  <!-- offload efficiency factor -->
  <xsl:param name="offload_efficiency"/>

  <td class="tcell" align="right">

  <xsl:if test="$offload_efficiency and $offload_efficiency != 0">

    <!-- put value -->
    <xsl:value-of 
       select="concat(format-number((1 - (1 div $offload_efficiency)) * 100, 
                      '#0.00'), '%')"/>
  </xsl:if>
  
  </td>

</xsl:template>


</xsl:stylesheet>
