Rem
Rem $Header: rdbms/admin/dbmsfspi.sql /main/3 2009/07/01 21:38:41 kkunchit Exp $
Rem
Rem dbmsfspi.sql
Rem
Rem Copyright (c) 2008, 2009, Oracle and/or its affiliates. 
Rem All rights reserved. 
Rem
Rem    NAME
Rem      dbmsfspi.sql - DBFS SFS reference provider specification.
Rem
Rem    DESCRIPTION
Rem      DBFS SFS reference provider specification.
Rem
Rem    NOTES
Rem      Specification for the "dbms_dbfs_sfs" and "dbms_dbfs_sfs_admin"
Rem      packages. The SFS provider implements POSIX filesystem semantics.
Rem
Rem
Rem    MODIFIED   (MM/DD/YY)
Rem    kkunchit    09/16/08 - Created
Rem



/* ----------------------------- dbms_dbfs_sfs ----------------------------- */
/*
 * The package "dbms_dbfs_sfs" is a store provider for the DBFS API and thus
 * conforms to the DBFS SPI defined in "dbms_dbfs_content_spi".
 *
 */

create or replace package dbms_dbfs_sfs
    authid current_user
as



    /*
     * Table/filesystem descriptors:
     *
     * A "table_t" is a record that describes a POSIX filesystem table
     * (in any schema) that is available for use a POSIX store.
     *
     * A "filesystem_t" is a record that describes a POSIX filesystem
     * registered for use by the current user.
     *
     * Clients can query this API for the list of available store
     * tables, determine which ones are suitable for their use, and
     * register one or more of these tables as stores.
     *
     */

    type table_t        is record (
        schema_name         varchar2(32),
        table_name          varchar2(32),
        ptable_name         varchar2(32),
        version#            varchar2(32),
        created             timestamp,
        formatted           timestamp,
        properties          dbms_dbfs_content_properties_t
    );
    type tables_t is table of table_t;


    type volume_t       is record (
        schema_name         varchar2(32),
        table_name          varchar2(32),
        volume_name         varchar2(128),
        created             timestamp,
        from_volume         varchar2(128),
        from_snapshot       varchar2(128)
    );
    type volumes_t is table of volume_t;


    type snapshot_t     is record (
        schema_name         varchar2(32),
        table_name          varchar2(32),
        volume_name         varchar2(128),
        snapshot_name       varchar2(128),
        created             timestamp
    );
    type snapshots_t is table of snapshot_t;


    type filesystem_t   is record (
        store_name          varchar2(32),
        schema_name         varchar2(32),
        table_name          varchar2(32),
        volume_name         varchar2(128),
        snapshot_name       varchar2(128),
        created             timestamp
    );
    type filesystems_t is table of filesystem_t;



    /*
     * List all available store tables and their properties.
     *
     */

    function    listTables
            return  tables_t
                pipelined;


    /*
     * List all volumes available for POSIX store tables.
     *
     */

    function    listVolumes
            return  volumes_t
                pipelined;


    /*
     * List all snapshots available for POSIX store tables.
     *
     */

    function    listSnapshots
            return  snapshots_t
                pipelined;


    /*
     * List all registered POSIX filesystems.
     *
     */

    function    listFilesystems
            return  filesystems_t
                pipelined;



    /*
     * Lookup store features (see dbms_dbfs_content.feature_XXX). Lookup
     * store id.
     *
     * A store ID identifies a provider-specific store, across
     * registrations and mounts, but independent of changes to the store
     * contents.
     *
     * I.e. changes to the store table(s) should be reflected in the
     * store ID, but re-initialization of the same store table(s) should
     * preserve the store ID.
     *
     * Providers should also return a "version" (either specific to a
     * provider package, or to an individual store) based on a standard
     * <a.b.c> naming convention (for <major>, <minor>, and <patch>
     * components).
     *
     */

    function    getFeatures(
        store_name          in      varchar2)
            return  integer;

    function    getStoreId(
        store_name          in      varchar2)
            return  number;

    function    getVersion(
        store_name          in      varchar2)
            return  varchar2;



    /*
     * Lookup pathnames by (store_name, std_guid) or (store_mount,
     * std_guid) tuples.
     *
     * If the underlying "std_guid" is found in the underlying store,
     * this function returns the store-qualified pathname.
     *
     * If the "std_guid" is unknown, a "null" value is returned. Clients
     * are expected to handle this as appropriate.
     *
     */

    function    getPathByStoreId(
        store_name          in      varchar2,
        guid                in      integer)
            return  varchar2;



    /*
     * DBFS SPI: space usage.
     *
     * Clients can query filesystem space usage statistics via the
     * "spaceUsage()" method. Providers are expected to support this
     * method for their stores (and to make a best effort determination
     * of space usage---esp. if the store consists of multiple
     * tables/indexes/lobs, etc.).
     *
     * "blksize" is the natural tablespace blocksize that holds the
     * store---if multiple tablespaces with different blocksizes are
     * used, any valid blocksize is acceptable.
     *
     * "tbytes" is the total size of the store in bytes, and "fbytes" is
     * the free/unused size of the store in bytes. These values are
     * computed over all segments that comprise the store.
     *
     * "nfile", "ndir", "nlink", and "nref" count the number of
     * currently available files, directories, links, and references in
     * the store.
     *
     * Since database objects are dynamically growable, it is not easy
     * to estimate the division between "free" space and "used" space.
     *
     */

    procedure   spaceUsage(
        store_name  in              varchar2,
        blksize     out             integer,
        tbytes      out             integer,
        fbytes      out             integer,
        nfile       out             integer,
        ndir        out             integer,
        nlink       out             integer,
        nref        out             integer);



    /*
     * DBFS SPI: notes on pathnames.
     *
     * All pathnames used in the SPI are store-qualified, i.e. a 2-tuple
     * of the form (store_name, pathname) (where the pathname is rooted
     * within the store namespace).
     *
     *
     * Stores/providers that support contentID-based access (see
     * "feature_content_id") also support a form of addressing that is
     * not based on pathnames. Items are identified by an explicit store
     * name, a "null" pathname, and possibly a contentID specified as a
     * parameter or via the "opt_content_id" property.
     *
     * Not all operations are supported with contentID-based access, and
     * applications should depend only on the simplest create/delete
     * functionality being available.
     *
     */



    /*
     * DBFS SPI: creation operations
     *
     * The SPI must allow the DBFS API to create directory, file, link,
     * and reference elements (subject to store feature support).
     *
     *
     * All of the creation methods require a valid pathname (see the
     * special exemption for contentID-based access below), and can
     * optionally specify properties to be associated with the pathname
     * as it is created. It is also possible for clients to fetch-back
     * item properties after the creation completes (so that
     * automatically generated properties (e.g. "std_creation_time") are
     * immediately available to clients (the exact set of properties
     * fetched back is controlled by the various "prop_xxx" bitmasks in
     * "prop_flags").
     *
     *
     * Links and references require an additional pathname to associate
     * with the primary pathname.
     *
     * File pathnames can optionally specify a BLOB value to use to
     * initially populate the underlying file content (the provided BLOB
     * may be any valid lob: temporary or permanent). On creation, the
     * underlying lob is returned to the client (if "prop_data" is
     * specified in "prop_flags").
     *
     * Non-directory pathnames require that their parent directory be
     * created first. Directory pathnames themselves can be recursively
     * created (i.e. the pathname hierarchy leading up to a directory
     * can be created in one call).
     *
     *
     * Attempts to create paths that already exist is an error; the one
     * exception is pathnames that are "soft-deleted" (see below for
     * delete operations)---in these cases, the soft-deleted item is
     * implicitly purged, and the new item creation is attempted.
     *
     *
     * Stores/providers that support contentID-based access accept an
     * explicit store name and a "null" path to create a new element.
     * The contentID generated for this element is available via the
     * "opt_content_id" property (contentID-based creation automatically
     * implies "prop_opt" in "prop_flags").
     *
     * The newly created element may also have an internally generated
     * pathname (if "feature_lazy_path" is not supported) and this path
     * is available via the "std_canonical_path" property.
     *
     * Only file elements are candidates for contentID-based access.
     *
     */

    procedure   createFile(
        store_name  in              varchar2,
        path        in              varchar2,
        properties  in out nocopy   dbms_dbfs_content_properties_t,
        content     in out nocopy   blob,
        prop_flags  in              integer,
        ctx         in              dbms_dbfs_content_context_t);

    procedure   createLink(
        store_name  in              varchar2,
        srcPath     in              varchar2,
        dstPath     in              varchar2,
        properties  in out nocopy   dbms_dbfs_content_properties_t,
        prop_flags  in              integer,
        ctx         in              dbms_dbfs_content_context_t);

    procedure   createReference(
        store_name  in              varchar2,
        srcPath     in              varchar2,
        dstPath     in              varchar2,
        properties  in out nocopy   dbms_dbfs_content_properties_t,
        prop_flags  in              integer,
        ctx         in              dbms_dbfs_content_context_t);

    procedure   createDirectory(
        store_name  in              varchar2,
        path        in              varchar2,
        properties  in out nocopy   dbms_dbfs_content_properties_t,
        prop_flags  in              integer,
        recurse     in              integer,
        ctx         in              dbms_dbfs_content_context_t);



    /*
     * DBFS SPI: deletion operations
     *
     * The SPI must allow the DBFS API to delete directory, file, link,
     * and reference elements (subject to store feature support).
     *
     *
     * By default, the deletions are "permanent" (get rid of the
     * successfully deleted items on transaction commit), but stores may
     * also support "soft-delete" features. If requested by the client,
     * soft-deleted items are retained by the store (but not typically
     * visible in normal listings or searches).
     *
     * Soft-deleted items can be "restore"d, or explicitly purged.
     *
     *
     * Directory pathnames can be recursively deleted (i.e. the pathname
     * hierarchy below a directory can be deleted in one call).
     * Non-recursive deletions can be performed only on empty
     * directories. Recursive soft-deletions apply the soft-delete to
     * all of the items being deleted.
     *
     *
     * Individual pathnames (or all soft-deleted pathnames under a
     * directory) can be restored or purged via the restore and purge
     * methods.
     *
     *
     * Providers that support filtering can use the provider "filter" to
     * identify subsets of items to delete---this makes most sense for
     * bulk operations (deleteDirectory, restoreAll, purgeAll), but all
     * of the deletion-related operations accept a "filter" argument.
     *
     *
     * Stores/providers that support contentID-based access can also
     * allow file items to be deleted by specifying their contentID.
     *
     */

    procedure   deleteFile(
        store_name  in              varchar2,
        path        in              varchar2,
        filter      in              varchar2,
        soft_delete in              integer,
        ctx         in              dbms_dbfs_content_context_t);

    procedure   deleteContent(
        store_name  in              varchar2,
        contentID   in              raw,
        filter      in              varchar2,
        soft_delete in              integer,
        ctx         in              dbms_dbfs_content_context_t);

    procedure   deleteDirectory(
        store_name  in              varchar2,
        path        in              varchar2,
        filter      in              varchar2,
        soft_delete in              integer,
        recurse     in              integer,
        ctx         in              dbms_dbfs_content_context_t);

    procedure   restorePath(
        store_name  in              varchar2,
        path        in              varchar2,
        filter      in              varchar2,
        ctx         in              dbms_dbfs_content_context_t);

    procedure   purgePath(
        store_name  in              varchar2,
        path        in              varchar2,
        filter      in              varchar2,
        ctx         in              dbms_dbfs_content_context_t);

    procedure   restoreAll(
        store_name  in              varchar2,
        path        in              varchar2,
        filter      in              varchar2,
        ctx         in              dbms_dbfs_content_context_t);

    procedure   purgeAll(
        store_name  in              varchar2,
        path        in              varchar2,
        filter      in              varchar2,
        ctx         in              dbms_dbfs_content_context_t);



    /*
     * DBFS SPI: path get/put operations.
     *
     * Existing path items can be accessed (for query or for update) and
     * modified via simple get/put methods.
     *
     * All pathnames allow their metadata (i.e. properties) to be
     * read/modified. On completion of the call, the client can request
     * (via "prop_flags") specific properties to be fetched as well.
     *
     * File pathnames allow their data (i.e. content) to be
     * read/modified. On completion of the call, the client can request
     * (via the "prop_data" bitmaks in "prop_flags") a new BLOB locator
     * that can be used to continue data access.
     *
     * Files can also be read/written without using BLOB locators, by
     * explicitly specifying logical offsets/buffer-amounts and a
     * suitably sized buffer.
     *
     *
     * Update accesses must specify the "forUpdate" flag. Access to link
     * pathnames can be implicitly and internally deferenced by stores
     * (subject to feature support) if the "deref" flag is
     * specified---however, this is dangerous since symbolic links are
     * not always resolvable.
     *
     *
     * The read methods (i.e. "getPath" where "forUpdate" is "false"
     * also accepts a valid "asof" timestamp parameter that can be used
     * by stores to implement "as of" style flashback queries. Mutating
     * versions of the "getPath" and the "putPath" methods do not
     * support as-of modes of operation.
     *
     *
     * "getPathNowait" implies a "forUpdate", and, if implemented (see
     * "feature_nowait"), allows providers to return an exception
     * (ORA-54) rather than wait for row locks.
     *
     */

    procedure   getPath(
        store_name  in              varchar2,
        path        in              varchar2,
        properties  in out nocopy   dbms_dbfs_content_properties_t,
        content     out    nocopy   blob,
        item_type   out             integer,
        prop_flags  in              integer,
        forUpdate   in              integer,
        deref       in              integer,
        ctx         in              dbms_dbfs_content_context_t);

    procedure   getPathNowait(
        store_name  in              varchar2,
        path        in              varchar2,
        properties  in out nocopy   dbms_dbfs_content_properties_t,
        content     out    nocopy   blob,
        item_type   out             integer,
        prop_flags  in              integer,
        deref       in              integer,
        ctx         in              dbms_dbfs_content_context_t);

    procedure   getPath(
        store_name  in              varchar2,
        path        in              varchar2,
        properties  in out nocopy   dbms_dbfs_content_properties_t,
        amount      in out          number,
        offset      in              number,
        buffer      out    nocopy   raw,
        prop_flags  in              integer,
        ctx         in              dbms_dbfs_content_context_t);

    procedure   getPath(
        store_name  in              varchar2,
        path        in              varchar2,
        properties  in out nocopy   dbms_dbfs_content_properties_t,
        amount      in out          number,
        offset      in              number,
        buffers     out    nocopy   dbms_dbfs_content_raw_t,
        prop_flags  in              integer,
        ctx         in              dbms_dbfs_content_context_t);

    procedure   putPath(
        store_name  in              varchar2,
        path        in              varchar2,
        properties  in out nocopy   dbms_dbfs_content_properties_t,
        content     in out nocopy   blob,
        item_type   out             integer,
        prop_flags  in              integer,
        ctx         in              dbms_dbfs_content_context_t);

    procedure   putPath(
        store_name  in              varchar2,
        path        in              varchar2,
        properties  in out nocopy   dbms_dbfs_content_properties_t,
        amount      in              number,
        offset      in              number,
        buffer      in              raw,
        prop_flags  in              integer,
        ctx         in              dbms_dbfs_content_context_t);

    procedure   putPath(
        store_name  in              varchar2,
        path        in              varchar2,
        properties  in out nocopy   dbms_dbfs_content_properties_t,
        written     out             number,
        offset      in              number,
        buffers     in              dbms_dbfs_content_raw_t,
        prop_flags  in              integer,
        ctx         in              dbms_dbfs_content_context_t);



    /*
     * DBFS SPI: rename/move operations.
     *
     * Pathnames can be renamed or moved, possibly across directory
     * hierarchies and mount-points, but within the same store.
     *
     *
     * Non-directory pathnames previously accessible via "oldPath" are
     * renamed as a single item subsequently accessible via "newPath";
     * assuming that "newPath" does not already exist.
     *
     * If "newPath" exists and is not a directory, the rename implicitly
     * deletes the existing item before renaming "oldPath". If "newPath"
     * exists and is a directory, "oldPath" is moved into the target
     * directory.
     *
     *
     * Directory pathnames previously accessible via "oldPath" are
     * renamed by moving the directory and all of its children to
     * "newPath" (if it does not already exist) or as children of
     * "newPath" (if it exists and is a directory).
     *
     *
     * Stores/providers that support contentID-based access and lazy
     * pathname binding also support the "setPath" method that
     * associates an existing "contentID" with a new "path".
     *
     */

    procedure   renamePath(
        store_name  in              varchar2,
        oldPath     in              varchar2,
        newPath     in              varchar2,
        properties  in out nocopy   dbms_dbfs_content_properties_t,
        ctx         in              dbms_dbfs_content_context_t);

    procedure   setPath(
        store_name  in              varchar2,
        contentID   in              raw,
        path        in              varchar2,
        properties  in out nocopy   dbms_dbfs_content_properties_t,
        ctx         in              dbms_dbfs_content_context_t);



    /*
     * DBFS SPI: directory navigation and search.
     *
     * The DBFS API can list or search the contents of directory
     * pathnames, optionally recursing into sub-directories, optionally
     * seeing soft-deleted items, optionally using flashback "as of" a
     * provided timestamp, and optionally filtering items in/out within
     * the store based on list/search predicates.
     *
     */

    function    list(
        store_name  in              varchar2,
        path        in              varchar2,
        filter      in              varchar2,
        recurse     in              integer,
        ctx         in              dbms_dbfs_content_context_t)
            return  dbms_dbfs_content_list_items_t
                pipelined;

    function    search(
        store_name  in              varchar2,
        path        in              varchar2,
        filter      in              varchar2,
        recurse     in              integer,
        ctx         in              dbms_dbfs_content_context_t)
            return  dbms_dbfs_content_list_items_t
                pipelined;



    /*
     * DBFS SPI: locking operations.
     *
     * Clients of the DBFS API can apply user-level locks to any valid
     * pathname (subject to store feature support), associate the lock
     * with user-data, and subsequently unlock these pathnames.
     *
     * The status of locked items is available via various optional
     * properties (see "opt_lock*" above).
     *
     *
     * It is the responsibility of the store (assuming it supports
     * user-defined lock checking) to ensure that lock/unlock operations
     * are performed in a consistent manner.
     *
     */

    procedure   lockPath(
        store_name  in              varchar2,
        path        in              varchar2,
        lock_type   in              integer,
        lock_data   in              varchar2,
        ctx         in              dbms_dbfs_content_context_t);

    procedure   unlockPath(
        store_name  in              varchar2,
        path        in              varchar2,
        ctx         in              dbms_dbfs_content_context_t);



    /*
     * DBFS SPI: access checks.
     *
     * Check if a given pathname (store_name, path, pathtype) can be
     * manipulated by "operation (see the various
     * "dbms_dbfs_content.op_xxx" opcodes) by "principal".
     *
     * This is a convenience function for the DBFS API; a store that
     * supports access control still internally performs these checks to
     * guarantee security.
     *
     */

    function    checkAccess(
        store_name  in              varchar2,
        path        in              varchar2,
        pathtype    in              integer,
        operation   in              varchar2,
        principal   in              varchar2)
            return  integer;



    /*
     * Create and register a new POSIX store.
     *
     *
     * Create a new POSIX store "store_name" in schema "schema_name"
     * (defaulting to the current schema) as table "tbl_name", with the
     * table (and internal indexes) in tablespace "tbl_tbs" (defaulting
     * to the schema's default tablespace), and its lob column in
     * tablespace "lob_tbs" (defaulting to "tbl_tbs").
     *
     * If "tbl_name" is not specified, an internally generated name is
     * used.
     *
     * If "use_bf" is true, a basicfile lob is used, otherwise a
     * securefile lob is used.
     *
     *
     * "props" is a table of (name, value, typecode) tuples that can be
     * used to configure the store properties. Currently, no such
     * properties are defined or used, but the placeholder exists for
     * future versions of the reference implementation.
     *
     *
     * If the "create_only" argument is "true", the filesystem is
     * created, but not registered with the current user---a separate
     * call to "dbms_dbfs_sfs_admin.registerFilesystem" (by the same
     * users or by other users) is needed to make the filesystem visible
     * for provider operations.
     *
     *
     * If "use_objects" is true, a single base-table with an object-type
     * column (using a nested table) is created to back the new
     * filesystem. Otherwise, a pair of (parent, child) tables is used
     * to back the filesystem. In any case, the object type nested table
     * or the child table is used only for user-defined properties.
     *
     *
     * If "with_grants" is true, DML and query access permissions are
     * granted to the "dbfs_role" as part of creating the filesystem.
     * Otherwise, explicit grants (or existing permissions) are required
     * to be able to access the filesystem.
     *
     *
     * All of the remaining arguments to the procedure deal with special
     * storage options for the file content lobs or to control the
     * partitioning strategy of the filesystem tables. Settings that do
     * not apply (to basicfiles or partitioning strategy) will be
     * silently ignored and coerced to sane values.
     *
     *
     * The procedure executes like a DDL (i.e. auto-commits before and
     * after its execution).
     *
     *
     * "createStore" is a wrapper around "createFilesystem".
     *
     */

    -- "compression" parameter
    compression_default constant varchar2(32)   := '';
    compression_low     constant varchar2(32)   := 'low';
    compression_medium  constant varchar2(32)   := 'medium';
    compression_high    constant varchar2(32)   := 'high';

    -- "encryption" parameter
    encryption_default  constant varchar2(32)   := '';
    encryption_3des168  constant varchar2(32)   := '3DES168';
    encryption_aes128   constant varchar2(32)   := 'AES128';
    encryption_aes192   constant varchar2(32)   := 'AES192';
    encryption_aes256   constant varchar2(32)   := 'AES256';

    -- "npartitions" parameter
    default_partitions  constant integer        := 16;

    -- "partition_key" parameter
    partition_by_item   constant integer        := 1;
    partition_by_path   constant integer        := 2;
    partition_by_guid   constant integer        := 3;

    procedure   createFilesystem(
        store_name      in          varchar2,
        schema_name     in          varchar2    default null,
        tbl_name        in          varchar2    default null,
        tbl_tbs         in          varchar2    default null,
        lob_tbs         in          varchar2    default null,
        use_bf          in          boolean     default false,
        properties      in          dbms_dbfs_content_properties_t
                                                default null,
        create_only     in          boolean     default false,
        use_objects     in          boolean     default false,
        with_grants     in          boolean     default false,
        do_dedup        in          boolean     default false,
        do_compress     in          boolean     default false,
        compression     in          varchar2    default compression_default,
        do_encrypt      in          boolean     default false,
        encryption      in          varchar2    default encryption_default,
        do_partition    in          boolean     default false,
        npartitions     in          number      default default_partitions,
        partition_key   in          number      default partition_by_item,
        partition_guidi in          boolean     default false,
        partition_pathi in          boolean     default false,
        partition_prop  in          boolean     default true);

    procedure   createStore(
        store_name  in              varchar2,
        tbl_name    in              varchar2    default null,
        tbs_name    in              varchar2    default null,
        use_bf      in              boolean     default false,
        stgopts     in              varchar2    default '');



    /*
     * Export a filesystem (for general cross-schema use) by granting
     * suitable permissions to the tables underlying the filesystem to
     * the "dbfs_role".
     *
     * These methods can be successfully invoked only by those users who
     * are capable of granting the necessary privileges, i.e. either by
     * the users who own the underlying filesystem tables, or
     * sufficiently privileged users who can grant access across
     * schemas.
     *
     */

    procedure   exportTable(
        schema_name in              varchar2    default null,
        tbl_name    in              varchar2,
        toUserRole  in              varchar2    default 'DBFS_ROLE');

    procedure   exportFilesystem(
        store_name  in              varchar2,
        toUserRole  in              varchar2    default 'DBFS_ROLE');



    /*
     * (Re)initialize a POSIX store.
     *
     *
     * The table associated with the POSIX store "store_name" is
     * truncated and reinitialized with a single "root" directory entry.
     *
     *
     * The procedure executes like a DDL (i.e. auto-commits before and
     * after its execution).
     *
     */

    procedure   initFS(
        store_name  in              varchar2);


    /*
     * Unregister and drop a POSIX store.
     *
     *
     * If the specified store table is registered by the current user,
     * it will be unregistered from the DBFS API and the POSIX metadata
     * tables.
     *
     * Subsequent to unregistration, an attempt will be made to store
     * table(s). This operation may fail if other users are currently
     * using this store table---in these cases, it is necessary to also
     * execute "dbms_dbfs_sfs.dropFilesystem" or
     * "dbms_dbfs_sfs_admin.unregisterFilesystem" as these users first
     * before the actual store table can be dropped.
     *
     * Finally, the user attempting a drop of the tables underlying the
     * store must actually have the privileges to complete the drop
     * operation (either as the owner of the tables, or as a
     * sufficiently privileged user for cross-schema operations).
     *
     * The procedure executes like a DDL (i.e. auto-commits before and
     * after its execution).
     *
     */

    procedure   dropFilesystem(
        schema_name in              varchar2    default null,
        tbl_name    in              varchar2);

    procedure   dropFilesystem(
        store_name  in              varchar2);



    /*
     * Snapshot operations.
     *
     * All snapshot operations must specify a valid store name
     * "store_name" and a valid snapshot name "snap_name". Additionally,
     * the methods accept a volume name "vol_name" (which may be "null"
     * or defaulted to "main"---these refer to the primary volume);
     * non-default volumes are not currently supported.
     *
     * Since "snap_name" may be used as a pathname component, it should
     * conform to the standard rules about such names (i.e. no embedded
     * "/" characters).
     *
     * All operations also accept a "do_wait" parameter (default "true")
     * which controls whether the invoking session waits for other
     * active transactions to finish, or if it exits immediately if the
     * specified operation cannot be initiated.
     *
     * All operations execute like DDL (i.e. auto-commit before and
     * after their execution).
     *
     *
     * "createSnapshot" creates a new snapshot on the specified
     * store/volume.
     *
     * "revertSnapshot" drops all snapshots/changes in the specified
     * store/volume more recent than the specified snapshot.
     *
     * "dropSnapshot" drops the specified snapshot from the specified
     * store/volume.
     *
     */

    procedure   createSnapshot(
        store_name  in              varchar2,
        snap_name   in              varchar2,
        vol_name    in              varchar2    default 'main',
        do_wait     in              boolean     default true);

    procedure   revertSnapshot(
        store_name  in              varchar2,
        snap_name   in              varchar2,
        vol_name    in              varchar2    default 'main',
        do_wait     in              boolean     default true);

    procedure   dropSnapshot(
        store_name  in              varchar2,
        snap_name   in              varchar2,
        vol_name    in              varchar2    default 'main',
        do_wait     in              boolean     default true,
        recurse     in              boolean     default false);



    /*
     * Filesystem/store operations.
     *
     * Filesystem operations are used to map a (schema, table, volume,
     * snapshot) tuple to a filesystem (aka store) name. A filesystem is
     * accessible via the DBFS API as a store/mount.
     *
     */

    procedure   registerFilesystem(
        store_name  in              varchar2,
        schema_name in              varchar2,
        tbl_name    in              varchar2,
        vol_name    in              varchar2    default 'main',
        snap_name   in              varchar2    default null);

    procedure   unregisterFilesystem(
        store_name  in              varchar2);
end;
/
show errors;

create or replace public synonym dbms_dbfs_sfs
    for sys.dbms_dbfs_sfs;

grant execute on dbms_dbfs_sfs
    to dbfs_role;



/* -------------------------- dbms_dbfs_sfs_admin -------------------------- */
/*
 * The package "dbms_dbfs_sfs_admin" defines provider-specific
 * administrative operations for the POSIX filesystems managed by
 * "dbms_dbfs_sfs".
 *
 * For the most part, the terms "POSIX filesystems" and "POSIX stores" are
 * synonymous.
 *
 * POSIX stores are identified by a (store_owner, store_name) tuple, where
 * "store_owner" is the name of the "session_user" user that creates a store
 * and "store_name" is the specified store name.
 *
 * Store owners are implicit, and a user will not be able to see or access
 * another user's stores via the store APIs.
 *
 */

create or replace package dbms_dbfs_sfs_admin
    authid definer
as



    /*
     * Bless the current session user with sufficient _direct_
     * privileges to enable them to use various DBFS API types in their
     * internal tables.
     *
     * This procedure is not meant to be invoked directly, but is a
     * helper function available for use with the
     * "dbms_dbfs_sfs.createFilesystem" method.
     *
     *
     */

    procedure   blessUser;



    /*
     * Helper function for store creation (only for use with
     * dbms_dbfs_sfs.createFilesystem).
     *
     */

    function    defaultTablespace(
        uname       in          varchar2)
            return  varchar2;



    /*
     * Helper function for looking up a store id (only for use with
     * dbms_dbfs_sfs.getStoreId).
     *
     */

    function    getStoreId(
        schema_name in      varchar2,
        tbl_name    in      varchar2)
            return  number;



    /*
     * Helper function for space usage queries (only for use with
     * dbms_dbfs_sfs.spaceUsage).
     *
     */

    procedure   spaceUsage(
        schema_name in          varchar2,
        tbl_name    in          varchar2,
        blksize     out         integer,
        tbytes      out         integer,
        fbytes      out         integer);



    /*
     * Create a POSIX store.
     *
     * Helper function for store creation (only for use with
     * dbms_dbfs_sfs.createFilesystem).
     *
     *
     * Add a newly created POSIX table to the list of known POSIX
     * tables. At this stage, no store is registered to a particular
     * owner as an accessible filesystem (use "registerFilesystem", if
     * needed).
     *
     * The (schema_name, tbl_name) and its identifier (tabid) must both
     * be database-wide unique. If the table uses object-types,
     * "ptbl_name" should be "null", else the name of a valid properties
     * table.
     *
     */

    procedure   createFilesystem(
        tabid       in              number,
        schema_name in              varchar2,
        tbl_name    in              varchar2,
        ptbl_name   in              varchar2,
        version     in              varchar2,
        properties  in              dbms_dbfs_content_properties_t
                                                default null);



    /*
     * Register a POSIX store.
     *
     *
     * Register an already created POSIX store table
     * (dbms_dbfs_sfs.createFilesystem) in schema "schema_name", table
     * "tbl_name", as a new filesystem named "store_name".
     *
     * The new filesystem/store can optionally be volume/snapshot
     * qualified (by default the "main" volume and current snapshot are
     * used).
     *
     * The same table can be registered as different filesystems by the
     * same/different user as long as the "store_name" and (schema_name,
     * tbl_name, volume_name, snapshot_name) tuple are unique per-user.
     *
     */

    procedure   registerFilesystem(
        store_name      in          varchar2,
        schema_name     in          varchar2,
        tbl_name        in          varchar2,
        volume_name     in          varchar2    default 'main',
        snapshot_name   in          varchar2    default null);



    /*
     * Unregister a POSIX store.
     *
     *
     * The store is removed from the metadata tables, unregistered from
     * the DBFS API, but the underlying filesystem itself is otherwise
     * untouched.
     *
     */

    procedure   unregisterFilesystem(
        store       in              varchar2);



    /*
     * Initialize a POSIX store.
     *
     * Helper function for store initialization (only for use with
     * dbms_dbfs_sfs.initFS).
     *
     *
     * Remove all volumes and snapshots associated with a POSIX
     * filesystem table, and update its "formatted" timestamp.
     *
     */

    procedure   initFilesystem(
        schema_name in              varchar2,
        tbl_name    in              varchar2);



    /*
     * Drop a POSIX store.
     *
     * Helper function for store creation (only for use with
     * dbms_dbfs_sfs.dropFilesystem).
     *
     *
     * Remove a POSIX filesystem table from the list of known POSIX
     * filesystem tables.
     *
     * The table underlying the store must not be in use (i.e. all
     * filesystems referring to this table must have been unregistered
     * prior to this call).
     *
     */

    procedure   dropFilesystem(
        schema_name in              varchar2,
        tbl_name    in              varchar2);



    /*
     * Snapshot operations.
     *
     * Helper functions for snapshot operations meant to be invoked only
     * from "dbms_dbfs_sfs".
     *
     */

    procedure   createSnapshot(
        store_name  in              varchar2,
        snap_name   in              varchar2,
        vol_name    in              varchar2    default 'main');

    procedure   revertSnapshot(
        store_name  in              varchar2,
        snap_name   in              varchar2,
        vol_name    in              varchar2    default 'main');

    procedure   dropSnapshot(
        store_name  in              varchar2,
        snap_name   in              varchar2,
        vol_name    in              varchar2    default 'main');



    /*
     * Drop _all_ POSIX filesystem tables.
     *
     * Action to be invoked only during cleanup/downgrade.
     *
     */

    procedure   drop_all_tables;
end;
/
show errors;

create or replace public synonym dbms_dbfs_sfs_admin
    for sys.dbms_dbfs_sfs_admin;

grant execute on dbms_dbfs_sfs_admin
    to dbfs_role;



