#!/bin/sh
#
# $Header: ldap/schema/oid/create_subscriber.sh /main/8 2009/05/25 14:20:34 fwu Exp $
#
# create_subscriber.sh
#
# Copyright (c) 2002, 2009, Oracle and/or its affiliates. All rights reserved. 
#
#    NAME
#      create_subscriber.sh : creates the default subscriber
#
#    DESCRIPTION
#      This script creates a new subscriber in the DIT. The new
#       subscriber can be optionally marked as the default subscriber.
#      This script assumes that the parent DIT of the subscriber is 
#       already created.
#
#    NOTES
#
#    MODIFIED   (MM/DD/YY)
#    fwu           05/22/09 - Revert ldifmigrator.bat changes
#    ntregres      05/14/09 - Fix 8497498
#    fwu           12/28/06 - Use ldifmigrator from SRCHOME until bug 5734836
#                             fixed
#    jwei          12/12/03 - add log_file parameter 
#    sshrivas      07/11/02 - New Substitution variable
#    jwei          04/23/02 - fix for whitespace in subscriber name
#    stlee         04/08/02 - stlee_bug-2310185
#    akolli        04/01/02 - Creation


usage() {

cat << USAGE_MESSAGE
usage: ${0} -host <host> -port <port>
            -default <yes/no>
            -SubscriberName <SubscriberName>
            -SubscriberNamingAttribute <SubscriberNamingAttribute>
            -SubscriberObjectclass <SubscriberObjectclass>
            -SubscriberParentDN <SubscriberParentDN>
            -CurrentUserDN <CurrentUserDN>
            -CurrentUserPassword <CurrentUserPassword>
             

mandatory arguments:
  SubscriberName: name of the subscriber (e.g. acme)

Default values are:
  host: $HOST
  port: 389
  default: no
  SubscriberNamingAttribute: o
  SubscriberObjectclass: organization
  SubscriberParentDN: dc=com
  CurrentUserDN: cn=orcladmin
  CurrentUserPassword: welcome
USAGE_MESSAGE
}

host=$HOST
port=389
default="no"
SubscriberNamingAttribute="o"
SubscriberObjectclass="organization"
SubscriberParentDN="dc=com"
CurrentUserDN="cn=orcladmin"
CurrentUserPassword="welcome"

# parce command line input
while [ $# -gt 0 ] ; do
  case $1 in

    -host) host=$2;
      if [ -z "${host}" ] ; then
         host=$HOST ;
      else
         shift;
      fi;;

    -port) port=$2;
      if [ -z "${port}" ] ; then
         port="389" ;
      else
         shift;
      fi;;

    -default) default=$2;
      if [ -z "${default}" ] ; then
         default="no" ;
      else
         shift;
      fi;;

    -SubscriberName) SubscriberName=$2;
      if [ -z "${SubscriberName}" ] ; then
         echo "Error: SubscriberName not specified." ; 
         usage ; 
         exit 3 ;
      else
         shift;
      fi;;

    -SubscriberNamingAttribute) SubscriberNamingAttribute=$2;
      if [ -z "${SubscriberNamingAttribute}" ] ; then
         SubscriberNamingAttribute="o" ;
      else
         shift;
      fi;;

    -SubscriberObjectclass) SubscriberObjectclass=$2;
      if [ -z "${SubscriberObjectclass}" ] ; then
         SubscriberObjectclass="organization" ;
      else
         shift;
      fi;;

    -SubscriberParentDN) SubscriberParentDN=$2;
      if [ -z "${SubscriberParentDN}" ] ; then
         SubscriberParentDN="dc=com" ;
      else
         shift;
      fi;;

    -CurrentUserDN) CurrentUserDN=$2;
      if [ -z "${CurrentUserDN}" ] ; then
         CurrentUserDN="cn=orcladmin" ;
      else
         shift;
      fi;;

    -CurrentUserPassword) CurrentUserPassword=$2;
      if [ -z "${CurrentUserPassword}" ] ; then
         CurrentUserPassword="welcome" ;
      else
         shift;
      fi;;

    -help) usage ; exit 1 ;;

    *) echo "$1 Not Found " ; usage ; exit 2 ;;

  esac
  shift
done

# make sure subscriber name is specified
if [ -z "${SubscriberName}" ] ; then
         echo "Error: SubscriberName not specified." ; 
         usage ; 
         exit 3 ;
fi;

# setup other variables
SubscriberDN="$SubscriberNamingAttribute=$SubscriberName,$SubscriberParentDN"
RootOracleContextDN="cn=oraclecontext"
OracleContextDN="cn=oraclecontext,$SubscriberDN"
OracleContextParentDN="$SubscriberDN"
groupSearchBase="cn=groups,$SubscriberDN"

echo "Creating subscriber: " $SubscriberName
flat_sbs_name="$SubscriberName.flt"
ldif_file_name="$SubscriberName.ldif"

# remove the flat sbs file just in case
rm -f "$flat_sbs_name"
perl flattenlst.pl -i oidSubscriberCreate.lst -o "$flat_sbs_name"

# if default subscriber then also setup root O.C.
if [ "$default" = "yes" ]
then
  cat oidDefaultSubscriberConfig.sbs >> "$flat_sbs_name";
fi;

echo "Performing variable substitution for subscriber..."
# remove the ldif file just in case
rm -f "$ldif_file_name"
# run the ldifmigrator to get the substituted data file
# Use ldifmigrator from SRCHOME until bug 5734836 is fixed
$SRCHOME/bin/ldifmigrator input_file="$flat_sbs_name" \
   output_file="$ldif_file_name" \
   log_file="ldifmig$SubscriberName.log" \
   s_SubscriberObjectclass="$SubscriberObjectclass" \
   s_SubscriberNamingAttribute="$SubscriberNamingAttribute" \
   s_SubscriberName="$SubscriberName" \
   s_SubscriberParentDN="$SubscriberParentDN" \
   s_SubscriberDN="$SubscriberDN" \
   s_RootOracleContextDN="$RootOracleContextDN" \
   s_OracleContextDN="$OracleContextDN" \
   s_OracleContextParentDN="$OracleContextParentDN" \
   s_CurrentUserDN="$CurrentUserDN" \
   s_GroupSearchBase="$groupSearchBase"

ldifmigret=$?
if [ $ldifmigret -ne 0 ] ; then
  echo "LDIF substitution failed with error code $ldifmigret.";
  exit  $ldifmigret;
fi

# load the subscriber into OID
echo "loading subscriber data into OID..."
log_file_name="$SubscriberName.log"
err_file_name="$SubscriberName.err"
rm -f "$log_file_name"
rm -f "$err_file_name"
$ORACLE_HOME/bin/ldapmodify -h $host -p $port -D $CurrentUserDN -w $CurrentUserPassword -o "$err_file_name" -v -c -f  "$ldif_file_name" > "$log_file_name" 2>&1

ldapmodret=$?
if [ $ldapmodret -ne 0 ] ; then
  echo "ldapmodify returned error code $ldapmodret.";
  exit  $ldapmodret;
fi

# return success
exit 0
 
