#!/bin/sh
#
# $Header: ldap/schema/oid/designate_default_subscriber.sh /main/7 2009/05/25 14:20:34 fwu Exp $
#
# designate_default_subscriber.sh
#
# Copyright (c) 2002, 2009, Oracle and/or its affiliates. All rights reserved. 
#
#    NAME
#      designate_default_subscriber.sh : designates an existing entry in
#                        the DIT as the default subscriber.
#
#    DESCRIPTION
#      This script can be used to designate a default subscriber for a custom
#       DIT.
#
#    NOTES
#
#    MODIFIED   (MM/DD/YY)
#    fwu           05/22/09 - Revert ldifmigrator.bat changes
#    ntregres      05/14/09 - Fix 8497498
#    fwu           12/28/06 - Use ldifmigrator from SRCHOME until bug 5734836
#                             fixed
#    akolli        07/30/02 - fix bug 2458519
#    sshrivas      07/11/02 - New Substitution variable
#    jwei          04/24/02 - fix whitespace problem
#    stlee         04/08/02 - stlee_bug-2310185
#    akolli        04/01/02 - Creation


usage() {

cat << USAGE_MESSAGE
usage: ${0} -host <host> -port <port>
            -SubscriberDN <SubscriberDN>
            -SubscriberNamingAttribute <SubscriberNamingAttribute>
            -CurrentUserDN <CurrentUserDN>
            -CurrentUserPassword <CurrentUserPassword>
             

mandatory arguments:
  SubscriberDN: the distinguished name of the entry that should be made 
                   the default subscriber.

Default values are:
  host: $HOST
  port: 389
  CurrentUserDN: cn=orcladmin
  CurrentUserPassword: welcome
USAGE_MESSAGE
}

host=$HOST
port=389
CurrentUserDN="cn=orcladmin"
CurrentUserPassword="welcome"

# parce command line input
while [ $# -gt 0 ] ; do
  case $1 in

    -host) host=$2;
      if [ -z "${host}" ] ; then
         host=$HOST ;
      else
         shift;
      fi;;

    -port) port=$2;
      if [ -z "${port}" ] ; then
         port="389" ;
      else
         shift;
      fi;;

    -SubscriberDN) SubscriberDN=$2;
      if [ -z "${SubscriberDN}" ] ; then
         echo "Error: SubscriberDN not specified." ; 
         usage ; 
         exit 3 ;
      else
         shift;
      fi;;

    -SubscriberNamingAttribute) SubscriberNamingAttribute=$2;
      if [ -z "${SubscriberNamingAttribute}" ] ; then
         echo "Error: SubscriberNamingAttribute not specified.";
         usage ;
         exit 4 ;
      else
         shift;
      fi;;

    -CurrentUserDN) CurrentUserDN=$2;
      if [ -z "${CurrentUserDN}" ] ; then
         CurrentUserDN="cn=orcladmin" ;
      else
         shift;
      fi;;

    -CurrentUserPassword) CurrentUserPassword=$2;
      if [ -z "${CurrentUserPassword}" ] ; then
         CurrentUserPassword="welcome" ;
      else
         shift;
      fi;;

    -help) usage ; exit 1 ;;

    *) echo "$1 Not Found " ; usage ; exit 2 ;;

  esac
  shift
done

# make sure subscriber name is specified
if [ -z "${SubscriberDN}" ] ; then
         echo "Error: SubscriberDN not specified." ; 
         usage ; 
         exit 3 ;
fi;

# setup other variables
rdn_attr=`echo $SubscriberDN | cut -f 1 -d=`;
if [ -z "${SubscriberNamingAttribute}" ] ; then
  SubscriberNamingAttribute=$rdn_attr
else 
  if [ $rdn_attr != $SubscriberNamingAttribute ] ; then
     echo "Warning: rdn value is $rdn_attr is different from $SubscriberNamingAttribute"
  fi;
fi;

SubscriberParentDN=`echo $SubscriberDN  | cut -f 2- -d, -s`
if [ -z "${SubscriberParentDN}" ] ; then
  echo "Warning: Unable to determine parent of subscriber: $SubscriberDN";
  SubscriberParentDN=$SubscriberDN
fi;

SubscriberName=`echo  $SubscriberDN   | cut -f 1 -d, | cut -f 2 -d= `
RootOracleContextDN="cn=oraclecontext"
OracleContextDN="cn=oraclecontext,$SubscriberDN"
OracleContextParentDN="$SubscriberDN"
groupSearchBase="cn=groups,$SubscriberDN"

echo "Designating entry $SubscriberDN as default subscriber. ($SubscriberName)" 
flat_sbs_name="$SubscriberName.flt"
ldif_file_name="$SubscriberName.ldif"

# remove the flat sbs file just in case
rm -f "$flat_sbs_name"
perl flattenlst.pl -i oidDefaultSubscriberDesignate.lst -o "$flat_sbs_name"

echo "Performing variable substitution for subscriber..."
# remove the ldif file just in case
rm -f "$ldif_file_name"
# run the ldifmigrator to get the substituted data file
$SRCHOME/bin/ldifmigrator input_file="$flat_sbs_name" \
                             output_file="$ldif_file_name" \
   s_SubscriberNamingAttribute="$SubscriberNamingAttribute" \
   s_SubscriberParentDN="$SubscriberParentDN" \
   s_SubscriberDN="$SubscriberDN" \
   s_RootOracleContextDN="$RootOracleContextDN" \
   s_OracleContextDN="$OracleContextDN" \
   s_OracleContextParentDN="$OracleContextParentDN" \
   s_CurrentUserDN="$CurrentUserDN" \
   s_GroupSearchBase="$groupSearchBase"

ldifmigret=$?
if [ $ldifmigret -ne 0 ] ; then
  echo "LDIF substitution failed with error code $ldifmigret.";
  exit  $ldifmigret;
fi

# load the subscriber into OID
echo "loading subscriber data into OID..."
log_file_name="$SubscriberName.log"
err_file_name="$SubscriberName.err"
rm -f "$log_file_name"
rm -f "$err_file_name"
$ORACLE_HOME/bin/ldapmodify -h $host -p $port -D $CurrentUserDN -w $CurrentUserPassword -o "$err_file_name" -v -c -f  "$ldif_file_name" > "$log_file_name" 2>&1

ldapmodret=$?
if [ $ldapmodret -ne 0 ] ; then
  echo "ldapmodify returned error code $ldapmodret.";
  exit  $ldapmodret;
fi

# return success
exit 0
