#!/usr/local/bin/perl
# 
# $Header: alertlog_util.pl 27-oct-2003.13:19:53 jsoule Exp $
#
# alertlog_util.pl
# 
# Copyright (c) 2002, 2003, Oracle Corporation.  All rights reserved.  
#
#    NAME
#      alertlog_util.pl - Alert Log monitoring utilities
#
#    DESCRIPTION
#      utilities for reading the alert log
#
#    NOTES
#      <other useful comments, qualifications, etc.>
#
#    MODIFIED   (MM/DD/YY)
#    jsoule      10/27/03 - reduce severity of alert log file deduction to 
#    jsoule      08/29/03 - allow either alert log file format 
#    jsoule      11/06/02 - jsoule_bug-2649529_main
#    jsoule      11/06/02 - Creation
# 

use strict;

require "emd_common.pl";

#
# subroutine get_absolute_dump_dest
#  Construct the (absolute) directory for a dump destination parameter.
#
# $_[0] - background_dump_dest, user_dump_dest, or core_dump_dest parameter
#           value
# $_[1] - $ORACLE_HOME value
#

sub get_absolute_dump_dest
{
    my ($dump_dest, $ohome) = @_;

    #
    # Replace any '?' in dump_dest with $ORACLE_HOME
    #
    $dump_dest =~ s/^\?/$ohome/g;

    if (EMAGENT_isPerlDebugEnabled())
    {
        EMAGENT_PERL_DEBUG("Using dump destination: $dump_dest.");
    }

    return $dump_dest;
}

#
# subroutine get_alert_log_directory
#  Construct the directory containing the alert log.
#
# $_[0] - background_dump_dest, user_dump_dest, or core_dump_dest parameter
#           value
# $_[1] - $ORACLE_HOME value
#

sub get_alert_log_directory
{
    return get_absolute_dump_dest(@_);
}

#
# subroutine get_alert_log_filename
#  Build the filename for the alert log.
#  Return the empty string if the alert log file cannot be found.
#
# $_[0] - $ORACLE_SID value
# #_[1] - absolute path directory containing the alert log
#

sub get_alert_log_filename
{
    my ($osid, $dump_dest_dir) = @_;
    my $alert_log_filename_candidate = "alert_".$osid.".log";

    # UNIX style
    if (EMAGENT_isPerlDebugEnabled())
    {
        EMAGENT_PERL_DEBUG("Seeking out alert log file $alert_log_filename_candidate.");
    }
    if (-f compose_absolute_alert_log_filename($dump_dest_dir,
                                               $alert_log_filename_candidate))
    {
        if (EMAGENT_isPerlDebugEnabled())
        {
            EMAGENT_PERL_DEBUG("Found file $alert_log_filename_candidate.");
        }
        return $alert_log_filename_candidate;
    }
    else
    {
        # Windows-style
        $alert_log_filename_candidate = $osid."alrt.log";
        if (EMAGENT_isPerlDebugEnabled())
        {
            EMAGENT_PERL_DEBUG("Seeking out alert log file $alert_log_filename_candidate.");
        }
        if (-f compose_absolute_alert_log_filename($dump_dest_dir,
                                                   $alert_log_filename_candidate))
        {
            if (EMAGENT_isPerlDebugEnabled())
            {
                EMAGENT_PERL_DEBUG("Found file $alert_log_filename_candidate.");
            }
            return $alert_log_filename_candidate;
        }
        else
        {
            $alert_log_filename_candidate = "";
            EMAGENT_PERL_INFO("Unable to locate alert log file in directory: $dump_dest_dir.");
        }
    }

    return $alert_log_filename_candidate;
}

#
# subroutine compose_absolute_alert_log_filename
#  Compose an absolute alert log filename from its components.
#  Return the empty string if there is a problem.
#
# $_[0] - alert log directory
# $_[1] - alert log filename
#

sub compose_absolute_alert_log_filename
{
    my ($alert_log_directory, $alert_log_filename) = @_;

    if ($alert_log_filename)
    {
        return $alert_log_directory.($^O =~ m/MSWin32/ ? "\\" : "\/").$alert_log_filename;
    }
    else
    {
        return "";
    }
}

#
# subroutine get_absolute_alert_log_filename
#  Get the absolute alert log filename
#
# $_[0] - background_dump_dest parameter value
# $_[1] - $ORACLE_HOME value
# $_[2] - $ORACLE_SID value
#

sub get_absolute_alert_log_filename
{
    my ($bg_dump_dest, $ohome, $osid) = @_;
    my $alert_log_directory = get_alert_log_directory($bg_dump_dest, $ohome);

    return compose_absolute_alert_log_filename(
             $alert_log_directory,
             get_alert_log_filename($osid, $alert_log_directory));
}
