--
-- $Header: onepdat.pls 03-may-2000.10:54:20 dthompso Exp $
--
-- onepdat.pls
--
-- Copyright (c) Oracle Corporation 1900, 2000. All Rights Reserved.
--
--   NAME
--     onepdat.pls - cwm$olap$dim$attribute
--
--   DESCRIPTION
--     Definition of dimension attributes.
--
--   RETURNS
--
--   NOTES
--     
--
--   MODIFIED   (MM/DD/YY)
--   dthompso    05/03/00 - Level attributes must be qualified by level name
--   dthompso    04/27/00 - Initial Version
--   dthompso    01/00/00 - Creation
--


create or replace package cwm$olap$dim$attribute as

  -- Create a new attribute of a dimension (or all levels).
  --
  -- param owner                     owner of the dimension
  -- param dimension_name            name of the dimension
  -- param attribute_name            name for the attribute
  -- param display_name              display name for the attribute
  -- param description               descriptive text for the attribute
  -- 
  -- raise no_access_privileges      if no privileges to edit the dimension
  -- raise dimension_not_found       if dimension doesn't exist
  -- raise invalid_name              if attribute name is invalid
  -- raise attribute_already_exists  if an attribute with name already exists
  procedure create_dimension_attribute(owner varchar2
                                     , dimension_name  varchar2
                                     , attribute_name  varchar2
                                     , display_name    varchar2
                                     , description     varchar2);

  -- Delete a dimension attribute.
  --
  -- param owner                 owner of the dimension
  -- param dimension_name        name of the dimension
  -- param attribute_name        name of the attribute
  --
  -- raise no_access_privileges  if no privileges to edit the dimension
  -- raise attribute_not_found   if attribute doesn't exist
  procedure drop_dimension_attribute(owner varchar2
                                   , dimension_name varchar2
                                   , attribute_name varchar2);
                                   

  -- Set new descriptive text for a dimension attribute.
  --
  -- param owner                 owner of the dimension
  -- param dimension_name        name of the dimension
  -- param attribute_name        name of the attribute
  -- param description           new descriptive text for the attribute
  --
  -- raise no_access_privileges  if no privileges to edit the dimension
  -- raise attribute_not_found   if attribute doesn't exist
  procedure set_description(owner varchar2
                          , dimension_name varchar2
                          , attribute_name varchar2
                          , description varchar2);

  -- Set a new display name for a dimension attribute.
  --
  -- param owner                 owner of the dimension
  -- param dimension_name        name of the dimension
  -- param attribute_name        name of the attribute
  -- param display_name          new display name for the attribute
  --
  -- raise no_access_privileges  if no privileges to edit the dimension
  -- raise attribute_not_found   if attribute doesn't exist 
  procedure set_display_name(owner varchar2
                           , dimension_name varchar2
                           , attribute_name varchar2
                           , display_name  varchar2);

  -- Add a level attribute to the dimension attribute.
  --
  -- param owner                     owner of the dimension
  -- param dimension_name            name of the dimension
  -- param dim_attribute_name        name of the dimension attribute
  -- param level_name                name of the level containing the level attribute
  -- param lvl_attribute_name        name of the level attribute
  --
  -- raise no_access_privileges      if no privileges to edit the dimension
  -- raise dimension_not_found       if dimension doesn't exist
  -- raise attribute_not_found       if either attribute doesn't exist 
  -- raise attribute_already_exists  if level attribute already added
  procedure add_level_attribute(owner varchar2
                              , dimension_name varchar2
                              , dim_attribute_name varchar2
                              , level_name varchar2
                              , lvl_attribute_name varchar2);

  -- Remove a level attribute from the dimension attribute.
  --
  -- param owner                     owner of the dimension
  -- param dimension_name            name of the dimension
  -- param dim_attribute_name        name of the dimension attribute
  -- param level_name                name of the level containing the level attribute
  -- param lvl_attribute_name        name of the level attribute
  --
  -- raise no_access_privileges      if no privileges to edit the dimension
  -- raise dimension_not_found       if dimension doesn't exist
  -- raise attribute_not_found       if attribute doesn't exist 
  -- raise attribute_use_not_found   if the use does not exist
  procedure remove_level_attribute(owner varchar2
                                 , dimension_name varchar2
                                 , dim_attribute_name varchar2
                                 , level_name varchar2
                                 , lvl_attribute_name varchar2);
                                        

  -- Lock the dimension attribute
  --
  -- param owner                 owner of the dimension
  -- param dimension_name        name of the dimension
  -- param attribute_name        name of the attribute
  -- param wait_for_lock         wait for lock if acquired by other user
  --
  -- raise no_access_privileges  if no privileges to edit the dimension
  -- raise attribute_not_found   if attribute doesn't exist 
  -- raise failed_to_gain_lock   if lock could not be acquired
  procedure lock_dimension_attribute(owner varchar2
                                   , dimension_name varchar2
                                   , attribute_name varchar2
                                   , wait_for_lock boolean := false);
                                   
                                   
  -- Internal method to return the id of the dimension attribute
  --
  -- param owner                 owner of the dimension
  -- param dimension_name        name of the dimension
  -- param attribute_name        name of the attribute
  --
  -- raise attribute_not_found   if attribute doesn't exist 
  function get_dimensionattribute_id(owner varchar2
                                   , dimension_name varchar2
                                   , attribute_name varchar2) return number;

end;
/
