--
-- $Header: onepmsr.pls 30-may-2000.16:02:00 dthompso Exp $
--
-- onepmsr.pls
--
-- Copyright (c) Oracle Corporation 1900, 2000. All Rights Reserved.
--
--   NAME
--     onepmsr.pls - cwm$olap$measure
--
--   DESCRIPTION
--     Define logical measure metadata including mapping to fact table and
--     specifying default aggregation methods.
--
--   RETURNS
--
--   NOTES
--     
--
--   MODIFIED   (MM/DD/YY)
--   dthompso    05/30/00 - Correct Exceptions raised
--   dthompso    05/24/00 - set_column_map new parameter override_mappings.
--   dthompso    04/27/00 - Initial Version
--   dthompso    01/00/00 - Creation
--


create or replace package cwm$olap$measure as
 
  -- Create a measure
  --
  -- param cube_owner            owner of the cube
  -- param cube_name             name of the cube
  -- param measure_name          name of the measure
  -- param display_name          display name for the measure
  -- param description           description text for the measure
  --
  -- raise measure_already_exists if measure already exists
  -- raise invalid_name           if name is empty or null
  -- raise no_access_privileges  insufficient access privilege
  procedure create_measure(cube_owner varchar2
                      , cube_name varchar2
                      , measure_name varchar2
                      , display_name varchar2
                      , description varchar2);

  -- Set a measure column mapping, the cube must be mapped to a fact table using map_cube
  --
  -- param cube_owner            owner of the cube
  -- param cube_name             name of the cube
  -- param measure_name          name of the measure
  -- param fact_table_owner      fact table owner
  -- param fact_table_name       fact table name containing the column
  -- param column_name           column to map
  --
  -- raise cube_not_found        if cube is not defined
  -- raise measure_not_found     if measure is not defined
  -- raise table_not_defined     if table is not defined
  -- raise fact_table_not_defined if cube not mapped to fact table
  -- raise no_access_privileges  insufficient access privilege to access table in schema
  -- raise column_not_found      if column does not exist in fact table
  -- raise cannot_override_mappings  if cube is mapped to different fact table
  procedure set_column_map(cube_owner varchar2
                             , cube_name varchar2
                             , measure_name varchar2
                             , fact_table_owner varchar2
                             , fact_table_name varchar2
			     , column_name VARCHAR2);

  -- Set display name for a measure
  --
  -- param cube_owner            owner of the cube
  -- param cube_name             name of the cube
  -- param measure_name          name of the measure
  -- param display_name          display name for the measure
  --
  -- raise cube_not_found        if cube is not defined
  -- raise measure_not_found     if measure is not defined
  procedure set_display_name(owner        varchar2
                            ,cube_name    varchar2
                            ,measure_name varchar2
                            ,display_name varchar2);


  -- Set display name for a measure
  --
  -- param cube_owner            owner of the cube
  -- param cube_name             name of the cube
  -- param measure_name          name of the measure
  -- param description           description for the measure
  --
  -- raise cube_not_found        if cube is not defined
  -- raise measure_not_found     if measure is not defined
  procedure set_description(owner        varchar2
                           ,cube_name    varchar2
                           ,measure_name varchar2
                           ,description  varchar2);


  -- Remove a measure column mapping, the cube must be mapped to a fact table using map_cube
  --
  -- param cube_owner            owner of the cube
  -- param cube_name             name of the cube
  -- param measure_name          name of the measure
  -- param fact_table_owner      fact table owner
  -- param fact_table_name       fact table name containing the column
  -- param column_name           column to map
  --
  -- raise cube_not_found        if cube is not defined
  -- raise measure_not_found     if measure is not defined
  -- raise table_not_defined     if table is not defined
  -- raise no_access_privileges  insufficient access privilege to access table in schema
  -- raise column_not_found      if column does not exist in fact table
  -- raise column_map_not_found  if column is not mapped to measure
  procedure remove_column_map(owner           varchar2
                             ,cube_name       varchar2
                             ,measure_name    varchar2
                             ,table_owner     varchar2
                             ,fact_table_name varchar2
                             ,column_name     varchar2);

  -- Drop a measure
  --
  -- param cube_owner            owner of the cube
  -- param cube_name             name of the cube
  -- param measure_name          name of the measure
  --
  -- raise cube_not_found        if cube is not defined
  -- raise measure_not_found     if measure is not defined
  -- raise no_access_privileges  insufficient access privilege
  procedure drop_measure(owner          varchar2
                        ,cube_name      varchar2
                        ,measure_name   varchar2);


  -- Set default aggregation method for measure
  --
  -- param cube_owner            owner of the cube
  -- param cube_name             name of the cube
  -- param measure_name          name of the measure
  -- param function_use_id       identifier for use of a function
  -- param dimension_owner       owner of the dimension
  -- param dimension_name        name for the dimension
  -- param dimension_alias       alias for a dimension
  --
  -- raise cube_not_found             if cube is not defined
  -- raise measure_not_found          if measure is not defined
  -- raise no_access_privileges       insufficient access privilege
  -- raise dimension_not_found        if dimension is not defined
  -- raise invalid_aggregation_method invalid function use identifier
  procedure set_default_aggregation_method(cube_owner      varchar2
                                          ,cube_name       varchar2
                      ,measure_name    varchar2
                      ,function_use_id number
                      ,dimension_owner varchar2
                      ,dimension_name  varchar2
                      ,dimension_alias varchar2);

  -- Lock the measure metadata for update
  --
  -- param owner                 owner of the cube
  -- param cube_name             name of the cube
  -- param measure_name          name of the measure
  -- param wait_for_lock         wait for lock if acquired by other user
  --
  -- raise no_access_privileges  insufficient access privilege
  -- raise measure_not_found     if measure is not defined
  -- raise failed_to_gain_lock   if lock could not be acquired
  procedure lock_measure(owner varchar2
                       , cube_name varchar2
                       , measure_name varchar2
                       , wait_for_lock boolean := false);
                    

  -- PRIVATE internal operation get the internal measure id
  function get_measure_id(cube_owner varchar2
                        , cube_name varchar2
                        , measure_name varchar2)
                          return number;
                             
end;
/
