Rem
Rem $Header: emcore/source/oracle/sysman/emdrep/sql/core/latest/availability/availability_master_agent_pkgbody.sql /st_emcore_10.2.0.4.2db11.2/2 2008/12/09 16:44:43 joyoon Exp $
Rem
Rem availability_master_agent_pkgbody.sql
Rem
Rem Copyright (c) 2003, 2008, Oracle and/or its affiliates.
Rem All rights reserved. 
Rem
Rem    NAME
Rem      availability_master_agent_pkgbody.sql - <one-line expansion of the name>
Rem
Rem    DESCRIPTION
Rem      For multi-agent targets like clusters, one of the agents is chosen
Rem      as a master agent. Master agent is responsible for monitoring and
Rem      managing the target for the period during which it is master. This
Rem      package contains the implementation of master agent handling.
Rem    NOTES
Rem      <other useful comments, qualifications, etc.>
Rem
Rem    MODIFIED   (MM/DD/YY)
Rem    joyoon      12/05/08 - Bug:7577467 - handle instance deletion 
Rem    jsadras     09/23/08 - Bug:7426046, add dbms_assert calls
Rem    rkpandey    09/15/05 - Bug:4608719 - Add more details in error logging 
Rem    rkpandey    06/30/05 - Bug 4332952: support handling of misconfigured agents
Rem                           having multiple rac instances on single agent
Rem    rpinnama    05/08/05 - Bug 4302515: support change url req and targets 
Rem                           rowset in different files 
Rem    neearora    03/15/05 - Added check during registration of callback to 
Rem                           prevent SQL Injection 
Rem    neearora    03/03/05 - Added entry for emdw_log 
Rem    ramalhot    03/03/05 - changed the code for MGMT_CHANGE_AGENT_URL  
Rem    ramalhot    02/17/05 - added handle_agent_port_change 
Rem    streddy     01/30/05 - Handle non-unique cluster membership case 
Rem    ramalhot    01/17/05 - g_member_guid->g_contains_guid
Rem    ramalhot    08/26/04 - cutover to new assoc tables 
Rem    dsahrawa    06/22/04 - use get_job_schedule_record 
Rem    streddy     11/30/03 - Added blackout functions 
Rem    mbhoopat    11/16/03 - master-changed callback changes 
Rem    streddy     10/04/03 - Always use agent timezone for ts calculations 
Rem    streddy     09/25/03 - Handle unreachable changes correctly 
Rem    skini       09/16/03 - Hook into the job system when master agent 
Rem    rpinnama    09/08/03 - Add purge policy for MGMT_MASTER_AGENT 
Rem    skini       09/09/03 - Change in schedule_record structure 
Rem    streddy     07/22/03 - Handle agent up / instance down scenario
Rem    streddy     06/30/03 - Handle cluster instance availability changes
Rem    streddy     06/11/03 - streddy_master_agent_support
Rem    streddy     04/24/03 - Created
Rem

CREATE OR REPLACE PACKAGE BODY em_master_agent AS

-- forward declaration
PROCEDURE EXEC_MASTER_CHANGED_CALLBACKS(v_target_guid IN RAW);
FUNCTION  IS_TARGET_IN_BLACKOUT(v_target_guid IN RAW) RETURN BOOLEAN;

--
-- PURPOSE
--   Initializes master agent for a target. This is called
--   when a target is added to the repository for the firt
--   time.
--
PROCEDURE initialize_master_agent(v_target_guid IN RAW,
                                  v_agent_guid  IN RAW,
                                  v_timestamp   IN DATE)
IS
  l_exists NUMBER;
BEGIN
  IF (EMDW_LOG.p_is_info_set)THEN
     EMDW_LOG.info('initialize_master_agent:Entry v_target_guid = ' || v_target_guid ||
                    ' v_agent_guid = ' || v_agent_guid ||
                    ' v_timestamp = '  || v_timestamp, MODULE_NAME);
  END IF;
  -- Initialize only for non-existing targets
  SELECT 1 INTO l_exists
    FROM mgmt_master_agent
   WHERE target_guid = v_target_guid
     AND ROWNUM < 2;

  IF (EMDW_LOG.p_is_info_set)THEN
     EMDW_LOG.info('initialize_master_agent:Exit', MODULE_NAME);
  END IF;

EXCEPTION
  WHEN NO_DATA_FOUND THEN
    INSERT INTO mgmt_master_agent
         (target_guid, agent_guid, start_timestamp)
    VALUES
         (v_target_guid, v_agent_guid, v_timestamp);

  IF (EMDW_LOG.p_is_info_set)THEN
     EMDW_LOG.info('initialize_master_agent:Exit NO_DATA_FOUND', MODULE_NAME);
  END IF;
END initialize_master_agent;


-- PURPOSE
--    Sets a new master agent
PROCEDURE set_new_master_agent(v_target_guid IN RAW,
                               v_agent_guid IN RAW,
                               v_timestamp  IN DATE)
IS 
  l_curr_emd_url MGMT_TARGETS.emd_url%TYPE;
  l_new_emd_url MGMT_TARGETS.emd_url%TYPE;
  l_agent_tzrgn  MGMT_TARGETS.timezone_region%TYPE;
  l_timestamp   DATE;
BEGIN
  IF (EMDW_LOG.p_is_info_set)THEN
     EMDW_LOG.info('set_new_master_agent:Entry v_target_guid = ' || v_target_guid ||
                    ' v_agent_guid = ' || v_agent_guid ||
                    ' v_timestamp = '  || v_timestamp, MODULE_NAME);
  END IF;

  SELECT emd_url, timezone_region 
    INTO l_new_emd_url, l_agent_tzrgn
    FROM MGMT_TARGETS WHERE target_guid=v_agent_guid;

   -- Get the current master url of the target
   SELECT emd_url INTO l_curr_emd_url
     FROM mgmt_targets
    WHERE target_guid = v_target_guid;

   -- Don't change the master if it is the same
   IF (l_curr_emd_url = l_new_emd_url) THEN
      IF (EMDW_LOG.p_is_debug_set) THEN
        EMDW_LOG.debug('set_new_master_agent: Old and new url same', MODULE_NAME);
      END IF;

      IF (EMDW_LOG.p_is_info_set)THEN
         EMDW_LOG.info('set_new_master_agent:Exit  Old and new url same', MODULE_NAME);
      END IF;

      RETURN;
   END IF;

   l_timestamp := mgmt_global.sysdate_tzrgn(l_agent_tzrgn);


  INSERT INTO mgmt_master_agent
       (target_guid, agent_guid, start_timestamp)
  VALUES
       (v_target_guid, v_agent_guid, l_timestamp);

  -- Add a new entry into the job system's emd queue
  MGMT_JOB_ENGINE.handle_emd_state_change(l_new_emd_url, 1);

  -- Call all registered callbacks
  exec_master_changed_callbacks(v_target_guid);

  -- Somebody included agent_guid as part of mgmt_metric_errors PK as a 
  -- hack (I think beacon guys). This is causing metric errors not to
  -- get cleared in Multi-agent case. 
  -- Manually clearing all non-response-status metric_errors as part of
  -- the switch so that the new master will send the correct status
  -- again.
  FOR met IN (SELECT e.metric_guid metric_guid , coll_name, agent_guid
                FROM mgmt_current_metric_errors e,
                     mgmt_metrics m
               WHERE e.target_guid = v_target_guid
                 AND e.metric_guid = m.metric_guid
	         AND m.metric_name <> mgmt_global.G_AVAIL_METRIC_NAME)
  LOOP
      BEGIN
        INSERT INTO MGMT_METRIC_ERRORS
           (target_guid, metric_guid, coll_name, collection_timestamp, agent_guid)
        VALUES
           (v_target_guid, met.metric_guid, met.coll_name, l_timestamp, met.agent_guid);
      EXCEPTION
          -- ingore duplicate records
          WHEN DUP_VAL_ON_INDEX OR MGMT_GLOBAL.DUPLICATE_RECORD THEN 
	    NULL;
      END;
  END LOOP;

  IF (EMDW_LOG.p_is_info_set)THEN
     EMDW_LOG.info('set_new_master_agent:Exit', MODULE_NAME);
  END IF;
END set_new_master_agent;



-- PURPOSE
--   Processes a new master agent selection. This contains common set
--   set of tasks we need to do for both OMS mediated monitoring
--   and Agent mediated monitoring.
--     o Closes the previous master agent record
--     o Submits a job to make previous master a standby
--     o Submits a job to set the new master agent
-- This is called from BEFORE INSERT TRIGGER of mgmt_master_agent table
PROCEDURE process_new_master_agent(v_target_guid IN RAW,
                                   v_new_master_guid IN RAW,
                                   v_timestamp IN DATE)
IS
  l_new_host_name     MGMT_TARGETS.host_name%TYPE;
  l_new_emd_url       MGMT_TARGETS.emd_url%TYPE;
  l_mon_mode          MGMT_TARGETS.monitoring_mode%TYPE;
  l_rep_side_avail    MGMT_TARGETS.rep_side_avail%TYPE;
  l_old_master_guid   MGMT_TARGETS.target_guid%TYPE;
  l_old_emd_url       MGMT_TARGETS.emd_url%TYPE;
  l_curr_status       MGMT_CURRENT_AVAILABILITY.current_status%TYPE;
  l_rs_metric_guid    MGMT_METRICS.metric_guid%TYPE;
  l_agent_marker      MGMT_AVAILABILITY_MARKER.marker_timestamp%TYPE;
BEGIN
  IF (EMDW_LOG.p_is_info_set)THEN
     EMDW_LOG.info('process_new_master_agent:Entry v_target_guid = ' || v_target_guid ||
                    ' v_new_master_guid = ' || v_new_master_guid ||
                    ' v_timestamp = '  || v_timestamp, MODULE_NAME);
  END IF;

  IF (EMDW_LOG.p_is_debug_set) THEN
      EMDW_LOG.debug('process_new_master_agent: '
                           || v_new_master_guid ||
                          '> as master for <' || v_target_guid ||'>', MODULE_NAME);
  END IF;


  -- update the previous master's end_timestamp to the 
  -- new master agent's start_timestamp
  UPDATE mgmt_master_agent
     SET end_timestamp = v_timestamp
   WHERE target_guid   = v_target_guid
     AND end_timestamp IS NULL
   RETURNING agent_guid INTO l_old_master_guid;

   -- Get the new master agent's hostname and URL
   SELECT host_name, emd_url
     INTO l_new_host_name, l_new_emd_url
     FROM mgmt_targets
    WHERE target_guid = v_new_master_guid;

   -- Fetch the old URL of the target before we set a new one
   SELECT emd_url, monitoring_mode, rep_side_avail
     INTO l_old_emd_url, l_mon_mode, l_rep_side_avail
     FROM mgmt_targets
    WHERE target_guid = v_target_guid;


   -- If the old URL and the new master URL are different, then
   -- change the URL.
   IF (l_new_emd_url <> l_old_emd_url) THEN


      IF (EMDW_LOG.p_is_debug_set) THEN
           EMDW_LOG.debug('process_new_master:Changing the target emd_url', MODULE_NAME);
      END IF;

     -- Insert an entry in change_agent_url to fool the
     -- mgmt_targets update trigger 
     emd_loader.set_emd_url_change(l_old_emd_url, l_new_emd_url);
  
     -- Make the entry in mgmt_targets point to the new master
     UPDATE mgmt_targets
        SET host_name = l_new_host_name, emd_url = l_new_emd_url
      WHERE target_guid = v_target_guid;

     -- remove the change url entry
     emd_loader.clear_emd_url_change(l_old_emd_url, l_new_emd_url);

     -- make the previous master standby for OMS mediated case
     IF (l_mon_mode = mgmt_global.G_MON_MODE_OMS_MEDIATED) THEN

       IF (EMDW_LOG.p_is_debug_set) THEN
           EMDW_LOG.debug('process_new_master: submittig standby job', MODULE_NAME);
       END IF;

        em_master_agent.submit_standby_agent_job(v_target_guid,
                                                  l_old_emd_url,
                                                  v_timestamp);
     END IF;

     IF (EMDW_LOG.p_is_debug_set) THEN
           EMDW_LOG.debug('process_new_master: submittig master job', MODULE_NAME);
     END IF;

     -- Inform the agent to be the new master.
     em_master_agent.submit_master_agent_job(v_target_guid, 
                                             v_new_master_guid,
                                             v_timestamp);
   END IF; -- l_new_emd_url <> l_old_emd_url

   -- If the target is marked as UNREACHABLE, then CLEAR it. This
   -- is needed to handle the following scenario: all agents become
   -- unreachable and hence the target becomes unreachable. Now, one
   -- of the agents which is not the current master becomes the new
   -- master agent. In this case, we need to clear the unreachability
   -- during master agent change. Don't do this targets whose 
   -- availability is calculated on the repository side because the
   -- background dbms_job would do this anyways.
   SELECT current_status 
     INTO l_curr_status
     FROM mgmt_current_availability
    WHERE target_guid = v_target_guid;

   IF (l_curr_status = mgmt_global.G_STATUS_UNREACHABLE AND
       l_rep_side_avail = 0) THEN

      SELECT DISTINCT m.metric_guid
        INTO l_rs_metric_guid
        FROM mgmt_targets t, mgmt_metrics m
       WHERE t.target_guid = v_target_guid
         AND t.target_type = m.target_type
         AND m.metric_name = mgmt_global.G_AVAIL_METRIC_NAME
         AND m.metric_column = mgmt_global.G_AVAIL_METRIC_COLUMN;

      -- Temporarily disable master-agent checks. This is because
      -- we may be doing this in the context of a standby agent
      p_ignore_master_check := TRUE;

      -- Use the new master agent's marker timestamp
      SELECT marker_timestamp
        INTO l_agent_marker
        FROM mgmt_availability_marker
       WHERE target_guid = v_new_master_guid;

      INSERT INTO MGMT_SEVERITY
        (target_guid, metric_guid, collection_timestamp, 
        severity_code)
      VALUES
        (v_target_guid, l_rs_metric_guid, l_agent_marker,
         mgmt_global.G_SEVERITY_UNREACHABLE_CLEAR);
  
      -- Enable the master-agent check again
      p_ignore_master_check := FALSE;
   END IF;


  IF (EMDW_LOG.p_is_info_set)THEN
     EMDW_LOG.info('process_new_master_agent:Exit', MODULE_NAME);
  END IF;
END process_new_master_agent;

--
-- Finds a new master agent.
--
FUNCTION find_new_master_agent(v_target_guid IN RAW,
                               v_force_change IN BOOLEAN,
                               v_curr_master_guid IN RAW)
  RETURN  RAW
IS
  l_new_master_guid MGMT_TARGETS.target_guid%TYPE;
  l_is_cluster      NUMBER;
  l_curr_master_status NUMBER;
BEGIN
  IF (EMDW_LOG.p_is_info_set)THEN
     EMDW_LOG.info('find_new_master_agent:Entry v_target_guid = ' || v_target_guid ||
                    ' v_curr_master_guid = '  || v_curr_master_guid, MODULE_NAME);
  END IF;

  -- Figure out if this is a cluster target
  BEGIN
    SELECT to_number(p.property_value) 
      INTO l_is_cluster
      FROM MGMT_TYPE_PROPERTIES p, MGMT_TARGETS t
     WHERE t.target_guid = v_target_guid
       AND t.target_type = p.target_type
       AND p.property_name = mgmt_global.G_IS_CLUSTER_PROP;
  EXCEPTION
         WHEN OTHERS THEN
           l_is_cluster := 0;
  END;
  
  
  IF (l_is_cluster <> 0) THEN

    IF (EMDW_LOG.p_is_debug_set) THEN
       EMDW_LOG.debug('In find_new_agent for cluster case', MODULE_NAME);
    END IF;

    -- Get the master for cluster targets by taking instance availability
    -- into consideration. Note that if a target is currently in UP state,
    -- then the agent MUST be in UP state. So, we are optimizing the query
    -- using this knowledge.
    BEGIN
    SELECT agent_guid INTO l_new_master_guid
      FROM (SELECT t2.target_guid agent_guid
              FROM mgmt_target_assocs mt,
                   mgmt_targets t1,
                   mgmt_targets t2,
                   mgmt_current_availability avail
             WHERE mt.source_target_guid = v_target_guid
               AND mt.assoc_target_guid = avail.target_guid
               AND mt.assoc_guid = MGMT_ASSOC.g_contains_guid
               AND avail.current_status = mgmt_global.G_STATUS_UP
               AND mt.assoc_target_guid = t1.target_guid
               AND t1.emd_url = t2.emd_url
               AND t2.target_type = mgmt_global.G_AGENT_TARGET_TYPE
               AND t2.target_guid <> v_curr_master_guid)
      WHERE ROWNUM = 1;

      -- return the new master if we found one

      IF (EMDW_LOG.p_is_info_set)THEN
        EMDW_LOG.info('find_new_master_agent:Exit = '|| l_new_master_guid, MODULE_NAME);
      END IF;

      RETURN l_new_master_guid;
    EXCEPTION
      WHEN NO_DATA_FOUND THEN
         -- There is no agent that is currently up, let us leave the
         -- master agent alone
         IF (EMDW_LOG.p_is_debug_set) THEN
            EMDW_LOG.debug('cluster case: No new master agent found - ' ||
                       ' All instances down for ' || v_target_guid, MODULE_NAME);
         END IF;
    END;

    -- If this is not a force change, then check if the current
    -- master is UP, then don't force a change
    IF (v_force_change = FALSE) THEN
      BEGIN
        SELECT current_status
          INTO l_curr_master_status
          FROM mgmt_current_availability
         WHERE target_guid = v_curr_master_guid;
        IF (l_curr_master_status = mgmt_global.G_STATUS_UP) THEN
           RETURN NULL;        
        END IF;
      EXCEPTION
         WHEN NO_DATA_FOUND THEN
           -- agent deletion case
           NULL;
      END;
    END IF; -- IF (v_force_change = FALSE)
  END IF; -- IF (l_is_cluster <> 0) THEN


  -- Now try to find an agent that is simply up
  IF (EMDW_LOG.p_is_debug_set) THEN
     EMDW_LOG.debug('In find_new_agent for non-cluster case', MODULE_NAME);
  END IF;

  -- Get the master for non-cluster targets
  BEGIN
    SELECT agent_guid INTO l_new_master_guid
      FROM (SELECT assoc.agent_guid agent_guid
              FROM mgmt_target_agent_assoc  assoc,
                   mgmt_current_availability avail
             WHERE assoc.target_guid = v_target_guid
               AND assoc.agent_guid <> v_curr_master_guid
               AND assoc.agent_guid  = avail.target_guid
               AND avail.current_status = mgmt_global.G_STATUS_UP)
      WHERE ROWNUM = 1;
  EXCEPTION
      WHEN NO_DATA_FOUND THEN
         -- There is no agent that is currently up, let us leave the
         -- master agent alone
         IF (EMDW_LOG.p_is_debug_set) THEN
            EMDW_LOG.debug('No new master agent found - ' ||
                       ' All agents down for ' || v_target_guid, MODULE_NAME);
         END IF;
  END;

  IF (EMDW_LOG.p_is_info_set)THEN
     EMDW_LOG.info('find_new_master_agent:Exit = '|| l_new_master_guid, MODULE_NAME);
  END IF;

  RETURN l_new_master_guid;

END find_new_master_agent;


-- 
-- PURPOSE
--   Sets the specified agent to the new master agent
--
PROCEDURE change_master_agent(v_target_guid IN RAW,
                              v_curr_master_guid IN RAW,
                              v_timestamp   IN DATE,
                              v_force_change IN BOOLEAN,
                              v_new_master_guid OUT RAW)
IS
BEGIN
  IF (EMDW_LOG.p_is_info_set)THEN
     EMDW_LOG.info('change_master_agent:Entry v_target_guid = ' || v_target_guid ||
                   ' v_curr_master_guid = '  || v_curr_master_guid ||
                   ' v_timestamp = '         || v_timestamp, MODULE_NAME);
  END IF;

  v_new_master_guid := find_new_master_agent(v_target_guid, 
                                             v_force_change,
                                             v_curr_master_guid);

  -- if there is no new master available, then don't do anything
  IF (v_new_master_guid IS NULL) THEN

    IF (EMDW_LOG.p_is_info_set)THEN
       EMDW_LOG.info('change_master_agent:Exit v_new_master_guid is NULL', MODULE_NAME);
    END IF;
    RETURN;
  END IF;

  IF (EMDW_LOG.p_is_debug_set) THEN
      EMDW_LOG.debug('change_master_agent: Setting agent<'
                           || v_new_master_guid ||
                          '> as master for <' || v_target_guid ||'>', MODULE_NAME);
  END IF;

  -- Don't change the master agent if we selected the same agent
  -- during instance down times
  IF (v_new_master_guid <> v_curr_master_guid) THEN
    set_new_master_agent(v_target_guid, v_new_master_guid, v_timestamp);
  END IF;

  IF (EMDW_LOG.p_is_info_set)THEN
     EMDW_LOG.info('change_master_agent:Exit v_new_master_guid = ' || 
                    v_new_master_guid, MODULE_NAME);
  END IF;
END change_master_agent;


--
--
-- PURPOSE
--   Process master agent changes for oms_mediated targets.
-- DESCRIPTION
--   For OMS mediated targets, master agent changes happen on
--   the OMS side in the procedure. If an agent becomes UP,
--   and if the current master agent is down for a target, then
--   this agent is set as master agent for that target. If an
--   agent goes into non-UP state, then a new master agent is
--   selected if this agent is the current master agent.
--
PROCEDURE process_oms_med_target(v_target_guid IN RAW,
                                 v_agent_guid  IN RAW,
                                 v_new_status  IN NUMBER,
                                 v_timestamp   IN DATE,
                                 v_curr_master_guid IN RAW,
                                 v_curr_master_status IN NUMBER)
IS
   l_new_master_guid    MGMT_TARGETS.target_guid%TYPE;
BEGIN
  IF (EMDW_LOG.p_is_info_set)THEN
     EMDW_LOG.info('process_oms_med_target:Entry v_target_guid = ' || v_target_guid ||
                   ' v_agent_guid = '         || v_agent_guid       ||
                   ' v_new_status = '         || v_new_status       ||     
                   ' v_timestamp = '          || v_timestamp        ||
                   ' v_curr_master_guid = '   || v_curr_master_guid ||
                   ' v_curr_master_status = ' || v_curr_master_status, MODULE_NAME);
  END IF;


   IF (EMDW_LOG.p_is_debug_set) THEN
       EMDW_LOG.debug('process_oms_med_target: agent status ' || v_new_status ||
                            '; master status =' || v_curr_master_status, MODULE_NAME);
   END IF;

   -- set this agent as master agent if the current master agent is not UP
   IF (v_new_status = mgmt_global.G_STATUS_UP) THEN

     IF (v_agent_guid <> v_curr_master_guid AND
         v_curr_master_status <> mgmt_global.G_STATUS_UP) THEN
       IF (EMDW_LOG.p_is_debug_set) THEN
          EMDW_LOG.debug('Setting agent: <' || v_agent_guid ||
                              '> as master for <' || v_target_guid, MODULE_NAME);
       END IF;

       set_new_master_agent(v_target_guid, v_agent_guid, v_timestamp);
    END IF;

    -- If the agent status is not UP and this agent was the master agent,
    -- then find an agent that is currently UP and set that as the new
    -- master agent. Note that we treat all non-UP states equally because
    -- if we have someother UP agent, that would be a better master agent
    -- than a non-UP agent
    ELSIF (v_curr_master_guid = v_agent_guid) THEN
      change_master_agent(v_target_guid, v_agent_guid, v_timestamp,
                          FALSE, l_new_master_guid);
    END IF;

  IF (EMDW_LOG.p_is_info_set)THEN
     EMDW_LOG.info('process_oms_med_target:Exit', MODULE_NAME);
  END IF;
END process_oms_med_target;

--
-- PURPOSE
--   Process master agent changes for agent_mediated targets
-- DESCRIPTION
--   For agent mediated case, master agent changes happen on the agent
--   side. But when the current master agent becomes unreachable, the
--   target will be marked as unreachable even though someother agent
--   is still up.
PROCEDURE process_agent_med_target(v_target_guid IN RAW,
                                   v_agent_guid  IN RAW,
                                   v_new_status  IN NUMBER,
                                   v_timestamp   IN DATE,
                                   v_curr_master_guid IN RAW,
                                   v_curr_master_status IN NUMBER)
IS
   l_new_master_guid    MGMT_TARGETS.target_guid%TYPE;
BEGIN
  IF (EMDW_LOG.p_is_info_set)THEN
     EMDW_LOG.info('process_agent_med_target:Entry v_target_guid = ' || v_target_guid ||
                   ' v_agent_guid = '         || v_agent_guid       ||
                   ' v_new_status = '         || v_new_status       ||     
                   ' v_timestamp = '          || v_timestamp        ||
                   ' v_curr_master_guid = '   || v_curr_master_guid ||
                   ' v_curr_master_status = ' || v_curr_master_status, MODULE_NAME);
  END IF;
   -- Master agent changes for all cases other then UNREACHABLE is performed
   -- on the agent side
   IF (v_agent_guid = v_curr_master_guid AND
       v_new_status = mgmt_global.G_STATUS_UNREACHABLE) THEN
      change_master_agent(v_target_guid, v_agent_guid, v_timestamp,
                          TRUE, l_new_master_guid);
   END IF;
 
  IF (EMDW_LOG.p_is_info_set)THEN
     EMDW_LOG.info('process_agent_med_target:Exit', MODULE_NAME);
  END IF;
END process_agent_med_target;


--
-- PURPOSE
--   Processes agent availability changes. If this agent is monitoring any
--   multi-agent targets, then master agent settings are appropriately
--   updated for these targets.
--   This is called from two places:
--   [1] From update_current_availability
--   [2] From ping manager when an agent is detected as unreachable
--
PROCEDURE process_agent_avail_change(v_agent_guid IN RAW,       
                                     v_agent_new_status IN NUMBER,
                                     v_timestamp  IN DATE)
IS
   l_curr_master_status MGMT_CURRENT_AVAILABILITY.current_status%TYPE;
   l_curr_master_guid   MGMT_TARGETS.target_guid%TYPE;
   l_new_master_guid    MGMT_TARGETS.target_guid%TYPE;	
BEGIN
  IF (EMDW_LOG.p_is_info_set)THEN
     EMDW_LOG.info('process_agent_avail_change:Entry v_agent_guid = ' || v_agent_guid ||
                   ' v_agent_new_status = ' || v_agent_new_status  ||     
                   ' v_timestamp = '        || v_timestamp, MODULE_NAME);
  END IF;

   IF (EMDW_LOG.p_is_debug_set) THEN
     EMDW_LOG.debug('BEGIN: process_agent_avail_change for : ' || 
                           v_agent_guid, MODULE_NAME);
   END IF;

   -- For each mutli-agent target for which this agent is enlisted as
   -- an agent, update the master agent settings as needed
   FOR tgt in (SELECT a.target_guid target_guid, t.emd_url emd_url,
                      t.target_name target_name, 
                      t.monitoring_mode monitoring_mode
                 FROM mgmt_target_agent_assoc a, mgmt_targets t
                WHERE a.agent_guid = v_agent_guid
                  AND a.target_guid = t.target_guid)
   LOOP
      IF (EMDW_LOG.p_is_debug_set) THEN
        EMDW_LOG.debug('Processing target : ' || tgt.target_name, MODULE_NAME);
      END IF;

      -- If the target is in blackout, then don't change
      -- master agent
      IF (IS_TARGET_IN_BLACKOUT(tgt.target_guid)) THEN
        IF (EMDW_LOG.p_is_debug_set) THEN
          EMDW_LOG.debug('Skipping .. target in blackout', MODULE_NAME);
        END IF;
        GOTO NEXT_TGT;
      END IF;

      -- Get the current master agent of the target
      SELECT target_guid INTO l_curr_master_guid
        FROM mgmt_targets
       WHERE emd_url = tgt.emd_url
         AND target_type = mgmt_global.G_AGENT_TARGET_TYPE;

      SELECT current_status INTO l_curr_master_status
        FROM mgmt_current_availability
       WHERE target_guid = l_curr_master_guid;


      IF (tgt.monitoring_mode = mgmt_global.G_MON_MODE_OMS_MEDIATED) THEN
           process_oms_med_target(tgt.target_guid,
                                  v_agent_guid,
                                  v_agent_new_status,
                                  v_timestamp,
                                  l_curr_master_guid,
                                  l_curr_master_status);
      ELSIF (tgt.monitoring_mode = mgmt_global.G_MON_MODE_AGENT_MEDIATED) THEN
           process_agent_med_target(tgt.target_guid,
                                    v_agent_guid,
                                    v_agent_new_status,
                                    v_timestamp,  
                                    l_curr_master_guid,
                                    l_curr_master_status);

      END IF;
     <<NEXT_TGT>>
      NULL;
   END LOOP; 

   IF (EMDW_LOG.p_is_debug_set) THEN
     EMDW_LOG.debug('DONE: process_agent_avail_change', MODULE_NAME);
   END IF;


  IF (EMDW_LOG.p_is_info_set)THEN
     EMDW_LOG.info('process_agent_avail_change:Exit', MODULE_NAME);
  END IF;
EXCEPTION
  WHEN OTHERS THEN
   mgmt_log.log_error(v_module_name_in => 'MASTER_AGENT',
                      v_error_code_in  =>  0,
                      v_error_msg_in   => 'process_agent_avail_change: v_agent_guid = ' || v_agent_guid ||
                                          ' v_agent_new_status = ' || v_agent_new_status  ||
                                          ' v_timestamp = '        || v_timestamp ||
                                          ' Error - '||SUBSTR(SQLERRM, 1, 2000));
 
END process_agent_avail_change;


--
-- PURPOSE
--  Processes cluster instance availability changes for OMS mediated
--  Mode. If the cluster instance on the current master goes down, then
--  this procedure initiates re-mastering. Similarly, if the current
--  master is DOWN and if the cluster instance status is UP, then
--  the cluster instance's agent is picked as new master.
--
PROCEDURE process_cluster_inst_changes(v_target_guid IN RAW,
                                       v_new_status  IN NUMBER,
                                       v_timestamp   IN DATE)       
IS 
   l_inst_agent_guid   MGMT_TARGETS.target_guid%TYPE;
   l_curr_master_guid  MGMT_TARGETS.target_guid%TYPE;
   l_cluster_guid      MGMT_TARGETS.target_guid%TYPE;
   l_up_inst_count     NUMBER;
BEGIN
  IF (EMDW_LOG.p_is_info_set)THEN
     EMDW_LOG.info('process_cluster_inst_changes:Entry v_target_guid = ' || v_target_guid ||
                   ' v_new_status = ' || v_new_status  ||     
                   ' v_timestamp = '  || v_timestamp, MODULE_NAME);
  END IF;

   IF (EMDW_LOG.p_is_debug_set) THEN
     EMDW_LOG.debug('In process_cluster_inst_changes for ' ||
                          v_target_guid, MODULE_NAME);
   END IF;


   -- Get the agent of the cluster instance
   SELECT t2.target_guid 
     INTO l_inst_agent_guid
     FROM mgmt_targets t1, mgmt_targets t2
    WHERE t1.target_guid = v_target_guid
      AND t1.emd_url = t2.emd_url
      AND t2.target_type = mgmt_global.G_AGENT_TARGET_TYPE;

   -- It is possible that the cluser instance is a member
   -- of more than one cluster parent target. Get all the
   -- cluster parents and iterate over them
   FOR ct IN (
       SELECT  tm.source_target_guid target_guid
         FROM  mgmt_target_assocs tm, 
               mgmt_type_properties p, 
               mgmt_targets ct
        WHERE tm.assoc_target_guid = v_target_guid
          AND tm.assoc_guid = MGMT_ASSOC.g_contains_guid
          AND ct.target_guid = tm.source_target_guid
          AND ct.monitoring_mode = mgmt_global.G_MON_MODE_OMS_MEDIATED
          AND p.target_type = ct.target_type
          AND p.property_name = MGMT_GLOBAL.G_IS_CLUSTER_PROP)
   LOOP
     -- process each parent individually

     l_cluster_guid := ct.target_guid;

     -- Get current master for the cluster target
     SELECT ma.agent_guid
       INTO l_curr_master_guid
       FROM mgmt_master_agent ma
      WHERE ma.target_guid = l_cluster_guid
        AND ma.end_timestamp IS NULL;

     -- Skip master agent change if target is blacked out
     IF (IS_TARGET_IN_BLACKOUT(l_cluster_guid)) THEN
       IF (EMDW_LOG.p_is_debug_set) THEN
          EMDW_LOG.debug('Skipping .. target in blackout', MODULE_NAME);
        END IF;

        IF (EMDW_LOG.p_is_info_set)THEN
           EMDW_LOG.info('process_cluster_inst_changes:Exit Target is blacked out', MODULE_NAME);
        END IF;

        RETURN;
     END IF;

     -- There are only scenarios we need to handle:
     -- [1] Cluster instance's agent is the current master AND
     --     cluster instance went DOWN.
     -- [2] Cluster instance's agent is not the current master AND
     --     cluster instance is UP and the instance on the current
     --     master is DOWN.
   
     --  [1] Instance's agent is the current master
     IF (l_curr_master_guid = l_inst_agent_guid) THEN

        -- If the instance is not UP, then remaster
        IF (v_new_status <> mgmt_global.G_STATUS_UP) THEN

          IF (EMDW_LOG.p_is_debug_set) THEN
             EMDW_LOG.debug('Cluster instance status is down; ' ||
                        'Initiating re-mastering for :' || l_cluster_guid, MODULE_NAME);
          END IF;

          process_oms_med_target(l_cluster_guid,
                                 l_curr_master_guid,
                                 mgmt_global.G_STATUS_DOWN,
                                 v_timestamp,
                                 l_curr_master_guid,
                                 mgmt_global.G_STATUS_DOWN);
        END IF;
     ELSIF (v_new_status = mgmt_global.G_STATUS_UP) THEN
        -- Get the count of cluster instances that are currently up
        -- on this agent.
        SELECT count(*) INTO l_up_inst_count
          FROM mgmt_target_assocs mt,
               mgmt_targets t1,
               mgmt_targets t2,
               mgmt_current_availability avail
         WHERE mt.source_target_guid = l_cluster_guid
           AND mt.assoc_target_guid = t1.target_guid
           AND mt.assoc_guid = MGMT_ASSOC.g_contains_guid
           AND t1.emd_url = t2.emd_url
           AND t2.target_guid = l_curr_master_guid
           AND mt.assoc_target_guid = avail.target_guid
           AND avail.current_status = mgmt_global.G_STATUS_UP;


        -- Pick the cluster instance's agent as the new master agent
        IF (l_up_inst_count = 0) THEN
  
          IF (EMDW_LOG.p_is_debug_set) THEN
            EMDW_LOG.debug('In process_cluster_inst_changes: Changing '||
                                 ' master agent to ' || l_inst_agent_guid, MODULE_NAME);
          END IF;

          IF (l_curr_master_guid <> l_inst_agent_guid) THEN
            set_new_master_agent(l_cluster_guid, l_inst_agent_guid, v_timestamp);
          END IF;
        END IF;
     END IF;
   END LOOP;

  IF (EMDW_LOG.p_is_info_set)THEN
     EMDW_LOG.info('process_cluster_inst_changes:Exit', MODULE_NAME);
  END IF;
END process_cluster_inst_changes;

-- 
-- PURPOSE
--   Returns true if the specified emd_url is the agent_url
--   of the specified target for the specified timestamp
--
FUNCTION is_master_agent_for_ts(v_target_guid IN RAW,
                                v_emd_url IN VARCHAR2,
                                v_timestamp IN DATE)
  RETURN BOOLEAN 
IS
  l_master_guid  MGMT_TARGETS.target_guid%TYPE;
  l_mon_mode     MGMT_TARGETS.monitoring_mode%TYPE;
BEGIN
  IF (EMDW_LOG.p_is_info_set)THEN
     EMDW_LOG.info('is_master_agent_for_ts:Entry v_target_guid = ' || v_target_guid ||
                   ' v_emd_url = ' || v_emd_url  ||     
                   ' v_timestamp = '  || v_timestamp, MODULE_NAME);
  END IF;

  -- If no emd_url set, then it is one of the system generated
  -- severities
  IF (v_emd_url IS NULL) THEN
     IF (EMDW_LOG.p_is_debug_set) THEN
       EMDW_LOG.debug('is_master_agent: NULL URL', MODULE_NAME);
     END IF;

     IF (EMDW_LOG.p_is_info_set)THEN
        EMDW_LOG.info('is_master_agent_for_ts:Exit return Value = TRUE', MODULE_NAME);
     END IF;

    RETURN TRUE;
  END IF;

  -- Don't do master-agent check if we are inserting severities for
  -- handling unreachability changes
  IF (p_ignore_master_check) THEN
     IF (EMDW_LOG.p_is_debug_set) THEN
       EMDW_LOG.debug('is_master_agent: master agent check disabled', MODULE_NAME);
     END IF;

     IF (EMDW_LOG.p_is_info_set)THEN
        EMDW_LOG.info('is_master_agent_for_ts:Exit return Value = TRUE', MODULE_NAME);
     END IF;

    RETURN TRUE;
  END IF;

  -- Check for vanilla single agent case
  SELECT monitoring_mode
    INTO l_mon_mode
    FROM mgmt_targets
   WHERE target_guid = v_target_guid;

  IF (l_mon_mode NOT IN (MGMT_GLOBAL.G_MON_MODE_OMS_MEDIATED,
                         MGMT_GLOBAL.G_MON_MODE_AGENT_MEDIATED)) THEN
     IF (EMDW_LOG.p_is_debug_set) THEN
       EMDW_LOG.debug('is_master_agent: Vanilla single agent target' 
                            || v_target_guid, MODULE_NAME);
     END IF;

     IF (EMDW_LOG.p_is_info_set)THEN
        EMDW_LOG.info('is_master_agent_for_ts:Exit return Value = TRUE', MODULE_NAME);
     END IF;

     RETURN TRUE;
  END IF;

  BEGIN
    -- Now do the master agent check
    SELECT m.agent_guid INTO l_master_guid
      FROM mgmt_master_agent m, mgmt_targets t
     WHERE m.target_guid = v_target_guid
       AND start_timestamp <= v_timestamp
       AND (end_timestamp  > v_timestamp OR
            end_timestamp IS NULL)
       AND m.agent_guid = t.target_guid
       AND t.emd_url = v_emd_url;

     IF (EMDW_LOG.p_is_debug_set) THEN
       EMDW_LOG.debug(SUBSTR(v_emd_url, 1, 200) ||
          ' is the master agent for ' || v_target_guid, MODULE_NAME);
     END IF;

     IF (EMDW_LOG.p_is_info_set)THEN
        EMDW_LOG.info('is_master_agent_for_ts:Exit return Value = TRUE', MODULE_NAME);
     END IF;

     RETURN TRUE;
   EXCEPTION
     WHEN NO_DATA_FOUND THEN
       IF (EMDW_LOG.p_is_debug_set) THEN
         EMDW_LOG.debug(SUBSTR(v_emd_url, 1, 200) ||
            ' is NOT the master agent for ' || v_target_guid, MODULE_NAME);
       END IF;

       IF (EMDW_LOG.p_is_info_set)THEN
           EMDW_LOG.info('is_master_agent_for_ts:Exit return Value = FALSE', MODULE_NAME);
       END IF;

       RETURN FALSE;
   END;

END is_master_agent_for_ts;

--
-- PURPOSE
--   Submits a job to mark an agent as the master agent
--
PROCEDURE submit_master_agent_job(v_target_guid IN RAW,
                                  v_agent_guid  IN RAW,
                                  v_timestamp   IN DATE)
IS
  l_target_name  MGMT_TARGETS.target_name%TYPE;
  l_target_type  MGMT_TARGETS.target_type%TYPE;
  l_monitoring_mode  VARCHAR2(2);
  l_job_params MGMT_JOB_PARAM_LIST;
  l_job_targets MGMT_JOB_TARGET_LIST;
  l_schedule MGMT_JOB_SCHEDULE_RECORD;
  l_execution_id RAW(16);
  l_job_name VARCHAR2(64);
  l_job_id RAW(16);
  l_agent_url  MGMT_TARGETS.emd_url%TYPE;
  l_agent_name MGMT_TARGETS.target_name%TYPE;
BEGIN
  IF (EMDW_LOG.p_is_info_set)THEN
     EMDW_LOG.info('submit_master_agent_job:Entry v_target_guid = ' || v_target_guid ||
                   ' v_agent_guid = ' || v_agent_guid  ||     
                   ' v_timestamp = '  || v_timestamp, MODULE_NAME);
  END IF;

   IF (EMDW_LOG.p_is_debug_set) THEN
      EMDW_LOG.debug('in submit_master_agent', MODULE_NAME);
   END IF;

  SELECT target_name, target_type, DECODE(monitoring_mode, 1, '1', '2')
    INTO l_target_name, l_target_type, l_monitoring_mode
    FROM mgmt_targets
   WHERE target_guid = v_target_guid;

   SELECT target_name, emd_url INTO l_agent_name, l_agent_url
     FROM mgmt_targets
    WHERE target_guid = v_agent_guid;

  l_job_targets := MGMT_JOB_TARGET_LIST();
  l_job_targets.extend(1);
  l_job_targets(1) := MGMT_JOB_TARGET_RECORD(l_agent_name, 
                                             mgmt_global.G_AGENT_TARGET_TYPE);


  l_job_params := MGMT_JOB_PARAM_LIST();
  l_job_params.extend(6);
  l_job_params(1) := MGMT_JOB_PARAM_RECORD('target_name', 1, 
                                            l_target_name, null);

  l_job_params(2) := MGMT_JOB_PARAM_RECORD('target_type', 1, 
                                            l_target_type, null);

  l_job_params(3) := MGMT_JOB_PARAM_RECORD('target_guid', 1, 
                                            v_target_guid, null);


  l_job_params(4) := MGMT_JOB_PARAM_RECORD('monitoring_mode', 1, 
                                           l_monitoring_mode,  null);


  l_job_params(5) := MGMT_JOB_PARAM_RECORD('emd_url', 1, 
                                            l_agent_url, null);

  l_job_params(6) := MGMT_JOB_PARAM_RECORD('timestamp', 1, 
                                            v_timestamp, null);


  -- schedule immediately
  l_schedule := MGMT_JOBS.get_job_schedule_record(MGMT_JOBS.ONE_TIME_FREQUENCY_CODE,
                                         SYSDATE, null, 0, 0, 0, null, null,
                                         MGMT_JOBS.TIMEZONE_REPOSITORY, 0, 0,
                                         null);
  
  l_job_name := 'SETMASTERAGENT_JOB_' || SYS_GUID();

  -- Submit a job that runs just once 
  MGMT_JOBS.submit_job(l_job_name,
                        'This job sets master agent',
                        'SetMasterAgent',
                        l_job_targets,
                        l_job_params,
                        l_schedule,
                        l_job_id,
                        l_execution_id,
                        null,
                        MGMT_JOB_ENGINE.SYSTEM_JOB_RETRY);


  IF (EMDW_LOG.p_is_info_set)THEN
     EMDW_LOG.info('submit_master_agent_job:Exit', MODULE_NAME);
  END IF;

END;



--
-- PURPOSE
--   Submits a job to mark an agent as a standby agent
--
PROCEDURE submit_standby_agent_job(v_target_guid IN RAW,
                                   v_emd_url IN VARCHAR2,
                                   v_timestamp IN DATE)
IS
  l_target_name  MGMT_TARGETS.target_name%TYPE;
  l_target_type  MGMT_TARGETS.target_type%TYPE;
BEGIN
  IF (EMDW_LOG.p_is_info_set)THEN
     EMDW_LOG.info('submit_standby_agent_job:Entry v_target_guid = ' || v_target_guid ||
                   ' v_emd_url = ' || v_emd_url  ||     
                   ' v_timestamp = '  || v_timestamp, MODULE_NAME);
  END IF;

  SELECT target_name, target_type
    INTO l_target_name, l_target_type
    FROM mgmt_targets
   WHERE target_guid = v_target_guid;

   submit_standby_agent_job(l_target_name, l_target_type, v_emd_url, v_timestamp);

  IF (EMDW_LOG.p_is_info_set)THEN
     EMDW_LOG.info('submit_standby_agent_job:Exit', MODULE_NAME);
  END IF;
END submit_standby_agent_job;

--
-- PURPOSE
--   Submits a job to mark an agent as a standby agent
--
PROCEDURE submit_standby_agent_job(v_target_name IN VARCHAR2,
                            v_target_type IN VARCHAR2,
                            v_emd_url IN VARCHAR2,
                            v_timestamp IN DATE)
IS
  l_job_params MGMT_JOB_PARAM_LIST;
  l_job_targets MGMT_JOB_TARGET_LIST;
  l_schedule MGMT_JOB_SCHEDULE_RECORD;
  l_execution_id RAW(16);
  l_job_name VARCHAR2(64);
  l_job_id RAW(16);
  l_agent_name mgmt_targets.target_name%TYPE;
BEGIN
  IF (EMDW_LOG.p_is_info_set)THEN
     EMDW_LOG.info('submit_standby_agent_job:Entry v_target_name = ' || v_target_name ||
                   ' v_target_type = ' || v_target_type ||     
                   ' v_emd_url = '     || v_emd_url     ||     
                   ' v_timestamp = '   || v_timestamp, MODULE_NAME);
  END IF;

   SELECT target_name
     INTO l_agent_name
     FROM mgmt_targets
    WHERE target_type = mgmt_global.G_AGENT_TARGET_TYPE
      AND emd_url = v_emd_url;

  l_job_targets := MGMT_JOB_TARGET_LIST();
  l_job_targets.extend(1);
  l_job_targets(1) := MGMT_JOB_TARGET_RECORD(l_agent_name, 
                               mgmt_global.G_AGENT_TARGET_TYPE);

  l_job_params := MGMT_JOB_PARAM_LIST();
  l_job_params.extend(4);
  l_job_params(1) := MGMT_JOB_PARAM_RECORD('target_name', 1, 
                                            v_target_name, null);

  l_job_params(2) := MGMT_JOB_PARAM_RECORD('target_type', 1, 
                                            v_target_type, null);

  l_job_params(3) := MGMT_JOB_PARAM_RECORD('emd_url', 1, 
                                            v_emd_url, null);

  l_job_params(4) := MGMT_JOB_PARAM_RECORD('timestamp', 1, 
                                            v_timestamp, null);

  -- schedule immediately
  l_schedule := MGMT_JOBS.get_job_schedule_record(MGMT_JOBS.ONE_TIME_FREQUENCY_CODE,
                                         SYSDATE, null, 0, 0, 0, null, null,
                                         MGMT_JOBS.TIMEZONE_REPOSITORY, 0, 0,
                                         null);
  
  l_job_name := 'SETSTANDBYAGENT_JOB_' || SYS_GUID();

  MGMT_JOBS.submit_job(l_job_name,
                        'This is a set standby agent job',
                        'SetStandbyAgent',
                        l_job_targets,
                        l_job_params,
                        l_schedule,
                        l_job_id,
                        l_execution_id,
                        null,
                        MGMT_JOB_ENGINE.SYSTEM_JOB_RETRY);

  IF (EMDW_LOG.p_is_info_set)THEN
     EMDW_LOG.info('submit_standby_agent_job:Exit', MODULE_NAME);
  END IF;
END submit_standby_agent_job;


--
-- PURPOSE
--  Given a target guid and an agent guid, this procedure processes the
--  target and marks the agent as standby if a master is already present
--  for the target.
--  This is called when a new agent is added for a multi-agent target.
PROCEDURE update_agent_mon_status(v_target_guid IN RAW,
                                  v_agent_guid  IN RAW)
IS
  l_master_guid  MGMT_TARGETS.target_guid%TYPE;
  l_agent_tzrgn  MGMT_TARGETS.timezone_region%TYPE;
  l_agent_url    MGMT_TARGETS.emd_url%TYPE;     
  l_mon_mode    MGMT_TARGETS.monitoring_mode%TYPE;
BEGIN
  IF (EMDW_LOG.p_is_info_set)THEN
     EMDW_LOG.info('update_agent_mon_status:Entry v_target_guid = ' || v_target_guid ||
                   ' v_tgent_guid = ' || v_agent_guid, MODULE_NAME);
  END IF;

 SELECT monitoring_mode
   INTO l_mon_mode
   FROM MGMT_TARGETS
  WHERE target_guid = v_target_guid;

  IF (EMDW_LOG.p_is_debug_set) THEN
    EMDW_LOG.debug('In update_agent_mon_status', MODULE_NAME);
  END IF;  

  IF (l_mon_mode <> MGMT_GLOBAL.G_MON_MODE_OMS_MEDIATED) THEN

     IF (EMDW_LOG.p_is_info_set)THEN
        EMDW_LOG.info('update_agent_mon_status:Exit Monitoring_Mode <> ' || 
                       MGMT_GLOBAL.G_MON_MODE_OMS_MEDIATED, MODULE_NAME);
     END IF;

     RETURN;
  END IF;

  BEGIN
   SELECT agent_guid INTO l_master_guid
     FROM mgmt_master_agent
    WHERE target_guid = v_target_guid
      AND end_timestamp IS NULL;
  EXCEPTION
   -- There is no master, don't make this dude standby; this can
   -- happen if none of the agents reported an UP yet.
   WHEN NO_DATA_FOUND THEN
     NULL;
  END;

  -- if the agent is not current master, then make it standby
  IF (l_master_guid <> v_agent_guid) THEN
    SELECT timezone_region, emd_url 
      INTO l_agent_tzrgn, l_agent_url
      FROM mgmt_targets
     WHERE target_guid = v_agent_guid;

    IF (EMDW_LOG.p_is_debug_set) THEN
      EMDW_LOG.debug('Submitting a standby job', MODULE_NAME);
    END IF;  

     submit_standby_agent_job(v_target_guid, l_agent_url,
            mgmt_global.sysdate_tzrgn(l_agent_tzrgn));
  END IF;

  IF (EMDW_LOG.p_is_debug_set) THEN
    EMDW_LOG.debug('Leaving update_agent_mon_status', MODULE_NAME);
  END IF;  


  IF (EMDW_LOG.p_is_info_set)THEN
     EMDW_LOG.info('update_agent_mon_status:Exit', MODULE_NAME);
  END IF;
END update_agent_mon_status;


-- Selects a new master agent if needed when an agent
-- is getting deleted
PROCEDURE process_agent_deletion(v_agent_guid IN RAW)
IS
  l_new_master_guid    MGMT_TARGETS.target_guid%TYPE;
  l_target_tzrgn  MGMT_TARGETS.timezone_region%TYPE;
  l_timestamp     DATE;
  l_mon_mode      MGMT_TARGETS.monitoring_mode%TYPE;
BEGIN
  IF (EMDW_LOG.p_is_info_set)THEN
     EMDW_LOG.info('process_agent_deletion:Entry v_agent_guid = ' || v_agent_guid, MODULE_NAME);
  END IF;

   IF (EMDW_LOG.p_is_debug_set) THEN
      EMDW_LOG.debug('In process_agent_deletion:', MODULE_NAME);
   END IF;

   DELETE mgmt_target_agent_assoc 
    WHERE agent_guid = v_agent_guid;

   -- for each target for which this is a master agent 
   FOR tgt IN (SELECT target_guid FROM mgmt_master_agent
                WHERE agent_guid = v_agent_guid
                  AND end_timestamp IS NULL)
   LOOP
     SELECT timezone_region, monitoring_mode
       INTO l_target_tzrgn, l_mon_mode
       FROM mgmt_targets
      WHERE target_guid = tgt.target_guid;

     l_timestamp := mgmt_global.sysdate_tzrgn(l_target_tzrgn);
     change_master_agent(tgt.target_guid, v_agent_guid, l_timestamp,
                         TRUE, l_new_master_guid);

     -- if there are no masters, then select another down target as the
     -- master so that our agent does get deleted; note that when a 
     -- agent is deleted, all the targets are also automatically deleted
     IF (l_new_master_guid IS NULL) THEN
        IF (EMDW_LOG.p_is_debug_set) THEN
           EMDW_LOG.debug('Did not find a up master,' ||
                                'finding a non-up master', MODULE_NAME);
        END IF;

       BEGIN
         SELECT agent_guid INTO l_new_master_guid
           FROM (SELECT agent_guid
                   FROM mgmt_target_agent_assoc
                  WHERE target_guid = tgt.target_guid
                    AND agent_guid <> v_agent_guid)
           WHERE ROWNUM = 1;

         IF (EMDW_LOG.p_is_debug_set) THEN
            EMDW_LOG.debug('Picking a new master', MODULE_NAME);
         END IF;

         set_new_master_agent(tgt.target_guid, l_new_master_guid, l_timestamp);
       EXCEPTION
          WHEN NO_DATA_FOUND THEN
          -- this is the last agent for the multi-agent target
          NULL;
       END;
     END IF; -- new_master_guid IS NULL

   END LOOP;

  IF (EMDW_LOG.p_is_info_set)THEN
     EMDW_LOG.info('process_agent_deletion:Exit', MODULE_NAME);
  END IF;
END process_agent_deletion;

-- Select a new master agent or instance if needed when an instance 
-- is getting deleted
PROCEDURE process_instance_deletion(v_instance_target_guid IN RAW)
IS
  l_new_master_guid    MGMT_TARGETS.target_guid%TYPE;
  l_target_tzrgn  MGMT_TARGETS.timezone_region%TYPE;
  l_timestamp     DATE;
  l_mon_mode      MGMT_TARGETS.monitoring_mode%TYPE;
  l_up_inst_count     NUMBER;
BEGIN
   IF (EMDW_LOG.p_is_info_set) THEN
     EMDW_LOG.info('process_instance_deletion:Entry v_instance_target_guid = ' || 
                   v_instance_target_guid, MODULE_NAME);
   END IF;

   -- Check the given instance_target_guid is master instance.
   -- In 10.2.0.5, we do not have master instance_target_guid in mgmt_master_agent
   -- so that we need to join tables for checking it. In case of deletion of non 
   -- master instance, we do not have to change master agent.

   -- for each target for which this is a master instance 
   FOR tgt IN (SELECT ma.target_guid, ma.agent_guid
                 FROM mgmt_target_assocs ta,
                      mgmt_master_agent ma, 
                      mgmt_targets t1, 
                      mgmt_targets t2
                WHERE ta.source_target_guid = ma.target_guid
                  AND ta.assoc_target_guid = v_instance_target_guid
                  AND ta.assoc_guid = MGMT_ASSOC.g_contains_guid
                  AND t2.emd_url = t1.emd_url
                  AND t1.target_guid = v_instance_target_guid 
                  AND t2.target_guid = ta.source_target_guid
                  AND ma.end_timestamp IS NULL)
   LOOP
     SELECT timezone_region
       INTO l_target_tzrgn
       FROM mgmt_targets
      WHERE target_guid = tgt.target_guid;

     IF (EMDW_LOG.p_is_debug_set) THEN
         EMDW_LOG.debug('In process_instance_deletion:', MODULE_NAME);
     END IF;

     l_timestamp := mgmt_global.sysdate_tzrgn(l_target_tzrgn);

     change_master_agent(tgt.target_guid, tgt.agent_guid, l_timestamp,
                       TRUE, l_new_master_guid);  
       
     IF (l_new_master_guid is not null) THEN 
        IF (EMDW_LOG.p_is_debug_set) THEN
           EMDW_LOG.debug('found a new master agent:' || l_new_master_guid,  MODULE_NAME);
        END IF;
     ELSE
        -- If there is no good candidate for master, select another down target as master
       IF (EMDW_LOG.p_is_debug_set) THEN
          EMDW_LOG.debug('Did not find a up master,' ||
                               'finding a non-up master', MODULE_NAME);
       END IF;

       BEGIN
         SELECT agent_guid INTO l_new_master_guid
           FROM (SELECT agent_guid
                   FROM mgmt_target_agent_assoc
                  WHERE target_guid = tgt.target_guid
                    AND agent_guid <> tgt.agent_guid)
           WHERE ROWNUM = 1;

         IF (EMDW_LOG.p_is_debug_set) THEN
            EMDW_LOG.debug('Picking a new master', MODULE_NAME);
         END IF;

         set_new_master_agent(tgt.target_guid, l_new_master_guid, l_timestamp);
       EXCEPTION
          WHEN NO_DATA_FOUND THEN
          -- this is the last agent for the multi-agent target
          NULL;
       END;
     END IF; 
   END LOOP;

  IF (EMDW_LOG.p_is_info_set)THEN
     EMDW_LOG.info('process_instance_deletion:Exit', MODULE_NAME);
  END IF;
END process_instance_deletion;


-- Adds a master changed callback
PROCEDURE ADD_MASTER_CHANGED_CALLBACK(v_target_name IN VARCHAR2,
                                      v_target_type IN VARCHAR2,
                                      v_callback_name IN VARCHAR2)
IS
l_error_message VARCHAR2(1000);
BEGIN
  IF (EMDW_LOG.p_is_info_set)THEN
     EMDW_LOG.info('add_master_changed_callback:Entry v_target_name = ' || v_target_name ||
                   ' v_target_type = '   || v_target_type ||
                   ' v_callback_name = ' || v_callback_name, MODULE_NAME);
  END IF;
  
  EM_CHECK.check_not_null(v_callback_name,'v_callback_name');

  IF NOT EM_CHECK.is_valid_signature(v_callback_name,
                    mgmt_short_string_array('VARCHAR2','VARCHAR2','VARCHAR2'),
                    l_error_message)
  THEN
    IF (EMDW_LOG.p_is_info_set)THEN
       EMDW_LOG.info('add_master_changed_callback:Exit Invalid callback name or invalid parameters', MODULE_NAME);
    END IF;

    raise_application_error(MGMT_GLOBAL.INVALID_PARAMS_ERR,l_error_message) ;
  END IF ;


  INSERT INTO mgmt_master_changed_callback
     (target_name, target_type, callback_name)
  VALUES
     (NVL(v_target_name, ' '), v_target_type, v_callback_name);

  IF (EMDW_LOG.p_is_info_set)THEN
     EMDW_LOG.info('add_master_changed_callback:Exit', MODULE_NAME);
  END IF;
END;

-- Deletes a master changed callback
PROCEDURE DELETE_MASTER_CHANGED_CALLBACK(v_target_name IN VARCHAR2,
                                         v_target_type IN VARCHAR2,
                                         v_callback_name IN VARCHAR2
                                                          DEFAULT NULL)
IS
BEGIN
  IF (EMDW_LOG.p_is_info_set)THEN
     EMDW_LOG.info('delete_master_changed_callback:Entry v_target_name = ' || v_target_name ||
                   ' v_target_type = '   || v_target_type ||
                   ' v_callback_name = ' || v_callback_name, MODULE_NAME);
  END IF;

  DELETE mgmt_master_changed_callback
   WHERE target_name = NVL(v_target_name, ' ')
     AND target_type = v_target_type
     AND callback_name = NVL(v_callback_name, callback_name);

  IF (EMDW_LOG.p_is_info_set)THEN
     EMDW_LOG.info('delete_master_changed_callback:Exit', MODULE_NAME);
  END IF;
END;


-- Execute master changed callbacks
PROCEDURE EXEC_MASTER_CHANGED_CALLBACKS(v_target_guid IN RAW)
IS
  l_target_name mgmt_targets.target_name%TYPE;
  l_target_type mgmt_targets.target_type%TYPE;
  l_emd_url     mgmt_targets.emd_url%TYPE;
BEGIN
  IF (EMDW_LOG.p_is_info_set)THEN
     EMDW_LOG.info('exec_master_changed_callbacks:Entry v_target_guid = ' || v_target_guid, MODULE_NAME);
  END IF;

  IF (EMDW_LOG.p_is_debug_set) THEN
     EMDW_LOG.debug('In exec_master_changed_callbacks', MODULE_NAME); 
  END IF;

  SELECT target_name, target_type, emd_url
    INTO l_target_name, l_target_type, l_emd_url
    FROM mgmt_targets
   WHERE target_guid = v_target_guid;

  FOR callback IN (SELECT callback_name 
                    FROM  mgmt_master_changed_callback
                   WHERE  (target_name = l_target_name OR
                           target_name = ' ')
                    AND   target_type = l_target_type)
  LOOP
    BEGIN
      EXECUTE IMMEDIATE 
        'BEGIN ' || DBMS_ASSERT.SQL_OBJECT_NAME(callback.callback_name) || '(:1, :2, :3); END; ' USING
         l_target_name, l_target_type, l_emd_url;
    EXCEPTION
      -- log the errors
      WHEN OTHERS THEN
          mgmt_log.log_error(v_module_name_in => 'MASTER_AGENT',
                             v_error_code_in  =>  0,
                             v_error_msg_in   => 'exec_master_changed_callbacks: callback: '||callback.callback_name||
                                                 'target_guid: '||v_target_guid||
                                                 'Error - '||SUBSTR(SQLERRM, 1, 2000));
    END;
  END LOOP;

  IF (EMDW_LOG.p_is_info_set)THEN
     EMDW_LOG.info('exec_master_changed_callbacks:Exit', MODULE_NAME);
  END IF;
END;

PROCEDURE INSTANCE_PRE_DEL_CALLBACK(v_target_name IN VARCHAR2,
                                    v_target_type IN VARCHAR2,
                                    v_target_guid IN RAW)
IS
BEGIN

  IF (EMDW_LOG.p_is_info_set)THEN
     emdw_log.debug('instance_pre_del_callback:Enter. Target ('
                   || v_target_name || ', '
                   || v_target_type || ')', MODULE_NAME);
  END IF;

  process_instance_deletion(v_target_guid);

  IF (EMDW_LOG.p_is_info_set)THEN
     emdw_log.debug('instance_pre_del_callback:Exit', MODULE_NAME);
  END IF;

END;

-- Check if the given target is blacked out
FUNCTION IS_TARGET_IN_BLACKOUT(v_target_guid IN RAW) 
  RETURN BOOLEAN
IS
  l_target_guid       MGMT_TARGETS.target_guid%TYPE;
  l_curr_status       MGMT_CURRENT_AVAILABILITY.current_status%TYPE;
BEGIN
  IF (EMDW_LOG.p_is_info_set)THEN
     EMDW_LOG.info('is_target_in_blackout:Entry v_target_guid = ' || v_target_guid, MODULE_NAME);
  END IF;

   -- Check the blackout target list to see if this target is 
   -- being blacked out
   BEGIN
     SELECT guid
       INTO l_target_guid
       FROM TABLE(CAST(p_blacked_out_targets AS MGMT_GUID_ARRAY)) bl
      WHERE bl.guid = v_target_guid;

     IF (EMDW_LOG.p_is_debug_set) THEN
        EMDW_LOG.debug(v_target_guid || ' is found in blackout list', MODULE_NAME);
     END IF;

     IF (EMDW_LOG.p_is_info_set)THEN
        EMDW_LOG.info('is_target_in_blackout:Exit return value = TRUE', MODULE_NAME);
     END IF;

     RETURN TRUE;
   EXCEPTION
     WHEN NO_DATA_FOUND THEN
       NULL;
   END;

   -- Check the current status of the target
   SELECT current_status 
     INTO l_curr_status
     FROM mgmt_current_availability
    WHERE target_guid = v_target_guid;

   IF (l_curr_status = mgmt_global.G_STATUS_BLACKOUT) THEN

     IF (EMDW_LOG.p_is_info_set)THEN
        EMDW_LOG.info('is_target_in_blackout:Exit return value = TRUE', MODULE_NAME);
     END IF;

     return TRUE;
   END IF;


   IF (EMDW_LOG.p_is_info_set)THEN
      EMDW_LOG.info('is_target_in_blackout:Exit return value = FALSE', MODULE_NAME);
   END IF;

   RETURN FALSE;

END IS_TARGET_IN_BLACKOUT;

-- Selects a new master agent for the specified target
PROCEDURE select_master_agent(v_target_guid IN RAW,
                              v_timestamp IN DATE)
IS
  l_master_agent_guid   MGMT_TARGETS.target_guid%TYPE;        
  l_master_agent_status MGMT_CURRENT_AVAILABILITY.current_status%TYPE;
  l_instance_status     MGMT_CURRENT_AVAILABILITY.current_status%TYPE;
BEGIN
  IF (EMDW_LOG.p_is_info_set)THEN
     EMDW_LOG.info('select_master_agent:Entry v_target_guid = ' || v_target_guid ||
                   ' v_timestamp = ' || v_timestamp, MODULE_NAME);
  END IF;
  -- Get the current master agent status
  SELECT av.current_status, ma.agent_guid
    INTO l_master_agent_status, l_master_agent_guid
    FROM mgmt_master_agent ma, mgmt_current_availability av
   WHERE ma.target_guid = v_target_guid
     AND ma.end_timestamp IS NULL
     AND ma.agent_guid = av.target_guid;

  -- If the current master is not up, then change the master
  IF (l_master_agent_status <> mgmt_global.G_STATUS_UP) THEN
     process_oms_med_target(v_target_guid,
                            l_master_agent_guid,
                            l_master_agent_status,
                            v_timestamp,
                            l_master_agent_guid,
                            l_master_agent_status);

     IF (EMDW_LOG.p_is_info_set)THEN
        EMDW_LOG.info('select_master_agent:Exit', MODULE_NAME);
     END IF;

     IF (EMDW_LOG.p_is_info_set)THEN
        EMDW_LOG.info('select_master_agent:Exit master_agent_status <> ' ||
                       mgmt_global.G_STATUS_UP, MODULE_NAME);
     END IF;

     RETURN;
  END IF;

  -- If this target has a cluster member on this agent, then check the
  -- status of the instance
  BEGIN
    SELECT DECODE (count(*), 0, MGMT_GLOBAL.G_STATUS_DOWN, MGMT_GLOBAL.G_STATUS_UP)
      INTO l_instance_status
      FROM mgmt_target_assocs tm,
           mgmt_targets  ct,
           mgmt_type_properties p,
           mgmt_targets t1,
           mgmt_targets t2,
           mgmt_current_availability av
     WHERE tm.source_target_guid = v_target_guid
       AND tm.assoc_guid = MGMT_ASSOC.g_contains_guid
       AND ct.target_guid = tm.source_target_guid
       AND p.target_type = ct.target_type
       AND p.property_name = MGMT_GLOBAL.G_IS_CLUSTER_PROP 
       AND tm.assoc_target_guid = t1.target_guid
       AND t1.emd_url = t2.emd_url
       AND t2.target_guid = l_master_agent_guid
       AND tm.assoc_target_guid = av.target_guid
       AND av.current_status = MGMT_GLOBAL.G_STATUS_UP;
  EXCEPTION
    WHEN NO_DATA_FOUND THEN
       -- this target does not have cluster members
       NULL;
  END;

  -- Passing l_instance_status as master-agent current status to force a change  
  IF (l_instance_status <> mgmt_global.G_STATUS_UP) THEN
     process_oms_med_target(v_target_guid,
                            l_master_agent_guid,
                            l_instance_status,
                            v_timestamp,
                            l_master_agent_guid,
                            l_instance_status);
  END IF;

  IF (EMDW_LOG.p_is_info_set)THEN
     EMDW_LOG.info('select_master_agent:Exit', MODULE_NAME);
  END IF;
EXCEPTION
  WHEN OTHERS THEN
          mgmt_log.log_error(v_module_name_in => 'MASTER_AGENT',
                             v_error_code_in  =>  0,
                             v_error_msg_in   =>  'select_master_agent: v_target_guid = ' || v_target_guid ||
                                                  ' v_timestamp = ' || v_timestamp||
                                                  'Error - '||SUBSTR(SQLERRM, 1, 2000));
    IF (EMDW_LOG.p_is_error_set)THEN
      EMDW_LOG.error('select_master_agent:Exception for Target guid '||v_target_guid || SQLERRM , MODULE_NAME);
    END IF;
END select_master_agent;

-- Sets the blacked out target list
PROCEDURE set_blackout_target_list(v_target_list IN MGMT_GUID_ARRAY)
IS
BEGIN
  IF (EMDW_LOG.p_is_info_set)THEN
     EMDW_LOG.info('set_blackout_target_list:Entry', MODULE_NAME);
  END IF;

   p_blacked_out_targets := v_target_list;

  IF (EMDW_LOG.p_is_info_set)THEN
     EMDW_LOG.info('set_blackout_target_list:Exit', MODULE_NAME);
  END IF;
END set_blackout_target_list;

-- Clears the list of blacked out target list
PROCEDURE clear_blackout_target_list
IS
  l_target_guid mgmt_targets.target_guid%TYPE;
  l_curr_status       MGMT_CURRENT_AVAILABILITY.current_status%TYPE;
BEGIN
  IF (EMDW_LOG.p_is_info_set)THEN
     EMDW_LOG.info('clear_blackout_target_list:Entry', MODULE_NAME);
  END IF;

  IF (p_blacked_out_targets IS NULL) THEN

    IF (EMDW_LOG.p_is_info_set)THEN
       EMDW_LOG.info('clear_blackout_target_list:Exit No blacked out target', MODULE_NAME);
    END IF;

    RETURN;
  END IF;

  -- Select a new master for the targets that are not
  -- not in BLACKOUT state. 
  IF (p_blacked_out_targets.COUNT > 0) THEN
    FOR idx IN 1..p_blacked_out_targets.COUNT LOOP
      l_target_guid := p_blacked_out_targets(idx).guid;

      SELECT current_status 
        INTO l_curr_status
        FROM mgmt_current_availability
       WHERE target_guid = l_target_guid;

      -- If the current status is not blackout, then find a new
      -- master agent if needed
      IF (l_curr_status <> mgmt_global.G_STATUS_BLACKOUT) THEN
        select_master_agent(l_target_guid, sysdate);
      END IF;
    END LOOP;
  END IF;

  -- Free the blackout list
  p_blacked_out_targets.delete;
  p_blacked_out_targets := NULL;

  IF (EMDW_LOG.p_is_info_set)THEN
     EMDW_LOG.info('clear_blackout_target_list:Exit', MODULE_NAME);
  END IF;
END clear_blackout_target_list;


  PROCEDURE MASTER_AGENT_HIST_PURGE(pcb_params IN OUT MGMT_PURGE_CALLBACK_PARAMS) IS

    l_purge_rowids t_rowid_list;
    l_rows_purged NUMBER := 0;
    CURSOR master_agent_hist_cursor (c_target_guid RAW,
                                     c_purge_till_time DATE) IS
      SELECT ma.ROWID
        FROM MGMT_MASTER_AGENT ma
       WHERE ma.target_guid = c_target_guid
         AND ma.end_timestamp <= c_purge_till_time;
  BEGIN
    IF (EMDW_LOG.p_is_info_set)THEN
        EMDW_LOG.info('master_agent_hist_purge:Entry', MODULE_NAME);
    END IF;

    OPEN master_agent_hist_cursor(pcb_params.target_guid, pcb_params.purge_upto_time);
    LOOP
      -- Bulk collect row ids
      FETCH master_agent_hist_cursor BULK COLLECT INTO l_purge_rowids LIMIT 500;

      -- Exit when no more row ids fetched
      EXIT WHEN l_purge_rowids.COUNT <= 0;

      -- BULK delete
      FORALL i IN l_purge_rowids.FIRST..l_purge_rowids.LAST
        DELETE FROM MGMT_MASTER_AGENT WHERE ROWID = l_purge_rowids(i);

      COMMIT;

      l_rows_purged := l_rows_purged + l_purge_rowids.COUNT;

      l_purge_rowids.DELETE;

    END LOOP;

    -- Close the cursor if open
    IF (master_agent_hist_cursor%ISOPEN) THEN
      CLOSE master_agent_hist_cursor;
    END IF;

    pcb_params.callback_result := 0;
    pcb_params.rows_processed  := l_rows_purged;
    pcb_params.error_code      := 0;
    pcb_params.error_msg       := NULL;


    IF (EMDW_LOG.p_is_info_set)THEN
       EMDW_LOG.info('master_agent_hist_purge:Exit', MODULE_NAME);
    END IF;

  END MASTER_AGENT_HIST_PURGE;

  -- This procedure handles the case when agent
  -- port number is changed. 
  PROCEDURE handle_agent_port_change
  (
    p_old_agent_guid IN mgmt_targets.target_guid%TYPE,
    p_new_agent_guid IN mgmt_targets.target_guid%TYPE
  )
  IS
    l_target_tzrgn  mgmt_targets.timezone_region%TYPE;
    l_timestamp     DATE;
    l_mon_mode      mgmt_targets.monitoring_mode%TYPE;
  BEGIN
    IF (EMDW_LOG.p_is_info_set)THEN
       EMDW_LOG.info('handle_agent_port_change:Entry p_old_agent_guid = ' || p_old_agent_guid ||
                     ' p_new_agent_guid = ' || p_new_agent_guid,  MODULE_NAME);
    END IF;
    
    DELETE mgmt_target_agent_assoc 
     WHERE agent_guid = p_old_agent_guid;

    -- for each target for which this is a master agent 
   UPDATE mgmt_master_agent
           SET agent_guid = p_new_agent_guid
    WHERE agent_guid = p_old_agent_guid
           AND end_timestamp IS NULL; 

    IF (EMDW_LOG.p_is_info_set)THEN
       EMDW_LOG.info('handle_agent_port_change:Exit',  MODULE_NAME);
    END IF;

  END handle_agent_port_change;

END em_master_agent;

/

show errors

