Rem
Rem $Header: blackout_engine_pkgdef.sql 27-may-2008.13:52:31 rrawat Exp $
Rem
Rem blackout_engine_pkgdef.sql
Rem
Rem Copyright (c) 2002, 2005, Oracle. All rights reserved.  
Rem
Rem    NAME
Rem      blackout_engine_pkgdef.sql - Internal blackout routines
Rem
Rem    DESCRIPTION
Rem       This contains internal blackout routines called by the sdk
Rem       and UI
Rem
Rem    NOTES
Rem
Rem    MODIFIED 
Rem    rawat   05/23/08  - Bug-7114090
Rem    rrawat  07/20/07 - Bug-5497731
Rem    rrawat  07/20/07 - Bug-5497731
Rem    skini   09/21/05 - Make log_error public for access to triggers 
Rem    skini   09/01/05 - Add purge policy on MGMT_BLACKOUTS 
Rem    dcawley 08/15/05 - Change type for callback 
Rem    pmodi   07/21/05 - Change submit_discard_state_job signature to accpet 
Rem                       job name prefix 
Rem    skini   07/18/05 - Add error code for discard state submit error 
Rem    pmodi   07/08/05 - Bug:4399250 - Use NLS message id for blackout 
Rem    mfidanb 06/24/05 - submit discard_state_job must support proxy targets 
Rem    ssukava 06/03/05 - made api to_blackout_time public
Rem    skini   04/07/05 - Add do_update flag to update_blackout_status 
Rem    skini   03/21/05 - Handle internal severity errors 
Rem    skini   03/06/05 - Add API to end agent-side blackout 
Rem    dcawley 01/26/05 - Change privilege callback 
Rem    rzazuet 01/12/05 - Add host/group change callbacks 
Rem    snakai  10/31/04 - make submit_discard_state_job public 
Rem    kmanick 09/20/04 - add user_model callbacks
Rem    skini   09/02/04 - Add start_blackout_window,end_blackout_window 
Rem    rzazuet 05/18/04 - Add create_immediate_blackout 
Rem    jsadras 03/17/04 - Added handle_user_deletion and get_blackout_count 
Rem    jsadras 03/11/04 - 
Rem    skini   12/12/03 - Fix bug 3300351 
Rem    streddy 09/25/03 - Added status to get_blackout_details 
Rem    rpinnam 09/08/03 - Add purge policy to purge blackout target state 
Rem    skini   08/28/03 - Add duration source 
Rem    rpatti  02/26/03 - cleanup on target delete
Rem    skini   01/17/03 - Allow agent-side bos to be deleted
Rem    skini   01/16/03 - Add overloaded submit_discard_state_job
Rem    skini   12/26/02 - Change in agent APIs for stop and discardState
Rem    skini   12/05/02 - Generate messages for blackout severities
Rem    skini   12/03/02 - Introduce modificationFailed, and 
Rem                        partialModifyFailed states
Rem    skini   11/22/02 - Add to_agent_time
Rem    skini   11/26/02 - Change membership API
Rem    skini   11/15/02 - Fix REM_PENDING for edit
Rem    skini   11/14/02 - Use changed signature of revoke_priv callback
Rem    skini   09/20/02 - Use actual stat time and duration
Rem    skini   09/18/02 - blackout windows
Rem    skini   08/26/02 - Introduce START_PROCESSING
Rem    skini   08/26/02 - 
Rem    skini   08/23/02 - Finish up edit
Rem    skini   08/20/02 - New blackout states
Rem    skini   08/13/02 - Continue development, finish edit
Rem    skini   08/02/02 - 
Rem    skini   08/01/02 - 
Rem    skini   07/31/02 - Implement stop
Rem    skini   07/24/02 - Change constants
Rem    skini   07/23/02 - Logging, implement stop
Rem    skini   07/18/02 - Blackout APIs
Rem    skini   07/17/02 - Continue blackout development
Rem    skini   06/20/02 - skini_blackouts
Rem    skini   06/17/02 - Created
Rem

CREATE OR REPLACE PACKAGE MGMT_BLACKOUT_ENGINE AS

MODULE_NAME constant VARCHAR2(11) := 'EM.Blackout';

TYPE STRING_CURSOR IS REF CURSOR;
TYPE TARGET_CURSOR IS REF CURSOR;
TYPE BLACKOUT_CURSOR IS REF CURSOR;

-- Possible values for "duration source"
DURATION_SOURCE_ENDTIME CONSTANT NUMBER(1) := 1;
DURATION_SOURCE_DURATION CONSTANT NUMBER(1) := 2;
DURATION_SOURCE_INDEFINITE CONSTANT NUMBER(1) := 3;

-- Possible internal error codes
BLACKOUT_NOT_EXIST constant NUMBER(2) := 1;
BLACKOUT_NOT_STOPPED constant NUMBER(2) := 2;
DUPLICATE_RECORDS_INSERT constant NUMBER(2) := 3;
UNEXPECTED_STATE constant NUMBER(2) := 4;
START_JOB_NOT_FOUND constant NUMBER(2) := 5;
WINDOWS_NOT_FOUND constant NUMBER(2) := 6;
--     Could not submit a discard state job
DISCARD_STATE_ERROR constant NUMBER(2) := 7;

-- Constants for target blackout status
-- (in the mgmt_blackout_flat_targets and
--    mgmt_blackout_state tables)
IN_BLACKOUT constant NUMBER(2) := 1;
NOT_IN_BLACKOUT constant NUMBER(2) := 0;

-- Constants for blackout target edit states (used by the
-- modify operation only)
-- The target has not been edited
NO_EDITS constant NUMBER(1) := 0;
-- The target needs to be added
ADD_PENDING constant NUMBER(1) := 1;  
-- The target needs to be removed
REM_PENDING constant NUMBER(1) := 2;
-- The target and all its members need to be removed
REM_REC_PENDING constant NUMBER(1) := 3;  
-- "Include members" has been added
INCLUDE_ADD_PENDING constant NUMBER(1) := 4;  
-- "Include members" has been removed
INCLUDE_REM_PENDING constant NUMBER(1) := 5; 

-- Constants for blackout states
BLK_STATE_START_PENDING constant NUMBER(1) := 0;
BLK_STATE_START_PROCESSING constant NUMBER(1) := 1;
BLK_STATE_START_PARTIAL constant NUMBER(1) := 2;
BLK_STATE_START_FAILED constant NUMBER(1) := 3;
BLK_STATE_STARTED constant NUMBER(1) := 4;
BLK_STATE_STOP_PENDING constant NUMBER(1) := 5;
-- The stop of the blackout did not succeed on any target.
-- Hence, all targets are blacked out
BLK_STATE_STOP_FAILED constant NUMBER(1) := 6;
-- The stop of the blackout succeeded on some targets
-- but not on others. Hence, some targets are blacked
-- out but not others
BLK_STATE_STOP_PARTIAL constant NUMBER(1) := 7;
-- Removal of targets during edit failed for all
-- targets. Hence, all targets are still blacked out
BLK_STATE_EDIT_FAILED constant NUMBER(1) := 8;
-- The edit succeeded on some targets but not on
-- others. Hence, some targets are blacked
-- out but not others
BLK_STATE_EDIT_PARTIAL constant NUMBER(1) := 9;
BLK_STATE_STOPPED constant NUMBER(2) := 10;
BLK_STATE_ENDED constant NUMBER(2) := 11;
BLK_STATE_END_PARTIAL constant NUMBER(2) := 12;
BLK_STATE_MODIFY_PENDING constant NUMBER(2) := 13;

-- Constants for blackout processing "job" states
BLK_JSTATE_START_PROCESSING constant NUMBER(1) := 0;
BLK_JSTATE_START_FAILED constant NUMBER(1) := 1;
BLK_JSTATE_STARTED constant NUMBER(1) := 2;
BLK_JSTATE_STOP_PROCESSING constant NUMBER(1) := 3;
BLK_JSTATE_STOP_FAILED constant NUMBER(1) := 4;
BLK_JSTATE_STOPPED constant NUMBER(1) := 5;

-- Constants for edit modes: the second argument to
-- update_blackout_status()
BLK_MODE_NONE constant NUMBER(1) := 1;
BLK_MODE_STOP constant NUMBER(1) := 2;
BLK_MODE_EDIT constant NUMBER(1) := 3;

-- Constants for frequency codes
ONE_TIME_FREQUENCY_CODE constant NUMBER(1) := MGMT_JOB_ENGINE.ONE_TIME_FREQUENCY_CODE;
INTERVAL_FREQUENCY_CODE constant NUMBER(1) := MGMT_JOB_ENGINE.INTERVAL_FREQUENCY_CODE;
DAILY_FREQUENCY_CODE constant NUMBER(1) := MGMT_JOB_ENGINE.DAILY_FREQUENCY_CODE;
WEEK_FREQUENCY_CODE constant NUMBER(1) := MGMT_JOB_ENGINE.WEEK_FREQUENCY_CODE;
MONTH_FREQUENCY_CODE constant NUMBER(1) := MGMT_JOB_ENGINE.MONTH_FREQUENCY_CODE;
YEAR_FREQUENCY_CODE constant NUMBER(1) := MGMT_JOB_ENGINE.YEAR_FREQUENCY_CODE;

-- Timezone constants
TIMEZONE_REPOSITORY constant NUMBER(1) := MGMT_JOB_ENGINE.TIMEZONE_REPOSITORY;
TIMEZONE_SPECIFIED constant NUMBER(1) := MGMT_JOB_ENGINE.TIMEZONE_SPECIFIED;
TIMEZONE_RGN_SPECIFIED constant NUMBER(1) := MGMT_JOB_ENGINE.TIMEZONE_RGN_SPECIFIED;

-- Messages for BO start and end severities
BO_START_MSG constant VARCHAR2(32) := 'Blackout started on this target';
BO_END_MSG constant VARCHAR2(32) := 'Blackout ended on this target';

BO_START_MSG_NLSID CONSTANT VARCHAR2(32) := 'EM__SYS__BO_START' ;
BO_END_MSG_NLSID CONSTANT VARCHAR2(32) := 'EM__SYS__BO_END' ;

-- The format used to convert timestamps to strings in the 
-- blackout subsystem
TIMESTAMP_FORMAT constant VARCHAR2(26) := 'mm/dd/yyyy hh:mi:ss pm';

-- Default sleep times in secs
BLK_STOP_SLEEP_TIME constant NUMBER(2) := 15;
BLK_STOP_TOTAL_SLEEP_TIME constant NUMBER(3) := 300;

-- Blackout Start/Stop Message and NLS ID
BLK_START_MSG CONSTANT VARCHAR2(1000) := 'Blackout started by {0}. Reason: {1}. Name: {2}.' ;
BLK_STOP_MSG CONSTANT VARCHAR2(1000) := 'Blackout ({0}) ended.' ;

BLK_START_MSG_NLSID CONSTANT VARCHAR2(128) := 'EM__SYS__BLK_START';
BLK_STOP_MSG_NLSID CONSTANT VARCHAR2(128) := 'EM__SYS__BLK_STOP';

-- Create a new blackout
FUNCTION create_blackout(p_blackout_name IN VARCHAR2,
                         p_description IN VARCHAR2,
                         p_schedule IN MGMT_BLACKOUT_SCHEDULE_RECORD,
                         p_reason IN VARCHAR2,
                         p_job_flag IN NUMBER,
                         p_targets IN MGMT_BLACKOUT_TARGET_LIST)
     RETURN RAW;

-- Create an immediate blackout with indefinite duration
FUNCTION create_immediate_blackout(p_blackout_name IN VARCHAR2,
                                   p_description IN VARCHAR2,
                                   p_reason IN VARCHAR2,
                                   p_job_flag IN NUMBER,
                                   p_targets IN MGMT_BLACKOUT_TARGET_LIST,
                                   p_caller_execution_id RAW DEFAULT NULL)
     RETURN RAW;

-- Process all blackouts (for the given target only) whose
-- start time is before p_start_time that haven't already
-- started
PROCEDURE start_blackouts_on_target(p_target_guid RAW,
                                    p_start_time DATE);

-- Process all blackouts (for the given target only) whose
-- end time is before p_end_time that haven't already
-- ended
PROCEDURE end_blackouts_on_target(p_target_guid RAW,
                                  p_end_time DATE);

-- Return all blackout windows for the specified targets,
-- along with the "current" query time
FUNCTION get_blackout_windows(p_target_guid RAW,
                              p_query_time_out OUT DATE)
    RETURN BLACKOUT_CURSOR;

-- Generate a blackout guid, given the blackout name,
-- creator and created-thru (emd url). 
FUNCTION generate_blackout_guid(p_blackout_name VARCHAR2,
                                p_creator VARCHAR2,
                                p_created_thru VARCHAR2) RETURN RAW;

-- Mark the blackout with the specified guid as started in the
-- repository. This is always called by a job (for now, anyway),
-- so takes in a job id and execution_id. This procedure flattens
-- out all the targets associated with the blackout and sets a vector
-- parameter that the job will eventually iterate over. This parameter
-- contains a list of emd's that the next set of steps will 
-- contact in order to start the blackout on those emd's.
PROCEDURE process_start_blackout(p_blackout_guid RAW,
                                 p_job_id RAW,
                                 p_execution_id RAW,
                                 p_job_param_name VARCHAR2);


-- End the specified blackout (its duration has expired)
PROCEDURE process_end_blackout(p_blackout_guid RAW);

-- Process the stop of the blackout on all targets with the
-- specific emd_url. Called from the stopBlackout job
-- p_state is the result of the stop operation; set to
-- BLK_JSTATE_STOPPED or BLK_JSTATE_STOP_FAILED.
-- p_stop_ts is the agent time corresponding to the stop.
PROCEDURE process_stop_blackout(p_blackout_guid RAW,
                                p_emd_url VARCHAR2,
                                p_state NUMBER,
                                p_message VARCHAR2,
                                p_stop_ts DATE);
 
-- Stop the specified blackout (the blackout has not yet expired)
PROCEDURE stop_blackout(p_blackout_guid RAW);

-- Internal procedure to stop an agent-side blackout
PROCEDURE stop_agent_side_blackout(p_blackout_guid RAW,
                                   p_end_time DATE DEFAULT NULL);

-- Stop the specified blackout (the blackout has not yet expired)
-- It Waits for the blackout to get out of start processing state before it stops it.
-- The stop of the blackout may still happen asynchronously.
-- p_sleep_time is the time to sleep between status checks (in secs).
-- p_total_sleep_time is the total sleep time in secs.
PROCEDURE stop_blackout(p_blackout_guid RAW,
                        p_sleep_time NUMBER,
                        p_total_sleep_time NUMBER);

-- Delete the specified blackout (only if it has expired)
PROCEDURE delete_blackout(p_blackout_guid RAW);

-- Start a blackout window on a target. Called from the
-- blackout trigger code to process windows when jobs run
-- out of order    
PROCEDURE start_blackout_window(p_blackout_guid RAW,
                                p_occurrence_number NUMBER,
                                p_target_guid RAW);

-- End a blackout window on a target. Called from the
-- blackout trigger code to process windows when jobs run
-- out of order   
PROCEDURE end_blackout_window(p_blackout_guid RAW,
                              p_occurrence_number NUMBER,
                              p_target_guid RAW,
                              p_end_ts DATE);

-- Update job status for all targets on the specified emd
-- for the specified blackout, during start processing of
-- the blackout
-- p_state is the state to update.
PROCEDURE update_start_job_status(p_blackout_guid RAW,
                                  p_emd_url VARCHAR2,
                                  p_state NUMBER,
                                  p_message VARCHAR2);

-- Update job status for all targets on the specified emd
-- for the specified blackout, during stop processing of
-- the blackout
-- p_state is the state to update.
-- p_stop_ts is the agent time corresponding to the stop.
PROCEDURE update_stop_job_status(p_blackout_guid RAW,
                                 p_emd_url VARCHAR2,
                                 p_state NUMBER,
                                 p_message VARCHAR2,
                                 p_stop_ts DATE);

-- Set the status of the blackout from its component targets
PROCEDURE set_blackout_status(p_blackout_guid RAW,
                              p_mode NUMBER);

-- Obtain the details for the specified blackout and emd_url
-- p_blackout_details_out is a cursor containing 
--   (blackout_name, created_by, blackout_status).
-- p_end_time_out will contain the blackout end time (for this occurrence)
-- p_targets_out is a cursor consisting of three columns:
-- (target_name, target_type, target_guid) for the 
-- blackout targets on the specified emd_url. These are the
-- current targets for the blackout
PROCEDURE get_blackout_details(p_blackout_guid RAW,
                               p_emd_url VARCHAR2,
                               p_blackout_details_out OUT BLACKOUT_CURSOR,
                               p_start_time_out OUT DATE,
                               p_end_time_out OUT DATE,
                               p_targets_out OUT TARGET_CURSOR,
                               p_blackout_job_state IN NUMBER DEFAULT -1);


-- Attempt to get the "reason id" for the blackout from the 
-- reason table. If the reason does not exist, throw an
-- "invalid reason" exception
PROCEDURE set_reason(p_blackout_guid RAW, p_reason VARCHAR2);

-- Edit the specified blackout. If any of the parameters is set
-- to null, the specific attribute of the blackout is not affected.
PROCEDURE edit_blackout(p_blackout_guid IN RAW,
                        p_description IN VARCHAR2,
                        p_schedule IN MGMT_BLACKOUT_SCHEDULE_RECORD,
                        p_reason IN VARCHAR2,
                        p_job_flag IN NUMBER,
                        p_targets_add IN MGMT_BLACKOUT_TARGET_LIST,
                        p_targets_remove IN MGMT_BLACKOUT_TARGET_LIST);

-- Cleanup targets after a stop operation, and update the blackout status
PROCEDURE cleanup_after_edit(p_blackout_guid IN RAW);

-- Called when a group's or host's membership changes (ie from MODIFY_GROUP
-- or from delete_target)
PROCEDURE handle_membership_change(p_container_guid RAW);

-- Callback for the user model when privileges are granted to a target
PROCEDURE grant_priv_callback(p_user_names IN SMP_EMD_STRING_ARRAY,
                              p_priv_names IN SMP_EMD_STRING_ARRAY,
                              p_target_guids IN MGMT_USER_GUID_ARRAY,
                              p_type IN NUMBER);

-- Callback for the user model when privileges are revoked from a target
PROCEDURE revoke_priv_callback(p_user_names IN SMP_EMD_STRING_ARRAY,
                               p_priv_names IN SMP_EMD_STRING_ARRAY,
                               p_target_guids IN MGMT_USER_GUID_ARRAY,
                               p_type IN NUMBER);

-- Check that the current user can modify the specified blackout
-- If p_check_agent_bo is false, then a check to see whether
-- the blackout is an agent-side blackout is not made. In general,
-- most operations such as edit and stop cannot be applied to
-- agent-side blackouts. If p_check_agent_bo is true (the default),
-- then an error is raised if the blackout is an agent-side blackout.
PROCEDURE check_modify_privs(p_blackout_guid RAW,
                             p_check_agent_bo BOOLEAN DEFAULT TRUE);
                             
-- Return true if the target is blacked out at the specified
-- time. 
FUNCTION is_blacked_out(p_target_guid RAW,
                        p_timestamp DATE) RETURN BOOLEAN;

-- This is public for internal use only. 
-- Convert the speecified timestamp to agent time
FUNCTION to_agent_time(p_blackout_guid RAW, 
                       p_target_guid RAW,
                       p_timestamp DATE) RETURN DATE;

-- Submit a discardState job for all targets in a blackout
-- with the specified emd url. For use by triggers only
PROCEDURE submit_discard_state_job(p_blackout_guid RAW,
                                   p_emd_url VARCHAR2,
                                   p_stop_time DATE);

-- This is public for internal use only. 
--
-- PROCEDURE: clean_blk_on_tgt_del
--
-- PURPOSE:
--   Cleanup blackouts as needed whenever a target is deleted
--     The blackout tables that have target info in them are marked
--     as exceptions and will not be cleaned by the admin routines.
--     We remove the target from blackouts and  check if any of these blackouts 
--     that are started/Scheduled have no more corresponding  
--     targets in the flat targets/target details table and call 
--     stop blackout on these. If the blackout is 
--        scheduled - this will work ok.
--        in progress - this will work ok.
--
--   IN Parameters:
--     target_name_in : target name of the deleted target
--     target_type_in : target type of the deleted target
--     target_guid_in : target guid of the deleted target
--
--
--   OUT Parameters:
--     NONE
PROCEDURE clean_blk_on_tgt_del
   (target_name_in IN VARCHAR2,
    target_type_in IN VARCHAR2,
    target_guid_in IN RAW);

  -- Purge policy callback procedure for blackout state
  TYPE t_rowid_list IS TABLE OF ROWID INDEX BY BINARY_INTEGER;

PROCEDURE BLACKOUT_STATE_PURGE(pcb_params IN OUT MGMT_PURGE_CALLBACK_PARAMS);
PROCEDURE BLACKOUT_PURGE(pcb_params IN OUT MGMT_PURGE_CALLBACK_PARAMS);

--
-- PROCEDURE : get_user_blackouts
--
-- PURPOSE
--
--   this is a callback which will be registered to the user_model pkg 
--   user_model pkg will call this procedure to get a list of blackouts which the 
--   user has
--  
-- PARAMETERS
--
--  user_name_in     - name of the user
--  user_objects_out - list of blackouts will be appended to user_objects_out
--  type_in          - type of user model callback
--
-- NOTES
--
  
PROCEDURE get_user_blackouts(user_name_in      IN VARCHAR2,
                             user_objects_out  OUT MGMT_USER_OBJECTS,
                             type_in           IN NUMBER);

-- PROCEDURE : drop_user_blackouts
--
-- PURPOSE
--
--   this is a callback which will be registered to the user_model pkg 
--   user_model pkg will call this procedure to reassign the blackouts
--   owned by a user to the repository owner.
--  
-- PARAMETERS
--
--  user_name_in     - name of the user
--  type_in          - type of user model callback
--
-- NOTES
--
        
PROCEDURE drop_user_blackouts(user_name_in      IN VARCHAR2,
                              type_in           IN NUMBER);

-- PROCEDURE : reassign_user_blackouts
--
-- PURPOSE
--
--   this is a callback which will be registered to the user_model pkg 
--   user_model pkg will call this procedure to reassign the blackouts
--   owned by a user to a new user.
--  
-- PARAMETERS
--
--  user_name_in     - name of the user
--  new_user_name_in - the new user name
--  type_in          - type of user model callback
--
-- NOTES
--
        
PROCEDURE reassign_user_blackouts(user_name_in      IN VARCHAR2,
                                  new_user_name_in  IN VARCHAR2,
                                  type_in           IN NUMBER);


--
-- PROCEDURE : check_reassign_user_blackouts
--
-- PURPOSE
--
--   this is a callback which will be registered to the user_model pkg 
--   user_model pkg will call this procedure to to check if the blackouts owned 
--   by a user can be reassigned to another user
--  
-- PARAMETERS
--
--  user_name_in     - name of the user
--  new_user_name_in - the new user name
--  user_objects_out - returns the list of blackouts which are incompatable
--  type_in          - type of user model callback
--
-- NOTES
--
        
PROCEDURE check_reassign_user_blackouts(user_name_in      IN VARCHAR2,
                                        new_user_name_in  IN VARCHAR2,
                                        user_objects_out  OUT MGMT_USER_OBJECTS,
                                        type_in           IN NUMBER);


-- This is public for internal use only. 
--
-- Submit a discardState job for a set of targets
-- with the specified emd url. For use by triggers only
--
PROCEDURE submit_discard_state_job(p_emd_url VARCHAR2,
                                   p_target_names MGMT_JOB_VECTOR_PARAMS,
                                   p_target_types MGMT_JOB_VECTOR_PARAMS,
                                   p_stop_time DATE);

-- This is public for internal use only. 
--
-- Submit a discardState job for a set of targets
-- with the specified emd url. This version accepts a proxyTarget boolean
-- which is "N" by default.
--
-- Job name prefix have been added so that same code could be used by
-- different module to submit discard state job with own prefix.
-- Keep job prefix length with-in 30 character.
PROCEDURE submit_discard_state_job(p_emd_url VARCHAR2,
                                   p_target_names MGMT_JOB_VECTOR_PARAMS,
                                   p_target_types MGMT_JOB_VECTOR_PARAMS,
                                   p_stop_time DATE,
                                   p_is_proxy VARCHAR2,
                                   p_job_name_prefix VARCHAR2 DEFAULT 'BLK_DISCST_JOB');

--
-- PROCEDURE: host_change_callback
-- PURPOSE: recompute flat target list of blackouts that are in scheduled state
--          and submitted against hosts with include_members=1.
--          when a target gets added/removed to/from a host.
--
-- IN PARAMETERS: see sdk_assoc_pkgdef
--                       p_assoc_def_name VARCHAR2,
  --                     p_source_target_name VARCHAR2,
  --                     p_source_target_type VARCHAR2,
  --                     p_assoc_target_name  VARCHAR2,
  --                     p_assoc_target_type  VARCHAR2,
  --                     p_scope_target_name  VARCHAR2,
  --                     p_scope_target_type  VARCHAR2;
-- OUT PARAMETER: none
-- 
PROCEDURE host_change_callback(p_assoc_def_name VARCHAR2,
                               p_source_target_name VARCHAR2,
                               p_source_target_type VARCHAR2,
                               p_assoc_target_name  VARCHAR2,
                               p_assoc_target_type  VARCHAR2,
                               p_scope_target_name  VARCHAR2,
                               p_scope_target_type  VARCHAR2);


--
-- PROCEDURE: group_change_callback
-- PURPOSE: recompute flat target list of blackouts that are in scheduled state
--          and submitted against groups with include_members=1
--          when a target gets added/removed to/from a group.
--
-- IN PARAMETERS: see sdk_assoc_pkgdef
--                       p_assoc_def_name VARCHAR2,
  --                     p_source_target_name VARCHAR2,
  --                     p_source_target_type VARCHAR2,
  --                     p_assoc_target_name  VARCHAR2,
  --                     p_assoc_target_type  VARCHAR2,
  --                     p_scope_target_name  VARCHAR2,
  --                     p_scope_target_type  VARCHAR2;
-- OUT PARAMETER: none
-- 
PROCEDURE group_change_callback(p_assoc_def_name VARCHAR2,
                                p_source_target_name VARCHAR2,
                                p_source_target_type VARCHAR2,
                                p_assoc_target_name  VARCHAR2,
                                p_assoc_target_type  VARCHAR2,
                                p_scope_target_name  VARCHAR2,
                                p_scope_target_type  VARCHAR2);

-- Reinsert blackout severity using the max colection timestamp
FUNCTION reinsert_blackout_severity(p_blackout_guid RAW, 
                                    p_target_guid RAW,
                                    p_resp_metric_guid RAW,
                                    p_severity_code NUMBER,
                                    p_msg VARCHAR2,
                                    p_msg_nlsid VARCHAR2 DEFAULT NULL,
                                    p_msg_params VARCHAR2 DEFAULT NULL) RETURN DATE;

-- Compute and return the new status of the blackout.
FUNCTION compute_blackout_status(p_blackout_guid RAW, p_mode NUMBER,
                                 p_last_start_time IN OUT DATE,
                                 p_last_end_time IN OUT DATE) 
        RETURN NUMBER;

-- Lock all blackouts corresponding to the specified array of windows
PROCEDURE lock_windows(l_windows MGMT_BLACKOUT_WINDOW_ARRAY);

-- Return the specified time, assumed to be in repository timezone,
-- into the blackout timezone
FUNCTION to_blackout_time(p_timestamp DATE, p_timezone_info NUMBER,
                          p_tzoffset NUMBER,
                          p_tzregion VARCHAR2) RETURN DATE ;

-- Log system error log message
PROCEDURE log_error(p_error_code NUMBER, p_error_message VARCHAR2);


END MGMT_BLACKOUT_ENGINE;
/
show errors;
