Rem $Header: chronos_rollup_pkgbody.sql 01-sep-2005.16:31:44 eporter Exp $
Rem
Rem chronos_rollup_pkgbody.sql
Rem
Rem Copyright (c) 2002, 2005, Oracle. All rights reserved.  
Rem
Rem    NAME
Rem      chronos_rollup_pkgbody.sql - file for chronos rollups
Rem
Rem    DESCRIPTION
Rem      package in em rep for chronos rollup procedures
Rem
Rem    NOTES
Rem      <other useful comments, qualifications, etc.>
Rem
Rem    MODIFIED   (MM/DD/YY)
Rem    eporter     08/31/05 - Allow database_time to be null 
Rem    scgrover    07/07/05 - add extended sql trace 
Rem    yxie        06/06/05 - changes depends_on to supports_eum_on 
Rem    eporter     04/20/05 - Bug 4308759 - bootstrap rollup exception
Rem    yxie        04/15/05 - remove mgmt_target_memberships table
Rem    eporter     03/11/05 - Fix bootstrap rollup locking 
Rem    eporter     02/24/05 - Add rollups for incomplete and parts of pages 
Rem    eporter     12/15/04 - fix bootstrap dist rollups 
Rem    eporter     12/01/04 - Bug 3849061 - Rollup into new columns 
Rem    eporter     09/09/04 - Adding bootstrap procedures 
Rem    jhsy        08/13/04 - add database rollup 
Rem    snakai      02/09/04 - report successful cleanups as info msgs 
Rem    snakai      01/08/04 - fix retrieval of rollup timestamps
Rem    mashukla    02/11/03 - add server timings rollups 
Rem    mashukla    11/15/02 - add server timings rollups
Rem    rpinnama    05/15/02 - rpinnama_reorg_rep_scripts
Rem    rpinnama    05/15/02 - Created
Rem

----------------------------------------------------------------------
--
--
CREATE OR REPLACE PACKAGE BODY EMD_RT_ROLLUP AS

PROCEDURE DBMSJOB_EXTENDED_SQL_TRACE_ON(p_value IN BOOLEAN) IS
BEGIN  
  MGMT_SQLTRACE.EXTENDED_SQL_TRACE_ON(EST_RT_ROLLUP_NAME, p_value);
END DBMSJOB_EXTENDED_SQL_TRACE_ON;

--
--
----------------------------------------------------------------------
PROCEDURE ROLLUP IS

  v_target_guid            p_target_guid_list_type;
  v_last_load_time         p_date_list_type;
  v_start_rc               PLS_INTEGER;

  CURSOR target_cursor IS
    SELECT mt.target_guid, last_rt_load_time
      FROM MGMT_TARGETS mt, MGMT_TARGET_ROLLUP_TIMES mtrt
      WHERE mtrt.rollup_table_name = p_url_1hour_table
        AND mt.target_guid = mtrt.target_guid 
        AND (TRUNC(mt.last_rt_load_time, 'HH24') 
             > mtrt.rollup_timestamp);

BEGIN

  MGMT_SQLTRACE.EXTENDED_SQL_TRACE(EST_RT_ROLLUP_NAME);

  EMD_INITIALIZE();
  p_rollup_start_time := SYSDATE;

  OPEN target_cursor;
  LOOP

    FETCH target_cursor BULK COLLECT INTO v_target_guid,
                                          v_last_load_time
                                          LIMIT p_target_batchsize;
    EXIT WHEN NOT v_target_guid.EXISTS(1);

    -- Loop through the targets returned from the target_cursor.
    FOR i IN v_target_guid.FIRST..v_target_guid.LAST 
    LOOP
      
      -- Make sure we have all the target's information
      IF ( EMD_GET_TARGET_INFO(v_target_guid(i)) ) THEN

        p_cur_target_guid := v_target_guid(i);

        -- Calculate the new rollup timestamps.
        p_cur_hour_rollup := TRUNC(v_last_load_time(i), 'HH24') - 1/24;
        p_cur_day_rollup := TRUNC(p_cur_hour_rollup, 'DD') - 1;

        -- Calculate the purge boundaries.
        -- The rollup timestamps mark the beginning of the the latest
        -- rollup and thus they count for 1 unit (hour or day).  Hence,
        -- 1 unit is substracted from the keep window.
        p_cur_hour_purge := LEAST(p_cur_day_rollup,
                p_cur_hour_rollup - ((p_cur_hour_keep-1)/24));
        p_cur_dist_hour_purge := LEAST(p_cur_day_rollup,
                p_cur_hour_rollup - ((p_cur_dist_hour_keep-1)/24));
        p_cur_day_purge := 
                p_cur_day_rollup - (p_cur_day_keep-1);
        p_cur_dist_day_purge := 
                p_cur_day_rollup - (p_cur_dist_day_keep-1);

        -- Keep track of the starting total rowcount 
        v_start_rc := p_total_rc;

        -- Run hourly rollups
        EMD_URL_1HOUR_ROLLUP;
        EMD_URL_DIST_1HOUR_ROLLUP;
        EMD_DOM_1HOUR_ROLLUP;
        EMD_DOM_DIST_1HOUR_ROLLUP;
        EMD_IP_1HOUR_ROLLUP;
        EMD_IP_DIST_1HOUR_ROLLUP;
        EMD_PR_MAPPING_1HOUR_ROLLUP;
        EMD_INC_LOADS_1HOUR_ROLLUP;

        -- Run daily rollups and purges
        IF (p_cur_day_rollup > 
               p_cur_rollup_timestamps(p_url_1day_idx)) THEN
          EMD_URL_1DAY_ROLLUP;
          EMD_URL_1DAY_PURGE;
        END IF;
        IF (p_cur_day_rollup > 
               p_cur_rollup_timestamps(p_url_dist_1day_idx)) THEN
          EMD_URL_DIST_1DAY_ROLLUP;
          EMD_URL_DIST_1DAY_PURGE;
        END IF;
        IF (p_cur_day_rollup > 
               p_cur_rollup_timestamps(p_dom_1day_idx)) THEN
          EMD_DOM_1DAY_ROLLUP;
          EMD_DOM_1DAY_PURGE;
        END IF;
        IF (p_cur_day_rollup > 
               p_cur_rollup_timestamps(p_dom_dist_1day_idx)) THEN
          EMD_DOM_DIST_1DAY_ROLLUP;
          EMD_DOM_DIST_1DAY_PURGE;
        END IF;
        IF (p_cur_day_rollup > 
               p_cur_rollup_timestamps(p_ip_1day_idx)) THEN
          EMD_IP_1DAY_ROLLUP;
          EMD_IP_1DAY_PURGE;
        END IF;
        IF (p_cur_day_rollup > 
               p_cur_rollup_timestamps(p_ip_dist_1day_idx)) THEN
          EMD_IP_DIST_1DAY_ROLLUP;
          EMD_IP_DIST_1DAY_PURGE;
        END IF;
        IF (p_cur_day_rollup > 
               p_cur_rollup_timestamps(p_pr_mapping_1day_idx)) THEN
          EMD_PR_MAPPING_1DAY_ROLLUP;
          EMD_PR_MAPPING_1DAY_PURGE;
        END IF;
        IF (p_cur_day_rollup > 
               p_cur_rollup_timestamps(p_inc_loads_1day_idx)) THEN
          EMD_INC_LOADS_1DAY_ROLLUP;
          EMD_INC_LOADS_1DAY_PURGE;
        END IF;

        -- Run hourly purges
        EMD_URL_1HOUR_PURGE;
        EMD_URL_DIST_1HOUR_PURGE;
        EMD_DOM_1HOUR_PURGE;
        EMD_DOM_DIST_1HOUR_PURGE;
        EMD_IP_1HOUR_PURGE;
        EMD_IP_DIST_1HOUR_PURGE;
        EMD_PR_MAPPING_1HOUR_PURGE;
        EMD_INC_LOADS_1HOUR_PURGE;

        -- Purge the raw table
        EMD_RAW_PURGE;

      ELSE

        -- IMPORTANT: The user needs to be alerted of this error as
        -- it stops all rollups/purges for the target.
        MGMT_LOG.LOG_ERROR('RTROLLUP', null,
          'Fatal error fetching rollup timestamps (target_guid='
          || v_target_guid(i) || ')');   
      END IF;

    END LOOP;

  END LOOP;

  v_target_guid.DELETE;
  v_last_load_time.DELETE;

  IF target_cursor%ISOPEN THEN
    CLOSE target_cursor;
  END IF;

  EMD_WRITE_LOG;

EXCEPTION

  WHEN OTHERS THEN
    ROLLBACK;
    IF target_cursor%ISOPEN THEN
      CLOSE target_cursor;
    END IF;
    v_target_guid.DELETE;
    v_last_load_time.DELETE;
    MGMT_LOG.LOG_ERROR('RTROLLUP', null,
      'Fatal error (see related errors): ' || 
      SUBSTR(SQLERRM, 1, 1100));

END ROLLUP;
----------------------------------------------------------------------

----------------------------------------------------------------------
PROCEDURE BOOTSTRAP_ROLLUP(target_name_in IN VARCHAR2,
                           target_type_in IN VARCHAR2,
                           result         OUT NUMBER,
                           err_desc       OUT VARCHAR2 ) IS

  v_target_guid            p_target_guid_list_type;
  v_last_load_time         p_date_list_type;
  v_start_rc               PLS_INTEGER;
  v_bootstrap_record       p_bootstrap_record_type;
  v_bootstrap_log          p_bootstrap_log_type;
  v_lockhandle             VARCHAR2(100);
  v_result                 NUMBER;

  -- Select the guids that have any new data for this target
  CURSOR target_cursor IS
    SELECT mt.target_guid, mt.last_rt_load_time
      FROM MGMT_TARGETS mt, MGMT_RT_BOOTSTRAP_TIMES mrbt,
           MGMT_TARGET_ASSOCS m, MGMT_TARGETS mt2
      WHERE mt.target_guid          = mrbt.target_guid 
        AND m.assoc_target_guid     = mt.target_guid
        AND mt.last_rt_load_time    > mrbt.rollup_timestamp
        AND mt2.target_name         = target_name_in
        AND mt2.target_type         = target_type_in
        AND m.source_target_guid    = mt2.target_guid
        AND m.assoc_guid            = MGMT_WEBSITE_TARGET.supports_eum_on_guid;

BEGIN
  
  EMD_BOOTSTRAP_INITIALIZE();

  -- Init counters
  v_bootstrap_log.total_rc           := 0;
  v_bootstrap_log.url_rollup_rc      := 0;
  v_bootstrap_log.url_dist_rollup_rc := 0;
  v_bootstrap_log.dom_rollup_rc      := 0;
  v_bootstrap_log.dom_dist_rollup_rc := 0;
  v_bootstrap_log.ip_rollup_rc       := 0;
  v_bootstrap_log.ip_dist_rollup_rc  := 0;
  v_bootstrap_log.rollup_start_time  := SYSDATE;

  OPEN target_cursor;
  LOOP

    FETCH target_cursor BULK COLLECT INTO v_target_guid,
                                          v_last_load_time
                                          LIMIT p_target_batchsize;
    EXIT WHEN NOT v_target_guid.EXISTS(1);

    -- Loop through the targets returned from the target_cursor.
    FOR i IN v_target_guid.FIRST..v_target_guid.LAST 
    LOOP
      

    dbms_lock.allocate_unique('ORA_CHRONOS_'||v_target_guid(i), v_lockhandle);
    v_result := dbms_lock.request(v_lockhandle);

    SELECT rollup_timestamp 
      INTO v_bootstrap_record.last_rollup_time
      FROM MGMT_RT_BOOTSTRAP_TIMES
      WHERE target_guid = v_target_guid(i);


        -- Check if filter values are defined for the target
        BEGIN
          SELECT NVL(TO_NUMBER(property_value), p_curdef_max_response_time)
            INTO v_bootstrap_record.max_response_time
            FROM MGMT_TARGET_PROPERTIES 
            WHERE target_guid = v_target_guid(i)
             AND property_name = p_elapsed_time_param;
        EXCEPTION
          WHEN NO_DATA_FOUND THEN
            v_bootstrap_record.max_response_time := p_curdef_max_response_time;
          WHEN OTHERS THEN RAISE;
        END;
        v_bootstrap_record.target_guid := v_target_guid(i);

        -- Calculate the new rollup timestamps.
        v_bootstrap_record.cur_rollup_time := v_last_load_time(i);

        -- Keep track of the starting total rowcount 
        v_start_rc := v_bootstrap_log.total_rc;
	
        -- Run hourly rollups
        EMD_URL_BOOTSTRAP_ROLLUP(v_bootstrap_record, v_bootstrap_log);
        EMD_URL_DIST_BOOTSTRAP_ROLLUP(v_bootstrap_record, v_bootstrap_log);
        EMD_DOM_BOOTSTRAP_ROLLUP(v_bootstrap_record, v_bootstrap_log);
        EMD_DOM_DIST_BOOTSTRAP_ROLLUP(v_bootstrap_record, v_bootstrap_log);
        EMD_IP_BOOTSTRAP_ROLLUP(v_bootstrap_record, v_bootstrap_log);
        EMD_IP_DIST_BOOTSTRAP_ROLLUP(v_bootstrap_record, v_bootstrap_log);


        UPDATE MGMT_RT_BOOTSTRAP_TIMES
          SET rollup_timestamp = v_bootstrap_record.cur_rollup_time
          WHERE target_guid = v_bootstrap_record.target_guid;

      v_result := dbms_lock.release(v_lockhandle);

    END LOOP;

  END LOOP;

  v_target_guid.DELETE;
  v_last_load_time.DELETE;

  IF target_cursor%ISOPEN THEN
    CLOSE target_cursor;
  END IF;

  EMD_WRITE_BOOTSTRAP_LOG(v_bootstrap_log);
  result := 0;

EXCEPTION

  WHEN OTHERS THEN
    ROLLBACK;
    IF target_cursor%ISOPEN THEN
      CLOSE target_cursor;
    END IF;
    v_target_guid.DELETE;
    v_last_load_time.DELETE;
    v_result := dbms_lock.release(v_lockhandle);
    MGMT_LOG.LOG_ERROR('RTBOOTSTRAPROLLUP', null,
      'Fatal error (see related errors): ' || 
      SUBSTR(SQLERRM, 1, 1100));
    result := SQLCODE;
    err_desc := 'Fatal error while processing samples: ' || 
                SUBSTR(SQLERRM, 1, 1100);

END BOOTSTRAP_ROLLUP;
----------------------------------------------------------------------

----------------------------------------------------------------------
PROCEDURE EMD_INITIALIZE IS

BEGIN

  -- Get the default keep window values
  BEGIN  
    SELECT NVL(TO_NUMBER(parameter_value), p_default_raw_keep)
      INTO p_curdef_raw_keep
      FROM MGMT_PARAMETERS 
      WHERE parameter_name = p_raw_keep_param;
  EXCEPTION
    WHEN NO_DATA_FOUND THEN
      p_curdef_raw_keep := p_default_raw_keep;
    WHEN OTHERS THEN RAISE;
  END;
  BEGIN  
    SELECT NVL(TO_NUMBER(parameter_value), p_default_hour_keep)
      INTO p_curdef_hour_keep
      FROM MGMT_PARAMETERS 
      WHERE parameter_name = p_hour_keep_param;
  EXCEPTION
    WHEN NO_DATA_FOUND THEN
      p_curdef_hour_keep := p_default_hour_keep;
    WHEN OTHERS THEN RAISE;
  END;
  BEGIN  
    SELECT NVL(TO_NUMBER(parameter_value), p_default_day_keep)
      INTO p_curdef_day_keep
      FROM MGMT_PARAMETERS 
      WHERE parameter_name = p_day_keep_param;
  EXCEPTION
    WHEN NO_DATA_FOUND THEN
      p_curdef_day_keep := p_default_day_keep;
    WHEN OTHERS THEN RAISE;
  END;
  BEGIN  
    SELECT NVL(TO_NUMBER(parameter_value), p_default_dist_hour_keep)
      INTO p_curdef_dist_hour_keep
      FROM MGMT_PARAMETERS 
      WHERE parameter_name = p_dist_hour_keep_param;
  EXCEPTION
    WHEN NO_DATA_FOUND THEN
      p_curdef_dist_hour_keep := p_default_dist_hour_keep;
    WHEN OTHERS THEN RAISE;
  END;
  BEGIN  
    SELECT NVL(TO_NUMBER(parameter_value), p_default_dist_day_keep)
      INTO p_curdef_dist_day_keep
      FROM MGMT_PARAMETERS 
      WHERE parameter_name = p_dist_day_keep_param;
  EXCEPTION
    WHEN NO_DATA_FOUND THEN
      p_curdef_dist_day_keep := p_default_dist_day_keep;
    WHEN OTHERS THEN RAISE;
  END;

  -- Get default filter values
  BEGIN  
    SELECT NVL(TO_NUMBER(parameter_value),
               p_default_max_response_time)
      INTO p_curdef_max_response_time
      FROM MGMT_PARAMETERS 
      WHERE parameter_name = p_elapsed_time_param;
  EXCEPTION
    WHEN NO_DATA_FOUND THEN
      p_curdef_max_response_time := p_default_max_response_time;
    WHEN OTHERS THEN RAISE;
  END;

  -- Init counters
  p_total_rc                     := 0;
  p_raw_purge_rc                 := 0;
  p_url_1hour_rollup_rc          := 0;
  p_url_dist_1hour_rollup_rc     := 0;
  p_url_1day_rollup_rc           := 0;
  p_url_dist_1day_rollup_rc      := 0;
  p_url_1hour_purge_rc           := 0;
  p_url_dist_1hour_purge_rc      := 0;
  p_url_1day_purge_rc            := 0;
  p_url_dist_1day_purge_rc       := 0;
  p_dom_1hour_rollup_rc          := 0;
  p_dom_dist_1hour_rollup_rc     := 0;
  p_dom_1day_rollup_rc           := 0;
  p_dom_dist_1day_rollup_rc      := 0;
  p_dom_1hour_purge_rc           := 0;
  p_dom_dist_1hour_purge_rc      := 0;
  p_dom_1day_purge_rc            := 0;
  p_dom_dist_1day_purge_rc       := 0;
  p_ip_1hour_rollup_rc           := 0;
  p_ip_dist_1hour_rollup_rc      := 0;
  p_ip_1day_rollup_rc            := 0;
  p_ip_dist_1day_rollup_rc       := 0;
  p_ip_1hour_purge_rc            := 0;
  p_ip_dist_1hour_purge_rc       := 0;
  p_ip_1day_purge_rc             := 0;
  p_ip_dist_1day_purge_rc        := 0;

EXCEPTION

  WHEN OTHERS THEN
      MGMT_LOG.LOG_ERROR('RTROLLUP', null,
      'Fatal error during initialization: ' || SUBSTR(SQLERRM, 1, 1100));
    RAISE;

END EMD_INITIALIZE;
----------------------------------------------------------------------

----------------------------------------------------------------------
PROCEDURE EMD_BOOTSTRAP_INITIALIZE IS

BEGIN


  -- Get default filter values
  BEGIN  
    SELECT NVL(TO_NUMBER(parameter_value),
               p_default_max_response_time)
      INTO p_curdef_max_response_time
      FROM MGMT_PARAMETERS 
      WHERE parameter_name = p_elapsed_time_param;
  EXCEPTION
    WHEN NO_DATA_FOUND THEN
      p_curdef_max_response_time := p_default_max_response_time;
    WHEN OTHERS THEN RAISE;
  END;

EXCEPTION

  WHEN OTHERS THEN
      MGMT_LOG.LOG_ERROR('RTBOOTSTRAPROLLUP', null,
      'Fatal error during initialization: ' || SUBSTR(SQLERRM, 1, 1100));
    RAISE;

END EMD_BOOTSTRAP_INITIALIZE;
----------------------------------------------------------------------

----------------------------------------------------------------------
FUNCTION EMD_GET_TARGET_INFO( cur_target_guid IN RAW ) 
  RETURN BOOLEAN IS

  CURSOR timestamp_cursor IS
    SELECT rollup_table_name, rollup_timestamp, rowid
      FROM MGMT_TARGET_ROLLUP_TIMES
      WHERE target_guid = cur_target_guid AND
        rollup_table_name LIKE 'MGMT_RT_%';

  v_tabname     p_varchar2_256_list_type;
  v_rollupts    p_date_list_type;
  v_rowid       p_rowid_list_type;
  v_counter     PLS_INTEGER;

BEGIN 

  -- Get the rollup timestamps
  OPEN timestamp_cursor;
  FETCH timestamp_cursor BULK COLLECT INTO 
    v_tabname, v_rollupts, v_rowid;
  CLOSE timestamp_cursor;

  IF v_tabname IS NULL OR v_rollupts IS NULL OR v_rowid IS NULL
     OR (v_tabname.COUNT < p_num_rollup_tables) THEN
    v_tabname.DELETE;
    v_rollupts.DELETE;
    v_rowid.DELETE;  
    MGMT_LOG.LOG_ERROR('RTROLLUP', null,
        'Invalid state (missing timestamps): ' || SUBSTR(SQLERRM, 1, 1000));
    RETURN FALSE;      
  END IF;

  -- Clear the arrays
  FOR v_counter IN 1..p_num_rollup_tables LOOP
    p_cur_rollup_timestamps(v_counter) := NULL;
    p_cur_rollup_rowids(v_counter) := NULL;
  END LOOP;

  -- Process the timestamps
  FOR v_counter IN v_tabname.FIRST..v_tabname.LAST LOOP
    IF v_tabname(v_counter) = p_url_1hour_table THEN
      p_cur_rollup_timestamps(p_url_1hour_idx) := v_rollupts(v_counter);
      p_cur_rollup_rowids(p_url_1hour_idx) := v_rowid(v_counter);
    ELSIF v_tabname(v_counter) = p_url_1day_table THEN
      p_cur_rollup_timestamps(p_url_1day_idx) := v_rollupts(v_counter);
      p_cur_rollup_rowids(p_url_1day_idx) := v_rowid(v_counter);
    ELSIF v_tabname(v_counter) = p_url_dist_1hour_table THEN
      p_cur_rollup_timestamps(p_url_dist_1hour_idx) := v_rollupts(v_counter);
      p_cur_rollup_rowids(p_url_dist_1hour_idx) := v_rowid(v_counter);
    ELSIF v_tabname(v_counter) = p_url_dist_1day_table THEN
      p_cur_rollup_timestamps(p_url_dist_1day_idx) := v_rollupts(v_counter);
      p_cur_rollup_rowids(p_url_dist_1day_idx) := v_rowid(v_counter);
    ELSIF v_tabname(v_counter) = p_dom_1hour_table THEN
      p_cur_rollup_timestamps(p_dom_1hour_idx) := v_rollupts(v_counter);
      p_cur_rollup_rowids(p_dom_1hour_idx) := v_rowid(v_counter);
    ELSIF v_tabname(v_counter) = p_dom_1day_table THEN
      p_cur_rollup_timestamps(p_dom_1day_idx) := v_rollupts(v_counter);
      p_cur_rollup_rowids(p_dom_1day_idx) := v_rowid(v_counter);
    ELSIF v_tabname(v_counter) = p_dom_dist_1hour_table THEN
      p_cur_rollup_timestamps(p_dom_dist_1hour_idx) := v_rollupts(v_counter);
      p_cur_rollup_rowids(p_dom_dist_1hour_idx) := v_rowid(v_counter);
    ELSIF v_tabname(v_counter) = p_dom_dist_1day_table THEN
      p_cur_rollup_timestamps(p_dom_dist_1day_idx) := v_rollupts(v_counter);
      p_cur_rollup_rowids(p_dom_dist_1day_idx) := v_rowid(v_counter);
    ELSIF v_tabname(v_counter) = p_ip_1hour_table THEN
      p_cur_rollup_timestamps(p_ip_1hour_idx) := v_rollupts(v_counter);
      p_cur_rollup_rowids(p_ip_1hour_idx) := v_rowid(v_counter);
    ELSIF v_tabname(v_counter) = p_ip_1day_table THEN
      p_cur_rollup_timestamps(p_ip_1day_idx) := v_rollupts(v_counter);
      p_cur_rollup_rowids(p_ip_1day_idx) := v_rowid(v_counter);
    ELSIF v_tabname(v_counter) = p_ip_dist_1hour_table THEN
      p_cur_rollup_timestamps(p_ip_dist_1hour_idx) := v_rollupts(v_counter);
      p_cur_rollup_rowids(p_ip_dist_1hour_idx) := v_rowid(v_counter);
    ELSIF v_tabname(v_counter) = p_ip_dist_1day_table THEN
      p_cur_rollup_timestamps(p_ip_dist_1day_idx) := v_rollupts(v_counter);
      p_cur_rollup_rowids(p_ip_dist_1day_idx) := v_rowid(v_counter);
    ELSIF v_tabname(v_counter) = p_pr_mapping_1hour_table THEN
      p_cur_rollup_timestamps(p_pr_mapping_1hour_idx) := v_rollupts(v_counter);
      p_cur_rollup_rowids(p_pr_mapping_1hour_idx) := v_rowid(v_counter);
    ELSIF v_tabname(v_counter) = p_pr_mapping_1day_table THEN
      p_cur_rollup_timestamps(p_pr_mapping_1day_idx) := v_rollupts(v_counter);
      p_cur_rollup_rowids(p_pr_mapping_1day_idx) := v_rowid(v_counter);
    ELSIF v_tabname(v_counter) = p_inc_loads_1hour_table THEN
      p_cur_rollup_timestamps(p_inc_loads_1hour_idx) := v_rollupts(v_counter);
      p_cur_rollup_rowids(p_inc_loads_1hour_idx) := v_rowid(v_counter);
    ELSIF v_tabname(v_counter) = p_inc_loads_1day_table THEN
      p_cur_rollup_timestamps(p_inc_loads_1day_idx) := v_rollupts(v_counter);
      p_cur_rollup_rowids(p_inc_loads_1day_idx) := v_rowid(v_counter);
    END IF;
  END LOOP;

  -- Not needed anymore
  v_tabname.DELETE;
  v_rollupts.DELETE;
  v_rowid.DELETE;  

  -- Verify that we got all the tables
  FOR v_counter IN 1..p_num_rollup_tables LOOP
    IF p_cur_rollup_timestamps(v_counter) IS NULL OR
       p_cur_rollup_rowids(v_counter) IS NULL THEN
      MGMT_LOG.LOG_ERROR('RTROLLUP', null,
        'Invalid state (bad timestamps): ' || SUBSTR(SQLERRM, 1, 1000));
      RETURN FALSE;      
    END IF;
  END LOOP;

  -- Get any keep window values defined for the target 
  BEGIN
    SELECT NVL(TO_NUMBER(property_value), p_curdef_raw_keep)
      INTO p_cur_raw_keep
      FROM MGMT_TARGET_PROPERTIES 
      WHERE target_guid = cur_target_guid
        AND property_name = p_raw_keep_param;
  EXCEPTION
    WHEN NO_DATA_FOUND THEN
      p_cur_raw_keep := p_curdef_raw_keep;
    WHEN OTHERS THEN RAISE;
  END;     
  BEGIN
    SELECT NVL(TO_NUMBER(property_value), p_curdef_hour_keep)
      INTO p_cur_hour_keep
      FROM MGMT_TARGET_PROPERTIES 
      WHERE target_guid = cur_target_guid
        AND property_name = p_hour_keep_param;
  EXCEPTION
    WHEN NO_DATA_FOUND THEN
      p_cur_hour_keep := p_curdef_hour_keep;
    WHEN OTHERS THEN RAISE;
  END;     
  BEGIN
    SELECT NVL(TO_NUMBER(property_value), p_curdef_day_keep)
      INTO p_cur_day_keep
      FROM MGMT_TARGET_PROPERTIES 
      WHERE target_guid = cur_target_guid
        AND property_name = p_day_keep_param;
  EXCEPTION
    WHEN NO_DATA_FOUND THEN
      p_cur_day_keep := p_curdef_day_keep;
    WHEN OTHERS THEN RAISE;
  END;     
  BEGIN
    SELECT NVL(TO_NUMBER(property_value), p_curdef_dist_hour_keep)
      INTO p_cur_dist_hour_keep
      FROM MGMT_TARGET_PROPERTIES 
      WHERE target_guid = cur_target_guid
        AND property_name = p_dist_hour_keep_param;
  EXCEPTION
    WHEN NO_DATA_FOUND THEN
      p_cur_dist_hour_keep := p_curdef_dist_hour_keep;
    WHEN OTHERS THEN RAISE;
  END;     
  BEGIN
    SELECT NVL(TO_NUMBER(property_value), p_curdef_dist_day_keep)
      INTO p_cur_dist_day_keep
      FROM MGMT_TARGET_PROPERTIES 
      WHERE target_guid = cur_target_guid
        AND property_name = p_dist_day_keep_param;
  EXCEPTION
    WHEN NO_DATA_FOUND THEN
      p_cur_dist_day_keep := p_curdef_dist_day_keep;
    WHEN OTHERS THEN RAISE;
  END;     

  -- Check if filter values are defined for the target
  BEGIN
    SELECT NVL(TO_NUMBER(property_value), p_curdef_max_response_time)
      INTO p_cur_max_response_time
      FROM MGMT_TARGET_PROPERTIES 
      WHERE target_guid = cur_target_guid
        AND property_name = p_elapsed_time_param;
  EXCEPTION
    WHEN NO_DATA_FOUND THEN
      p_cur_max_response_time := p_curdef_max_response_time;
    WHEN OTHERS THEN RAISE;
  END;     

  RETURN TRUE;

EXCEPTION

  WHEN OTHERS THEN
    IF timestamp_cursor%ISOPEN THEN
       CLOSE timestamp_cursor;
    END IF;     
    v_tabname.DELETE;
    v_rollupts.DELETE;
    v_rowid.DELETE;
    MGMT_LOG.LOG_ERROR('RTROLLUP', null,
      'Fatal error getting target info: ' || SUBSTR(SQLERRM, 1, 1100));
    RETURN FALSE;

END EMD_GET_TARGET_INFO;
----------------------------------------------------------------------

----------------------------------------------------------------------
PROCEDURE EMD_WRITE_LOG IS

BEGIN
  MGMT_LOG.LOG_PERFORMANCE('RTROLLUP', (SYSDATE-p_rollup_start_time)*(24*60*60*1000),
             p_rollup_start_time, 'Y', 'RECORDS', p_total_rc);
  MGMT_LOG.LOG_PERFORMANCE('RTROLLUP RAW PURGE', (SYSDATE-p_rollup_start_time)*(24*60*60*1000),
             p_rollup_start_time, 'N', 'RECORDS', p_raw_purge_rc);
  MGMT_LOG.LOG_PERFORMANCE('RTROLLUP URL 1HOUR', (SYSDATE-p_rollup_start_time)*(24*60*60*1000),
             p_rollup_start_time, 'N', 'RECORDS', p_url_1hour_rollup_rc);
  MGMT_LOG.LOG_PERFORMANCE('RTROLLUP URL 1HOUR PURGE',
             (SYSDATE-p_rollup_start_time)*(24*60*60*1000), p_rollup_start_time, 'N', 'RECORDS',
             p_url_1hour_purge_rc);
  MGMT_LOG.LOG_PERFORMANCE('RTROLLUP URL DIST 1HOUR',
             (SYSDATE-p_rollup_start_time)*(24*60*60*1000), p_rollup_start_time, 'N', 'RECORDS',
             p_url_dist_1hour_rollup_rc);
  MGMT_LOG.LOG_PERFORMANCE('RTROLLUP URL DIST 1HOUR PURGE',
             (SYSDATE-p_rollup_start_time)*(24*60*60*1000), p_rollup_start_time, 'N', 'RECORDS',
             p_url_dist_1hour_purge_rc);
  MGMT_LOG.LOG_PERFORMANCE('RTROLLUP URL 1DAY', (SYSDATE-p_rollup_start_time)*(24*60*60*1000),
             p_rollup_start_time, 'N', 'RECORDS', p_url_1day_rollup_rc);
  MGMT_LOG.LOG_PERFORMANCE('RTROLLUP URL 1DAY PURGE',
             (SYSDATE-p_rollup_start_time)*(24*60*60*1000), p_rollup_start_time, 'N', 'RECORDS',
             p_url_1day_purge_rc);
  MGMT_LOG.LOG_PERFORMANCE('RTROLLUP URL DIST 1DAY',
             (SYSDATE-p_rollup_start_time)*(24*60*60*1000), p_rollup_start_time, 'N', 'RECORDS',
             p_url_dist_1day_rollup_rc);
  MGMT_LOG.LOG_PERFORMANCE('RTROLLUP URL DIST 1DAY PURGE',
             (SYSDATE-p_rollup_start_time)*(24*60*60*1000), p_rollup_start_time, 'N', 'RECORDS',
             p_url_dist_1day_purge_rc);

  MGMT_LOG.LOG_PERFORMANCE('RTROLLUP DOMAIN 1HOUR', (SYSDATE-p_rollup_start_time)*(24*60*60*1000), 
             p_rollup_start_time, 'N', 'RECORDS', p_dom_1hour_rollup_rc);
  MGMT_LOG.LOG_PERFORMANCE('RTROLLUP DOMAIN 1HOUR PURGE',
             (SYSDATE-p_rollup_start_time)*(24*60*60*1000), p_rollup_start_time, 'N', 'RECORDS',
             p_dom_1hour_purge_rc);
  MGMT_LOG.LOG_PERFORMANCE('RTROLLUP DOMAIN DIST 1HOUR',
             (SYSDATE-p_rollup_start_time)*(24*60*60*1000), p_rollup_start_time, 'N', 'RECORDS',
             p_dom_dist_1hour_rollup_rc);
  MGMT_LOG.LOG_PERFORMANCE('RTROLLUP DOMAIN DIST 1HOUR PURGE',
             (SYSDATE-p_rollup_start_time)*(24*60*60*1000), p_rollup_start_time, 'N', 'RECORDS',
             p_dom_dist_1hour_purge_rc);
  MGMT_LOG.LOG_PERFORMANCE('RTROLLUP DOMAIN 1DAY', (SYSDATE-p_rollup_start_time)*(24*60*60*1000),
             p_rollup_start_time, 'N', 'RECORDS', p_dom_1day_rollup_rc);
  MGMT_LOG.LOG_PERFORMANCE('RTROLLUP DOMAIN 1DAY PURGE',
             (SYSDATE-p_rollup_start_time)*(24*60*60*1000), p_rollup_start_time, 'N', 'RECORDS',
             p_dom_1day_purge_rc);
  MGMT_LOG.LOG_PERFORMANCE('RTROLLUP DOMAIN DIST 1DAY',
             (SYSDATE-p_rollup_start_time)*(24*60*60*1000), p_rollup_start_time, 'N', 'RECORDS',
             p_dom_dist_1day_rollup_rc);
  MGMT_LOG.LOG_PERFORMANCE('RTROLLUP DOMAIN DIST 1DAY PURGE',
             (SYSDATE-p_rollup_start_time)*(24*60*60*1000), p_rollup_start_time, 'N', 'RECORDS',
             p_dom_dist_1day_purge_rc);
  MGMT_LOG.LOG_PERFORMANCE('RTROLLUP IP 1HOUR', (SYSDATE-p_rollup_start_time)*(24*60*60*1000),
             p_rollup_start_time, 'N', 'RECORDS', p_ip_1hour_rollup_rc);
  MGMT_LOG.LOG_PERFORMANCE('RTROLLUP IP 1HOUR PURGE',
             (SYSDATE-p_rollup_start_time)*(24*60*60*1000), p_rollup_start_time, 'N', 'RECORDS',
             p_ip_1hour_purge_rc);
  MGMT_LOG.LOG_PERFORMANCE('RTROLLUP IP DIST 1HOUR',
             (SYSDATE-p_rollup_start_time)*(24*60*60*1000), p_rollup_start_time, 'N', 'RECORDS',
             p_ip_dist_1hour_rollup_rc);
  MGMT_LOG.LOG_PERFORMANCE('RTROLLUP IP DIST 1HOUR PURGE',
             (SYSDATE-p_rollup_start_time)*(24*60*60*1000), p_rollup_start_time, 'N', 'RECORDS',
             p_ip_dist_1hour_purge_rc);
  MGMT_LOG.LOG_PERFORMANCE('RTROLLUP IP 1DAY', (SYSDATE-p_rollup_start_time)*(24*60*60*1000),
             p_rollup_start_time, 'N', 'RECORDS', p_ip_1day_rollup_rc);
  MGMT_LOG.LOG_PERFORMANCE('RTROLLUP IP 1DAY PURGE',
             (SYSDATE-p_rollup_start_time)*(24*60*60*1000), p_rollup_start_time, 'N', 'RECORDS',
             p_ip_1day_purge_rc);
  MGMT_LOG.LOG_PERFORMANCE('RTROLLUP IP DIST 1DAY',
             (SYSDATE-p_rollup_start_time)*(24*60*60*1000), p_rollup_start_time, 'N', 'RECORDS',
             p_ip_dist_1day_rollup_rc);
  MGMT_LOG.LOG_PERFORMANCE('RTROLLUP IP DIST 1DAY PURGE',
             (SYSDATE-p_rollup_start_time)*(24*60*60*1000), p_rollup_start_time, 'N', 'RECORDS',
             p_ip_dist_1day_purge_rc);

  MGMT_LOG.LOG_PERFORMANCE('RTROLLUP PR MAPPING 1HOUR',
             (SYSDATE-p_rollup_start_time)*(24*60*60*1000), p_rollup_start_time, 'N', 'RECORDS', p_pr_mapping_1hour_rollup_rc);
  MGMT_LOG.LOG_PERFORMANCE('RTROLLUP PR MAPPING 1HOUR PURGE',
             (SYSDATE-p_rollup_start_time)*(24*60*60*1000), p_rollup_start_time, 'N', 'RECORDS', p_pr_mapping_1hour_purge_rc);
  MGMT_LOG.LOG_PERFORMANCE('RTROLLUP PR MAPPING 1DAY',
             (SYSDATE-p_rollup_start_time)*(24*60*60*1000), p_rollup_start_time, 'N', 'RECORDS', p_pr_mapping_1day_rollup_rc);
  MGMT_LOG.LOG_PERFORMANCE('RTROLLUP PR MAPPING 1DAY PURGE',
             (SYSDATE-p_rollup_start_time)*(24*60*60*1000), p_rollup_start_time, 'N', 'RECORDS', p_pr_mapping_1day_purge_rc);
  MGMT_LOG.LOG_PERFORMANCE('RTROLLUP INCOMPLETE LOADS 1HOUR',
             (SYSDATE-p_rollup_start_time)*(24*60*60*1000), p_rollup_start_time, 'N', 'RECORDS', p_inc_loads_1hour_rollup_rc);
  MGMT_LOG.LOG_PERFORMANCE('RTROLLUP INCOMPLETE LOADS 1HOUR PURGE',
             (SYSDATE-p_rollup_start_time)*(24*60*60*1000), p_rollup_start_time, 'N', 'RECORDS', p_inc_loads_1hour_purge_rc);
  MGMT_LOG.LOG_PERFORMANCE('RTROLLUP INCOMPLETE LOADS 1DAY',
             (SYSDATE-p_rollup_start_time)*(24*60*60*1000), p_rollup_start_time, 'N', 'RECORDS', p_inc_loads_1day_rollup_rc);
  MGMT_LOG.LOG_PERFORMANCE('RTROLLUP INCOMPLETE LOADS 1DAY PURGE',
             (SYSDATE-p_rollup_start_time)*(24*60*60*1000), p_rollup_start_time, 'N', 'RECORDS', p_inc_loads_1day_purge_rc);
EXCEPTION

  WHEN OTHERS THEN
    ROLLBACK;
    MGMT_LOG.LOG_ERROR('RTROLLUP', null,
      'Error while writing rollup stats to database.');

END EMD_WRITE_LOG;
----------------------------------------------------------------------

----------------------------------------------------------------------
PROCEDURE EMD_WRITE_BOOTSTRAP_LOG(
                  bootstrap_log_in    IN p_bootstrap_log_type) IS

BEGIN
  MGMT_LOG.LOG_PERFORMANCE('RTBOOTSTRAPROLLUP', (SYSDATE-bootstrap_log_in.rollup_start_time)*(24*60*60*1000),
             bootstrap_log_in.rollup_start_time, 'Y', 'RECORDS', bootstrap_log_in.total_rc);
  MGMT_LOG.LOG_PERFORMANCE('RTBOOTSTRAPROLLUP URL', (SYSDATE-bootstrap_log_in.rollup_start_time)*(24*60*60*1000),
             bootstrap_log_in.rollup_start_time, 'N', 'RECORDS', bootstrap_log_in.url_rollup_rc);
  MGMT_LOG.LOG_PERFORMANCE('RTBOOTSTRAPROLLUP URL DIST',
             (SYSDATE-bootstrap_log_in.rollup_start_time)*(24*60*60*1000), bootstrap_log_in.rollup_start_time, 'N', 'RECORDS',
             bootstrap_log_in.url_dist_rollup_rc);
  MGMT_LOG.LOG_PERFORMANCE('RTBOOTSTRAPROLLUP DOMAIN', (SYSDATE-bootstrap_log_in.rollup_start_time)*(24*60*60*1000), 
             bootstrap_log_in.rollup_start_time, 'N', 'RECORDS', bootstrap_log_in.dom_rollup_rc);
  MGMT_LOG.LOG_PERFORMANCE('RTBOOTSTRAPROLLUP DOMAIN DIST',
             (SYSDATE-bootstrap_log_in.rollup_start_time)*(24*60*60*1000), bootstrap_log_in.rollup_start_time, 'N', 'RECORDS',
             bootstrap_log_in.dom_dist_rollup_rc);
  MGMT_LOG.LOG_PERFORMANCE('RTBOOTSTRAPROLLUP IP', (SYSDATE-bootstrap_log_in.rollup_start_time)*(24*60*60*1000),
             bootstrap_log_in.rollup_start_time, 'N', 'RECORDS', bootstrap_log_in.ip_rollup_rc);
  MGMT_LOG.LOG_PERFORMANCE('RTBOOTSTRAPROLLUP IP DIST',
             (SYSDATE-bootstrap_log_in.rollup_start_time)*(24*60*60*1000), bootstrap_log_in.rollup_start_time, 'N', 'RECORDS',
             bootstrap_log_in.ip_dist_rollup_rc);
EXCEPTION

  WHEN OTHERS THEN
    ROLLBACK;
    MGMT_LOG.LOG_ERROR('RTBOOTSTRAPROLLUP', null,
      'Error while writing rollup stats to database.');

END EMD_WRITE_BOOTSTRAP_LOG;
----------------------------------------------------------------------

----------------------------------------------------------------------
PROCEDURE EMD_RAW_PURGE IS

  v_purge_rowid p_rowid_list_type;
  v_rc PLS_INTEGER := 0;

  CURSOR purge_cursor ( c_target_guid RAW,
                        c_max_purge_time DATE ) IS
    SELECT rowid
      FROM MGMT_RT_METRICS_RAW
      WHERE target_guid = c_target_guid
        AND collection_timestamp < c_max_purge_time;
BEGIN  

  -- p_cur_hour_rollup marks the beginning of the latest rollup hour,
  -- i.e. it already counts for 1 hour.  Hence, 1 is substracted from
  -- the keep window.
  OPEN purge_cursor( p_cur_target_guid, 
                     p_cur_hour_rollup - ((p_cur_raw_keep-1)/24) );
  LOOP

    FETCH purge_cursor BULK COLLECT INTO v_purge_rowid 
      LIMIT p_purge_batchsize;

    -- Exit loop when there's no more data to purge
    EXIT WHEN v_purge_rowid.COUNT <= 0;

    -- Bulk delete by rowid.
    FORALL i IN v_purge_rowid.FIRST..v_purge_rowid.LAST
      DELETE FROM MGMT_RT_METRICS_RAW 
        WHERE rowid = v_purge_rowid(i);

    COMMIT;

    -- Keep track of the number of inserts
    v_rc := v_rc + v_purge_rowid.COUNT;

    -- Clear the rowid list
    v_purge_rowid.DELETE;

  END LOOP;

  IF purge_cursor%ISOPEN THEN
    CLOSE purge_cursor;
  END IF;

  -- Update the global row counters
  p_total_rc := p_total_rc + v_rc;
  p_raw_purge_rc := p_raw_purge_rc + v_rc;

EXCEPTION

  WHEN OTHERS THEN
    ROLLBACK;
    IF purge_cursor%ISOPEN THEN
      CLOSE purge_cursor;
    END IF;
    v_purge_rowid.DELETE;
    MGMT_LOG.LOG_ERROR('RTROLLUP', null,
      'Error during raw purge: ' || SUBSTR(SQLERRM, 1, 1100));

END EMD_RAW_PURGE;
----------------------------------------------------------------------

----------------------    HOURLY ROLLUPS    --------------------------

----------------------------------------------------------------------
PROCEDURE EMD_URL_1HOUR_ROLLUP IS 

  v_rollup_rec      p_hour_rollup_record_type;
  v_rc              PLS_INTEGER := 0; 

  CURSOR rollup_cursor ( c_target_guid RAW,
                         c_prev_rollup DATE,
                         c_cur_rollup DATE,
                         c_max_elapsed_time NUMBER ) IS    
    SELECT TRUNC(collection_timestamp, 'HH24'), metric_name,
      url_filename, url_base,
      COUNT(target_guid), 
      AVG(elapsed_time), MIN(elapsed_time), 
      MAX(elapsed_time), VARIANCE(elapsed_time),
      AVG(server_latency_time), MIN(server_latency_time), 
      MAX(server_latency_time), VARIANCE(server_latency_time), 
      AVG(database_time), MIN(database_time), 
      MAX(database_time), VARIANCE(database_time)
    FROM MGMT_RT_METRICS_RAW
    WHERE target_guid = c_target_guid
      AND TRUNC(collection_timestamp, 'HH24') > c_prev_rollup 
      AND TRUNC(collection_timestamp, 'HH24') <= c_cur_rollup
      AND elapsed_time <= c_max_elapsed_time
    GROUP BY TRUNC(collection_timestamp, 'HH24'), metric_name, 
      url_filename, url_base;

BEGIN

  OPEN rollup_cursor( p_cur_target_guid, 
                      p_cur_rollup_timestamps(p_url_1hour_idx),
                      p_cur_hour_rollup,
                      p_cur_max_response_time );
  LOOP

    -- Process batchsize number of rows per commit.
    FETCH rollup_cursor BULK COLLECT INTO v_rollup_rec.rollup_time,
                                          v_rollup_rec.metric_name,
                                          v_rollup_rec.key_val,
                                          v_rollup_rec.url_base,
                                          v_rollup_rec.hits,
                                          v_rollup_rec.avg_val,
                                          v_rollup_rec.min_val,
                                          v_rollup_rec.max_val,
                                          v_rollup_rec.var_val,
                                          v_rollup_rec.svr_avg_val,
                                          v_rollup_rec.svr_min_val,
                                          v_rollup_rec.svr_max_val,
                                          v_rollup_rec.svr_var_val,
                                          v_rollup_rec.db_avg_val,
                                          v_rollup_rec.db_min_val,
                                          v_rollup_rec.db_max_val,
                                          v_rollup_rec.db_var_val
                                          LIMIT p_rollup_batchsize;

    -- Exit if no rollup records are generated.
    EXIT WHEN NOT v_rollup_rec.rollup_time.EXISTS(1);

    -- Bulk inserts
    FORALL i IN v_rollup_rec.rollup_time.FIRST..v_rollup_rec.rollup_time.LAST
      INSERT INTO MGMT_RT_URL_1HOUR
      ( 
        target_guid, metric_name, 
        rollup_timestamp,
        url_filename,  
        url_link, hits,
        response_time_average, response_time_minimum, 
        response_time_maximum, response_time_sdev, 
        response_time_variance,
        server_time_average, server_time_minimum, 
        server_time_maximum, server_time_sdev, 
        server_time_variance,
        db_time_average, db_time_minimum, 
        db_time_maximum, db_time_sdev, 
        db_time_variance
      )
      VALUES
      ( 
        p_cur_target_guid, v_rollup_rec.metric_name(i),
        v_rollup_rec.rollup_time(i),
        v_rollup_rec.key_val(i), v_rollup_rec.url_base(i) || v_rollup_rec.key_val(i), 
        v_rollup_rec.hits(i), 
        v_rollup_rec.avg_val(i), v_rollup_rec.min_val(i), v_rollup_rec.max_val(i),  
        SQRT(v_rollup_rec.var_val(i)), v_rollup_rec.var_val(i),
        v_rollup_rec.svr_avg_val(i), v_rollup_rec.svr_min_val(i), v_rollup_rec.svr_max_val(i),  
        SQRT(v_rollup_rec.svr_var_val(i)), v_rollup_rec.svr_var_val(i),
        v_rollup_rec.db_avg_val(i), v_rollup_rec.db_min_val(i), v_rollup_rec.db_max_val(i),  
        SQRT(v_rollup_rec.db_var_val(i)), v_rollup_rec.db_var_val(i)
      );

    COMMIT;

    -- Keep track of the number of inserts
    v_rc := v_rc + v_rollup_rec.key_val.COUNT;

    -- Free temporary tables
    v_rollup_rec.rollup_time.DELETE;
    v_rollup_rec.metric_name.DELETE;
    v_rollup_rec.key_val.DELETE;
    v_rollup_rec.url_base.DELETE;
    v_rollup_rec.hits.DELETE;
    v_rollup_rec.avg_val.DELETE;
    v_rollup_rec.min_val.DELETE;
    v_rollup_rec.max_val.DELETE;
    v_rollup_rec.var_val.DELETE;
    v_rollup_rec.svr_avg_val.DELETE;
    v_rollup_rec.svr_min_val.DELETE;
    v_rollup_rec.svr_max_val.DELETE;
    v_rollup_rec.svr_var_val.DELETE;
    v_rollup_rec.db_avg_val.DELETE;
    v_rollup_rec.db_min_val.DELETE;
    v_rollup_rec.db_max_val.DELETE;
    v_rollup_rec.db_var_val.DELETE;

    -- Check cursor status here because cursor may return not found on a
    -- bulk fetch, even though records were retrieved.
    EXIT WHEN rollup_cursor%NOTFOUND;

  END LOOP; 

  -- delete all records from the table that are covered by this rollup 
  DELETE FROM MGMT_RT_URL_BOOTSTRAP
  WHERE target_guid = p_cur_target_guid
    AND rollup_timestamp <= p_cur_hour_rollup;

  -- Update the rollup timestamp
  UPDATE MGMT_TARGET_ROLLUP_TIMES
    SET rollup_timestamp = p_cur_hour_rollup
    WHERE rowid = p_cur_rollup_rowids(p_url_1hour_idx);

  COMMIT;

  IF rollup_cursor%ISOPEN THEN
    CLOSE rollup_cursor;
  END IF;

  -- Update the global row counters
  p_total_rc := p_total_rc + v_rc;
  p_url_1hour_rollup_rc := p_url_1hour_rollup_rc + v_rc;

EXCEPTION

  WHEN DUP_VAL_ON_INDEX THEN
    -- Partial rollup detected.

    -- Rollback any partial work performed in this procedure
    ROLLBACK;

    -- Free all resources used
    IF rollup_cursor%ISOPEN THEN
      CLOSE rollup_cursor;
    END IF;
    v_rollup_rec.rollup_time.DELETE;
    v_rollup_rec.metric_name.DELETE;
    v_rollup_rec.key_val.DELETE;
    v_rollup_rec.url_base.DELETE;
    v_rollup_rec.hits.DELETE;
    v_rollup_rec.avg_val.DELETE;
    v_rollup_rec.min_val.DELETE;
    v_rollup_rec.max_val.DELETE;
    v_rollup_rec.var_val.DELETE;
    v_rollup_rec.svr_avg_val.DELETE;
    v_rollup_rec.svr_min_val.DELETE;
    v_rollup_rec.svr_max_val.DELETE;
    v_rollup_rec.svr_var_val.DELETE;
    v_rollup_rec.db_avg_val.DELETE;
    v_rollup_rec.db_min_val.DELETE;
    v_rollup_rec.db_max_val.DELETE;
    v_rollup_rec.db_var_val.DELETE;

    -- Cleanup all partial work.  If the cleanup fails, update the rollup
    -- timestamp so that the rollup doesn't get stuck forever.
    IF (EMD_URL_1HOUR_CLEANUP = TRUE) THEN
      MGMT_LOG.LOG_ERROR('RTROLLUP', null,
        'Successfully cleaned up partial 1hour URL rollup (target=' ||
        p_cur_target_guid || ')', 'MGMT', NULL, NULL, NULL, MGMT_GLOBAL.G_INFO);
    ELSE
      MGMT_LOG.LOG_ERROR('RTROLLUP', null,
        'Unable to cleanup the partial 1hour URL rollup (target=' ||
        p_cur_target_guid || ')');
      BEGIN
        UPDATE MGMT_TARGET_ROLLUP_TIMES
          SET rollup_timestamp = p_cur_hour_rollup
          WHERE rowid = p_cur_rollup_rowids(p_url_1hour_idx);
        COMMIT;
        MGMT_LOG.LOG_ERROR('RTROLLUP', null,
          'Rollups for the 1hour URL period [' ||
          TO_CHAR(p_cur_rollup_timestamps(p_url_1hour_idx),
          'MM:DD:YYYY HH24:MI') || ', ' ||
          TO_CHAR(p_cur_hour_rollup, 'MM:DD:YYYY HH24:MI') || '] 
          cannot be performed (target=' || p_cur_target_guid || ')');
      EXCEPTION
        WHEN OTHERS THEN
          MGMT_LOG.LOG_ERROR('RTROLLUP', null,
            'Fatal Error: Unable to set the 1hour URL timestamp (target=' ||
            p_cur_target_guid || '). Error: ' || SUBSTR(SQLERRM, 1, 1000));
          RAISE;
      END;
    END IF;

  WHEN OTHERS THEN
    -- Assumes permanent error and attempts to update the rollup timestamp,
    -- thus losing the rollup period in question.
    -- This should be improved upon, we need to determine if the error is
    -- permanent or transient and only update the timestamp if it's permanent.

    -- Rollback any partial work
    ROLLBACK;

    -- Free all resources used
    IF rollup_cursor%ISOPEN THEN
      CLOSE rollup_cursor;
    END IF;
    v_rollup_rec.rollup_time.DELETE;
    v_rollup_rec.metric_name.DELETE;
    v_rollup_rec.key_val.DELETE;
    v_rollup_rec.url_base.DELETE;
    v_rollup_rec.hits.DELETE;
    v_rollup_rec.avg_val.DELETE;
    v_rollup_rec.min_val.DELETE;
    v_rollup_rec.max_val.DELETE;
    v_rollup_rec.var_val.DELETE;
    v_rollup_rec.svr_avg_val.DELETE;
    v_rollup_rec.svr_min_val.DELETE;
    v_rollup_rec.svr_max_val.DELETE;
    v_rollup_rec.svr_var_val.DELETE;
    v_rollup_rec.db_avg_val.DELETE;
    v_rollup_rec.db_min_val.DELETE;
    v_rollup_rec.db_max_val.DELETE;
    v_rollup_rec.db_var_val.DELETE;

    MGMT_LOG.LOG_ERROR('RTROLLUP', null,
      'The following unrecoverable error occurred while performing the' ||
      ' 1hour URL rollup (target=' || p_cur_target_guid || ').  Attempting to' ||
      ' skip over rollup period [' ||
      TO_CHAR(p_cur_rollup_timestamps(p_url_1hour_idx),
      'MM:DD:YYYY HH24:MI') || ', ' ||
      TO_CHAR(p_cur_hour_rollup,
      'MM:DD:YYYY HH24:MI') || '].  Error: ' || SUBSTR(SQLERRM, 1, 1000));

    BEGIN
      UPDATE MGMT_TARGET_ROLLUP_TIMES
        SET rollup_timestamp = p_cur_hour_rollup
        WHERE rowid = p_cur_rollup_rowids(p_url_1hour_idx);
      COMMIT;
      MGMT_LOG.LOG_ERROR('RTROLLUP', null,
        'Successfully skipped over period.  1hour URL Rollup data will not be' ||
        ' generated for this period (target=' || p_cur_target_guid || ').');
    EXCEPTION
      WHEN OTHERS THEN
        MGMT_LOG.LOG_ERROR('RTROLLUP', null,
          'Fatal Error: Unable to set the 1hour URL timestamp (target=' ||
          p_cur_target_guid || '). Error: ' || SUBSTR(SQLERRM, 1, 1000));
        RAISE;
    END;

END EMD_URL_1HOUR_ROLLUP;
----------------------------------------------------------------------

----------------------------------------------------------------------
PROCEDURE EMD_URL_DIST_1HOUR_ROLLUP IS

  v_rollup_rec      p_dist_rollup_record_type;
  v_rc              PLS_INTEGER := 0; 

  CURSOR et_rollup_cursor ( c_target_guid RAW,
                         c_prev_rollup DATE,
                         c_cur_rollup DATE,
                         c_max_elapsed_time NUMBER ) IS
    SELECT TRUNC(collection_timestamp, 'HH24'), metric_name,
      url_filename, 
      COUNT(target_guid),
      ROUND(elapsed_time/1000)
    FROM MGMT_RT_METRICS_RAW
    WHERE target_guid = c_target_guid
      AND TRUNC(collection_timestamp, 'HH24') > c_prev_rollup 
      AND TRUNC(collection_timestamp, 'HH24') <= c_cur_rollup
      AND elapsed_time <= c_max_elapsed_time
    GROUP BY TRUNC(collection_timestamp, 'HH24'), metric_name, url_filename,
          ROUND(elapsed_time/1000);

  CURSOR svt_rollup_cursor ( c_target_guid RAW,
                         c_prev_rollup DATE,
                         c_cur_rollup DATE,
                         c_max_elapsed_time NUMBER ) IS
    SELECT TRUNC(collection_timestamp, 'HH24'), metric_name,
      url_filename, 
      COUNT(target_guid),
      ROUND(server_latency_time/1000)
    FROM MGMT_RT_METRICS_RAW
    WHERE target_guid = c_target_guid
      AND TRUNC(collection_timestamp, 'HH24') > c_prev_rollup 
      AND TRUNC(collection_timestamp, 'HH24') <= c_cur_rollup
      AND elapsed_time <= c_max_elapsed_time
    GROUP BY TRUNC(collection_timestamp, 'HH24'), metric_name, url_filename,
          ROUND(server_latency_time/1000);
  
  CURSOR db_rollup_cursor ( c_target_guid RAW,
                         c_prev_rollup DATE,
                         c_cur_rollup DATE,
                         c_max_elapsed_time NUMBER ) IS
    SELECT TRUNC(collection_timestamp, 'HH24'), metric_name,
      url_filename, 
      COUNT(target_guid),
      ROUND(database_time/1000)
    FROM MGMT_RT_METRICS_RAW
    WHERE target_guid = c_target_guid
      AND TRUNC(collection_timestamp, 'HH24') > c_prev_rollup 
      AND TRUNC(collection_timestamp, 'HH24') <= c_cur_rollup
      AND elapsed_time <= c_max_elapsed_time
      AND database_time IS NOT NULL
    GROUP BY TRUNC(collection_timestamp, 'HH24'), metric_name, url_filename,
          ROUND(database_time/1000);

  BEGIN

  OPEN et_rollup_cursor( p_cur_target_guid, 
                      p_cur_rollup_timestamps(p_url_dist_1hour_idx),
                      p_cur_hour_rollup,
                      p_cur_max_response_time );
  LOOP

    -- Process batchsize number of rows per commit.
  FETCH et_rollup_cursor BULK COLLECT INTO v_rollup_rec.rollup_time,
                                          v_rollup_rec.metric_name,
                                          v_rollup_rec.key_val,
                                          v_rollup_rec.hits,
                                          v_rollup_rec.dist_period
                                          LIMIT p_rollup_batchsize;
    EXIT WHEN NOT v_rollup_rec.rollup_time.EXISTS(1);

    -- Bulk inserts
    FORALL i IN v_rollup_rec.rollup_time.FIRST..v_rollup_rec.rollup_time.LAST

          INSERT INTO MGMT_RT_URL_DIST_1HOUR
          ( 
            target_guid, metric_name, 
            rollup_timestamp,
            url_filename, 
            hits, num_seconds, dist_value_type
          )
          VALUES
          ( 
            p_cur_target_guid, v_rollup_rec.metric_name(i),
            v_rollup_rec.rollup_time(i),
            v_rollup_rec.key_val(i), 
            v_rollup_rec.hits(i), 
            v_rollup_rec.dist_period(i), p_et_dist_type
          ); 
    COMMIT;

    -- Keep track of the number of inserts
    v_rc := v_rc + v_rollup_rec.key_val.COUNT;

    -- Free temporary tables
    v_rollup_rec.rollup_time.DELETE;
    v_rollup_rec.metric_name.DELETE;
    v_rollup_rec.key_val.DELETE;
    v_rollup_rec.hits.DELETE;
    v_rollup_rec.dist_period.DELETE;

    EXIT WHEN et_rollup_cursor%NOTFOUND;
  END LOOP;

  OPEN svt_rollup_cursor( p_cur_target_guid, 
                      p_cur_rollup_timestamps(p_url_dist_1hour_idx),
                      p_cur_hour_rollup,
                      p_cur_max_response_time );
  LOOP

    -- Process batchsize number of rows per commit.
  FETCH svt_rollup_cursor BULK COLLECT INTO v_rollup_rec.rollup_time,
                                          v_rollup_rec.metric_name,
                                          v_rollup_rec.key_val,
                                          v_rollup_rec.hits,
                                          v_rollup_rec.dist_period
                                          LIMIT p_rollup_batchsize;

    EXIT WHEN NOT v_rollup_rec.rollup_time.EXISTS(1);
    -- Bulk inserts
    FORALL i IN v_rollup_rec.rollup_time.FIRST..v_rollup_rec.rollup_time.LAST

          INSERT INTO MGMT_RT_URL_DIST_1HOUR
          ( 
            target_guid, metric_name, 
            rollup_timestamp,
            url_filename, 
            hits, num_seconds, dist_value_type
          )
          VALUES
          ( 
            p_cur_target_guid, v_rollup_rec.metric_name(i),
            v_rollup_rec.rollup_time(i),
            v_rollup_rec.key_val(i), 
            v_rollup_rec.hits(i), 
            v_rollup_rec.dist_period(i), p_svt_dist_type
          ); 
    COMMIT;

    -- Keep track of the number of inserts
    v_rc := v_rc + v_rollup_rec.key_val.COUNT;

    -- Free temporary tables
    v_rollup_rec.rollup_time.DELETE;
    v_rollup_rec.metric_name.DELETE;
    v_rollup_rec.key_val.DELETE;
    v_rollup_rec.hits.DELETE;
    v_rollup_rec.dist_period.DELETE;

    EXIT WHEN svt_rollup_cursor%NOTFOUND;
  END LOOP;

  OPEN db_rollup_cursor( p_cur_target_guid, 
                      p_cur_rollup_timestamps(p_url_dist_1hour_idx),
                      p_cur_hour_rollup,
                      p_cur_max_response_time );
  LOOP

    -- Process batchsize number of rows per commit.
  FETCH db_rollup_cursor BULK COLLECT INTO v_rollup_rec.rollup_time,
                                          v_rollup_rec.metric_name,
                                          v_rollup_rec.key_val,
                                          v_rollup_rec.hits,
                                          v_rollup_rec.dist_period
                                          LIMIT p_rollup_batchsize;

    EXIT WHEN NOT v_rollup_rec.rollup_time.EXISTS(1);
    -- Bulk inserts
    FORALL i IN v_rollup_rec.rollup_time.FIRST..v_rollup_rec.rollup_time.LAST

          INSERT INTO MGMT_RT_URL_DIST_1HOUR
          ( 
            target_guid, metric_name, 
            rollup_timestamp,
            url_filename, 
            hits, num_seconds, dist_value_type
          )
          VALUES
          ( 
            p_cur_target_guid, v_rollup_rec.metric_name(i),
            v_rollup_rec.rollup_time(i),
            v_rollup_rec.key_val(i), 
            v_rollup_rec.hits(i), 
            v_rollup_rec.dist_period(i), p_db_dist_type
          ); 
    COMMIT;

    -- Keep track of the number of inserts
    v_rc := v_rc + v_rollup_rec.key_val.COUNT;

    -- Free temporary tables
    v_rollup_rec.rollup_time.DELETE;
    v_rollup_rec.metric_name.DELETE;
    v_rollup_rec.key_val.DELETE;
    v_rollup_rec.hits.DELETE;
    v_rollup_rec.dist_period.DELETE;

    EXIT WHEN db_rollup_cursor%NOTFOUND;
  END LOOP;

  -- delete all records from the table that are covered by this rollup 
  DELETE FROM MGMT_RT_URL_DIST_BOOTSTRAP
  WHERE target_guid = p_cur_target_guid
    AND rollup_timestamp <= p_cur_hour_rollup;

  -- Update the rollup timestamp
  UPDATE MGMT_TARGET_ROLLUP_TIMES
    SET rollup_timestamp = p_cur_hour_rollup
    WHERE rowid = p_cur_rollup_rowids(p_url_dist_1hour_idx);

  COMMIT;

  IF et_rollup_cursor%ISOPEN THEN
    CLOSE et_rollup_cursor;
  END IF;
  IF svt_rollup_cursor%ISOPEN THEN
    CLOSE svt_rollup_cursor;
  END IF;
  IF db_rollup_cursor%ISOPEN THEN
    CLOSE db_rollup_cursor;
  END IF;

  -- Update the global row counters
  p_total_rc := p_total_rc + v_rc;
  p_url_dist_1hour_rollup_rc := p_url_dist_1hour_rollup_rc + v_rc;

  EXCEPTION
  

  WHEN DUP_VAL_ON_INDEX THEN
    -- Partial rollup detected.

    -- Rollback any partial work performed in this procedure
    ROLLBACK;

    -- Free all resources used
    IF et_rollup_cursor%ISOPEN THEN
      CLOSE et_rollup_cursor;
    END IF;
    IF svt_rollup_cursor%ISOPEN THEN
      CLOSE svt_rollup_cursor;
    END IF;
    IF db_rollup_cursor%ISOPEN THEN
      CLOSE db_rollup_cursor;
    END IF;

    v_rollup_rec.rollup_time.DELETE;
    v_rollup_rec.metric_name.DELETE;
    v_rollup_rec.key_val.DELETE;
    v_rollup_rec.hits.DELETE;
    v_rollup_rec.dist_period.DELETE;

    -- Cleanup all partial work.  If the cleanup fails, update the rollup
    -- timestamp so that the rollup doesn't get stuck forever.
    IF (EMD_URL_DIST_1HOUR_CLEANUP = TRUE) THEN
      MGMT_LOG.LOG_ERROR('RTROLLUP', null,
        'Successfully cleaned up partial 1hour dist URL rollup (target=' ||
        p_cur_target_guid || ')', 'MGMT', NULL, NULL, NULL, MGMT_GLOBAL.G_INFO);
    ELSE
      MGMT_LOG.LOG_ERROR('RTROLLUP', null,
        'Unable to cleanup the partial 1hour dist URL rollup (target=' ||
        p_cur_target_guid || ')');
      BEGIN
        UPDATE MGMT_TARGET_ROLLUP_TIMES
          SET rollup_timestamp = p_cur_hour_rollup
          WHERE rowid = p_cur_rollup_rowids(p_url_dist_1hour_idx);
        COMMIT;
        MGMT_LOG.LOG_ERROR('RTROLLUP', null,
          'Rollups for the 1hour dist URL period [' || 
          TO_CHAR(p_cur_rollup_timestamps(p_url_dist_1hour_idx),
          'MM:DD:YYYY HH24:MI') || ', ' ||
          TO_CHAR(p_cur_hour_rollup, 'MM:DD:YYYY HH24:MI') || '] 
          cannot be performed (target=' || p_cur_target_guid || ')');
      EXCEPTION
        WHEN OTHERS THEN
          MGMT_LOG.LOG_ERROR('RTROLLUP', null,
            'Fatal Error: Unable to set the 1hour dist URL timestamp (target=' ||
            p_cur_target_guid || '). Error: ' || SUBSTR(SQLERRM, 1, 1000));
          RAISE;
      END;
    END IF;

  WHEN OTHERS THEN
    -- Assumes permanent error and attempts to update the rollup timestamp,
    -- thus losing the rollup period in question.
    -- This should be improved upon, we need to determine if the error is
    -- permanent or transient and only update the timestamp if it's permanent.

    -- Rollback any partial work
    ROLLBACK;

    -- Free all resources used
    IF et_rollup_cursor%ISOPEN THEN
      CLOSE et_rollup_cursor;
    END IF; 
    IF svt_rollup_cursor%ISOPEN THEN
      CLOSE svt_rollup_cursor;
    END IF;
    IF db_rollup_cursor%ISOPEN THEN
      CLOSE db_rollup_cursor;
    END IF;
    v_rollup_rec.rollup_time.DELETE;
    v_rollup_rec.metric_name.DELETE;
    v_rollup_rec.key_val.DELETE;
    v_rollup_rec.hits.DELETE;
    v_rollup_rec.dist_period.DELETE;

    MGMT_LOG.LOG_ERROR('RTROLLUP', null,
      'The following unrecoverable error occurred while performing the' ||
      ' 1hour dist URL rollup (target=' || p_cur_target_guid || ').  Attempting to' ||
      ' skip over rollup period [' || 
      TO_CHAR(p_cur_rollup_timestamps(p_url_dist_1hour_idx),
      'MM:DD:YYYY HH24:MI') || ', ' ||
      TO_CHAR(p_cur_hour_rollup,
      'MM:DD:YYYY HH24:MI') || '].  Error: ' || SUBSTR(SQLERRM, 1, 1000));
   
    BEGIN
      UPDATE MGMT_TARGET_ROLLUP_TIMES
        SET rollup_timestamp = p_cur_hour_rollup
        WHERE rowid = p_cur_rollup_rowids(p_url_dist_1hour_idx);
      COMMIT;
      MGMT_LOG.LOG_ERROR('RTROLLUP', null,
        'Successfully skipped over period.  1hour dist URL rollup data will not be' ||
        ' generated for this period (target=' || p_cur_target_guid || ').');
    EXCEPTION
      WHEN OTHERS THEN
        MGMT_LOG.LOG_ERROR('RTROLLUP', null,
          'Fatal Error: Unable to set the 1hour dist URL timestamp (target=' ||
          p_cur_target_guid || '). Error: ' || SUBSTR(SQLERRM, 1, 1000));
        RAISE;
    END;

END EMD_URL_DIST_1HOUR_ROLLUP;
----------------------------------------------------------------------

----------------------------------------------------------------------
PROCEDURE EMD_DOM_1HOUR_ROLLUP IS

  v_rollup_rec      p_hour_rollup_record_type;
  v_rc              PLS_INTEGER := 0;

  CURSOR rollup_cursor ( c_target_guid RAW,
                         c_prev_rollup DATE,
                         c_cur_rollup DATE,
                         c_max_elapsed_time NUMBER ) IS
    SELECT TRUNC(collection_timestamp, 'HH24'), metric_name,
      visitor_domain,
      SUBSTR(visitor_ip, 1, INSTR(visitor_ip, '.', 1, 3)-1),
      BITAND(visitor_ip_num, p_subnet_mask),
      COUNT(visitor_domain), 
      AVG(elapsed_time), MIN(elapsed_time), 
      MAX(elapsed_time), VARIANCE(elapsed_time),
      AVG(server_latency_time), MIN(server_latency_time), 
      MAX(server_latency_time), VARIANCE(server_latency_time), 
      AVG(database_time), MIN(database_time), 
      MAX(database_time), VARIANCE(database_time)
    FROM MGMT_RT_METRICS_RAW
    WHERE target_guid = c_target_guid
      AND TRUNC(collection_timestamp, 'HH24') > c_prev_rollup
      AND TRUNC(collection_timestamp, 'HH24') <= c_cur_rollup
      AND elapsed_time <= c_max_elapsed_time
    GROUP BY TRUNC(collection_timestamp, 'HH24'), metric_name, 
      visitor_domain, 
      SUBSTR(visitor_ip, 1, INSTR(visitor_ip, '.', 1, 3)-1),
      BITAND(visitor_ip_num, p_subnet_mask);

BEGIN

  OPEN rollup_cursor( p_cur_target_guid, 
                      p_cur_rollup_timestamps(p_dom_1hour_idx),
                      p_cur_hour_rollup,
                      p_cur_max_response_time );
  LOOP

    -- Process batchsize number of rows per commit.
    FETCH rollup_cursor BULK COLLECT INTO v_rollup_rec.rollup_time,
                                          v_rollup_rec.metric_name,
                                          v_rollup_rec.key_val,
                                          v_rollup_rec.subnet,
                                          v_rollup_rec.subnet_num,
                                          v_rollup_rec.hits,
                                          v_rollup_rec.avg_val,
                                          v_rollup_rec.min_val,
                                          v_rollup_rec.max_val,
                                          v_rollup_rec.var_val,
                                          v_rollup_rec.svr_avg_val,
                                          v_rollup_rec.svr_min_val,
                                          v_rollup_rec.svr_max_val,
                                          v_rollup_rec.svr_var_val,
                                          v_rollup_rec.db_avg_val,
                                          v_rollup_rec.db_min_val,
                                          v_rollup_rec.db_max_val,
                                          v_rollup_rec.db_var_val
                                          LIMIT p_rollup_batchsize;

    -- Exit if no rollup records are generated.
    EXIT WHEN NOT v_rollup_rec.rollup_time.EXISTS(1);

    -- Bulk inserts
    FORALL i IN v_rollup_rec.rollup_time.FIRST..v_rollup_rec.rollup_time.LAST
      INSERT INTO MGMT_RT_DOMAIN_1HOUR
      ( 
        target_guid, metric_name, 
        rollup_timestamp,
	visitor_domain,
        visitor_subnet, visitor_subnet_num, hits, 
        response_time_average, 
        response_time_minimum, response_time_maximum, 
        response_time_sdev, response_time_variance,
        server_time_average, server_time_minimum, 
        server_time_maximum, server_time_sdev, 
        server_time_variance,
        db_time_average, db_time_minimum, 
        db_time_maximum, db_time_sdev, 
        db_time_variance
      )
      VALUES
      ( 
        p_cur_target_guid, v_rollup_rec.metric_name(i),
        v_rollup_rec.rollup_time(i),
        v_rollup_rec.key_val(i), 
	v_rollup_rec.subnet(i), v_rollup_rec.subnet_num(i),
        v_rollup_rec.hits(i), v_rollup_rec.avg_val(i), 
        v_rollup_rec.min_val(i), v_rollup_rec.max_val(i),  
        SQRT(v_rollup_rec.var_val(i)), v_rollup_rec.var_val(i),
        v_rollup_rec.svr_avg_val(i), v_rollup_rec.svr_min_val(i), 
        v_rollup_rec.svr_max_val(i), SQRT(v_rollup_rec.svr_var_val(i)), 
        v_rollup_rec.svr_var_val(i),
        v_rollup_rec.db_avg_val(i), v_rollup_rec.db_min_val(i), 
        v_rollup_rec.db_max_val(i), SQRT(v_rollup_rec.db_var_val(i)), 
        v_rollup_rec.db_var_val(i)
      );

    COMMIT;

    -- Keep track of the number of inserts
    v_rc := v_rc + v_rollup_rec.key_val.COUNT;

    -- Free temporary tables
    v_rollup_rec.rollup_time.DELETE;
    v_rollup_rec.metric_name.DELETE;
    v_rollup_rec.key_val.DELETE;
    v_rollup_rec.subnet.DELETE;
    v_rollup_rec.subnet_num.DELETE;
    v_rollup_rec.hits.DELETE;
    v_rollup_rec.avg_val.DELETE;
    v_rollup_rec.min_val.DELETE;
    v_rollup_rec.max_val.DELETE;
    v_rollup_rec.var_val.DELETE;
    v_rollup_rec.svr_avg_val.DELETE;
    v_rollup_rec.svr_min_val.DELETE;
    v_rollup_rec.svr_max_val.DELETE;
    v_rollup_rec.svr_var_val.DELETE;
    v_rollup_rec.db_avg_val.DELETE;
    v_rollup_rec.db_min_val.DELETE;
    v_rollup_rec.db_max_val.DELETE;
    v_rollup_rec.db_var_val.DELETE;

    -- Check cursor status here because cursor may return not found on a
    -- bulk fetch, even though records were retrieved.
    EXIT WHEN rollup_cursor%NOTFOUND;

  END LOOP; 

  -- delete all records from the table that are covered by the other rollup 
  DELETE FROM MGMT_RT_DOMAIN_BOOTSTRAP
  WHERE target_guid = p_cur_target_guid
    AND rollup_timestamp <= p_cur_hour_rollup;

  -- Update the rollup timestamp
  UPDATE MGMT_TARGET_ROLLUP_TIMES
    SET rollup_timestamp = p_cur_hour_rollup
    WHERE rowid = p_cur_rollup_rowids(p_dom_1hour_idx);

  COMMIT;

  IF rollup_cursor%ISOPEN THEN
    CLOSE rollup_cursor;
  END IF;

  -- Update the global row counters
  p_total_rc := p_total_rc + v_rc;
  p_dom_1hour_rollup_rc := p_dom_1hour_rollup_rc + v_rc;

EXCEPTION

  WHEN DUP_VAL_ON_INDEX THEN
    -- Partial rollup detected.

    -- Rollback any partial work performed in this procedure
    ROLLBACK;

    -- Free all resources used
    IF rollup_cursor%ISOPEN THEN
      CLOSE rollup_cursor;
    END IF;
    v_rollup_rec.rollup_time.DELETE;
    v_rollup_rec.metric_name.DELETE;
    v_rollup_rec.key_val.DELETE;
    v_rollup_rec.subnet.DELETE;
    v_rollup_rec.hits.DELETE;
    v_rollup_rec.avg_val.DELETE;
    v_rollup_rec.min_val.DELETE;
    v_rollup_rec.max_val.DELETE;
    v_rollup_rec.var_val.DELETE;
    v_rollup_rec.svr_avg_val.DELETE;
    v_rollup_rec.svr_min_val.DELETE;
    v_rollup_rec.svr_max_val.DELETE;
    v_rollup_rec.svr_var_val.DELETE;
    v_rollup_rec.db_avg_val.DELETE;
    v_rollup_rec.db_min_val.DELETE;
    v_rollup_rec.db_max_val.DELETE;
    v_rollup_rec.db_var_val.DELETE;

    -- Cleanup all partial work.  If the cleanup fails, update the rollup
    -- timestamp so that the rollup doesn't get stuck forever.
    IF (EMD_DOM_1HOUR_CLEANUP = TRUE) THEN
      MGMT_LOG.LOG_ERROR('RTROLLUP', null,
        'Successfully cleaned up partial dom 1hour rollup (target=' ||
        p_cur_target_guid || ')', 'MGMT', NULL, NULL, NULL, MGMT_GLOBAL.G_INFO);
    ELSE
      MGMT_LOG.LOG_ERROR('RTROLLUP', null,
        'Unable to cleanup the partial dom 1hour rollup (target=' ||
        p_cur_target_guid || ')');
      BEGIN
        UPDATE MGMT_TARGET_ROLLUP_TIMES
          SET rollup_timestamp = p_cur_hour_rollup
          WHERE rowid = p_cur_rollup_rowids(p_dom_1hour_idx);
        COMMIT;
        MGMT_LOG.LOG_ERROR('RTROLLUP', null,
          'Dom 1hour rollups for the period [' ||
          TO_CHAR(p_cur_rollup_timestamps(p_dom_1hour_idx),
          'MM:DD:YYYY HH24:MI') || ', ' ||
          TO_CHAR(p_cur_hour_rollup, 'MM:DD:YYYY HH24:MI') || '] 
          cannot be performed (target=' || p_cur_target_guid || ')');
      EXCEPTION
        WHEN OTHERS THEN
          MGMT_LOG.LOG_ERROR('RTROLLUP', null,
            'Fatal Error: Unable to set the dom 1hour timestamp (target=' ||
            p_cur_target_guid || '). Error: ' || SUBSTR(SQLERRM, 1, 1000));
          RAISE;
      END;
    END IF;

  WHEN OTHERS THEN
    -- Assumes permanent error and attempts to update the rollup timestamp,
    -- thus losing the rollup period in question.
    -- This should be improved upon, we need to determine if the error is
    -- permanent or transient and only update the timestamp if it's permanent.
   
    -- Rollback any partial work
    ROLLBACK;

    -- Free all resources used
    IF rollup_cursor%ISOPEN THEN
      CLOSE rollup_cursor;
    END IF;
    v_rollup_rec.rollup_time.DELETE;
    v_rollup_rec.metric_name.DELETE;
    v_rollup_rec.key_val.DELETE;
    v_rollup_rec.subnet.DELETE;
    v_rollup_rec.subnet_num.DELETE;
    v_rollup_rec.hits.DELETE;
    v_rollup_rec.avg_val.DELETE;
    v_rollup_rec.min_val.DELETE;
    v_rollup_rec.max_val.DELETE;
    v_rollup_rec.var_val.DELETE;
    v_rollup_rec.svr_avg_val.DELETE;
    v_rollup_rec.svr_min_val.DELETE;
    v_rollup_rec.svr_max_val.DELETE;
    v_rollup_rec.svr_var_val.DELETE;
    v_rollup_rec.db_avg_val.DELETE;
    v_rollup_rec.db_min_val.DELETE;
    v_rollup_rec.db_max_val.DELETE;
    v_rollup_rec.db_var_val.DELETE;

    MGMT_LOG.LOG_ERROR('RTROLLUP', null,
      'The following unrecoverable error occurred while performing the' ||
      ' dom 1hour rollup (target=' || p_cur_target_guid || ').  Attempting to' ||
      ' skip over rollup period [' || 
      TO_CHAR(p_cur_rollup_timestamps(p_dom_1hour_idx),
      'MM:DD:YYYY HH24:MI') || ', ' ||
      TO_CHAR(p_cur_hour_rollup,
      'MM:DD:YYYY HH24:MI') || '].  Error: ' || SUBSTR(SQLERRM, 1, 1000));
   
    BEGIN
      UPDATE MGMT_TARGET_ROLLUP_TIMES
        SET rollup_timestamp = p_cur_hour_rollup
        WHERE rowid = p_cur_rollup_rowids(p_dom_1hour_idx);
      COMMIT;
      MGMT_LOG.LOG_ERROR('RTROLLUP', null,
        'Successfully skipped over period.  Dom 1hour rollup data will not be' ||
        ' generated for this period (target=' || p_cur_target_guid || ').');
    EXCEPTION
      WHEN OTHERS THEN
        MGMT_LOG.LOG_ERROR('RTROLLUP', null,
          'Fatal Error: Unable to set the dom 1hour timestamp (target=' ||
          p_cur_target_guid || '). Error: ' || SUBSTR(SQLERRM, 1, 1000));
        RAISE;
    END;

END EMD_DOM_1HOUR_ROLLUP;
----------------------------------------------------------------------

----------------------------------------------------------------------
PROCEDURE EMD_DOM_DIST_1HOUR_ROLLUP IS

  v_rollup_rec      p_dist_rollup_record_type;
  v_rc              PLS_INTEGER := 0; 

  CURSOR et_rollup_cursor ( c_target_guid RAW,
                         c_prev_rollup DATE,
                         c_cur_rollup DATE,
                         c_max_elapsed_time NUMBER ) IS
    SELECT TRUNC(collection_timestamp, 'HH24'), metric_name,
      visitor_domain, 
      SUBSTR(visitor_ip, 1, INSTR(visitor_ip, '.', 1, 3)-1),
      BITAND(visitor_ip_num, p_subnet_mask),
      COUNT(target_guid),
      ROUND(elapsed_time/1000)
    FROM MGMT_RT_METRICS_RAW
    WHERE target_guid = c_target_guid
      AND TRUNC(collection_timestamp, 'HH24') > c_prev_rollup 
      AND TRUNC(collection_timestamp, 'HH24') <= c_cur_rollup
      AND elapsed_time <= c_max_elapsed_time
    GROUP BY TRUNC(collection_timestamp, 'HH24'), visitor_domain,
      metric_name,
      SUBSTR(visitor_ip, 1, INSTR(visitor_ip, '.', 1, 3)-1),
      BITAND(visitor_ip_num, p_subnet_mask),
      ROUND(elapsed_time/1000);

  CURSOR svt_rollup_cursor ( c_target_guid RAW,
                         c_prev_rollup DATE,
                         c_cur_rollup DATE,
                         c_max_elapsed_time NUMBER ) IS
    SELECT TRUNC(collection_timestamp, 'HH24'), metric_name,
      visitor_domain, 
      SUBSTR(visitor_ip, 1, INSTR(visitor_ip, '.', 1, 3)-1),
      BITAND(visitor_ip_num, p_subnet_mask),
      COUNT(target_guid),
      ROUND(server_latency_time/1000)
    FROM MGMT_RT_METRICS_RAW
    WHERE target_guid = c_target_guid
      AND TRUNC(collection_timestamp, 'HH24') > c_prev_rollup 
      AND TRUNC(collection_timestamp, 'HH24') <= c_cur_rollup
      AND elapsed_time <= c_max_elapsed_time
    GROUP BY TRUNC(collection_timestamp, 'HH24'), visitor_domain,
      metric_name,
      SUBSTR(visitor_ip, 1, INSTR(visitor_ip, '.', 1, 3)-1),
      BITAND(visitor_ip_num, p_subnet_mask),
      ROUND(server_latency_time/1000);

  CURSOR db_rollup_cursor ( c_target_guid RAW,
                         c_prev_rollup DATE,
                         c_cur_rollup DATE,
                         c_max_elapsed_time NUMBER ) IS
    SELECT TRUNC(collection_timestamp, 'HH24'), metric_name,
      visitor_domain, 
      SUBSTR(visitor_ip, 1, INSTR(visitor_ip, '.', 1, 3)-1),
      BITAND(visitor_ip_num, p_subnet_mask),
      COUNT(target_guid),
      ROUND(database_time/1000)
    FROM MGMT_RT_METRICS_RAW
    WHERE target_guid = c_target_guid
      AND TRUNC(collection_timestamp, 'HH24') > c_prev_rollup 
      AND TRUNC(collection_timestamp, 'HH24') <= c_cur_rollup
      AND elapsed_time <= c_max_elapsed_time
      AND database_time IS NOT NULL
    GROUP BY TRUNC(collection_timestamp, 'HH24'), visitor_domain,
      metric_name,
      SUBSTR(visitor_ip, 1, INSTR(visitor_ip, '.', 1, 3)-1),
      BITAND(visitor_ip_num, p_subnet_mask),
      ROUND(database_time/1000);

BEGIN

  OPEN et_rollup_cursor( p_cur_target_guid, 
                      p_cur_rollup_timestamps(p_dom_dist_1hour_idx),
                      p_cur_hour_rollup,
                      p_cur_max_response_time );
  LOOP

    FETCH et_rollup_cursor BULK COLLECT INTO v_rollup_rec.rollup_time,
                                          v_rollup_rec.metric_name,
                                          v_rollup_rec.key_val,
                                          v_rollup_rec.subnet,
                                          v_rollup_rec.subnet_num,
                                          v_rollup_rec.hits,
                                          v_rollup_rec.dist_period
                                          LIMIT p_rollup_batchsize;

    -- Exit if no rollup records are generated
    EXIT WHEN NOT v_rollup_rec.rollup_time.EXISTS(1);

    -- Bulk inserts
    FORALL i IN v_rollup_rec.rollup_time.FIRST..v_rollup_rec.rollup_time.LAST
          INSERT INTO MGMT_RT_DOMAIN_DIST_1HOUR
          ( 
            target_guid, metric_name, 
            rollup_timestamp,
            visitor_domain, visitor_subnet, visitor_subnet_num, 
            hits, num_seconds, dist_value_type
          )
          VALUES
          ( 
            p_cur_target_guid, v_rollup_rec.metric_name(i),
            v_rollup_rec.rollup_time(i),
            v_rollup_rec.key_val(i), 
            v_rollup_rec.subnet(i), v_rollup_rec.subnet_num(i),
            v_rollup_rec.hits(i), v_rollup_rec.dist_period(i), p_et_dist_type
          ); 

    COMMIT;

    -- Keep track of the number of inserts
    v_rc := v_rc + v_rollup_rec.key_val.COUNT;

    -- Free temporary tables
    v_rollup_rec.rollup_time.DELETE;
    v_rollup_rec.metric_name.DELETE;
    v_rollup_rec.key_val.DELETE;
    v_rollup_rec.subnet.DELETE;
    v_rollup_rec.subnet_num.DELETE;
    v_rollup_rec.hits.DELETE;
    v_rollup_rec.dist_period.DELETE;

    -- Check cursor status here because cursor may return not found on a
    -- bulk fetch, even though records were retrieved.
    EXIT WHEN et_rollup_cursor%NOTFOUND;

  END LOOP; 

  OPEN svt_rollup_cursor( p_cur_target_guid, 
                      p_cur_rollup_timestamps(p_url_dist_1hour_idx),
                      p_cur_hour_rollup,
                      p_cur_max_response_time );
  LOOP

    -- Process batchsize number of rows per commit.
  FETCH svt_rollup_cursor BULK COLLECT INTO v_rollup_rec.rollup_time,
                                          v_rollup_rec.metric_name,
                                          v_rollup_rec.key_val,
                                          v_rollup_rec.subnet,
                                          v_rollup_rec.subnet_num,
                                          v_rollup_rec.hits,
                                          v_rollup_rec.dist_period
                                          LIMIT p_rollup_batchsize;

    EXIT WHEN NOT v_rollup_rec.rollup_time.EXISTS(1);
    -- Bulk inserts
    FORALL i IN v_rollup_rec.rollup_time.FIRST..v_rollup_rec.rollup_time.LAST
          INSERT INTO MGMT_RT_DOMAIN_DIST_1HOUR
          ( 
            target_guid, metric_name, 
            rollup_timestamp,
            visitor_domain, visitor_subnet, visitor_subnet_num, 
            hits, num_seconds, dist_value_type
          )
          VALUES
          ( 
            p_cur_target_guid, v_rollup_rec.metric_name(i),
            v_rollup_rec.rollup_time(i),
            v_rollup_rec.key_val(i), 
            v_rollup_rec.subnet(i),
            v_rollup_rec.subnet_num(i),
            v_rollup_rec.hits(i), 
            v_rollup_rec.dist_period(i), p_svt_dist_type
          ); 
    COMMIT;

    -- Keep track of the number of inserts
    v_rc := v_rc + v_rollup_rec.key_val.COUNT;

    -- Free temporary tables
    v_rollup_rec.rollup_time.DELETE;
    v_rollup_rec.metric_name.DELETE;
    v_rollup_rec.key_val.DELETE;
    v_rollup_rec.subnet.DELETE;
    v_rollup_rec.subnet_num.DELETE;
    v_rollup_rec.hits.DELETE;
    v_rollup_rec.dist_period.DELETE;

    EXIT WHEN svt_rollup_cursor%NOTFOUND;

  END LOOP;

  OPEN db_rollup_cursor( p_cur_target_guid, 
                      p_cur_rollup_timestamps(p_url_dist_1hour_idx),
                      p_cur_hour_rollup,
                      p_cur_max_response_time );
  LOOP

    -- Process batchsize number of rows per commit.
  FETCH db_rollup_cursor BULK COLLECT INTO v_rollup_rec.rollup_time,
                                          v_rollup_rec.metric_name,
                                          v_rollup_rec.key_val,
                                          v_rollup_rec.subnet,
                                          v_rollup_rec.subnet_num,
                                          v_rollup_rec.hits,
                                          v_rollup_rec.dist_period
                                          LIMIT p_rollup_batchsize;

    EXIT WHEN NOT v_rollup_rec.rollup_time.EXISTS(1);
    -- Bulk inserts
    FORALL i IN v_rollup_rec.rollup_time.FIRST..v_rollup_rec.rollup_time.LAST
          INSERT INTO MGMT_RT_DOMAIN_DIST_1HOUR
          ( 
            target_guid, metric_name, 
            rollup_timestamp,
            visitor_domain, visitor_subnet, visitor_subnet_num, 
            hits, num_seconds, dist_value_type
          )
          VALUES
          ( 
            p_cur_target_guid, v_rollup_rec.metric_name(i),
            v_rollup_rec.rollup_time(i),
            v_rollup_rec.key_val(i), 
            v_rollup_rec.subnet(i),
            v_rollup_rec.subnet_num(i),
            v_rollup_rec.hits(i), 
            v_rollup_rec.dist_period(i), p_db_dist_type
          ); 
    COMMIT;

    -- Keep track of the number of inserts
    v_rc := v_rc + v_rollup_rec.key_val.COUNT;

    -- Free temporary tables
    v_rollup_rec.rollup_time.DELETE;
    v_rollup_rec.metric_name.DELETE;
    v_rollup_rec.key_val.DELETE;
    v_rollup_rec.subnet.DELETE;
    v_rollup_rec.subnet_num.DELETE;
    v_rollup_rec.hits.DELETE;
    v_rollup_rec.dist_period.DELETE;

    EXIT WHEN db_rollup_cursor%NOTFOUND;

  END LOOP;

  -- delete all records from the table that are covered by this rollup 
  DELETE FROM MGMT_RT_DOMAIN_DIST_BOOTSTRAP
  WHERE target_guid = p_cur_target_guid
    AND rollup_timestamp <= p_cur_hour_rollup;

  -- Update the rollup timestamp
  UPDATE MGMT_TARGET_ROLLUP_TIMES
    SET rollup_timestamp = p_cur_hour_rollup
    WHERE rowid = p_cur_rollup_rowids(p_dom_dist_1hour_idx);

  COMMIT;

  IF et_rollup_cursor%ISOPEN THEN
    CLOSE et_rollup_cursor;
  END IF;
  IF svt_rollup_cursor%ISOPEN THEN
    CLOSE svt_rollup_cursor;
  END IF;
  IF db_rollup_cursor%ISOPEN THEN
    CLOSE db_rollup_cursor;
  END IF;

  -- Update the global row counters
  p_total_rc := p_total_rc + v_rc;
  p_dom_dist_1hour_rollup_rc := p_dom_dist_1hour_rollup_rc + v_rc;

EXCEPTION

  WHEN DUP_VAL_ON_INDEX THEN
    -- Partial rollup detected.

    -- Rollback any partial work performed in this procedure
    ROLLBACK;

    -- Free all resources used
    IF et_rollup_cursor%ISOPEN THEN
      CLOSE et_rollup_cursor;
    END IF;
    IF svt_rollup_cursor%ISOPEN THEN
      CLOSE svt_rollup_cursor;
    END IF;
    IF db_rollup_cursor%ISOPEN THEN
      CLOSE db_rollup_cursor;
    END IF;

    v_rollup_rec.rollup_time.DELETE;
    v_rollup_rec.metric_name.DELETE;
    v_rollup_rec.key_val.DELETE;
    v_rollup_rec.subnet.DELETE;
    v_rollup_rec.subnet_num.DELETE;
    v_rollup_rec.hits.DELETE;
    v_rollup_rec.dist_period.DELETE;

    -- Cleanup all partial work.  If the cleanup fails, update the rollup
    -- timestamp so that the rollup doesn't get stuck forever.
    IF (EMD_DOM_DIST_1HOUR_CLEANUP = TRUE) THEN
      MGMT_LOG.LOG_ERROR('RTROLLUP', null,
        'Successfully cleaned up partial dom 1hour dist rollup (target=' ||
        p_cur_target_guid || ')', 'MGMT', NULL, NULL, NULL, MGMT_GLOBAL.G_INFO);
    ELSE
      MGMT_LOG.LOG_ERROR('RTROLLUP', null,
        'Unable to cleanup the partial dom 1hour dist rollup (target=' ||
        p_cur_target_guid || ')');
      BEGIN
        UPDATE MGMT_TARGET_ROLLUP_TIMES
          SET rollup_timestamp = p_cur_hour_rollup
          WHERE rowid = p_cur_rollup_rowids(p_dom_dist_1hour_idx);
        COMMIT;
        MGMT_LOG.LOG_ERROR('RTROLLUP', null,
          'Dom 1hour dist rollups for the period [' ||
          TO_CHAR(p_cur_rollup_timestamps(p_dom_dist_1hour_idx),
          'MM:DD:YYYY HH24:MI') || ', ' ||
          TO_CHAR(p_cur_hour_rollup, 'MM:DD:YYYY HH24:MI') || '] 
          cannot be performed (target=' || p_cur_target_guid || ')');
      EXCEPTION
        WHEN OTHERS THEN
          MGMT_LOG.LOG_ERROR('RTROLLUP', null,
            'Fatal Error: Unable to set the dom 1hour dist timestamp (target=' ||
            p_cur_target_guid || '). Error: ' || SUBSTR(SQLERRM, 1, 1000));
          RAISE;
      END;
    END IF;

  WHEN OTHERS THEN
    -- Assumes permanent error and attempts to update the rollup timestamp,
    -- thus losing the rollup period in question.
    -- This should be improved upon, we need to determine if the error is
    -- permanent or transient and only update the timestamp if it's permanent.
   
    -- Rollback any partial work
    ROLLBACK;

    -- Free all resources used
    IF et_rollup_cursor%ISOPEN THEN
      CLOSE et_rollup_cursor;
    END IF;
    IF svt_rollup_cursor%ISOPEN THEN
      CLOSE svt_rollup_cursor;
    END IF;
    IF db_rollup_cursor%ISOPEN THEN
      CLOSE db_rollup_cursor;
    END IF;

    v_rollup_rec.rollup_time.DELETE;
    v_rollup_rec.metric_name.DELETE;
    v_rollup_rec.key_val.DELETE;
    v_rollup_rec.subnet.DELETE;
    v_rollup_rec.subnet_num.DELETE;
    v_rollup_rec.hits.DELETE;
    v_rollup_rec.dist_period.DELETE;

    MGMT_LOG.LOG_ERROR('RTROLLUP', null,
      'The following unrecoverable error occurred while performing the' ||
      ' dom 1hour dist rollup (target=' || p_cur_target_guid || ').  Attempting to' ||
      ' skip over rollup period [' ||
      TO_CHAR(p_cur_rollup_timestamps(p_dom_dist_1hour_idx),
      'MM:DD:YYYY HH24:MI') || ', ' ||
      TO_CHAR(p_cur_hour_rollup,
      'MM:DD:YYYY HH24:MI') || '].  Error: ' || SUBSTR(SQLERRM, 1, 1000));

    BEGIN
      UPDATE MGMT_TARGET_ROLLUP_TIMES
        SET rollup_timestamp = p_cur_hour_rollup
        WHERE rowid = p_cur_rollup_rowids(p_dom_dist_1hour_idx);
      COMMIT;
      MGMT_LOG.LOG_ERROR('RTROLLUP', null,
        'Successfully skipped over period.  Dom 1hour dist rollup data will not be' ||
        ' generated for this period (target=' || p_cur_target_guid || ').');
    EXCEPTION
      WHEN OTHERS THEN
        MGMT_LOG.LOG_ERROR('RTROLLUP', null,
          'Fatal Error: Unable to set the dom 1hour dist timestamp (target=' ||
          p_cur_target_guid || '). Error: ' || SUBSTR(SQLERRM, 1, 1000));
        RAISE;
    END;

END EMD_DOM_DIST_1HOUR_ROLLUP;
----------------------------------------------------------------------

----------------------------------------------------------------------
PROCEDURE EMD_IP_1HOUR_ROLLUP IS
 
  v_rollup_rec      p_hour_rollup_record_type;
  v_rc              PLS_INTEGER := 0; 

  CURSOR rollup_cursor ( c_target_guid RAW,
                         c_prev_rollup DATE,
                         c_cur_rollup DATE,
                         c_max_elapsed_time NUMBER ) IS
    SELECT TRUNC(collection_timestamp, 'HH24'), metric_name,
      NVL(visitor_node, visitor_ip),
      COUNT(target_guid), AVG(elapsed_time), 
      MIN(elapsed_time), MAX(elapsed_time), VARIANCE(elapsed_time),
      AVG(server_latency_time), MIN(server_latency_time), 
      MAX(server_latency_time), VARIANCE(server_latency_time), 
      AVG(database_time), MIN(database_time), 
      MAX(database_time), VARIANCE(database_time)
    FROM MGMT_RT_METRICS_RAW
    WHERE target_guid = c_target_guid
      AND TRUNC(collection_timestamp, 'HH24') > c_prev_rollup 
      AND TRUNC(collection_timestamp, 'HH24') <= c_cur_rollup
      AND elapsed_time <= c_max_elapsed_time
    GROUP BY TRUNC(collection_timestamp, 'HH24'), metric_name,
      NVL(visitor_node, visitor_ip);

BEGIN

  OPEN rollup_cursor( p_cur_target_guid, 
                      p_cur_rollup_timestamps(p_ip_1hour_idx),
                      p_cur_hour_rollup,
                      p_cur_max_response_time );
  LOOP

    FETCH rollup_cursor BULK COLLECT INTO v_rollup_rec.rollup_time,
                                          v_rollup_rec.metric_name,
                                          v_rollup_rec.key_val,
                                          v_rollup_rec.hits,
                                          v_rollup_rec.avg_val,
                                          v_rollup_rec.min_val,
                                          v_rollup_rec.max_val,
                                          v_rollup_rec.var_val,
                                          v_rollup_rec.svr_avg_val,
                                          v_rollup_rec.svr_min_val,
                                          v_rollup_rec.svr_max_val,
                                          v_rollup_rec.svr_var_val,
                                          v_rollup_rec.db_avg_val,
                                          v_rollup_rec.db_min_val,
                                          v_rollup_rec.db_max_val,
                                          v_rollup_rec.db_var_val
                                          LIMIT p_rollup_batchsize;

    -- Exit if no rollup records are generated.
    EXIT WHEN NOT v_rollup_rec.rollup_time.EXISTS(1);

    -- Bulk inserts  
    FORALL i IN v_rollup_rec.rollup_time.FIRST..v_rollup_rec.rollup_time.LAST
      INSERT INTO MGMT_RT_IP_1HOUR
      ( 
        target_guid, metric_name, 
        rollup_timestamp,
	visitor_node,
        hits, response_time_average, 
        response_time_minimum, response_time_maximum, 
        response_time_sdev, response_time_variance,
        server_time_average, server_time_minimum, 
        server_time_maximum, server_time_sdev, 
        server_time_variance, 
        db_time_average, db_time_minimum, 
        db_time_maximum, db_time_sdev, 
        db_time_variance
      )
      VALUES
      ( 
        p_cur_target_guid, v_rollup_rec.metric_name(i),
        v_rollup_rec.rollup_time(i),
        v_rollup_rec.key_val(i), 
        v_rollup_rec.hits(i), v_rollup_rec.avg_val(i), 
        v_rollup_rec.min_val(i), v_rollup_rec.max_val(i),  
        SQRT(v_rollup_rec.var_val(i)), 
        v_rollup_rec.var_val(i) ,
        v_rollup_rec.svr_avg_val(i), v_rollup_rec.svr_min_val(i), 
        v_rollup_rec.svr_max_val(i), SQRT(v_rollup_rec.svr_var_val(i)),
        v_rollup_rec.svr_var_val(i),  
        v_rollup_rec.db_avg_val(i), v_rollup_rec.db_min_val(i), 
        v_rollup_rec.db_max_val(i), SQRT(v_rollup_rec.db_var_val(i)),
        v_rollup_rec.db_var_val(i) 
      );

    COMMIT;

    -- Keep track of the number of inserts
    v_rc := v_rc + v_rollup_rec.key_val.COUNT;

    -- Free temporary tables
    v_rollup_rec.rollup_time.DELETE;
    v_rollup_rec.metric_name.DELETE;
    v_rollup_rec.key_val.DELETE;
    v_rollup_rec.hits.DELETE;
    v_rollup_rec.avg_val.DELETE;
    v_rollup_rec.min_val.DELETE;
    v_rollup_rec.max_val.DELETE;
    v_rollup_rec.var_val.DELETE;
    v_rollup_rec.svr_avg_val.DELETE;
    v_rollup_rec.svr_min_val.DELETE;
    v_rollup_rec.svr_max_val.DELETE;
    v_rollup_rec.svr_var_val.DELETE;
    v_rollup_rec.db_avg_val.DELETE;
    v_rollup_rec.db_min_val.DELETE;
    v_rollup_rec.db_max_val.DELETE;
    v_rollup_rec.db_var_val.DELETE;

    -- Check cursor status here because cursor may return not found on a
    -- bulk fetch, even though records were retrieved.
    EXIT WHEN rollup_cursor%NOTFOUND;

  END LOOP; 

  -- delete all records from the table that are covered by this rollup 
  DELETE FROM MGMT_RT_IP_BOOTSTRAP
  WHERE target_guid = p_cur_target_guid
    AND rollup_timestamp <= p_cur_hour_rollup;

  -- Update the rollup timestamp
  UPDATE MGMT_TARGET_ROLLUP_TIMES
    SET rollup_timestamp = p_cur_hour_rollup
    WHERE rowid = p_cur_rollup_rowids(p_ip_1hour_idx);

  COMMIT;

  IF rollup_cursor%ISOPEN THEN
    CLOSE rollup_cursor;
  END IF;

  -- Update the global row counters
  p_total_rc := p_total_rc + v_rc;
  p_ip_1hour_rollup_rc := p_ip_1hour_rollup_rc + v_rc;

EXCEPTION

  WHEN DUP_VAL_ON_INDEX THEN
    -- Partial rollup detected.

    -- Rollback any partial work performed in this procedure
    ROLLBACK;

    -- Free all resources used
    IF rollup_cursor%ISOPEN THEN
      CLOSE rollup_cursor;
    END IF;
    v_rollup_rec.rollup_time.DELETE;
    v_rollup_rec.metric_name.DELETE;
    v_rollup_rec.key_val.DELETE;
    v_rollup_rec.hits.DELETE;
    v_rollup_rec.avg_val.DELETE;
    v_rollup_rec.min_val.DELETE;
    v_rollup_rec.max_val.DELETE;
    v_rollup_rec.var_val.DELETE;
    v_rollup_rec.svr_avg_val.DELETE;
    v_rollup_rec.svr_min_val.DELETE;
    v_rollup_rec.svr_max_val.DELETE;
    v_rollup_rec.svr_var_val.DELETE;
    v_rollup_rec.db_avg_val.DELETE;
    v_rollup_rec.db_min_val.DELETE;
    v_rollup_rec.db_max_val.DELETE;
    v_rollup_rec.db_var_val.DELETE;

    -- Cleanup all partial work.  If the cleanup fails, update the rollup
    -- timestamp so that the rollup doesn't get stuck forever.
    IF (EMD_IP_1HOUR_CLEANUP = TRUE) THEN
      MGMT_LOG.LOG_ERROR('RTROLLUP', null,
        'Successfully cleaned up partial ip 1hour rollup (target=' ||
        p_cur_target_guid || ')', 'MGMT', NULL, NULL, NULL, MGMT_GLOBAL.G_INFO);
    ELSE
      MGMT_LOG.LOG_ERROR('RTROLLUP', null,
        'Unable to cleanup the partial ip 1hour rollup (target=' ||
        p_cur_target_guid || ')');
      BEGIN
        UPDATE MGMT_TARGET_ROLLUP_TIMES
          SET rollup_timestamp = p_cur_hour_rollup
          WHERE rowid = p_cur_rollup_rowids(p_ip_1hour_idx);
        COMMIT;
        MGMT_LOG.LOG_ERROR('RTROLLUP', null,
          'IP 1hour rollups for the period [' ||
          TO_CHAR(p_cur_rollup_timestamps(p_ip_1hour_idx),
          'MM:DD:YYYY HH24:MI') || ', ' ||
          TO_CHAR(p_cur_hour_rollup, 'MM:DD:YYYY HH24:MI') || ']
          cannot be performed (target=' || p_cur_target_guid || ')');
      EXCEPTION
        WHEN OTHERS THEN
          MGMT_LOG.LOG_ERROR('RTROLLUP', null,
            'Fatal Error: Unable to set the IP 1hour timestamp (target=' ||
            p_cur_target_guid || '). Error: ' || SUBSTR(SQLERRM, 1, 1000));
          RAISE;
      END;
    END IF;

  WHEN OTHERS THEN
    -- Assumes permanent error and attempts to update the rollup timestamp,
    -- thus losing the rollup period in question.
    -- This should be improved upon, we need to determine if the error is
    -- permanent or transient and only update the timestamp if it's permanent.
   
    -- Rollback any partial work
    ROLLBACK;

    -- Free all resources used
    IF rollup_cursor%ISOPEN THEN
      CLOSE rollup_cursor;
    END IF;
    v_rollup_rec.rollup_time.DELETE;
    v_rollup_rec.metric_name.DELETE;
    v_rollup_rec.key_val.DELETE;
    v_rollup_rec.hits.DELETE;
    v_rollup_rec.avg_val.DELETE;
    v_rollup_rec.min_val.DELETE;
    v_rollup_rec.max_val.DELETE;
    v_rollup_rec.var_val.DELETE;
    v_rollup_rec.svr_avg_val.DELETE;
    v_rollup_rec.svr_min_val.DELETE;
    v_rollup_rec.svr_max_val.DELETE;
    v_rollup_rec.svr_var_val.DELETE;
    v_rollup_rec.db_avg_val.DELETE;
    v_rollup_rec.db_min_val.DELETE;
    v_rollup_rec.db_max_val.DELETE;
    v_rollup_rec.db_var_val.DELETE;

    MGMT_LOG.LOG_ERROR('RTROLLUP', null,
      'The following unrecoverable error occurred while performing the' ||
      ' IP 1hour rollup (target=' || p_cur_target_guid || ').  Attempting to' ||
      ' skip over rollup period [' ||
      TO_CHAR(p_cur_rollup_timestamps(p_ip_1hour_idx),
      'MM:DD:YYYY HH24:MI') || ', ' ||
      TO_CHAR(p_cur_hour_rollup,
      'MM:DD:YYYY HH24:MI') || '].  Error: ' || SUBSTR(SQLERRM, 1, 1000));

    BEGIN
      UPDATE MGMT_TARGET_ROLLUP_TIMES
        SET rollup_timestamp = p_cur_hour_rollup
        WHERE rowid = p_cur_rollup_rowids(p_ip_1hour_idx);
      COMMIT;
      MGMT_LOG.LOG_ERROR('RTROLLUP', null,
        'Successfully skipped over period. IP 1hour rollup data will not be' ||
        ' generated for this period (target=' || p_cur_target_guid || ').');
    EXCEPTION
      WHEN OTHERS THEN
        MGMT_LOG.LOG_ERROR('RTROLLUP', null,
          'Fatal Error: Unable to set the ip 1hour timestamp (target=' ||
          p_cur_target_guid || '). Error: ' || SUBSTR(SQLERRM, 1, 1000));
        RAISE;
    END;

END EMD_IP_1HOUR_ROLLUP;
----------------------------------------------------------------------

----------------------------------------------------------------------
PROCEDURE EMD_IP_DIST_1HOUR_ROLLUP IS

  v_rollup_rec      p_dist_rollup_record_type;
  v_rc              PLS_INTEGER := 0; 

  CURSOR et_rollup_cursor ( c_target_guid RAW,
                         c_prev_rollup DATE,
                         c_cur_rollup DATE,
                         c_max_elapsed_time NUMBER ) IS
    SELECT TRUNC(collection_timestamp, 'HH24'), metric_name,
      NVL(visitor_node, visitor_ip),
      COUNT(target_guid),
      ROUND(elapsed_time/1000)
    FROM MGMT_RT_METRICS_RAW
    WHERE target_guid = c_target_guid
      AND TRUNC(collection_timestamp, 'HH24') > c_prev_rollup 
      AND TRUNC(collection_timestamp, 'HH24') <= c_cur_rollup
      AND elapsed_time <= c_max_elapsed_time
    GROUP BY TRUNC(collection_timestamp, 'HH24'), metric_name, 
      NVL(visitor_node, visitor_ip),
      ROUND(elapsed_time/1000);

  CURSOR svt_rollup_cursor ( c_target_guid RAW,
                         c_prev_rollup DATE,
                         c_cur_rollup DATE,
                         c_max_elapsed_time NUMBER ) IS
    SELECT TRUNC(collection_timestamp, 'HH24'), metric_name,
      NVL(visitor_node, visitor_ip),
      COUNT(target_guid),
      ROUND(server_latency_time/1000)
    FROM MGMT_RT_METRICS_RAW
    WHERE target_guid = c_target_guid
      AND TRUNC(collection_timestamp, 'HH24') > c_prev_rollup 
      AND TRUNC(collection_timestamp, 'HH24') <= c_cur_rollup
      AND elapsed_time <= c_max_elapsed_time
    GROUP BY TRUNC(collection_timestamp, 'HH24'), metric_name, 
      NVL(visitor_node, visitor_ip),
      ROUND(server_latency_time/1000);

  CURSOR db_rollup_cursor ( c_target_guid RAW,
                         c_prev_rollup DATE,
                         c_cur_rollup DATE,
                         c_max_elapsed_time NUMBER ) IS
    SELECT TRUNC(collection_timestamp, 'HH24'), metric_name,
      NVL(visitor_node, visitor_ip),
      COUNT(target_guid),
      ROUND(database_time/1000)
    FROM MGMT_RT_METRICS_RAW
    WHERE target_guid = c_target_guid
      AND TRUNC(collection_timestamp, 'HH24') > c_prev_rollup 
      AND TRUNC(collection_timestamp, 'HH24') <= c_cur_rollup
      AND elapsed_time <= c_max_elapsed_time
      AND database_time IS NOT NULL
    GROUP BY TRUNC(collection_timestamp, 'HH24'), metric_name, 
      NVL(visitor_node, visitor_ip),
      ROUND(database_time/1000);

BEGIN

  OPEN et_rollup_cursor( p_cur_target_guid, 
                      p_cur_rollup_timestamps(p_ip_dist_1hour_idx),
                      p_cur_hour_rollup,
                      p_cur_max_response_time );
  LOOP

    FETCH et_rollup_cursor BULK COLLECT INTO v_rollup_rec.rollup_time,
                                          v_rollup_rec.metric_name,
                                          v_rollup_rec.key_val,
                                          v_rollup_rec.hits,
                                          v_rollup_rec.dist_period
                                          LIMIT p_rollup_batchsize;

    -- Exit if no rollup records are generated.
    EXIT WHEN NOT v_rollup_rec.rollup_time.EXISTS(1);

    -- Bulk inserts
    FORALL i IN v_rollup_rec.rollup_time.FIRST..v_rollup_rec.rollup_time.LAST
          INSERT INTO MGMT_RT_IP_DIST_1HOUR
          ( 
            target_guid, metric_name, 
            rollup_timestamp,
            visitor_node, 
            hits, num_seconds,dist_value_type
          )
          VALUES
          ( 
            p_cur_target_guid, v_rollup_rec.metric_name(i),
            v_rollup_rec.rollup_time(i),
            v_rollup_rec.key_val(i), 
            v_rollup_rec.hits(i),v_rollup_rec.dist_period(i) ,p_et_dist_type
          ); 

    COMMIT;

    -- Keep track of the number of inserts
    v_rc := v_rc + v_rollup_rec.key_val.COUNT;

    -- Free temporary tables
    v_rollup_rec.rollup_time.DELETE;
    v_rollup_rec.metric_name.DELETE;
    v_rollup_rec.key_val.DELETE;
    v_rollup_rec.hits.DELETE;
    v_rollup_rec.dist_period.DELETE;

    -- Check cursor status here because cursor may return not found on a
    -- bulk fetch, even though records were retrieved.
    EXIT WHEN et_rollup_cursor%NOTFOUND;

  END LOOP; 

  OPEN svt_rollup_cursor( p_cur_target_guid, 
                      p_cur_rollup_timestamps(p_url_dist_1hour_idx),
                      p_cur_hour_rollup,
                      p_cur_max_response_time );
  LOOP

    -- Process batchsize number of rows per commit.
  FETCH svt_rollup_cursor BULK COLLECT INTO v_rollup_rec.rollup_time,
                                          v_rollup_rec.metric_name,
                                          v_rollup_rec.key_val,
                                          v_rollup_rec.hits,
                                          v_rollup_rec.dist_period
                                          LIMIT p_rollup_batchsize;

    EXIT WHEN NOT v_rollup_rec.rollup_time.EXISTS(1);
    -- Bulk inserts
    FORALL i IN v_rollup_rec.rollup_time.FIRST..v_rollup_rec.rollup_time.LAST

          INSERT INTO MGMT_RT_IP_DIST_1HOUR
          ( 
            target_guid, metric_name, 
            rollup_timestamp,
            visitor_node, 
            hits, num_seconds, dist_value_type
          )
          VALUES
          ( 
            p_cur_target_guid, v_rollup_rec.metric_name(i),
            v_rollup_rec.rollup_time(i),
            v_rollup_rec.key_val(i), 
            v_rollup_rec.hits(i), 
            v_rollup_rec.dist_period(i), p_svt_dist_type
          ); 
    COMMIT;

    -- Keep track of the number of inserts
    v_rc := v_rc + v_rollup_rec.key_val.COUNT;

    -- Free temporary tables
    v_rollup_rec.rollup_time.DELETE;
    v_rollup_rec.metric_name.DELETE;
    v_rollup_rec.key_val.DELETE;
    v_rollup_rec.hits.DELETE;
    v_rollup_rec.dist_period.DELETE;

    EXIT WHEN svt_rollup_cursor%NOTFOUND;

  END LOOP;

  OPEN db_rollup_cursor( p_cur_target_guid, 
                      p_cur_rollup_timestamps(p_url_dist_1hour_idx),
                      p_cur_hour_rollup,
                      p_cur_max_response_time );
  LOOP

    -- Process batchsize number of rows per commit.
  FETCH db_rollup_cursor BULK COLLECT INTO v_rollup_rec.rollup_time,
                                          v_rollup_rec.metric_name,
                                          v_rollup_rec.key_val,
                                          v_rollup_rec.hits,
                                          v_rollup_rec.dist_period
                                          LIMIT p_rollup_batchsize;

    EXIT WHEN NOT v_rollup_rec.rollup_time.EXISTS(1);
    -- Bulk inserts
    FORALL i IN v_rollup_rec.rollup_time.FIRST..v_rollup_rec.rollup_time.LAST

          INSERT INTO MGMT_RT_IP_DIST_1HOUR
          ( 
            target_guid, metric_name, 
            rollup_timestamp,
            visitor_node, 
            hits, num_seconds, dist_value_type
          )
          VALUES
          ( 
            p_cur_target_guid, v_rollup_rec.metric_name(i),
            v_rollup_rec.rollup_time(i),
            v_rollup_rec.key_val(i), 
            v_rollup_rec.hits(i), 
            v_rollup_rec.dist_period(i), p_db_dist_type
          ); 
    COMMIT;

    -- Keep track of the number of inserts
    v_rc := v_rc + v_rollup_rec.key_val.COUNT;

    -- Free temporary tables
    v_rollup_rec.rollup_time.DELETE;
    v_rollup_rec.metric_name.DELETE;
    v_rollup_rec.key_val.DELETE;
    v_rollup_rec.hits.DELETE;
    v_rollup_rec.dist_period.DELETE;

    EXIT WHEN db_rollup_cursor%NOTFOUND;

  END LOOP;

  -- delete all records from the table that are covered by this rollup 
  DELETE FROM MGMT_RT_IP_DIST_BOOTSTRAP
  WHERE target_guid = p_cur_target_guid
    AND rollup_timestamp <= p_cur_hour_rollup;

  -- Update the rollup timestamp
  UPDATE MGMT_TARGET_ROLLUP_TIMES
    SET rollup_timestamp = p_cur_hour_rollup
    WHERE rowid = p_cur_rollup_rowids(p_ip_dist_1hour_idx);

  COMMIT;

  -- Free all resources used
  IF et_rollup_cursor%ISOPEN THEN
    CLOSE et_rollup_cursor;
  END IF;
  IF svt_rollup_cursor%ISOPEN THEN
    CLOSE svt_rollup_cursor;
  END IF;
  IF db_rollup_cursor%ISOPEN THEN
    CLOSE db_rollup_cursor;
  END IF;

  -- Update the global row counters
  p_total_rc := p_total_rc + v_rc;
  p_ip_dist_1hour_rollup_rc := p_ip_dist_1hour_rollup_rc + v_rc;

EXCEPTION

  WHEN DUP_VAL_ON_INDEX THEN
    -- Partial rollup detected.

    -- Rollback any partial work performed in this procedure
    ROLLBACK;

    -- Free all resources used
    IF et_rollup_cursor%ISOPEN THEN
      CLOSE et_rollup_cursor;
    END IF;
    IF svt_rollup_cursor%ISOPEN THEN
      CLOSE svt_rollup_cursor;
    END IF;
    IF db_rollup_cursor%ISOPEN THEN
      CLOSE db_rollup_cursor;
    END IF;

    v_rollup_rec.rollup_time.DELETE;
    v_rollup_rec.metric_name.DELETE;
    v_rollup_rec.key_val.DELETE;
    v_rollup_rec.hits.DELETE;
    v_rollup_rec.dist_period.DELETE;

    -- Cleanup all partial work.  If the cleanup fails, update the rollup
    -- timestamp so that the rollup doesn't get stuck forever.
    IF (EMD_IP_DIST_1HOUR_CLEANUP = TRUE) THEN
      MGMT_LOG.LOG_ERROR('RTROLLUP', null,
        'Successfully cleaned up partial ip dist 1hour rollup (target=' ||
        p_cur_target_guid || ')', 'MGMT', NULL, NULL, NULL, MGMT_GLOBAL.G_INFO);
    ELSE
      MGMT_LOG.LOG_ERROR('RTROLLUP', null,
        'Unable to cleanup the partial ip dist 1hour rollup (target=' ||
        p_cur_target_guid || ')');
      BEGIN
        UPDATE MGMT_TARGET_ROLLUP_TIMES
          SET rollup_timestamp = p_cur_hour_rollup
          WHERE rowid = p_cur_rollup_rowids(p_ip_dist_1hour_idx);
        COMMIT;
        MGMT_LOG.LOG_ERROR('RTROLLUP', null,
          'IP dist 1hour rollups for the period [' ||
          TO_CHAR(p_cur_rollup_timestamps(p_ip_dist_1hour_idx),
          'MM:DD:YYYY HH24:MI') || ', ' ||
          TO_CHAR(p_cur_hour_rollup, 'MM:DD:YYYY HH24:MI') || '] 
          cannot be performed (target=' || p_cur_target_guid || ')');
      EXCEPTION
        WHEN OTHERS THEN
          MGMT_LOG.LOG_ERROR('RTROLLUP', null,
            'Fatal Error: Unable to set the ip dist 1hour timestamp (target=' ||
            p_cur_target_guid || '). Error: ' || SUBSTR(SQLERRM, 1, 1000));
          RAISE;
      END;
    END IF;

  WHEN OTHERS THEN
    -- Assumes permanent error and attempts to update the rollup timestamp,
    -- thus losing the rollup period in question.
    -- This should be improved upon, we need to determine if the error is
    -- permanent or transient and only update the timestamp if it's permanent.
   
    -- Rollback any partial work
    ROLLBACK;

    -- Free all resources used
    IF et_rollup_cursor%ISOPEN THEN
      CLOSE et_rollup_cursor;
    END IF;
    IF svt_rollup_cursor%ISOPEN THEN
      CLOSE svt_rollup_cursor;
    END IF;
    IF db_rollup_cursor%ISOPEN THEN
      CLOSE db_rollup_cursor;
    END IF;

    v_rollup_rec.rollup_time.DELETE;
    v_rollup_rec.metric_name.DELETE;
    v_rollup_rec.key_val.DELETE;
    v_rollup_rec.hits.DELETE;
    v_rollup_rec.dist_period.DELETE;

    MGMT_LOG.LOG_ERROR('RTROLLUP', null,
      'The following unrecoverable error occurred while performing the' ||
      ' ip dist 1hour rollup (target=' || p_cur_target_guid || ').  Attempting to' ||
      ' skip over rollup period [' || 
      TO_CHAR(p_cur_rollup_timestamps(p_ip_dist_1hour_idx),
      'MM:DD:YYYY HH24:MI') || ', ' ||
      TO_CHAR(p_cur_hour_rollup,
      'MM:DD:YYYY HH24:MI') || '].  Error: ' || SUBSTR(SQLERRM, 1, 1000));

    BEGIN
      UPDATE MGMT_TARGET_ROLLUP_TIMES
        SET rollup_timestamp = p_cur_hour_rollup
        WHERE rowid = p_cur_rollup_rowids(p_ip_dist_1hour_idx);
      COMMIT;
      MGMT_LOG.LOG_ERROR('RTROLLUP', null,
        'Successfully skipped over period.  IP dist 1hour rollup data will not be' ||
        ' generated for this period (target=' || p_cur_target_guid || ').');
    EXCEPTION
      WHEN OTHERS THEN
        MGMT_LOG.LOG_ERROR('RTROLLUP', null,
          'Fatal Error: Unable to set the ip dist 1hour timestamp (target=' ||
          p_cur_target_guid || '). Error: ' || SUBSTR(SQLERRM, 1, 1000));
        RAISE;
    END;

END EMD_IP_DIST_1HOUR_ROLLUP;
----------------------------------------------------------------------

----------------------------------------------------------------------
PROCEDURE EMD_PR_MAPPING_1HOUR_ROLLUP IS
 
  v_rollup_rec      p_pr_map_rollup_record_type;
  v_rc              PLS_INTEGER := 0; 

  CURSOR rollup_cursor ( c_target_guid RAW,
                         c_prev_rollup DATE,
                         c_cur_rollup DATE) IS
    SELECT TRUNC(aggregate_hour_timestamp, 'HH24'), page_url, 
           request_url,
      SUM(num_cache_hits), 
      SUM(num_cache_hits * cache_hits_avg_svr_time),
      SUM(num_non_cache_hits), 
      SUM(num_non_cache_hits * non_cache_hits_avg_svr_time)
    FROM MGMT_RT_PR_MAPPING
    WHERE target_guid = c_target_guid
      AND TRUNC(aggregate_hour_timestamp, 'HH24') > c_prev_rollup 
      AND TRUNC(aggregate_hour_timestamp, 'HH24') <= c_cur_rollup
    GROUP BY TRUNC(aggregate_hour_timestamp, 'HH24'), page_url, request_url;

BEGIN

  OPEN rollup_cursor( p_cur_target_guid, 
                      p_cur_rollup_timestamps(p_pr_mapping_1hour_idx),
                      p_cur_hour_rollup );
  LOOP

    FETCH rollup_cursor BULK COLLECT INTO v_rollup_rec.rollup_time,
                                          v_rollup_rec.page_url,
                                          v_rollup_rec.request_url,
                                          v_rollup_rec.cache_hits,
                                          v_rollup_rec.ch_svr_sum,
                                          v_rollup_rec.non_cache_hits,
                                          v_rollup_rec.nch_svr_sum
                                          LIMIT p_rollup_batchsize;

    -- Exit if no rollup records are generated.
    EXIT WHEN NOT v_rollup_rec.rollup_time.EXISTS(1);

    -- Bulk inserts  
    FORALL i IN v_rollup_rec.rollup_time.FIRST..v_rollup_rec.rollup_time.LAST
      INSERT INTO MGMT_RT_PR_MAPPING_1HOUR
      ( 
        target_guid, 
        page_url,
        request_url,
        num_cache_hits,
        cache_hits_avg_svr_time,
        num_non_cache_hits,
        non_cache_hits_avg_svr_time,
        rollup_timestamp
      )
      VALUES
      ( 
        p_cur_target_guid,
        v_rollup_rec.page_url(i),
        v_rollup_rec.request_url(i),
        v_rollup_rec.cache_hits(i),
        DECODE(v_rollup_rec.cache_hits(i),
               0, 0,
               v_rollup_rec.ch_svr_sum(i)/v_rollup_rec.cache_hits(i)),
        v_rollup_rec.non_cache_hits(i),
        DECODE(v_rollup_rec.non_cache_hits(i),
               0, 0,
               v_rollup_rec.nch_svr_sum(i)/v_rollup_rec.non_cache_hits(i)),
        v_rollup_rec.rollup_time(i)
      );

    COMMIT;

    -- Keep track of the number of inserts
    v_rc := v_rc + v_rollup_rec.page_url.COUNT;

    -- Free temporary tables
    v_rollup_rec.page_url.DELETE;
    v_rollup_rec.request_url.DELETE;
    v_rollup_rec.cache_hits.DELETE;
    v_rollup_rec.ch_svr_sum.DELETE;
    v_rollup_rec.non_cache_hits.DELETE;
    v_rollup_rec.nch_svr_sum.DELETE;
    v_rollup_rec.rollup_time.DELETE;

    -- Check cursor status here because cursor may return not found on a
    -- bulk fetch, even though records were retrieved.
    EXIT WHEN rollup_cursor%NOTFOUND;

  END LOOP; 

  -- Update the rollup timestamp
  UPDATE MGMT_TARGET_ROLLUP_TIMES
    SET rollup_timestamp = p_cur_hour_rollup
    WHERE rowid = p_cur_rollup_rowids(p_pr_mapping_1hour_idx);

  COMMIT;

  IF rollup_cursor%ISOPEN THEN
    CLOSE rollup_cursor;
  END IF;

  -- Update the global row counters
  p_total_rc := p_total_rc + v_rc;
  p_pr_mapping_1hour_rollup_rc := p_pr_mapping_1hour_rollup_rc + v_rc;

EXCEPTION

  WHEN DUP_VAL_ON_INDEX THEN
    -- Partial rollup detected.

    -- Rollback any partial work performed in this procedure
    ROLLBACK;

    -- Free all resources used
    IF rollup_cursor%ISOPEN THEN
      CLOSE rollup_cursor;
    END IF;
    v_rollup_rec.page_url.DELETE;
    v_rollup_rec.request_url.DELETE;
    v_rollup_rec.cache_hits.DELETE;
    v_rollup_rec.ch_svr_sum.DELETE;
    v_rollup_rec.non_cache_hits.DELETE;
    v_rollup_rec.nch_svr_sum.DELETE;
    v_rollup_rec.rollup_time.DELETE;

    -- Cleanup all partial work.  If the cleanup fails, update the rollup
    -- timestamp so that the rollup doesn't get stuck forever.
    IF (EMD_PR_MAPPING_1HOUR_CLEANUP = TRUE) THEN
      MGMT_LOG.LOG_ERROR('RTROLLUP', null,
        'Successfully cleaned up partial PR mapping 1hour rollup (target=' ||
        p_cur_target_guid || ')', 'MGMT', NULL, NULL, NULL, MGMT_GLOBAL.G_INFO);
    ELSE
      MGMT_LOG.LOG_ERROR('RTROLLUP', null,
        'Unable to cleanup the partial PR mapping 1hour rollup (target=' ||
        p_cur_target_guid || ')');
      BEGIN
        UPDATE MGMT_TARGET_ROLLUP_TIMES
          SET rollup_timestamp = p_cur_hour_rollup
          WHERE rowid = p_cur_rollup_rowids(p_pr_mapping_1hour_idx);
        COMMIT;
        MGMT_LOG.LOG_ERROR('RTROLLUP', null,
          'PR Mapping 1hour rollups for the period [' ||
          TO_CHAR(p_cur_rollup_timestamps(p_pr_mapping_1hour_idx),
          'MM:DD:YYYY HH24:MI') || ', ' ||
          TO_CHAR(p_cur_hour_rollup, 'MM:DD:YYYY HH24:MI') || ']
          cannot be performed (target=' || p_cur_target_guid || ')');
      EXCEPTION
        WHEN OTHERS THEN
          MGMT_LOG.LOG_ERROR('RTROLLUP', null,
            'Fatal Error: Unable to set the PR Mapping 1hour timestamp (target=' ||
            p_cur_target_guid || '). Error: ' || SUBSTR(SQLERRM, 1, 1000));
          RAISE;
      END;
    END IF;

  WHEN OTHERS THEN
    -- Assumes permanent error and attempts to update the rollup timestamp,
    -- thus losing the rollup period in question.
    -- This should be improved upon, we need to determine if the error is
    -- permanent or transient and only update the timestamp if it's permanent.
   
    -- Rollback any partial work
    ROLLBACK;

    -- Free all resources used
    IF rollup_cursor%ISOPEN THEN
      CLOSE rollup_cursor;
    END IF;
    v_rollup_rec.page_url.DELETE;
    v_rollup_rec.request_url.DELETE;
    v_rollup_rec.cache_hits.DELETE;
    v_rollup_rec.ch_svr_sum.DELETE;
    v_rollup_rec.non_cache_hits.DELETE;
    v_rollup_rec.nch_svr_sum.DELETE;
    v_rollup_rec.rollup_time.DELETE;

    MGMT_LOG.LOG_ERROR('RTROLLUP', null,
      'The following unrecoverable error occurred while performing the' ||
      ' PR Mapping 1hour rollup (target=' || p_cur_target_guid || ').  Attempting to' ||
      ' skip over rollup period [' ||
      TO_CHAR(p_cur_rollup_timestamps(p_pr_mapping_1hour_idx),
      'MM:DD:YYYY HH24:MI') || ', ' ||
      TO_CHAR(p_cur_hour_rollup,
      'MM:DD:YYYY HH24:MI') || '].  Error: ' || SUBSTR(SQLERRM, 1, 1000));

    BEGIN
      UPDATE MGMT_TARGET_ROLLUP_TIMES
        SET rollup_timestamp = p_cur_hour_rollup
        WHERE rowid = p_cur_rollup_rowids(p_pr_mapping_1hour_idx);
      COMMIT;
      MGMT_LOG.LOG_ERROR('RTROLLUP', null,
        'Successfully skipped over period. PR Mapping 1hour rollup data will not be' ||
        ' generated for this period (target=' || p_cur_target_guid || ').');
    EXCEPTION
      WHEN OTHERS THEN
        MGMT_LOG.LOG_ERROR('RTROLLUP', null,
          'Fatal Error: Unable to set the PR Mapping 1hour timestamp (target=' ||
          p_cur_target_guid || '). Error: ' || SUBSTR(SQLERRM, 1, 1000));
        RAISE;
    END;

END EMD_PR_MAPPING_1HOUR_ROLLUP;
----------------------------------------------------------------------

----------------------------------------------------------------------
PROCEDURE EMD_INC_LOADS_1HOUR_ROLLUP IS
 
  v_rollup_rec      p_inc_loads_rollup_record_type;
  v_rc              PLS_INTEGER := 0; 

  CURSOR rollup_cursor ( c_target_guid RAW,
                         c_prev_rollup DATE,
                         c_cur_rollup DATE) IS
    SELECT TRUNC(aggregate_hour_timestamp, 'HH24'), page_url,
      SUM(num_incomplete_loads), 
      ROUND(SUM(num_incomplete_loads * avg_server_time) / 
            SUM(num_incomplete_loads), 2)
    FROM MGMT_RT_INCOMPLETE_LOADS
    WHERE target_guid = c_target_guid
      AND TRUNC(aggregate_hour_timestamp, 'HH24') > c_prev_rollup 
      AND TRUNC(aggregate_hour_timestamp, 'HH24') <= c_cur_rollup
    GROUP BY TRUNC(aggregate_hour_timestamp, 'HH24'), page_url;

BEGIN

  OPEN rollup_cursor( p_cur_target_guid, 
                      p_cur_rollup_timestamps(p_inc_loads_1hour_idx),
                      p_cur_hour_rollup );
  LOOP

    FETCH rollup_cursor BULK COLLECT INTO v_rollup_rec.rollup_time,
                                          v_rollup_rec.page_url,
                                          v_rollup_rec.inc_loads,
                                          v_rollup_rec.svr_avg
                                          LIMIT p_rollup_batchsize;

    -- Exit if no rollup records are generated.
    EXIT WHEN NOT v_rollup_rec.rollup_time.EXISTS(1);

    -- Bulk inserts  
    FORALL i IN v_rollup_rec.rollup_time.FIRST..v_rollup_rec.rollup_time.LAST
      INSERT INTO MGMT_RT_INCOMPLETE_LOADS_1HOUR
      ( 
        target_guid, 
        page_url,
        num_incomplete_loads,
        avg_server_time,
        rollup_timestamp
      )
      VALUES
      ( 
        p_cur_target_guid,
        v_rollup_rec.page_url(i),
        v_rollup_rec.inc_loads(i),
        v_rollup_rec.svr_avg(i),
        v_rollup_rec.rollup_time(i)
      );

    COMMIT;

    -- Keep track of the number of inserts
    v_rc := v_rc + v_rollup_rec.page_url.COUNT;

    -- Free temporary tables
    v_rollup_rec.page_url.DELETE;
    v_rollup_rec.inc_loads.DELETE;
    v_rollup_rec.svr_avg.DELETE;
    v_rollup_rec.rollup_time.DELETE;

    -- Check cursor status here because cursor may return not found on a
    -- bulk fetch, even though records were retrieved.
    EXIT WHEN rollup_cursor%NOTFOUND;

  END LOOP; 

  -- Update the rollup timestamp
  UPDATE MGMT_TARGET_ROLLUP_TIMES
    SET rollup_timestamp = p_cur_hour_rollup
    WHERE rowid = p_cur_rollup_rowids(p_inc_loads_1hour_idx);

  COMMIT;

  IF rollup_cursor%ISOPEN THEN
    CLOSE rollup_cursor;
  END IF;

  -- Update the global row counters
  p_total_rc := p_total_rc + v_rc;
  p_inc_loads_1hour_rollup_rc := p_inc_loads_1hour_rollup_rc + v_rc;

EXCEPTION

  WHEN DUP_VAL_ON_INDEX THEN
    -- Partial rollup detected.

    -- Rollback any partial work performed in this procedure
    ROLLBACK;

    -- Free all resources used
    IF rollup_cursor%ISOPEN THEN
      CLOSE rollup_cursor;
    END IF;
    v_rollup_rec.page_url.DELETE;
    v_rollup_rec.inc_loads.DELETE;
    v_rollup_rec.svr_avg.DELETE;
    v_rollup_rec.rollup_time.DELETE;

    -- Cleanup all partial work.  If the cleanup fails, update the rollup
    -- timestamp so that the rollup doesn't get stuck forever.
    IF (EMD_INC_LOADS_1HOUR_CLEANUP = TRUE) THEN
      MGMT_LOG.LOG_ERROR('RTROLLUP', null,
        'Successfully cleaned up partial incomplete loads 1hour rollup (target=' ||
        p_cur_target_guid || ')', 'MGMT', NULL, NULL, NULL, MGMT_GLOBAL.G_INFO);
    ELSE
      MGMT_LOG.LOG_ERROR('RTROLLUP', null,
        'Unable to cleanup the partial incomplete loads 1hour rollup (target=' ||
        p_cur_target_guid || ')');
      BEGIN
        UPDATE MGMT_TARGET_ROLLUP_TIMES
          SET rollup_timestamp = p_cur_hour_rollup
          WHERE rowid = p_cur_rollup_rowids(p_inc_loads_1hour_idx);
        COMMIT;
        MGMT_LOG.LOG_ERROR('RTROLLUP', null,
          'Incomplete Loads 1hour rollups for the period [' ||
          TO_CHAR(p_cur_rollup_timestamps(p_inc_loads_1hour_idx),
          'MM:DD:YYYY HH24:MI') || ', ' ||
          TO_CHAR(p_cur_hour_rollup, 'MM:DD:YYYY HH24:MI') || ']
          cannot be performed (target=' || p_cur_target_guid || ')');
      EXCEPTION
        WHEN OTHERS THEN
          MGMT_LOG.LOG_ERROR('RTROLLUP', null,
            'Fatal Error: Unable to set the Incomplete Loads 1hour timestamp (target=' ||
            p_cur_target_guid || '). Error: ' || SUBSTR(SQLERRM, 1, 1000));
          RAISE;
      END;
    END IF;

  WHEN OTHERS THEN
    -- Assumes permanent error and attempts to update the rollup timestamp,
    -- thus losing the rollup period in question.
    -- This should be improved upon, we need to determine if the error is
    -- permanent or transient and only update the timestamp if it's permanent.
   
    -- Rollback any partial work
    ROLLBACK;

    -- Free all resources used
    IF rollup_cursor%ISOPEN THEN
      CLOSE rollup_cursor;
    END IF;
    v_rollup_rec.page_url.DELETE;
    v_rollup_rec.inc_loads.DELETE;
    v_rollup_rec.svr_avg.DELETE;
    v_rollup_rec.rollup_time.DELETE;

    MGMT_LOG.LOG_ERROR('RTROLLUP', null,
      'The following unrecoverable error occurred while performing the' ||
      ' Incomplete Loads 1hour rollup (target=' || p_cur_target_guid || ').  Attempting to' ||
      ' skip over rollup period [' ||
      TO_CHAR(p_cur_rollup_timestamps(p_inc_loads_1hour_idx),
      'MM:DD:YYYY HH24:MI') || ', ' ||
      TO_CHAR(p_cur_hour_rollup,
      'MM:DD:YYYY HH24:MI') || '].  Error: ' || SUBSTR(SQLERRM, 1, 1000));

    BEGIN
      UPDATE MGMT_TARGET_ROLLUP_TIMES
        SET rollup_timestamp = p_cur_hour_rollup
        WHERE rowid = p_cur_rollup_rowids(p_inc_loads_1hour_idx);
      COMMIT;
      MGMT_LOG.LOG_ERROR('RTROLLUP', null,
        'Successfully skipped over period. Incomplete Loads 1hour rollup data will not be' ||
        ' generated for this period (target=' || p_cur_target_guid || ').');
    EXCEPTION
      WHEN OTHERS THEN
        MGMT_LOG.LOG_ERROR('RTROLLUP', null,
          'Fatal Error: Unable to set the Incomplete Loads 1hour timestamp (target=' ||
          p_cur_target_guid || '). Error: ' || SUBSTR(SQLERRM, 1, 1000));
        RAISE;
    END;
END EMD_INC_LOADS_1HOUR_ROLLUP;
----------------------------------------------------------------------

---------------------    BOOTSTRAP ROLLUPS   -------------------------

----------------------------------------------------------------------
PROCEDURE EMD_URL_BOOTSTRAP_ROLLUP(
                  bootstrap_record_in IN p_bootstrap_record_type,
                  bootstrap_log_in    IN OUT p_bootstrap_log_type) IS 

  v_rollup_rec      p_hour_rollup_record_type;
  v_rc              PLS_INTEGER := 0; 
  v_last_rollup     DATE; -- the time of the last rollup

  CURSOR rollup_cursor ( c_target_guid RAW,
                         c_prev_rollup DATE,
                         c_cur_rollup DATE,
                         c_last_rollup DATE, 
                         c_max_elapsed_time NUMBER ) IS    
    SELECT TRUNC(collection_timestamp, 'HH24'), metric_name,
      url_filename, url_base,
      COUNT(target_guid), 
      AVG(elapsed_time), MIN(elapsed_time), 
      MAX(elapsed_time), VARIANCE(elapsed_time),
      AVG(server_latency_time), MIN(server_latency_time), 
      MAX(server_latency_time), VARIANCE(server_latency_time), 
      AVG(database_time), MIN(database_time), 
      MAX(database_time), VARIANCE(database_time)
    FROM MGMT_RT_METRICS_RAW
    WHERE target_guid = c_target_guid
      AND TRUNC(collection_timestamp, 'HH24') >  c_last_rollup
      AND collection_timestamp >= TRUNC(c_prev_rollup, 'HH24') 
      AND       collection_timestamp          <= c_cur_rollup
      AND elapsed_time <= c_max_elapsed_time
    GROUP BY TRUNC(collection_timestamp, 'HH24'), metric_name, 
      url_filename, url_base;

BEGIN

  SELECT rollup_timestamp
  INTO v_last_rollup
  FROM MGMT_TARGET_ROLLUP_TIMES
  WHERE target_guid = bootstrap_record_in.target_guid
    AND rollup_table_name = 'MGMT_RT_URL_1HOUR';

  DELETE FROM MGMT_RT_URL_BOOTSTRAP
    WHERE target_guid = bootstrap_record_in.target_guid
      AND rollup_timestamp >= TRUNC(bootstrap_record_in.last_rollup_time, 'HH24')
      AND rollup_timestamp <= bootstrap_record_in.cur_rollup_time;

  OPEN rollup_cursor( bootstrap_record_in.target_guid, 
                      bootstrap_record_in.last_rollup_time,
                      bootstrap_record_in.cur_rollup_time,
                      v_last_rollup,
                      bootstrap_record_in.max_response_time );

  LOOP

    -- Process batchsize number of rows per commit.
    FETCH rollup_cursor BULK COLLECT INTO v_rollup_rec.rollup_time,
                                          v_rollup_rec.metric_name,
                                          v_rollup_rec.key_val,
                                          v_rollup_rec.url_base,
                                          v_rollup_rec.hits,
                                          v_rollup_rec.avg_val,
                                          v_rollup_rec.min_val,
                                          v_rollup_rec.max_val,
                                          v_rollup_rec.var_val,
                                          v_rollup_rec.svr_avg_val,
                                          v_rollup_rec.svr_min_val,
                                          v_rollup_rec.svr_max_val,
                                          v_rollup_rec.svr_var_val,
                                          v_rollup_rec.db_avg_val,
                                          v_rollup_rec.db_min_val,
                                          v_rollup_rec.db_max_val,
                                          v_rollup_rec.db_var_val
                                          LIMIT p_rollup_batchsize;

    -- Exit if no rollup records are generated.
    EXIT WHEN NOT v_rollup_rec.rollup_time.EXISTS(1);

    -- Bulk inserts
    FORALL i IN v_rollup_rec.rollup_time.FIRST..v_rollup_rec.rollup_time.LAST
      INSERT INTO MGMT_RT_URL_BOOTSTRAP
      ( 
        target_guid, metric_name, 
        rollup_timestamp,
        url_filename,  
        url_link, hits,
        response_time_average, response_time_minimum, 
        response_time_maximum, response_time_sdev, 
        response_time_variance,
        server_time_average, server_time_minimum, 
        server_time_maximum, server_time_sdev, 
        server_time_variance,
        db_time_average, db_time_minimum, 
        db_time_maximum, db_time_sdev, 
        db_time_variance
      )
      VALUES
      ( 
        bootstrap_record_in.target_guid, v_rollup_rec.metric_name(i),
        v_rollup_rec.rollup_time(i),
        v_rollup_rec.key_val(i), v_rollup_rec.url_base(i) || v_rollup_rec.key_val(i), 
        v_rollup_rec.hits(i), 
        v_rollup_rec.avg_val(i), v_rollup_rec.min_val(i), v_rollup_rec.max_val(i),  
        SQRT(v_rollup_rec.var_val(i)), v_rollup_rec.var_val(i),
        v_rollup_rec.svr_avg_val(i), v_rollup_rec.svr_min_val(i), v_rollup_rec.svr_max_val(i),  
        SQRT(v_rollup_rec.svr_var_val(i)), v_rollup_rec.svr_var_val(i),
        v_rollup_rec.db_avg_val(i), v_rollup_rec.db_min_val(i), v_rollup_rec.db_max_val(i),  
        SQRT(v_rollup_rec.db_var_val(i)), v_rollup_rec.db_var_val(i)
      );

    COMMIT;

    -- Keep track of the number of inserts
    v_rc := v_rc + v_rollup_rec.key_val.COUNT;

    -- Free temporary tables
    v_rollup_rec.rollup_time.DELETE;
    v_rollup_rec.metric_name.DELETE;
    v_rollup_rec.key_val.DELETE;
    v_rollup_rec.url_base.DELETE;
    v_rollup_rec.hits.DELETE;
    v_rollup_rec.avg_val.DELETE;
    v_rollup_rec.min_val.DELETE;
    v_rollup_rec.max_val.DELETE;
    v_rollup_rec.var_val.DELETE;
    v_rollup_rec.svr_avg_val.DELETE;
    v_rollup_rec.svr_min_val.DELETE;
    v_rollup_rec.svr_max_val.DELETE;
    v_rollup_rec.svr_var_val.DELETE;
    v_rollup_rec.db_avg_val.DELETE;
    v_rollup_rec.db_min_val.DELETE;
    v_rollup_rec.db_max_val.DELETE;
    v_rollup_rec.db_var_val.DELETE;

    -- Check cursor status here because cursor may return not found on a
    -- bulk fetch, even though records were retrieved.
    EXIT WHEN rollup_cursor%NOTFOUND;

  END LOOP; 

  IF rollup_cursor%ISOPEN THEN
    CLOSE rollup_cursor;
  END IF;

  -- Update the global row counters
  bootstrap_log_in.total_rc := bootstrap_log_in.total_rc + v_rc;
  bootstrap_log_in.url_rollup_rc := bootstrap_log_in.url_rollup_rc + v_rc;

EXCEPTION

  WHEN OTHERS THEN
    -- Assumes permanent error and attempts to update the rollup timestamp,
    -- thus losing the rollup period in question.
    -- This should be improved upon, we need to determine if the error is
    -- permanent or transient and only update the timestamp if it's permanent.

    -- Rollback any partial work
    ROLLBACK;

    -- Free all resources used
    IF rollup_cursor%ISOPEN THEN
      CLOSE rollup_cursor;
    END IF;
    v_rollup_rec.rollup_time.DELETE;
    v_rollup_rec.metric_name.DELETE;
    v_rollup_rec.key_val.DELETE;
    v_rollup_rec.url_base.DELETE;
    v_rollup_rec.hits.DELETE;
    v_rollup_rec.avg_val.DELETE;
    v_rollup_rec.min_val.DELETE;
    v_rollup_rec.max_val.DELETE;
    v_rollup_rec.var_val.DELETE;
    v_rollup_rec.svr_avg_val.DELETE;
    v_rollup_rec.svr_min_val.DELETE;
    v_rollup_rec.svr_max_val.DELETE;
    v_rollup_rec.svr_var_val.DELETE;
    v_rollup_rec.db_avg_val.DELETE;
    v_rollup_rec.db_min_val.DELETE;
    v_rollup_rec.db_max_val.DELETE;
    v_rollup_rec.db_var_val.DELETE;

    MGMT_LOG.LOG_ERROR('RTBOOTSTRAPROLLUP', null,
      'The following unrecoverable error occurred while performing the' ||
      ' bootstrap URL rollup (target=' || bootstrap_record_in.target_guid || ').  Attempting to' ||
      ' skip over rollup period [' ||
      TO_CHAR(bootstrap_record_in.last_rollup_time,
      'MM:DD:YYYY HH24:MI') || ', ' ||
      TO_CHAR(bootstrap_record_in.cur_rollup_time,
      'MM:DD:YYYY HH24:MI') || '].  Error: ' || SUBSTR(SQLERRM, 1, 1000));
    RAISE;

END EMD_URL_BOOTSTRAP_ROLLUP;
----------------------------------------------------------------------

----------------------------------------------------------------------
PROCEDURE EMD_URL_DIST_BOOTSTRAP_ROLLUP(
                  bootstrap_record_in IN p_bootstrap_record_type,
                  bootstrap_log_in    IN OUT p_bootstrap_log_type) IS

  v_rollup_rec      p_dist_rollup_record_type;
  v_rc              PLS_INTEGER := 0; 
  v_last_rollup     DATE; -- the time of the last rollup

  CURSOR et_rollup_cursor ( c_target_guid RAW,
                         c_prev_rollup DATE,
                         c_cur_rollup DATE,
                         c_last_rollup DATE, 
                         c_max_elapsed_time NUMBER ) IS
    SELECT TRUNC(collection_timestamp, 'HH24'), metric_name,
      url_filename, 
      COUNT(target_guid),
      ROUND(elapsed_time/1000)
    FROM MGMT_RT_METRICS_RAW
    WHERE target_guid = c_target_guid
      AND TRUNC(collection_timestamp, 'HH24') >  c_last_rollup
      AND collection_timestamp >= TRUNC(c_prev_rollup, 'HH24') 
      AND       collection_timestamp          <= c_cur_rollup
      AND elapsed_time <= c_max_elapsed_time
    GROUP BY TRUNC(collection_timestamp, 'HH24'), metric_name, url_filename,
          ROUND(elapsed_time/1000);

  CURSOR svt_rollup_cursor ( c_target_guid RAW,
                         c_prev_rollup DATE,
                         c_cur_rollup DATE,
                         c_last_rollup DATE, 
                         c_max_elapsed_time NUMBER ) IS
    SELECT TRUNC(collection_timestamp, 'HH24'), metric_name,
      url_filename, 
      COUNT(target_guid),
      ROUND(server_latency_time/1000)
    FROM MGMT_RT_METRICS_RAW
    WHERE target_guid = c_target_guid
      AND TRUNC(collection_timestamp, 'HH24') >  c_last_rollup
      AND collection_timestamp >= TRUNC(c_prev_rollup, 'HH24') 
      AND       collection_timestamp          <= c_cur_rollup
      AND elapsed_time <= c_max_elapsed_time
    GROUP BY TRUNC(collection_timestamp, 'HH24'), metric_name, url_filename,
          ROUND(server_latency_time/1000);

  CURSOR db_rollup_cursor ( c_target_guid RAW,
                         c_prev_rollup DATE,
                         c_cur_rollup DATE,
                         c_last_rollup DATE, 
                         c_max_elapsed_time NUMBER ) IS
    SELECT TRUNC(collection_timestamp, 'HH24'), metric_name,
      url_filename, 
      COUNT(target_guid),
      ROUND(database_time/1000)
    FROM MGMT_RT_METRICS_RAW
    WHERE target_guid = c_target_guid
      AND TRUNC(collection_timestamp, 'HH24') >  c_last_rollup
      AND collection_timestamp >= TRUNC(c_prev_rollup, 'HH24') 
      AND       collection_timestamp          <= c_cur_rollup
      AND elapsed_time <= c_max_elapsed_time
      AND database_time IS NOT NULL
    GROUP BY TRUNC(collection_timestamp, 'HH24'), metric_name, url_filename,
          ROUND(database_time/1000);

  BEGIN

  SELECT rollup_timestamp
  INTO v_last_rollup
  FROM MGMT_TARGET_ROLLUP_TIMES
  WHERE target_guid = bootstrap_record_in.target_guid
    AND rollup_table_name = 'MGMT_RT_URL_DIST_1HOUR';

  DELETE FROM MGMT_RT_URL_DIST_BOOTSTRAP
    WHERE target_guid = bootstrap_record_in.target_guid
      AND rollup_timestamp >= TRUNC(bootstrap_record_in.last_rollup_time, 'HH24')
      AND rollup_timestamp <= bootstrap_record_in.cur_rollup_time;

  OPEN et_rollup_cursor( bootstrap_record_in.target_guid, 
                      bootstrap_record_in.last_rollup_time,
                      bootstrap_record_in.cur_rollup_time,
                      v_last_rollup,
                      bootstrap_record_in.max_response_time );
  LOOP

    -- Process batchsize number of rows per commit.
  FETCH et_rollup_cursor BULK COLLECT INTO v_rollup_rec.rollup_time,
                                          v_rollup_rec.metric_name,
                                          v_rollup_rec.key_val,
                                          v_rollup_rec.hits,
                                          v_rollup_rec.dist_period
                                          LIMIT p_rollup_batchsize;
    EXIT WHEN NOT v_rollup_rec.rollup_time.EXISTS(1);

    -- Bulk inserts
    FORALL i IN v_rollup_rec.rollup_time.FIRST..v_rollup_rec.rollup_time.LAST

          INSERT INTO MGMT_RT_URL_DIST_BOOTSTRAP
          ( 
            target_guid, metric_name, 
            rollup_timestamp,
            url_filename, 
            hits, num_seconds, dist_value_type
          )
          VALUES
          ( 
            bootstrap_record_in.target_guid, v_rollup_rec.metric_name(i),
            v_rollup_rec.rollup_time(i),
            v_rollup_rec.key_val(i), 
            v_rollup_rec.hits(i), 
            v_rollup_rec.dist_period(i), p_et_dist_type
          ); 
    COMMIT;

    -- Keep track of the number of inserts
    v_rc := v_rc + v_rollup_rec.key_val.COUNT;

    -- Free temporary tables
    v_rollup_rec.rollup_time.DELETE;
    v_rollup_rec.metric_name.DELETE;
    v_rollup_rec.key_val.DELETE;
    v_rollup_rec.hits.DELETE;
    v_rollup_rec.dist_period.DELETE;

    EXIT WHEN et_rollup_cursor%NOTFOUND;
  END LOOP;

  OPEN svt_rollup_cursor( bootstrap_record_in.target_guid, 
                      bootstrap_record_in.last_rollup_time,
                      bootstrap_record_in.cur_rollup_time,
                      v_last_rollup,
                      bootstrap_record_in.max_response_time );
  LOOP

    -- Process batchsize number of rows per commit.
  FETCH svt_rollup_cursor BULK COLLECT INTO v_rollup_rec.rollup_time,
                                          v_rollup_rec.metric_name,
                                          v_rollup_rec.key_val,
                                          v_rollup_rec.hits,
                                          v_rollup_rec.dist_period
                                          LIMIT p_rollup_batchsize;

    EXIT WHEN NOT v_rollup_rec.rollup_time.EXISTS(1);
    -- Bulk inserts
    FORALL i IN v_rollup_rec.rollup_time.FIRST..v_rollup_rec.rollup_time.LAST

          INSERT INTO MGMT_RT_URL_DIST_BOOTSTRAP
          ( 
            target_guid, metric_name, 
            rollup_timestamp,
            url_filename, 
            hits, num_seconds, dist_value_type
          )
          VALUES
          ( 
            bootstrap_record_in.target_guid, v_rollup_rec.metric_name(i),
            v_rollup_rec.rollup_time(i),
            v_rollup_rec.key_val(i), 
            v_rollup_rec.hits(i), 
            v_rollup_rec.dist_period(i), p_svt_dist_type
          ); 
    COMMIT;

    -- Keep track of the number of inserts
    v_rc := v_rc + v_rollup_rec.key_val.COUNT;

    -- Free temporary tables
    v_rollup_rec.rollup_time.DELETE;
    v_rollup_rec.metric_name.DELETE;
    v_rollup_rec.key_val.DELETE;
    v_rollup_rec.hits.DELETE;
    v_rollup_rec.dist_period.DELETE;

    EXIT WHEN svt_rollup_cursor%NOTFOUND;
  END LOOP;

  OPEN db_rollup_cursor( bootstrap_record_in.target_guid, 
                      bootstrap_record_in.last_rollup_time,
                      bootstrap_record_in.cur_rollup_time,
                      v_last_rollup,
                      bootstrap_record_in.max_response_time );
  LOOP

    -- Process batchsize number of rows per commit.
  FETCH db_rollup_cursor BULK COLLECT INTO v_rollup_rec.rollup_time,
                                          v_rollup_rec.metric_name,
                                          v_rollup_rec.key_val,
                                          v_rollup_rec.hits,
                                          v_rollup_rec.dist_period
                                          LIMIT p_rollup_batchsize;

    EXIT WHEN NOT v_rollup_rec.rollup_time.EXISTS(1);
    -- Bulk inserts
    FORALL i IN v_rollup_rec.rollup_time.FIRST..v_rollup_rec.rollup_time.LAST

          INSERT INTO MGMT_RT_URL_DIST_BOOTSTRAP
          ( 
            target_guid, metric_name, 
            rollup_timestamp,
            url_filename, 
            hits, num_seconds, dist_value_type
          )
          VALUES
          ( 
            bootstrap_record_in.target_guid, v_rollup_rec.metric_name(i),
            v_rollup_rec.rollup_time(i),
            v_rollup_rec.key_val(i), 
            v_rollup_rec.hits(i), 
            v_rollup_rec.dist_period(i), p_db_dist_type
          ); 
    COMMIT;

    -- Keep track of the number of inserts
    v_rc := v_rc + v_rollup_rec.key_val.COUNT;

    -- Free temporary tables
    v_rollup_rec.rollup_time.DELETE;
    v_rollup_rec.metric_name.DELETE;
    v_rollup_rec.key_val.DELETE;
    v_rollup_rec.hits.DELETE;
    v_rollup_rec.dist_period.DELETE;

    EXIT WHEN db_rollup_cursor%NOTFOUND;
  END LOOP;

  COMMIT;

  IF et_rollup_cursor%ISOPEN THEN
    CLOSE et_rollup_cursor;
  END IF;
  IF svt_rollup_cursor%ISOPEN THEN
    CLOSE svt_rollup_cursor;
  END IF;
  IF db_rollup_cursor%ISOPEN THEN
    CLOSE db_rollup_cursor;
  END IF;

  -- Update the global row counters
  bootstrap_log_in.total_rc := bootstrap_log_in.total_rc + v_rc;
  bootstrap_log_in.url_dist_rollup_rc := bootstrap_log_in.url_dist_rollup_rc + v_rc;

  EXCEPTION

  WHEN OTHERS THEN
    -- Assumes permanent error and attempts to update the rollup timestamp,
    -- thus losing the rollup period in question.
    -- This should be improved upon, we need to determine if the error is
    -- permanent or transient and only update the timestamp if it's permanent.

    -- Rollback any partial work
    ROLLBACK;

    -- Free all resources used
    IF et_rollup_cursor%ISOPEN THEN
      CLOSE et_rollup_cursor;
    END IF; 
    IF svt_rollup_cursor%ISOPEN THEN
      CLOSE svt_rollup_cursor;
    END IF;
    IF db_rollup_cursor%ISOPEN THEN
      CLOSE db_rollup_cursor;
    END IF;
    v_rollup_rec.rollup_time.DELETE;
    v_rollup_rec.metric_name.DELETE;
    v_rollup_rec.key_val.DELETE;
    v_rollup_rec.hits.DELETE;
    v_rollup_rec.dist_period.DELETE;

    MGMT_LOG.LOG_ERROR('RTBOOTSTRAPROLLUP', null,
      'The following unrecoverable error occurred while performing the' ||
      ' bootstrap dist URL rollup (target=' || bootstrap_record_in.target_guid || ').  Attempting to' ||
      ' skip over rollup period [' || 
      TO_CHAR(bootstrap_record_in.last_rollup_time,
      'MM:DD:YYYY HH24:MI') || ', ' ||
      TO_CHAR(bootstrap_record_in.cur_rollup_time,
      'MM:DD:YYYY HH24:MI') || '].  Error: ' || SUBSTR(SQLERRM, 1, 1000));
    RAISE;

END EMD_URL_DIST_BOOTSTRAP_ROLLUP;
----------------------------------------------------------------------

----------------------------------------------------------------------
PROCEDURE EMD_DOM_BOOTSTRAP_ROLLUP(
                  bootstrap_record_in IN p_bootstrap_record_type,
                  bootstrap_log_in    IN OUT p_bootstrap_log_type) IS

  v_rollup_rec      p_hour_rollup_record_type;
  v_rc              PLS_INTEGER := 0;
  v_last_rollup     DATE; -- the time of the last rollup

  CURSOR rollup_cursor ( c_target_guid RAW,
                         c_prev_rollup DATE,
                         c_cur_rollup DATE,
                         c_last_rollup DATE, 
                         c_max_elapsed_time NUMBER ) IS
    SELECT TRUNC(collection_timestamp, 'HH24'), metric_name,
      visitor_domain,
      SUBSTR(visitor_ip, 1, INSTR(visitor_ip, '.', 1, 3)-1),
      BITAND(visitor_ip_num, p_subnet_mask),
      COUNT(visitor_domain), 
      AVG(elapsed_time), MIN(elapsed_time), 
      MAX(elapsed_time), VARIANCE(elapsed_time),
      AVG(server_latency_time), MIN(server_latency_time), 
      MAX(server_latency_time), VARIANCE(server_latency_time), 
      AVG(database_time), MIN(database_time), 
      MAX(database_time), VARIANCE(database_time)
    FROM MGMT_RT_METRICS_RAW
    WHERE target_guid = c_target_guid
      AND TRUNC(collection_timestamp, 'HH24') >  c_last_rollup
      AND collection_timestamp >= TRUNC(c_prev_rollup, 'HH24')
      AND       collection_timestamp          <= c_cur_rollup
      AND elapsed_time <= c_max_elapsed_time
    GROUP BY TRUNC(collection_timestamp, 'HH24'), metric_name, 
      visitor_domain, 
      SUBSTR(visitor_ip, 1, INSTR(visitor_ip, '.', 1, 3)-1),
      BITAND(visitor_ip_num, p_subnet_mask);

BEGIN

  SELECT rollup_timestamp
  INTO v_last_rollup
  FROM MGMT_TARGET_ROLLUP_TIMES
  WHERE target_guid = bootstrap_record_in.target_guid
    AND rollup_table_name = 'MGMT_RT_DOMAIN_1HOUR';

  DELETE FROM MGMT_RT_DOMAIN_BOOTSTRAP
    WHERE target_guid = bootstrap_record_in.target_guid
      AND rollup_timestamp >= TRUNC(bootstrap_record_in.last_rollup_time, 'HH24')
      AND rollup_timestamp <= bootstrap_record_in.cur_rollup_time;

  OPEN rollup_cursor( bootstrap_record_in.target_guid, 
                      bootstrap_record_in.last_rollup_time,
                      bootstrap_record_in.cur_rollup_time,
                      v_last_rollup,
                      bootstrap_record_in.max_response_time );
  LOOP

    -- Process batchsize number of rows per commit.
    FETCH rollup_cursor BULK COLLECT INTO v_rollup_rec.rollup_time,
                                          v_rollup_rec.metric_name,
                                          v_rollup_rec.key_val,
                                          v_rollup_rec.subnet,
                                          v_rollup_rec.subnet_num,
                                          v_rollup_rec.hits,
                                          v_rollup_rec.avg_val,
                                          v_rollup_rec.min_val,
                                          v_rollup_rec.max_val,
                                          v_rollup_rec.var_val,
                                          v_rollup_rec.svr_avg_val,
                                          v_rollup_rec.svr_min_val,
                                          v_rollup_rec.svr_max_val,
                                          v_rollup_rec.svr_var_val,
                                          v_rollup_rec.db_avg_val,
                                          v_rollup_rec.db_min_val,
                                          v_rollup_rec.db_max_val,
                                          v_rollup_rec.db_var_val
                                          LIMIT p_rollup_batchsize;

    -- Exit if no rollup records are generated.
    EXIT WHEN NOT v_rollup_rec.rollup_time.EXISTS(1);

    -- Bulk inserts
    FORALL i IN v_rollup_rec.rollup_time.FIRST..v_rollup_rec.rollup_time.LAST
      INSERT INTO MGMT_RT_DOMAIN_BOOTSTRAP
      ( 
        target_guid, metric_name, 
        rollup_timestamp,
	visitor_domain,
        visitor_subnet, visitor_subnet_num, hits, 
        response_time_average, 
        response_time_minimum, response_time_maximum, 
        response_time_sdev, response_time_variance,
        server_time_average, server_time_minimum, 
        server_time_maximum, server_time_sdev, 
        server_time_variance,
        db_time_average, db_time_minimum, 
        db_time_maximum, db_time_sdev, 
        db_time_variance
      )
      VALUES
      ( 
        bootstrap_record_in.target_guid, v_rollup_rec.metric_name(i),
        v_rollup_rec.rollup_time(i),
        v_rollup_rec.key_val(i), 
	v_rollup_rec.subnet(i), v_rollup_rec.subnet_num(i), 
        v_rollup_rec.hits(i), v_rollup_rec.avg_val(i), 
        v_rollup_rec.min_val(i), v_rollup_rec.max_val(i),  
        SQRT(v_rollup_rec.var_val(i)), v_rollup_rec.var_val(i),
        v_rollup_rec.svr_avg_val(i), v_rollup_rec.svr_min_val(i), 
        v_rollup_rec.svr_max_val(i), SQRT(v_rollup_rec.svr_var_val(i)), 
        v_rollup_rec.svr_var_val(i),
        v_rollup_rec.db_avg_val(i), v_rollup_rec.db_min_val(i), 
        v_rollup_rec.db_max_val(i), SQRT(v_rollup_rec.db_var_val(i)), 
        v_rollup_rec.db_var_val(i)
      );

    COMMIT;

    -- Keep track of the number of inserts
    v_rc := v_rc + v_rollup_rec.key_val.COUNT;

    -- Free temporary tables
    v_rollup_rec.rollup_time.DELETE;
    v_rollup_rec.metric_name.DELETE;
    v_rollup_rec.key_val.DELETE;
    v_rollup_rec.subnet.DELETE;
    v_rollup_rec.subnet_num.DELETE;
    v_rollup_rec.hits.DELETE;
    v_rollup_rec.avg_val.DELETE;
    v_rollup_rec.min_val.DELETE;
    v_rollup_rec.max_val.DELETE;
    v_rollup_rec.var_val.DELETE;
    v_rollup_rec.svr_avg_val.DELETE;
    v_rollup_rec.svr_min_val.DELETE;
    v_rollup_rec.svr_max_val.DELETE;
    v_rollup_rec.svr_var_val.DELETE;
    v_rollup_rec.db_avg_val.DELETE;
    v_rollup_rec.db_min_val.DELETE;
    v_rollup_rec.db_max_val.DELETE;
    v_rollup_rec.db_var_val.DELETE;

    -- Check cursor status here because cursor may return not found on a
    -- bulk fetch, even though records were retrieved.
    EXIT WHEN rollup_cursor%NOTFOUND;

  END LOOP;

  IF rollup_cursor%ISOPEN THEN
    CLOSE rollup_cursor;
  END IF;

  -- Update the global row counters
  bootstrap_log_in.total_rc := bootstrap_log_in.total_rc + v_rc;
  bootstrap_log_in.dom_rollup_rc := bootstrap_log_in.dom_rollup_rc + v_rc;

EXCEPTION

  WHEN OTHERS THEN
    -- Assumes permanent error and attempts to update the rollup timestamp,
    -- thus losing the rollup period in question.
    -- This should be improved upon, we need to determine if the error is
    -- permanent or transient and only update the timestamp if it's permanent.
   
    -- Rollback any partial work
    ROLLBACK;

    -- Free all resources used
    IF rollup_cursor%ISOPEN THEN
      CLOSE rollup_cursor;
    END IF;
    v_rollup_rec.rollup_time.DELETE;
    v_rollup_rec.metric_name.DELETE;
    v_rollup_rec.key_val.DELETE;
    v_rollup_rec.subnet.DELETE;
    v_rollup_rec.subnet_num.DELETE;
    v_rollup_rec.hits.DELETE;
    v_rollup_rec.avg_val.DELETE;
    v_rollup_rec.min_val.DELETE;
    v_rollup_rec.max_val.DELETE;
    v_rollup_rec.var_val.DELETE;
    v_rollup_rec.svr_avg_val.DELETE;
    v_rollup_rec.svr_min_val.DELETE;
    v_rollup_rec.svr_max_val.DELETE;
    v_rollup_rec.svr_var_val.DELETE;
    v_rollup_rec.db_avg_val.DELETE;
    v_rollup_rec.db_min_val.DELETE;
    v_rollup_rec.db_max_val.DELETE;
    v_rollup_rec.db_var_val.DELETE;

    MGMT_LOG.LOG_ERROR('RTBOOTSTRAPROLLUP', null,
      'The following unrecoverable error occurred while performing the' ||
      ' dom bootstrap rollup (target=' || bootstrap_record_in.target_guid || ').  Attempting to' ||
      ' skip over rollup period [' || 
      TO_CHAR(bootstrap_record_in.last_rollup_time,
      'MM:DD:YYYY HH24:MI') || ', ' ||
      TO_CHAR(bootstrap_record_in.cur_rollup_time,
      'MM:DD:YYYY HH24:MI') || '].  Error: ' || SUBSTR(SQLERRM, 1, 1000));
    RAISE;

END EMD_DOM_BOOTSTRAP_ROLLUP;
----------------------------------------------------------------------

----------------------------------------------------------------------
PROCEDURE EMD_DOM_DIST_BOOTSTRAP_ROLLUP(
                  bootstrap_record_in IN p_bootstrap_record_type,
                  bootstrap_log_in    IN OUT p_bootstrap_log_type) IS

  v_rollup_rec      p_dist_rollup_record_type;
  v_rc              PLS_INTEGER := 0; 
  v_last_rollup     DATE; -- the time of the last rollup

  CURSOR et_rollup_cursor ( c_target_guid RAW,
                         c_prev_rollup DATE,
                         c_cur_rollup DATE,
                         c_last_rollup DATE, 
                         c_max_elapsed_time NUMBER ) IS
    SELECT TRUNC(collection_timestamp, 'HH24'), metric_name,
      visitor_domain, 
      SUBSTR(visitor_ip, 1, INSTR(visitor_ip, '.', 1, 3)-1),
      BITAND(visitor_ip_num, p_subnet_mask),
      COUNT(target_guid),
      ROUND(elapsed_time/1000)
    FROM MGMT_RT_METRICS_RAW
    WHERE target_guid = c_target_guid
      AND TRUNC(collection_timestamp, 'HH24') >  c_last_rollup
      AND collection_timestamp >= TRUNC(c_prev_rollup, 'HH24')
      AND       collection_timestamp          <= c_cur_rollup
      AND elapsed_time <= c_max_elapsed_time
    GROUP BY TRUNC(collection_timestamp, 'HH24'), visitor_domain,
      metric_name,
      SUBSTR(visitor_ip, 1, INSTR(visitor_ip, '.', 1, 3)-1),
      BITAND(visitor_ip_num, p_subnet_mask),
      ROUND(elapsed_time/1000);

  CURSOR svt_rollup_cursor ( c_target_guid RAW,
                         c_prev_rollup DATE,
                         c_cur_rollup DATE,
                         c_last_rollup DATE, 
                         c_max_elapsed_time NUMBER ) IS
    SELECT TRUNC(collection_timestamp, 'HH24'), metric_name,
      visitor_domain, 
      SUBSTR(visitor_ip, 1, INSTR(visitor_ip, '.', 1, 3)-1),
      BITAND(visitor_ip_num, p_subnet_mask),
      COUNT(target_guid),
      ROUND(server_latency_time/1000)
    FROM MGMT_RT_METRICS_RAW
    WHERE target_guid = c_target_guid
      AND TRUNC(collection_timestamp, 'HH24') >  c_last_rollup
      AND collection_timestamp >= TRUNC(c_prev_rollup, 'HH24') 
      AND       collection_timestamp          <= c_cur_rollup
      AND elapsed_time <= c_max_elapsed_time
    GROUP BY TRUNC(collection_timestamp, 'HH24'), visitor_domain,
      metric_name,
      SUBSTR(visitor_ip, 1, INSTR(visitor_ip, '.', 1, 3)-1),
      BITAND(visitor_ip_num, p_subnet_mask),
      ROUND(server_latency_time/1000);

  CURSOR db_rollup_cursor ( c_target_guid RAW,
                         c_prev_rollup DATE,
                         c_cur_rollup DATE,
                         c_last_rollup DATE, 
                         c_max_elapsed_time NUMBER ) IS
    SELECT TRUNC(collection_timestamp, 'HH24'), metric_name,
      visitor_domain, 
      SUBSTR(visitor_ip, 1, INSTR(visitor_ip, '.', 1, 3)-1),
      BITAND(visitor_ip_num, p_subnet_mask),
      COUNT(target_guid),
      ROUND(database_time/1000)
    FROM MGMT_RT_METRICS_RAW
    WHERE target_guid = c_target_guid
      AND TRUNC(collection_timestamp, 'HH24') >  c_last_rollup
      AND collection_timestamp >= TRUNC(c_prev_rollup, 'HH24') 
      AND       collection_timestamp          <= c_cur_rollup
      AND elapsed_time <= c_max_elapsed_time
      AND database_time IS NOT NULL
    GROUP BY TRUNC(collection_timestamp, 'HH24'), visitor_domain,
      metric_name,
      SUBSTR(visitor_ip, 1, INSTR(visitor_ip, '.', 1, 3)-1),
      BITAND(visitor_ip_num, p_subnet_mask),
      ROUND(database_time/1000);

BEGIN

  SELECT rollup_timestamp
  INTO v_last_rollup
  FROM MGMT_TARGET_ROLLUP_TIMES
  WHERE target_guid = bootstrap_record_in.target_guid
    AND rollup_table_name = 'MGMT_RT_DOMAIN_DIST_1HOUR';

  DELETE FROM MGMT_RT_DOMAIN_DIST_BOOTSTRAP
    WHERE target_guid = bootstrap_record_in.target_guid
      AND rollup_timestamp >= TRUNC(bootstrap_record_in.last_rollup_time, 'HH24')
      AND rollup_timestamp <= bootstrap_record_in.cur_rollup_time;

  OPEN et_rollup_cursor( bootstrap_record_in.target_guid, 
                      bootstrap_record_in.last_rollup_time,
                      bootstrap_record_in.cur_rollup_time,
                      v_last_rollup,
                      bootstrap_record_in.max_response_time );
  LOOP

    FETCH et_rollup_cursor BULK COLLECT INTO v_rollup_rec.rollup_time,
                                          v_rollup_rec.metric_name,
                                          v_rollup_rec.key_val,
                                          v_rollup_rec.subnet,
                                          v_rollup_rec.subnet_num,
                                          v_rollup_rec.hits,
                                          v_rollup_rec.dist_period
                                          LIMIT p_rollup_batchsize;

    -- Exit if no rollup records are generated
    EXIT WHEN NOT v_rollup_rec.rollup_time.EXISTS(1);

    -- Bulk inserts
    FORALL i IN v_rollup_rec.rollup_time.FIRST..v_rollup_rec.rollup_time.LAST
          INSERT INTO MGMT_RT_DOMAIN_DIST_BOOTSTRAP
          ( 
            target_guid, metric_name, 
            rollup_timestamp,
            visitor_domain, visitor_subnet, visitor_subnet_num, 
            hits, num_seconds, dist_value_type
          )
          VALUES
          ( 
            bootstrap_record_in.target_guid, v_rollup_rec.metric_name(i),
            v_rollup_rec.rollup_time(i),
            v_rollup_rec.key_val(i), 
            v_rollup_rec.subnet(i), v_rollup_rec.subnet_num(i),
            v_rollup_rec.hits(i), v_rollup_rec.dist_period(i), p_et_dist_type
          ); 

    COMMIT;

    -- Keep track of the number of inserts
    v_rc := v_rc + v_rollup_rec.key_val.COUNT;

    -- Free temporary tables
    v_rollup_rec.rollup_time.DELETE;
    v_rollup_rec.metric_name.DELETE;
    v_rollup_rec.key_val.DELETE;
    v_rollup_rec.subnet.DELETE;
    v_rollup_rec.subnet_num.DELETE;
    v_rollup_rec.hits.DELETE;
    v_rollup_rec.dist_period.DELETE;

    -- Check cursor status here because cursor may return not found on a
    -- bulk fetch, even though records were retrieved.
    EXIT WHEN et_rollup_cursor%NOTFOUND;

  END LOOP; 

  OPEN svt_rollup_cursor( bootstrap_record_in.target_guid, 
                      bootstrap_record_in.last_rollup_time,
                      bootstrap_record_in.cur_rollup_time,
                      v_last_rollup,
                      bootstrap_record_in.max_response_time );
  LOOP

    -- Process batchsize number of rows per commit.
  FETCH svt_rollup_cursor BULK COLLECT INTO v_rollup_rec.rollup_time,
                                          v_rollup_rec.metric_name,
                                          v_rollup_rec.key_val,
                                          v_rollup_rec.subnet,
                                          v_rollup_rec.subnet_num,
                                          v_rollup_rec.hits,
                                          v_rollup_rec.dist_period
                                          LIMIT p_rollup_batchsize;

    EXIT WHEN NOT v_rollup_rec.rollup_time.EXISTS(1);
    -- Bulk inserts
    FORALL i IN v_rollup_rec.rollup_time.FIRST..v_rollup_rec.rollup_time.LAST
          INSERT INTO MGMT_RT_DOMAIN_DIST_BOOTSTRAP
          ( 
            target_guid, metric_name, 
            rollup_timestamp,
            visitor_domain, visitor_subnet, visitor_subnet_num, 
            hits, num_seconds, dist_value_type
          )
          VALUES
          ( 
            bootstrap_record_in.target_guid, v_rollup_rec.metric_name(i),
            v_rollup_rec.rollup_time(i),
            v_rollup_rec.key_val(i), 
            v_rollup_rec.subnet(i),
            v_rollup_rec.subnet_num(i),
            v_rollup_rec.hits(i), 
            v_rollup_rec.dist_period(i), p_svt_dist_type
          ); 
    COMMIT;

    -- Keep track of the number of inserts
    v_rc := v_rc + v_rollup_rec.key_val.COUNT;

    -- Free temporary tables
    v_rollup_rec.rollup_time.DELETE;
    v_rollup_rec.metric_name.DELETE;
    v_rollup_rec.key_val.DELETE;
    v_rollup_rec.subnet.DELETE;
    v_rollup_rec.subnet_num.DELETE;
    v_rollup_rec.hits.DELETE;
    v_rollup_rec.dist_period.DELETE;

    EXIT WHEN svt_rollup_cursor%NOTFOUND;

  END LOOP;

  OPEN db_rollup_cursor( bootstrap_record_in.target_guid, 
                      bootstrap_record_in.last_rollup_time,
                      bootstrap_record_in.cur_rollup_time,
                      v_last_rollup,
                      bootstrap_record_in.max_response_time );
  LOOP

    -- Process batchsize number of rows per commit.
  FETCH db_rollup_cursor BULK COLLECT INTO v_rollup_rec.rollup_time,
                                          v_rollup_rec.metric_name,
                                          v_rollup_rec.key_val,
                                          v_rollup_rec.subnet,
                                          v_rollup_rec.subnet_num,
                                          v_rollup_rec.hits,
                                          v_rollup_rec.dist_period
                                          LIMIT p_rollup_batchsize;

    EXIT WHEN NOT v_rollup_rec.rollup_time.EXISTS(1);
    -- Bulk inserts
    FORALL i IN v_rollup_rec.rollup_time.FIRST..v_rollup_rec.rollup_time.LAST
          INSERT INTO MGMT_RT_DOMAIN_DIST_BOOTSTRAP
          ( 
            target_guid, metric_name, 
            rollup_timestamp,
            visitor_domain, visitor_subnet, visitor_subnet_num, 
            hits, num_seconds, dist_value_type
          )
          VALUES
          ( 
            bootstrap_record_in.target_guid, v_rollup_rec.metric_name(i),
            v_rollup_rec.rollup_time(i),
            v_rollup_rec.key_val(i), 
            v_rollup_rec.subnet(i),
            v_rollup_rec.subnet_num(i),
            v_rollup_rec.hits(i), 
            v_rollup_rec.dist_period(i), p_db_dist_type
          ); 
    COMMIT;

    -- Keep track of the number of inserts
    v_rc := v_rc + v_rollup_rec.key_val.COUNT;

    -- Free temporary tables
    v_rollup_rec.rollup_time.DELETE;
    v_rollup_rec.metric_name.DELETE;
    v_rollup_rec.key_val.DELETE;
    v_rollup_rec.subnet.DELETE;
    v_rollup_rec.subnet_num.DELETE;
    v_rollup_rec.hits.DELETE;
    v_rollup_rec.dist_period.DELETE;

    EXIT WHEN db_rollup_cursor%NOTFOUND;

  END LOOP;

  IF et_rollup_cursor%ISOPEN THEN
    CLOSE et_rollup_cursor;
  END IF;
  IF svt_rollup_cursor%ISOPEN THEN
    CLOSE svt_rollup_cursor;
  END IF;
  IF db_rollup_cursor%ISOPEN THEN
    CLOSE db_rollup_cursor;
  END IF;

  -- Update the global row counters
  bootstrap_log_in.total_rc := bootstrap_log_in.total_rc + v_rc;
  bootstrap_log_in.dom_dist_rollup_rc := bootstrap_log_in.dom_dist_rollup_rc + v_rc;

EXCEPTION

  WHEN OTHERS THEN
    -- Assumes permanent error and attempts to update the rollup timestamp,
    -- thus losing the rollup period in question.
    -- This should be improved upon, we need to determine if the error is
    -- permanent or transient and only update the timestamp if it's permanent.
   
    -- Rollback any partial work
    ROLLBACK;

    -- Free all resources used
    IF et_rollup_cursor%ISOPEN THEN
      CLOSE et_rollup_cursor;
    END IF;
    IF svt_rollup_cursor%ISOPEN THEN
      CLOSE svt_rollup_cursor;
    END IF;
    IF db_rollup_cursor%ISOPEN THEN
      CLOSE db_rollup_cursor;
    END IF;

    v_rollup_rec.rollup_time.DELETE;
    v_rollup_rec.metric_name.DELETE;
    v_rollup_rec.key_val.DELETE;
    v_rollup_rec.subnet.DELETE;
    v_rollup_rec.subnet_num.DELETE;
    v_rollup_rec.hits.DELETE;
    v_rollup_rec.dist_period.DELETE;

    MGMT_LOG.LOG_ERROR('RTBOOTSTRAPROLLUP', null,
      'The following unrecoverable error occurred while performing the' ||
      ' dom 1hour dist rollup (target=' || bootstrap_record_in.target_guid || ').  Attempting to' ||
      ' skip over rollup period [' ||
      TO_CHAR(bootstrap_record_in.last_rollup_time,
      'MM:DD:YYYY HH24:MI') || ', ' ||
      TO_CHAR(bootstrap_record_in.cur_rollup_time,
      'MM:DD:YYYY HH24:MI') || '].  Error: ' || SUBSTR(SQLERRM, 1, 1000));
    RAISE;

END EMD_DOM_DIST_BOOTSTRAP_ROLLUP;
----------------------------------------------------------------------

----------------------------------------------------------------------
PROCEDURE EMD_IP_BOOTSTRAP_ROLLUP(
                  bootstrap_record_in IN p_bootstrap_record_type,
                  bootstrap_log_in    IN OUT p_bootstrap_log_type) IS
 
  v_rollup_rec      p_hour_rollup_record_type;
  v_rc              PLS_INTEGER := 0; 
  v_last_rollup     DATE; -- the time of the last rollup

  CURSOR rollup_cursor ( c_target_guid RAW,
                         c_prev_rollup DATE,
                         c_cur_rollup DATE,
                         c_last_rollup DATE, 
                         c_max_elapsed_time NUMBER ) IS
    SELECT TRUNC(collection_timestamp, 'HH24'), metric_name,
      NVL(visitor_node, visitor_ip),
      COUNT(target_guid), AVG(elapsed_time), 
      MIN(elapsed_time), MAX(elapsed_time), VARIANCE(elapsed_time),
      AVG(server_latency_time), MIN(server_latency_time), 
      MAX(server_latency_time), VARIANCE(server_latency_time), 
      AVG(database_time), MIN(database_time), 
      MAX(database_time), VARIANCE(database_time)
    FROM MGMT_RT_METRICS_RAW
    WHERE target_guid = c_target_guid
      AND TRUNC(collection_timestamp, 'HH24') >  c_last_rollup
      AND collection_timestamp >= TRUNC(c_prev_rollup, 'HH24') 
      AND       collection_timestamp          <= c_cur_rollup
      AND elapsed_time <= c_max_elapsed_time
    GROUP BY TRUNC(collection_timestamp, 'HH24'), metric_name,
      NVL(visitor_node, visitor_ip);

BEGIN

  SELECT rollup_timestamp
  INTO v_last_rollup
  FROM MGMT_TARGET_ROLLUP_TIMES
  WHERE target_guid = bootstrap_record_in.target_guid
    AND rollup_table_name = 'MGMT_RT_IP_1HOUR';

  DELETE FROM MGMT_RT_IP_BOOTSTRAP
    WHERE target_guid = bootstrap_record_in.target_guid
      AND rollup_timestamp >= TRUNC(bootstrap_record_in.last_rollup_time, 'HH24')
      AND rollup_timestamp <= bootstrap_record_in.cur_rollup_time;

  OPEN rollup_cursor( bootstrap_record_in.target_guid, 
                      bootstrap_record_in.last_rollup_time,
                      bootstrap_record_in.cur_rollup_time,
                      v_last_rollup,
                      bootstrap_record_in.max_response_time );
  LOOP

    FETCH rollup_cursor BULK COLLECT INTO v_rollup_rec.rollup_time,
                                          v_rollup_rec.metric_name,
                                          v_rollup_rec.key_val,
                                          v_rollup_rec.hits,
                                          v_rollup_rec.avg_val,
                                          v_rollup_rec.min_val,
                                          v_rollup_rec.max_val,
                                          v_rollup_rec.var_val,
                                          v_rollup_rec.svr_avg_val,
                                          v_rollup_rec.svr_min_val,
                                          v_rollup_rec.svr_max_val,
                                          v_rollup_rec.svr_var_val,
                                          v_rollup_rec.db_avg_val,
                                          v_rollup_rec.db_min_val,
                                          v_rollup_rec.db_max_val,
                                          v_rollup_rec.db_var_val
                                          LIMIT p_rollup_batchsize;

    -- Exit if no rollup records are generated.
    EXIT WHEN NOT v_rollup_rec.rollup_time.EXISTS(1);

    -- Bulk inserts  
    FORALL i IN v_rollup_rec.rollup_time.FIRST..v_rollup_rec.rollup_time.LAST
      INSERT INTO MGMT_RT_IP_BOOTSTRAP
      ( 
        target_guid, metric_name, 
        rollup_timestamp,
	visitor_node,
        hits, response_time_average, 
        response_time_minimum, response_time_maximum, 
        response_time_sdev, response_time_variance,
        server_time_average, server_time_minimum, 
        server_time_maximum, server_time_sdev, 
        server_time_variance, 
        db_time_average, db_time_minimum, 
        db_time_maximum, db_time_sdev, 
        db_time_variance
      )
      VALUES
      ( 
        bootstrap_record_in.target_guid, v_rollup_rec.metric_name(i),
        v_rollup_rec.rollup_time(i),
        v_rollup_rec.key_val(i), 
        v_rollup_rec.hits(i), v_rollup_rec.avg_val(i), 
        v_rollup_rec.min_val(i), v_rollup_rec.max_val(i),  
        SQRT(v_rollup_rec.var_val(i)), 
        v_rollup_rec.var_val(i) ,
        v_rollup_rec.svr_avg_val(i), v_rollup_rec.svr_min_val(i), 
        v_rollup_rec.svr_max_val(i), SQRT(v_rollup_rec.svr_var_val(i)) ,
        v_rollup_rec.svr_var_val(i),  
        v_rollup_rec.db_avg_val(i), v_rollup_rec.db_min_val(i), 
        v_rollup_rec.db_max_val(i), SQRT(v_rollup_rec.db_var_val(i)),
        v_rollup_rec.db_var_val(i)  
      );

    COMMIT;

    -- Keep track of the number of inserts
    v_rc := v_rc + v_rollup_rec.key_val.COUNT;

    -- Free temporary tables
    v_rollup_rec.rollup_time.DELETE;
    v_rollup_rec.metric_name.DELETE;
    v_rollup_rec.key_val.DELETE;
    v_rollup_rec.hits.DELETE;
    v_rollup_rec.avg_val.DELETE;
    v_rollup_rec.min_val.DELETE;
    v_rollup_rec.max_val.DELETE;
    v_rollup_rec.var_val.DELETE;
    v_rollup_rec.svr_avg_val.DELETE;
    v_rollup_rec.svr_min_val.DELETE;
    v_rollup_rec.svr_max_val.DELETE;
    v_rollup_rec.svr_var_val.DELETE;
    v_rollup_rec.db_avg_val.DELETE;
    v_rollup_rec.db_min_val.DELETE;
    v_rollup_rec.db_max_val.DELETE;
    v_rollup_rec.db_var_val.DELETE;

    -- Check cursor status here because cursor may return not found on a
    -- bulk fetch, even though records were retrieved.
    EXIT WHEN rollup_cursor%NOTFOUND;

  END LOOP; 

  IF rollup_cursor%ISOPEN THEN
    CLOSE rollup_cursor;
  END IF;

  -- Update the global row counters
  bootstrap_log_in.total_rc := bootstrap_log_in.total_rc + v_rc;
  bootstrap_log_in.ip_rollup_rc := bootstrap_log_in.ip_rollup_rc + v_rc;

EXCEPTION

  WHEN OTHERS THEN
    -- Assumes permanent error and attempts to update the rollup timestamp,
    -- thus losing the rollup period in question.
    -- This should be improved upon, we need to determine if the error is
    -- permanent or transient and only update the timestamp if it's permanent.
   
    -- Rollback any partial work
    ROLLBACK;

    -- Free all resources used
    IF rollup_cursor%ISOPEN THEN
      CLOSE rollup_cursor;
    END IF;
    v_rollup_rec.rollup_time.DELETE;
    v_rollup_rec.metric_name.DELETE;
    v_rollup_rec.key_val.DELETE;
    v_rollup_rec.hits.DELETE;
    v_rollup_rec.avg_val.DELETE;
    v_rollup_rec.min_val.DELETE;
    v_rollup_rec.max_val.DELETE;
    v_rollup_rec.var_val.DELETE;
    v_rollup_rec.svr_avg_val.DELETE;
    v_rollup_rec.svr_min_val.DELETE;
    v_rollup_rec.svr_max_val.DELETE;
    v_rollup_rec.svr_var_val.DELETE;
    v_rollup_rec.db_avg_val.DELETE;
    v_rollup_rec.db_min_val.DELETE;
    v_rollup_rec.db_max_val.DELETE;
    v_rollup_rec.db_var_val.DELETE;

    MGMT_LOG.LOG_ERROR('RTBOOTSTRAPROLLUP', null,
      'The following unrecoverable error occurred while performing the' ||
      ' IP bootstrap rollup (target=' || bootstrap_record_in.target_guid || ').  Attempting to' ||
      ' skip over rollup period [' ||
      TO_CHAR(bootstrap_record_in.last_rollup_time,
      'MM:DD:YYYY HH24:MI') || ', ' ||
      TO_CHAR(bootstrap_record_in.cur_rollup_time,
      'MM:DD:YYYY HH24:MI') || '].  Error: ' || SUBSTR(SQLERRM, 1, 1000));
    RAISE;

END EMD_IP_BOOTSTRAP_ROLLUP;
----------------------------------------------------------------------

----------------------------------------------------------------------
PROCEDURE EMD_IP_DIST_BOOTSTRAP_ROLLUP(
                  bootstrap_record_in IN p_bootstrap_record_type,
                  bootstrap_log_in    IN OUT p_bootstrap_log_type) IS

  v_rollup_rec      p_dist_rollup_record_type;
  v_rc              PLS_INTEGER := 0; 
  v_last_rollup     DATE; -- the time of the last rollup

  CURSOR et_rollup_cursor ( c_target_guid RAW,
                         c_prev_rollup DATE,
                         c_cur_rollup DATE,
                         c_last_rollup DATE, 
                         c_max_elapsed_time NUMBER ) IS
    SELECT TRUNC(collection_timestamp, 'HH24'), metric_name,
      NVL(visitor_node, visitor_ip),
      COUNT(target_guid),
      ROUND(elapsed_time/1000)
    FROM MGMT_RT_METRICS_RAW
    WHERE target_guid = c_target_guid
      AND TRUNC(collection_timestamp, 'HH24') >  c_last_rollup
      AND collection_timestamp >= TRUNC(c_prev_rollup, 'HH24') 
      AND       collection_timestamp          <= c_cur_rollup
      AND elapsed_time <= c_max_elapsed_time
    GROUP BY TRUNC(collection_timestamp, 'HH24'), metric_name, 
      NVL(visitor_node, visitor_ip),
      ROUND(elapsed_time/1000);

  CURSOR svt_rollup_cursor ( c_target_guid RAW,
                         c_prev_rollup DATE,
                         c_cur_rollup DATE,
                         c_last_rollup DATE, 
                         c_max_elapsed_time NUMBER ) IS
    SELECT TRUNC(collection_timestamp, 'HH24'), metric_name,
      NVL(visitor_node, visitor_ip),
      COUNT(target_guid),
      ROUND(server_latency_time/1000)
    FROM MGMT_RT_METRICS_RAW
    WHERE target_guid = c_target_guid
      AND TRUNC(collection_timestamp, 'HH24') >  c_last_rollup
      AND collection_timestamp >= TRUNC(c_prev_rollup, 'HH24') 
      AND       collection_timestamp          <= c_cur_rollup
      AND elapsed_time <= c_max_elapsed_time
    GROUP BY TRUNC(collection_timestamp, 'HH24'), metric_name, 
      NVL(visitor_node, visitor_ip),
      ROUND(server_latency_time/1000);

  CURSOR db_rollup_cursor ( c_target_guid RAW,
                         c_prev_rollup DATE,
                         c_cur_rollup DATE,
                         c_last_rollup DATE, 
                         c_max_elapsed_time NUMBER ) IS
    SELECT TRUNC(collection_timestamp, 'HH24'), metric_name,
      NVL(visitor_node, visitor_ip),
      COUNT(target_guid),
      ROUND(database_time/1000)
    FROM MGMT_RT_METRICS_RAW
    WHERE target_guid = c_target_guid
      AND TRUNC(collection_timestamp, 'HH24') >  c_last_rollup
      AND collection_timestamp >= TRUNC(c_prev_rollup, 'HH24') 
      AND       collection_timestamp          <= c_cur_rollup
      AND elapsed_time <= c_max_elapsed_time
      AND database_time IS NOT NULL	
    GROUP BY TRUNC(collection_timestamp, 'HH24'), metric_name, 
      NVL(visitor_node, visitor_ip),
      ROUND(database_time/1000);

BEGIN

  SELECT rollup_timestamp
  INTO v_last_rollup
  FROM MGMT_TARGET_ROLLUP_TIMES
  WHERE target_guid = bootstrap_record_in.target_guid
    AND rollup_table_name = 'MGMT_RT_IP_DIST_1HOUR';

  DELETE FROM MGMT_RT_IP_DIST_BOOTSTRAP
    WHERE target_guid = bootstrap_record_in.target_guid
      AND rollup_timestamp >= TRUNC(bootstrap_record_in.last_rollup_time, 'HH24')
      AND rollup_timestamp <= bootstrap_record_in.cur_rollup_time;

  OPEN et_rollup_cursor( bootstrap_record_in.target_guid, 
                      bootstrap_record_in.last_rollup_time,
                      bootstrap_record_in.cur_rollup_time,
                      v_last_rollup,
                      bootstrap_record_in.max_response_time );
  LOOP

    FETCH et_rollup_cursor BULK COLLECT INTO v_rollup_rec.rollup_time,
                                          v_rollup_rec.metric_name,
                                          v_rollup_rec.key_val,
                                          v_rollup_rec.hits,
                                          v_rollup_rec.dist_period
                                          LIMIT p_rollup_batchsize;

    -- Exit if no rollup records are generated.
    EXIT WHEN NOT v_rollup_rec.rollup_time.EXISTS(1);

    -- Bulk inserts
    FORALL i IN v_rollup_rec.rollup_time.FIRST..v_rollup_rec.rollup_time.LAST
          INSERT INTO MGMT_RT_IP_DIST_BOOTSTRAP
          ( 
            target_guid, metric_name, 
            rollup_timestamp,
            visitor_node, 
            hits, num_seconds,dist_value_type
          )
          VALUES
          ( 
            bootstrap_record_in.target_guid, v_rollup_rec.metric_name(i),
            v_rollup_rec.rollup_time(i),
            v_rollup_rec.key_val(i), 
            v_rollup_rec.hits(i),v_rollup_rec.dist_period(i) ,p_et_dist_type
          ); 

    COMMIT;

    -- Keep track of the number of inserts
    v_rc := v_rc + v_rollup_rec.key_val.COUNT;

    -- Free temporary tables
    v_rollup_rec.rollup_time.DELETE;
    v_rollup_rec.metric_name.DELETE;
    v_rollup_rec.key_val.DELETE;
    v_rollup_rec.hits.DELETE;
    v_rollup_rec.dist_period.DELETE;

    -- Check cursor status here because cursor may return not found on a
    -- bulk fetch, even though records were retrieved.
    EXIT WHEN et_rollup_cursor%NOTFOUND;

  END LOOP; 

  OPEN svt_rollup_cursor( bootstrap_record_in.target_guid, 
                      bootstrap_record_in.last_rollup_time,
                      bootstrap_record_in.cur_rollup_time,
                      v_last_rollup,
                      bootstrap_record_in.max_response_time );
  LOOP

    -- Process batchsize number of rows per commit.
  FETCH svt_rollup_cursor BULK COLLECT INTO v_rollup_rec.rollup_time,
                                          v_rollup_rec.metric_name,
                                          v_rollup_rec.key_val,
                                          v_rollup_rec.hits,
                                          v_rollup_rec.dist_period
                                          LIMIT p_rollup_batchsize;

    EXIT WHEN NOT v_rollup_rec.rollup_time.EXISTS(1);
    -- Bulk inserts
    FORALL i IN v_rollup_rec.rollup_time.FIRST..v_rollup_rec.rollup_time.LAST

          INSERT INTO MGMT_RT_IP_DIST_BOOTSTRAP
          ( 
            target_guid, metric_name, 
            rollup_timestamp,
            visitor_node, 
            hits, num_seconds, dist_value_type
          )
          VALUES
          ( 
            bootstrap_record_in.target_guid, v_rollup_rec.metric_name(i),
            v_rollup_rec.rollup_time(i),
            v_rollup_rec.key_val(i), 
            v_rollup_rec.hits(i), 
            v_rollup_rec.dist_period(i), p_svt_dist_type
          ); 
    COMMIT;

    -- Keep track of the number of inserts
    v_rc := v_rc + v_rollup_rec.key_val.COUNT;

    -- Free temporary tables
    v_rollup_rec.rollup_time.DELETE;
    v_rollup_rec.metric_name.DELETE;
    v_rollup_rec.key_val.DELETE;
    v_rollup_rec.hits.DELETE;
    v_rollup_rec.dist_period.DELETE;

    EXIT WHEN svt_rollup_cursor%NOTFOUND;

  END LOOP;

  OPEN db_rollup_cursor( bootstrap_record_in.target_guid, 
                      bootstrap_record_in.last_rollup_time,
                      bootstrap_record_in.cur_rollup_time,
                      v_last_rollup,
                      bootstrap_record_in.max_response_time );
  LOOP

    -- Process batchsize number of rows per commit.
  FETCH db_rollup_cursor BULK COLLECT INTO v_rollup_rec.rollup_time,
                                          v_rollup_rec.metric_name,
                                          v_rollup_rec.key_val,
                                          v_rollup_rec.hits,
                                          v_rollup_rec.dist_period
                                          LIMIT p_rollup_batchsize;

    EXIT WHEN NOT v_rollup_rec.rollup_time.EXISTS(1);
    -- Bulk inserts
    FORALL i IN v_rollup_rec.rollup_time.FIRST..v_rollup_rec.rollup_time.LAST

          INSERT INTO MGMT_RT_IP_DIST_BOOTSTRAP
          ( 
            target_guid, metric_name, 
            rollup_timestamp,
            visitor_node, 
            hits, num_seconds, dist_value_type
          )
          VALUES
          ( 
            bootstrap_record_in.target_guid, v_rollup_rec.metric_name(i),
            v_rollup_rec.rollup_time(i),
            v_rollup_rec.key_val(i), 
            v_rollup_rec.hits(i), 
            v_rollup_rec.dist_period(i), p_db_dist_type
          ); 
    COMMIT;

    -- Keep track of the number of inserts
    v_rc := v_rc + v_rollup_rec.key_val.COUNT;

    -- Free temporary tables
    v_rollup_rec.rollup_time.DELETE;
    v_rollup_rec.metric_name.DELETE;
    v_rollup_rec.key_val.DELETE;
    v_rollup_rec.hits.DELETE;
    v_rollup_rec.dist_period.DELETE;

    EXIT WHEN db_rollup_cursor%NOTFOUND;

  END LOOP;

  -- Free all resources used
  IF svt_rollup_cursor%ISOPEN THEN
    CLOSE svt_rollup_cursor;
  END IF;
  IF et_rollup_cursor%ISOPEN THEN
    CLOSE et_rollup_cursor;
  END IF;
  IF db_rollup_cursor%ISOPEN THEN
    CLOSE db_rollup_cursor;
  END IF;

  -- Update the global row counters
  bootstrap_log_in.total_rc := bootstrap_log_in.total_rc + v_rc;
  bootstrap_log_in.ip_dist_rollup_rc := bootstrap_log_in.ip_dist_rollup_rc + v_rc;

EXCEPTION

  WHEN OTHERS THEN
    -- Assumes permanent error and attempts to update the rollup timestamp,
    -- thus losing the rollup period in question.
    -- This should be improved upon, we need to determine if the error is
    -- permanent or transient and only update the timestamp if it's permanent.
   
    -- Rollback any partial work
    ROLLBACK;

    -- Free all resources used
    IF et_rollup_cursor%ISOPEN THEN
      CLOSE et_rollup_cursor;
    END IF;
    IF svt_rollup_cursor%ISOPEN THEN
      CLOSE svt_rollup_cursor;
    END IF;
    IF db_rollup_cursor%ISOPEN THEN
      CLOSE db_rollup_cursor;
    END IF;

    v_rollup_rec.rollup_time.DELETE;
    v_rollup_rec.metric_name.DELETE;
    v_rollup_rec.key_val.DELETE;
    v_rollup_rec.hits.DELETE;
    v_rollup_rec.dist_period.DELETE;

    MGMT_LOG.LOG_ERROR('RTBOOTSTRAPROLLUP', null,
      'The following unrecoverable error occurred while performing the' ||
      ' ip dist 1hour rollup (target=' || bootstrap_record_in.target_guid || ').  Attempting to' ||
      ' skip over rollup period [' || 
      TO_CHAR(bootstrap_record_in.last_rollup_time,
      'MM:DD:YYYY HH24:MI') || ', ' ||
      TO_CHAR(bootstrap_record_in.cur_rollup_time,
      'MM:DD:YYYY HH24:MI') || '].  Error: ' || SUBSTR(SQLERRM, 1, 1000));
    RAISE;

END EMD_IP_DIST_BOOTSTRAP_ROLLUP;
----------------------------------------------------------------------

-----------------------    DAILY ROLLUPS    --------------------------

----------------------------------------------------------------------
PROCEDURE EMD_URL_1DAY_ROLLUP IS

  v_rollup_rec      p_day_rollup_record_type;
  v_rc              PLS_INTEGER := 0; 

  CURSOR rollup_cursor( c_target_guid RAW,
                        c_prev_rollup DATE,
                        c_cur_rollup DATE) IS
    SELECT TRUNC(rollup_timestamp, 'DD'), metric_name,
      url_filename, url_link,
      SUM(hits), 
      MIN(response_time_minimum),
      MAX(response_time_maximum), 
      SUM(hits * response_time_average),
      SUM((response_time_variance * (hits-1)) + 
          (hits * POWER(response_time_average, 2))),
      MIN(server_time_minimum),
      MAX(server_time_maximum), 
      SUM(hits * server_time_average),
      SUM((server_time_variance * (hits-1)) + 
          (hits * POWER(server_time_average, 2))),
      MIN(db_time_minimum),
      MAX(db_time_maximum),
      SUM(hits * db_time_average),
      SUM((db_time_variance * (hits-1)) + 
          (hits * POWER(db_time_average, 2)))
    FROM MGMT_RT_URL_1HOUR
    WHERE target_guid = c_target_guid 
      AND TRUNC(rollup_timestamp, 'DD') > c_prev_rollup
      AND TRUNC(rollup_timestamp, 'DD') <= c_cur_rollup
    GROUP BY TRUNC(rollup_timestamp, 'DD'), metric_name,
      url_filename, url_link;

BEGIN

  OPEN rollup_cursor( p_cur_target_guid, 
                      p_cur_rollup_timestamps(p_url_1day_idx),
                      p_cur_day_rollup ); 
  LOOP

    -- Process batchsize number of rows per commit.
    FETCH rollup_cursor BULK COLLECT INTO v_rollup_rec.rollup_time,
                                          v_rollup_rec.metric_name,
                                          v_rollup_rec.key_val,
                                          v_rollup_rec.url_link,
                                          v_rollup_rec.hits,
                                          v_rollup_rec.min_val,
                                          v_rollup_rec.max_val,
                                          v_rollup_rec.sumx_val,
                                          v_rollup_rec.sumx2_val,
                                          v_rollup_rec.svr_min_val,
                                          v_rollup_rec.svr_max_val,
                                          v_rollup_rec.svr_sumx_val,
                                          v_rollup_rec.svr_sumx2_val,
                                          v_rollup_rec.db_min_val,
                                          v_rollup_rec.db_max_val,
                                          v_rollup_rec.db_sumx_val,
                                          v_rollup_rec.db_sumx2_val
                                          LIMIT p_rollup_batchsize;

    -- Exit loop if no rollup records are generated.
    EXIT WHEN NOT v_rollup_rec.rollup_time.EXISTS(1);

    -- Bulk inserts
    FORALL i IN v_rollup_rec.rollup_time.FIRST..v_rollup_rec.rollup_time.LAST
      INSERT INTO MGMT_RT_URL_1DAY
      ( 
        target_guid, metric_name, 
        rollup_timestamp,
        url_filename, 
        url_link, 
        hits, 
        response_time_average, 
        response_time_minimum, 
        response_time_maximum, 
        response_time_sdev, 
        response_time_variance,
        server_time_average, 
        server_time_minimum, 
        server_time_maximum, 
        server_time_sdev, 
        server_time_variance, 
        db_time_average, 
        db_time_minimum, 
        db_time_maximum, 
        db_time_sdev, 
        db_time_variance
      )
      VALUES
      ( 
        p_cur_target_guid, v_rollup_rec.metric_name(i),
        v_rollup_rec.rollup_time(i),
        v_rollup_rec.key_val(i), 
        v_rollup_rec.url_link(i), 
        v_rollup_rec.hits(i), 
        DECODE(v_rollup_rec.hits(i), 
               0, 0,
               v_rollup_rec.sumx_val(i)/v_rollup_rec.hits(i)),
        v_rollup_rec.min_val(i), 
        v_rollup_rec.max_val(i),  
        DECODE(v_rollup_rec.hits(i), 
               0, 0,
               1, 0,
               SQRT(((v_rollup_rec.hits(i) * v_rollup_rec.sumx2_val(i) - 
                       POWER(v_rollup_rec.sumx_val(i),2))
                      / (v_rollup_rec.hits(i) * (v_rollup_rec.hits(i) - 1))))),
        DECODE(v_rollup_rec.hits(i), 
               0, 0,
               1, 0,
               ((v_rollup_rec.hits(i) * v_rollup_rec.sumx2_val(i) - 
                 POWER(v_rollup_rec.sumx_val(i),2))
                / (v_rollup_rec.hits(i) * (v_rollup_rec.hits(i) - 1)))),
        DECODE(v_rollup_rec.hits(i), 
               0, 0,
               v_rollup_rec.svr_sumx_val(i)/v_rollup_rec.hits(i)),
        v_rollup_rec.svr_min_val(i), 
        v_rollup_rec.svr_max_val(i),  
        DECODE(v_rollup_rec.hits(i), 
               0, 0,
               1, 0,
               SQRT(((v_rollup_rec.hits(i) * v_rollup_rec.svr_sumx2_val(i) - 
                       POWER(v_rollup_rec.svr_sumx_val(i),2))
                      / (v_rollup_rec.hits(i) * (v_rollup_rec.hits(i) - 1))))),
        DECODE(v_rollup_rec.hits(i), 
               0, 0,
               1, 0,
               ((v_rollup_rec.hits(i) * v_rollup_rec.svr_sumx2_val(i) - 
                 POWER(v_rollup_rec.svr_sumx_val(i),2))
                / (v_rollup_rec.hits(i) * (v_rollup_rec.hits(i) - 1)))),
        DECODE(v_rollup_rec.hits(i), 
               0, 0,
               v_rollup_rec.db_sumx_val(i)/v_rollup_rec.hits(i)),
        v_rollup_rec.db_min_val(i), 
        v_rollup_rec.db_max_val(i),  
        DECODE(v_rollup_rec.hits(i), 
               0, 0,
               1, 0,
               SQRT(((v_rollup_rec.hits(i) * v_rollup_rec.db_sumx2_val(i) - 
                       POWER(v_rollup_rec.db_sumx_val(i),2))
                      / (v_rollup_rec.hits(i) * (v_rollup_rec.hits(i) - 1))))),
        DECODE(v_rollup_rec.hits(i), 
               0, 0,
               1, 0,
               ((v_rollup_rec.hits(i) * v_rollup_rec.db_sumx2_val(i) - 
                 POWER(v_rollup_rec.db_sumx_val(i),2))
                / (v_rollup_rec.hits(i) * (v_rollup_rec.hits(i) - 1))))
      );

    COMMIT;

    -- Keep track of the number of inserts
    v_rc := v_rc + v_rollup_rec.key_val.COUNT;

    -- Free temporary tables
    v_rollup_rec.rollup_time.DELETE;
    v_rollup_rec.metric_name.DELETE;
    v_rollup_rec.key_val.DELETE;
    v_rollup_rec.url_link.DELETE;
    v_rollup_rec.hits.DELETE;
    v_rollup_rec.min_val.DELETE;
    v_rollup_rec.max_val.DELETE;
    v_rollup_rec.sumx_val.DELETE;
    v_rollup_rec.sumx2_val.DELETE;
    v_rollup_rec.svr_min_val.DELETE;
    v_rollup_rec.svr_max_val.DELETE;
    v_rollup_rec.svr_sumx_val.DELETE;
    v_rollup_rec.svr_sumx2_val.DELETE;
    v_rollup_rec.db_min_val.DELETE;
    v_rollup_rec.db_max_val.DELETE;
    v_rollup_rec.db_sumx_val.DELETE;
    v_rollup_rec.db_sumx2_val.DELETE;

    -- Check cursor status here because cursor may return not found on a
    -- bulk fetch, even though records were retrieved.
    EXIT WHEN rollup_cursor%NOTFOUND;

  END LOOP; 

  -- Update the rollup timestamp
  UPDATE MGMT_TARGET_ROLLUP_TIMES
    SET rollup_timestamp = p_cur_day_rollup
    WHERE rowid = p_cur_rollup_rowids(p_url_1day_idx);

  COMMIT;

  IF rollup_cursor%ISOPEN THEN
    CLOSE rollup_cursor;
  END IF;

  -- Update the global row counters
  p_total_rc := p_total_rc + v_rc;
  p_url_1day_rollup_rc := p_url_1day_rollup_rc + v_rc;

EXCEPTION

  WHEN DUP_VAL_ON_INDEX THEN
    -- Partial rollup detected.
 
   -- Rollback any partial work
    ROLLBACK;

    -- Free all resources used
    IF rollup_cursor%ISOPEN THEN
      CLOSE rollup_cursor;
    END IF;
    v_rollup_rec.rollup_time.DELETE;
    v_rollup_rec.metric_name.DELETE;
    v_rollup_rec.key_val.DELETE;
    v_rollup_rec.url_link.DELETE;
    v_rollup_rec.hits.DELETE;
    v_rollup_rec.min_val.DELETE;
    v_rollup_rec.max_val.DELETE;
    v_rollup_rec.sumx_val.DELETE;
    v_rollup_rec.sumx2_val.DELETE;
    v_rollup_rec.svr_min_val.DELETE;
    v_rollup_rec.svr_max_val.DELETE;
    v_rollup_rec.svr_sumx_val.DELETE;
    v_rollup_rec.svr_sumx2_val.DELETE;
    v_rollup_rec.db_min_val.DELETE;
    v_rollup_rec.db_max_val.DELETE;
    v_rollup_rec.db_sumx_val.DELETE;
    v_rollup_rec.db_sumx2_val.DELETE;

    -- Cleanup all partial work.  If the cleanup fails, update the rollup
    -- timestamp so that the rollup doesn't get stuck forever.
    IF (EMD_URL_1DAY_CLEANUP = TRUE) THEN
      MGMT_LOG.LOG_ERROR('RTROLLUP', null,
        'Successfully cleaned up partial url 1day rollup (target=' ||
        p_cur_target_guid || ')', 'MGMT', NULL, NULL, NULL, MGMT_GLOBAL.G_INFO);
    ELSE
      MGMT_LOG.LOG_ERROR('RTROLLUP', null,
        'Unable to cleanup the partial url 1day rollup (target=' ||
        p_cur_target_guid || ')');
      BEGIN
        UPDATE MGMT_TARGET_ROLLUP_TIMES
          SET rollup_timestamp = p_cur_day_rollup
          WHERE rowid = p_cur_rollup_rowids(p_url_1day_idx);
        COMMIT;
        MGMT_LOG.LOG_ERROR('RTROLLUP', null,
          'URL 1day rollups for the period [' ||
          TO_CHAR(p_cur_rollup_timestamps(p_url_1day_idx),
          'MM:DD:YYYY') || ', ' ||
          TO_CHAR(p_cur_day_rollup, 'MM:DD:YYYY') || '] 
          cannot be performed (target=' || p_cur_target_guid || ')');
      EXCEPTION
        WHEN OTHERS THEN
          MGMT_LOG.LOG_ERROR('RTROLLUP', null,
            'Fatal Error: Unable to set the url 1day timestamp (target=' ||
            p_cur_target_guid || '). Error: ' || SUBSTR(SQLERRM, 1, 1000));
          RAISE;
      END;
    END IF;

  WHEN OTHERS THEN
    -- Assumes permanent error and attempts to update the rollup timestamp,
    -- thus losing the rollup period in question.
    -- This should be improved upon, we need to determine if the error is
    -- permanent or transient and only update the timestamp if it's permanent.
   
    -- Rollback any partial work
    ROLLBACK;

    -- Free all resources used
    IF rollup_cursor%ISOPEN THEN
      CLOSE rollup_cursor;
    END IF;
    v_rollup_rec.rollup_time.DELETE;
    v_rollup_rec.metric_name.DELETE;
    v_rollup_rec.key_val.DELETE;
    v_rollup_rec.url_base.DELETE;
    v_rollup_rec.hits.DELETE;
    v_rollup_rec.min_val.DELETE;
    v_rollup_rec.max_val.DELETE;
    v_rollup_rec.sumx_val.DELETE;
    v_rollup_rec.sumx2_val.DELETE;
    v_rollup_rec.svr_min_val.DELETE;
    v_rollup_rec.svr_max_val.DELETE;
    v_rollup_rec.svr_sumx_val.DELETE;
    v_rollup_rec.svr_sumx2_val.DELETE;
    v_rollup_rec.db_min_val.DELETE;
    v_rollup_rec.db_max_val.DELETE;
    v_rollup_rec.db_sumx_val.DELETE;
    v_rollup_rec.db_sumx2_val.DELETE;

    MGMT_LOG.LOG_ERROR('RTROLLUP', null,
      'The following unrecoverable error occurred while performing the' ||
      ' url 1day rollup (target=' || p_cur_target_guid || ').  Attempting to' ||
      ' skip over rollup period [' || 
      TO_CHAR(p_cur_rollup_timestamps(p_url_1day_idx),
      'MM:DD:YYYY') || ', ' ||
      TO_CHAR(p_cur_day_rollup,
      'MM:DD:YYYY') || '].  Error: ' || SUBSTR(SQLERRM, 1, 1000));

    BEGIN
      UPDATE MGMT_TARGET_ROLLUP_TIMES
        SET rollup_timestamp = p_cur_day_rollup
        WHERE rowid = p_cur_rollup_rowids(p_url_1day_idx);
      COMMIT;
      MGMT_LOG.LOG_ERROR('RTROLLUP', null,
        'Successfully skipped over period.  URL 1day rollup data will not be' ||
        ' generated for this period (target=' || p_cur_target_guid || ').');
    EXCEPTION
      WHEN OTHERS THEN
        MGMT_LOG.LOG_ERROR('RTROLLUP', null,
          'Fatal Error: Unable to set the url 1day timestamp (target=' ||
          p_cur_target_guid || '). Error: ' || SUBSTR(SQLERRM, 1, 1000));
        RAISE;
    END;

END EMD_URL_1DAY_ROLLUP;
----------------------------------------------------------------------

----------------------------------------------------------------------
PROCEDURE EMD_URL_DIST_1DAY_ROLLUP IS

  v_rollup_rec      p_dist_rollup_record_type;
  v_rc              PLS_INTEGER := 0; 

  CURSOR rollup_cursor( c_target_guid RAW,
                        c_prev_rollup DATE,
                        c_cur_rollup DATE) IS
    SELECT TRUNC(rollup_timestamp, 'DD'), metric_name,
      url_filename,
      SUM(hits), num_seconds, dist_value_type
    FROM MGMT_RT_URL_DIST_1HOUR
    WHERE target_guid = c_target_guid
      AND TRUNC(rollup_timestamp, 'DD') > c_prev_rollup
      AND TRUNC(rollup_timestamp, 'DD') <= c_cur_rollup
    GROUP BY TRUNC(rollup_timestamp, 'DD'), metric_name, url_filename, num_seconds, dist_value_type;

BEGIN

  OPEN rollup_cursor( p_cur_target_guid, 
                      p_cur_rollup_timestamps(p_url_dist_1day_idx),
                      p_cur_day_rollup ); 
  LOOP

    -- Process batchsize number of rows per commit.
    FETCH rollup_cursor BULK COLLECT INTO v_rollup_rec.rollup_time,
                                          v_rollup_rec.metric_name,
                                          v_rollup_rec.key_val,
                                          v_rollup_rec.hits,
                                          v_rollup_rec.dist_period,
                                          v_rollup_rec.dist_type 
                                          LIMIT p_rollup_batchsize;

    -- Exit loop if no rollup records are generated.
    EXIT WHEN NOT v_rollup_rec.rollup_time.EXISTS(1);

    -- Bulk inserts
    FORALL i IN v_rollup_rec.rollup_time.FIRST..v_rollup_rec.rollup_time.LAST
          INSERT INTO MGMT_RT_URL_DIST_1DAY
          ( 
            target_guid, metric_name, 
            rollup_timestamp,
            url_filename, 
            hits, num_seconds,dist_value_type
          )
          VALUES
          ( 
            p_cur_target_guid, v_rollup_rec.metric_name(i),
            v_rollup_rec.rollup_time(i),
            v_rollup_rec.key_val(i), 
            v_rollup_rec.hits(i),
            v_rollup_rec.dist_period(i),
            v_rollup_rec.dist_type(i)
          ); 

    COMMIT;

    -- Keep track of the number of inserts
    v_rc := v_rc + v_rollup_rec.key_val.COUNT;

    -- Free temporary tables
    v_rollup_rec.rollup_time.DELETE;
    v_rollup_rec.metric_name.DELETE;
    v_rollup_rec.key_val.DELETE;
    v_rollup_rec.hits.DELETE;
    v_rollup_rec.dist_period.DELETE;
    v_rollup_rec.dist_type.DELETE;

    -- Check cursor status here because cursor may return not found on a
    -- bulk fetch, even though records were retrieved.
    EXIT WHEN rollup_cursor%NOTFOUND;

  END LOOP; 

  -- Update the rollup timestamp
  UPDATE MGMT_TARGET_ROLLUP_TIMES
    SET rollup_timestamp = p_cur_day_rollup
    WHERE rowid = p_cur_rollup_rowids(p_url_dist_1day_idx);

  COMMIT;

  IF rollup_cursor%ISOPEN THEN
    CLOSE rollup_cursor;
  END IF;

  -- Update the global row counters
  p_total_rc := p_total_rc + v_rc;
  p_url_dist_1day_rollup_rc := p_url_dist_1day_rollup_rc + v_rc;

EXCEPTION

  WHEN DUP_VAL_ON_INDEX THEN
    -- Partial rollup detected.

    -- Rollback any partial work
    ROLLBACK;

    -- Free all resources used
    IF rollup_cursor%ISOPEN THEN
      CLOSE rollup_cursor;
    END IF;
    v_rollup_rec.rollup_time.DELETE;
    v_rollup_rec.metric_name.DELETE;
    v_rollup_rec.key_val.DELETE;
    v_rollup_rec.hits.DELETE;
    v_rollup_rec.dist_period.DELETE;
    v_rollup_rec.dist_type.DELETE;

    -- Cleanup all partial work.  If the cleanup fails, update the rollup
    -- timestamp so that the rollup doesn't get stuck forever.
    IF (EMD_URL_DIST_1DAY_CLEANUP = TRUE) THEN
      MGMT_LOG.LOG_ERROR('RTROLLUP', null,
        'Successfully cleaned up partial url dist 1day rollup (target=' ||
        p_cur_target_guid || ')', 'MGMT', NULL, NULL, NULL, MGMT_GLOBAL.G_INFO);
    ELSE
      MGMT_LOG.LOG_ERROR('RTROLLUP', null,
        'Unable to cleanup the partial url dist 1day rollup (target=' ||
        p_cur_target_guid || ')');
      BEGIN
        UPDATE MGMT_TARGET_ROLLUP_TIMES
          SET rollup_timestamp = p_cur_day_rollup
          WHERE rowid = p_cur_rollup_rowids(p_url_dist_1day_idx);
        COMMIT;
        MGMT_LOG.LOG_ERROR('RTROLLUP', null,
          'URL dist 1day rollups for the period [' ||
          TO_CHAR(p_cur_rollup_timestamps(p_url_dist_1day_idx),
          'MM:DD:YYYY') || ', ' ||
          TO_CHAR(p_cur_day_rollup, 'MM:DD:YYYY') || ']
          cannot be performed (target=' || p_cur_target_guid || ')');
      EXCEPTION
        WHEN OTHERS THEN
          MGMT_LOG.LOG_ERROR('RTROLLUP', null,
            'Fatal Error: Unable to set the url dist 1day timestamp (target=' ||
            p_cur_target_guid || '). Error: ' || SUBSTR(SQLERRM, 1, 1900));
          RAISE;
      END;
    END IF;

  WHEN OTHERS THEN
    -- Assumes permanent error and attempts to update the rollup timestamp,
    -- thus losing the rollup period in question.
    -- This should be improved upon, we need to determine if the error is
    -- permanent or transient and only update the timestamp if it's permanent.
   
    -- Rollback any partial work
    ROLLBACK;

    -- Free all resources used
    IF rollup_cursor%ISOPEN THEN
      CLOSE rollup_cursor;
    END IF;
    v_rollup_rec.rollup_time.DELETE;
    v_rollup_rec.metric_name.DELETE;
    v_rollup_rec.key_val.DELETE;
    v_rollup_rec.hits.DELETE;
    v_rollup_rec.dist_period.DELETE;
    v_rollup_rec.dist_type.DELETE;

    MGMT_LOG.LOG_ERROR('RTROLLUP', null,
      'The following unrecoverable error occurred while performing the' ||
      ' url dist 1day rollup (target=' || p_cur_target_guid || ').  Attempting to' ||
      ' skip over rollup period [' ||
      TO_CHAR(p_cur_rollup_timestamps(p_url_dist_1day_idx),
      'MM:DD:YYYY') || ', ' ||
      TO_CHAR(p_cur_day_rollup,
      'MM:DD:YYYY') || '].  Error: ' || SUBSTR(SQLERRM, 1, 1000));

    BEGIN
      UPDATE MGMT_TARGET_ROLLUP_TIMES
        SET rollup_timestamp = p_cur_day_rollup
        WHERE rowid = p_cur_rollup_rowids(p_url_dist_1day_idx);
      COMMIT;
      MGMT_LOG.LOG_ERROR('RTROLLUP', null,
        'Successfully skipped over period.  URL dist 1day rollup data will not be' ||
        ' generated for this period (target=' || p_cur_target_guid || ').');
    EXCEPTION
      WHEN OTHERS THEN
        MGMT_LOG.LOG_ERROR('RTROLLUP', null,
          'Fatal Error: Unable to set the url dist 1day timestamp (target=' ||
          p_cur_target_guid || '). Error: ' || SUBSTR(SQLERRM, 1, 1000));
        RAISE;
    END;

END EMD_URL_DIST_1DAY_ROLLUP;
----------------------------------------------------------------------

----------------------------------------------------------------------
PROCEDURE EMD_DOM_1DAY_ROLLUP IS

  v_rollup_rec      p_day_rollup_record_type;
  v_rc              PLS_INTEGER := 0; 

  CURSOR rollup_cursor( c_target_guid RAW,
                        c_prev_rollup DATE,
                        c_cur_rollup DATE) IS
    SELECT TRUNC(rollup_timestamp, 'DD'), metric_name,
      visitor_domain, visitor_subnet, visitor_subnet_num,
      SUM(hits),
      MIN(response_time_minimum),
      MAX(response_time_maximum), 
      SUM(hits * response_time_average),
      SUM((response_time_variance * (hits-1)) + 
          (hits * POWER(response_time_average, 2))),
      MIN(server_time_minimum),
      MAX(server_time_maximum), 
      SUM(hits * server_time_average),
      SUM((server_time_variance * (hits-1)) + 
          (hits * POWER(server_time_average, 2))),
      MIN(db_time_minimum),
      MAX(db_time_maximum), 
      SUM(hits * db_time_average),
      SUM((db_time_variance * (hits-1)) + 
          (hits * POWER(db_time_average, 2)))
    FROM MGMT_RT_DOMAIN_1HOUR
    WHERE target_guid = c_target_guid 
      AND TRUNC(rollup_timestamp, 'DD') > c_prev_rollup
      AND TRUNC(rollup_timestamp, 'DD') <= c_cur_rollup
    GROUP BY TRUNC(rollup_timestamp, 'DD'), metric_name,
      visitor_domain, visitor_subnet, visitor_subnet_num;

BEGIN

  OPEN rollup_cursor( p_cur_target_guid, 
                      p_cur_rollup_timestamps(p_dom_1day_idx),
                      p_cur_day_rollup ); 
  LOOP

    -- Process batchsize number of rows per commit.
    FETCH rollup_cursor BULK COLLECT INTO v_rollup_rec.rollup_time,
                                          v_rollup_rec.metric_name,
                                          v_rollup_rec.key_val,
                                          v_rollup_rec.subnet,
                                          v_rollup_rec.subnet_num,
                                          v_rollup_rec.hits,
                                          v_rollup_rec.min_val,
                                          v_rollup_rec.max_val,
                                          v_rollup_rec.sumx_val,
                                          v_rollup_rec.sumx2_val,
                                          v_rollup_rec.svr_min_val,
                                          v_rollup_rec.svr_max_val,
                                          v_rollup_rec.svr_sumx_val,
                                          v_rollup_rec.svr_sumx2_val,
                                          v_rollup_rec.db_min_val,
                                          v_rollup_rec.db_max_val,
                                          v_rollup_rec.db_sumx_val,
                                          v_rollup_rec.db_sumx2_val
                                          LIMIT p_rollup_batchsize;

    -- Exit loop if no rollup records are generated.
    EXIT WHEN NOT v_rollup_rec.rollup_time.EXISTS(1);

    -- Bulk inserts
    FORALL i IN v_rollup_rec.rollup_time.FIRST..v_rollup_rec.rollup_time.LAST
      INSERT INTO MGMT_RT_DOMAIN_1DAY
      ( 
        target_guid, metric_name, 
        rollup_timestamp,
        visitor_domain, visitor_subnet, visitor_subnet_num,
        hits, 
        response_time_average, 
        response_time_minimum, 
        response_time_maximum, 
        response_time_sdev, 
        response_time_variance,
        server_time_average, 
        server_time_minimum, 
        server_time_maximum, 
        server_time_sdev, 
        server_time_variance, 
        db_time_average, 
        db_time_minimum, 
        db_time_maximum, 
        db_time_sdev, 
        db_time_variance
      )
      VALUES
      ( 
        p_cur_target_guid, v_rollup_rec.metric_name(i),
        v_rollup_rec.rollup_time(i),
        v_rollup_rec.key_val(i), 
        v_rollup_rec.subnet(i), 
        v_rollup_rec.subnet_num(i), 
        v_rollup_rec.hits(i), 
        DECODE(v_rollup_rec.hits(i), 
               0, 0,
               v_rollup_rec.sumx_val(i)/v_rollup_rec.hits(i)),
        v_rollup_rec.min_val(i), 
        v_rollup_rec.max_val(i),  
        DECODE(v_rollup_rec.hits(i), 
               0, 0,
               1, 0,
               SQRT(((v_rollup_rec.hits(i) * v_rollup_rec.sumx2_val(i) - 
                       POWER(v_rollup_rec.sumx_val(i),2))
                      / (v_rollup_rec.hits(i) * (v_rollup_rec.hits(i) - 1))))),
        DECODE(v_rollup_rec.hits(i), 
               0, 0,
               1, 0,
               ((v_rollup_rec.hits(i) * v_rollup_rec.sumx2_val(i) - 
                 POWER(v_rollup_rec.sumx_val(i),2))
                / (v_rollup_rec.hits(i) * (v_rollup_rec.hits(i) - 1)))),
        DECODE(v_rollup_rec.hits(i), 
               0, 0,
               v_rollup_rec.svr_sumx_val(i)/v_rollup_rec.hits(i)),
        v_rollup_rec.svr_min_val(i), 
        v_rollup_rec.svr_max_val(i),  
        DECODE(v_rollup_rec.hits(i), 
               0, 0,
               1, 0,
               SQRT(((v_rollup_rec.hits(i) * v_rollup_rec.svr_sumx2_val(i) - 
                       POWER(v_rollup_rec.svr_sumx_val(i),2))
                      / (v_rollup_rec.hits(i) * (v_rollup_rec.hits(i) - 1))))),
        DECODE(v_rollup_rec.hits(i), 
               0, 0,
               1, 0,
               ((v_rollup_rec.hits(i) * v_rollup_rec.svr_sumx2_val(i) - 
                 POWER(v_rollup_rec.svr_sumx_val(i),2))
                / (v_rollup_rec.hits(i) * (v_rollup_rec.hits(i) - 1)))),
        DECODE(v_rollup_rec.hits(i), 
               0, 0,
               v_rollup_rec.db_sumx_val(i)/v_rollup_rec.hits(i)),
        v_rollup_rec.db_min_val(i), 
        v_rollup_rec.db_max_val(i),  
        DECODE(v_rollup_rec.hits(i), 
               0, 0,
               1, 0,
               SQRT(((v_rollup_rec.hits(i) * v_rollup_rec.db_sumx2_val(i) - 
                       POWER(v_rollup_rec.db_sumx_val(i),2))
                      / (v_rollup_rec.hits(i) * (v_rollup_rec.hits(i) - 1))))),
        DECODE(v_rollup_rec.hits(i), 
               0, 0,
               1, 0,
               ((v_rollup_rec.hits(i) * v_rollup_rec.db_sumx2_val(i) - 
                 POWER(v_rollup_rec.db_sumx_val(i),2))
                / (v_rollup_rec.hits(i) * (v_rollup_rec.hits(i) - 1))))
      );

    COMMIT;

    -- Keep track of the number of inserts
    v_rc := v_rc + v_rollup_rec.key_val.COUNT;

    -- Free temporary tables
    v_rollup_rec.rollup_time.DELETE;
    v_rollup_rec.metric_name.DELETE;
    v_rollup_rec.key_val.DELETE;
    v_rollup_rec.subnet.DELETE;
    v_rollup_rec.subnet_num.DELETE;
    v_rollup_rec.hits.DELETE;
    v_rollup_rec.min_val.DELETE;
    v_rollup_rec.max_val.DELETE;
    v_rollup_rec.sumx_val.DELETE;
    v_rollup_rec.sumx2_val.DELETE;
    v_rollup_rec.svr_min_val.DELETE;
    v_rollup_rec.svr_max_val.DELETE;
    v_rollup_rec.svr_sumx_val.DELETE;
    v_rollup_rec.svr_sumx2_val.DELETE;
    v_rollup_rec.db_min_val.DELETE;
    v_rollup_rec.db_max_val.DELETE;
    v_rollup_rec.db_sumx_val.DELETE;
    v_rollup_rec.db_sumx2_val.DELETE;

    -- Check cursor status here because cursor may return not found on a
    -- bulk fetch, even though records were retrieved.
    EXIT WHEN rollup_cursor%NOTFOUND;

  END LOOP; 

  -- Update the rollup timestamp
  UPDATE MGMT_TARGET_ROLLUP_TIMES
    SET rollup_timestamp = p_cur_day_rollup
    WHERE rowid = p_cur_rollup_rowids(p_dom_1day_idx);

  COMMIT;

  IF rollup_cursor%ISOPEN THEN
    CLOSE rollup_cursor;
  END IF;

  -- Update the global row counters
  p_total_rc := p_total_rc + v_rc;
  p_dom_1day_rollup_rc := p_dom_1day_rollup_rc + v_rc;

EXCEPTION

  WHEN DUP_VAL_ON_INDEX THEN
    -- Partial rollup detected.

    -- Rollback any partial work
    ROLLBACK;

    -- Free all resources used
    IF rollup_cursor%ISOPEN THEN
      CLOSE rollup_cursor;
    END IF;
    v_rollup_rec.rollup_time.DELETE;
    v_rollup_rec.metric_name.DELETE;
    v_rollup_rec.key_val.DELETE;
    v_rollup_rec.subnet.DELETE;
    v_rollup_rec.subnet_num.DELETE;
    v_rollup_rec.hits.DELETE;
    v_rollup_rec.min_val.DELETE;
    v_rollup_rec.max_val.DELETE;
    v_rollup_rec.sumx_val.DELETE;
    v_rollup_rec.sumx2_val.DELETE;
    v_rollup_rec.svr_min_val.DELETE;
    v_rollup_rec.svr_max_val.DELETE;
    v_rollup_rec.svr_sumx_val.DELETE;
    v_rollup_rec.svr_sumx2_val.DELETE;
    v_rollup_rec.db_min_val.DELETE;
    v_rollup_rec.db_max_val.DELETE;
    v_rollup_rec.db_sumx_val.DELETE;
    v_rollup_rec.db_sumx2_val.DELETE;

    -- Cleanup all partial work.  If the cleanup fails, update the rollup
    -- timestamp so that the rollup doesn't get stuck forever.
    IF (EMD_DOM_1DAY_CLEANUP = TRUE) THEN
      MGMT_LOG.LOG_ERROR('RTROLLUP', null,
        'Successfully cleaned up partial dom 1day rollup (target=' ||
        p_cur_target_guid || ')', 'MGMT', NULL, NULL, NULL, MGMT_GLOBAL.G_INFO);
    ELSE
      MGMT_LOG.LOG_ERROR('RTROLLUP', null,
        'Unable to cleanup the partial dom 1day rollup (target=' ||
        p_cur_target_guid || ')');
      BEGIN
        UPDATE MGMT_TARGET_ROLLUP_TIMES
          SET rollup_timestamp = p_cur_day_rollup
          WHERE rowid = p_cur_rollup_rowids(p_dom_1day_idx);
        COMMIT;
        MGMT_LOG.LOG_ERROR('RTROLLUP', null,
          'Dom 1day rollups for the period [' ||
          TO_CHAR(p_cur_rollup_timestamps(p_dom_1day_idx),
          'MM:DD:YYYY') || ', ' ||
          TO_CHAR(p_cur_day_rollup, 'MM:DD:YYYY') || '] 
          cannot be performed (target=' || p_cur_target_guid || ')');
      EXCEPTION
        WHEN OTHERS THEN
          MGMT_LOG.LOG_ERROR('RTROLLUP', null,
            'Fatal Error: Unable to set the dom 1day timestamp (target=' ||
            p_cur_target_guid || '). Error: ' || SUBSTR(SQLERRM, 1, 1000));
          RAISE;
      END;
    END IF;

  WHEN OTHERS THEN
    -- Assumes permanent error and attempts to update the rollup timestamp,
    -- thus losing the rollup period in question.
    -- This should be improved upon, we need to determine if the error is
    -- permanent or transient and only update the timestamp if it's permanent.
   
    -- Rollback any partial work
    ROLLBACK;

    -- Free all resources used
    IF rollup_cursor%ISOPEN THEN
      CLOSE rollup_cursor;
    END IF;
    v_rollup_rec.rollup_time.DELETE;
    v_rollup_rec.metric_name.DELETE;
    v_rollup_rec.key_val.DELETE;
    v_rollup_rec.subnet.DELETE;
    v_rollup_rec.subnet_num.DELETE;
    v_rollup_rec.hits.DELETE;
    v_rollup_rec.min_val.DELETE;
    v_rollup_rec.max_val.DELETE;
    v_rollup_rec.sumx_val.DELETE;
    v_rollup_rec.sumx2_val.DELETE;
    v_rollup_rec.svr_min_val.DELETE;
    v_rollup_rec.svr_max_val.DELETE;
    v_rollup_rec.svr_sumx_val.DELETE;
    v_rollup_rec.svr_sumx2_val.DELETE;
    v_rollup_rec.db_min_val.DELETE;
    v_rollup_rec.db_max_val.DELETE;
    v_rollup_rec.db_sumx_val.DELETE;
    v_rollup_rec.db_sumx2_val.DELETE;

    MGMT_LOG.LOG_ERROR('RTROLLUP', null,
      'The following unrecoverable error occurred while performing the' ||
      ' dom 1day rollup (target=' || p_cur_target_guid || ').  Attempting to' ||
      ' skip over rollup period [' || 
      TO_CHAR(p_cur_rollup_timestamps(p_dom_1day_idx),
      'MM:DD:YYYY') || ', ' ||
      TO_CHAR(p_cur_day_rollup,
      'MM:DD:YYYY') || '].  Error: ' || SUBSTR(SQLERRM, 1, 1000));

    BEGIN
      UPDATE MGMT_TARGET_ROLLUP_TIMES
        SET rollup_timestamp = p_cur_day_rollup
        WHERE rowid = p_cur_rollup_rowids(p_dom_1day_idx);
      COMMIT;
      MGMT_LOG.LOG_ERROR('RTROLLUP', null,
        'Successfully skipped over period.  Dom 1day rollup data will not be' ||
        ' generated for this period (target=' || p_cur_target_guid || ').');
    EXCEPTION
      WHEN OTHERS THEN
        MGMT_LOG.LOG_ERROR('RTROLLUP', null,
          'Fatal Error: Unable to set the dom 1day timestamp (target=' ||
          p_cur_target_guid || '). Error: ' || SUBSTR(SQLERRM, 1, 1000));
        RAISE;
    END;

END EMD_DOM_1DAY_ROLLUP;
----------------------------------------------------------------------

----------------------------------------------------------------------
PROCEDURE EMD_DOM_DIST_1DAY_ROLLUP IS

  v_rollup_rec      p_dist_rollup_record_type;
  v_rc              PLS_INTEGER := 0; 

  CURSOR rollup_cursor( c_target_guid RAW,
                        c_prev_rollup DATE,
                        c_cur_rollup DATE) IS
    SELECT TRUNC(rollup_timestamp, 'DD'), metric_name,
      visitor_domain, visitor_subnet, visitor_subnet_num,
      SUM(hits), 
      num_seconds,dist_value_type
    FROM MGMT_RT_DOMAIN_DIST_1HOUR
    WHERE target_guid = c_target_guid
      AND TRUNC(rollup_timestamp, 'DD') > c_prev_rollup
      AND TRUNC(rollup_timestamp, 'DD') <= c_cur_rollup
    GROUP BY 
      TRUNC(rollup_timestamp, 'DD'), metric_name, visitor_domain, 
          visitor_subnet, visitor_subnet_num, num_seconds,dist_value_type;

BEGIN

  OPEN rollup_cursor( p_cur_target_guid, 
                      p_cur_rollup_timestamps(p_dom_dist_1day_idx),
                      p_cur_day_rollup ); 
  LOOP

    -- Process batchsize number of rows per commit.
    FETCH rollup_cursor BULK COLLECT INTO v_rollup_rec.rollup_time,
                                          v_rollup_rec.metric_name,
                                          v_rollup_rec.key_val,
                                          v_rollup_rec.subnet,
                                          v_rollup_rec.subnet_num,
                                          v_rollup_rec.hits,
                                          v_rollup_rec.dist_period,
                                          v_rollup_rec.dist_type
                                          LIMIT p_rollup_batchsize;

    -- Exit loop if no rollup records are generated.
    EXIT WHEN NOT v_rollup_rec.rollup_time.EXISTS(1);

    -- Bulk inserts
    FORALL i IN v_rollup_rec.rollup_time.FIRST..v_rollup_rec.rollup_time.LAST
          INSERT INTO MGMT_RT_DOMAIN_DIST_1DAY
          ( 
            target_guid, metric_name, 
            rollup_timestamp,
            visitor_domain, visitor_subnet, visitor_subnet_num, 
            hits, num_seconds, dist_value_type
          )
          VALUES
          ( 
            p_cur_target_guid, v_rollup_rec.metric_name(i),
            v_rollup_rec.rollup_time(i),
            v_rollup_rec.key_val(i),
            v_rollup_rec.subnet(i), 
            v_rollup_rec.subnet_num(i), 
            v_rollup_rec.hits(i),
            v_rollup_rec.dist_period(i),
            v_rollup_rec.dist_type(i)
          ); 

    COMMIT;

    -- Keep track of the number of inserts
    v_rc := v_rc + v_rollup_rec.key_val.COUNT;

    -- Free temporary tables
    v_rollup_rec.rollup_time.DELETE;
    v_rollup_rec.metric_name.DELETE;
    v_rollup_rec.key_val.DELETE;
    v_rollup_rec.subnet.DELETE;
    v_rollup_rec.subnet_num.DELETE;
    v_rollup_rec.hits.DELETE;
    v_rollup_rec.dist_period.DELETE;
    v_rollup_rec.dist_type.DELETE;

    -- Check cursor status here because cursor may return not found on a
    -- bulk fetch, even though records were retrieved.
    EXIT WHEN rollup_cursor%NOTFOUND;

  END LOOP; 

  -- Update the rollup timestamp
  UPDATE MGMT_TARGET_ROLLUP_TIMES
    SET rollup_timestamp = p_cur_day_rollup
    WHERE rowid = p_cur_rollup_rowids(p_dom_dist_1day_idx);

  COMMIT;

  IF rollup_cursor%ISOPEN THEN
    CLOSE rollup_cursor;
  END IF;

  -- Update the global row counters
  p_total_rc := p_total_rc + v_rc;
  p_dom_dist_1day_rollup_rc := p_dom_dist_1day_rollup_rc + v_rc;

EXCEPTION

  WHEN DUP_VAL_ON_INDEX THEN
    -- Partial rollup detected.

    -- Rollback any partial work
    ROLLBACK;

    -- Free all resources used
    IF rollup_cursor%ISOPEN THEN
      CLOSE rollup_cursor;
    END IF;
    v_rollup_rec.rollup_time.DELETE;
    v_rollup_rec.metric_name.DELETE;
    v_rollup_rec.key_val.DELETE;
    v_rollup_rec.subnet.DELETE;
    v_rollup_rec.subnet_num.DELETE;
    v_rollup_rec.hits.DELETE;
    v_rollup_rec.dist_period.DELETE;
    v_rollup_rec.dist_type.DELETE;

    -- Cleanup all partial work.  If the cleanup fails, update the rollup
    -- timestamp so that the rollup doesn't get stuck forever.
    IF (EMD_DOM_DIST_1DAY_CLEANUP = TRUE) THEN
      MGMT_LOG.LOG_ERROR('RTROLLUP', null,
        'Successfully cleaned up partial dom dist 1day rollup (target=' ||
        p_cur_target_guid || ')', 'MGMT', NULL, NULL, NULL, MGMT_GLOBAL.G_INFO);
    ELSE
      MGMT_LOG.LOG_ERROR('RTROLLUP', null,
        'Unable to cleanup the partial dom dist 1day rollup (target=' ||
        p_cur_target_guid || ')');
      BEGIN
        UPDATE MGMT_TARGET_ROLLUP_TIMES
          SET rollup_timestamp = p_cur_day_rollup
          WHERE rowid = p_cur_rollup_rowids(p_dom_dist_1day_idx);
        COMMIT;
        MGMT_LOG.LOG_ERROR('RTROLLUP', null,
          'Dom dist 1day rollups for the period [' ||
          TO_CHAR(p_cur_rollup_timestamps(p_dom_dist_1day_idx),
          'MM:DD:YYYY') || ', ' ||
          TO_CHAR(p_cur_day_rollup, 'MM:DD:YYYY') || '] 
          cannot be performed (target=' || p_cur_target_guid || ')');
      EXCEPTION
        WHEN OTHERS THEN
          MGMT_LOG.LOG_ERROR('RTROLLUP', null,
            'Fatal Error: Unable to set the dom dist 1day timestamp (target=' ||
            p_cur_target_guid || '). Error: ' || SUBSTR(SQLERRM, 1, 1000));
          RAISE;
      END;
    END IF;

  WHEN OTHERS THEN
    -- Assumes permanent error and attempts to update the rollup timestamp,
    -- thus losing the rollup period in question.
    -- This should be improved upon, we need to determine if the error is
    -- permanent or transient and only update the timestamp if it's permanent.
   
    -- Rollback any partial work
    ROLLBACK;

    -- Free all resources used
    IF rollup_cursor%ISOPEN THEN
      CLOSE rollup_cursor;
    END IF;
    v_rollup_rec.rollup_time.DELETE;
    v_rollup_rec.metric_name.DELETE;
    v_rollup_rec.key_val.DELETE;
    v_rollup_rec.subnet.DELETE;
    v_rollup_rec.subnet_num.DELETE;
    v_rollup_rec.hits.DELETE;
    v_rollup_rec.dist_period.DELETE;
    v_rollup_rec.dist_type.DELETE;

    MGMT_LOG.LOG_ERROR('RTROLLUP', null,
      'The following unrecoverable error occurred while performing the' ||
      ' dom dist 1day rollup (target=' || p_cur_target_guid || ').  Attempting to' ||
      ' skip over rollup period [' || 
      TO_CHAR(p_cur_rollup_timestamps(p_dom_dist_1day_idx),
      'MM:DD:YYYY') || ', ' ||
      TO_CHAR(p_cur_day_rollup,
      'MM:DD:YYYY') || '].  Error: ' || SUBSTR(SQLERRM, 1, 1000));

    BEGIN
      UPDATE MGMT_TARGET_ROLLUP_TIMES
        SET rollup_timestamp = p_cur_day_rollup
        WHERE rowid = p_cur_rollup_rowids(p_dom_dist_1day_idx);
      COMMIT;
      MGMT_LOG.LOG_ERROR('RTROLLUP', null,
        'Successfully skipped over period.  Dom dist 1day rollup data will not be' ||
        ' generated for this period (target=' || p_cur_target_guid || ').');
    EXCEPTION
      WHEN OTHERS THEN
        MGMT_LOG.LOG_ERROR('RTROLLUP', null,
          'Fatal Error: Unable to set the dom dist 1day timestamp (target=' ||
          p_cur_target_guid || '). Error: ' || SUBSTR(SQLERRM, 1, 1000));
        RAISE;
    END;

END EMD_DOM_DIST_1DAY_ROLLUP;
----------------------------------------------------------------------

----------------------------------------------------------------------
PROCEDURE EMD_IP_1DAY_ROLLUP IS

  v_rollup_rec      p_day_rollup_record_type;
  v_rc              PLS_INTEGER := 0; 

  CURSOR rollup_cursor( c_target_guid RAW,
                        c_prev_rollup DATE,
                        c_cur_rollup DATE) IS
    SELECT TRUNC(rollup_timestamp, 'DD'), metric_name,
      visitor_node,
      SUM(hits),
      MIN(response_time_minimum),
      MAX(response_time_maximum), 
      SUM(hits * response_time_average),
      SUM((response_time_variance * (hits-1)) + 
          (hits * POWER(response_time_average, 2))),
      MIN(server_time_minimum),
      MAX(server_time_maximum), 
      SUM(hits * server_time_average),
      SUM((server_time_variance * (hits-1)) + 
          (hits * POWER(server_time_average, 2))),
      MIN(db_time_minimum),
      MAX(db_time_maximum), 
      SUM(hits * db_time_average),
      SUM((db_time_variance * (hits-1)) + 
          (hits * POWER(db_time_average, 2)))
    FROM MGMT_RT_IP_1HOUR
    WHERE target_guid = c_target_guid 
      AND TRUNC(rollup_timestamp, 'DD') > c_prev_rollup
      AND TRUNC(rollup_timestamp, 'DD') <= c_cur_rollup
    GROUP BY TRUNC(rollup_timestamp, 'DD'), metric_name, visitor_node;

BEGIN

  OPEN rollup_cursor( p_cur_target_guid, 
                      p_cur_rollup_timestamps(p_ip_1day_idx),
                      p_cur_day_rollup ); 
  LOOP

    -- Process batchsize number of rows per commit.
    FETCH rollup_cursor BULK COLLECT INTO v_rollup_rec.rollup_time,
                                          v_rollup_rec.metric_name,
                                          v_rollup_rec.key_val,
                                          v_rollup_rec.hits,
                                          v_rollup_rec.min_val,
                                          v_rollup_rec.max_val,
                                          v_rollup_rec.sumx_val,
                                          v_rollup_rec.sumx2_val,
                                          v_rollup_rec.svr_min_val,
                                          v_rollup_rec.svr_max_val,
                                          v_rollup_rec.svr_sumx_val,
                                          v_rollup_rec.svr_sumx2_val,
                                          v_rollup_rec.db_min_val,
                                          v_rollup_rec.db_max_val,
                                          v_rollup_rec.db_sumx_val,
                                          v_rollup_rec.db_sumx2_val
                                          LIMIT p_rollup_batchsize;

    -- Exit loop if no rollup records are generated.
    EXIT WHEN NOT v_rollup_rec.rollup_time.EXISTS(1);

    -- Bulk inserts
    FORALL i IN v_rollup_rec.rollup_time.FIRST..v_rollup_rec.rollup_time.LAST
      INSERT INTO MGMT_RT_IP_1DAY
      ( 
        target_guid, metric_name, 
        rollup_timestamp,
	visitor_node,
        hits, 
        response_time_average, 
        response_time_minimum, 
        response_time_maximum, 
        response_time_sdev, 
        response_time_variance,
        server_time_average, 
        server_time_minimum, 
        server_time_maximum, 
        server_time_sdev, 
        server_time_variance, 
        db_time_average, 
        db_time_minimum, 
        db_time_maximum, 
        db_time_sdev, 
        db_time_variance
      )
      VALUES
      ( 
        p_cur_target_guid, v_rollup_rec.metric_name(i),
        v_rollup_rec.rollup_time(i),
        v_rollup_rec.key_val(i), 
        v_rollup_rec.hits(i), 
        DECODE(v_rollup_rec.hits(i), 
               0, 0,
               v_rollup_rec.sumx_val(i)/v_rollup_rec.hits(i)),
        v_rollup_rec.min_val(i), 
        v_rollup_rec.max_val(i),  
        DECODE(v_rollup_rec.hits(i), 
               0, 0,
               1, 0,
               SQRT(((v_rollup_rec.hits(i) * v_rollup_rec.sumx2_val(i) - 
                       POWER(v_rollup_rec.sumx_val(i),2))
                      / (v_rollup_rec.hits(i) * (v_rollup_rec.hits(i) - 1))))),
        DECODE(v_rollup_rec.hits(i), 
               0, 0,
               1, 0,
               ((v_rollup_rec.hits(i) * v_rollup_rec.sumx2_val(i) - 
                 POWER(v_rollup_rec.sumx_val(i),2))
                / (v_rollup_rec.hits(i) * (v_rollup_rec.hits(i) - 1)))),
        DECODE(v_rollup_rec.hits(i), 
               0, 0,
               v_rollup_rec.svr_sumx_val(i)/v_rollup_rec.hits(i)),
        v_rollup_rec.svr_min_val(i), 
        v_rollup_rec.svr_max_val(i),  
        DECODE(v_rollup_rec.hits(i), 
               0, 0,
               1, 0,
               SQRT(((v_rollup_rec.hits(i) * v_rollup_rec.svr_sumx2_val(i) - 
                       POWER(v_rollup_rec.svr_sumx_val(i),2))
                      / (v_rollup_rec.hits(i) * (v_rollup_rec.hits(i) - 1))))),
        DECODE(v_rollup_rec.hits(i), 
               0, 0,
               1, 0,
               ((v_rollup_rec.hits(i) * v_rollup_rec.svr_sumx2_val(i) - 
                 POWER(v_rollup_rec.svr_sumx_val(i),2))
                / (v_rollup_rec.hits(i) * (v_rollup_rec.hits(i) - 1)))),
        DECODE(v_rollup_rec.hits(i), 
               0, 0,
               v_rollup_rec.db_sumx_val(i)/v_rollup_rec.hits(i)),
        v_rollup_rec.db_min_val(i), 
        v_rollup_rec.db_max_val(i),  
        DECODE(v_rollup_rec.hits(i), 
               0, 0,
               1, 0,
               SQRT(((v_rollup_rec.hits(i) * v_rollup_rec.db_sumx2_val(i) - 
                       POWER(v_rollup_rec.db_sumx_val(i),2))
                      / (v_rollup_rec.hits(i) * (v_rollup_rec.hits(i) - 1))))),
        DECODE(v_rollup_rec.hits(i), 
               0, 0,
               1, 0,
               ((v_rollup_rec.hits(i) * v_rollup_rec.db_sumx2_val(i) - 
                 POWER(v_rollup_rec.db_sumx_val(i),2))
                / (v_rollup_rec.hits(i) * (v_rollup_rec.hits(i) - 1))))
      );

    COMMIT;

    -- Keep track of the number of inserts
    v_rc := v_rc + v_rollup_rec.key_val.COUNT;

    -- Free temporary tables
    v_rollup_rec.rollup_time.DELETE;
    v_rollup_rec.metric_name.DELETE;
    v_rollup_rec.key_val.DELETE;
    v_rollup_rec.hits.DELETE;
    v_rollup_rec.min_val.DELETE;
    v_rollup_rec.max_val.DELETE;
    v_rollup_rec.sumx_val.DELETE;
    v_rollup_rec.sumx2_val.DELETE;
    v_rollup_rec.svr_min_val.DELETE;
    v_rollup_rec.svr_max_val.DELETE;
    v_rollup_rec.svr_sumx_val.DELETE;
    v_rollup_rec.svr_sumx2_val.DELETE;
    v_rollup_rec.db_min_val.DELETE;
    v_rollup_rec.db_max_val.DELETE;
    v_rollup_rec.db_sumx_val.DELETE;
    v_rollup_rec.db_sumx2_val.DELETE;

    -- Check cursor status here because cursor may return not found on a
    -- bulk fetch, even though records were retrieved.
    EXIT WHEN rollup_cursor%NOTFOUND;

  END LOOP; 

  -- Update the rollup timestamp
  UPDATE MGMT_TARGET_ROLLUP_TIMES
    SET rollup_timestamp = p_cur_day_rollup
    WHERE rowid = p_cur_rollup_rowids(p_ip_1day_idx);

  COMMIT;

  IF rollup_cursor%ISOPEN THEN
    CLOSE rollup_cursor;
  END IF;

  -- Update the global row counters
  p_total_rc := p_total_rc + v_rc;
  p_ip_1day_rollup_rc := p_ip_1day_rollup_rc + v_rc;

EXCEPTION

  WHEN DUP_VAL_ON_INDEX THEN
    -- Partial rollup detected.

    -- Rollback any partial work
    ROLLBACK;

    -- Free all resources used
    IF rollup_cursor%ISOPEN THEN
      CLOSE rollup_cursor;
    END IF;
    v_rollup_rec.rollup_time.DELETE;
    v_rollup_rec.metric_name.DELETE;
    v_rollup_rec.key_val.DELETE;
    v_rollup_rec.hits.DELETE;
    v_rollup_rec.min_val.DELETE;
    v_rollup_rec.max_val.DELETE;
    v_rollup_rec.sumx_val.DELETE;
    v_rollup_rec.sumx2_val.DELETE;
    v_rollup_rec.svr_min_val.DELETE;
    v_rollup_rec.svr_max_val.DELETE;
    v_rollup_rec.svr_sumx_val.DELETE;
    v_rollup_rec.svr_sumx2_val.DELETE;
    v_rollup_rec.db_min_val.DELETE;
    v_rollup_rec.db_max_val.DELETE;
    v_rollup_rec.db_sumx_val.DELETE;
    v_rollup_rec.db_sumx2_val.DELETE;

    -- Cleanup all partial work.  If the cleanup fails, update the rollup
    -- timestamp so that the rollup doesn't get stuck forever.
    IF (EMD_IP_1DAY_CLEANUP = TRUE) THEN
      MGMT_LOG.LOG_ERROR('RTROLLUP', null,
        'Successfully cleaned up partial ip 1day rollup (target=' ||
        p_cur_target_guid || ')', 'MGMT', NULL, NULL, NULL, MGMT_GLOBAL.G_INFO);
    ELSE
      MGMT_LOG.LOG_ERROR('RTROLLUP', null,
        'Unable to cleanup the partial ip 1day rollup (target=' ||
        p_cur_target_guid || ')');
      BEGIN
        UPDATE MGMT_TARGET_ROLLUP_TIMES
          SET rollup_timestamp = p_cur_day_rollup
          WHERE rowid = p_cur_rollup_rowids(p_ip_1day_idx);
        COMMIT;
        MGMT_LOG.LOG_ERROR('RTROLLUP', null,
          'IP 1day rollups for the period [' ||
          TO_CHAR(p_cur_rollup_timestamps(p_ip_1day_idx),
          'MM:DD:YYYY') || ', ' ||
          TO_CHAR(p_cur_day_rollup, 'MM:DD:YYYY') || ']
          cannot be performed (target=' || p_cur_target_guid || ')');
      EXCEPTION
        WHEN OTHERS THEN
          MGMT_LOG.LOG_ERROR('RTROLLUP', null,
            'Fatal Error: Unable to set the ip 1day timestamp (target=' ||
            p_cur_target_guid || '). Error: ' || SUBSTR(SQLERRM, 1, 1000));
          RAISE;
      END;
    END IF;

  WHEN OTHERS THEN
    -- Assumes permanent error and attempts to update the rollup timestamp,
    -- thus losing the rollup period in question.
    -- This should be improved upon, we need to determine if the error is
    -- permanent or transient and only update the timestamp if it's permanent.
   
    -- Rollback any partial work
    ROLLBACK;

    -- Free all resources used
    IF rollup_cursor%ISOPEN THEN
      CLOSE rollup_cursor;
    END IF;
    v_rollup_rec.rollup_time.DELETE;
    v_rollup_rec.metric_name.DELETE;
    v_rollup_rec.key_val.DELETE;
    v_rollup_rec.hits.DELETE;
    v_rollup_rec.min_val.DELETE;
    v_rollup_rec.max_val.DELETE;
    v_rollup_rec.sumx_val.DELETE;
    v_rollup_rec.sumx2_val.DELETE;
    v_rollup_rec.svr_min_val.DELETE;
    v_rollup_rec.svr_max_val.DELETE;
    v_rollup_rec.svr_sumx_val.DELETE;
    v_rollup_rec.svr_sumx2_val.DELETE;
    v_rollup_rec.db_min_val.DELETE;
    v_rollup_rec.db_max_val.DELETE;
    v_rollup_rec.db_sumx_val.DELETE;
    v_rollup_rec.db_sumx2_val.DELETE;

    MGMT_LOG.LOG_ERROR('RTROLLUP', null,
      'The following unrecoverable error occurred while performing the' ||
      ' ip 1day rollup (target=' || p_cur_target_guid || ').  Attempting to' ||
      ' skip over rollup period [' || 
      TO_CHAR(p_cur_rollup_timestamps(p_ip_1day_idx),
      'MM:DD:YYYY') || ', ' ||
      TO_CHAR(p_cur_day_rollup,
      'MM:DD:YYYY') || '].  Error: ' || SUBSTR(SQLERRM, 1, 1000));

    BEGIN
      UPDATE MGMT_TARGET_ROLLUP_TIMES
        SET rollup_timestamp = p_cur_day_rollup
        WHERE rowid = p_cur_rollup_rowids(p_ip_1day_idx);
      COMMIT;
      MGMT_LOG.LOG_ERROR('RTROLLUP', null,
        'Successfully skipped over period.  IP 1day rollup data will not be' ||
        ' generated for this period (target=' || p_cur_target_guid || ').');
    EXCEPTION
      WHEN OTHERS THEN
        MGMT_LOG.LOG_ERROR('RTROLLUP', null,
          'Fatal Error: Unable to set the ip 1day timestamp (target=' ||
          p_cur_target_guid || '). Error: ' || SUBSTR(SQLERRM, 1, 1000));
        RAISE;
    END;

END EMD_IP_1DAY_ROLLUP;
----------------------------------------------------------------------

----------------------------------------------------------------------
PROCEDURE EMD_IP_DIST_1DAY_ROLLUP IS

  v_rollup_rec      p_dist_rollup_record_type;
  v_rc              PLS_INTEGER := 0; 

  CURSOR rollup_cursor( c_target_guid RAW,
                        c_prev_rollup DATE,
                        c_cur_rollup DATE) IS
    SELECT TRUNC(rollup_timestamp, 'DD'), metric_name,
      visitor_node,
      SUM(hits), 
      num_seconds, dist_value_type
    FROM MGMT_RT_IP_DIST_1HOUR
    WHERE target_guid = c_target_guid
      AND TRUNC(rollup_timestamp, 'DD') > c_prev_rollup
      AND TRUNC(rollup_timestamp, 'DD') <= c_cur_rollup
    GROUP BY 
      TRUNC(rollup_timestamp, 'DD'), metric_name, visitor_node, num_seconds,dist_value_type;
BEGIN

  OPEN rollup_cursor( p_cur_target_guid, 
                      p_cur_rollup_timestamps(p_ip_dist_1day_idx),
                      p_cur_day_rollup ); 
  LOOP

    -- Process batchsize number of rows per commit.
    FETCH rollup_cursor BULK COLLECT INTO v_rollup_rec.rollup_time,
                                          v_rollup_rec.metric_name,
                                          v_rollup_rec.key_val,
                                          v_rollup_rec.hits,
                                          v_rollup_rec.dist_period,
                                          v_rollup_rec.dist_type
                                          LIMIT p_rollup_batchsize;

    -- Exit loop if no rollup records are generated.
    EXIT WHEN NOT v_rollup_rec.rollup_time.EXISTS(1);

    -- Bulk inserts
    FORALL i IN v_rollup_rec.rollup_time.FIRST..v_rollup_rec.rollup_time.LAST
          INSERT INTO MGMT_RT_IP_DIST_1DAY
          ( 
            target_guid, metric_name, 
            rollup_timestamp,
            visitor_node,
            hits, num_seconds,dist_value_type
          )
          VALUES
          ( 
            p_cur_target_guid, v_rollup_rec.metric_name(i),
            v_rollup_rec.rollup_time(i),
            v_rollup_rec.key_val(i), 
            v_rollup_rec.hits(i),
            v_rollup_rec.dist_period(i),
            v_rollup_rec.dist_type(i)
          ); 

    COMMIT;

    -- Keep track of the number of inserts
    v_rc := v_rc + v_rollup_rec.key_val.COUNT;

    -- Free temporary tables
    v_rollup_rec.rollup_time.DELETE;
    v_rollup_rec.metric_name.DELETE;
    v_rollup_rec.key_val.DELETE;
    v_rollup_rec.hits.DELETE;
    v_rollup_rec.dist_period.DELETE;
    v_rollup_rec.dist_type.DELETE;

    -- Check cursor status here because cursor may return not found on a
    -- bulk fetch, even though records were retrieved.
    EXIT WHEN rollup_cursor%NOTFOUND;

  END LOOP; 

  -- Update the rollup timestamp
  UPDATE MGMT_TARGET_ROLLUP_TIMES
    SET rollup_timestamp = p_cur_day_rollup
    WHERE rowid = p_cur_rollup_rowids(p_ip_dist_1day_idx);

  COMMIT;

  IF rollup_cursor%ISOPEN THEN
    CLOSE rollup_cursor;
  END IF;

  -- Update the global row counters
  p_total_rc := p_total_rc + v_rc;
  p_ip_dist_1day_rollup_rc := p_ip_dist_1day_rollup_rc + v_rc;

EXCEPTION

  WHEN DUP_VAL_ON_INDEX THEN
    -- Partial rollup detected.

    -- Rollback any partial work
    ROLLBACK;

    -- Free all resources used
    IF rollup_cursor%ISOPEN THEN
      CLOSE rollup_cursor;
    END IF;
    v_rollup_rec.rollup_time.DELETE;
    v_rollup_rec.metric_name.DELETE;
    v_rollup_rec.key_val.DELETE;
    v_rollup_rec.hits.DELETE;
    v_rollup_rec.dist_period.DELETE;
    v_rollup_rec.dist_type.DELETE;

    -- Cleanup all partial work.  If the cleanup fails, update the rollup
    -- timestamp so that the rollup doesn't get stuck forever.
    IF (EMD_IP_DIST_1DAY_CLEANUP = TRUE) THEN
      MGMT_LOG.LOG_ERROR('RTROLLUP', null,
        'Successfully cleaned up partial ip dist 1day rollup (target=' ||
        p_cur_target_guid || ')', 'MGMT', NULL, NULL, NULL, MGMT_GLOBAL.G_INFO);
    ELSE
      MGMT_LOG.LOG_ERROR('RTROLLUP', null,
        'Unable to cleanup the partial ip dist 1day rollup (target=' ||
        p_cur_target_guid || ')');
      BEGIN
        UPDATE MGMT_TARGET_ROLLUP_TIMES
          SET rollup_timestamp = p_cur_day_rollup
          WHERE rowid = p_cur_rollup_rowids(p_ip_dist_1day_idx);
        COMMIT;
        MGMT_LOG.LOG_ERROR('RTROLLUP', null,
          'IP dist 1day rollups for the period [' ||
          TO_CHAR(p_cur_rollup_timestamps(p_ip_dist_1day_idx),
          'MM:DD:YYYY') || ', ' ||
          TO_CHAR(p_cur_day_rollup, 'MM:DD:YYYY') || '] 
          cannot be performed (target=' || p_cur_target_guid || ')');
      EXCEPTION
        WHEN OTHERS THEN
          MGMT_LOG.LOG_ERROR('RTROLLUP', null,
            'Fatal Error: Unable to set the ip dist 1day timestamp (target=' ||
            p_cur_target_guid || '). Error: ' || SUBSTR(SQLERRM, 1, 1000));
          RAISE;
      END;
    END IF;

  WHEN OTHERS THEN
    -- Assumes permanent error and attempts to update the rollup timestamp,
    -- thus losing the rollup period in question.
    -- This should be improved upon, we need to determine if the error is
    -- permanent or transient and only update the timestamp if it's permanent.
   
    -- Rollback any partial work
    ROLLBACK;

    -- Free all resources used
    IF rollup_cursor%ISOPEN THEN
      CLOSE rollup_cursor;
    END IF;
    v_rollup_rec.rollup_time.DELETE;
    v_rollup_rec.metric_name.DELETE;
    v_rollup_rec.key_val.DELETE;
    v_rollup_rec.hits.DELETE;
    v_rollup_rec.dist_period.DELETE;
    v_rollup_rec.dist_type.DELETE;

    MGMT_LOG.LOG_ERROR('RTROLLUP', null,
      'The following unrecoverable error occurred while performing the' ||
      ' ip dist 1day rollup (target=' || p_cur_target_guid || ').  Attempting to' ||
      ' skip over rollup period [' || 
      TO_CHAR(p_cur_rollup_timestamps(p_ip_dist_1day_idx),
      'MM:DD:YYYY') || ', ' ||
      TO_CHAR(p_cur_day_rollup,
      'MM:DD:YYYY') || '].  Error: ' || SUBSTR(SQLERRM, 1, 1000));

    BEGIN
      UPDATE MGMT_TARGET_ROLLUP_TIMES
        SET rollup_timestamp = p_cur_day_rollup
        WHERE rowid = p_cur_rollup_rowids(p_ip_dist_1day_idx);
      COMMIT;
      MGMT_LOG.LOG_ERROR('RTROLLUP', null,
        'Successfully skipped over period.  IP dist 1day Rollup data will not be' ||
        ' generated for this period (target=' || p_cur_target_guid || ').');
    EXCEPTION
      WHEN OTHERS THEN
        MGMT_LOG.LOG_ERROR('RTROLLUP', null,
          'Fatal Error: Unable to set the ip dist 1day timestamp (target=' ||
          p_cur_target_guid || '). Error: ' || SUBSTR(SQLERRM, 1, 1000));
        RAISE;
    END;

END EMD_IP_DIST_1DAY_ROLLUP;
----------------------------------------------------------------------

----------------------------------------------------------------------
PROCEDURE EMD_PR_MAPPING_1DAY_ROLLUP IS
 
  v_rollup_rec      p_pr_map_rollup_record_type;
  v_rc              PLS_INTEGER := 0; 

  CURSOR rollup_cursor ( c_target_guid RAW,
                         c_prev_rollup DATE,
                         c_cur_rollup DATE) IS
    SELECT TRUNC(rollup_timestamp, 'DD'), page_url, 
           request_url,
      SUM(num_cache_hits), 
      SUM(num_cache_hits * cache_hits_avg_svr_time),
      SUM(num_non_cache_hits), 
      SUM(num_non_cache_hits * non_cache_hits_avg_svr_time)
    FROM MGMT_RT_PR_MAPPING_1HOUR
    WHERE target_guid = c_target_guid
      AND TRUNC(rollup_timestamp, 'DD') > c_prev_rollup 
      AND TRUNC(rollup_timestamp, 'DD') <= c_cur_rollup
    GROUP BY TRUNC(rollup_timestamp, 'DD'), page_url, request_url;

BEGIN

  OPEN rollup_cursor( p_cur_target_guid, 
                      p_cur_rollup_timestamps(p_pr_mapping_1day_idx),
                      p_cur_day_rollup );
  LOOP

    FETCH rollup_cursor BULK COLLECT INTO v_rollup_rec.rollup_time,
                                          v_rollup_rec.page_url,
                                          v_rollup_rec.request_url,
                                          v_rollup_rec.cache_hits,
                                          v_rollup_rec.ch_svr_sum,
                                          v_rollup_rec.non_cache_hits,
                                          v_rollup_rec.nch_svr_sum
                                          LIMIT p_rollup_batchsize;

    -- Exit if no rollup records are generated.
    EXIT WHEN NOT v_rollup_rec.rollup_time.EXISTS(1);

    -- Bulk inserts  
    FORALL i IN v_rollup_rec.rollup_time.FIRST..v_rollup_rec.rollup_time.LAST
      INSERT INTO MGMT_RT_PR_MAPPING_1DAY
      ( 
        target_guid, 
        page_url,
        request_url,
        num_cache_hits,
        cache_hits_avg_svr_time,
        num_non_cache_hits,
        non_cache_hits_avg_svr_time,
        rollup_timestamp
      )
      VALUES
      ( 
        p_cur_target_guid,
        v_rollup_rec.page_url(i),
        v_rollup_rec.request_url(i),
        v_rollup_rec.cache_hits(i),
        DECODE(v_rollup_rec.cache_hits(i),
               0, 0,
               v_rollup_rec.ch_svr_sum(i)/v_rollup_rec.cache_hits(i)),
        v_rollup_rec.non_cache_hits(i),
        DECODE(v_rollup_rec.non_cache_hits(i),
               0, 0,
               v_rollup_rec.nch_svr_sum(i)/v_rollup_rec.non_cache_hits(i)),
        v_rollup_rec.rollup_time(i)
      );

    COMMIT;

    -- Keep track of the number of inserts
    v_rc := v_rc + v_rollup_rec.page_url.COUNT;

    -- Free temporary tables
    v_rollup_rec.page_url.DELETE;
    v_rollup_rec.request_url.DELETE;
    v_rollup_rec.cache_hits.DELETE;
    v_rollup_rec.ch_svr_sum.DELETE;
    v_rollup_rec.non_cache_hits.DELETE;
    v_rollup_rec.nch_svr_sum.DELETE;
    v_rollup_rec.rollup_time.DELETE;

    -- Check cursor status here because cursor may return not found on a
    -- bulk fetch, even though records were retrieved.
    EXIT WHEN rollup_cursor%NOTFOUND;

  END LOOP; 

  -- Update the rollup timestamp
  UPDATE MGMT_TARGET_ROLLUP_TIMES
    SET rollup_timestamp = p_cur_day_rollup
    WHERE rowid = p_cur_rollup_rowids(p_pr_mapping_1day_idx);

  COMMIT;

  IF rollup_cursor%ISOPEN THEN
    CLOSE rollup_cursor;
  END IF;

  -- Update the global row counters
  p_total_rc := p_total_rc + v_rc;
  p_pr_mapping_1day_rollup_rc := p_pr_mapping_1day_rollup_rc + v_rc;

EXCEPTION

  WHEN DUP_VAL_ON_INDEX THEN
    -- Partial rollup detected.

    -- Rollback any partial work performed in this procedure
    ROLLBACK;

    -- Free all resources used
    IF rollup_cursor%ISOPEN THEN
      CLOSE rollup_cursor;
    END IF;
    v_rollup_rec.page_url.DELETE;
    v_rollup_rec.request_url.DELETE;
    v_rollup_rec.cache_hits.DELETE;
    v_rollup_rec.ch_svr_sum.DELETE;
    v_rollup_rec.non_cache_hits.DELETE;
    v_rollup_rec.nch_svr_sum.DELETE;
    v_rollup_rec.rollup_time.DELETE;

    -- Cleanup all partial work.  If the cleanup fails, update the rollup
    -- timestamp so that the rollup doesn't get stuck forever.
    IF (EMD_PR_MAPPING_1DAY_CLEANUP = TRUE) THEN
      MGMT_LOG.LOG_ERROR('RTROLLUP', null,
        'Successfully cleaned up partial PR mapping 1day rollup (target=' ||
        p_cur_target_guid || ')', 'MGMT', NULL, NULL, NULL, MGMT_GLOBAL.G_INFO);
    ELSE
      MGMT_LOG.LOG_ERROR('RTROLLUP', null,
        'Unable to cleanup the partial PR mapping 1day rollup (target=' ||
        p_cur_target_guid || ')');
      BEGIN
        UPDATE MGMT_TARGET_ROLLUP_TIMES
          SET rollup_timestamp = p_cur_day_rollup
          WHERE rowid = p_cur_rollup_rowids(p_pr_mapping_1day_idx);
        COMMIT;
        MGMT_LOG.LOG_ERROR('RTROLLUP', null,
          'PR Mapping 1day rollups for the period [' ||
          TO_CHAR(p_cur_rollup_timestamps(p_pr_mapping_1day_idx),
          'MM:DD:YYYY HH24:MI') || ', ' ||
          TO_CHAR(p_cur_day_rollup, 'MM:DD:YYYY HH24:MI') || ']
          cannot be performed (target=' || p_cur_target_guid || ')');
      EXCEPTION
        WHEN OTHERS THEN
          MGMT_LOG.LOG_ERROR('RTROLLUP', null,
            'Fatal Error: Unable to set the PR Mapping 1day timestamp (target=' ||
            p_cur_target_guid || '). Error: ' || SUBSTR(SQLERRM, 1, 1000));
          RAISE;
      END;
    END IF;

  WHEN OTHERS THEN
    -- Assumes permanent error and attempts to update the rollup timestamp,
    -- thus losing the rollup period in question.
    -- This should be improved upon, we need to determine if the error is
    -- permanent or transient and only update the timestamp if it's permanent.
   
    -- Rollback any partial work
    ROLLBACK;

    -- Free all resources used
    IF rollup_cursor%ISOPEN THEN
      CLOSE rollup_cursor;
    END IF;
    v_rollup_rec.page_url.DELETE;
    v_rollup_rec.request_url.DELETE;
    v_rollup_rec.cache_hits.DELETE;
    v_rollup_rec.ch_svr_sum.DELETE;
    v_rollup_rec.non_cache_hits.DELETE;
    v_rollup_rec.nch_svr_sum.DELETE;
    v_rollup_rec.rollup_time.DELETE;

    MGMT_LOG.LOG_ERROR('RTROLLUP', null,
      'The following unrecoverable error occurred while performing the' ||
      ' PR Mapping 1day rollup (target=' || p_cur_target_guid || ').  Attempting to' ||
      ' skip over rollup period [' ||
      TO_CHAR(p_cur_rollup_timestamps(p_pr_mapping_1day_idx),
      'MM:DD:YYYY HH24:MI') || ', ' ||
      TO_CHAR(p_cur_day_rollup,
      'MM:DD:YYYY HH24:MI') || '].  Error: ' || SUBSTR(SQLERRM, 1, 1000));

    BEGIN
      UPDATE MGMT_TARGET_ROLLUP_TIMES
        SET rollup_timestamp = p_cur_day_rollup
        WHERE rowid = p_cur_rollup_rowids(p_pr_mapping_1day_idx);
      COMMIT;
      MGMT_LOG.LOG_ERROR('RTROLLUP', null,
        'Successfully skipped over period. PR Mapping 1day rollup data will not be' ||
        ' generated for this period (target=' || p_cur_target_guid || ').');
    EXCEPTION
      WHEN OTHERS THEN
        MGMT_LOG.LOG_ERROR('RTROLLUP', null,
          'Fatal Error: Unable to set the PR Mapping 1day timestamp (target=' ||
          p_cur_target_guid || '). Error: ' || SUBSTR(SQLERRM, 1, 1000));
        RAISE;
    END;

END EMD_PR_MAPPING_1DAY_ROLLUP;
----------------------------------------------------------------------

----------------------------------------------------------------------
PROCEDURE EMD_INC_LOADS_1DAY_ROLLUP IS
 
  v_rollup_rec      p_inc_loads_rollup_record_type;
  v_rc              PLS_INTEGER := 0; 

  CURSOR rollup_cursor ( c_target_guid RAW,
                         c_prev_rollup DATE,
                         c_cur_rollup DATE) IS
    SELECT TRUNC(rollup_timestamp, 'DD'), page_url,
      SUM(num_incomplete_loads), 
      ROUND(SUM(num_incomplete_loads * avg_server_time) / 
            SUM(num_incomplete_loads), 2)
    FROM MGMT_RT_INCOMPLETE_LOADS_1HOUR
    WHERE target_guid = c_target_guid
      AND TRUNC(rollup_timestamp, 'DD') > c_prev_rollup 
      AND TRUNC(rollup_timestamp, 'DD') <= c_cur_rollup
    GROUP BY TRUNC(rollup_timestamp, 'DD'), page_url;

BEGIN

  OPEN rollup_cursor( p_cur_target_guid, 
                      p_cur_rollup_timestamps(p_inc_loads_1day_idx),
                      p_cur_day_rollup );
  LOOP

    FETCH rollup_cursor BULK COLLECT INTO v_rollup_rec.rollup_time,
                                          v_rollup_rec.page_url,
                                          v_rollup_rec.inc_loads,
                                          v_rollup_rec.svr_avg
                                          LIMIT p_rollup_batchsize;

    -- Exit if no rollup records are generated.
    EXIT WHEN NOT v_rollup_rec.rollup_time.EXISTS(1);

    -- Bulk inserts  
    FORALL i IN v_rollup_rec.rollup_time.FIRST..v_rollup_rec.rollup_time.LAST
      INSERT INTO MGMT_RT_INCOMPLETE_LOADS_1DAY
      ( 
        target_guid, 
        page_url,
        num_incomplete_loads,
        avg_server_time,
        rollup_timestamp
      )
      VALUES
      ( 
        p_cur_target_guid,
        v_rollup_rec.page_url(i),
        v_rollup_rec.inc_loads(i),
        v_rollup_rec.svr_avg(i),
        v_rollup_rec.rollup_time(i)
      );

    COMMIT;

    -- Keep track of the number of inserts
    v_rc := v_rc + v_rollup_rec.page_url.COUNT;

    -- Free temporary tables
    v_rollup_rec.page_url.DELETE;
    v_rollup_rec.inc_loads.DELETE;
    v_rollup_rec.svr_avg.DELETE;
    v_rollup_rec.rollup_time.DELETE;

    -- Check cursor status here because cursor may return not found on a
    -- bulk fetch, even though records were retrieved.
    EXIT WHEN rollup_cursor%NOTFOUND;

  END LOOP; 

  -- Update the rollup timestamp
  UPDATE MGMT_TARGET_ROLLUP_TIMES
    SET rollup_timestamp = p_cur_day_rollup
    WHERE rowid = p_cur_rollup_rowids(p_inc_loads_1day_idx);

  COMMIT;

  IF rollup_cursor%ISOPEN THEN
    CLOSE rollup_cursor;
  END IF;

  -- Update the global row counters
  p_total_rc := p_total_rc + v_rc;
  p_inc_loads_1day_rollup_rc := p_inc_loads_1day_rollup_rc + v_rc;

EXCEPTION

  WHEN DUP_VAL_ON_INDEX THEN
    -- Partial rollup detected.

    -- Rollback any partial work performed in this procedure
    ROLLBACK;

    -- Free all resources used
    IF rollup_cursor%ISOPEN THEN
      CLOSE rollup_cursor;
    END IF;
    v_rollup_rec.page_url.DELETE;
    v_rollup_rec.inc_loads.DELETE;
    v_rollup_rec.svr_avg.DELETE;
    v_rollup_rec.rollup_time.DELETE;

    -- Cleanup all partial work.  If the cleanup fails, update the rollup
    -- timestamp so that the rollup doesn't get stuck forever.
    IF (EMD_INC_LOADS_1DAY_CLEANUP = TRUE) THEN
      MGMT_LOG.LOG_ERROR('RTROLLUP', null,
        'Successfully cleaned up partial incomplete loads 1day rollup (target=' ||
        p_cur_target_guid || ')', 'MGMT', NULL, NULL, NULL, MGMT_GLOBAL.G_INFO);
    ELSE
      MGMT_LOG.LOG_ERROR('RTROLLUP', null,
        'Unable to cleanup the partial incomplete loads 1day rollup (target=' ||
        p_cur_target_guid || ')');
      BEGIN
        UPDATE MGMT_TARGET_ROLLUP_TIMES
          SET rollup_timestamp = p_cur_day_rollup
          WHERE rowid = p_cur_rollup_rowids(p_inc_loads_1day_idx);
        COMMIT;
        MGMT_LOG.LOG_ERROR('RTROLLUP', null,
          'Incomplete Loads 1day rollups for the period [' ||
          TO_CHAR(p_cur_rollup_timestamps(p_inc_loads_1day_idx),
          'MM:DD:YYYY HH24:MI') || ', ' ||
          TO_CHAR(p_cur_day_rollup, 'MM:DD:YYYY HH24:MI') || ']
          cannot be performed (target=' || p_cur_target_guid || ')');
      EXCEPTION
        WHEN OTHERS THEN
          MGMT_LOG.LOG_ERROR('RTROLLUP', null,
            'Fatal Error: Unable to set the Incomplete Loads 1day timestamp (target=' ||
            p_cur_target_guid || '). Error: ' || SUBSTR(SQLERRM, 1, 1000));
          RAISE;
      END;
    END IF;

  WHEN OTHERS THEN
    -- Assumes permanent error and attempts to update the rollup timestamp,
    -- thus losing the rollup period in question.
    -- This should be improved upon, we need to determine if the error is
    -- permanent or transient and only update the timestamp if it's permanent.
   
    -- Rollback any partial work
    ROLLBACK;

    -- Free all resources used
    IF rollup_cursor%ISOPEN THEN
      CLOSE rollup_cursor;
    END IF;
    v_rollup_rec.page_url.DELETE;
    v_rollup_rec.inc_loads.DELETE;
    v_rollup_rec.svr_avg.DELETE;
    v_rollup_rec.rollup_time.DELETE;

    MGMT_LOG.LOG_ERROR('RTROLLUP', null,
      'The following unrecoverable error occurred while performing the' ||
      ' Incomplete Loads 1day rollup (target=' || p_cur_target_guid || ').  Attempting to' ||
      ' skip over rollup period [' ||
      TO_CHAR(p_cur_rollup_timestamps(p_inc_loads_1day_idx),
      'MM:DD:YYYY HH24:MI') || ', ' ||
      TO_CHAR(p_cur_day_rollup,
      'MM:DD:YYYY HH24:MI') || '].  Error: ' || SUBSTR(SQLERRM, 1, 1000));

    BEGIN
      UPDATE MGMT_TARGET_ROLLUP_TIMES
        SET rollup_timestamp = p_cur_day_rollup
        WHERE rowid = p_cur_rollup_rowids(p_inc_loads_1day_idx);
      COMMIT;
      MGMT_LOG.LOG_ERROR('RTROLLUP', null,
        'Successfully skipped over period. Incomplete Loads 1day rollup data will not be' ||
        ' generated for this period (target=' || p_cur_target_guid || ').');
    EXCEPTION
      WHEN OTHERS THEN
        MGMT_LOG.LOG_ERROR('RTROLLUP', null,
          'Fatal Error: Unable to set the Incomplete Loads 1day timestamp (target=' ||
          p_cur_target_guid || '). Error: ' || SUBSTR(SQLERRM, 1, 1000));
        RAISE;
    END;
END EMD_INC_LOADS_1DAY_ROLLUP;
----------------------------------------------------------------------

-----------------------    HOURLY PURGES    --------------------------

----------------------------------------------------------------------
PROCEDURE EMD_URL_1HOUR_PURGE IS 

  v_purge_rowid     p_rowid_list_type;
  v_rc              PLS_INTEGER := 0; 

  CURSOR purge_cursor ( c_target_guid RAW,
                        c_max_purge_time DATE ) IS
    SELECT rowid
      FROM MGMT_RT_URL_1HOUR
      WHERE target_guid = c_target_guid
        AND rollup_timestamp < c_max_purge_time;

BEGIN

  OPEN purge_cursor( p_cur_target_guid, p_cur_hour_purge );
  LOOP

    -- Process batchsize number of rows per commit.
    FETCH purge_cursor BULK COLLECT INTO v_purge_rowid 
      LIMIT p_purge_batchsize;

    -- Exit when nothing to purge
    EXIT WHEN v_purge_rowid.COUNT <= 0;  
 
    -- Bulk delete by rowid
    FORALL i IN v_purge_rowid.FIRST..v_purge_rowid.LAST
      DELETE FROM MGMT_RT_URL_1HOUR 
        WHERE rowid = v_purge_rowid(i);

    COMMIT;

    -- Keep track of the number of deleted rows
    v_rc := v_rc + v_purge_rowid.COUNT;

    -- Clear the rowid list
    v_purge_rowid.DELETE;

  END LOOP;

  IF purge_cursor%ISOPEN THEN
    CLOSE purge_cursor;
  END IF;

  -- Update the global row counters
  p_total_rc := p_total_rc + v_rc;
  p_url_1hour_purge_rc := p_url_1hour_purge_rc + v_rc;

EXCEPTION
  WHEN OTHERS THEN
    ROLLBACK;
    IF purge_cursor%ISOPEN THEN
      CLOSE purge_cursor;
    END IF;
    v_purge_rowid.DELETE;
    MGMT_LOG.LOG_ERROR('PURGE', null,
      'Error during url 1hour: ' || SUBSTR(SQLERRM, 1, 1100));

END EMD_URL_1HOUR_PURGE;
----------------------------------------------------------------------

----------------------------------------------------------------------
PROCEDURE EMD_URL_DIST_1HOUR_PURGE IS 

  v_purge_rowid     p_rowid_list_type;
  v_rc              PLS_INTEGER := 0; 

  CURSOR purge_cursor ( c_target_guid RAW,
                        c_max_purge_time DATE ) IS
    SELECT rowid
      FROM MGMT_RT_URL_DIST_1HOUR
      WHERE target_guid = c_target_guid
        AND rollup_timestamp < c_max_purge_time;

BEGIN

  OPEN purge_cursor( p_cur_target_guid, p_cur_dist_hour_purge );
  LOOP

    -- Process batchsize number of rows per commit.
    FETCH purge_cursor BULK COLLECT INTO v_purge_rowid 
      LIMIT p_purge_batchsize;

    -- Exit when nothing to purge
    EXIT WHEN v_purge_rowid.COUNT <= 0;  
 
    -- Bulk delete by rowid
    FORALL i IN v_purge_rowid.FIRST..v_purge_rowid.LAST
      DELETE FROM MGMT_RT_URL_DIST_1HOUR 
        WHERE rowid = v_purge_rowid(i);

    COMMIT;

    -- Keep track of the number of deleted rows
    v_rc := v_rc + v_purge_rowid.COUNT;

    -- Clear the rowid list
    v_purge_rowid.DELETE;  

  END LOOP;

  IF purge_cursor%ISOPEN THEN
    CLOSE purge_cursor;
  END IF;

  -- Update the global row counters
  p_total_rc := p_total_rc + v_rc;
  p_url_dist_1hour_purge_rc := p_url_dist_1hour_purge_rc + v_rc;

EXCEPTION
  WHEN OTHERS THEN
    ROLLBACK;
    IF purge_cursor%ISOPEN THEN
      CLOSE purge_cursor;
    END IF;
    v_purge_rowid.DELETE;
    MGMT_LOG.LOG_ERROR('PURGE', null,
      'Error during url dist 1hour: ' || SUBSTR(SQLERRM, 1, 1100));
END EMD_URL_DIST_1HOUR_PURGE;
----------------------------------------------------------------------

----------------------------------------------------------------------
PROCEDURE EMD_DOM_1HOUR_PURGE IS 

  v_purge_rowid     p_rowid_list_type;
  v_rc              PLS_INTEGER := 0; 

  CURSOR purge_cursor ( c_target_guid RAW,
                        c_max_purge_time DATE ) IS
    SELECT rowid
      FROM MGMT_RT_DOMAIN_1HOUR
      WHERE target_guid = c_target_guid
        AND rollup_timestamp < c_max_purge_time;

BEGIN

  OPEN purge_cursor( p_cur_target_guid, p_cur_hour_purge );
  LOOP

    -- Process batchsize number of rows per commit.
    FETCH purge_cursor BULK COLLECT INTO v_purge_rowid 
      LIMIT p_purge_batchsize;

    -- Exit when nothing to purge
    EXIT WHEN v_purge_rowid.COUNT <= 0;  
 
    -- Bulk delete by rowid
    FORALL i IN v_purge_rowid.FIRST..v_purge_rowid.LAST
      DELETE FROM MGMT_RT_DOMAIN_1HOUR 
        WHERE rowid = v_purge_rowid(i);

    COMMIT;

    -- Keep track of the number of deleted rows
    v_rc := v_rc + v_purge_rowid.COUNT;

    -- Clear the rowid list
    v_purge_rowid.DELETE;

  END LOOP;

  IF purge_cursor%ISOPEN THEN
    CLOSE purge_cursor;
  END IF;

  -- Update the global row counters
  p_total_rc := p_total_rc + v_rc;
  p_dom_1hour_purge_rc := p_dom_1hour_purge_rc + v_rc;

EXCEPTION
  WHEN OTHERS THEN
    ROLLBACK;
    IF purge_cursor%ISOPEN THEN
      CLOSE purge_cursor;
    END IF;
    v_purge_rowid.DELETE;
    MGMT_LOG.LOG_ERROR('PURGE', null,
      'Error during dom 1hour: ' || SUBSTR(SQLERRM, 1, 1100));
END EMD_DOM_1HOUR_PURGE;
----------------------------------------------------------------------

----------------------------------------------------------------------
PROCEDURE EMD_DOM_DIST_1HOUR_PURGE IS 

  v_purge_rowid     p_rowid_list_type;
  v_rc              PLS_INTEGER := 0; 

  CURSOR purge_cursor ( c_target_guid RAW,
                        c_max_purge_time DATE ) IS
    SELECT rowid
      FROM MGMT_RT_DOMAIN_DIST_1HOUR
      WHERE target_guid = c_target_guid
        AND rollup_timestamp < c_max_purge_time;

BEGIN

  OPEN purge_cursor( p_cur_target_guid, p_cur_dist_hour_purge );
  LOOP

    -- Process batchsize number of rows per commit.
    FETCH purge_cursor BULK COLLECT INTO v_purge_rowid 
      LIMIT p_purge_batchsize;

    -- Exit when nothing to purge
    EXIT WHEN v_purge_rowid.COUNT <= 0;  
 
    -- Bulk delete by rowid
    FORALL i IN v_purge_rowid.FIRST..v_purge_rowid.LAST
      DELETE FROM MGMT_RT_DOMAIN_DIST_1HOUR 
        WHERE rowid = v_purge_rowid(i);

    COMMIT;

    -- Keep track of the number of deleted rows
    v_rc := v_rc + v_purge_rowid.COUNT;

    -- Clear the rowid list
    v_purge_rowid.DELETE;  

  END LOOP;

  IF purge_cursor%ISOPEN THEN
    CLOSE purge_cursor;
  END IF;

  -- Update the global row counters
  p_total_rc := p_total_rc + v_rc;
  p_dom_dist_1hour_purge_rc := p_dom_dist_1hour_purge_rc + v_rc;

EXCEPTION
  WHEN OTHERS THEN
    ROLLBACK;
    IF purge_cursor%ISOPEN THEN
      CLOSE purge_cursor;
    END IF;
    v_purge_rowid.DELETE;
    MGMT_LOG.LOG_ERROR('PURGE', null,
      'Error during dom dist 1hour: ' || SUBSTR(SQLERRM, 1, 1100));
END EMD_DOM_DIST_1HOUR_PURGE;
----------------------------------------------------------------------

----------------------------------------------------------------------
PROCEDURE EMD_IP_1HOUR_PURGE IS 

  v_purge_rowid     p_rowid_list_type;
  v_rc              PLS_INTEGER := 0; 

  CURSOR purge_cursor ( c_target_guid RAW,
                        c_max_purge_time DATE ) IS
    SELECT rowid
      FROM MGMT_RT_IP_1HOUR
      WHERE target_guid = c_target_guid
        AND rollup_timestamp < c_max_purge_time;

BEGIN

  OPEN purge_cursor( p_cur_target_guid, p_cur_hour_purge );
  LOOP

    -- Process batchsize number of rows per commit.
    FETCH purge_cursor BULK COLLECT INTO v_purge_rowid 
      LIMIT p_purge_batchsize;

    -- Exit when nothing to purge
    EXIT WHEN v_purge_rowid.COUNT <= 0;  
 
    -- Bulk delete by rowid
    FORALL i IN v_purge_rowid.FIRST..v_purge_rowid.LAST
      DELETE FROM MGMT_RT_IP_1HOUR 
        WHERE rowid = v_purge_rowid(i);

    COMMIT;

    -- Keep track of the number of deleted rows
    v_rc := v_rc + v_purge_rowid.COUNT;

    -- Clear the rowid list
    v_purge_rowid.DELETE;

  END LOOP;

  IF purge_cursor%ISOPEN THEN
    CLOSE purge_cursor;
  END IF;

  -- Update the global row counters
  p_total_rc := p_total_rc + v_rc;
  p_ip_1hour_purge_rc := p_ip_1hour_purge_rc + v_rc;

EXCEPTION
  WHEN OTHERS THEN
    ROLLBACK;
    IF purge_cursor%ISOPEN THEN
      CLOSE purge_cursor;
    END IF;
    v_purge_rowid.DELETE;
    MGMT_LOG.LOG_ERROR('PURGE', null,
      'Error during ip 1hour: ' || SUBSTR(SQLERRM, 1, 1100));
END EMD_IP_1HOUR_PURGE;
----------------------------------------------------------------------

----------------------------------------------------------------------
PROCEDURE EMD_IP_DIST_1HOUR_PURGE IS 

  v_purge_rowid     p_rowid_list_type;
  v_rc              PLS_INTEGER := 0; 

  CURSOR purge_cursor ( c_target_guid RAW,
                        c_max_purge_time DATE ) IS
    SELECT rowid
      FROM MGMT_RT_IP_DIST_1HOUR
      WHERE target_guid = c_target_guid
        AND rollup_timestamp < c_max_purge_time;

BEGIN

  OPEN purge_cursor( p_cur_target_guid, p_cur_dist_hour_purge );
  LOOP

    -- Process batchsize number of rows per commit.
    FETCH purge_cursor BULK COLLECT INTO v_purge_rowid 
      LIMIT p_purge_batchsize;

    -- Exit when nothing to purge
    EXIT WHEN v_purge_rowid.COUNT <= 0;  
 
    -- Bulk delete by rowid
    FORALL i IN v_purge_rowid.FIRST..v_purge_rowid.LAST
      DELETE FROM MGMT_RT_IP_DIST_1HOUR 
        WHERE rowid = v_purge_rowid(i);

    COMMIT;

    -- Keep track of the number of deleted rows
    v_rc := v_rc + v_purge_rowid.COUNT;

    -- Clear the rowid list
    v_purge_rowid.DELETE;  

  END LOOP;

  IF purge_cursor%ISOPEN THEN
    CLOSE purge_cursor;
  END IF;

  -- Update the global row counters
  p_total_rc := p_total_rc + v_rc;
  p_ip_dist_1hour_purge_rc := p_ip_dist_1hour_purge_rc + v_rc;

EXCEPTION
  WHEN OTHERS THEN
    ROLLBACK;
    IF purge_cursor%ISOPEN THEN
      CLOSE purge_cursor;
    END IF;
    v_purge_rowid.DELETE;
    MGMT_LOG.LOG_ERROR('PURGE', null,
      'Error during ip dist 1hour: ' || SUBSTR(SQLERRM, 1, 1100));
END EMD_IP_DIST_1HOUR_PURGE;
----------------------------------------------------------------------

----------------------------------------------------------------------
PROCEDURE EMD_PR_MAPPING_1HOUR_PURGE IS 

  v_purge_rowid     p_rowid_list_type;
  v_rc              PLS_INTEGER := 0; 

  CURSOR purge_cursor ( c_target_guid RAW,
                        c_max_purge_time DATE ) IS
    SELECT rowid
      FROM MGMT_RT_PR_MAPPING_1HOUR
      WHERE target_guid = c_target_guid
        AND rollup_timestamp < c_max_purge_time;

  CURSOR raw_purge_cursor ( c_target_guid RAW,
                            c_max_purge_time DATE ) IS
    SELECT rowid
      FROM MGMT_RT_PR_MAPPING
      WHERE target_guid = c_target_guid
        AND aggregate_hour_timestamp < c_max_purge_time;

BEGIN

  OPEN purge_cursor( p_cur_target_guid, p_cur_hour_purge );
  LOOP

    -- Process batchsize number of rows per commit.
    FETCH purge_cursor BULK COLLECT INTO v_purge_rowid 
      LIMIT p_purge_batchsize;

    -- Exit when nothing to purge
    EXIT WHEN v_purge_rowid.COUNT <= 0;  
 
    -- Bulk delete by rowid
    FORALL i IN v_purge_rowid.FIRST..v_purge_rowid.LAST
      DELETE FROM MGMT_RT_PR_MAPPING_1HOUR 
        WHERE rowid = v_purge_rowid(i);

    COMMIT;

    -- Keep track of the number of deleted rows
    v_rc := v_rc + v_purge_rowid.COUNT;

    -- Clear the rowid list
    v_purge_rowid.DELETE;

  END LOOP;

  OPEN raw_purge_cursor( p_cur_target_guid, p_cur_hour_purge );
  LOOP

    -- Process batchsize number of rows per commit.
    FETCH raw_purge_cursor BULK COLLECT INTO v_purge_rowid 
      LIMIT p_purge_batchsize;

    -- Exit when nothing to purge
    EXIT WHEN v_purge_rowid.COUNT <= 0;  
 
    -- Bulk delete by rowid
    FORALL i IN v_purge_rowid.FIRST..v_purge_rowid.LAST
      DELETE FROM MGMT_RT_PR_MAPPING
        WHERE rowid = v_purge_rowid(i);

    COMMIT;

    -- Clear the rowid list
    v_purge_rowid.DELETE;

  END LOOP;

  IF purge_cursor%ISOPEN THEN
    CLOSE purge_cursor;
  END IF;

  -- Update the global row counters
  p_total_rc := p_total_rc + v_rc;
  p_pr_mapping_1hour_purge_rc := p_pr_mapping_1hour_purge_rc + v_rc;

EXCEPTION
  WHEN OTHERS THEN
    ROLLBACK;
    IF purge_cursor%ISOPEN THEN
      CLOSE purge_cursor;
    END IF;
    v_purge_rowid.DELETE;
    MGMT_LOG.LOG_ERROR('PURGE', null,
      'Error during pr mapping 1hour: ' || SUBSTR(SQLERRM, 1, 1100));
END EMD_PR_MAPPING_1HOUR_PURGE;
----------------------------------------------------------------------

----------------------------------------------------------------------
PROCEDURE EMD_INC_LOADS_1HOUR_PURGE IS 

  v_purge_rowid     p_rowid_list_type;
  v_rc              PLS_INTEGER := 0; 

  CURSOR purge_cursor ( c_target_guid RAW,
                        c_max_purge_time DATE ) IS
    SELECT rowid
      FROM MGMT_RT_INCOMPLETE_LOADS_1HOUR
      WHERE target_guid = c_target_guid
        AND rollup_timestamp < c_max_purge_time;

  CURSOR raw_purge_cursor ( c_target_guid RAW,
                            c_max_purge_time DATE ) IS
    SELECT rowid
      FROM MGMT_RT_INCOMPLETE_LOADS
      WHERE target_guid = c_target_guid
        AND aggregate_hour_timestamp < c_max_purge_time;

BEGIN

  OPEN purge_cursor( p_cur_target_guid, p_cur_hour_purge );
  LOOP

    -- Process batchsize number of rows per commit.
    FETCH purge_cursor BULK COLLECT INTO v_purge_rowid 
      LIMIT p_purge_batchsize;

    -- Exit when nothing to purge
    EXIT WHEN v_purge_rowid.COUNT <= 0;  
 
    -- Bulk delete by rowid
    FORALL i IN v_purge_rowid.FIRST..v_purge_rowid.LAST
      DELETE FROM MGMT_RT_INCOMPLETE_LOADS_1HOUR 
        WHERE rowid = v_purge_rowid(i);

    COMMIT;

    -- Keep track of the number of deleted rows
    v_rc := v_rc + v_purge_rowid.COUNT;

    -- Clear the rowid list
    v_purge_rowid.DELETE;

  END LOOP;

  OPEN raw_purge_cursor( p_cur_target_guid, p_cur_hour_purge );
  LOOP

    -- Process batchsize number of rows per commit.
    FETCH raw_purge_cursor BULK COLLECT INTO v_purge_rowid 
      LIMIT p_purge_batchsize;

    -- Exit when nothing to purge
    EXIT WHEN v_purge_rowid.COUNT <= 0;  
 
    -- Bulk delete by rowid
    FORALL i IN v_purge_rowid.FIRST..v_purge_rowid.LAST
      DELETE FROM MGMT_RT_INCOMPLETE_LOADS
        WHERE rowid = v_purge_rowid(i);

    COMMIT;

    -- Clear the rowid list
    v_purge_rowid.DELETE;

  END LOOP;

  IF purge_cursor%ISOPEN THEN
    CLOSE purge_cursor;
  END IF;

  -- Update the global row counters
  p_total_rc := p_total_rc + v_rc;
  p_inc_loads_1hour_purge_rc := p_inc_loads_1hour_purge_rc + v_rc;

EXCEPTION
  WHEN OTHERS THEN
    ROLLBACK;
    IF purge_cursor%ISOPEN THEN
      CLOSE purge_cursor;
    END IF;
    v_purge_rowid.DELETE;
    MGMT_LOG.LOG_ERROR('PURGE', null,
      'Error during incomplete loads 1hour: ' || SUBSTR(SQLERRM, 1, 1100));
END EMD_INC_LOADS_1HOUR_PURGE;
----------------------------------------------------------------------

------------------------    DAILY PURGES    --------------------------

----------------------------------------------------------------------
PROCEDURE EMD_URL_1DAY_PURGE IS

  v_purge_rowid     p_rowid_list_type;
  v_rc              PLS_INTEGER := 0; 

  CURSOR purge_cursor ( c_target_guid RAW,
                        c_max_purge_time DATE ) IS
    SELECT rowid
      FROM MGMT_RT_URL_1DAY
      WHERE target_guid = c_target_guid
        AND rollup_timestamp < c_max_purge_time;

BEGIN

  OPEN purge_cursor( p_cur_target_guid, p_cur_day_purge );
  LOOP

    -- Process batchsize number of rows per commit.
    FETCH purge_cursor BULK COLLECT INTO v_purge_rowid 
      LIMIT p_purge_batchsize;

    -- Exit when nothing to purge
    EXIT WHEN v_purge_rowid.COUNT <= 0;
 
    -- Bulk delete by rowid
    FORALL i IN v_purge_rowid.FIRST..v_purge_rowid.LAST
      DELETE FROM MGMT_RT_URL_1DAY 
        WHERE rowid = v_purge_rowid(i);

    COMMIT;

    -- Keep track of the number of deleted rows
    v_rc := v_rc + v_purge_rowid.COUNT;

    -- Clear the rowid list
    v_purge_rowid.DELETE;  

  END LOOP;

  IF purge_cursor%ISOPEN THEN
    CLOSE purge_cursor;
  END IF;

  -- Update the global row counters
  p_total_rc := p_total_rc + v_rc;
  p_url_1day_purge_rc := p_url_1day_purge_rc + v_rc;

EXCEPTION
  WHEN OTHERS THEN
    ROLLBACK;
    IF purge_cursor%ISOPEN THEN
      CLOSE purge_cursor;
    END IF;
    v_purge_rowid.DELETE;
    MGMT_LOG.LOG_ERROR('PURGE', null,
      'Error during url 1day: ' || SUBSTR(SQLERRM, 1, 1100));
END EMD_URL_1DAY_PURGE;
----------------------------------------------------------------------

----------------------------------------------------------------------
PROCEDURE EMD_URL_DIST_1DAY_PURGE IS 

  v_purge_rowid     p_rowid_list_type;
  v_rc              PLS_INTEGER := 0; 

  CURSOR purge_cursor ( c_target_guid RAW,
                        c_max_purge_time DATE ) IS
    SELECT rowid
      FROM MGMT_RT_URL_DIST_1DAY
      WHERE target_guid = c_target_guid
        AND rollup_timestamp < c_max_purge_time;

BEGIN

  OPEN purge_cursor( p_cur_target_guid, p_cur_dist_day_purge );
  LOOP

    -- Process batchsize number of rows per commit.
    FETCH purge_cursor BULK COLLECT INTO v_purge_rowid 
      LIMIT p_purge_batchsize;

    -- Exit when nothing to purge
    EXIT WHEN v_purge_rowid.COUNT <= 0;
 
    -- Bulk delete by rowid
    FORALL i IN v_purge_rowid.FIRST..v_purge_rowid.LAST
      DELETE FROM MGMT_RT_URL_DIST_1DAY 
        WHERE rowid = v_purge_rowid(i);

    COMMIT;

    -- Keep track of the number of deleted rows
    v_rc := v_rc + v_purge_rowid.COUNT;

    -- Clear the rowid list
    v_purge_rowid.DELETE;

  END LOOP;

  IF purge_cursor%ISOPEN THEN
    CLOSE purge_cursor;
  END IF;

  -- Update the global row counters
  p_total_rc := p_total_rc + v_rc;
  p_url_dist_1day_purge_rc := p_url_dist_1day_purge_rc + v_rc;

EXCEPTION
  WHEN OTHERS THEN
    ROLLBACK;
    IF purge_cursor%ISOPEN THEN
      CLOSE purge_cursor;
    END IF;
    v_purge_rowid.DELETE;
    MGMT_LOG.LOG_ERROR('PURGE', null,
      'Error during url dist 1day: ' || SUBSTR(SQLERRM, 1, 1100));
END EMD_URL_DIST_1DAY_PURGE;
----------------------------------------------------------------------

----------------------------------------------------------------------
PROCEDURE EMD_DOM_1DAY_PURGE IS 

  v_purge_rowid     p_rowid_list_type;
  v_rc              PLS_INTEGER := 0; 

  CURSOR purge_cursor ( c_target_guid RAW,
                        c_max_purge_time DATE ) IS
    SELECT rowid
      FROM MGMT_RT_DOMAIN_1DAY
      WHERE target_guid = c_target_guid
        AND rollup_timestamp < c_max_purge_time;

BEGIN

  OPEN purge_cursor( p_cur_target_guid, p_cur_day_purge );
  LOOP

    -- Process batchsize number of rows per commit.
    FETCH purge_cursor BULK COLLECT INTO v_purge_rowid 
      LIMIT p_purge_batchsize;

    -- Exit when nothing to purge
    EXIT WHEN v_purge_rowid.COUNT <= 0;
 
    -- Bulk delete by rowid
    FORALL i IN v_purge_rowid.FIRST..v_purge_rowid.LAST
      DELETE FROM MGMT_RT_DOMAIN_1DAY 
        WHERE rowid = v_purge_rowid(i);

    COMMIT;

    -- Keep track of the number of deleted rows
    v_rc := v_rc + v_purge_rowid.COUNT;

    -- Clear the rowid list
    v_purge_rowid.DELETE;  

  END LOOP;

  IF purge_cursor%ISOPEN THEN
    CLOSE purge_cursor;
  END IF;

  -- Update the global row counters
  p_total_rc := p_total_rc + v_rc;
  p_dom_1day_purge_rc := p_dom_1day_purge_rc + v_rc;

EXCEPTION
  WHEN OTHERS THEN
    ROLLBACK;
    IF purge_cursor%ISOPEN THEN
      CLOSE purge_cursor;
    END IF;
    v_purge_rowid.DELETE;
    MGMT_LOG.LOG_ERROR('PURGE', null,
      'Error during dom 1day: ' || SUBSTR(SQLERRM, 1, 1100));
END EMD_DOM_1DAY_PURGE;
----------------------------------------------------------------------

----------------------------------------------------------------------
PROCEDURE EMD_DOM_DIST_1DAY_PURGE IS

  v_purge_rowid     p_rowid_list_type;
  v_rc              PLS_INTEGER := 0; 

  CURSOR purge_cursor ( c_target_guid RAW,
                        c_max_purge_time DATE ) IS
    SELECT rowid
      FROM MGMT_RT_DOMAIN_DIST_1DAY
      WHERE target_guid = c_target_guid
        AND rollup_timestamp < c_max_purge_time;

BEGIN

  OPEN purge_cursor( p_cur_target_guid, p_cur_dist_day_purge );
  LOOP

    -- Process batchsize number of rows per commit.
    FETCH purge_cursor BULK COLLECT INTO v_purge_rowid 
      LIMIT p_purge_batchsize;

    -- Exit when nothing to purge
    EXIT WHEN v_purge_rowid.COUNT <= 0;
 
    -- Bulk delete by rowid
    FORALL i IN v_purge_rowid.FIRST..v_purge_rowid.LAST
      DELETE FROM MGMT_RT_DOMAIN_DIST_1DAY 
        WHERE rowid = v_purge_rowid(i);

    COMMIT;

    -- Keep track of the number of deleted rows
    v_rc := v_rc + v_purge_rowid.COUNT;

    -- Clear the rowid list
    v_purge_rowid.DELETE;

  END LOOP;

  IF purge_cursor%ISOPEN THEN
    CLOSE purge_cursor;
  END IF;

  -- Update the global row counters
  p_total_rc := p_total_rc + v_rc;
  p_dom_dist_1day_purge_rc := p_dom_dist_1day_purge_rc + v_rc;

EXCEPTION
  WHEN OTHERS THEN
    ROLLBACK;
    IF purge_cursor%ISOPEN THEN
      CLOSE purge_cursor;
    END IF;
    v_purge_rowid.DELETE;
    MGMT_LOG.LOG_ERROR('PURGE', null,
      'Error during dom dist 1day: ' || SUBSTR(SQLERRM, 1, 1100));
END EMD_DOM_DIST_1DAY_PURGE;
----------------------------------------------------------------------

----------------------------------------------------------------------
PROCEDURE EMD_IP_1DAY_PURGE IS 

  v_purge_rowid     p_rowid_list_type;
  v_rc              PLS_INTEGER := 0; 

  CURSOR purge_cursor ( c_target_guid RAW,
                        c_max_purge_time DATE ) IS
    SELECT rowid
      FROM MGMT_RT_IP_1DAY
      WHERE target_guid = c_target_guid
        AND rollup_timestamp < c_max_purge_time;

BEGIN

  OPEN purge_cursor( p_cur_target_guid, p_cur_day_purge );
  LOOP

    -- Process batchsize number of rows per commit.
    FETCH purge_cursor BULK COLLECT INTO v_purge_rowid 
      LIMIT p_purge_batchsize;

    -- Exit when nothing to purge
    EXIT WHEN v_purge_rowid.COUNT <= 0;

    -- Bulk delete by rowid
    FORALL i IN v_purge_rowid.FIRST..v_purge_rowid.LAST
      DELETE FROM MGMT_RT_IP_1DAY 
        WHERE rowid = v_purge_rowid(i);

    COMMIT;

    -- Keep track of the number of deleted rows
    v_rc := v_rc + v_purge_rowid.COUNT;

    -- Clear the rowid list
    v_purge_rowid.DELETE;  

  END LOOP;

  IF purge_cursor%ISOPEN THEN
    CLOSE purge_cursor;
  END IF;

  -- Update the global row counters
  p_total_rc := p_total_rc + v_rc;
  p_ip_1day_purge_rc := p_ip_1day_purge_rc + v_rc;

EXCEPTION
  WHEN OTHERS THEN
    ROLLBACK;
    IF purge_cursor%ISOPEN THEN
      CLOSE purge_cursor;
    END IF;
    v_purge_rowid.DELETE;
    MGMT_LOG.LOG_ERROR('PURGE', null,
      'Error during ip 1day: ' || SUBSTR(SQLERRM, 1, 1100));
END EMD_IP_1DAY_PURGE;
----------------------------------------------------------------------

----------------------------------------------------------------------
PROCEDURE EMD_IP_DIST_1DAY_PURGE IS 

  v_purge_rowid     p_rowid_list_type;
  v_rc              PLS_INTEGER := 0; 

  CURSOR purge_cursor ( c_target_guid RAW,
                        c_max_purge_time DATE ) IS
    SELECT rowid
      FROM MGMT_RT_IP_DIST_1DAY
      WHERE target_guid = c_target_guid
        AND rollup_timestamp < c_max_purge_time;

BEGIN

  OPEN purge_cursor( p_cur_target_guid, p_cur_dist_day_purge );
  LOOP

    -- Process batchsize number of rows per commit.
    FETCH purge_cursor BULK COLLECT INTO v_purge_rowid 
      LIMIT p_purge_batchsize;

    -- Exit when nothing to purge
    EXIT WHEN v_purge_rowid.COUNT <= 0;
 
    -- Bulk delete by rowid
    FORALL i IN v_purge_rowid.FIRST..v_purge_rowid.LAST
      DELETE FROM MGMT_RT_IP_DIST_1DAY 
        WHERE rowid = v_purge_rowid(i);

    COMMIT;

    -- Keep track of the number of deleted rows
    v_rc := v_rc + v_purge_rowid.COUNT;

    -- Clear the rowid list
    v_purge_rowid.DELETE;

  END LOOP;

  IF purge_cursor%ISOPEN THEN
    CLOSE purge_cursor;
  END IF;
 
  -- Update the global row counters
  p_total_rc := p_total_rc + v_rc;
  p_ip_dist_1day_purge_rc := p_ip_dist_1day_purge_rc + v_rc;

EXCEPTION

  WHEN OTHERS THEN
    ROLLBACK;
    IF purge_cursor%ISOPEN THEN
      CLOSE purge_cursor;
    END IF;
    v_purge_rowid.DELETE;
    MGMT_LOG.LOG_ERROR('PURGE', null,
      'Error during ip dist 1day: ' || SUBSTR(SQLERRM, 1, 1100));
END EMD_IP_DIST_1DAY_PURGE;
----------------------------------------------------------------------

----------------------------------------------------------------------
PROCEDURE EMD_PR_MAPPING_1DAY_PURGE IS 

  v_purge_rowid     p_rowid_list_type;
  v_rc              PLS_INTEGER := 0; 

  CURSOR purge_cursor ( c_target_guid RAW,
                        c_max_purge_time DATE ) IS
    SELECT rowid
      FROM MGMT_RT_PR_MAPPING_1DAY
      WHERE target_guid = c_target_guid
        AND rollup_timestamp < c_max_purge_time;

BEGIN

  OPEN purge_cursor( p_cur_target_guid, p_cur_day_purge );
  LOOP

    -- Process batchsize number of rows per commit.
    FETCH purge_cursor BULK COLLECT INTO v_purge_rowid 
      LIMIT p_purge_batchsize;

    -- Exit when nothing to purge
    EXIT WHEN v_purge_rowid.COUNT <= 0;  
 
    -- Bulk delete by rowid
    FORALL i IN v_purge_rowid.FIRST..v_purge_rowid.LAST
      DELETE FROM MGMT_RT_PR_MAPPING_1DAY
        WHERE rowid = v_purge_rowid(i);

    COMMIT;

    -- Keep track of the number of deleted rows
    v_rc := v_rc + v_purge_rowid.COUNT;

    -- Clear the rowid list
    v_purge_rowid.DELETE;

  END LOOP;

  IF purge_cursor%ISOPEN THEN
    CLOSE purge_cursor;
  END IF;

  -- Update the global row counters
  p_total_rc := p_total_rc + v_rc;
  p_pr_mapping_1day_purge_rc := p_pr_mapping_1day_purge_rc + v_rc;

EXCEPTION
  WHEN OTHERS THEN
    ROLLBACK;
    IF purge_cursor%ISOPEN THEN
      CLOSE purge_cursor;
    END IF;
    v_purge_rowid.DELETE;
    MGMT_LOG.LOG_ERROR('PURGE', null,
      'Error during pr mapping 1day: ' || SUBSTR(SQLERRM, 1, 1100));
END EMD_PR_MAPPING_1DAY_PURGE;
----------------------------------------------------------------------

----------------------------------------------------------------------
PROCEDURE EMD_INC_LOADS_1DAY_PURGE IS 

  v_purge_rowid     p_rowid_list_type;
  v_rc              PLS_INTEGER := 0; 

  CURSOR purge_cursor ( c_target_guid RAW,
                        c_max_purge_time DATE ) IS
    SELECT rowid
      FROM MGMT_RT_INCOMPLETE_LOADS_1DAY
      WHERE target_guid = c_target_guid
        AND rollup_timestamp < c_max_purge_time;

BEGIN

  OPEN purge_cursor( p_cur_target_guid, p_cur_day_purge );
  LOOP

    -- Process batchsize number of rows per commit.
    FETCH purge_cursor BULK COLLECT INTO v_purge_rowid 
      LIMIT p_purge_batchsize;

    -- Exit when nothing to purge
    EXIT WHEN v_purge_rowid.COUNT <= 0;  
 
    -- Bulk delete by rowid
    FORALL i IN v_purge_rowid.FIRST..v_purge_rowid.LAST
      DELETE FROM MGMT_RT_INCOMPLETE_LOADS_1DAY
        WHERE rowid = v_purge_rowid(i);

    COMMIT;

    -- Keep track of the number of deleted rows
    v_rc := v_rc + v_purge_rowid.COUNT;

    -- Clear the rowid list
    v_purge_rowid.DELETE;

  END LOOP;

  IF purge_cursor%ISOPEN THEN
    CLOSE purge_cursor;
  END IF;

  -- Update the global row counters
  p_total_rc := p_total_rc + v_rc;
  p_inc_loads_1day_purge_rc := p_inc_loads_1day_purge_rc + v_rc;

EXCEPTION
  WHEN OTHERS THEN
    ROLLBACK;
    IF purge_cursor%ISOPEN THEN
      CLOSE purge_cursor;
    END IF;
    v_purge_rowid.DELETE;
    MGMT_LOG.LOG_ERROR('PURGE', null,
      'Error during incomplete loads 1day: ' || SUBSTR(SQLERRM, 1, 1100));
END EMD_INC_LOADS_1DAY_PURGE;
----------------------------------------------------------------------

-------------------   PARTIAL ROLLUP CLEAMUP    ----------------------

----------------------------------------------------------------------
FUNCTION EMD_URL_1HOUR_CLEANUP RETURN BOOLEAN IS 

  v_cleanup_rowid   p_rowid_list_type;
  v_rc              PLS_INTEGER := 0; 

  CURSOR cleanup_cursor ( c_target_guid RAW,
                          c_last_good_rollup DATE ) IS
    SELECT rowid
      FROM MGMT_RT_URL_1HOUR
      WHERE target_guid = c_target_guid
        AND rollup_timestamp > c_last_good_rollup;

BEGIN

  OPEN cleanup_cursor( p_cur_target_guid, 
                       p_cur_rollup_timestamps(p_url_1hour_idx) );
  LOOP

    FETCH cleanup_cursor BULK COLLECT INTO v_cleanup_rowid 
      LIMIT p_purge_batchsize;

    EXIT WHEN v_cleanup_rowid.COUNT <= 0;
 
    -- Bulk delete by rowid
    FORALL i IN v_cleanup_rowid.FIRST..v_cleanup_rowid.LAST
      DELETE FROM MGMT_RT_URL_1HOUR 
        WHERE rowid = v_cleanup_rowid(i);

    COMMIT;

    -- Keep track of the number of deleted rows
    v_rc := v_rc + v_cleanup_rowid.COUNT;

    -- Clear the rowid list
    v_cleanup_rowid.DELETE;

  END LOOP;

  IF cleanup_cursor%ISOPEN THEN
    CLOSE cleanup_cursor;
  END IF;

  RETURN TRUE;

EXCEPTION

  WHEN NO_DATA_FOUND THEN
    ROLLBACK;
    IF cleanup_cursor%ISOPEN THEN
      CLOSE cleanup_cursor;
    END IF;
    v_cleanup_rowid.DELETE;
    RETURN TRUE;
  
  WHEN OTHERS THEN
    ROLLBACK;
    IF cleanup_cursor%ISOPEN THEN
      CLOSE cleanup_cursor;
    END IF;
    v_cleanup_rowid.DELETE;
    MGMT_LOG.LOG_ERROR('RTROLLUP', null,
      'Error during url 1hour cleanup: ' || SUBSTR(SQLERRM, 1, 1100));
    RETURN FALSE;

END EMD_URL_1HOUR_CLEANUP;
----------------------------------------------------------------------

----------------------------------------------------------------------
FUNCTION EMD_URL_DIST_1HOUR_CLEANUP RETURN BOOLEAN IS 

  v_cleanup_rowid   p_rowid_list_type;
  v_rc              PLS_INTEGER := 0; 

  CURSOR cleanup_cursor ( c_target_guid RAW,
                          c_last_good_rollup DATE ) IS
    SELECT rowid
      FROM MGMT_RT_URL_DIST_1HOUR
      WHERE target_guid = c_target_guid
        AND rollup_timestamp > c_last_good_rollup;

BEGIN

  OPEN cleanup_cursor( p_cur_target_guid, 
                       p_cur_rollup_timestamps(p_url_dist_1hour_idx) );
  LOOP

    FETCH cleanup_cursor BULK COLLECT INTO v_cleanup_rowid 
      LIMIT p_purge_batchsize;

    EXIT WHEN v_cleanup_rowid.COUNT <= 0;
 
    -- Bulk delete by rowid
    FORALL i IN v_cleanup_rowid.FIRST..v_cleanup_rowid.LAST
      DELETE FROM MGMT_RT_URL_DIST_1HOUR 
        WHERE rowid = v_cleanup_rowid(i);

    COMMIT;

    -- Keep track of the number of deleted rows
    v_rc := v_rc + v_cleanup_rowid.COUNT;

    -- Clear the rowid list
    v_cleanup_rowid.DELETE;

  END LOOP;

  IF cleanup_cursor%ISOPEN THEN
    CLOSE cleanup_cursor;
  END IF;
  RETURN TRUE;

EXCEPTION

  WHEN NO_DATA_FOUND THEN
    ROLLBACK;
    IF cleanup_cursor%ISOPEN THEN
      CLOSE cleanup_cursor;
    END IF;
    v_cleanup_rowid.DELETE;
    RETURN TRUE;
  
  WHEN OTHERS THEN
    ROLLBACK;
    IF cleanup_cursor%ISOPEN THEN
      CLOSE cleanup_cursor;
    END IF;
    v_cleanup_rowid.DELETE;
    MGMT_LOG.LOG_ERROR('RTROLLUP', null,
      'Error during url dist 1hour cleanup: ' || SUBSTR(SQLERRM, 1, 1100));
    RETURN FALSE;

END EMD_URL_DIST_1HOUR_CLEANUP;
----------------------------------------------------------------------

----------------------------------------------------------------------
FUNCTION EMD_DOM_1HOUR_CLEANUP RETURN BOOLEAN IS 

  v_cleanup_rowid   p_rowid_list_type;
  v_rc              PLS_INTEGER := 0; 

  CURSOR cleanup_cursor ( c_target_guid RAW,
                          c_last_good_rollup DATE ) IS
    SELECT rowid
      FROM MGMT_RT_DOMAIN_1HOUR
      WHERE target_guid = c_target_guid
        AND rollup_timestamp > c_last_good_rollup;

BEGIN

  OPEN cleanup_cursor( p_cur_target_guid, 
                       p_cur_rollup_timestamps(p_dom_1hour_idx) );
  LOOP

    FETCH cleanup_cursor BULK COLLECT INTO v_cleanup_rowid 
      LIMIT p_purge_batchsize;

    EXIT WHEN v_cleanup_rowid.COUNT <= 0;
 
    -- Bulk delete by rowid
    FORALL i IN v_cleanup_rowid.FIRST..v_cleanup_rowid.LAST
      DELETE FROM MGMT_RT_DOMAIN_1HOUR 
        WHERE rowid = v_cleanup_rowid(i);

    COMMIT;

    -- Keep track of the number of deleted rows
    v_rc := v_rc + v_cleanup_rowid.COUNT;

    -- Clear the rowid list
    v_cleanup_rowid.DELETE;

  END LOOP;

  IF cleanup_cursor%ISOPEN THEN
    CLOSE cleanup_cursor;
  END IF;
  RETURN TRUE;

EXCEPTION

  WHEN NO_DATA_FOUND THEN
    ROLLBACK;
    IF cleanup_cursor%ISOPEN THEN
      CLOSE cleanup_cursor;
    END IF;
    v_cleanup_rowid.DELETE;
    RETURN TRUE;
  
  WHEN OTHERS THEN
    ROLLBACK;
    IF cleanup_cursor%ISOPEN THEN
      CLOSE cleanup_cursor;
    END IF;
    v_cleanup_rowid.DELETE;
    MGMT_LOG.LOG_ERROR('RTROLLUP', null,
      'Error during dom 1hour cleanup: ' || SUBSTR(SQLERRM, 1, 1100));
    RETURN FALSE;

END EMD_DOM_1HOUR_CLEANUP;
----------------------------------------------------------------------

----------------------------------------------------------------------
FUNCTION EMD_DOM_DIST_1HOUR_CLEANUP RETURN BOOLEAN IS 

  v_cleanup_rowid   p_rowid_list_type;
  v_rc              PLS_INTEGER := 0; 

  CURSOR cleanup_cursor ( c_target_guid RAW,
                          c_last_good_rollup DATE ) IS
    SELECT rowid
      FROM MGMT_RT_DOMAIN_DIST_1HOUR
      WHERE target_guid = c_target_guid
        AND rollup_timestamp > c_last_good_rollup;

BEGIN

  OPEN cleanup_cursor( p_cur_target_guid, 
                       p_cur_rollup_timestamps(p_dom_dist_1hour_idx) );
  LOOP

    FETCH cleanup_cursor BULK COLLECT INTO v_cleanup_rowid 
      LIMIT p_purge_batchsize;

    EXIT WHEN v_cleanup_rowid.COUNT <= 0;
 
    -- Bulk delete by rowid
    FORALL i IN v_cleanup_rowid.FIRST..v_cleanup_rowid.LAST
      DELETE FROM MGMT_RT_DOMAIN_DIST_1HOUR 
        WHERE rowid = v_cleanup_rowid(i);

    COMMIT;

    -- Keep track of the number of deleted rows
    v_rc := v_rc + v_cleanup_rowid.COUNT;

    -- Clear the rowid list
    v_cleanup_rowid.DELETE;

  END LOOP;

  IF cleanup_cursor%ISOPEN THEN
    CLOSE cleanup_cursor;
  END IF;
  RETURN TRUE;

EXCEPTION

  WHEN NO_DATA_FOUND THEN
    ROLLBACK;
    IF cleanup_cursor%ISOPEN THEN
      CLOSE cleanup_cursor;
    END IF;
    v_cleanup_rowid.DELETE;
    RETURN TRUE;
  
  WHEN OTHERS THEN
    ROLLBACK;
    IF cleanup_cursor%ISOPEN THEN
      CLOSE cleanup_cursor;
    END IF;
    v_cleanup_rowid.DELETE;
    MGMT_LOG.LOG_ERROR('RTROLLUP', null,
      'Error during dom dist 1hour cleanup: ' || SUBSTR(SQLERRM, 1, 1100));
    RETURN FALSE;

END EMD_DOM_DIST_1HOUR_CLEANUP;
----------------------------------------------------------------------

----------------------------------------------------------------------
FUNCTION EMD_IP_1HOUR_CLEANUP RETURN BOOLEAN IS 

  v_cleanup_rowid   p_rowid_list_type;
  v_rc              PLS_INTEGER := 0; 

  CURSOR cleanup_cursor ( c_target_guid RAW,
                          c_last_good_rollup DATE ) IS
    SELECT rowid
      FROM MGMT_RT_IP_1HOUR
      WHERE target_guid = c_target_guid
        AND rollup_timestamp > c_last_good_rollup;

BEGIN

  OPEN cleanup_cursor( p_cur_target_guid, 
                       p_cur_rollup_timestamps(p_ip_1hour_idx) );
  LOOP

    FETCH cleanup_cursor BULK COLLECT INTO v_cleanup_rowid 
      LIMIT p_purge_batchsize;

    EXIT WHEN v_cleanup_rowid.COUNT <= 0;
 
    -- Bulk delete by rowid
    FORALL i IN v_cleanup_rowid.FIRST..v_cleanup_rowid.LAST
      DELETE FROM MGMT_RT_IP_1HOUR 
        WHERE rowid = v_cleanup_rowid(i);

    COMMIT;

    -- Keep track of the number of deleted rows
    v_rc := v_rc + v_cleanup_rowid.COUNT;

    -- Clear the rowid list
    v_cleanup_rowid.DELETE;

  END LOOP;

  IF cleanup_cursor%ISOPEN THEN
    CLOSE cleanup_cursor;
  END IF;
  RETURN TRUE;

EXCEPTION

  WHEN NO_DATA_FOUND THEN
    ROLLBACK;
    IF cleanup_cursor%ISOPEN THEN
      CLOSE cleanup_cursor;
    END IF;
    v_cleanup_rowid.DELETE;
    RETURN TRUE;

  WHEN OTHERS THEN
    ROLLBACK;
    IF cleanup_cursor%ISOPEN THEN
      CLOSE cleanup_cursor;
    END IF;
    v_cleanup_rowid.DELETE;
    MGMT_LOG.LOG_ERROR('RTROLLUP', null,
      'Error during ip 1hour cleanup: ' || SUBSTR(SQLERRM, 1, 1100));
    RETURN FALSE;

END EMD_IP_1HOUR_CLEANUP;
----------------------------------------------------------------------

----------------------------------------------------------------------
FUNCTION EMD_IP_DIST_1HOUR_CLEANUP RETURN BOOLEAN IS

  v_cleanup_rowid   p_rowid_list_type;
  v_rc              PLS_INTEGER := 0;

  CURSOR cleanup_cursor ( c_target_guid RAW,
                          c_last_good_rollup DATE ) IS
    SELECT rowid
      FROM MGMT_RT_IP_DIST_1HOUR
      WHERE target_guid = c_target_guid
        AND rollup_timestamp > c_last_good_rollup;

BEGIN

  OPEN cleanup_cursor( p_cur_target_guid, 
                       p_cur_rollup_timestamps(p_ip_dist_1hour_idx) );
  LOOP

    FETCH cleanup_cursor BULK COLLECT INTO v_cleanup_rowid 
      LIMIT p_purge_batchsize;

    EXIT WHEN v_cleanup_rowid.COUNT <= 0;
 
    -- Bulk delete by rowid
    FORALL i IN v_cleanup_rowid.FIRST..v_cleanup_rowid.LAST
      DELETE FROM MGMT_RT_IP_DIST_1HOUR 
        WHERE rowid = v_cleanup_rowid(i);

    COMMIT;

    -- Keep track of the number of deleted rows
    v_rc := v_rc + v_cleanup_rowid.COUNT;

    -- Clear the rowid list
    v_cleanup_rowid.DELETE;

  END LOOP;

  IF cleanup_cursor%ISOPEN THEN
    CLOSE cleanup_cursor;
  END IF;
  RETURN TRUE;

EXCEPTION

  WHEN NO_DATA_FOUND THEN
    ROLLBACK;
    IF cleanup_cursor%ISOPEN THEN
      CLOSE cleanup_cursor;
    END IF;
    v_cleanup_rowid.DELETE;
    RETURN TRUE;
  
  WHEN OTHERS THEN
    ROLLBACK;
    IF cleanup_cursor%ISOPEN THEN
      CLOSE cleanup_cursor;
    END IF;
    v_cleanup_rowid.DELETE;
    MGMT_LOG.LOG_ERROR('RTROLLUP', null,
      'Error during ip dist 1hour cleanup: ' || SUBSTR(SQLERRM, 1, 1100));
    RETURN FALSE;

END EMD_IP_DIST_1HOUR_CLEANUP;
----------------------------------------------------------------------

----------------------------------------------------------------------
FUNCTION EMD_URL_1DAY_CLEANUP RETURN BOOLEAN IS 

  v_cleanup_rowid   p_rowid_list_type;
  v_rc              PLS_INTEGER := 0; 

  CURSOR cleanup_cursor ( c_target_guid RAW,
                          c_last_good_rollup DATE ) IS
    SELECT rowid
      FROM MGMT_RT_URL_1DAY
      WHERE target_guid = c_target_guid
        AND rollup_timestamp > c_last_good_rollup;

BEGIN

  OPEN cleanup_cursor( p_cur_target_guid, 
                       p_cur_rollup_timestamps(p_url_1day_idx) );
  LOOP

    FETCH cleanup_cursor BULK COLLECT INTO v_cleanup_rowid 
      LIMIT p_purge_batchsize;

    EXIT WHEN v_cleanup_rowid.COUNT <= 0;
 
    -- Bulk delete by rowid
    FORALL i IN v_cleanup_rowid.FIRST..v_cleanup_rowid.LAST
      DELETE FROM MGMT_RT_URL_1DAY
        WHERE rowid = v_cleanup_rowid(i);

    COMMIT;

    -- Keep track of the number of deleted rows
    v_rc := v_rc + v_cleanup_rowid.COUNT;

    -- Clear the rowid list
    v_cleanup_rowid.DELETE;

  END LOOP;

  IF cleanup_cursor%ISOPEN THEN
    CLOSE cleanup_cursor;
  END IF;
  RETURN TRUE;

EXCEPTION

  WHEN NO_DATA_FOUND THEN
    ROLLBACK;
    IF cleanup_cursor%ISOPEN THEN
      CLOSE cleanup_cursor;
    END IF;
    v_cleanup_rowid.DELETE;
    RETURN TRUE;
  
  WHEN OTHERS THEN
    ROLLBACK;
    IF cleanup_cursor%ISOPEN THEN
      CLOSE cleanup_cursor;
    END IF;
    v_cleanup_rowid.DELETE;
    MGMT_LOG.LOG_ERROR('RTROLLUP', null,
      'Error during url 1day cleanup: ' || SUBSTR(SQLERRM, 1, 1100));
    RETURN FALSE;

END EMD_URL_1DAY_CLEANUP;
----------------------------------------------------------------------

----------------------------------------------------------------------
FUNCTION EMD_URL_DIST_1DAY_CLEANUP RETURN BOOLEAN IS 

  v_cleanup_rowid   p_rowid_list_type;
  v_rc              PLS_INTEGER := 0; 

  CURSOR cleanup_cursor ( c_target_guid RAW,
                          c_last_good_rollup DATE ) IS
    SELECT rowid
      FROM MGMT_RT_URL_DIST_1DAY
      WHERE target_guid = c_target_guid
        AND rollup_timestamp > c_last_good_rollup;

BEGIN

  OPEN cleanup_cursor( p_cur_target_guid, 
                       p_cur_rollup_timestamps(p_url_dist_1day_idx) );
  LOOP

    FETCH cleanup_cursor BULK COLLECT INTO v_cleanup_rowid 
      LIMIT p_purge_batchsize;

    EXIT WHEN v_cleanup_rowid.COUNT <= 0;
 
    -- Bulk delete by rowid
    FORALL i IN v_cleanup_rowid.FIRST..v_cleanup_rowid.LAST
      DELETE FROM MGMT_RT_URL_DIST_1DAY
        WHERE rowid = v_cleanup_rowid(i);

    COMMIT;

    -- Keep track of the number of deleted rows
    v_rc := v_rc + v_cleanup_rowid.COUNT;

    -- Clear the rowid list
    v_cleanup_rowid.DELETE;

  END LOOP;

  IF cleanup_cursor%ISOPEN THEN
    CLOSE cleanup_cursor;
  END IF;
  RETURN TRUE;

EXCEPTION

  WHEN NO_DATA_FOUND THEN
    ROLLBACK;
    IF cleanup_cursor%ISOPEN THEN
      CLOSE cleanup_cursor;
    END IF;
    v_cleanup_rowid.DELETE;
    RETURN TRUE;
  
  WHEN OTHERS THEN
    ROLLBACK;
    IF cleanup_cursor%ISOPEN THEN
      CLOSE cleanup_cursor;
    END IF;
    v_cleanup_rowid.DELETE;
    MGMT_LOG.LOG_ERROR('RTROLLUP', null,
      'Error during url dist 1day cleanup: ' || SUBSTR(SQLERRM, 1, 1100));
    RETURN FALSE;

END EMD_URL_DIST_1DAY_CLEANUP;
----------------------------------------------------------------------

----------------------------------------------------------------------
FUNCTION EMD_DOM_1DAY_CLEANUP RETURN BOOLEAN IS 

  v_cleanup_rowid   p_rowid_list_type;
  v_rc              PLS_INTEGER := 0; 

  CURSOR cleanup_cursor ( c_target_guid RAW,
                          c_last_good_rollup DATE ) IS
    SELECT rowid
      FROM MGMT_RT_DOMAIN_1DAY
      WHERE target_guid = c_target_guid
        AND rollup_timestamp > c_last_good_rollup;

BEGIN

  OPEN cleanup_cursor( p_cur_target_guid, 
                       p_cur_rollup_timestamps(p_dom_1day_idx) );
  LOOP

    FETCH cleanup_cursor BULK COLLECT INTO v_cleanup_rowid 
      LIMIT p_purge_batchsize;

    EXIT WHEN v_cleanup_rowid.COUNT <= 0;
 
    -- Bulk delete by rowid
    FORALL i IN v_cleanup_rowid.FIRST..v_cleanup_rowid.LAST
      DELETE FROM MGMT_RT_DOMAIN_1DAY
        WHERE rowid = v_cleanup_rowid(i);

    COMMIT;

    -- Keep track of the number of deleted rows
    v_rc := v_rc + v_cleanup_rowid.COUNT;

    -- Clear the rowid list
    v_cleanup_rowid.DELETE;

  END LOOP;

  IF cleanup_cursor%ISOPEN THEN
    CLOSE cleanup_cursor;
  END IF;
  RETURN TRUE;

EXCEPTION

  WHEN NO_DATA_FOUND THEN
    ROLLBACK;
    IF cleanup_cursor%ISOPEN THEN
      CLOSE cleanup_cursor;
    END IF;
    v_cleanup_rowid.DELETE;
    RETURN TRUE;
  
  WHEN OTHERS THEN
    ROLLBACK;
    IF cleanup_cursor%ISOPEN THEN
      CLOSE cleanup_cursor;
    END IF;
    v_cleanup_rowid.DELETE;
    MGMT_LOG.LOG_ERROR('RTROLLUP', null,
      'Error during dom 1day cleanup: ' || SUBSTR(SQLERRM, 1, 1100));
    RETURN FALSE;

END EMD_DOM_1DAY_CLEANUP;
----------------------------------------------------------------------

----------------------------------------------------------------------
FUNCTION EMD_DOM_DIST_1DAY_CLEANUP RETURN BOOLEAN IS 

  v_cleanup_rowid   p_rowid_list_type;
  v_rc              PLS_INTEGER := 0; 

  CURSOR cleanup_cursor ( c_target_guid RAW,
                          c_last_good_rollup DATE ) IS
    SELECT rowid
      FROM MGMT_RT_DOMAIN_DIST_1DAY
      WHERE target_guid = c_target_guid
        AND rollup_timestamp > c_last_good_rollup;

BEGIN

  OPEN cleanup_cursor( p_cur_target_guid, 
                       p_cur_rollup_timestamps(p_dom_dist_1day_idx) );
  LOOP

    FETCH cleanup_cursor BULK COLLECT INTO v_cleanup_rowid 
      LIMIT p_purge_batchsize;

    EXIT WHEN v_cleanup_rowid.COUNT <= 0;
 
    -- Bulk delete by rowid
    FORALL i IN v_cleanup_rowid.FIRST..v_cleanup_rowid.LAST
      DELETE FROM MGMT_RT_DOMAIN_DIST_1DAY
        WHERE rowid = v_cleanup_rowid(i);

    COMMIT;

    -- Keep track of the number of deleted rows
    v_rc := v_rc + v_cleanup_rowid.COUNT;

    -- Clear the rowid list
    v_cleanup_rowid.DELETE;

  END LOOP;

  IF cleanup_cursor%ISOPEN THEN
    CLOSE cleanup_cursor;
  END IF;
  RETURN TRUE;

EXCEPTION

  WHEN NO_DATA_FOUND THEN
    ROLLBACK;
    IF cleanup_cursor%ISOPEN THEN
      CLOSE cleanup_cursor;
    END IF;
    v_cleanup_rowid.DELETE;
    RETURN TRUE;

  WHEN OTHERS THEN
    ROLLBACK;
    IF cleanup_cursor%ISOPEN THEN
      CLOSE cleanup_cursor;
    END IF;
    v_cleanup_rowid.DELETE;
    MGMT_LOG.LOG_ERROR('RTROLLUP', null,
      'Error during dom dist 1day cleanup: ' || SUBSTR(SQLERRM, 1, 1100));
    RETURN FALSE;

END EMD_DOM_DIST_1DAY_CLEANUP;
----------------------------------------------------------------------

----------------------------------------------------------------------
FUNCTION EMD_IP_1DAY_CLEANUP RETURN BOOLEAN IS 

  v_cleanup_rowid   p_rowid_list_type;
  v_rc              PLS_INTEGER := 0; 

  CURSOR cleanup_cursor ( c_target_guid RAW,
                          c_last_good_rollup DATE ) IS
    SELECT rowid
      FROM MGMT_RT_IP_1DAY
      WHERE target_guid = c_target_guid
        AND rollup_timestamp > c_last_good_rollup;

BEGIN

  OPEN cleanup_cursor( p_cur_target_guid, 
                       p_cur_rollup_timestamps(p_ip_1day_idx) );
  LOOP

    FETCH cleanup_cursor BULK COLLECT INTO v_cleanup_rowid 
      LIMIT p_purge_batchsize;

    EXIT WHEN v_cleanup_rowid.COUNT <= 0;
 
    -- Bulk delete by rowid
    FORALL i IN v_cleanup_rowid.FIRST..v_cleanup_rowid.LAST
      DELETE FROM MGMT_RT_IP_1DAY
        WHERE rowid = v_cleanup_rowid(i);

    COMMIT;

    -- Keep track of the number of deleted rows
    v_rc := v_rc + v_cleanup_rowid.COUNT;

    -- Clear the rowid list
    v_cleanup_rowid.DELETE;

  END LOOP;

  IF cleanup_cursor%ISOPEN THEN
    CLOSE cleanup_cursor;
  END IF;
  RETURN TRUE;

EXCEPTION

  WHEN NO_DATA_FOUND THEN
    ROLLBACK;
    IF cleanup_cursor%ISOPEN THEN
      CLOSE cleanup_cursor;
    END IF;
    v_cleanup_rowid.DELETE;
    RETURN TRUE;
  
  WHEN OTHERS THEN
    ROLLBACK;
    IF cleanup_cursor%ISOPEN THEN
      CLOSE cleanup_cursor;
    END IF;
    v_cleanup_rowid.DELETE;
    MGMT_LOG.LOG_ERROR('RTROLLUP', null,
      'Error during ip 1day cleanup: ' || SUBSTR(SQLERRM, 1, 1100));
    RETURN FALSE;

END EMD_IP_1DAY_CLEANUP;
----------------------------------------------------------------------

----------------------------------------------------------------------
FUNCTION EMD_IP_DIST_1DAY_CLEANUP RETURN BOOLEAN IS 

  v_cleanup_rowid   p_rowid_list_type;
  v_rc              PLS_INTEGER := 0; 

  CURSOR cleanup_cursor ( c_target_guid RAW,
                          c_last_good_rollup DATE ) IS
    SELECT rowid
      FROM MGMT_RT_IP_DIST_1DAY
      WHERE target_guid = c_target_guid
        AND rollup_timestamp > c_last_good_rollup;

BEGIN

  OPEN cleanup_cursor( p_cur_target_guid, 
                       p_cur_rollup_timestamps(p_ip_dist_1day_idx) );
  LOOP

    FETCH cleanup_cursor BULK COLLECT INTO v_cleanup_rowid 
      LIMIT p_purge_batchsize;

    EXIT WHEN v_cleanup_rowid.COUNT <= 0;
 
    -- Bulk delete by rowid
    FORALL i IN v_cleanup_rowid.FIRST..v_cleanup_rowid.LAST
      DELETE FROM MGMT_RT_IP_DIST_1DAY
        WHERE rowid = v_cleanup_rowid(i);

    COMMIT;

    -- Keep track of the number of deleted rows
    v_rc := v_rc + v_cleanup_rowid.COUNT;

    -- Clear the rowid list
    v_cleanup_rowid.DELETE;

  END LOOP;

  IF cleanup_cursor%ISOPEN THEN
    CLOSE cleanup_cursor;
  END IF;
  RETURN TRUE;

EXCEPTION

  WHEN NO_DATA_FOUND THEN
    ROLLBACK;
    IF cleanup_cursor%ISOPEN THEN
      CLOSE cleanup_cursor;
    END IF;
    v_cleanup_rowid.DELETE;
    RETURN TRUE;
  
  WHEN OTHERS THEN
    ROLLBACK;
    IF cleanup_cursor%ISOPEN THEN
      CLOSE cleanup_cursor;
    END IF;
    v_cleanup_rowid.DELETE;
    MGMT_LOG.LOG_ERROR('RTROLLUP', null,
      'Error during ip dist 1day cleanup: ' || SUBSTR(SQLERRM, 1, 1100));
    RETURN FALSE;

END EMD_IP_DIST_1DAY_CLEANUP;
----------------------------------------------------------------------

----------------------------------------------------------------------
FUNCTION EMD_PR_MAPPING_1HOUR_CLEANUP RETURN BOOLEAN IS 

  v_cleanup_rowid   p_rowid_list_type;
  v_rc              PLS_INTEGER := 0; 

  CURSOR cleanup_cursor ( c_target_guid RAW,
                          c_last_good_rollup DATE ) IS
    SELECT rowid
      FROM MGMT_RT_PR_MAPPING_1HOUR
      WHERE target_guid = c_target_guid
        AND rollup_timestamp > c_last_good_rollup;

BEGIN

  OPEN cleanup_cursor( p_cur_target_guid, 
                       p_cur_rollup_timestamps(p_pr_mapping_1hour_idx) );
  LOOP

    FETCH cleanup_cursor BULK COLLECT INTO v_cleanup_rowid 
      LIMIT p_purge_batchsize;

    EXIT WHEN v_cleanup_rowid.COUNT <= 0;
 
    -- Bulk delete by rowid
    FORALL i IN v_cleanup_rowid.FIRST..v_cleanup_rowid.LAST
      DELETE FROM MGMT_RT_PR_MAPPING_1HOUR 
        WHERE rowid = v_cleanup_rowid(i);

    COMMIT;

    -- Keep track of the number of deleted rows
    v_rc := v_rc + v_cleanup_rowid.COUNT;

    -- Clear the rowid list
    v_cleanup_rowid.DELETE;

  END LOOP;

  IF cleanup_cursor%ISOPEN THEN
    CLOSE cleanup_cursor;
  END IF;
  RETURN TRUE;

EXCEPTION

  WHEN NO_DATA_FOUND THEN
    ROLLBACK;
    IF cleanup_cursor%ISOPEN THEN
      CLOSE cleanup_cursor;
    END IF;
    v_cleanup_rowid.DELETE;
    RETURN TRUE;

  WHEN OTHERS THEN
    ROLLBACK;
    IF cleanup_cursor%ISOPEN THEN
      CLOSE cleanup_cursor;
    END IF;
    v_cleanup_rowid.DELETE;
    MGMT_LOG.LOG_ERROR('RTROLLUP', null,
      'Error during pr mapping 1hour cleanup: ' || SUBSTR(SQLERRM, 1, 1100));
    RETURN FALSE;

END EMD_PR_MAPPING_1HOUR_CLEANUP;
----------------------------------------------------------------------

----------------------------------------------------------------------
FUNCTION EMD_INC_LOADS_1HOUR_CLEANUP RETURN BOOLEAN IS 

  v_cleanup_rowid   p_rowid_list_type;
  v_rc              PLS_INTEGER := 0; 

  CURSOR cleanup_cursor ( c_target_guid RAW,
                          c_last_good_rollup DATE ) IS
    SELECT rowid
      FROM MGMT_RT_INCOMPLETE_LOADS_1HOUR
      WHERE target_guid = c_target_guid
        AND rollup_timestamp > c_last_good_rollup;

BEGIN

  OPEN cleanup_cursor( p_cur_target_guid, 
                       p_cur_rollup_timestamps(p_inc_loads_1hour_idx) );
  LOOP

    FETCH cleanup_cursor BULK COLLECT INTO v_cleanup_rowid 
      LIMIT p_purge_batchsize;

    EXIT WHEN v_cleanup_rowid.COUNT <= 0;
 
    -- Bulk delete by rowid
    FORALL i IN v_cleanup_rowid.FIRST..v_cleanup_rowid.LAST
      DELETE FROM MGMT_RT_INCOMPLETE_LOADS_1HOUR 
        WHERE rowid = v_cleanup_rowid(i);

    COMMIT;

    -- Keep track of the number of deleted rows
    v_rc := v_rc + v_cleanup_rowid.COUNT;

    -- Clear the rowid list
    v_cleanup_rowid.DELETE;

  END LOOP;

  IF cleanup_cursor%ISOPEN THEN
    CLOSE cleanup_cursor;
  END IF;
  RETURN TRUE;

EXCEPTION

  WHEN NO_DATA_FOUND THEN
    ROLLBACK;
    IF cleanup_cursor%ISOPEN THEN
      CLOSE cleanup_cursor;
    END IF;
    v_cleanup_rowid.DELETE;
    RETURN TRUE;

  WHEN OTHERS THEN
    ROLLBACK;
    IF cleanup_cursor%ISOPEN THEN
      CLOSE cleanup_cursor;
    END IF;
    v_cleanup_rowid.DELETE;
    MGMT_LOG.LOG_ERROR('RTROLLUP', null,
      'Error during incomplete loads 1hour cleanup: ' || SUBSTR(SQLERRM, 1, 1100));
    RETURN FALSE;

END EMD_INC_LOADS_1HOUR_CLEANUP;
----------------------------------------------------------------------

----------------------------------------------------------------------
FUNCTION EMD_PR_MAPPING_1DAY_CLEANUP RETURN BOOLEAN IS 

  v_cleanup_rowid   p_rowid_list_type;
  v_rc              PLS_INTEGER := 0; 

  CURSOR cleanup_cursor ( c_target_guid RAW,
                          c_last_good_rollup DATE ) IS
    SELECT rowid
      FROM MGMT_RT_PR_MAPPING_1DAY
      WHERE target_guid = c_target_guid
        AND rollup_timestamp > c_last_good_rollup;

BEGIN

  OPEN cleanup_cursor( p_cur_target_guid, 
                       p_cur_rollup_timestamps(p_pr_mapping_1day_idx) );
  LOOP

    FETCH cleanup_cursor BULK COLLECT INTO v_cleanup_rowid 
      LIMIT p_purge_batchsize;

    EXIT WHEN v_cleanup_rowid.COUNT <= 0;
 
    -- Bulk delete by rowid
    FORALL i IN v_cleanup_rowid.FIRST..v_cleanup_rowid.LAST
      DELETE FROM MGMT_RT_PR_MAPPING_1DAY
        WHERE rowid = v_cleanup_rowid(i);

    COMMIT;

    -- Keep track of the number of deleted rows
    v_rc := v_rc + v_cleanup_rowid.COUNT;

    -- Clear the rowid list
    v_cleanup_rowid.DELETE;

  END LOOP;

  IF cleanup_cursor%ISOPEN THEN
    CLOSE cleanup_cursor;
  END IF;
  RETURN TRUE;

EXCEPTION

  WHEN NO_DATA_FOUND THEN
    ROLLBACK;
    IF cleanup_cursor%ISOPEN THEN
      CLOSE cleanup_cursor;
    END IF;
    v_cleanup_rowid.DELETE;
    RETURN TRUE;

  WHEN OTHERS THEN
    ROLLBACK;
    IF cleanup_cursor%ISOPEN THEN
      CLOSE cleanup_cursor;
    END IF;
    v_cleanup_rowid.DELETE;
    MGMT_LOG.LOG_ERROR('RTROLLUP', null,
      'Error during pr mapping 1day cleanup: ' || SUBSTR(SQLERRM, 1, 1100));
    RETURN FALSE;

END EMD_PR_MAPPING_1DAY_CLEANUP;
----------------------------------------------------------------------

----------------------------------------------------------------------
FUNCTION EMD_INC_LOADS_1DAY_CLEANUP RETURN BOOLEAN IS 

  v_cleanup_rowid   p_rowid_list_type;
  v_rc              PLS_INTEGER := 0; 

  CURSOR cleanup_cursor ( c_target_guid RAW,
                          c_last_good_rollup DATE ) IS
    SELECT rowid
      FROM MGMT_RT_INCOMPLETE_LOADS_1DAY
      WHERE target_guid = c_target_guid
        AND rollup_timestamp > c_last_good_rollup;

BEGIN

  OPEN cleanup_cursor( p_cur_target_guid, 
                       p_cur_rollup_timestamps(p_inc_loads_1day_idx) );
  LOOP

    FETCH cleanup_cursor BULK COLLECT INTO v_cleanup_rowid 
      LIMIT p_purge_batchsize;

    EXIT WHEN v_cleanup_rowid.COUNT <= 0;
 
    -- Bulk delete by rowid
    FORALL i IN v_cleanup_rowid.FIRST..v_cleanup_rowid.LAST
      DELETE FROM MGMT_RT_INCOMPLETE_LOADS_1DAY
        WHERE rowid = v_cleanup_rowid(i);

    COMMIT;

    -- Keep track of the number of deleted rows
    v_rc := v_rc + v_cleanup_rowid.COUNT;

    -- Clear the rowid list
    v_cleanup_rowid.DELETE;

  END LOOP;

  IF cleanup_cursor%ISOPEN THEN
    CLOSE cleanup_cursor;
  END IF;
  RETURN TRUE;

EXCEPTION

  WHEN NO_DATA_FOUND THEN
    ROLLBACK;
    IF cleanup_cursor%ISOPEN THEN
      CLOSE cleanup_cursor;
    END IF;
    v_cleanup_rowid.DELETE;
    RETURN TRUE;

  WHEN OTHERS THEN
    ROLLBACK;
    IF cleanup_cursor%ISOPEN THEN
      CLOSE cleanup_cursor;
    END IF;
    v_cleanup_rowid.DELETE;
    MGMT_LOG.LOG_ERROR('RTROLLUP', null,
      'Error during incomplete loads 1day cleanup: ' || SUBSTR(SQLERRM, 1, 1100));
    RETURN FALSE;

END EMD_INC_LOADS_1DAY_CLEANUP;
----------------------------------------------------------------------

END EMD_RT_ROLLUP;
/
show errors;
