create or replace package MGMT_IP_SAMPLES
as

  TYPE CURSOR_TYPE IS REF CURSOR;

  PROCEDURE LOAD;
  PROCEDURE LOAD_NLS_STRINGS;

  PROCEDURE sample_chart_with_plsql (
                result_cursor_out        OUT CURSOR_TYPE,
                tgt_guid_in              IN  MGMT_IP_TGT_GUID_ARRAY,
                start_date_in            IN DATE DEFAULT NULL,
                end_date_in              IN DATE DEFAULT NULL
                );

end MGMT_IP_SAMPLES;
/
show errors

-- package to load samples
create or replace package body MGMT_IP_SAMPLES
as

--
-- example of a pl/sql procedure call used with Chart from SQL element
--
PROCEDURE sample_chart_with_plsql (
                result_cursor_out        OUT CURSOR_TYPE,
                tgt_guid_in              IN  MGMT_IP_TGT_GUID_ARRAY,
                start_date_in            IN  DATE DEFAULT NULL,
                end_date_in              IN  DATE DEFAULT NULL
                ) is 
  query_string   VARCHAR(6000);
  where_targets  VARCHAR(2000) := ' ';
BEGIN

  IF tgt_guid_in.COUNT > 0 THEN
    
      IF tgt_guid_in.COUNT = 1 
      THEN
        where_targets := ' where tgt.target_guid = HEXTORAW(''' ||
          tgt_guid_in(1) || ''') ';
      ELSE
        where_targets := ' where (';
        
        FOR i IN 1..tgt_guid_in.COUNT
        LOOP 
          where_targets := where_targets || ' tgt.target_guid = HEXTORAW(''' || tgt_guid_in(i) || ''') ' ;
          IF i < tgt_guid_in.COUNT
          THEN
            where_targets := where_targets || ' or ' ;
          END IF;
        END LOOP;
        
        where_targets := where_targets || ') ';

      END IF;
    
    END IF;

    query_string := 'select tgt.target_type, count(*) ';
    query_string := query_string || ' from mgmt$target tgt ';
    query_string := query_string || where_targets;
    query_string := query_string || '  group by tgt.target_type';

    OPEN result_cursor_out for query_string;
    
END sample_chart_with_plsql;

PROCEDURE load is 
  l_target_types  MGMT_IP_TARGET_TYPES;
  l_param_classes MGMT_IP_PARAM_CLASSES;
  l_param_values  MGMT_IP_PARAM_VALUE_LIST;
  l_targets       MGMT_IP_TARGET_LIST;
  l_report_guid   RAW(16);
  l_element_guid  RAW(16);
  l_curr_order    NUMBER;
  l_curr_row      NUMBER;
BEGIN


    -- Report Definitions
    
    -------------------------------------------------------------------
    -- All Targets - a most basic report
    -------------------------------------------------------------------
    
    l_report_guid := mgmt_ip.create_report_definition (
      p_title_nlsid                   => 'IPMSGSAMP_All_Targets_Title',
      p_description_nlsid             => 'IPMSGSAMP_All_Targets_Description',
      p_owner                         => 'SYSMAN',
      p_category_nlsid                => 'IPMSGSAMP_SAMPLES', 
      p_sub_category_nlsid            => 'IPMSG_TARGETS',
      p_late_binding_target_types     =>  null,
      p_late_binding_multi_targets    =>  0,
      p_show_table_of_contents        =>  1, 
      p_system_report                 =>  1
        );


    l_curr_order := 1;
    l_curr_row := 1;

    -- register this named sql statement
    --
    mgmt_ip.register_sql_statement (
                p_name  =>'oracle.sysman.eml.ip.render.TARGETS_TABLE_SQL',
                p_sql_statement => 'select target_type, count(*) from mgmt$target group by target_type'
                );

    l_param_values := MGMT_IP_PARAM_VALUE_LIST();
    l_param_values.extend(5);
    l_param_values(1) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.NamedSqlStatement',
      'oracle.sysman.eml.ip.render.TARGETS_TABLE_SQL');
    l_param_values(2) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.ChartParamController.width',
      '200');
    l_param_values(3) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.ChartParamController.height',
      '200');
    l_param_values(4) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.ChartParamController.chartType',
      'pieChart');
    l_param_values(5) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.ChartParamController.pieShowSlicePercentLabels',
      'true');

    -- Create an element definition for this element so that users can
    -- include it in their own reports.  Set the default paramters 
    -- to our original values that the produce ships with.
    --
    mgmt_ip.add_chart_from_sql_elem_def(
          p_element_name_nlsid  => 'IPMSGSAMP_ALL_TARGETS_CHART',
          p_element_type_nlsid  => 'IPMSG_ANY_TARGET_TYPE',
          p_description_nlsid   => 'IPMSGSAMP_all_target_pie_chart',
          p_internal_use_only   => 0, 
          p_default_parameters  => l_param_values
          );

    -- Now add an element of the newly created type to this report.
    --
    l_element_guid := mgmt_ip.add_element_to_report_def (
      p_report_guid         => l_report_guid,
      p_element_name_nlsid  => 'IPMSGSAMP_ALL_TARGETS_CHART',
      p_element_type_nlsid  => 'IPMSG_ANY_TARGET_TYPE',
      p_header_nlsid        => 'IPMSGSAMP_target_types_chart_header',
      p_element_order       => l_curr_order, 
      p_element_row         => l_curr_row, 
      p_parameters          => null,
      p_targets             => null
        );      
      l_curr_order := l_curr_order + 1;

    l_param_values := MGMT_IP_PARAM_VALUE_LIST();

    mgmt_ip.register_sql_statement (
                p_name  =>'oracle.sysman.eml.ip.render.elem.table.component.ALL_TARGETS_TABLE_SQL',
                p_sql_statement => 'select TARGET_TYPE, count(*) from mgmt$target group by TARGET_TYPE'
                );

    l_param_values.extend(1);
    l_param_values(1) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.NamedSqlStatement',
      'oracle.sysman.eml.ip.render.elem.table.component.ALL_TARGETS_TABLE_SQL');
    l_param_values.extend(1);
    l_param_values(2) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.TableRender.targetTypeColumns',
      'TARGET_TYPE');

    mgmt_ip.add_table_from_sql_elem_def(
          p_element_name_nlsid  => 'IPMSGSAMP_ALL_TARGETS_TABLE',
          p_element_type_nlsid  => 'IPMSG_ANY_TARGET_TYPE',
          p_description_nlsid   => 'IPMSGSAMP_all_target_table',
          p_internal_use_only   => 0, 
          p_default_parameters  => l_param_values
          );

    l_element_guid := mgmt_ip.add_element_to_report_def (
      p_report_guid         => l_report_guid,
      p_element_name_nlsid  => 'IPMSGSAMP_ALL_TARGETS_TABLE',
      p_element_type_nlsid  => 'IPMSG_ANY_TARGET_TYPE',
      p_header_nlsid        => 'IPMSGSAMP_targets_table_header',
      p_element_order       => l_curr_order, 
      p_element_row         => l_curr_row, 
      p_parameters          => null,
      p_targets             => null
        );

    -------------------------------------------------------------------
    -- current status all metrics with user database selection
    -------------------------------------------------------------------

    l_target_types := MGMT_IP_TARGET_TYPES();
    l_target_types.extend(1);
    l_target_types(1) := 'oracle_database';
    
    l_report_guid := mgmt_ip.create_report_definition (
      p_title_nlsid                   => 'Current_status: user selects a database target',
      p_description_nlsid             => 'Current_status of all metrics for selected database',
      p_owner                         => 'SYSMAN',
      p_category_nlsid                => 'IPMSGSAMP_SAMPLES', 
      p_sub_category_nlsid            => 'IPMSG_TARGETS',
      p_late_binding_target_types     =>  l_target_types,
      p_late_binding_multi_targets    =>  0,
      p_show_table_of_contents        =>  0, 
      p_system_report                 =>  1
        );

    l_curr_order := 1;
    l_curr_row := 1;

    -- add this element to allow the user to choose targets for this report
    -- that will be stored with the report definition
    --
    -- this element does not get rendered - it is simply to select targets
    -- from the Set Properties option when targets are stored with the report      
    l_element_guid := mgmt_ip.add_element_to_report_def(
      p_report_guid         => l_report_guid,
      p_element_name_nlsid  => 'IPMSG_DB_TARGET_SELECTOR_ELEMENT',
      p_element_type_nlsid  => 'IPMSG_DATABASE_TARGET_TYPE',
      p_header_nlsid        => 'IPMSGSAMP_select_database_targets_for_report',
      p_element_order       => l_curr_order, 
      p_element_row         => l_curr_row, 
      p_parameters          => null,
      p_targets             => null
        );
      l_curr_order := l_curr_order + 1;


    mgmt_ip.register_sql_statement (
                p_name  =>'oracle.sysman.eml.ip.render.elem.table.component.IPMSGSAMP_USER_SELECTS_DATABASE',
                p_sql_statement => 
        '  select distinct met.metric_name, met.metric_column, ' 
        || '       s.collection_timestamp, s.severity_code SEVERITY_COLUMN, s.key_value, '
        || '       s.severity_guid, s.message, s.message_nlsid, s.message_params,'
        || '       met.metric_label, met.column_label '
        || '  from mgmt_targets tgt, mgmt_metrics met, mgmt_current_severity s '
        || ' where tgt.target_guid = ??EMIP_BIND_TARGET_GUID?? and '
        || '       tgt.target_guid = s.target_guid and '
        || '       s.severity_code in (20, 25) and '
        || '       met.metric_guid = s.metric_guid '
        || '   AND met.type_meta_ver = tgt.type_meta_ver'
        || '   AND (met.category_prop_1 = tgt.category_prop_1 OR met.category_prop_1 = '' '')'
        || '   AND (met.category_prop_2 = tgt.category_prop_2 OR met.category_prop_2 = '' '')'
        || '   AND (met.category_prop_3 = tgt.category_prop_3 OR met.category_prop_3 = '' '')'
        || '   AND (met.category_prop_4 = tgt.category_prop_4 OR met.category_prop_4 = '' '')'
        || '   AND (met.category_prop_5 = tgt.category_prop_5 OR met.category_prop_5 = '' '')'
        || ' order by collection_timestamp desc'
                );

    l_param_values := MGMT_IP_PARAM_VALUE_LIST();
    l_param_values.extend(2);
    l_param_values(1) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.NamedSqlStatement',
      'oracle.sysman.eml.ip.render.elem.table.component.IPMSGSAMP_USER_SELECTS_DATABASE');

    l_param_values(2) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.TableRender.severityColumn',
      'SEVERITY_COLUMN');

    mgmt_ip.add_table_from_sql_elem_def(
          p_element_name_nlsid  => 'IPMSGSAMP_USER_SELECTS_DATABASE',
          p_element_type_nlsid  => 'IPMSG_DATABASE_TARGET_TYPE',
          p_description_nlsid   => 'IPMSGSAMP_SEVERITIES',
          p_internal_use_only   => 0, 
          p_default_parameters  => l_param_values,
          p_query_views         => 0
          );

    l_element_guid := mgmt_ip.add_element_to_report_def (
      p_report_guid         => l_report_guid,
      p_element_name_nlsid  => 'IPMSGSAMP_USER_SELECTS_DATABASE',
      p_element_type_nlsid  => 'IPMSG_DATABASE_TARGET_TYPE',
      p_header_nlsid        => 'IPMSGSAMP_targets_table_header',
      p_element_order       => l_curr_order, 
      p_element_row         => l_curr_row, 
      p_parameters          => null,
      p_targets             => null
        );

    -------------------------------------------------------------------
    -- Database Targets - multiple target substitution
    -------------------------------------------------------------------
    l_report_guid := mgmt_ip.create_report_definition (
      p_title_nlsid                   => 'Database_Targets',
      p_description_nlsid             => 'Database_Targets_Description',
      p_owner                         => 'SYSMAN',
      p_category_nlsid                => 'IPMSGSAMP_SAMPLES', 
      p_sub_category_nlsid            => 'IPMSG_TARGETS',
      p_late_binding_target_types     =>  null,
      p_late_binding_multi_targets    =>  0,
      p_show_table_of_contents        =>  0, 
      p_system_report                 =>  1
        );
      
    l_curr_order := 1;
    l_curr_row := 1;

    -- this element does not get rendered - it is simply to select targets
    -- from the Set Properties option when targets are stored with the report      
    l_element_guid := mgmt_ip.add_element_to_report_def(
      p_report_guid         => l_report_guid,
      p_element_name_nlsid  => 'IPMSG_DB_TARGET_SELECTOR_ELEMENT',
      p_element_type_nlsid  => 'IPMSG_DATABASE_TARGET_TYPE',
      p_header_nlsid        => 'IPMSGSAMP_database_target_selector_desc',
      p_element_order       => l_curr_order, 
      p_element_row         => l_curr_row, 
      p_parameters          => null,
      p_targets             => null
        );
      l_curr_order := l_curr_order + 1;


    mgmt_ip.register_sql_statement (
                p_name  =>'oracle.sysman.eml.ip.render.elem.table.component.IPMSGSAMP_DATABASE_MULTI_TARGETS',
                p_sql_statement => 
      'select target_name REPORTS_FILTER_TARGET_NAME, target_type REPORTS_FILTER_TARGET_TYPE, 
          host_name HOST_NAME 
        from mgmt$target 
        where target_guid in (??EMIP_BIND_COMMA_SEPARATED_TARGET_GUIDS??)'
                );

    l_param_values := MGMT_IP_PARAM_VALUE_LIST();
    l_param_values.extend(2);
    l_param_values(1) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.NamedSqlStatement',
      'oracle.sysman.eml.ip.render.elem.table.component.IPMSGSAMP_DATABASE_MULTI_TARGETS');

    l_param_values(2) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.TableRender.resourceBundle',
      'oracle.sysman.resources.eml.ip.IPMsg');
    mgmt_ip.add_table_from_sql_elem_def(
          p_element_name_nlsid  => 'IPMSGSAMP_DATABASE_MULTI_TARGETS',
          p_element_type_nlsid  => 'IPMSG_DATABASE_TARGET_TYPE',
          p_description_nlsid   => 'IPMSGSAMP_SEVERITIES',
          p_internal_use_only   => 0, 
          p_default_parameters  => l_param_values,
          p_query_views         => 1
          );

    -- targets (stored with report definition)
    l_targets := MGMT_IP_TARGET_LIST();
    l_targets.extend(3);
    l_targets(1) := MGMT_IP_TARGET_RECORD(
      'database',
      'oracle_database');
    l_targets(2) := MGMT_IP_TARGET_RECORD(
      'database2',
      'oracle_database');
    l_targets(3) := MGMT_IP_TARGET_RECORD(
      'database3',
      'oracle_database');

    l_element_guid := mgmt_ip.add_element_to_report_def (
      p_report_guid         => l_report_guid,
      p_element_name_nlsid  => 'IPMSGSAMP_DATABASE_MULTI_TARGETS',
      p_element_type_nlsid  => 'IPMSG_DATABASE_TARGET_TYPE',
      p_header_nlsid        => 'IPMSGSAMP_targets_table_header',
      p_element_order       => l_curr_order, 
      p_element_row         => l_curr_row, 
      p_parameters          => null,
      p_targets             => l_targets
        );

    -------------------------------------------------------------------
    -- Use separate rows example
    -------------------------------------------------------------------

    l_report_guid := mgmt_ip.create_report_definition (
      p_title_nlsid                   => 'IPMSGSAMP_SEPARATE_ROWS',
      p_description_nlsid             => 'IPMSGSAMP_SEPARATE_ROWS_Description',
      p_owner                         => 'SYSMAN',
      p_category_nlsid                => 'IPMSGSAMP_SAMPLES', 
      p_sub_category_nlsid            => 'IPMSG_TARGETS',
      p_late_binding_target_types     =>  null,
      p_late_binding_multi_targets    =>  0,
      p_show_table_of_contents        =>  1, 
      p_system_report                 =>  1
        );


    l_curr_order := 1;
    l_curr_row := 1;

    mgmt_ip.register_sql_statement (
                p_name  =>'oracle.sysman.eml.ip.render.elem.table.component.IPMSGSAMP_USE_SEPARATE_ROWS_TABLE',
                p_sql_statement => 
      'select ''X,Y,Z'' COL0, 5, ''U,V,W'' col2 from dual
       union
       select ''A,B,C'' COL0, 88, ''D,E,F'' col2 from dual'
                );

    l_param_values := MGMT_IP_PARAM_VALUE_LIST();
    l_param_values.extend(1);
    l_param_values(1) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.NamedSqlStatement',
      'oracle.sysman.eml.ip.render.elem.table.component.IPMSGSAMP_USE_SEPARATE_ROWS_TABLE');

    l_param_values.extend(1);
    l_param_values(2) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.TableRender.useSeparateRowsColumns',
      'COL0,COL2');

    mgmt_ip.add_table_from_sql_elem_def(
          p_element_name_nlsid  => 'IPMSGSAMP_USE_SEPARATE_ROWS_TABLE',
          p_element_type_nlsid  => 'IPMSG_ANY_TARGET_TYPE',
          p_description_nlsid   => 'IPMSGSAMP_USE_SEPARATE_ROWS_TABLE_DESC',
          p_internal_use_only   => 0, 
          p_default_parameters  => l_param_values
          );

    l_element_guid := mgmt_ip.add_element_to_report_def (
      p_report_guid         => l_report_guid,
      p_element_name_nlsid  => 'IPMSGSAMP_USE_SEPARATE_ROWS_TABLE',
      p_element_type_nlsid  => 'IPMSG_ANY_TARGET_TYPE',
      p_header_nlsid        => 'IPMSGSAMP_header',
      p_element_order       => l_curr_order, 
      p_element_row         => l_curr_row, 
      p_parameters          => null,
      p_targets             => null
        );

    -------------------------------------------------------------------
    -- Table from pl/sql - user selects database
    -------------------------------------------------------------------

    -- late binding for database target type
    l_target_types := MGMT_IP_TARGET_TYPES();
    l_target_types.extend(1);
    l_target_types(1) := 'oracle_database';

    l_report_guid := mgmt_ip.create_report_definition (
      p_title_nlsid                   => 'User_Chooses a Database Target:  column headers translated',
      p_description_nlsid             => 'Example_of an element where a user chooses a target of type database',
      p_owner                         => 'SYSMAN',
      p_category_nlsid                => 'IPMSGSAMP_SAMPLES', 
      p_sub_category_nlsid            => 'IPMSG_TARGETS',
      p_late_binding_target_types     =>  l_target_types,
      p_late_binding_multi_targets    =>  0,
      p_show_table_of_contents        =>  0, 
      p_system_report                 =>  1
        );
      
    l_curr_order := 1;
    l_curr_row := 1;

    l_param_values := MGMT_IP_PARAM_VALUE_LIST();
    l_param_values.extend(5);

    mgmt_ip.register_sql_statement (
                p_name  =>'oracle.sysman.eml.ip.render.elem.table.component.IPMSGSAMP_USER_SELECTS_DATABASE_COL_TRANS',
                p_sql_statement => 
      'BEGIN 
        DECLARE 
           TYPE CURSOR_TYPE IS REF CURSOR; 
           result_cursor_out CURSOR_TYPE; 
           tgt_guid_in  MGMT_IP_TGT_GUID_ARRAY; 
           start_date_in DATE DEFAULT NULL; 
           end_date_in  DATE DEFAULT NULL; 
           query_string   VARCHAR(600); 
        BEGIN 
            result_cursor_out := ??EMIP_BIND_RESULTS_CURSOR??; 
            tgt_guid_in := ??EMIP_BIND_TARGET_GUID_ARRAY??; 
            start_date_in := ??EMIP_BIND_START_DATE??; 
            end_date_in := ??EMIP_BIND_END_DATE??; 
      
          query_string := ''select tgt.target_name REPORTS_FILTER_TARGET_NAME, 
                            tgt.type_display_name REPORTS_FILTER_TARGET_TYPE, 
                            tgt.host_name HOST_NAME, tgt.load_timestamp 
                            from mgmt_targets tgt 
                            where target_guid = :1
                            order by HOST_NAME desc '';
      
          OPEN result_cursor_out for query_string using tgt_guid_in(1);
      END; 
    END;'
                );

    l_param_values := MGMT_IP_PARAM_VALUE_LIST();
    l_param_values.extend(5);
    l_param_values(1) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.NamedSqlStatement',
      'oracle.sysman.eml.ip.render.elem.table.component.IPMSGSAMP_USER_SELECTS_DATABASE_COL_TRANS');
    l_param_values(2) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.sqlStatementIsPlSql',
      'true');
    l_param_values(3) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.TableRender.resourceBundle',
      'oracle.sysman.resources.eml.ip.IPMsg');
    l_param_values(4) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.TableRender.initialSortColumn',
      'HOST_NAME');
    l_param_values(5) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.TableRender.initialSortOrder',
      'descending');

    mgmt_ip.add_table_from_sql_elem_def(
          p_element_name_nlsid  => 'IPMSGSAMP_USER_SELECTS_DATABASE_COL_TRANS',
          p_element_type_nlsid  => 'IPMSG_DATABASE_TARGET_TYPE',
          p_description_nlsid   => 'IPMSGSAMP_table_with_col_trans',
          p_internal_use_only   => 0, 
          p_default_parameters  => l_param_values,
          p_query_views         => 0
          );

    l_element_guid := mgmt_ip.add_element_to_report_def (
      p_report_guid         => l_report_guid,
      p_element_name_nlsid  => 'IPMSGSAMP_USER_SELECTS_DATABASE_COL_TRANS',
      p_element_type_nlsid  => 'IPMSG_DATABASE_TARGET_TYPE',
      p_header_nlsid        => 'IPMSGSAMP_targets_table_header',
      p_element_order       => l_curr_order, 
      p_element_row         => l_curr_row, 
      p_parameters          => null,
      p_targets             => null
        );
    l_curr_order := l_curr_order + 1;
    l_curr_row := l_curr_row + 1;

    -------------------------------------------------------------------
    -- Chart with single target substitution
    -------------------------------------------------------------------
    -- late binding for any target type
    l_target_types := MGMT_IP_TARGET_TYPES();
    l_target_types.extend(1);
    l_target_types(1) := 'any_type';

    l_report_guid := mgmt_ip.create_report_definition (
      p_title_nlsid                   => 'Chart_Single_Target Binding',
      p_description_nlsid             => 'Chart_Single_Target Binding Description',
      p_owner                         => 'SYSMAN',
      p_category_nlsid                => 'IPMSGSAMP_SAMPLES', 
      p_sub_category_nlsid            => 'IPMSG_TARGETS',
      p_late_binding_target_types     =>  l_target_types,
      p_late_binding_multi_targets    =>  0,
      p_show_table_of_contents        =>  0, 
      p_system_report                 =>  1
        );
      
    l_curr_order := 1;
    l_curr_row := 1;

    mgmt_ip.register_sql_statement (
                p_name  =>'oracle.sysman.eml.ip.render.elem.table.component.IPMSGSAMP_CHART_SINGLE_TARGET',
                p_sql_statement => 
      'select target_type, count(*) from mgmt$target 
        where target_guid=??EMIP_BIND_TARGET_GUID??
        group by target_type '
                );

    l_param_values := MGMT_IP_PARAM_VALUE_LIST();
    l_param_values.extend(3);
    l_param_values(1) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.NamedSqlStatement',
      'oracle.sysman.eml.ip.render.elem.table.component.IPMSGSAMP_CHART_SINGLE_TARGET');
    l_param_values(2) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.ChartParamController.width',
      '200');
    l_param_values(3) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.ChartParamController.height',
      '200');
    mgmt_ip.add_chart_from_sql_elem_def(
          p_element_name_nlsid  => 'IPMSGSAMP_CHART_SINGLE_TARGET',
          p_element_type_nlsid  => 'IPMSG_ANY_TARGET_TYPE',
          p_description_nlsid   => 'IPMSGSAMP_single_target_pie_chart',
          p_internal_use_only   => 0, 
          p_default_parameters  => l_param_values
          );

    l_element_guid := mgmt_ip.add_element_to_report_def (
      p_report_guid         => l_report_guid,
      p_element_name_nlsid  => 'IPMSGSAMP_CHART_SINGLE_TARGET',
      p_element_type_nlsid  => 'IPMSG_ANY_TARGET_TYPE',
      p_header_nlsid        => null,
      p_element_order       => l_curr_order, 
      p_element_row         => l_curr_row, 
      p_parameters          => null,
      p_targets             => null
        );

    
    -------------------------------------------------------------------
    -- Chart with multiple target substitution
    -------------------------------------------------------------------
    l_report_guid := mgmt_ip.create_report_definition (
      p_title_nlsid                   => 'Chart_Multi_Target_Binding',
      p_description_nlsid             => 'Chart_Multi_Target_Binding_Description',
      p_owner                         => 'SYSMAN',
      p_category_nlsid                => 'IPMSGSAMP_SAMPLES', 
      p_sub_category_nlsid            => 'IPMSG_TARGETS',
      p_late_binding_target_types     =>  null,
      p_late_binding_multi_targets    =>  0,
      p_show_table_of_contents        =>  0, 
      p_system_report                 =>  1
        );

    l_curr_order := 1;
    l_curr_row := 1;

   
    mgmt_ip.register_sql_statement (
                p_name  =>'oracle.sysman.eml.ip.render.elem.table.component.IPMSGSAMP_CHART_MULTITARGET_MGMT_PLSQL',
                p_sql_statement => 
      'begin MGMT_IP_SAMPLES.sample_chart_with_plsql(
        ??EMIP_BIND_RESULTS_CURSOR??,
        ??EMIP_BIND_TARGET_GUID_ARRAY??,
        ??EMIP_BIND_START_DATE??,
        ??EMIP_BIND_END_DATE??); end;'
                );

    l_param_values := MGMT_IP_PARAM_VALUE_LIST();
    l_param_values.extend(4);
    l_param_values(1) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.NamedSqlStatement',
      'oracle.sysman.eml.ip.render.elem.table.component.IPMSGSAMP_CHART_MULTITARGET_MGMT_PLSQL');
    l_param_values(2) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.sqlStatementIsPlSql',
      'true');
    l_param_values(3) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.ChartParamController.width',
      '200');
    l_param_values(4) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.ChartParamController.height',
      '200');

    mgmt_ip.add_chart_from_sql_elem_def(
          p_element_name_nlsid  => 'IPMSGSAMP_CHART_MULTITARGET_MGMT_PLSQL',
          p_element_type_nlsid  => 'IPMSG_ANY_TARGET_TYPE',
          p_description_nlsid   => 'IPMSGSAMP_multi_target_pie_chart',
          p_internal_use_only   => 0, 
          p_default_parameters  => l_param_values,
          p_query_views         => 0
          );

    -- host target name different on different systems
    -- targets
    l_targets := MGMT_IP_TARGET_LIST();
    --    l_targets.extend(4);
    l_targets.extend(3);
    l_targets(1) := MGMT_IP_TARGET_RECORD(
      'database',
      'oracle_database');
    l_targets(2) := MGMT_IP_TARGET_RECORD(
      'database2',
      'oracle_database');
    l_targets(3) := MGMT_IP_TARGET_RECORD(
      'database3',
      'oracle_database');

    l_element_guid := mgmt_ip.add_element_to_report_def (
      p_report_guid         => l_report_guid,
      p_element_name_nlsid  => 'IPMSGSAMP_CHART_MULTITARGET_MGMT_PLSQL',
      p_element_type_nlsid  => 'IPMSG_ANY_TARGET_TYPE',
      p_header_nlsid        => null,
      p_element_order       => l_curr_order, 
      p_element_row         => l_curr_row, 
      p_parameters          => null,
      p_targets             => l_targets
        );
    l_curr_order := l_curr_order + 1;
    l_curr_row := l_curr_row + 1;


    -------------------------------------------------------------------
    -- Chart - end user selects a target  of any type
    -------------------------------------------------------------------

    -- late binding for any target type
    l_target_types := MGMT_IP_TARGET_TYPES();
    l_target_types.extend(1);
    l_target_types(1) := 'any_type';

    -- this report has a time period and the user can customize it
    l_param_values := MGMT_IP_PARAM_VALUE_LIST();
    l_param_values.extend(3);
    l_param_values(1) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.TimePeriodOption',
      'true');
    l_param_values(2) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.TimePeriodUserCust',
      'true');
    l_param_values(3) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.TimePeriodParam',
      '0:0');

    l_report_guid := mgmt_ip.create_report_definition (
      p_title_nlsid                   => 'Host Load_Chart',
      p_description_nlsid             => 'TimeseriesBarChart_where_user_selects_a_target',
      p_owner                         => 'SYSMAN',
      p_category_nlsid                => 'IPMSGSAMP_SAMPLES', 
      p_sub_category_nlsid            => 'Load Metrics',
      p_late_binding_target_types     =>  l_target_types,
      p_late_binding_multi_targets    =>  0,
      p_show_table_of_contents        =>  0, 
      p_system_report                 =>  1,
      p_parameters                    =>  l_param_values
        );

    l_curr_order := 1;
    l_curr_row := 1;
      
    mgmt_ip.register_sql_statement (
                p_name  =>'oracle.sysman.eml.ip.render.elem.table.component.IPMSGSAMP_CHART_TIMESERIES_HOST_LOAD',
                p_sql_statement => 
      'SELECT COLUMN_LABEL, ROLLUP_TIMESTAMP, AVERAGE 
        FROM MGMT$METRIC_HOURLY
        WHERE TARGET_GUID = ??EMIP_BIND_TARGET_GUID??
        AND METRIC_LABEL = ''Load''
        AND KEY_VALUE = '' '' 
        AND ROLLUP_TIMESTAMP > ??EMIP_BIND_START_DATE??
        AND ROLLUP_TIMESTAMP < ??EMIP_BIND_END_DATE??
        ORDER BY ROLLUP_TIMESTAMP'
                );

    l_param_values := MGMT_IP_PARAM_VALUE_LIST();
    l_param_values.extend(4);
    l_param_values(1) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.NamedSqlStatement',
      'oracle.sysman.eml.ip.render.elem.table.component.IPMSGSAMP_CHART_TIMESERIES_HOST_LOAD');
    l_param_values(2) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.ChartParamController.width',
      '400');
    l_param_values(3) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.ChartParamController.height',
      '200');
    l_param_values(4) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.ChartParamController.chartType',
      'timeSeriesBarChart');

    mgmt_ip.add_chart_from_sql_elem_def(
          p_element_name_nlsid  => 'IPMSGSAMP_CHART_TIMESERIES_HOST_LOAD',
          p_element_type_nlsid  => 'IPMSG_ANY_TARGET_TYPE',
          p_description_nlsid   => 'IPMSGSAMP_sample load metric graph',
          p_internal_use_only   => 0, 
          p_default_parameters  => l_param_values,
          p_query_views         => 0
          );

    l_element_guid := mgmt_ip.add_element_to_report_def (
      p_report_guid         => l_report_guid,
      p_element_name_nlsid  => 'IPMSGSAMP_CHART_TIMESERIES_HOST_LOAD',
      p_element_type_nlsid  => 'IPMSG_ANY_TARGET_TYPE',
      p_header_nlsid        => null,
      p_element_order       => l_curr_order, 
      p_element_row         => l_curr_row, 
      p_parameters          => null,
      p_targets             => null
        );
    l_curr_order := l_curr_order + 1;
    l_curr_row := l_curr_row + 1;



    -------------------------------------------------------------------
    -- Chart - end user selects a database target 
    -------------------------------------------------------------------

    -- late binding for database target
    l_target_types := MGMT_IP_TARGET_TYPES();
    l_target_types.extend(1);
    l_target_types(1) := 'oracle_database';

    l_report_guid := mgmt_ip.create_report_definition (
      p_title_nlsid                   => 'Database_Chart',
      p_description_nlsid             => 'Timeseries_Chart_where_user_selects_database',
      p_owner                         => 'SYSMAN',
      p_category_nlsid                => 'IPMSGSAMP_SAMPLES', 
      p_sub_category_nlsid            => 'Space Usage',
      p_late_binding_target_types     =>  l_target_types,
      p_late_binding_multi_targets    =>  0,
      p_show_table_of_contents        =>  0, 
      p_system_report                 =>  1
        );

    l_curr_order := 1;
    l_curr_row := 1;
      
    mgmt_ip.register_sql_statement (
                p_name  =>'oracle.sysman.eml.ip.render.elem.table.component.IPMSGSAMP_CHART_TIMESERIES_TABLESPACE_USED',
                p_sql_statement => 
      'BEGIN 
        DECLARE 
           TYPE CURSOR_TYPE IS REF CURSOR; 
           result_cursor_out CURSOR_TYPE; 
           tgt_guid_in  MGMT_IP_TGT_GUID_ARRAY; 
           start_date_in DATE DEFAULT NULL; 
           end_date_in  DATE DEFAULT NULL; 
           query_string   VARCHAR(2000); 
        BEGIN 
            result_cursor_out := ??EMIP_BIND_RESULTS_CURSOR??; 
            tgt_guid_in := ??EMIP_BIND_TARGET_GUID_ARRAY??; 
            start_date_in := ??EMIP_BIND_START_DATE??; 
            end_date_in := ??EMIP_BIND_END_DATE??; 

        query_string := ''SELECT
              d.key_value AS tbsp_name,
              d.rollup_timestamp AS time,
              round(d.value_average,2) AS metric_value
          FROM
              mgmt_metrics_1day d,
              mgmt_metrics m,
              mgmt_targets t,
              (SELECT cs.key_value AS tbsp_name
                 FROM mgmt_current_violation cs, mgmt_targets t, mgmt_metrics m
                WHERE t.target_guid = :1 AND
                      m.target_type=t.target_type AND
                      (m.category_prop_1 = t.category_prop_1 or 
          m.category_prop_1='''' '''') AND
                      (m.metric_name=''''problemTbsp'''' OR 
          m.metric_name=''''problemTbsp10iDct'''') AND
                      m.metric_column=''''pctUsed'''' AND
                      cs.target_guid=t.target_guid AND
                      cs.policy_guid=m.metric_guid) c
          WHERE
              t.target_guid = :1 AND
              m.target_type=t.target_type AND
              (m.metric_name=''''problemTbsp'''' OR m.metric_name=''''problemTbsp10iDct'''') AND
                 m.metric_column=''''pctUsed'''' AND
              (m.category_prop_1 = t.category_prop_1 or m.category_prop_1='''' '''') AND
              m.target_type=t.target_type AND
              d.target_guid=t.target_guid AND
              d.metric_guid=m.metric_guid AND
              d.key_value=c.tbsp_name 
          ORDER BY d.rollup_timestamp'';
              
              
          OPEN result_cursor_out for query_string using tgt_guid_in(1), tgt_guid_in(1);
      END; 
    END;'
    );

    l_param_values := MGMT_IP_PARAM_VALUE_LIST();
    l_param_values.extend(7);
    l_param_values(1) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.NamedSqlStatement',
      'oracle.sysman.eml.ip.render.elem.table.component.IPMSGSAMP_CHART_TIMESERIES_TABLESPACE_USED');
    l_param_values(2) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.sqlStatementIsPlSql',
      'true');
    l_param_values(3) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.ChartParamController.width',
      '400');
    l_param_values(4) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.ChartParamController.height',
      '200');
    l_param_values(5) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.ChartParamController.resourceBundle',
      'oracle.sysman.resources.MntrResourceBundle');
    l_param_values(6) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.ChartParamController.yAxisLabel',
      'PERCENTAGE_LABEL');
    l_param_values(7) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.ChartParamController.chartType',
      'timeSeriesChart');

    mgmt_ip.add_chart_from_sql_elem_def(
          p_element_name_nlsid  => 'IPMSGSAMP_CHART_TIMESERIES_TABLESPACE_USED',
          p_element_type_nlsid  => 'IPMSG_DATABASE_TARGET_TYPE',
          p_description_nlsid   => 'IPMSGSAMP_database_tablespeace_pctUsed_element',
          p_internal_use_only   => 0, 
          p_default_parameters  => l_param_values,
          p_query_views         => 0
          );

    l_element_guid := mgmt_ip.add_element_to_report_def (
      p_report_guid         => l_report_guid,
      p_element_name_nlsid  => 'IPMSGSAMP_CHART_TIMESERIES_TABLESPACE_USED',
      p_element_type_nlsid  => 'IPMSG_DATABASE_TARGET_TYPE',
      p_header_nlsid        => null,
      p_element_order       => l_curr_order, 
      p_element_row         => l_curr_row, 
      p_parameters          => null,
      p_targets             => null
        );
    l_curr_order := l_curr_order + 1;
    l_curr_row := l_curr_row + 1;



    -------------------------------------------------------------------
    -- Cache Hits
    -------------------------------------------------------------------
    
    -- late binding for database target type
    l_target_types := MGMT_IP_TARGET_TYPES();
    l_target_types.extend(1);
    l_target_types(1) := 'oracle_database';

    l_report_guid := mgmt_ip.create_report_definition (
      p_title_nlsid                   => 'IPMSGSAMP_Cache_Hit_Title',
      p_description_nlsid             => 'IPMSGSAMP_Cache_Hit_Description',
      p_owner                         => 'SYSMAN',
      p_category_nlsid                => 'IPMSGSAMP_SAMPLES', 
      p_sub_category_nlsid            => 'IPMSG_EFFICIENCY',
      p_late_binding_target_types     =>  l_target_types,
      p_late_binding_multi_targets    =>  0,
      p_show_table_of_contents        =>  0, 
      p_system_report                 =>  1
        );
      
    l_curr_order := 1;
    l_curr_row := 1;

    -- parameters
    l_param_values := MGMT_IP_PARAM_VALUE_LIST();
    l_param_values.extend(4);
    l_param_values(1) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.MetDetInternalTargetType',
      'oracle_database');
    l_param_values(2) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.MetDetSelectedMetric',
      'instance_efficiency');
    l_param_values(3) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.MetDetSelectedMetricColumn',
      'bufcachehit_pct');
    l_param_values(4) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.MetDetSelectedTimeframe',
      'byWeek');
    l_element_guid := mgmt_ip.add_element_to_report_def (
      p_report_guid         => l_report_guid,
      p_element_name_nlsid  => 'IPMSG_METRIC_DETAILS',
      p_element_type_nlsid  => 'IPMSG_ANY_TARGET_TYPE',
      p_header_nlsid        => 'IPMSGSAMP_Buff_Cache_Hit',
      p_element_order       => l_curr_order, 
      p_element_row         => l_curr_row, 
      p_parameters          => l_param_values,
      p_targets             => null
      );
    l_curr_order := l_curr_order + 1;
    l_curr_row := l_curr_row + 1;


   
    -- parameters
    l_param_values := MGMT_IP_PARAM_VALUE_LIST();
    l_param_values.extend(4);
    l_param_values(1) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.MetDetInternalTargetType',
      'oracle_database');
    l_param_values(2) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.MetDetSelectedMetric',
      'instance_efficiency');
    l_param_values(3) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.MetDetSelectedMetricColumn',
      'libcache_hit_pct');
    l_param_values(4) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.MetDetSelectedTimeframe',
      'byWeek');
    l_element_guid := mgmt_ip.add_element_to_report_def (
      p_report_guid         => l_report_guid,
      p_element_name_nlsid  => 'IPMSG_METRIC_DETAILS',
      p_element_type_nlsid  => 'IPMSG_ANY_TARGET_TYPE',
      p_header_nlsid        => 'IPMSGSAMP_Lib_Cache_Hit',
      p_element_order       => l_curr_order, 
      p_element_row         => l_curr_row, 
      p_parameters          => l_param_values,
      p_targets             => null
        );
    l_curr_order := l_curr_order + 1;
    l_curr_row := l_curr_row + 1;



    -------------------------------------------------------------------
    -- Host Metrics
    -------------------------------------------------------------------

    -- late binding for host target type
    l_target_types := MGMT_IP_TARGET_TYPES();
    l_target_types.extend(1);
    l_target_types(1) := 'host';

    l_report_guid := mgmt_ip.create_report_definition (
      p_title_nlsid                   => 'IPMSGSAMP_Host_Statistics',
      p_description_nlsid             => 'IPMSGSAMP_Key_statistics_for_host_target',
      p_owner                         => 'SYSMAN',
      p_category_nlsid                => 'IPMSGSAMP_SAMPLES', 
      p_sub_category_nlsid            => 'IPMSG_EFFICIENCY',
      p_late_binding_target_types     =>  l_target_types,
      p_late_binding_multi_targets    =>  0,
      p_show_table_of_contents        =>  0, 
      p_system_report                 =>  1
        );

    l_curr_order := 1;
    l_curr_row := 1;
     
    -- parameters
    l_param_values := MGMT_IP_PARAM_VALUE_LIST();
    l_param_values.extend(4);
    l_param_values(1) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.MetDetInternalTargetType',
      'host');
    l_param_values(2) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.MetDetSelectedMetric',
      'Load');
    l_param_values(3) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.MetDetSelectedMetricColumn',
      'cpuUtil');
    l_param_values(4) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.MetDetSelectedTimeframe',
      'byWeek');
    l_element_guid := mgmt_ip.add_element_to_report_def (
      p_report_guid         => l_report_guid,
      p_element_name_nlsid  => 'IPMSG_METRIC_DETAILS',
      p_element_type_nlsid  => 'IPMSG_ANY_TARGET_TYPE',
      p_header_nlsid        => 'IPMSGSAMP_Buff_Cache_Hit',
      p_element_order       => l_curr_order, 
      p_element_row         => l_curr_row, 
      p_parameters          => l_param_values,
      p_targets             => null
        );
    l_curr_order := l_curr_order + 1;
    l_curr_row := l_curr_row + 1;

    
    -- parameters
    l_param_values := MGMT_IP_PARAM_VALUE_LIST();
    l_param_values.extend(4);
    l_param_values(1) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.MetDetInternalTargetType',
      'oracle_database');
    l_param_values(2) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.MetDetSelectedMetric',
      'Load');
    l_param_values(3) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.MetDetSelectedMetricColumn',
      'swapUtil');
    l_param_values(4) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.MetDetSelectedTimeframe',
      'byWeek');
    l_element_guid := mgmt_ip.add_element_to_report_def (
      p_report_guid         => l_report_guid,
      p_element_name_nlsid  => 'IPMSG_METRIC_DETAILS',
      p_element_type_nlsid  => 'IPMSG_ANY_TARGET_TYPE',
      p_header_nlsid        => 'IPMSGSAMP_Lib_Cache_Hit',
      p_element_order       => l_curr_order, 
      p_element_row         => l_curr_row, 
      p_parameters          => l_param_values,
      p_targets             => null
        );
    l_curr_order := l_curr_order + 1;
    l_curr_row := l_curr_row + 1;


    -------------------------------------------------------------------
    -- Blackouts next week
    -------------------------------------------------------------------
    l_report_guid := mgmt_ip.create_report_definition (
      p_title_nlsid                   => 'IPMSGSAMP_Blackout_SCHEDULE',
      p_description_nlsid             => 'IPMSGSAMP_Blackout_SCHEDULE_Description',
      p_owner                         => 'SYSMAN',
      p_category_nlsid                => 'IPMSGSAMP_SAMPLES', 
      p_sub_category_nlsid            => 'IPMSG_BLACKOUTS',
      p_late_binding_target_types     =>  NULL,
      p_late_binding_multi_targets    =>  0,
      p_show_table_of_contents        =>  0, 
      p_system_report                 =>  1
        );

    l_curr_order := 1;
    l_curr_row := 1;

    mgmt_ip.register_sql_statement (
                p_name  =>'oracle.sysman.eml.ip.render.elem.table.component.IPMSGSAMP_Blackout_SCHEDULE',
                p_sql_statement => 
      'select ta.target_name, sc.start_time, bl.blackout_name, re.reason 
      from mgmt_blackouts bl, mgmt_blackout_reason re, 
        mgmt_blackout_schedule sc, mgmt_blackout_target_details td, mgmt_targets ta 
      where re.reason_id = bl.reason_id and bl.blackout_guid = sc.blackout_guid 
        and bl.blackout_guid = td.blackout_guid and ta.target_guid = td.target_guid 
        and sc.start_time > ??EMIP_BIND_START_DATE?? 
        and sc.start_time < ??EMIP_BIND_END_DATE?? 
      order by sc.start_time'      );

    l_param_values := MGMT_IP_PARAM_VALUE_LIST();
    l_param_values.extend(1);
    l_param_values(1) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.NamedSqlStatement',
      'oracle.sysman.eml.ip.render.elem.table.component.IPMSGSAMP_Blackout_SCHEDULE');

    mgmt_ip.add_table_from_sql_elem_def(
          p_element_name_nlsid  => 'IPMSGSAMP_Blackout_SCHEDULE',
          p_element_type_nlsid  => 'IPMSG_ANY_TARGET_TYPE',
          p_description_nlsid   => 'IPMSGSAMP_Blackout_SCHEDULE_Description',
          p_internal_use_only   => 0, 
          p_default_parameters  => l_param_values,
          p_query_views         => 0
          );
    
    l_element_guid := mgmt_ip.add_element_to_report_def (
      p_report_guid         => l_report_guid,
      p_element_name_nlsid  => 'IPMSGSAMP_Blackout_SCHEDULE',
      p_element_type_nlsid  => 'IPMSG_ANY_TARGET_TYPE',
      p_header_nlsid        => 'IPMSGSAMP_Blackout_SCHEDULE',
      p_element_order       => l_curr_order, 
      p_element_row         => l_curr_row, 
      p_parameters          => null,
      p_targets             => null
        );
    l_curr_order := l_curr_order + 1;
    l_curr_row := l_curr_row + 1;

    -------------------------------------------------------------------
    -- Alerts cleared in last week
    -------------------------------------------------------------------
    l_report_guid := mgmt_ip.create_report_definition (
      p_title_nlsid                   => 'Alerts_Cleared_Over_Time_Period',
      p_description_nlsid             => 'Alerts_Cleared_Over_Time_Period_Description',
      p_owner                         => 'SYSMAN',
      p_category_nlsid                => 'IPMSGSAMP_SAMPLES', 
      p_sub_category_nlsid            => 'IPMSG_ALERTS',
      p_late_binding_target_types     =>  NULL,
      p_late_binding_multi_targets    =>  0,
      p_show_table_of_contents        =>  0, 
      p_system_report                 =>  1
        );

    l_curr_order := 1;
    l_curr_row := 1;

    mgmt_ip.register_sql_statement (
                p_name  =>'oracle.sysman.eml.ip.render.elem.table.component.Alerts_Cleared_Over_Time_Period_Element',
                p_sql_statement => 
      'select tar.target_name, tar.type_display_name, 
        met.metric_name||'':''||met.metric_column, sev.message, 
          sev.load_timestamp, sev.severity_duration 
          from mgmt_severity sev, mgmt_targets tar, mgmt_metrics met 
          where sev.severity_code = ''15'' 
          and met.metric_guid = sev.metric_guid 
          and tar.target_guid = sev.target_guid 
          and sev.load_timestamp > ??EMIP_BIND_START_DATE?? 
          and sev.load_timestamp < ??EMIP_BIND_END_DATE??'
          );

    l_param_values := MGMT_IP_PARAM_VALUE_LIST();
    l_param_values.extend(1);
    l_param_values(1) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.NamedSqlStatement',
      'oracle.sysman.eml.ip.render.elem.table.component.Alerts_Cleared_Over_Time_Period_Element');

    mgmt_ip.add_table_from_sql_elem_def(
          p_element_name_nlsid  => 'Alerts_Cleared_Over_Time_Period_Element',
          p_element_type_nlsid  => 'IPMSG_ANY_TARGET_TYPE',
          p_description_nlsid   => 'Alerts_Cleared_Over_Time_Period_Element_Description',
          p_internal_use_only   => 0, 
          p_default_parameters  => l_param_values,
          p_query_views         => 0
          );
    
    l_element_guid := mgmt_ip.add_element_to_report_def (
      p_report_guid         => l_report_guid,
      p_element_name_nlsid  => 'Alerts_Cleared_Over_Time_Period_Element',
      p_element_type_nlsid  => 'IPMSG_ANY_TARGET_TYPE',
      p_header_nlsid        => 'Alerts_Cleared',
      p_element_order       => l_curr_order, 
      p_element_row         => l_curr_row, 
      p_parameters          => null,
      p_targets             => null
        );
    l_curr_order := l_curr_order + 1;
    l_curr_row := l_curr_row + 1;


    -------------------------------------------------------------------
    -- Filesystem Image Report
    -------------------------------------------------------------------
    l_report_guid := mgmt_ip.create_report_definition (
      p_title_nlsid                   => 'IPMSGSAMP_Filesystem_image_Title',
      p_description_nlsid             => 'IPMSGSAMP_Filesystem_image_Description',
      p_owner                         => 'SYSMAN',
      p_category_nlsid                => 'IPMSGSAMP_SAMPLES', 
      p_sub_category_nlsid            => '_FileSystemImages',
      p_late_binding_target_types     =>  NULL,
      p_late_binding_multi_targets    =>  0,
      p_show_table_of_contents        =>  0, 
      p_system_report                 =>  1
        );

    l_curr_order := 1;
    l_curr_row := 1;

    l_param_values := MGMT_IP_PARAM_VALUE_LIST();
    l_param_values.extend(2);
    l_param_values(1) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.image.filesystem_href',
      '/em/images/oracle_logo_red.gif');
    l_param_values(2) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.image.href',
      'http://www.oracle.com');
    mgmt_ip.add_em_image_elem_def(
          p_element_name_nlsid  => 'IPMSGSAMP_Oracle_Logo_Element',
          p_element_type_nlsid  => 'IPMSG_NO_TARGET_TYPE',
          p_description_nlsid   => 'IPMSGSAMP_Oracle_Logo_Element_Description',
          p_internal_use_only   => 0, 
          p_default_parameters  => l_param_values
          );

    l_element_guid := mgmt_ip.add_element_to_report_def (
      p_report_guid         => l_report_guid,
      p_element_name_nlsid  => 'IPMSGSAMP_Oracle_Logo_Element',
      p_element_type_nlsid  => 'IPMSG_NO_TARGET_TYPE',
      p_header_nlsid        => null,
      p_element_order       => l_curr_order, 
      p_element_row         => l_curr_row, 
      p_parameters          => null,
      p_targets             => null
        );
    l_curr_order := l_curr_order + 1;
    l_curr_row := l_curr_row + 1;

      
    -------------------------------------------------------------------
    -- Typical Text Elements
    -------------------------------------------------------------------
    l_report_guid := mgmt_ip.create_report_definition (
      p_title_nlsid                   => 'IPMSGSAMP_Typical_Text_element_examples',
      p_description_nlsid             => 'IPMSGSAMP_Contains Text elements with differing messages, style, and resource bundles -- all report level parameters are default',
      p_owner                         => 'SYSMAN',
      p_category_nlsid                => 'IPMSGSAMP_SAMPLES', 
      p_sub_category_nlsid            => '_Text',
      p_late_binding_target_types     =>  NULL,
      p_late_binding_multi_targets    =>  0,
      p_show_table_of_contents        =>  0, 
      p_system_report                 =>  1
        );

    l_curr_order := 1;
    l_curr_row := 1;

    l_element_guid := mgmt_ip.add_element_to_report_def (
      p_report_guid         => l_report_guid,
      p_element_name_nlsid  => 'IPMSG_STYLED_TEXT',
      p_element_type_nlsid  => 'IPMSG_ANY_TARGET_TYPE',
      p_header_nlsid        => 'IPMSGSAMP_Empty Message (No parameters)',
      p_element_order       => l_curr_order, 
      p_element_row         => l_curr_row, 
      p_parameters          => null,
      p_targets             => null
        );
    l_curr_order := l_curr_order + 1;
    l_curr_row := l_curr_row + 1;

    l_param_values := MGMT_IP_PARAM_VALUE_LIST();
    l_param_values.extend(1);
    l_param_values(1) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.TextParamBean.textMessage',
      'This is what a message in the text element might look like.');
    l_element_guid := mgmt_ip.add_element_to_report_def (
      p_report_guid         => l_report_guid,
      p_element_name_nlsid  => 'IPMSG_STYLED_TEXT',
      p_element_type_nlsid  => 'IPMSG_ANY_TARGET_TYPE',
      p_header_nlsid        => 'IPMSGSAMP_Default Style Message',
      p_element_order       => l_curr_order, 
      p_element_row         => l_curr_row, 
      p_parameters          => l_param_values,
      p_targets             => null
        );
    l_curr_order := l_curr_order + 1;
    l_curr_row := l_curr_row + 1;

    l_param_values := MGMT_IP_PARAM_VALUE_LIST();
    l_param_values.extend(2);
    l_param_values(1) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.TextParamBean.textMessage',
      'Tip-like element');
    l_param_values(2) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.TextParamBean.textStyle',
      'OraTipText');
    l_element_guid := mgmt_ip.add_element_to_report_def (
      p_report_guid         => l_report_guid,
      p_element_name_nlsid  => 'IPMSG_STYLED_TEXT',
      p_element_type_nlsid  => 'IPMSG_ANY_TARGET_TYPE',
      p_header_nlsid        => 'IPMSGSAMP_Tip-style Message',
      p_element_order       => l_curr_order, 
      p_element_row         => l_curr_row, 
      p_parameters          => l_param_values,
      p_targets             => null
        );
    l_curr_order := l_curr_order + 1;
    l_curr_row := l_curr_row + 1;

    l_param_values := MGMT_IP_PARAM_VALUE_LIST();
    l_param_values.extend(2);
    l_param_values(1) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.TextParamBean.textMessage',
      'I''m a tip-like message with an element header');
    l_param_values(2) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.TextParamBean.textStyle',
      'OraTipText');
    l_element_guid := mgmt_ip.add_element_to_report_def (
      p_report_guid         => l_report_guid,
      p_element_name_nlsid  => 'IPMSG_STYLED_TEXT',
      p_element_type_nlsid  => 'IPMSG_ANY_TARGET_TYPE',
      p_header_nlsid        => 'IPMSGSAMP_2 mesages: tip-like (with header) followed by instruction-like without header',
      p_element_order       => l_curr_order, 
      p_element_row         => l_curr_row, 
      p_parameters          => l_param_values,
      p_targets             => null
        );
    l_curr_order := l_curr_order + 1;
    l_curr_row := l_curr_row + 1;

    l_param_values := MGMT_IP_PARAM_VALUE_LIST();
    l_param_values.extend(2);
    l_param_values(1) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.TextParamBean.textMessage',
      'I''m a separate element without an element header');
    l_param_values(2) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.TextParamBean.textStyle',
      'OraInstructionText');
    l_element_guid := mgmt_ip.add_element_to_report_def (
      p_report_guid         => l_report_guid,
      p_element_name_nlsid  => 'IPMSG_STYLED_TEXT',
      p_element_type_nlsid  => 'IPMSG_ANY_TARGET_TYPE',
      p_header_nlsid        => null,
      p_element_order       => l_curr_order, 
      p_element_row         => l_curr_row, 
      p_parameters          => l_param_values,
      p_targets             => null
        );
    l_curr_order := l_curr_order + 1;
    l_curr_row := l_curr_row + 1;


    l_param_values := MGMT_IP_PARAM_VALUE_LIST();
    l_param_values.extend(3);
    l_param_values(1) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.TextParamBean.textMessage',
      'REPORTS_FILTER_TIP');
    l_param_values(2) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.TextParamBean.textStyle',
      'OraInstructionText');
    l_param_values(3) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.TextParamBean.textResourceBundleName',
      'oracle.sysman.resources.eml.ip.IPMsg');
    l_element_guid := mgmt_ip.add_element_to_report_def (
      p_report_guid         => l_report_guid,
      p_element_name_nlsid  => 'IPMSG_STYLED_TEXT',
      p_element_type_nlsid  => 'IPMSG_ANY_TARGET_TYPE',
      p_header_nlsid        => 'IPMSGSAMP_Message from IPMsg bundle',
      p_element_order       => l_curr_order, 
      p_element_row         => l_curr_row, 
      p_parameters          => l_param_values,
      p_targets             => null
        );
    l_curr_order := l_curr_order + 1;
    l_curr_row := l_curr_row + 1;

      
    -------------------------------------------------------------------
    -- Style and Header Test 1
    -------------------------------------------------------------------
    l_report_guid := mgmt_ip.create_report_definition (
      p_title_nlsid                   => 'Style and Header Test 1 (BLAF, headers)',
      p_description_nlsid             => 'BLAF and show headers are specified',
      p_owner                         => 'SYSMAN',
      p_category_nlsid                => 'IPMSGSAMP_SAMPLES', 
      p_sub_category_nlsid            => '_Style',
      p_late_binding_target_types     =>  NULL,
      p_late_binding_multi_targets    =>  0,
      p_show_table_of_contents        =>  0, 
      p_system_report                 =>  1,
      p_style                         => 'BLAF',
      p_show_navigation               => 1
        );

    l_curr_order := 1;
    l_curr_row := 1;

    l_param_values := MGMT_IP_PARAM_VALUE_LIST();
    l_param_values.extend(1);
    l_param_values(1) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.TextParamBean.textMessage',
      'Foo');
    l_element_guid := mgmt_ip.add_element_to_report_def (
      p_report_guid         => l_report_guid,
      p_element_name_nlsid  => 'IPMSG_STYLED_TEXT',
      p_element_type_nlsid  => 'IPMSG_ANY_TARGET_TYPE',
      p_header_nlsid        => null,
      p_element_order       => l_curr_order, 
      p_element_row         => l_curr_row, 
      p_parameters          => l_param_values,
      p_targets             => null
        );
    l_curr_order := l_curr_order + 1;
    l_curr_row := l_curr_row + 1;
      
    -------------------------------------------------------------------
    -- Style and Header Test 2
    -------------------------------------------------------------------
    l_report_guid := mgmt_ip.create_report_definition (
      p_title_nlsid                   => 'IPMSGSAMP_Style and Header Test 2 (DASHBOARD, headers)',
      p_description_nlsid             => 'IPMSGSAMP_DASHBOARD and show headers are specified',
      p_owner                         => 'SYSMAN',
      p_category_nlsid                => 'IPMSGSAMP_SAMPLES', 
      p_sub_category_nlsid            => '_Style',
      p_late_binding_target_types     =>  NULL,
      p_late_binding_multi_targets    =>  0,
      p_show_table_of_contents        =>  0, 
      p_system_report                 =>  1,
      p_style                         => 'DASHBOARD',
      p_show_navigation               => 1
        );

    l_curr_order := 1;
    l_curr_row := 1;

    l_param_values := MGMT_IP_PARAM_VALUE_LIST();
    l_param_values.extend(1);
    l_param_values(1) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.TextParamBean.textMessage',
      'Foo');
    l_element_guid := mgmt_ip.add_element_to_report_def (
      p_report_guid         => l_report_guid,
      p_element_name_nlsid  => 'IPMSG_STYLED_TEXT',
      p_element_type_nlsid  => 'IPMSG_ANY_TARGET_TYPE',
      p_header_nlsid        => null,
      p_element_order       => l_curr_order, 
      p_element_row         => l_curr_row, 
      p_parameters          => l_param_values,
      p_targets             => null
        );
    l_curr_order := l_curr_order + 1;
    l_curr_row := l_curr_row + 1;
      
    -------------------------------------------------------------------
    -- Style and Header Test 3
    -------------------------------------------------------------------
    l_report_guid := mgmt_ip.create_report_definition (
      p_title_nlsid                   => 'IPMSGSAMP_Style and Header Test 3 (BLAF, no headers)',
      p_description_nlsid             => 'IPMSGSAMP_BLAF and show no headers are specified',
      p_owner                         => 'SYSMAN',
      p_category_nlsid                => 'IPMSGSAMP_SAMPLES', 
      p_sub_category_nlsid            => '_Style',
      p_late_binding_target_types     =>  NULL,
      p_late_binding_multi_targets    =>  0,
      p_show_table_of_contents        =>  0, 
      p_system_report                 =>  1,
      p_style                         =>  'BLAF',
      p_show_navigation               =>  0
        );

    l_curr_order := 1;
    l_curr_row := 1;

    l_param_values := MGMT_IP_PARAM_VALUE_LIST();
    l_param_values.extend(1);
    l_param_values(1) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.TextParamBean.textMessage',
      'Foo');
    l_element_guid := mgmt_ip.add_element_to_report_def (
      p_report_guid         => l_report_guid,
      p_element_name_nlsid  => 'IPMSG_STYLED_TEXT',
      p_element_type_nlsid  => 'IPMSG_ANY_TARGET_TYPE',
      p_header_nlsid        => null,
      p_element_order       => l_curr_order, 
      p_element_row         => l_curr_row, 
      p_parameters          => l_param_values,
      p_targets             => null
        );
    l_curr_order := l_curr_order + 1;
    l_curr_row := l_curr_row + 1;
      
    -------------------------------------------------------------------
    -- Style and Header Test 4
    -------------------------------------------------------------------
    l_report_guid := mgmt_ip.create_report_definition (
      p_title_nlsid                   => 'IPMSGSAMP_Style and Header Test 4 (DASHBOARD, no headers)',
      p_description_nlsid             => 'IPMSGSAMP_DASHBOARD and show no headers are specified',
      p_owner                         => 'SYSMAN',
      p_category_nlsid                => 'IPMSGSAMP_SAMPLES', 
      p_sub_category_nlsid            => '_Style',
      p_late_binding_target_types     =>  NULL,
      p_late_binding_multi_targets    =>  0,
      p_show_table_of_contents        =>  0, 
      p_system_report                 =>  1,
      p_style                         =>  'DASHBOARD',
      p_show_navigation               =>  0
        );


    l_curr_order := 1;
    l_curr_row := 1;

    l_param_values := MGMT_IP_PARAM_VALUE_LIST();
    l_param_values.extend(1);
    l_param_values(1) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.TextParamBean.textMessage',
      'Foo');
    l_element_guid := mgmt_ip.add_element_to_report_def (
      p_report_guid         => l_report_guid,
      p_element_name_nlsid  => 'IPMSG_STYLED_TEXT',
      p_element_type_nlsid  => 'IPMSG_ANY_TARGET_TYPE',
      p_header_nlsid        => null,
      p_element_order       => l_curr_order, 
      p_element_row         => l_curr_row, 
      p_parameters          => l_param_values,
      p_targets             => null
        );
    l_curr_order := l_curr_order + 1;
    l_curr_row := l_curr_row + 1;



  -- Example of Built-in Element with Default Parameters:  select from dual
  --
    mgmt_ip.register_sql_statement (
                p_name  =>'oracle.sysman.eml.ip.render.elem.table.component.IPMSGSAMP_SELECT_FROM_DUAL',
                p_sql_statement => 
      'select * from dual'
          );

    l_param_values := MGMT_IP_PARAM_VALUE_LIST();
    l_param_values.extend(1);
    l_param_values(1) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.NamedSqlStatement',
      'oracle.sysman.eml.ip.render.elem.table.component.IPMSGSAMP_SELECT_FROM_DUAL');

    mgmt_ip.add_element_definition(
      p_element_name_nlsid  => 'IPMSGSAMP_SELECT_FROM_DUAL',
      p_element_type_nlsid  => 'IPMSG_NO_TARGET_TYPE',
      p_description_nlsid   => 'IPMSGSAMP_example_ootb_element_with_default_params',
      p_element_class_name  => 'oracle.sysman.eml.ip.render.elem.TableRenderController',
      p_internal_use_only   => 0, 
      p_param_classes       => null,
      p_default_parameters  => l_param_values,
      p_pack_name           => null
      );

    -------------------------------------------------------------------
    -- Example of user editable SQL aginst base tables - INTERNAL USE ONLY
    --
    --  This is used to test queries against base tables.  If you're
    --  querying against views, as is recommended, you can use the 
    --  normal 'user_chart_from_sql' element
    -------------------------------------------------------------------
    -- this element is for interactive testing purposes only.  integrators
    -- can use this to test sql statements.  once you are done testing you
    -- should use the pl/sql api (as is done in this file) to create report
    -- definitions/elements and use the 'chart_from_sql' element
    --
    l_param_classes := MGMT_IP_PARAM_CLASSES();
    l_param_classes.extend(2);
    l_param_classes(1) := 'oracle.sysman.eml.ip.render.elem.HeaderParamController';
    l_param_classes(2) := 'oracle.sysman.eml.ip.render.elem.ChartParamController';
    mgmt_ip.add_element_definition(
      p_element_name_nlsid  => 'INTERNALTEST_chart_from_sql',
      p_element_type_nlsid  => 'IPMSG_ANY_TARGET_TYPE',
      p_description_nlsid   => 'CAUTION:  this element can be used to test queryies against base tables -- but we should be using VIEWS!',
      p_element_class_name  => 'oracle.sysman.eml.ip.render.elem.ChartRenderController',
      p_internal_use_only   => 0, 
      p_param_classes       => l_param_classes,
      p_default_parameters  => null,
      p_pack_name           => null
      );  
    -------------------------------------------------------------------
    -- Example of user editable SQL aginst base tables - INTERNAL USE ONLY
    --
    --  This is used to test queries against base tables.  If you're
    --  querying against views, as is recommended, you can use the 
    --  normal 'user_table_from_sql' element
    -------------------------------------------------------------------
    -- this element is for interactive testing purposes only.  integrators
    -- can use this to test sql statements.  once you are done testing you
    -- should use the pl/sql api (as is done in this file) to create report
    -- definitions/elements and use the 'table_from_sql' element
    --
    l_param_classes := MGMT_IP_PARAM_CLASSES();
    l_param_classes.extend(2);
    l_param_classes(1) := 'oracle.sysman.eml.ip.render.elem.HeaderParamController';
    l_param_classes(2) := 'oracle.sysman.eml.ip.render.elem.SQLStatementParamController';
    mgmt_ip.add_element_definition(
      p_element_name_nlsid  => 'INTERNALTEST_table_from_sql',
      p_element_type_nlsid  => 'IPMSG_ANY_TARGET_TYPE',
      p_description_nlsid   => 'CAUTION:  this element can be used to test queryies against base tables -- but we should be using VIEWS!',
      p_element_class_name  => 'oracle.sysman.eml.ip.render.elem.TableRenderController',
      p_internal_use_only   => 0, 
      p_param_classes       => l_param_classes,
      p_default_parameters  => null,
      p_pack_name           => null
      );


    -------------------------------------------------------------------
    -- Example of table that renders as multiple tables based
    -- on a column that changes values.  This is the split column.
    -------------------------------------------------------------------
    
    l_report_guid := mgmt_ip.create_report_definition (
      p_title_nlsid                   => 'SplitColumnBreaksTableIntoSections',
      p_description_nlsid             => 'IPMSGSAMP_multi tables from one query',
      p_owner                         => 'SYSMAN',
      p_category_nlsid                => 'IPMSGSAMP_SAMPLES', 
      p_sub_category_nlsid            => 'IPMSG_TARGETS',
      p_late_binding_target_types     =>  null,
      p_late_binding_multi_targets    =>  0,
      p_show_table_of_contents        =>  1, 
      p_system_report                 =>  1
        );


    l_curr_order := 1;
    l_curr_row := 1;
    
    mgmt_ip.register_sql_statement (
                p_name  =>'oracle.sysman.eml.ip.render.elem.table.component.SPLIT_COLUMN_TARGETS_TABLE',
                p_sql_statement => 
      'select display_name, target_type TARGET_TYPE from mgmt$target order by target_type'
          );

    l_param_values := MGMT_IP_PARAM_VALUE_LIST();
    l_param_values.extend(1);
    l_param_values(1) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.NamedSqlStatement',
      'oracle.sysman.eml.ip.render.elem.table.component.SPLIT_COLUMN_TARGETS_TABLE');
    l_param_values.extend(1);
    l_param_values(2) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.TableRender.tableSplitColumn',
      'TARGET_TYPE');
    l_param_values.extend(1);
    l_param_values(3) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.TableRender.tableHeaderMsgID',
      'SET_TARGET_REPORT_INSTRUCTION');
    l_param_values.extend(1);
    l_param_values(4) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.TableRender.resourceBundle',
      'oracle.sysman.resources.eml.ip.IPMsg');

    mgmt_ip.add_table_from_sql_elem_def(
          p_element_name_nlsid  => 'SPLIT_COLUMN_TARGETS_TABLE',
          p_element_type_nlsid  => 'IPMSG_ANY_TARGET_TYPE',
          p_description_nlsid   => 'IPMSGSAMP_all_target_table',
          p_internal_use_only   => 0, 
          p_default_parameters  => l_param_values
          );

    l_element_guid := mgmt_ip.add_element_to_report_def (
      p_report_guid         => l_report_guid,
      p_element_name_nlsid  => 'SPLIT_COLUMN_TARGETS_TABLE',
      p_element_type_nlsid  => 'IPMSG_ANY_TARGET_TYPE',
      p_header_nlsid        => 'IPMSGSAMP_targets_table_header',
      p_element_order       => l_curr_order, 
      p_element_row         => l_curr_row, 
      p_parameters          => null,
      p_targets             => null
        );
  


    -------------------------------------------------------------------
    -- Echo bind variables
    -------------------------------------------------------------------

    -- late binding for any target type
    l_target_types := MGMT_IP_TARGET_TYPES();
    l_target_types.extend(1);
    l_target_types(1) := 'any_type';

    -- this report has a time period and the user can customize it
    l_param_values := MGMT_IP_PARAM_VALUE_LIST();
    l_param_values.extend(3);
    l_param_values(1) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.TimePeriodOption',
      'true');
    l_param_values(2) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.TimePeriodUserCust',
      'true');
    l_param_values(3) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.TimePeriodParam',
      '0:0');

    l_report_guid := mgmt_ip.create_report_definition (
      p_title_nlsid                   => 'EchoBindVariables',
      p_description_nlsid             => 'echo bind variables from query',
      p_owner                         => 'SYSMAN',
      p_category_nlsid                => 'IPMSGSAMP_SAMPLES', 
      p_sub_category_nlsid            => 'IPMSG_TARGETS',
      p_late_binding_target_types     =>  l_target_types,
      p_late_binding_multi_targets    =>  0,
      p_show_table_of_contents        =>  1, 
      p_system_report                 =>  0,
      p_parameters                    =>  l_param_values
        );


    l_curr_order := 1;
    l_curr_row := 1;
    
    mgmt_ip.register_sql_statement (
                p_name  =>'oracle.sysman.eml.ip.render.elem.table.component.ECHO_BINDS_TABLE',
                p_sql_statement => 
      'select ??EMIP_BIND_TARGET_GUID??           TARGET_GUID,
      ??EMIP_BIND_COMMA_SEPARATED_TARGET_GUIDS??  COMMA_SEPARATED_TARGET_GUIDS,
      ??EMIP_BIND_START_DATE??                    START_DATE,
      ??EMIP_BIND_END_DATE??                      END_DATE,
      ??EMIP_BIND_TIMEZONE_REGION??               TIMEZONE_REGION,
      ??EMIP_BIND_LOCALE_COUNTRY??                LOCALE_COUNTRY,
      ??EMIP_BIND_LOCALE_LANGUAGE??               LOCALE_LANGUAGE
      from dual'
          );

    l_param_values := MGMT_IP_PARAM_VALUE_LIST();
    l_param_values.extend(1);
    l_param_values(1) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.NamedSqlStatement',
      'oracle.sysman.eml.ip.render.elem.table.component.ECHO_BINDS_TABLE');

    mgmt_ip.add_table_from_sql_elem_def(
          p_element_name_nlsid  => 'ECHO_BINDS_TABLE',
          p_element_type_nlsid  => 'IPMSG_ANY_TARGET_TYPE',
          p_description_nlsid   => 'Echo bind variables',
          p_internal_use_only   => 0, 
          p_default_parameters  => l_param_values
          );

    l_element_guid := mgmt_ip.add_element_to_report_def (
      p_report_guid         => l_report_guid,
      p_element_name_nlsid  => 'ECHO_BINDS_TABLE',
      p_element_type_nlsid  => 'IPMSG_ANY_TARGET_TYPE',
      p_header_nlsid        => 'Echo bind variables',
      p_element_order       => l_curr_order, 
      p_element_row         => l_curr_row, 
      p_parameters          => null,
      p_targets             => null
        );


    -------------------------------------------------------------------
    -- Example of table that has column Group headers.
    -------------------------------------------------------------------

    l_report_guid := mgmt_ip.create_report_definition (
      p_title_nlsid                   => 'ColumnGroupHeaders_All_Targets_Title',
      p_description_nlsid             => 'IPMSGSAMP_column headers are grouped',
      p_owner                         => 'SYSMAN',
      p_category_nlsid                => 'IPMSGSAMP_SAMPLES', 
      p_sub_category_nlsid            => 'IPMSG_TARGETS',
      p_late_binding_target_types     =>  null,
      p_late_binding_multi_targets    =>  0,
      p_show_table_of_contents        =>  1, 
      p_system_report                 =>  1
        );


    l_curr_order := 1;
    l_curr_row := 1;
    
    mgmt_ip.register_sql_statement (
                p_name  =>'oracle.sysman.eml.ip.render.elem.table.component.COLGROUP_ALL_TARGETS_TABLE',
                p_sql_statement => 
      'select target_name, display_name, target_type, host_name, timezone_region, last_metric_load_time from mgmt$target order by target_type'
          );

    l_param_values := MGMT_IP_PARAM_VALUE_LIST();
    l_param_values.extend(1);
    l_param_values(1) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.NamedSqlStatement',
      'oracle.sysman.eml.ip.render.elem.table.component.COLGROUP_ALL_TARGETS_TABLE');
    l_param_values.extend(1);
    l_param_values(2) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.TableRender.columnGroupHeader1',
      'Header 1');
    l_param_values.extend(1);
    l_param_values(3) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.TableRender.columnGroupStartCol1',
      'TARGET_NAME');
    l_param_values.extend(1);
    l_param_values(4) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.TableRender.columnGroupEndCol1',
      'DISPLAY_NAME');

    l_param_values.extend(1);
    l_param_values(5) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.TableRender.columnGroupHeader2',
      'Header 2');
    l_param_values.extend(1);
    l_param_values(6) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.TableRender.columnGroupStartCol2',
      'TIMEZONE_REGION');
    l_param_values.extend(1);
    l_param_values(7) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.TableRender.columnGroupEndCol2',
      'LAST_METRIC_LOAD_TIME');

    mgmt_ip.add_table_from_sql_elem_def(
          p_element_name_nlsid  => 'COLGROUP_ALL_TARGETS_TABLE',
          p_element_type_nlsid  => 'IPMSG_ANY_TARGET_TYPE',
          p_description_nlsid   => 'IPMSGSAMP_all_target_table',
          p_internal_use_only   => 0, 
          p_default_parameters  => l_param_values
          );

    l_element_guid := mgmt_ip.add_element_to_report_def (
      p_report_guid         => l_report_guid,
      p_element_name_nlsid  => 'COLGROUP_ALL_TARGETS_TABLE',
      p_element_type_nlsid  => 'IPMSG_ANY_TARGET_TYPE',
      p_header_nlsid        => 'IPMSGSAMP_targets_table_header',
      p_element_order       => l_curr_order, 
      p_element_row         => l_curr_row, 
      p_parameters          => null,
      p_targets             => null
        );


    -------------------------------------------------------------------
    -- Example of table that has translated column values and will
    -- show a blank cell if a severity value is null or invalid
    -------------------------------------------------------------------

    l_report_guid := mgmt_ip.create_report_definition (
      p_title_nlsid                   => 'IPMSGSAMP_TranslateColumnValue',
      p_description_nlsid             => 'IPMSGSAMP_TranslateColumnValue',
      p_owner                         => 'SYSMAN',
      p_category_nlsid                => 'IPMSGSAMP_SAMPLES', 
      p_sub_category_nlsid            => 'EMIPID_TARGETS',
      p_late_binding_target_types     =>  null,
      p_late_binding_multi_targets    =>  0,
      p_show_table_of_contents        =>  0, 
      p_system_report                 =>  0
        );

    l_curr_order := 1;
    l_curr_row := 1;

    mgmt_ip.register_sql_statement (
                p_name  =>'oracle.sysman.eml.ip.render.elem.table.component.IPMSG_USER_TABLE_FROM_SQL_TEST2',
                p_sql_statement => 
      'select ''LIST_OF_MEMBERS'' COL0, 15 COL1, ''AVAILABLE_OBJECTS'' COL2 from dual
        union
       select ''TOTAL_MEMBER_TARGETS'' COL0, 20 COL1, ''SELECTED_OBJECTS'' COL2 from dual
        union
       select ''TOTAL_MEMBER_TARGETS'' COL0, null COL1, ''SELECTED_OBJECTS'' COL2 from dual'
          );

    l_param_values := MGMT_IP_PARAM_VALUE_LIST();
    l_param_values.extend(4);
    l_param_values(1) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.NamedSqlStatement',
      'oracle.sysman.eml.ip.render.elem.table.component.IPMSG_USER_TABLE_FROM_SQL_TEST2');
    l_param_values(2) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.TableRender.translateColumnValueColumns',
      'COL0,COL2');
    l_param_values(3) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.TableRender.severityColumn',
      'COL1');
    l_param_values(4) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.TableRender.resourceBundle',
      'oracle.sysman.resources.MntrResourceBundle');

    l_element_guid := mgmt_ip.add_element_to_report_def (
      p_report_guid         => l_report_guid,
      p_element_name_nlsid  => 'IPMSG_USER_TABLE_FROM_SQL',
      p_element_type_nlsid  => 'IPMSG_ANY_TARGET_TYPE',
      p_header_nlsid        => 'IPMSGSAMP_targets_table_header',
      p_element_order       => l_curr_order, 
      p_element_row         => l_curr_row, 
      p_parameters          => l_param_values,
      p_targets             => null
        );
    l_curr_order := l_curr_order + 1;
    l_curr_row := l_curr_row + 1;



    -------------------------------------------------------------------
    -- Show how key values of a timeseries report can be translated
    -- into viewers language
    -------------------------------------------------------------------

    l_report_guid := mgmt_ip.create_report_definition (
      p_title_nlsid                   => 'I18N Key values for Timeseries Chart',
      p_description_nlsid             => 'Timeseries_Chart',
      p_owner                         => 'SYSMAN',
      p_category_nlsid                => 'IPMSGSAMP_SAMPLES', 
      p_sub_category_nlsid            => 'IPMSG_TARGETS',
      p_late_binding_target_types     =>  null,
      p_late_binding_multi_targets    =>  0,
      p_show_table_of_contents        =>  0, 
      p_system_report                 =>  0
        );

    l_curr_order := 1;
    l_curr_row := 1;
      
    mgmt_ip.register_sql_statement (
                p_name  =>'oracle.sysman.eml.ip.render.elem.table.component.IPMSGSAMP_CHART_I18NKEY_NLSID',
                p_sql_statement => 
      'select ''GROUPS'', sysdate, 55 from dual 
      union
      select ''SUPPORT'', sysdate, 99 from dual
      union
      select ''GROUPS'', sysdate-1, 60 from dual
      union 
      select ''SUPPORT'', sysdate-1, 69 from dual'
          );

    l_param_values := MGMT_IP_PARAM_VALUE_LIST();
    l_param_values.extend(7);
    l_param_values(1) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.NamedSqlStatement',
      'oracle.sysman.eml.ip.render.elem.table.component.IPMSGSAMP_CHART_I18NKEY_NLSID');
    l_param_values(2) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.ChartParamController.width',
      '400');
    l_param_values(3) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.ChartParamController.height',
      '200');
    l_param_values(4) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.ChartParamController.resourceBundle',
      'oracle.sysman.resources.MntrResourceBundle');
    l_param_values(5) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.ChartParamController.yAxisLabel',
      'PERCENTAGE_LABEL');
    l_param_values(6) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.ChartParamController.chartType',
      'timeSeriesChart');
    l_param_values(7) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.ChartParamController.translateKeyValues',
      'true');

    mgmt_ip.add_chart_from_sql_elem_def(
          p_element_name_nlsid  => 'IPMSGSAMP_CHART_I18NKEY_NLSID',
          p_element_type_nlsid  => 'IPMSG_ANY_TARGET_TYPE',
          p_description_nlsid   => 'IPMSGSAMP_sample_timeseries_with_i18N',
          p_internal_use_only   => 0, 
          p_default_parameters  => l_param_values,
          p_query_views         => 0
          );

    l_element_guid := mgmt_ip.add_element_to_report_def (
      p_report_guid         => l_report_guid,
      p_element_name_nlsid  => 'IPMSGSAMP_CHART_I18NKEY_NLSID',
      p_element_type_nlsid  => 'IPMSG_ANY_TARGET_TYPE',
      p_header_nlsid        => null,
      p_element_order       => l_curr_order, 
      p_element_row         => l_curr_row, 
      p_parameters          => null,
      p_targets             => null
        );
    l_curr_order := l_curr_order + 1;
    l_curr_row := l_curr_row + 1;



    -------------------------------------------------------------------
    -- Show how icon filename can displayed in a given column 
    -- by using a decode statement to convert a status code to 
    -- an image name
    -------------------------------------------------------------------

    l_report_guid := mgmt_ip.create_report_definition (
      p_title_nlsid                   => 'IPMSGSAMP_IconFilenameColumn',
      p_description_nlsid             => 'IPMSGSAMP_IconFilenameColumn',
      p_owner                         => 'SYSMAN',
      p_category_nlsid                => 'IPMSGSAMP_SAMPLES', 
      p_sub_category_nlsid            => 'EMIPID_TARGETS',
      p_late_binding_target_types     =>  null,
      p_late_binding_multi_targets    =>  0,
      p_show_table_of_contents        =>  0, 
      p_system_report                 =>  0
        );

    l_curr_order := 1;
    l_curr_row := 1;

    mgmt_ip.register_sql_statement (
                p_name  =>'oracle.sysman.eml.ip.render.elem.table.component.IconFilenameNamedSQLStatementText',
                p_sql_statement => 
      'select ''ZOOM_IN'' COL0, decode(15, 15, ''/images/zoom-in.gif'', 20, ''/images/zoom-out.gif'', ''/images/zoom-on.gif'')  COL1 from dual
        union
       select ''ZOOM_OUT'' COL0, decode(20, 15, ''/images/zoom-in.gif'', 20, ''/images/zoom-out.gif'', ''/images/zoom-on.gif'') COL1 from dual
        union
       select ''ZOOM_ON'' COL0, decode(null, 15, ''/images/zoom-in.gif'', 20, ''/images/zoom-out.gif'', ''/images/zoom-on.gif'') COL1 from dual'
          );

    l_param_values := MGMT_IP_PARAM_VALUE_LIST();
    l_param_values.extend(2);
    l_param_values(1) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.NamedSqlStatement',
      'oracle.sysman.eml.ip.render.elem.table.component.IconFilenameNamedSQLStatementText');
    l_param_values(2) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.TableRender.imageFilenameColumns',
      'COL1');

    l_element_guid := mgmt_ip.add_element_to_report_def (
      p_report_guid         => l_report_guid,
      p_element_name_nlsid  => 'IPMSG_USER_TABLE_FROM_SQL',
      p_element_type_nlsid  => 'IPMSG_ANY_TARGET_TYPE',
      p_header_nlsid        => 'IPMSGSAMP_targets_table_header',
      p_element_order       => l_curr_order, 
      p_element_row         => l_curr_row, 
      p_parameters          => l_param_values,
      p_targets             => null
        );
    l_curr_order := l_curr_order + 1;
    l_curr_row := l_curr_row + 1;


    -------------------------------------------------------------------
    -- Severity Column Test tables
    -------------------------------------------------------------------

    l_report_guid := mgmt_ip.create_report_definition(
      p_title_nlsid                   => 'Severity Column Test',
      p_description_nlsid             => 'Severity Column is populated with a variety of valid and invalid values',
      p_owner                         => 'SYSMAN',
      p_category_nlsid                => 'IPMSGSAMP_SAMPLES', 
      p_sub_category_nlsid            => 'Table Tests',
      p_late_binding_target_types     =>  null,
      p_late_binding_multi_targets    =>  0,
      p_show_table_of_contents        =>  0, 
      p_system_report                 =>  1);

    l_param_values := MGMT_IP_PARAM_VALUE_LIST();
    l_param_values.extend(3);
    l_param_values(1) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.sqlStatement',
      'SELECT null SEVERITY, ''null''   VALUE, ''(Invalid)''          MEANING FROM dual 
       UNION
       SELECT -1   SEVERITY, ''-1''     VALUE, ''(Invalid)''          MEANING FROM dual 
       UNION 
       SELECT 0    SEVERITY, ''0''      VALUE, ''(Invalid)''          MEANING FROM dual 
       UNION 
       SELECT 0.5  SEVERITY, ''0.5''    VALUE, ''(Invalid)''          MEANING FROM dual 
       UNION 
       SELECT 1    SEVERITY, ''1''      VALUE, ''(Invalid)''          MEANING FROM dual 
       UNION 
       SELECT 10   SEVERITY, ''10''     VALUE, ''Comment''            MEANING FROM dual 
       UNION 
       SELECT 15   SEVERITY, ''15''     VALUE, ''Clear''              MEANING FROM dual 
       UNION 
       SELECT 18   SEVERITY, ''18''     VALUE, ''Information''        MEANING FROM dual 
       UNION 
       SELECT 20   SEVERITY, ''20''     VALUE, ''Warning''            MEANING FROM dual 
       UNION 
       SELECT 25   SEVERITY, ''25''     VALUE, ''Critical''           MEANING FROM dual 
       UNION 
       SELECT 115  SEVERITY, ''115''    VALUE, ''Unreachable Clear''  MEANING FROM dual 
       UNION 
       SELECT 125  SEVERITY, ''125''    VALUE, ''Unreachable Start''  MEANING FROM dual 
       UNION 
       SELECT 215  SEVERITY, ''215''    VALUE, ''Blackout End''       MEANING FROM dual 
       UNION 
       SELECT 225  SEVERITY, ''225''    VALUE, ''Blackout Start''     MEANING FROM dual 
       UNION 
       SELECT 315  SEVERITY, ''315''    VALUE, ''Metric Error Clear'' MEANING FROM dual 
       UNION 
       SELECT 325  SEVERITY, ''325''    VALUE, ''Metric Error Start'' MEANING FROM dual 
       UNION 
       SELECT 415  SEVERITY, ''415''    VALUE, ''No Beacons Clear''   MEANING FROM dual 
       UNION 
       SELECT 425  SEVERITY, ''425''    VALUE, ''No Beacons Start''   MEANING FROM dual 
       UNION 
       SELECT 515  SEVERITY, ''515''    VALUE, ''Unknown''            MEANING FROM dual');
    l_param_values(2) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.TableRender.numRowsToShow',
      '25');
    l_param_values(3) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.TableRender.severityColumn',
      'SEVERITY');

    l_element_guid := mgmt_ip.add_element_to_report_def(
      p_report_guid         => l_report_guid,
      p_element_name_nlsid  => 'IPMSG_USER_TABLE_FROM_SQL',
      p_element_type_nlsid  => 'IPMSG_ANY_TARGET_TYPE',
      p_header_nlsid        => 'Severity Column Test Table w/ Numbers',
      p_element_order       => 1, 
      p_element_row         => 1, 
      p_parameters          => l_param_values,
      p_targets             => null);

    l_param_values := MGMT_IP_PARAM_VALUE_LIST();
    l_param_values.extend(3);
    l_param_values(1) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.sqlStatement',
      'SELECT null         SEVERITY, ''null''       VALUE, ''(Invalid)''   MEANING FROM dual 
       UNION
       SELECT ''null''     SEVERITY, ''"null"''     VALUE, ''(Invalid)''   MEANING FROM dual 
       UNION
       SELECT ''-1''        SEVERITY, ''"-1"''      VALUE, ''(Invalid)''          MEANING FROM dual 
       UNION 
       SELECT ''0''        SEVERITY, ''"0"''        VALUE, ''(Invalid)''          MEANING FROM dual 
       UNION 
       SELECT ''0.5''      SEVERITY, ''"0.5"''      VALUE, ''(Invalid)''          MEANING FROM dual 
       UNION 
       SELECT ''1''        SEVERITY, ''"1"''        VALUE, ''(Invalid)''          MEANING FROM dual 
       UNION  
       SELECT ''10''       SEVERITY, ''"10"''       VALUE, ''Comment''            MEANING FROM dual 
       UNION 
       SELECT ''15''       SEVERITY, ''"15"''       VALUE, ''Clear''              MEANING FROM dual 
       UNION 
       SELECT ''18''       SEVERITY, ''"18"''       VALUE, ''Information''        MEANING FROM dual 
       UNION 
       SELECT ''20''       SEVERITY, ''"20"''       VALUE, ''Warning''            MEANING FROM dual 
       UNION 
       SELECT ''25''       SEVERITY, ''"25"''       VALUE, ''Critical''           MEANING FROM dual 
       UNION  
       SELECT ''115''      SEVERITY, ''"115"''      VALUE, ''Unreachable Clear''  MEANING FROM dual 
       UNION 
       SELECT ''125''      SEVERITY, ''"125"''      VALUE, ''Unreachable Start''  MEANING FROM dual 
       UNION 
       SELECT ''215''      SEVERITY, ''"215"''      VALUE, ''Blackout End''       MEANING FROM dual 
       UNION 
       SELECT ''225''      SEVERITY, ''"225"''      VALUE, ''Blackout Start''     MEANING FROM dual 
       UNION 
       SELECT ''315''      SEVERITY, ''"315"''      VALUE, ''Metric Error Clear'' MEANING FROM dual 
       UNION 
       SELECT ''325''      SEVERITY, ''"325"''      VALUE, ''Metric Error Start'' MEANING FROM dual 
       UNION 
       SELECT ''415''      SEVERITY, ''"415"''      VALUE, ''No Beacons Clear''   MEANING FROM dual 
       UNION 
       SELECT ''425''      SEVERITY, ''"425"''      VALUE, ''No Beacons Start''   MEANING FROM dual 
       UNION 
       SELECT ''515''      SEVERITY, ''"515"''      VALUE, ''Unknown''            MEANING FROM dual 
       UNION
       SELECT ''A''        SEVERITY, ''"A"''        VALUE, ''(Invalid)''   MEANING FROM dual 
       UNION
       SELECT ''Critical'' SEVERITY, ''"Critical"'' VALUE, ''(Invalid)''   MEANING FROM dual 
       UNION
       SELECT ''   25   '' SEVERITY, ''"   25   "'' VALUE, ''(Invalid)''   MEANING FROM dual');
    l_param_values(2) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.TableRender.numRowsToShow',
      '25');
    l_param_values(3) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.TableRender.severityColumn',
      'SEVERITY');

    l_element_guid := mgmt_ip.add_element_to_report_def(
      p_report_guid         => l_report_guid,
      p_element_name_nlsid  => 'IPMSG_USER_TABLE_FROM_SQL',
      p_element_type_nlsid  => 'IPMSG_ANY_TARGET_TYPE',
      p_header_nlsid        => 'Severity Column Test Table w/ Strings',
      p_element_order       => 1, 
      p_element_row         => 1, 
      p_parameters          => l_param_values,
      p_targets             => null);

    -------------------------------------------------------------------
    -- Availability Status Column Test tables
    -------------------------------------------------------------------

    l_report_guid := mgmt_ip.create_report_definition(
      p_title_nlsid                   => 'Availability Status Column Test',
      p_description_nlsid             => 'Availability Status Column is populated with a variety of valid and invalid values',
      p_owner                         => 'SYSMAN',
      p_category_nlsid                => 'IPMSGSAMP_SAMPLES', 
      p_sub_category_nlsid            => 'Table Tests',
      p_late_binding_target_types     =>  null,
      p_late_binding_multi_targets    =>  0,
      p_show_table_of_contents        =>  0, 
      p_system_report                 =>  1);

    l_param_values := MGMT_IP_PARAM_VALUE_LIST();
    l_param_values.extend(3);
    l_param_values(1) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.sqlStatement',
      'SELECT null AVAIL, ''null''   VALUE, ''(Invalid)''          MEANING FROM dual 
       UNION
       SELECT -1   AVAIL, ''-1''     VALUE, ''(Invalid)''          MEANING FROM dual 
       UNION 
       SELECT 0.5  AVAIL, ''0.5''    VALUE, ''(Invalid)''          MEANING FROM dual 
       UNION 
       SELECT 0    AVAIL, ''0''      VALUE, ''AVAIL_DOWN''          MEANING FROM dual 
       UNION 
       SELECT 1    AVAIL, ''1''      VALUE, ''AVAIL_UP''          MEANING FROM dual 
       UNION 
       SELECT 2   AVAIL, ''2''     VALUE, ''AVAIL_ERROR''            MEANING FROM dual 
       UNION 
       SELECT 3   AVAIL, ''3''     VALUE, ''AVAIL_AGENT_DOWN''              MEANING FROM dual 
       UNION 
       SELECT 4   AVAIL, ''4''     VALUE, ''AVAIL_UNREACHABLE''        MEANING FROM dual 
       UNION 
       SELECT 5   AVAIL, ''5''     VALUE, ''AVAIL_BLACKOUT''            MEANING FROM dual 
       UNION 
       SELECT 6   AVAIL, ''6''     VALUE, ''AVAIL_UNKNOWN''           MEANING FROM dual');
    l_param_values(2) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.TableRender.numRowsToShow',
      '25');
    l_param_values(3) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.TableRender.availabilityStatusColumn',
      'AVAIL');

    l_element_guid := mgmt_ip.add_element_to_report_def(
      p_report_guid         => l_report_guid,
      p_element_name_nlsid  => 'IPMSG_USER_TABLE_FROM_SQL',
      p_element_type_nlsid  => 'IPMSG_ANY_TARGET_TYPE',
      p_header_nlsid        => 'Availability Status Column Test Table w/ Numbers',
      p_element_order       => 1, 
      p_element_row         => 1, 
      p_parameters          => l_param_values,
      p_targets             => null);

    l_param_values := MGMT_IP_PARAM_VALUE_LIST();
    l_param_values.extend(3);
    l_param_values(1) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.sqlStatement',
      'SELECT null AVAIL, ''null''   VALUE, ''(Invalid)''          MEANING FROM dual 
       UNION
       SELECT ''-1''   AVAIL, ''-1''     VALUE, ''(Invalid)''          MEANING FROM dual 
       UNION 
       SELECT ''0.5''  AVAIL, ''0.5''    VALUE, ''(Invalid)''          MEANING FROM dual 
       UNION 
       SELECT ''0''    AVAIL, ''0''      VALUE, ''AVAIL_DOWN''          MEANING FROM dual 
       UNION 
       SELECT ''1''    AVAIL, ''1''      VALUE, ''AVAIL_UP''          MEANING FROM dual 
       UNION 
       SELECT ''2''   AVAIL, ''2''     VALUE, ''AVAIL_ERROR''            MEANING FROM dual 
       UNION 
       SELECT ''3''   AVAIL, ''3''     VALUE, ''AVAIL_AGENT_DOWN''              MEANING FROM dual 
       UNION 
       SELECT ''4''   AVAIL, ''4''     VALUE, ''AVAIL_UNREACHABLE''        MEANING FROM dual 
       UNION 
       SELECT ''5''   AVAIL, ''5''     VALUE, ''AVAIL_BLACKOUT''            MEANING FROM dual 
       UNION 
       SELECT ''6''   AVAIL, ''6''     VALUE, ''AVAIL_UNKNOWN''           MEANING FROM dual');
    l_param_values(2) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.TableRender.numRowsToShow',
      '25');
    l_param_values(3) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.TableRender.availabilityStatusColumn',
      'AVAIL');

    l_element_guid := mgmt_ip.add_element_to_report_def(
      p_report_guid         => l_report_guid,
      p_element_name_nlsid  => 'IPMSG_USER_TABLE_FROM_SQL',
      p_element_type_nlsid  => 'IPMSG_ANY_TARGET_TYPE',
      p_header_nlsid        => 'Availability Status Column Test Table w/ Strings',
      p_element_order       => 1, 
      p_element_row         => 1, 
      p_parameters          => l_param_values,
      p_targets             => null);

    -------------------------------------------------------------------
    -- Name_value_pair_table severity
    -------------------------------------------------------------------
    l_report_guid := mgmt_ip.create_report_definition (
      p_title_nlsid                   => 'Name Value Pair with Severity',
      p_description_nlsid             => 'Name Value Pair with Severity',
      p_owner                         => 'SYSMAN',
      p_category_nlsid                => 'IPMSGSAMP_SAMPLES', 
      p_sub_category_nlsid            => 'Table Tests',
      p_late_binding_target_types     =>  null,
      p_late_binding_multi_targets    =>  0,
      p_show_table_of_contents        =>  0, 
      p_system_report                 =>  1
        );


    l_curr_order := 1;
    l_curr_row := 1;

    mgmt_ip.register_sql_statement (
                p_name  =>'oracle.sysman.eml.ip.render.elem.table.component.IPMSG_USER_TABLE_FROM_SQL_TEST',
                p_sql_statement => 
      'select 
        89 "(Invalid null)", null "SEVERITY",
        89 "(Invalid -1)", -1 "SEVERITY",
        89 "(Invalid 1)", 1 "SEVERITY",
        89 "(Invalid 0.5)", 0.5 "SEVERITY",
        1005 "Information", 18 "SEVERITY",
        5 "Warning", 20 "SEVERITY", 
        5 "Critical", 25 "SEVERITY", 
        5 "Unreachable Clear", 115 "SEVERITY", 
        5 "Unreachable Start", 125 "SEVERITY", 
        5 "Blackout End", 215 "SEVERITY", 
        5 "Blackout Start", 225 "SEVERITY", 
        5 "Metric Error Clear", 315 "SEVERITY", 
        5 "Metric Error Start", 325 "SEVERITY", 
        5 "No Beacons Clear", 415 "SEVERITY", 
        5 "No Beacons Start", 425 "SEVERITY", 
        5 "Unknown", 515 "SEVERITY" 
      from dual'
      );

    l_param_values := MGMT_IP_PARAM_VALUE_LIST();
    l_param_values.extend(3);
    l_param_values(1) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.NamedSqlStatement',
      'oracle.sysman.eml.ip.render.elem.table.component.IPMSG_USER_TABLE_FROM_SQL_TEST');
    l_param_values(2) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.TableRender.nameValueDisplay',
      '1');
    l_param_values(3) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.TableRender.severityColumn',
      'SEVERITY');


    l_element_guid := mgmt_ip.add_element_to_report_def (
      p_report_guid         => l_report_guid,
      p_element_name_nlsid  => 'IPMSG_USER_TABLE_FROM_SQL',
      p_element_type_nlsid  => 'IPMSG_ANY_TARGET_TYPE',
      p_header_nlsid        => 'IPMSGSAMP_targets_table_header',
      p_element_order       => l_curr_order, 
      p_element_row         => l_curr_row, 
      p_parameters          => l_param_values,
      p_targets             => null
        );
    l_curr_order := l_curr_order + 1;
    l_curr_row := l_curr_row + 1;

    -------------------------------------------------------------------
    -- Name_value_pair_table availability status
    -------------------------------------------------------------------
    l_report_guid := mgmt_ip.create_report_definition (
      p_title_nlsid                   => 'Name Value Pair with Availability Status',
      p_description_nlsid             => 'Name Value Pair with Availability Status',
      p_owner                         => 'SYSMAN',
      p_category_nlsid                => 'IPMSGSAMP_SAMPLES', 
      p_sub_category_nlsid            => 'Table Tests',
      p_late_binding_target_types     =>  null,
      p_late_binding_multi_targets    =>  0,
      p_show_table_of_contents        =>  0, 
      p_system_report                 =>  1
        );


    l_curr_order := 1;
    l_curr_row := 1;

    mgmt_ip.register_sql_statement (
                p_name  =>'oracle.sysman.eml.ip.render.elem.table.component.IPMSG_USER_TABLE_FROM_SQL_TEST_AVAIL_STATUS',
                p_sql_statement => 
      'select 
        5 "(Invalid null)", null "AVAIL", 
        89 "(Invalid -1)", -1 "AVAIL",
        1005 "(Invalid 0.5)", 0.5 "AVAIL",
        1005 "AVAIL_DOWN", 0 "AVAIL",
        1005 "AVAIL_UP", 1 "AVAIL",
        1005 "AVAIL_ERROR", 2 "AVAIL",
        1005 "AVAIL_AGENT_DOWN", 3 "AVAIL",
        1005 "AVAIL_UNREACHABLE", 4 "AVAIL",
        1005 "AVAIL_BLACKOUT", 5 "AVAIL",
        1005 "AVAIL_UNKNOWN", 6 "AVAIL"
      from dual'
      );

    l_param_values := MGMT_IP_PARAM_VALUE_LIST();
    l_param_values.extend(3);
    l_param_values(1) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.NamedSqlStatement',
      'oracle.sysman.eml.ip.render.elem.table.component.IPMSG_USER_TABLE_FROM_SQL_TEST_AVAIL_STATUS');
    l_param_values(2) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.TableRender.nameValueDisplay',
      '1');
    l_param_values(3) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.TableRender.availabilityStatusColumn',
      'AVAIL');


    l_element_guid := mgmt_ip.add_element_to_report_def (
      p_report_guid         => l_report_guid,
      p_element_name_nlsid  => 'IPMSG_USER_TABLE_FROM_SQL',
      p_element_type_nlsid  => 'IPMSG_ANY_TARGET_TYPE',
      p_header_nlsid        => 'IPMSGSAMP_targets_table_header',
      p_element_order       => l_curr_order, 
      p_element_row         => l_curr_row, 
      p_parameters          => l_param_values,
      p_targets             => null
        );
    l_curr_order := l_curr_order + 1;
    l_curr_row := l_curr_row + 1;



    -------------------------------------------------------------------
    -- Vanilla label value pair data type test
    -------------------------------------------------------------------

    l_report_guid := mgmt_ip.create_report_definition(
      p_title_nlsid                   => 'Label Value Pair Data Types, Translate Column Values',
      p_description_nlsid             => 'Various data types selected',
      p_owner                         => 'SYSMAN',
      p_category_nlsid                => 'IPMSGSAMP_SAMPLES', 
      p_sub_category_nlsid            => 'Table Tests',
      p_late_binding_target_types     =>  null,
      p_late_binding_multi_targets    =>  0,
      p_show_table_of_contents        =>  0, 
      p_system_report                 =>  1);

    l_param_values := MGMT_IP_PARAM_VALUE_LIST();
    l_param_values.extend(4);
    l_param_values(1) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.sqlStatement',
      'select 
        null "null", 
        -1 "-1",
        50000 "50000",
        sysdate "sysdate",
        systimestamp "systimestamp",
        ''REPORTS_TABLE_GO_BUTTON'' "MSG1",
        ''TITLE'' "MSG2"
      from dual');
    l_param_values(2) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.TableRender.nameValueDisplay',
      '3');
    l_param_values(3) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.TableRender.resourceBundle',
      'oracle.sysman.resources.eml.ip.IPMsg');
    l_param_values(4) := MGMT_IP_PARAM_VALUE_RECORD(
      'oracle.sysman.eml.ip.render.elem.TableRender.translateColumnValueColumns',
      'MSG1,MSG2');

    l_element_guid := mgmt_ip.add_element_to_report_def(
      p_report_guid         => l_report_guid,
      p_element_name_nlsid  => 'IPMSG_USER_TABLE_FROM_SQL',
      p_element_type_nlsid  => 'IPMSG_ANY_TARGET_TYPE',
      p_header_nlsid        => 'Name Value Type Test - 3 columns',
      p_element_order       => 1, 
      p_element_row         => 1, 
      p_parameters          => l_param_values,
      p_targets             => null);


END load;

    
PROCEDURE load_nls_strings is 
BEGIN
    --   ----------------------------------------------------------------------------------------
    --   MGMT_MESSAGES (EM translated string table -- OOTB string insertions)
    --
    --   THIS IS FOR DEMO PURPOSES ONLY -- TRANSLATED STRINGS SHOULD BE STORED
    --   IN A .DLF FILE AND LOADED INTO THE REPOSITORY BY REP_SCHEMA.
    --
    --   THESE STRINGS WILL NEVER BE TRANSLATED.
    --   --------------------------------------------------------------------------------------
    --
    
    -- Report Definitions
    --
    insert into MGMT_MESSAGES (MESSAGE_ID, SUBSYSTEM, LANGUAGE_CODE, COUNTRY_CODE, MESSAGE) 
         values ('IPMSGSAMP_All_Targets_Title', 'REPORTS', 'en', ' ', 'All Targets');
    insert into MGMT_MESSAGES (MESSAGE_ID, SUBSYSTEM, LANGUAGE_CODE, COUNTRY_CODE, MESSAGE) 
         values ('IPMSGSAMP_All_Targets_Description', 'REPORTS', 'en', ' ', 'Listing of all targets');
    insert into MGMT_MESSAGES (MESSAGE_ID, SUBSYSTEM, LANGUAGE_CODE, COUNTRY_CODE, MESSAGE) 
         values ('IPMSGSAMP_ALL_TARGETS_CHART', 'REPORTS', 'en', ' ', 'All Targets Chart');
    insert into MGMT_MESSAGES (MESSAGE_ID, SUBSYSTEM, LANGUAGE_CODE, COUNTRY_CODE, MESSAGE) 
         values ('IPMSGSAMP_all_target_pie_chart', 'REPORTS', 'en', ' ', 'All Targets Pie Chart');
    insert into MGMT_MESSAGES (MESSAGE_ID, SUBSYSTEM, LANGUAGE_CODE, COUNTRY_CODE, MESSAGE) 
         values ('IPMSGSAMP_ALL_TARGETS_TABLE', 'REPORTS', 'en', ' ', 'All Targets Table');
      
    insert into MGMT_MESSAGES (MESSAGE_ID, SUBSYSTEM, LANGUAGE_CODE, COUNTRY_CODE, MESSAGE) 
         values ('IPMSGSAMP_Cache_Hit_Title', 'REPORTS', 'en', ' ', 'Cache Hits');         
    insert into MGMT_MESSAGES (MESSAGE_ID, SUBSYSTEM, LANGUAGE_CODE, COUNTRY_CODE, MESSAGE) 
         values ('IPMSGSAMP_Cache_Hit_Description', 'REPORTS', 'en', ' ', 'Cache hit details');
    
    insert into MGMT_MESSAGES (MESSAGE_ID, SUBSYSTEM, LANGUAGE_CODE, COUNTRY_CODE, MESSAGE) 
         values ('IPMSGSAMP_Host_Statistics', 'REPORTS', 'en', ' ', 'Host Statistics');         
    insert into MGMT_MESSAGES (MESSAGE_ID, SUBSYSTEM, LANGUAGE_CODE, COUNTRY_CODE, MESSAGE) 
         values ('IPMSGSAMP_Key_statistics_for_host_target', 'REPORTS', 'en', ' ', 'Key statistics for host target');

    insert into MGMT_MESSAGES (MESSAGE_ID, SUBSYSTEM, LANGUAGE_CODE, COUNTRY_CODE, MESSAGE) 
         values ('IPMSGSAMP_Name_value_pair_table', 'REPORTS', 'en', ' ', 'Name Value Pair table');         
    insert into MGMT_MESSAGES (MESSAGE_ID, SUBSYSTEM, LANGUAGE_CODE, COUNTRY_CODE, MESSAGE) 
         values ('IPMSGSAMP_name_value_pairs', 'REPORTS', 'en', ' ', 'Shows table displayed as name value pairs with severity column');
    
    insert into MGMT_MESSAGES (MESSAGE_ID, SUBSYSTEM, LANGUAGE_CODE, COUNTRY_CODE, MESSAGE) 
         values ('IPMSGSAMP_Blackout_SCHEDULE', 'REPORTS', 'en', ' ', 'Blackout Schedule');
    insert into MGMT_MESSAGES (MESSAGE_ID, SUBSYSTEM, LANGUAGE_CODE, COUNTRY_CODE, MESSAGE) 
         values ('IPMSGSAMP_Blackout_SCHEDULE_Description', 'REPORTS', 'en', ' ', 'Blackout schedule for given time period');
    
    insert into MGMT_MESSAGES (MESSAGE_ID, SUBSYSTEM, LANGUAGE_CODE, COUNTRY_CODE, MESSAGE) 
         values ('IPMSGSAMP_Filesystem_image_Title', 'REPORTS', 'en', ' ', 'Filesystem Image Report');
    insert into MGMT_MESSAGES (MESSAGE_ID, SUBSYSTEM, LANGUAGE_CODE, COUNTRY_CODE, MESSAGE) 
         values ('IPMSGSAMP_Filesystem_image_Description', 'REPORTS', 'en', ' ', 'This displays the Oracle Logo in red from the default em installed images');
    
    -- Element Headers
    --
    insert into MGMT_MESSAGES (MESSAGE_ID, SUBSYSTEM, LANGUAGE_CODE, COUNTRY_CODE, MESSAGE) 
         values ('IPMSGSAMP_target_types_chart_header', 'REPORTS', 'en', ' ', 'Target Types');
    insert into MGMT_MESSAGES (MESSAGE_ID, SUBSYSTEM, LANGUAGE_CODE, COUNTRY_CODE, MESSAGE) 
         values ('IPMSGSAMP_targets_table_header', 'REPORTS', 'en', ' ', 'Targets');
    
    insert into MGMT_MESSAGES (MESSAGE_ID, SUBSYSTEM, LANGUAGE_CODE, COUNTRY_CODE, MESSAGE) 
         values ('IPMSGSAMP_Buff_Cache_Hit', 'REPORTS', 'en', ' ', 'Buffer Cache Hit %');
    insert into MGMT_MESSAGES (MESSAGE_ID, SUBSYSTEM, LANGUAGE_CODE, COUNTRY_CODE, MESSAGE) 
         values ('IPMSGSAMP_Lib_Cache_Hit', 'REPORTS', 'en', ' ', 'Library Cache Hit %');
    
    insert into MGMT_MESSAGES (MESSAGE_ID, SUBSYSTEM, LANGUAGE_CODE, COUNTRY_CODE, MESSAGE) 
         values ('IPMSGSAMP_blackouts_next_week_header', 'REPORTS', 'en', ' ', 'Blackouts upcoming this week');

    insert into MGMT_MESSAGES (MESSAGE_ID, SUBSYSTEM, LANGUAGE_CODE, COUNTRY_CODE, MESSAGE) 
         values ('IPMSGSAMP_SELECT_FROM_DUAL', 'REPORTS', 'en', ' ', 'Select from Dual - On The Fly Element Creation');
    insert into MGMT_MESSAGES (MESSAGE_ID, SUBSYSTEM, LANGUAGE_CODE, COUNTRY_CODE, MESSAGE) 
         values ('IPMSGSAMP_example_ootb_element_with_default_params', 'REPORTS', 'en', ' ', 'Example of Built-in Element with Default Parameters');

    insert into MGMT_MESSAGES (MESSAGE_ID, SUBSYSTEM, LANGUAGE_CODE, COUNTRY_CODE, MESSAGE) 
         values ('IPMSGSAMP_SAMPLES', 'REPORTS', 'en', ' ', 'X Sample Report Definitions');

    insert into MGMT_MESSAGES (MESSAGE_ID, SUBSYSTEM, LANGUAGE_CODE, COUNTRY_CODE, MESSAGE) 
         values ('IPMSGSAMP_select_database_targets_for_report', 'REPORTS', 'en', ' ', 'Select Database Targets');

    insert into MGMT_MESSAGES (MESSAGE_ID, SUBSYSTEM, LANGUAGE_CODE, COUNTRY_CODE, MESSAGE) 
         values ('IPMSGSAMP_USER_SELECTS_DATABASE', 'REPORTS', 'en', ' ', 'User Select Database Target');

    insert into MGMT_MESSAGES (MESSAGE_ID, SUBSYSTEM, LANGUAGE_CODE, COUNTRY_CODE, MESSAGE) 
         values ('IPMSGSAMP_SEVERITIES', 'REPORTS', 'en', ' ', 'Severities');

    insert into MGMT_MESSAGES (MESSAGE_ID, SUBSYSTEM, LANGUAGE_CODE, COUNTRY_CODE, MESSAGE) 
         values ('IPMSGSAMP_database_target_selector_desc', 'REPORTS', 'en', ' ', 'Element for selecting database targets to include in report');

    insert into MGMT_MESSAGES (MESSAGE_ID, SUBSYSTEM, LANGUAGE_CODE, COUNTRY_CODE, MESSAGE) 
         values ('IPMSGSAMP_DATABASE_MULTI_TARGETS', 'REPORTS', 'en', ' ', 'Element for selecting database targets to include in report');
    insert into MGMT_MESSAGES (MESSAGE_ID, SUBSYSTEM, LANGUAGE_CODE, COUNTRY_CODE, MESSAGE) 
         values ('IPMSGSAMP_USER_SELECTS_DATABASE_COL_TRANS', 'REPORTS', 'en', ' ', 'End User selects database, column headers translated');
    insert into MGMT_MESSAGES (MESSAGE_ID, SUBSYSTEM, LANGUAGE_CODE, COUNTRY_CODE, MESSAGE) 
         values ('IPMSGSAMP_table_with_col_trans', 'REPORTS', 'en', ' ', 'End User selects database, column headers translated');

end load_nls_strings;


END MGMT_IP_SAMPLES;
/
show errors;

set serveroutput on;
call mgmt_ip_samples.load_nls_strings();
call mgmt_ip_samples.load();
show errors;
commit;
