Rem
Rem  ip_pkgbody_10_2_0_1_0.sql
Rem
Rem Copyright (c) 2004, 2005, Oracle. All rights reserved.  
Rem
Rem    NAME
Rem      ip_pkgbody_10_2_0_1_0.sql - <one-line expansion of the name>
Rem
Rem    DESCRIPTION
Rem      <short description of component this file declares/defines>
Rem
Rem    NOTES
Rem      tbd
Rem
Rem    MODIFIED   (MM/DD/YY)
Rem    lgloyd      07/28/05 - lgloyd_bug_4502618_plsql
Rem    lgloyd      03/31/05 - add api for regression tests 
Rem    jabramso    03/18/05 - add update_element_type 
Rem    jabramso    03/09/05 - update last published date on pub ver delete 
Rem    lgloyd      02/28/05 - register sql 
Rem    lgloyd      02/16/05 - help topic 
Rem    jabramso    01/11/05 - Add public option to create 
Rem    tzimmerm    12/21/04 - Purge publish jobs with stored copies 
Rem    jabramso    12/17/04 - Make sure report owner is upcase 
Rem    lgloyd      11/30/04 - fix param classes bug 
Rem    jabramso    12/06/04 - revoke all report priv 
Rem    lgloyd      11/26/04 - report params, version 
Rem    lgloyd      11/24/04 - add version to rep def 
Rem    jabramso    10/25/04 - Add check to see if user has publish to public 
Rem                           system privilege 
Rem    lgloyd      09/22/04 - jit types 
Rem    jabramso    09/23/04 - Add GET_GRANTEES_FOR_REPORT 
Rem    kmanicka    09/20/04 - add user model callbacks
Rem    lgloyd      09/08/04 - rep def jsp 
Rem    dcawley     08/31/04 - Call user model when report is deleted 
Rem    lgloyd      08/25/04 - alerts
Rem    tzimmerm    08/25/04 - 
Rem    tzimmerm    07/21/04 - Add IP purge functionality 
Rem    lgloyd      07/14/04 - pl/sql api for report defns 
Rem    dcawley     07/07/04 - Increase user name size 
Rem    tzimmerm    05/18/04 - Fix tgt type filter 
Rem    tzimmerm    05/21/04 - Add delete rpt by version 
Rem    tzimmerm    03/31/04 - Add support for hGrid Prior Copies column
Rem    tzimmerm    03/29/04 - 
Rem    tzimmerm    03/26/04 - Adding report delete functions
Rem    tzimmerm    03/18/04 - Implement locale support 
Rem    tzimmerm    03/17/04 - 
Rem    tzimmerm    03/10/04 - Add initial target support 
Rem    tzimmerm    03/08/04 - 
Rem    tzimmerm    03/05/04 - 
Rem    tzimmerm    03/03/04 - Add target tables & NLSID columns 
Rem    tzimmerm    02/27/04 - tzimmerm_reports_repos 
Rem    tzimmerm    02/17/04 - created
Rem

create or replace package body MGMT_IP
as

--  ----------------------------------------------------------------------------
--  Create a report definition
--  Input:
--      p_title_nlsid         report title (nlsid if translated or
--                            display string if not translated)
--      p_description_nlsid   description (nlsid if translated or
--                            display string if not translated)
--      p_owner               owner name
--      p_category_nlsid      category name (nlsid if translated or
--                            display string if not translated)
--      p_sub_category_nlsid  subcategory name (nlsid if translated or
--                            display string if not translated)
--      p_late_binding_target_types  - target type for late binding, 
--                            or null if not late binding
--      p_late_binding_multi_targets  - 1 if multiple targets are
--                            supported for late binding, otherwise 0
--      p_show_table_of_contents    1=show 0=hide
--      p_system_report       1=system report, 0=end user report      
--      p_custom_bean_class   optional class for customizing report
--      p_internal_use_only   1=internal use only, 0=available to administrators
--      p_pack_name           License pack assoc with this report defintion, or null
--      p_style               Style for rendering
--      p_show_navigation     Show navigation headers in report (tabs, etc)
--      p_product_name        Product name
--      p_component_name      Component name
--      p_version             Product version
--      p_parameters          Report wide parameters
--      p_public              Whether report will show up for public user
--      p_help_topic_id       Help topic id
--  Output:
--      returns guid for report definition
--  ----------------------------------------------------------------------------

FUNCTION create_report_definition (
                p_title_nlsid                       IN VARCHAR2,
                p_description_nlsid                 IN VARCHAR2,
                p_owner                             IN VARCHAR2,
                p_category_nlsid                    IN VARCHAR2,
                p_sub_category_nlsid                IN VARCHAR2,                
                p_late_binding_target_types         IN MGMT_IP_TARGET_TYPES DEFAULT NULL,
                p_late_binding_multi_targets        IN NUMBER DEFAULT 0,
                p_show_table_of_contents            IN NUMBER DEFAULT 0,
                p_system_report                     IN NUMBER DEFAULT 1,
                p_custom_report_bean_class          IN VARCHAR2 DEFAULT NULL,
                p_internal_use_only                 IN NUMBER DEFAULT 0,
                p_pack_name                         IN VARCHAR2 DEFAULT NULL,
                p_style                             IN VARCHAR2 DEFAULT 'BLAF',
                p_show_navigation                   IN NUMBER DEFAULT 1,
                p_product_name                      IN VARCHAR2 DEFAULT 'EM',
                p_component_name                    IN VARCHAR2 DEFAULT '',
                p_version                           IN VARCHAR2 DEFAULT '10.2.0.1.0',
                p_parameters                        IN MGMT_IP_PARAM_VALUE_LIST DEFAULT NULL,
                p_generate_context                  IN NUMBER DEFAULT 0,
                p_public_report                     IN NUMBER DEFAULT 0,
                p_help_topic_id                     IN VARCHAR2 DEFAULT NULL
                ) RETURN RAW IS
  l_report_definition_guid MGMT_IP_REPORT_DEF.report_guid%TYPE;
BEGIN

    l_report_definition_guid := sys_guid();
    
    return create_report_def_internal(
                l_report_definition_guid,
                p_title_nlsid,
                p_description_nlsid,
                p_owner,
                p_category_nlsid,
                p_sub_category_nlsid,                
                p_late_binding_target_types,
                p_late_binding_multi_targets,
                p_show_table_of_contents,
                p_system_report,
                p_custom_report_bean_class,
                p_internal_use_only,
                p_pack_name,
                p_style,
                p_show_navigation,
                p_product_name,
                p_component_name,
                p_version,
                p_parameters,
                p_generate_context,
                p_public_report,
                p_help_topic_id);

END;

--  ----------------------------------------------------------------------------
-- ORACLE INTERNAL USE ONLY: This api is provided to create report definitions 
--                       for regression tests
--  ----------------------------------------------------------------------------
FUNCTION create_report_def_internal (
                p_report_guid                       IN RAW,
                p_title_nlsid                       IN VARCHAR2,
                p_description_nlsid                 IN VARCHAR2,
                p_owner                             IN VARCHAR2,
                p_category_nlsid                    IN VARCHAR2,
                p_sub_category_nlsid                IN VARCHAR2,                
                p_late_binding_target_types         IN MGMT_IP_TARGET_TYPES DEFAULT NULL,
                p_late_binding_multi_targets        IN NUMBER DEFAULT 0,
                p_show_table_of_contents            IN NUMBER DEFAULT 0,
                p_system_report                     IN NUMBER DEFAULT 1,
                p_custom_report_bean_class          IN VARCHAR2 DEFAULT NULL,
                p_internal_use_only                 IN NUMBER DEFAULT 0,
                p_pack_name                         IN VARCHAR2 DEFAULT NULL,
                p_style                             IN VARCHAR2 DEFAULT 'BLAF',
                p_show_navigation                   IN NUMBER DEFAULT 1,
                p_product_name                      IN VARCHAR2 DEFAULT 'EM',
                p_component_name                    IN VARCHAR2 DEFAULT '',
                p_version                           IN VARCHAR2 DEFAULT '10.2.0.1.0',
                p_parameters                        IN MGMT_IP_PARAM_VALUE_LIST DEFAULT NULL,
                p_generate_context                  IN NUMBER DEFAULT 0,
                p_public_report                     IN NUMBER DEFAULT 0,
                p_help_topic_id                     IN VARCHAR2 DEFAULT NULL
                ) RETURN RAW IS
  l_is_late_binding NUMBER;
  l_owner VARCHAR2(256) := UPPER(TRIM(p_owner));
BEGIN
   
    IF p_late_binding_target_types IS NULL
    THEN
      l_is_late_binding := 0;
    ELSE 
      l_is_late_binding := 1;
    END IF;
    
    -- add element to MGMT_IP_REPORT_DEF_ELEMENTS
    BEGIN
      INSERT INTO MGMT_IP_REPORT_DEF
          (REPORT_GUID, 
            TITLE_NLSID, DESCRIPTION_NLSID,
            OWNER, 
            CATEGORY_NLSID, SUB_CATEGORY_NLSID,
            LAST_EDIT_TIME, 
            IS_JIT_TARGET, IS_JIT_MULTI_TARGET,
            ADD_TOC,
            SYSTEM_REPORT,
            CUSTOM_REPORT_BEAN_CLASS, 
            INTERNAL_ONLY,
            PACK_NAME,
            STYLE,
            SHOW_NAVIGATION,
            PRODUCT_NAME,
            COMPONENT_NAME,
            VERSION,
            GENERATE_CONTEXT,
            PUBLISH_TO_PUBLIC,
            HELP_TOPIC_ID)
      VALUES
          (p_report_guid, 
           p_title_nlsid, p_description_nlsid,
           l_owner,
           p_category_nlsid, p_sub_category_nlsid,
           sysdate,
           l_is_late_binding, p_late_binding_multi_targets,
           p_show_table_of_contents,
           p_system_report,
           p_custom_report_bean_class,
           p_internal_use_only,
           p_pack_name,
           p_style,
           p_show_navigation,
           p_product_name,
           p_component_name,
           p_version,
           p_generate_context,
           p_public_report,
           p_help_topic_id);
      
    EXCEPTION
          WHEN OTHERS THEN 
            RAISE;
    END;
    
    -- save parameters for report definition
   IF p_parameters IS NOT NULL
   THEN
     FOR i IN 1..p_parameters.COUNT LOOP
        BEGIN
            INSERT INTO MGMT_IP_REPORT_ELEM_PARAMS
                (REPORT_GUID, ELEMENT_GUID, 
                  PARAM, VALUE)
            VALUES
                (p_report_guid, p_report_guid,
                  p_parameters(i).param, p_parameters(i).value);
        EXCEPTION
            WHEN OTHERS THEN
              dbms_output.put_line('Error inserting parameter for report definition  ' ||
                    p_title_nlsid || ':' || 
                    p_description_nlsid);
              RAISE;
        END;
        END LOOP;
    END IF;     

    -- add late binding target types
   IF p_late_binding_target_types IS NOT NULL 
   THEN
     FOR i IN 1..p_late_binding_target_types.COUNT LOOP
        BEGIN
            INSERT INTO MGMT_IP_REPORT_DEF_JIT_TYPES
                (REPORT_GUID, TARGET_TYPE)
            VALUES
                (p_report_guid, p_late_binding_target_types(i));
        EXCEPTION
            WHEN OTHERS THEN
              dbms_output.put_line('Error inserting late binding target for report definition  ' ||
                    p_title_nlsid || ':' || 
                    p_late_binding_target_types(i));
              RAISE;
        END;
        END LOOP;
    END IF;

    -- grant report access to owner of report
    grant_report_to_user(l_owner, p_report_guid);
    
    return p_report_guid;
END;

--  ----------------------------------------------------------------------------
--  Add an element to a report definition
--  Input:
--        p_report_guid         guid to identify the report definition       
--        p_element_name_nlsid  element name (nlsid if translated or
--                              display string if not translated)
--        p_element_type_nlsid  element type name (nlsid if translated or
--                              display string if not translated) 
--        p_header_nlslid       element header (nlsid if translated or
--                              display string if not translated), or null
--        p_element_order       order of this element, 1 based
--        p_element_row         row for this element, 1 based
--        p_parameters          parameters for this element
--        p_targets             targets for this element
--        p_header_nlsid        element header (nlsid if translated or
--                              display string if not translated), or null
--                              [fix typo in earlier param]
--  Output:
--      returns guid for this element instance
--  ----------------------------------------------------------------------------
FUNCTION add_element_to_report_def(
                p_report_guid                       IN RAW,
                p_element_name_nlsid                IN VARCHAR2,
                p_element_type_nlsid                IN VARCHAR2,
                p_header_nlslid                     IN VARCHAR2 DEFAULT NULL,
                p_element_order                     IN NUMBER,
                p_element_row                       IN NUMBER,
                p_parameters                        IN MGMT_IP_PARAM_VALUE_LIST,
                p_targets                           IN MGMT_IP_TARGET_LIST,
                p_header_nlsid                      IN VARCHAR2 DEFAULT NULL
                ) RETURN RAW IS
  l_element_guid MGMT_IP_REPORT_DEF_ELEMENTS.element_guid%TYPE;
  l_header VARCHAR2(200);
BEGIN

    -- fix typo in header param name (backward compatible with misspelled version)
    l_header := p_header_nlslid;
    if p_header_nlsid is not null
    then
      l_header := p_header_nlsid;
    end if;
      
    l_element_guid := sys_guid();
    
    -- add element to MGMT_IP_REPORT_DEF_ELEMENTS
    BEGIN
      INSERT INTO MGMT_IP_REPORT_DEF_ELEMENTS
          (REPORT_GUID, ELEMENT_GUID, 
            ELEMENT_NAME_NLSID, ELEMENT_TYPE_NLSID,
            HEADER_NLSID, ELEMENT_ORDER, ELEMENT_ROW)
      VALUES
          (p_report_guid, l_element_guid, 
           p_element_name_nlsid, p_element_type_nlsid,
           l_header, p_element_order, p_element_row);
      
    EXCEPTION
          WHEN OTHERS THEN 
            RAISE;
    END;

    -- save parameters for element
   IF p_parameters IS NOT NULL
   THEN
     FOR i IN 1..p_parameters.COUNT LOOP
        BEGIN
            INSERT INTO MGMT_IP_REPORT_ELEM_PARAMS
                (REPORT_GUID, ELEMENT_GUID, 
                  PARAM, VALUE)
            VALUES
                (p_report_guid, l_element_guid,
                  p_parameters(i).param, p_parameters(i).value);
        EXCEPTION
            WHEN OTHERS THEN
              dbms_output.put_line('Error inserting parameter for element  ' ||
                    p_element_name_nlsid || ':' || 
                    p_element_type_nlsid);
              RAISE;
        END;
        END LOOP;
    END IF;     
      
    -- save targets for element
    IF p_targets IS NOT NULL 
    THEN
      FOR i IN 1..p_targets.COUNT LOOP
      BEGIN
          INSERT INTO MGMT_IP_REPORT_ELEM_TARGETS
              (REPORT_GUID, ELEMENT_GUID, 
               TARGET_NAME, TARGET_TYPE, TARGET_GUID)
          VALUES
              (p_report_guid, l_element_guid,
               p_targets(i).target_name, p_targets(i).target_type, null);
      EXCEPTION
          WHEN OTHERS THEN
            dbms_output.put_line('Error inserting target for element ' ||
                  p_element_name_nlsid || ':' || 
                  p_element_type_nlsid || ':' ||
                  p_targets(i).target_name || ':' ||
                  p_targets(i).target_type);
          RAISE;
      END;
      END LOOP;
    END IF;
    
    RETURN l_element_guid;

END;


--  ----------------------------------------------------------------------------
--  Add an element type.  An element type defines what target types are
--    compatiable with an element.
--   Input:
--    p_element_type_nlsid    element type name (nlsid if translated or
--                            display string if not translated)
--    p_target_types          list of target types assoc with this element type
--  ----------------------------------------------------------------------------

PROCEDURE add_element_type (
                p_element_type_nlsid              IN VARCHAR2,
                p_target_types                    IN MGMT_IP_TARGET_TYPES
                ) IS
BEGIN
    -- delete any existing entries for elements of this type
    --
    DELETE FROM MGMT_IP_ELEM_TARGET_TYPES 
      where ELEMENT_TYPE_NLSID=p_element_type_nlsid;
    
    -- insert element type and supported target types into MGMT_IP_ELEM_TARGET_TYPES
    --
    FOR i IN 1..p_target_types.COUNT LOOP
    BEGIN
        INSERT INTO MGMT_IP_ELEM_TARGET_TYPES
            (ELEMENT_TYPE_NLSID, TARGET_TYPE)
        VALUES
            (p_element_type_nlsid, p_target_types(i));
    EXCEPTION
        WHEN OTHERS THEN
          dbms_output.put_line('Error inserting target type for element type ' ||
                p_element_type_nlsid || ':' || 
                p_target_types(i));
        RAISE;
    END;
    END LOOP;
  
END;

--  ----------------------------------------------------------------------------
--  update an element type given the old target type and the new one.
--  An element type defines what target types are
--   compatiable with an element.
--   Input:
--    p_element_type_nlsid    element type name (nlsid if translated or
--                            display string if not translated)
--    p_target_type           The target type assoc with this element type to
--                            either add or delete.
--    p_add_or_remove         TRUE to add, FALSE to remove.
--  ----------------------------------------------------------------------------

PROCEDURE update_element_type (
                p_element_type_nlsid              IN VARCHAR2,
                p_target_type                     IN VARCHAR,
                p_add_or_remove                   IN BOOLEAN
                ) IS
BEGIN
    -- delete any existing entries for elements with the old type
    --
    IF p_add_or_remove = FALSE
    THEN
      BEGIN
        DELETE FROM MGMT_IP_ELEM_TARGET_TYPES 
          WHERE ELEMENT_TYPE_NLSID = p_element_type_nlsid
            AND TARGET_TYPE = p_target_type;
      END;
    ELSE
      BEGIN
        INSERT INTO MGMT_IP_ELEM_TARGET_TYPES
                    (ELEMENT_TYPE_NLSID, TARGET_TYPE)
             VALUES
                    (p_element_type_nlsid, p_target_type);
      END;
    END IF;
END;    

--  ----------------------------------------------------------------------------
--  Add an element definition
--   Input:
--        p_element_name_nlsid  element name (nlsid if translated or
--                              display string if not translated)
--        p_element_type_nlsid  element type name (nlsid if translated or
--                              display string if not translated) 
--        p_description_nlsid   description (nlsid if translated or
--                              display string if not translated), or null
--        p_element_class_name  name of the class which renders this element
--        p_internal_use_only   1=internal use only, 0=available to administrators
--        p_param_classes       list of parameter controller class names
--        p_default_parameters  list of default parameters
--        p_pack_name           License pack assoc with this report defintion, or null
--        p_product_name        Product name
--        p_component_name      Component name
--        p_version             Product version
--  ----------------------------------------------------------------------------
PROCEDURE add_element_definition (                
                p_element_name_nlsid                IN VARCHAR2,
                p_element_type_nlsid                IN VARCHAR2,
                p_description_nlsid                 IN VARCHAR2,
                p_element_class_name                IN VARCHAR2,
                p_internal_use_only                 IN NUMBER DEFAULT 0, 
                p_param_classes                     IN MGMT_IP_PARAM_CLASSES  DEFAULT NULL,
                p_default_parameters                IN MGMT_IP_PARAM_VALUE_LIST DEFAULT NULL,
                p_pack_name                         IN VARCHAR2 DEFAULT NULL,
                p_product_name                      IN VARCHAR2 DEFAULT 'EM',
                p_component_name                    IN VARCHAR2 DEFAULT '',
                p_version                           IN VARCHAR2 DEFAULT '10.2.0.1.0'
                ) IS
BEGIN
    -- delete any existing element definition
    -- (LG:  not sure is this is the correct behavior)
    DELETE FROM MGMT_IP_REPORT_ELEM_DEF 
      where ELEMENT_NAME_NLSID=p_element_name_nlsid
      and ELEMENT_TYPE_NLSID=p_element_type_nlsid;
    DELETE FROM MGMT_IP_ELEM_PARAM_CLASSES 
      where ELEMENT_NAME_NLSID=p_element_name_nlsid
      and ELEMENT_TYPE_NLSID=p_element_type_nlsid;
    DELETE FROM MGMT_IP_ELEM_DEFAULT_PARAMS 
      where ELEMENT_NAME_NLSID=p_element_name_nlsid
      and ELEMENT_TYPE_NLSID=p_element_type_nlsid;

    -- add this element definition to MGMT_IP_REPORT_ELEM_DEF
    BEGIN
      INSERT INTO MGMT_IP_REPORT_ELEM_DEF
          (ELEMENT_NAME_NLSID, ELEMENT_TYPE_NLSID, DESCRIPTION_NLSID,
            ELEMENT_CLASS, INTERNAL_ONLY, PACK_NAME, 
            PRODUCT_NAME, COMPONENT_NAME, VERSION)
      VALUES
          (p_element_name_nlsid, p_element_type_nlsid, p_description_nlsid,
           p_element_class_name, p_internal_use_only, p_pack_name,
           p_product_name, p_component_name, p_version);
      
    EXCEPTION
          WHEN OTHERS THEN 
            RAISE;
    END;
  
    -- insert the parameter classes
    IF p_param_classes IS NOT NULL 
    THEN
      FOR i IN 1..p_param_classes.COUNT LOOP
      BEGIN
          INSERT INTO MGMT_IP_ELEM_PARAM_CLASSES
              (ELEMENT_NAME_NLSID, ELEMENT_TYPE_NLSID, 
                ELEMENT_PARAM_CLASS, DISPLAY_ORDER)
          VALUES
              (p_element_name_nlsid, p_element_type_nlsid,
                p_param_classes(i), i);
      EXCEPTION
          WHEN OTHERS THEN
            dbms_output.put_line('Error inserting parameter class for element definition ' ||
                  p_element_name_nlsid || ':' || 
                  p_element_type_nlsid || ':' || 
                  p_param_classes(i));
            RAISE;
      END;
      END LOOP;
    END IF; 
  
  -- insert the default parameters
    IF p_default_parameters IS NOT NULL 
    THEN
      FOR i IN 1..p_default_parameters.COUNT LOOP
      BEGIN
          INSERT INTO MGMT_IP_ELEM_DEFAULT_PARAMS
              (ELEMENT_NAME_NLSID, ELEMENT_TYPE_NLSID, 
                PARAM, VALUE)
          VALUES
              (p_element_name_nlsid, p_element_type_nlsid,
                p_default_parameters(i).param, p_default_parameters(i).value);
      EXCEPTION
          WHEN OTHERS THEN
            dbms_output.put_line('Error inserting default parameter for element definition ' ||
                  p_element_name_nlsid || ':' || 
                  p_element_type_nlsid);
            RAISE;
      END;
      END LOOP;
    END IF;
    
END;


--  ----------------------------------------------------------------------------
--  Add a chart from SQL element definition (for querying against mgmt views)
--   Input:
--        p_element_name_nlsid  element name (nlsid if translated or
--                              display string if not translated)
--        p_element_type_nlsid  element type name (nlsid if translated or
--                              display string if not translated) 
--        p_description_nlsid   description (nlsid if translated or
--                              display string if not translated), or null
--        p_internal_use_only   1=internal use only, 0=available to administrators
--        p_param_classes       list of parameter controller class names
--        p_default_parameters  list of default parameters
--        p_pack_name           License pack assoc with this element defintion
--        p_product_name        Product name
--        p_component_name      Component name
--        p_version             Product version
--        p_query_views         SQL queries against mgmt views (true=1, false=0)
--  ----------------------------------------------------------------------------
PROCEDURE add_chart_from_sql_elem_def (                
                p_element_name_nlsid                IN VARCHAR2,
                p_element_type_nlsid                IN VARCHAR2,
                p_description_nlsid                 IN VARCHAR2,
                p_internal_use_only                 IN NUMBER DEFAULT 0, 
                p_param_classes                     IN MGMT_IP_PARAM_CLASSES  DEFAULT NULL,
                p_default_parameters                IN MGMT_IP_PARAM_VALUE_LIST DEFAULT NULL,
                p_pack_name                         IN VARCHAR2 DEFAULT NULL,
                p_product_name                      IN VARCHAR2 DEFAULT 'EM',
                p_component_name                    IN VARCHAR2 DEFAULT '',
                p_version                           IN VARCHAR2 DEFAULT '10.2.0.1.0',
                p_query_views                       IN NUMBER DEFAULT 1
                ) IS
    l_param_classes MGMT_IP_PARAM_CLASSES;
    l_render_class VARCHAR(1000);
    l_count NUMBER;
BEGIN

  l_count := 1;
  
  l_param_classes := MGMT_IP_PARAM_CLASSES();
  l_param_classes.extend(1);
  l_param_classes(l_count) := 'oracle.sysman.eml.ip.render.elem.HeaderParamController';
  l_count := l_count + 1;
  
  IF p_query_views = 1
  THEN
    l_param_classes.extend(1);
    l_param_classes(l_count) := 'oracle.sysman.eml.ip.render.elem.ChartParamController';
    l_count := l_count + 1;
       
    l_render_class := 'oracle.sysman.eml.ip.render.elem.UserChartRenderController';
  ELSE
    l_render_class := 'oracle.sysman.eml.ip.render.elem.ChartRenderController';
  END IF;
  
  IF p_param_classes IS NOT NULL 
  THEN
      FOR i IN 1..p_param_classes.COUNT LOOP
      BEGIN
          l_param_classes.extend(1);
          l_param_classes(l_count) := p_param_classes(i);
          l_count := l_count + 1;
      END;
      END LOOP;    
  END IF;

  mgmt_ip.add_element_definition(
      p_element_name_nlsid  => p_element_name_nlsid,
      p_element_type_nlsid  => p_element_type_nlsid,
      p_description_nlsid   => p_description_nlsid,
      p_element_class_name  => l_render_class,
      p_internal_use_only   => p_internal_use_only, 
      p_param_classes       => l_param_classes,
      p_default_parameters  => p_default_parameters,
      p_pack_name           => p_pack_name,
      p_product_name        => p_product_name,
      p_component_name      => p_component_name,
      p_version             => p_version
      );
END;

--  ----------------------------------------------------------------------------
--  Add a table from sql element definition (for querying against mgmt views)
--   Input:
--        p_element_name_nlsid  element name (nlsid if translated or
--                              display string if not translated)
--        p_element_type_nlsid  element type name (nlsid if translated or
--                              display string if not translated) 
--        p_description_nlsid   description (nlsid if translated or
--                              display string if not translated), or null
--        p_internal_use_only   1=internal use only, 0=available to administrators
--        p_param_classes       list of parameter controller class names
--        p_default_parameters  list of default parameters
--        p_pack_name           License pack assoc with this element defintion
--        p_product_name        Product name
--        p_component_name      Component name
--        p_version             Product version
--        p_query_views         SQL queries against mgmt views (true=1, false=0)
--  ----------------------------------------------------------------------------
PROCEDURE add_table_from_sql_elem_def (                
                p_element_name_nlsid                IN VARCHAR2,
                p_element_type_nlsid                IN VARCHAR2,
                p_description_nlsid                 IN VARCHAR2,
                p_internal_use_only                 IN NUMBER DEFAULT 0, 
                p_param_classes                     IN MGMT_IP_PARAM_CLASSES  DEFAULT NULL,
                p_default_parameters                IN MGMT_IP_PARAM_VALUE_LIST DEFAULT NULL,
                p_pack_name                         IN VARCHAR2 DEFAULT NULL,
                p_product_name                      IN VARCHAR2 DEFAULT 'EM',
                p_component_name                    IN VARCHAR2 DEFAULT '',
                p_version                           IN VARCHAR2 DEFAULT '10.2.0.1.0',
                p_query_views                       IN NUMBER DEFAULT 1
                ) IS
    l_param_classes MGMT_IP_PARAM_CLASSES;
    l_render_class VARCHAR(1000);
    l_count NUMBER;
BEGIN

  l_count := 1;
  
  l_param_classes := MGMT_IP_PARAM_CLASSES();
  l_param_classes.extend(1);
  l_param_classes(l_count) := 'oracle.sysman.eml.ip.render.elem.HeaderParamController';
  l_count := l_count + 1;

  IF p_query_views = 1
  THEN
    l_param_classes.extend(1);
    l_param_classes(l_count) := 'oracle.sysman.eml.ip.render.elem.SQLStatementParamController';
    l_count := l_count + 1;
    
    l_render_class := 'oracle.sysman.eml.ip.render.elem.UserTableRenderController';
  ELSE

    l_render_class := 'oracle.sysman.eml.ip.render.elem.TableRenderController';
  END IF;
  
  IF p_param_classes IS NOT NULL 
  THEN
      FOR i IN 1..p_param_classes.COUNT LOOP
      BEGIN
          l_param_classes.extend(1);
          l_param_classes(l_count) := p_param_classes(i);
          l_count := l_count + 1;
      END;
      END LOOP;    
  END IF;

  mgmt_ip.add_element_definition(
      p_element_name_nlsid  => p_element_name_nlsid,
      p_element_type_nlsid  => p_element_type_nlsid,
      p_description_nlsid   => p_description_nlsid,
      p_element_class_name  => l_render_class,
      p_internal_use_only   => p_internal_use_only, 
      p_param_classes       => l_param_classes,
      p_default_parameters  => p_default_parameters,
      p_pack_name           => p_pack_name,
      p_product_name        => p_product_name,
      p_component_name      => p_component_name,
      p_version             => p_version
      ); 
END;


--  ----------------------------------------------------------------------------
--  Add image element definition for an image file that ships with EM
--   Input:
--        p_element_name_nlsid  element name (nlsid if translated or
--                              display string if not translated)
--        p_element_type_nlsid  element type name (nlsid if translated or
--                              display string if not translated) 
--        p_description_nlsid   description (nlsid if translated or
--                              display string if not translated), or null
--        p_internal_use_only   1=internal use only, 0=available to administrators
--        p_default_parameters  list of default parameters
--        p_pack_name           License pack assoc with this element defintion
--        p_product_name        Product name
--        p_component_name      Component name
--        p_version             Product version
--  ----------------------------------------------------------------------------
PROCEDURE add_em_image_elem_def (                
                p_element_name_nlsid                IN VARCHAR2,
                p_element_type_nlsid                IN VARCHAR2,
                p_description_nlsid                 IN VARCHAR2,
                p_internal_use_only                 IN NUMBER DEFAULT 0, 
                p_default_parameters                IN MGMT_IP_PARAM_VALUE_LIST DEFAULT NULL,
                p_pack_name                         IN VARCHAR2 DEFAULT NULL,
                p_product_name                      IN VARCHAR2 DEFAULT 'EM',
                p_component_name                    IN VARCHAR2 DEFAULT '',
                p_version                           IN VARCHAR2 DEFAULT '10.2.0.1.0'
                ) IS
    l_param_classes MGMT_IP_PARAM_CLASSES;
BEGIN

    mgmt_ip.add_element_definition(
      p_element_name_nlsid  => p_element_name_nlsid,
      p_element_type_nlsid  => p_element_type_nlsid,
      p_description_nlsid   => p_description_nlsid,
      p_element_class_name  => 'oracle.sysman.eml.ip.render.elem.ImageRenderController',
      p_internal_use_only   => p_internal_use_only, 
      p_param_classes       => null,
      p_default_parameters  => p_default_parameters,
      p_pack_name           => p_pack_name,
      p_product_name        => p_product_name,
      p_component_name      => p_component_name,
      p_version             => p_version
      ); 

END;

--  ----------------------------------------------------------------------------
--  Register a named SQL statement
--  Input:
--      p_product_name        Product name
--      p_component_name      Component name
--      p_version             Product version
--      p_name                Statement name (must be unique)
--      p_sql_statement       SQL statement text
--  ----------------------------------------------------------------------------
PROCEDURE register_sql_statement (
                p_product_name                      IN VARCHAR2 DEFAULT NULL,
                p_component_name                    IN VARCHAR2 DEFAULT NULL,
                p_version                           IN VARCHAR2 DEFAULT NULL,
                p_name                              IN VARCHAR2 DEFAULT NULL,
                p_sql_statement                     IN VARCHAR2 DEFAULT NULL
                ) IS
BEGIN

    -- add this sql statement to MGMT_IP_SQL_STATEMENTS
    BEGIN
      INSERT INTO MGMT_IP_SQL_STATEMENTS
          (PRODUCT_NAME, COMPONENT_NAME, VERSION,
            NAME, SQL_STATEMENT)
      VALUES
          (p_product_name, p_component_name, p_version,
           p_name, p_sql_statement);
      
    EXCEPTION
          WHEN OTHERS THEN 
            RAISE;
    END;

END;

-- --------------------------------------------------------------------------------
-- IP INTERNAL USE ONLY:  
-- grant_report_to_user() 
--
--  user_name_in    EM username
--  report_guid_in  report GUID
-- --------------------------------------------------------------------------------

PROCEDURE grant_report_to_user (user_name_in IN VARCHAR2,
                                report_guid_in IN RAW)
IS
  uname VARCHAR2(256) := UPPER(user_name_in);
  usave VARCHAR2(256 ):= MGMT_USER.GET_CURRENT_EM_USER;
  
BEGIN
    SETEMUSERCONTEXT(MGMT_USER.GET_REPOSITORY_OWNER,
                     MGMT_USER.OP_SET_IDENTIFIER);
    mgmt_user.grant_priv(uname,
                         MGMT_USER.VIEW_REPORT, 
                         report_guid_in);
    SETEMUSERCONTEXT(usave, MGMT_USER.OP_SET_IDENTIFIER);
END GRANT_REPORT_TO_USER;


-- --------------------------------------------------------------------------------
-- IP INTERNAL USE ONLY:  
-- determine if a user has the publish to public system privilege
-- user_has_publish_to_public() 
--
--  user_name_in    EM username
-- --------------------------------------------------------------------------------

FUNCTION USER_HAS_PUBLISH_TO_PUBLIC (user_name_in IN VARCHAR2)
RETURN NUMBER
IS
  uname VARCHAR2(256) := UPPER(user_name_in);
  usave VARCHAR2(256 ):= MGMT_USER.GET_CURRENT_EM_USER;
  retcode NUMBER := 0;
  
BEGIN
    SETEMUSERCONTEXT(MGMT_USER.GET_REPOSITORY_OWNER,
                     MGMT_USER.OP_SET_IDENTIFIER);
    IF MGMT_USER.HAS_PRIV(uname, MGMT_USER.PUBLISH_REPORT) = MGMT_USER.USER_HAS_PRIV
    THEN
      retcode := 1;
    ELSE
      retcode := 0;
    END IF;
    SETEMUSERCONTEXT(usave, MGMT_USER.OP_SET_IDENTIFIER);
    RETURN retcode;
END USER_HAS_PUBLISH_TO_PUBLIC;

-- -----------------------------------------------------------------------------
-- IP INTERNAL USE ONLY:  
-- get_report_grants
--   
--
--  report_guid_in  report GUID
--  owner_out       The report owner
--  grantees_out    Cursor to retrieve list of users who have VIEW_REPORT
--                  on supplied report_guid
-- -----------------------------------------------------------------------------

PROCEDURE get_report_grants (report_guid_in IN RAW,
                             owner_out OUT VARCHAR2,
                             grantees_out OUT GRANT_CURSOR)
IS
  usave VARCHAR2(256 ):= MGMT_USER.GET_CURRENT_EM_USER;
  
BEGIN
    SETEMUSERCONTEXT(MGMT_USER.GET_REPOSITORY_OWNER,
                     MGMT_USER.OP_SET_IDENTIFIER);
    MGMT_USER.GET_REPORT_GRANTS(report_guid_in, owner_out, grantees_out);
    SETEMUSERCONTEXT(usave, MGMT_USER.OP_SET_IDENTIFIER);
END get_report_grants;


--  ----------------------------------------------------------------------------
--
--  IP INTERNAL USE ONLY:  
--  check_modify_report()
--
--  Throw an exception if the current user does not have privileges to
--  perform the current operation.
--  If p_allow_superuser is TRUE, then superusers will be able to 
--  perform the operation. If set to FALSE, then the user will 
--  be able to perform the operation only if the user is the owner of the report.
--
--  ----------------------------------------------------------------------------
PROCEDURE check_modify_report(report_id_in       RAW,
                              allow_superuser_in BOOLEAN DEFAULT TRUE) IS
report_owner VARCHAR2(256);
current_user VARCHAR2(256) := MGMT_USER.get_current_em_user();

BEGIN
    SELECT owner INTO report_owner FROM MGMT_IP_REPORT_DEF
       WHERE report_guid = report_id_in;

    IF current_user = report_owner THEN
        RETURN;
    END IF;

--  TODO:  Complete the following privilege check code
--
--    check_priv(MGMT_USER.FULL_JOB, current_user, 
--               report_id_in, allow_superuser_in);

--  IMPLEMENT THE EXCEPTION HANDLING CODE:
--      mgmt_global.invalid_job_err is an exception defined in 
--      emdrep/sql/core/latest/sdk/sdk_global_pkgdef.sql.   Reporting 
--      PL/SQL exceptions will have to be added here as well.
--      -----------------------------------------------------
--EXCEPTION
--    WHEN NO_DATA_FOUND THEN
--        raise_application_error(MGMT_GLOBAL.INVALID_JOB_ERR,
--          'The specified job/execution does not exist');

END check_modify_report ;    


PROCEDURE delete_report_by_title (report_owner_in    IN  VARCHAR2,
                                  report_title_in    IN  VARCHAR2,
                                  check_security     IN  INTEGER)
IS
   l_report_guid        RAW(16);

BEGIN

   SELECT def.report_guid INTO l_report_guid
   FROM MGMT_IP_REPORT_DEF def
   WHERE def.title_nlsid = report_title_in
     AND def.owner = report_owner_in;

   delete_report(l_report_guid, check_security);

END delete_report_by_title;

--  ----------------------------------------------------------------------------
--  IP INTERNAL USE ONLY:  
--   Delete a report (all versions and copies of the report)
--  ----------------------------------------------------------------------------
PROCEDURE delete_report (report_id_in       IN  VARCHAR2, 
                         check_security     IN  INTEGER)
                         
IS
    element_id_list     MGMT_USER_GUID_ARRAY; 
    purge_id            RAW(16);
    job_id              RAW(16);
    
BEGIN

    SELECT element_guid BULK COLLECT INTO element_id_list
    FROM MGMT_IP_REPORT_DEF_ELEMENTS elem
    WHERE elem.report_guid = report_id_in;
    
    --  Loop though and delete individual report elements
    --
    FOR i IN 1..element_id_list.COUNT 
    LOOP    
        delete_report_element( report_id_in, element_id_list(i) );        
    END LOOP;
    
    --  Delete the report purge policy
    --
    SELECT purge_policy_guid into purge_id from MGMT_IP_REPORT_DEF WHERE report_guid = report_id_in;
    IF purge_id IS NOT NULL
    THEN
        DELETE FROM MGMT_IP_PURGE_POLICY WHERE purge_policy_guid = purge_id;
    END IF;    
    
    --  Delete report email options 
    --
    DELETE FROM MGMT_IP_EMAIL_REPORT WHERE REPORT_GUID = report_id_in;
    
    -- Delete JIT target types    
    --
    DELETE FROM MGMT_IP_REPORT_DEF_JIT_TYPES WHERE REPORT_GUID = report_id_in;

    --  Delete the report access rules
    --  We revoke access to the report to everyone
    -- This is done by the mgmt_user package
    --     -- jaa
    
    --  The report publish job is deleted from java since any current executions
    --  need to be stopped first.   
    --
    
    --  Finally, delete the report definition record
    --
    DELETE FROM MGMT_IP_REPORT_DEF WHERE report_guid = report_id_in;

    -- Call the user model
    MGMT_USER.REPORT_DEFINITION_DELETED(report_id_in);

END delete_report;

--  ----------------------------------------------------------------------------
--  IP INTERNAL USE ONLY:  
--   Delete a single report element given its GUID
--  ----------------------------------------------------------------------------
PROCEDURE delete_report_element (report_id_in  VARCHAR2,
                                 element_id_in RAW)

IS
BEGIN
    --  Delete the element parameters
    DELETE FROM MGMT_IP_REPORT_ELEM_PARAMS 
    WHERE report_guid  = report_id_in
      AND element_guid = element_id_in;
    
    --  Delete the targets selected for rendering the element 
    DELETE FROM MGMT_IP_REPORT_ELEM_TARGETS 
    WHERE report_guid  = report_id_in
      AND element_guid = element_id_in;
    
-- ???   Is this correct?   Think about what stored data (all versions?) get 
--       deleted when the element is deleted.
    --  Delete the element's stored data 
    --
    DELETE FROM MGMT_IP_STORED_REPORT 
    WHERE report_guid  = report_id_in
      AND element_guid = element_id_in;

    --  Delete the report definition element record
    DELETE FROM MGMT_IP_REPORT_DEF_ELEMENTS
    WHERE report_guid  = report_id_in
      AND element_guid = element_id_in;
    
    
END delete_report_element;

--  ----------------------------------------------------------------------------
--  IP INTERNAL USE ONLY:  
--   Delete a single report version, specifically the report version's stored
--   data.
--  ----------------------------------------------------------------------------

PROCEDURE delete_report_by_version (report_id_in    IN  VARCHAR2,
                                    version_id_in   IN  VARCHAR2,
                                    check_security  IN  INTEGER)
IS
latest_pub DATE;
BEGIN

--    IF check_security = 1 THEN
--        check_modify_report (report_id_in);
--    END IF;

    DELETE FROM MGMT_IP_STORED_REPORT 
    WHERE report_guid  = report_id_in
      AND version_guid = version_id_in;

    SELECT max(generated_date) INTO latest_pub
      FROM MGMT_IP_STORED_REPORT
    WHERE  report_guid = report_id_in;

    UPDATE MGMT_IP_REPORT_DEF
       SET last_published = latest_pub
    WHERE  report_guid = report_id_in;
   

END delete_report_by_version;


--  ----------------------------------------------------------------------------
--   High level entrypoint for purge of EM reports
--  ----------------------------------------------------------------------------
PROCEDURE do_purge (pcb_params IN OUT MGMT_PURGE_CALLBACK_PARAMS)

IS
    report_id         RAW(16);
    purge_policy_id   RAW(16);
    num_versions      NUMBER := 0;
    num_intervals     NUMBER := 0;
    interval_type     NUMBER := 0;
    rows_purged       NUMBER := 0;
    
    CURSOR policyCursor IS
        SELECT REPORT_GUID, PURGE_POLICY_GUID 
        FROM MGMT_IP_REPORT_DEF
        WHERE PURGE_POLICY_GUID IS NOT NULL;                
    
BEGIN
    OPEN policyCursor;
    LOOP
        FETCH policyCursor INTO report_id, purge_policy_id;
        EXIT WHEN policyCursor%NOTFOUND;
        
        
        SELECT KEEP_NUM_VERSIONS, RETENTION_INTERVAL, NUM_INTERVALS into num_versions, interval_type, num_intervals
        FROM MGMT_IP_PURGE_POLICY
        WHERE PURGE_POLICY_GUID = purge_policy_id;
    
        --  Purge by number of verstions to keep
        --
        IF num_versions > 0
        THEN
          purge_by_keep_versions(report_id, 
                                 num_versions,
                                 rows_purged);          
        END IF;
        
        --  Purge by number of intrevals to keep (intervals can be days, weeks, months)
        --        
        IF num_intervals > 0
        THEN
          purge_by_keep_days(report_id, 
                        num_intervals,
                        interval_type,
                        rows_purged);
        END IF;
    END LOOP;
    
    -- Close the cursor if open
    IF (policyCursor%ISOPEN) THEN
      CLOSE policyCursor;
    END IF;    
    
    pcb_params.callback_result := 0;
    pcb_params.rows_processed  := rows_purged;
    pcb_params.error_code      := 0;
    pcb_params.error_msg       := NULL;
    
END do_purge;

PROCEDURE purge_by_keep_versions (report_id_in    IN  RAW, 
                                  num_versions_in IN  NUMBER,
                                  rows_purged     OUT NUMBER)
IS
  version_id               RAW(16);
  jobID                    RAW(16);
  gen_date                 DATE;
  sched_date               DATE;
  version_count            NUMBER := 1;
  del_job_exec_older_than  DATE;
  formatted_date           VARCHAR2(50);
  formattedSchedDate       VARCHAR2(50); 
  
  CURSOR reportVerionCursor IS
    select distinct version_guid, to_char(generated_date, 'mm-dd-yy hh:mi am') formatted_date, generated_date gen_date
    from MGMT_IP_STORED_REPORT
    where report_guid = report_id_in
    order by gen_date DESC; 

  CURSOR purgeRunsCursor IS
    select scheduled_time 
    from MGMT_JOB_EXEC_SUMMARY
    where job_id = jobID
      and scheduled_time < del_job_exec_older_than;    
      
BEGIN

    select schedule_guid into jobID 
    from MGMT_IP_REPORT_DEF
    where report_guid = report_id_in;
    
    rows_purged := 0;
    
    OPEN reportVerionCursor;
    LOOP
        FETCH reportVerionCursor INTO version_id, formatted_date, gen_date;
        EXIT WHEN reportVerionCursor%NOTFOUND;
        

        IF version_count > num_versions_in
        THEN
          --  If this is the first stored version to delete save the generated date to 
          --  use for deleting assoicated job runs.
          IF  rows_purged = 0
          THEN 
            del_job_exec_older_than := gen_date;
          END IF;
          
          --  Delete one stored report
          --
          DELETE FROM MGMT_IP_STORED_REPORT 
          WHERE REPORT_GUID    = report_id_in
            AND VERSION_GUID   = version_id
            AND GENERATED_DATE = gen_date; 
            
          rows_purged := rows_purged + 1;
          
        END IF;
        version_count := version_count + 1;  
    END LOOP;
    
    --  Now purge the job runs associated with the stored reports deleted above.
    --
    OPEN purgeRunsCursor;
    LOOP
        FETCH purgeRunsCursor INTO sched_date;
        EXIT WHEN purgeRunsCursor%NOTFOUND;

        --  Delete the job run 
        MGMT_JOB_ENGINE.delete_job_run (jobID, sched_date);        

    END LOOP;    
    
    -- Close the cursor if open
    --
    IF (purgeRunsCursor%ISOPEN) 
    THEN
      CLOSE purgeRunsCursor;
    END IF;     

    -- Close the cursor if open
    --
    IF (reportVerionCursor%ISOPEN) 
    THEN
      CLOSE reportVerionCursor;
    END IF;    
    
END purge_by_keep_versions;

-- purge_by_keep_days() - purge by an interval of days 
--
PROCEDURE purge_by_keep_days (report_id_in    IN  RAW, 
                              num_intervals   IN  NUMBER,
                              interval_type   IN  NUMBER,
                              rows_purged     OUT NUMBER)
IS
  jobID           RAW(16);
  schedDate       DATE;
  del_job_exec_older_than DATE;


  CURSOR purgeRunsCursor IS
    select scheduled_time 
    from MGMT_JOB_EXEC_SUMMARY
    where job_id = jobID
      and scheduled_time < sysdate - (interval_type * num_intervals); 
    
BEGIN
    rows_purged := 0;
  
    SELECT SYSDATE - (interval_type * num_intervals) INTO  del_job_exec_older_than FROM DUAL;
    
    SELECT count (*) into rows_purged 
    FROM MGMT_IP_STORED_REPORT
    WHERE REPORT_GUID = report_id_in
      AND GENERATED_DATE < SYSDATE - (interval_type * num_intervals) ;
    
    SELECT schedule_guid into jobID 
    FROM  MGMT_IP_REPORT_DEF
    WHERE report_guid = report_id_in;
    
    --  Purge the stored reports
    --
    DELETE FROM MGMT_IP_STORED_REPORT 
    WHERE REPORT_GUID = report_id_in
      AND GENERATED_DATE < SYSDATE - (interval_type * num_intervals) ;

    OPEN purgeRunsCursor;
    LOOP
        FETCH purgeRunsCursor INTO schedDate;
        EXIT WHEN purgeRunsCursor%NOTFOUND;
        --  Delete the job run 
        MGMT_JOB_ENGINE.delete_job_run (jobID, schedDate);
    END LOOP;
    
END purge_by_keep_days;


-- get alert cursors
--
PROCEDURE get_alert_data (
                tgt_guid_in              IN  MGMT_IP_TGT_GUID_ARRAY,
                severities_in            IN  MGMT_IP_STRING_ARRAY,
                metric_alert_cursor_out  OUT CURSOR_TYPE,
                avail_alert_cursor_out   OUT CURSOR_TYPE,
                metric_error_cursor_out  OUT CURSOR_TYPE)
IS
  query_metric_alerts   VARCHAR(6000);
  query_avail_alerts   VARCHAR(6000);
  query_metric_errors   VARCHAR(6000);

BEGIN
    -- metric alerts (critical, warning)
    --
    query_metric_alerts := getMetricAlertsSQL(tgt_guid_in, severities_in);
    IF query_metric_alerts is NOT NULL
    THEN
      OPEN metric_alert_cursor_out for query_metric_alerts;
    END IF;

    -- availability alerts (blackout, unreachable)
    --
    query_avail_alerts := getAvailAlertsSQL(tgt_guid_in, severities_in);
    IF query_avail_alerts is NOT NULL
    THEN
      OPEN avail_alert_cursor_out for query_avail_alerts;
    END IF;

    -- metric errors
    --
    query_metric_errors := getMetricErrorAlertSQL(tgt_guid_in, severities_in);
    IF query_metric_errors is NOT NULL
    THEN
      OPEN metric_error_cursor_out for query_metric_errors;
    END IF;

END get_alert_data;



  

-- Returns query to get metric alerts
--
--            // Column Mapping:
--            // 1=target name, 2=target_type,  3=target_guid
--            // 4=target display name, 5=target display type
--            // 6=severity code
--            // 7=duration 
--            // 8=message
--            // 9=timestamp string
--            // 10=timezone delta
--            // 11=metric_column
--            // 12=metric_name
--            // 13=key_value
--
FUNCTION getMetricAlertsSQL (
                tgt_guid_in        IN  MGMT_IP_TGT_GUID_ARRAY,
                severities_in      IN  MGMT_IP_STRING_ARRAY)
RETURN VARCHAR2 
IS
    query       VARCHAR(5000) := '';
    select_clause  VARCHAR(2000) := 'SELECT distinct ' ||
      ' tgt.target_name, ' ||
      ' tgt.target_type, ' ||
      ' tgt.target_guid, ' ||
      ' tgt.target_name, ' ||
      ' tgt.type_display_name, ' ||
      ' cs.severity_code, '||
      ' round((tgt_sysdate_wguid(tgt.target_guid) - ' ||
      '     cs.collection_timestamp)*24*60, 0), ' ||
      ' cs.message, ' ||
      ' cs.collection_timestamp, ' ||
      ' tgt.timezone_delta/60, ' ||
      ' met.metric_column, ' ||
      ' met.metric_name, ' ||
      ' cs.key_value ';
    from_clause VARCHAR(500) := ' FROM ' ||
      ' mgmt_targets tgt, ' ||
      ' mgmt_metrics met, ' ||
      ' mgmt_current_severity cs ';
    
    where_clause VARCHAR(2000) := 'WHERE ' ||    
      ' not (met.metric_name = ''Response'' ' ||
      '    and met.metric_column = ''Status'') ' || 
      ' and tgt.target_guid = cs.target_guid ' ||
      ' and met.metric_guid = cs.metric_guid ' ;
      
    where_severity VARCHAR(500) := ' ';

    where_targets VARCHAR(2000) := ' ';

    order_clause VARCHAR(200) := ' ORDER BY tgt.target_guid ';
  
    l_alert_severities    MGMT_IP_STRING_ARRAY;
    l_index               NUMBER := 1;
BEGIN

    l_alert_severities := MGMT_IP_STRING_ARRAY();
    l_index := 1;
    
    FOR i IN 1..severities_in.COUNT LOOP
      -- check for EmlConstants.SEV_CRITICAL
      IF severities_in(i) = '25' THEN
        l_alert_severities.extend(1);
        l_alert_severities(l_index) := '25';
        l_index := l_index + 1;
      END IF;

      -- check for EmlConstants.SEV_WARNING
      IF severities_in(i) = '20' THEN
        l_alert_severities.extend(1);
        l_alert_severities(l_index) := '20';
        l_index := l_index + 1;
      END IF;
    END LOOP;
    
    IF l_alert_severities.COUNT = 0
    THEN
      return null;
    END IF;

    where_targets := getTargetWhere(tgt_guid_in);
    
    IF (l_alert_severities.COUNT = 1)
    THEN
      where_severity := ' and cs.severity_code = ''' || l_alert_severities(1) || '''';    
    ELSE
      where_severity := ' and cs.severity_code in (';

      FOR i IN 1..l_alert_severities.COUNT
      LOOP
        IF i > 1
        THEN
          where_severity := where_severity || ', ';
        END IF;
        where_severity := where_severity || l_alert_severities(i);
      END LOOP;
      
      where_severity := where_severity || ') ';
    
    END IF;
    
    
   
    return select_clause || from_clause  || where_clause || where_severity || where_targets || order_clause; 

END getMetricAlertsSQL;


--            // Column Mapping:
--            // 1=target name, 2=target_type,  3=target_guid
--            // 4=target display name, 5=target display type
--            // 6=avail current status 
--            // 7=duration 
--            // 8=message
--            // 9=timestamp string
--            // 10=timezone delta
--
FUNCTION getAvailAlertsSQL (
                tgt_guid_in              IN  MGMT_IP_TGT_GUID_ARRAY,
                severities_in            IN  MGMT_IP_STRING_ARRAY)
RETURN VARCHAR2 
IS
    l_index               NUMBER := 1;
    l_avail_severities    MGMT_IP_STRING_ARRAY;

    query       VARCHAR(5000) := '';
    select_clause  VARCHAR(2000) := 'SELECT ' ||
        ' tgt.target_name, ' ||
        ' tgt.target_type, ' ||
        ' tgt.target_guid, ' || 
        ' tgt.target_name, ' ||
        ' tgt.type_display_name, ' ||
        ' avail.current_status, ' ||
        ' round((tgt_sysdate_wguid(tgt.target_guid) - ' ||
        '     avail.start_collection_timestamp)*24*60, 0), ' ||
        ' '' '' message, ' ||
        ' avail.start_collection_timestamp collection_timestamp_string, ' ||
        ' tgt.timezone_delta/60 ';

    from_clause VARCHAR(500) := ' FROM ' ||
        ' mgmt_targets tgt, ' ||
        ' mgmt_current_availability avail ';
    
    where_clause VARCHAR(2000) := 'WHERE ' ||    
        ' tgt.target_guid = avail.target_guid ';
      
    where_severity VARCHAR(500) := ' ';

    where_targets VARCHAR(2000) := ' ';

    order_clause VARCHAR(200) := ' ORDER BY tgt.target_guid ';

BEGIN
  l_avail_severities := MGMT_IP_STRING_ARRAY();
  
  l_index := 1;

  -- Blackout, Unreachable severities
  --
  FOR i IN 1..severities_in.COUNT
  LOOP
    -- blackout start (ignore) (EmlConstants.SEV_BLACKOUT_START)
    IF severities_in(i) = '225'
    THEN
      -- EmlConstants.AVAIL_BLACKOUT
      l_avail_severities.extend(1);
      l_avail_severities(l_index) := '5';
      l_index := l_index + 1;
    END IF;

    -- EmlConstants.SEV_UNREACHABLE_START
    IF severities_in(i) = '125' 
    THEN
      -- EmlConstants.AVAIL_ERROR
      l_avail_severities.extend(1);
      l_avail_severities(l_index) := '2';
      l_index := l_index + 1;
      -- EmlConstants.AVAIL_AGENT_DOWN
      l_avail_severities.extend(1);
      l_avail_severities(l_index) := '3';
      l_index := l_index + 1;
      -- EmlConstants.AVAIL_UNREACHABLE
      l_avail_severities.extend(1);
      l_avail_severities(l_index) := '4';
      l_index := l_index + 1;
      -- EmlConstants.AVAIL_UNKNOWN
      l_avail_severities.extend(1);
      l_avail_severities(l_index) := '6';
      l_index := l_index + 1;

    END IF;
  END LOOP;
    
    IF l_avail_severities.COUNT = 0 
    THEN
      return null;
    END IF;
  
    where_targets := getTargetWhere(tgt_guid_in);
    
    IF l_avail_severities.COUNT = 1
    THEN
      where_severity := ' and avail.current_status = ''' || l_avail_severities(1) || '''';    
    ELSE
      where_severity := ' and avail.current_status in (';

      FOR i IN 1..l_avail_severities.COUNT
      LOOP
        IF (i > 1)
        THEN
          where_severity := where_severity || ', ';
        END IF;
        where_severity := where_severity || l_avail_severities(i);
      END LOOP;
      
      where_severity := where_severity || ') ';
    
    END IF;
    
    return select_clause || from_clause  || where_clause || where_severity || where_targets || order_clause; 
  

END getAvailAlertsSQL;

--            // Column Mapping:
--            // 1=target name, 2=target_type,  3= target_guid
--            // 4=target display name, 5=target display type
--            // 6=duration 
--            // 7=message
--            // 8=timestamp string
--            // 9=timezone delta
--            // 10=metric name
--            // 11=metric column
--
FUNCTION getMetricErrorAlertSQL (
                tgt_guid_in        IN  MGMT_IP_TGT_GUID_ARRAY,
                severities_in      IN  MGMT_IP_STRING_ARRAY)
RETURN VARCHAR2 
IS
    l_count               NUMBER := 0;

    select_clause  VARCHAR(2000) := 'SELECT  ' ||
        ' tgt.target_name, ' ||
        ' tgt.target_type, ' ||
        ' tgt.target_guid, ' ||
        ' tgt.target_name, ' ||
        ' tgt.type_display_name, '||
        ' round((tgt_sysdate_wguid(tgt.target_guid) - ' ||
        '        err.collection_timestamp)*24*60, 0), ' ||
        ' err.metric_error_message, ' ||
        ' err.collection_timestamp collection_timestamp_string, ' ||
        ' tgt.timezone_delta/60, ' ||
        ' met.metric_name, ' ||
        ' met.metric_column ';

    from_clause VARCHAR(500) := ' FROM ' ||
        ' mgmt_targets tgt, ' ||
        ' mgmt_metrics met, ' ||
        ' mgmt_current_metric_errors err ';
    
    where_clause VARCHAR(2000) := 'WHERE ' ||    
        ' err.target_guid=tgt.target_guid and ' ||
        ' err.metric_guid=met.metric_guid AND ' ||
        ' err.metric_error_message IS NOT NULL ';
      
    where_targets VARCHAR(2000) := ' ';

    order_clause VARCHAR(200) := ' ORDER BY tgt.target_guid ';


BEGIN
   
    FOR i IN 1..severities_in.COUNT LOOP
      -- EmlConstants.SEV_METRIC_ERROR_START
      IF severities_in(i) = '325' THEN
        l_count := l_count + 1;
      END IF;
    END LOOP;
    
    IF l_count = 0 
    THEN
      return null;
    END IF;

    where_targets := getTargetWhere(tgt_guid_in);

    return select_clause || from_clause  || where_clause || where_targets || order_clause; 
    
END getMetricErrorAlertSQL;


FUNCTION getTargetWhere (
                tgt_guid_in        IN  MGMT_IP_TGT_GUID_ARRAY)
RETURN VARCHAR2 
IS
    where_targets VARCHAR(2000) := ' ';
BEGIN
  IF tgt_guid_in.COUNT > 0 THEN
    
      IF tgt_guid_in.COUNT = 1 
      THEN
        where_targets := ' and tgt.target_guid = HEXTORAW(''' ||
          tgt_guid_in(1) || ''') ';
      ELSE
        where_targets := 'and (';
        
        FOR i IN 1..tgt_guid_in.COUNT
        LOOP 
          where_targets := where_targets || ' tgt.target_guid = HEXTORAW(''' || tgt_guid_in(i) || ''') ' ;
          IF i < tgt_guid_in.COUNT
          THEN
            where_targets := where_targets || ' or ' ;
          END IF;
        END LOOP;
        
        where_targets := where_targets || ') ';

      END IF;
    
    END IF;

  return where_targets;
END getTargetWhere;

--
-- PROCEDURE : get_user_report_defs
--
-- PURPOSE
--
--   this is a callback which will be registered to the user_model pkg 
--   user_model pkg will call this procedure to get a list of report_defs which the 
--   user has
--  
-- PARAMETERS
--
--  user_name_in     - name of the user
--  user_objects_out - list of report_defs will be appended to user_objects_out
--  type_in          - type of user model callback
--
-- NOTES
--
  
PROCEDURE get_user_report_defs(user_name_in      IN VARCHAR2,
                               user_objects_out  OUT MGMT_USER_OBJECTS,
                               type_in           IN NUMBER) IS

l_user_name       VARCHAR2(256):= UPPER(TRIM(user_name_in));

i INTEGER := 0;
l_user_report_defs MGMT_USER_OBJECTS := MGMT_USER_OBJECTS();

CURSOR c_report_defs IS SELECT title_nlsid
FROM MGMT_IP_REPORT_DEF WHERE owner=l_user_name;
BEGIN
    FOR c IN c_report_defs
    LOOP
        l_user_report_defs.extend(1);
        i := i + 1;
        l_user_report_defs(i) := MGMT_USER_OBJECT(MGMT_USER.USER_OBJECT_REPORT_DEF,
                                                c.title_nlsid,
                                                null,
                                                MGMT_USER.SYNC_DROP_OBJECT);
    END LOOP;
    user_objects_out := l_user_report_defs;
END;

--
-- PROCEDURE : drop_user_report_defs
--
-- PURPOSE
--
--   this is a callback which will be registered to the user_model pkg 
--   user_model pkg will call this procedure to delete the report_defs
--   owned by a user.
--  
-- PARAMETERS
--
--  user_name_in     - name of the user
--  type_in          - type of user model callback
--
-- NOTES
--
PROCEDURE drop_user_report_defs(user_name_in      IN VARCHAR2,
                                type_in           IN NUMBER) IS
  
  l_user_name       VARCHAR2(256):= UPPER(TRIM(user_name_in));
                             
  CURSOR c_report_defs IS SELECT report_guid, title_nlsid
     FROM MGMT_IP_REPORT_DEF WHERE owner=l_user_name;
BEGIN 
    
    FOR c IN c_report_defs
    LOOP
        BEGIN
            delete_report(c.report_guid,1);
        EXCEPTION
        WHEN OTHERS THEN
            mgmt_log.log_error(MGMT_USER.USERMODEL_MODULE_NAME,
                               MGMT_GLOBAL.CALLBACK_FAILED_ERR,
                               'drop_user_report_defs ' || l_user_name || 
                               ' : ' || c.title_nlsid || ' : ' || SQLERRM);
        END;
    END LOOP;
END;


--
-- PROCEDURE : reassign_user_report_defs
--
-- PURPOSE
--
--   this is a callback which will be registered to the user_model pkg 
--   user_model pkg will call this procedure to reassign the report_defs
--   owned by a user to a new user.
--  
-- PARAMETERS
--
--  user_name_in     - name of the user
--  new_user_name_in - the new user name
--  type_in          - type of user model callback
--
-- NOTES
--
PROCEDURE reassign_user_report_defs(user_name_in      IN VARCHAR2,
                                    new_user_name_in  IN VARCHAR2,
                                    type_in           IN NUMBER) IS
  
  l_user_name       VARCHAR2(256) := UPPER(TRIM(user_name_in));
  l_new_user_name   VARCHAR2(256) := UPPER(TRIM(new_user_name_in));
                             
  CURSOR c_report_defs IS SELECT report_guid
     FROM MGMT_IP_REPORT_DEF WHERE owner=l_user_name;
BEGIN
    
    FOR c IN c_report_defs
    LOOP
        UPDATE MGMT_IP_REPORT_DEF
        SET    owner = l_new_user_name 
        WHERE  report_guid = c.report_guid;
        
        MGMT_USER.GRANT_PRIV(l_new_user_name, MGMT_USER.VIEW_REPORT, c.report_guid);
        
    END LOOP;
END;


END MGMT_IP;
/
show errors;


