Rem drv: <create type="pkgdefs" pos="rep_metric/rep_metric_pkgdef.sql+"/>
Rem
Rem $Header: notification_pkgdefs.sql 18-jul-2007.04:23:33 denath Exp $
Rem
Rem notification_pkgdefs.sql
Rem
Rem Copyright (c) 2002, 2007, Oracle. All rights reserved.  
Rem
Rem    NAME
Rem      notification_pkgdefs.sql - <one-line expansion of the name>
Rem
Rem    DESCRIPTION
Rem     This is the PL/SQL implementation of the Notification Manager
Rem      component of the Mozart Project
Rem
Rem    NOTES
Rem      Performance data logged by this package is gathered by the mgmt_admin package (self_monitor)
Rem
Rem    MODIFIED (MM/DD/YY)
Rem    denath    06/29/07 - Bug fix 5983944.Added
Rem                         DEF_DEV_PING_STATE_TIMEOUT.
Rem    neearora  06/20/07 - bug 6142074
Rem    neearora  05/03/07 - added API to get set global repeat settings
Rem    neearora  03/27/07 - added procedure QUEUE_NOTIF_INPUT
Rem    neearora  10/21/05 - Added support for Java notifications
Rem    dcawley   07/13/05 - Add callback for key value cleanup 
Rem    scgrover  07/07/05 - add extended sql trace 
Rem    gsbhatia  07/01/05 - New repmgr header impl 
Rem    gsbhatia  06/26/05 - New repmgr header impl 
Rem    dcawley   06/22/05 - Add array version for setting gateways 
Rem    jriel     05/17/05 - access to getqueue 
Rem    dcawley   03/24/05 - Set additional gateway parameters 
Rem    jriel     03/15/05 - queue rca task with normal type 
Rem    gsbhatia  02/13/05 - updating repmgr header 
Rem    gsbhatia  02/09/05 - updating repmgr header 
Rem    gsbhatia  02/07/05 - updating repmgr header 
Rem    dcawley   02/02/05 - Pass locale from OMS 
Rem    ramalhot  02/02/05 - changed signature for handle_target_deleted 
Rem    ktlaw     01/11/05 - add repmgr header 
Rem    dcawley   01/11/05 - Rename callback 
Rem    dcawley   01/06/05 - Add target deletion callback 
Rem    dcawley   12/16/04 - Add RCA notification type 
Rem    dcawley   12/03/04 - Do not use nlsid for plsql notifications 
Rem    dcawley   11/23/04 - Use exisiting type 
Rem    dcawley   10/22/04 - Add api for removing specific configurations 
Rem    dcawley   10/11/04 - Include target guid for groups 
Rem    jriel     09/28/04 - add proc to queue rca directly 
Rem    dcawley   09/27/04 - Change parameter for queuing violation 
Rem    dcawley   09/15/04 - Switch to violations table 
Rem    dcawley   09/06/04 - Add context to severity details 
Rem    jriel     09/08/04 - change params to queue_rca_retry 
Rem    dcawley   08/19/04 - Add os parameter to register oms 
Rem    jriel     08/06/04 - add queue rca 
Rem    jriel     07/20/04 - add RCA 
Rem    dcawley   07/22/04 - Allow states per metric 
Rem    dcawley   07/21/04 - Remove extra job 
Rem    dcawley   07/16/04 - Define constants for Corrective Actions 
Rem    dcawley   07/12/04 - Add procedure for getting job state details 
Rem    dcawley   06/14/04 - Changes for 10gR2 
Rem    aholser   05/13/04 - define types for performance monitoring 
Rem    dcawley   03/25/04 - Add timezone to schedule 
Rem    yaofeng   11/12/03 - show date in NLS format 
Rem    dcawley   10/16/03 - Add error for failing to find severity details 
Rem    dcawley   09/08/03 - Use timezone_region 
Rem    dcawley   09/03/03 - Add log_history 
Rem    dcawley   08/27/03 - Add new procedures for schedules 
Rem    rpinnama  07/25/03 - Move Queue initialization to post_creation
Rem    dcawley   07/23/03 - Log device specific delivery times
Rem    dcawley   07/18/03 - Change annotations
Rem    dcawley   07/16/03 - Change number of queues
Rem    dcawley   06/26/03 - Add UNREGISTER_OMS
Rem    dcawley   05/21/03 - Changes for moving delivery to OMS

CREATE OR REPLACE PACKAGE EMD_NOTIFICATION
IS

-- The module name for error and performance logging
CHECK_MODULE_NAME constant varchar2(50) := 'EMD_NOTIFICATION.HandleSeverities Subsystem';
DELIVER_MODULE_NAME constant varchar2(50) := 'EMD_NOTIFICATION.NotificationDelivery Subsystem';

GUID_TYPE_SEVERITY constant varchar2(1) := '0';
GUID_TYPE_JOB_STATE_CHANGE constant varchar2(1) := '1';
GUID_TYPE_REP_SEVERITY constant  varchar2(1) := '2';

-- These are used as the 'key column' for the notification methods performance 
-- metrics monitored by the emrep target. If new methods are added, they should 
-- also be added to this array and nls'd in 
-- source/oracle/sysman/resources/health/HealthResourceBundle with a key 
-- defined in source/oracle/sysman/health/HealthConstants
METHOD_NAMES constant SMP_EMD_STRING_ARRAY :=
                      SMP_EMD_STRING_ARRAY(
                        'EMAIL',
                        'OSCMD',
                        'PLSQL',
                        'SNMP',
                        'RCA',
                        'JAVA'
                      );
METHOD_COUNT constant NUMBER(2) := METHOD_NAMES.COUNT;

-- Performance name
PERF_NUM_SEVERITIES_CHECKED constant VARCHAR2(10) := 'Severities';
PERF_NUM_JOBSTATES_CHECKED constant VARCHAR2(18) := 'Job State Changes';

PERF_NUM_QUEUED constant VARCHAR2(6) := 'QUEUED';
PERF_NUM_EMAILS_QUEUED constant VARCHAR2(13) := 'EMAILS_QUEUED';
PERF_NUM_OSCMDS_QUEUED constant VARCHAR2(13) := 'OSCMDS_QUEUED';
PERF_NUM_PLSQLS_QUEUED constant VARCHAR2(13) := 'PLSQLS_QUEUED';
PERF_NUM_SNMPS_QUEUED constant VARCHAR2(12) := 'SNMPS_QUEUED';
PERF_NUM_RCAS_QUEUED constant VARCHAR2(11) := 'RCAS_QUEUED';
PERF_NUM_JAVA_QUEUED constant VARCHAR2(19) := 'JAVAS_QUEUED';

PERF_NUM_JOB_QUEUED constant VARCHAR2(10) := 'JOB_QUEUED';
PERF_NUM_JOB_EMAILS_QUEUED constant VARCHAR2(18) := 'JOB_EMAILS_QUEUED';
PERF_NUM_JOB_OSCMDS_QUEUED constant VARCHAR2(18) := 'JOB_OSCMDS_QUEUED';
PERF_NUM_JOB_PLSQLS_QUEUED constant VARCHAR2(18) := 'JOB_PLSQLS_QUEUED';
PERF_NUM_JOB_SNMPS_QUEUED constant VARCHAR2(18) := 'JOB_SNMPS_QUEUED';
PERF_NUM_JOB_JAVA_QUEUED constant VARCHAR2(19) := 'JOB_JAVAS_QUEUED';

PERF_WAITING_IN_QUEUE constant VARCHAR2(20) := 'Queued Notifications';
PERF_AVG_WAIT constant VARCHAR2(20) := 'Average Waiting Time';
PERF_TOTAL_WAIT constant VARCHAR2(20) := 'Total Waiting Time';
PERF_DELIVERY_TIME constant VARCHAR2(13) := 'Delivery Time';
PERF_TOTAL_DELIVERY_TIME constant VARCHAR2(20) := 'Total Delivery Time';
PERF_EMAIL_DELIVERY_TIME constant VARCHAR2(20) := 'EMAIL_DELIVERY_TIME';
PERF_EMAIL_TOTAL_DELIVERY_TIME constant VARCHAR2(26) := 'EMAIL_TOTAL_DELIVERY_TIME';
PERF_OSCMD_DELIVERY_TIME constant VARCHAR2(20) := 'OSCMD_DELIVERY_TIME';
PERF_OSCMD_TOTAL_DELIVERY_TIME constant VARCHAR2(26) := 'OSCMD_TOTAL_DELIVERY_TIME';
PERF_PLSQL_DELIVERY_TIME constant VARCHAR2(20) := 'PLSQL_DELIVERY_TIME';
PERF_PLSQL_TOTAL_DELIVERY_TIME constant VARCHAR2(26) := 'PLSQL_TOTAL_DELIVERY_TIME';
PERF_SNMP_DELIVERY_TIME constant VARCHAR2(20) := 'SNMP_DELIVERY_TIME';
PERF_SNMP_TOTAL_DELIVERY_TIME constant VARCHAR2(26) := 'SNMP_TOTAL_DELIVERY_TIME';
PERF_RCA_DELIVERY_TIME constant VARCHAR2(19) := 'RCA_DELIVERY_TIME';
PERF_RCA_TOTAL_DELIVERY_TIME constant VARCHAR2(25) := 'RCA_TOTAL_DELIVERY_TIME';
PERF_JAVA_DELIVERY_TIME constant VARCHAR2(22) := 'JAVA_DELIVERY_TIME';
PERF_JAVA_TOTAL_DELIVERY_TIME constant VARCHAR2(29) := 'JAVA_TOTAL_DELIVERY_TIME';

-- Device types

-- Email device types. The long email device type has more information
-- about a severity while the short email device type contains less information
-- and is more useful for a pager
EMAIL_DEVICE_TYPE CONSTANT NUMBER(1) := 1; -- backwards compatability
LONG_EMAIL_DEVICE_TYPE CONSTANT NUMBER(1) := 1;
SHORT_EMAIL_DEVICE_TYPE CONSTANT NUMBER(1) := 5;

PROGRAM_DEVICE_TYPE CONSTANT NUMBER(1) := 2;
SNMP_DEVICE_TYPE CONSTANT NUMBER(1) := 3;
PLSQL_DEVICE_TYPE CONSTANT NUMBER(1) := 4;
QUEUE_DEVICE_TYPE CONSTANT NUMBER(1) := 6;
RCA_DEVICE_TYPE CONSTANT NUMBER(1) := 7;
JAVA_DEVICE_TYPE CONSTANT NUMBER(1) := 9;
WINDOWS_PROGRAM_DEVICE_TYPE CONSTANT NUMBER(1) := 8;

-- Device status
DEVICE_UP constant number := 1;
DEVICE_DOWN constant number := 0;
DEVICE_BEING_PINGED constant number := 2;

-- Notification types
NORMAL_NOTIFICATION constant number := 1;
RETRY_NOTIFICATION constant number := 2;
ESCALATION_NOTIFICATION constant number := 3;
REPEAT_NOTIFICATION constant number := 4;
STOP_DEQUEUEING constant number := 5;
START_DEQUEUEING constant number := 6;

IS_PING constant number := 1;
IS_NOT_PING constant number := 0;

-- Notification source types
METRIC_SEVERITY constant number := 1;
POLICY_VIOLATION constant number := 2;
JOB_STATE_CHANGE constant number := 3;
METRIC_CA_STATE_CHANGE constant number := 4;
POLICY_CA_STATE_CHANGE constant number := 5;
RCA_SEVERITY constant number := 6;
RCA_RETRY_SEVERITY constant number := 7;
RCA_RETRY_EVENT constant number := 8;
RCA_COMPLETED_SEVERITY constant number := 9;

-- Parameters for queue sizes
NOTIFY_EMAIL_QUEUES constant varchar2(32) := 'NOTIFY_EMAIL_QUEUES';
NOTIFY_OSCMD_QUEUES constant varchar2(32) := 'NOTIFY_OSCMD_QUEUES';
NOTIFY_PLSQL_QUEUES constant varchar2(32) := 'NOTIFY_PLSQL_QUEUES';
NOTIFY_SNMP_QUEUES constant varchar2(32) := 'NOTIFY_SNMP_QUEUES';
NOTIFY_RCA_QUEUES constant varchar2(32) := 'NOTIFY_RCA_QUEUES';
NOTIFY_JAVA_QUEUES constant varchar2(32) := 'NOTIFY_JAVA_QUEUES';
NOTIFY_DEV_PING_STATE_TIMEOUT constant varchar2(32) := 'NOTIFY_DEV_PING_STATE_TIMEOUT';

DEFAULT_NOTIFY_EMAIL_QUEUES constant number := 6;
DEFAULT_NOTIFY_OSCMD_QUEUES constant number := 12;
DEFAULT_NOTIFY_PLSQL_QUEUES constant number := 3;
DEFAULT_NOTIFY_SNMP_QUEUES constant number := 3;

-- Default time to remain in ping status is 30 mins.
DEF_DEV_PING_STATE_TIMEOUT constant number := 1800; 
-- DO NOT modify this value as RCA only supports a single OMS processing
-- RCA tasks and therefore can not support multiple queues...
-- for 10gR2 RCA will only support a single queue to avoid having
-- to synchronize RCA tasks across OMSs
DEFAULT_NOTIFY_RCA_QUEUES constant number := 1;

DEFAULT_NOTIFY_JAVA_QUEUES constant number := 2;

-- Global repeat notification settings defaults
-- Global repeat notification settings
REPEAT_ENABLED_PARAM constant varchar2(50) := 'NOTIFICATION_REPEAT_ENABLED';
REPEAT_FREQUENCY_PARAM constant varchar2(50) := 'NOTIFICATION_REPEAT_FREQUENCY';
REPEAT_COUNT_PARAM constant varchar2(50) := 'NOTIFICATION_REPEAT_COUNT';
REPEAT_ENABLED_DEFAULT constant number := 0;
REPEAT_FREQUENCY_DEFAULT constant number := 15;
REPEAT_COUNT_DEFAULT constant number := 3;

-- Parameter for upper bound when checking for new notifications (seconds)
NOTIFY_CHECK_UPPER_BOUND constant varchar2(32) := 'NOTIFY_CHECK_UPPER_BOUND';
DEFAULT_NOTIFY_CHECK_UPR_BOUND constant varchar2(32) := 5;

-- Error code returned when the details of a device cannot be found
NO_DEVICE_ERR constant number := -1;
-- Error code returned when the details of a severity cannot be found
NO_SEVERITY_ERR constant number := -2;
-- Error code returned when the details of a state change cannot be found
NO_STATE_CHANGE_ERR constant number := -3;

EST_NOTIFICATION_NAME constant varchar2(32) := 'EST_NOTIFICATION';

QUEUE_TIMEOUT constant NUMBER := -25228;

PROCEDURE DBMSJOB_EXTENDED_SQL_TRACE_ON(p_value IN BOOLEAN);

-- PURPOSE
--  To initialize the queue table and get the number of queues which 
--  can be using in mapping user/object combinations to a queue
PROCEDURE INITIALIZE_QUEUES;

-- PURPOSE
--   Main procedure to go through violation table, execute notification rules
--   and queue notifications. This routine takes no parameters and should
--   be called periodically from a DBMS JOB.
PROCEDURE CHECK_FOR_SEVERITIES;

-- PURPOSE
--   To register an OMS with the notification system so that it can be assigned
--   some queues to handle
PROCEDURE REGISTER_OMS(oms_id_in IN NUMBER,
                       admin_qname_out OUT VARCHAR2,
                       windows_in IN NUMBER DEFAULT 0);

-- PURPOSE
--   To unregister an OMS from the notification system so that its queues 
--   can be reassigned
PROCEDURE UNREGISTER_OMS(oms_id_in IN NUMBER);

-- PURPOSE
--  To handle failover when an OMS goes down
PROCEDURE OMS_FAILOVER(oms_id_in IN NUMBER, 
                       last_timestamp_in IN DATE);

-- PURPOSE
--   Set the number of notification queues
--
-- PARAMETERS
--   V_NUMBER - the number of queues. 
--  
-- PURPOSE
--   Set the number of notification queues
--
-- PARAMETERS
--
PROCEDURE SET_NUMBER_QUEUES(
 v_num_email_qs IN NUMBER DEFAULT DEFAULT_NOTIFY_EMAIL_QUEUES,
 v_num_oscmd_qs IN NUMBER DEFAULT DEFAULT_NOTIFY_OSCMD_QUEUES,
 v_num_plsql_qs IN NUMBER DEFAULT DEFAULT_NOTIFY_PLSQL_QUEUES,
 v_num_snmp_qs IN NUMBER DEFAULT DEFAULT_NOTIFY_SNMP_QUEUES,
 v_num_java_qs IN NUMBER DEFAULT DEFAULT_NOTIFY_JAVA_QUEUES);

-- PURPOSE
--   To determine which queue has a notifiction ready for dequeueing
PROCEDURE QUEUE_READY(qnames_in IN SMP_EMD_STRING_ARRAY,
                      qtimeout_in IN NUMBER,
                      qname_out OUT VARCHAR2);

-- PURPOSE
--   Procedure to queue an RCA update task
PROCEDURE QUEUE_RCA_UPDATE(target_guid_in IN RAW,                          
                           severity_guid_in IN RAW, 
                           event_guid_in IN RAW, 
                           delay_in IN NUMBER);

-- PURPOSE
--   Procedure to queue an RCA  task (used during regression runs only)
PROCEDURE QUEUE_RCA_TASK(target_guid_in IN RAW,                          
                           severity_guid_in IN RAW,
                           rca_type_in IN NUMBER,
                           rca_mode_in IN VARCHAR2);
                           
-- PURPOSE
--   Procedure to queue a task with a delay
PROCEDURE QUEUE_WITH_DELAY(qname_in IN VARCHAR2,
                notification_in IN MGMT_NOTIFY_NOTIFICATION,
                delay_in IN NUMBER);
                
-- PURPOSE
--   To dequeue a notificaiton from a queue
PROCEDURE DEQUEUE(qname_in IN VARCHAR2,
                  qtimeout_in IN NUMBER,
                  qmsg_id_out OUT RAW,
                  notification_out OUT MGMT_NOTIFY_NOTIFICATION);

-- PURPOSE
--   Procedure to specify a gateway
--   
-- PARAMETERS
--   V_GATEWAYS - the names of the gateways through which outgoing emails 
--                are sent. The gateways are tried in order until a working
--                one is found
--   V_SMTP_PORTS - the ports of the SMTP services on the gateway machine
--   V_EMAIL_ADDRESS - the email address from which emails are sent
--   V_EMAIL_NAME - the name associated with the email address
--   V_SMTP_USER - the name of the SMTP user to be sent to the gateway for
--                 authentication
--   V_SMTP_PWD - the password of the SMTP user to be sent to the gateway for
--                 authentication
-- NOTE
--   The gateways all share the same SMTP username, SMTP password, email
--   address and email name. 
PROCEDURE SET_EMAIL_GATEWAYS(v_gateways      IN MGMT_MEDIUM_STRING_ARRAY,
                             v_smtp_ports    IN MGMT_INTEGER_ARRAY,
                             v_email_address IN VARCHAR2,
                             v_email_name    IN VARCHAR2,
                             v_smtp_user     IN VARCHAR2 DEFAULT '',
                             v_smtp_pwd      IN VARCHAR2 DEFAULT '');

-- PURPOSE
--   Procedure to specify a number of email gateways.
--
-- PARAMETERS
--   V_GATEWAYS - the names of the gateways through which outgoing emails 
--                are sent. The gateways are tried in order until a working
--                one is found
--   V_EMAIL_ADDRESS - the email address from which emails are sent
--   V_EMAIL_NAME - the name associated with the email address
--
PROCEDURE SET_EMAIL_GATEWAYS(v_gateways      IN SMP_EMD_STRING_ARRAY,
                             v_email_address IN VARCHAR2,
                             v_email_name    IN VARCHAR2);
-- PURPOSE
--   Procedure to add an administrator profile. The profile name must be less
--   than 30 characters, contain no spaces and conform to the Oracle naming
--   conventions for dictionary objects
-- PARAMETERS
--   V_PROFILE_NAME - the name of the administrator profile
--   V_DESCRIPTION - a description of the profile
-- NOTES
--   Only the super administrator should be able to add an administrator profile
PROCEDURE ADD_ADMIN_PROFILE(v_profile_name IN VARCHAR2,
                            v_description  IN VARCHAR2);

-- PURPOSE
--   Procedure to add the name of an administrator profile that will serve
--   as an escalation point.
-- PARAMETERS
--   V_PROFILE_NAME - the name of the administrator profile for which the
--                    escalation point is being set up
--   V_ESCALATION_PROFILE_NAME - the name of the escalation profile
--   V_ESCALATION_INTERVAL - the interval after which escalaltion should take
--                           place if a severity occurrence has not been
--                           cleared
-- NOTES
--
PROCEDURE SET_ESCALATION_PROFILE(v_profile_name            IN VARCHAR2,
                                 v_escalation_profile_name IN VARCHAR2,
                                 v_escalation_interval     IN NUMBER);

-- PURPOSE
--   Procedure to delete the name of an administrator profile that will serve
--   as an escalation point.
-- PARAMETERS
--   V_PROFILE_NAME - the name of the administrator profile for which the
--                    escalation point is being deleted
-- NOTES
--
PROCEDURE DELETE_ESCALATION_PROFILE(v_profile_name IN VARCHAR2);

-- PURPOSE
--   Procedure to add an email notification device to an administrator profile
-- PARAMETERS
--    V_DEVICE_NAME - the name of the device
--    V_PROFILE_NAME - the name of the admin profile
--    V_EMAIL_ADDRESS - the email address of the device
--    V_DESCRIPTION - the description of the device
PROCEDURE ADD_EMAIL_DEVICE(v_device_name   IN VARCHAR2,
                           v_profile_name  IN VARCHAR2,
                           v_email_address IN VARCHAR2,
                           v_type          IN NUMBER
                                           DEFAULT LONG_EMAIL_DEVICE_TYPE,
                           v_description   IN VARCHAR2 DEFAULT '');

-- PURPOSE
--   Procedure to change the email address of an existing email notification
--   device that was added via ADD_EMAIL_DEVICE
-- PARAMETERS
--    V_DEVICE_NAME - the name of the device
--    V_PROFILE_NAME - the name of the admin profile
--    V_EMAIL_ADDRESS - the new email address of the device
--    V_TYPE - the type of email 
--             LONG_EMAIL_DEVICE_TYPE - send long formatted emails which have
--                                      more information
--             SHORT_EMAIL_DEVICE_TYPE - send short formatted emails which have
--                                       less information and are more suitable
--                                       for pagers
--    V_DESCRIPTION - the description of the device
PROCEDURE EDIT_EMAIL_DEVICE(v_device_name   IN VARCHAR2,
                            v_profile_name  IN VARCHAR2,
                            v_email_address IN VARCHAR2,
                            v_type          IN NUMBER
                                            DEFAULT LONG_EMAIL_DEVICE_TYPE,
                            v_description   IN VARCHAR2 DEFAULT '');

-- PURPOSE
--   Procedure to add a program/script notification device to an administrator
--   profile
-- PARAMETERS
--    V_DEVICE_NAME - the name of the device
--    V_PROFILE_NAME - the name of the admin profile
--    V_PROGRAM - this is the name of the program/script to run which will look
--                after delivering the notification. Can be used to integrate 
--                other problem tracking systems
--    V_DESCRIPTION - the description of the device
--    V_ARGS - a list of command line arguments
PROCEDURE ADD_PROGRAM_DEVICE(v_device_name   IN VARCHAR2,
                             v_profile_name  IN VARCHAR2,
                             v_program       IN VARCHAR2,
                             v_description   IN VARCHAR2 DEFAULT '',
                             v_args          IN EMDSYSTEM_ARGS DEFAULT NULL);

-- PURPOSE
--   Procedure to change the program/script of an existing notification device
--   device that was added via ADD_PROGRAM_DEVICE
-- PARAMETERS
--    V_DEVICE_NAME - the name of the device
--    V_PROFILE_NAME - the name of the admin profile
--    V_PROGRAM - the new name of the program/script to run
--    V_DESCRIPTION - the description of the device
--    V_ARGS - a list of command line arguments
PROCEDURE EDIT_PROGRAM_DEVICE(v_device_name   IN VARCHAR2,
                              v_profile_name  IN VARCHAR2,
                              v_program       IN VARCHAR2,
                              v_description   IN VARCHAR2 DEFAULT '',
                              v_args          IN EMDSYSTEM_ARGS DEFAULT NULL);

-- PURPOSE
--   Procedure to add a java callback notification device to an administrator
--   profile
-- PARAMETERS
--    V_DEVICE_NAME - the name of the device
--    V_PROFILE_NAME - the name of the admin profile
--    V_CLASS - this is the name of fully qualified class name that will be invoked 
--                on delivering the notification. Can be used to integrate 
--                other problem tracking systems
--    V_DESCRIPTION - the description of the device
PROCEDURE ADD_JAVA_DEVICE(v_device_name   IN VARCHAR2,
                          v_profile_name  IN VARCHAR2,
                          v_class         IN VARCHAR2,
                          v_description   IN VARCHAR2 DEFAULT '',
                          v_context       IN MGMT_MEDIUM_STRING_TABLE DEFAULT NULL);

--
-- PROCEDURE: EDIT_JAVA_DEVICE
--
-- PURPOSE:
--   Modify the existing notification device of type Java.
--   Update p_class, p_description and context for the given p_device_name and p_profile_name
--   New context will override the previous context 
--
-- IN Parameters:
--   p_device_name  : Name of the notification device
--   p_profile_name : The name of the admin profile  
--   p_class      : Fully qualified name of the class implementing the notification interface
--   p_description  : Description for this device
--   p_context      : String array of context that will be passed to the notify method while delivering the notification.
--
-- OUT Parameters:
--    NONE
--
PROCEDURE EDIT_JAVA_DEVICE(v_device_name  IN VARCHAR2,
                           v_profile_name IN VARCHAR2,
                           v_class        IN VARCHAR2,
                           v_description  IN VARCHAR2 DEFAULT '',
                           v_context      IN MGMT_MEDIUM_STRING_TABLE DEFAULT NULL);


-- PURPOSE
--   Function to test a PL/SQL notification device to see if it can be
--   called
-- PARAMETERS
--   V_PLSQL_PROC - this is the name of the PL/SQL procedure to call
--   V_ERR_MSG - the reason for failure
FUNCTION TEST_PLSQL_DEVICE(v_plsql_proc IN VARCHAR2, v_err_msg OUT VARCHAR2) 
RETURN NUMBER; 

-- PURPOSE
--   Procedure to add a PL/SQL notification device to an administrator
--   profile
-- PARAMETERS
--    V_DEVICE_NAME - the name of the device
--    V_PROFILE_NAME - the name of the admin profile
--    V_PLSQL_PROC - this is the name of the PL/SQL procedure to call which will
--                   look after delivering the notification. This can be used 
--                   to integrate other problem tracking systems
--    V_DESCRIPTION - the description of the device
PROCEDURE ADD_PLSQL_DEVICE(v_device_name   IN VARCHAR2,
                           v_profile_name  IN VARCHAR2,
                           v_plsql_proc    IN VARCHAR2,
                           v_description   IN VARCHAR2 DEFAULT '');

-- PURPOSE
--   Procedure to change the name of the PL/SQL procedure of an existing 
--   notification device that was added via ADD_PLSQL_DEVICE
-- PARAMETERS
--    V_DEVICE_NAME - the name of the device
--    V_PROFILE_NAME - the name of the admin profile
--    V_PLSQL_PROC - the new name of the PL/SQL procedure to call
--    V_DESCRIPTION - the description of the device
PROCEDURE EDIT_PLSQL_DEVICE(v_device_name   IN VARCHAR2,
                            v_profile_name  IN VARCHAR2,
                            v_plsql_proc    IN VARCHAR2,
                            v_description   IN VARCHAR2 DEFAULT '');

-- PURPOSE
--   Procedure to add an SNMP notification device to an administrator profile
-- PARAMETERS
--    V_DEVICE_NAME - the name of the device
--    V_PROFILE_NAME - the name of the admin profile
--    V_SNMP_HOST - the name of the snmp host
--    V_SNMP_PORT - the port of the snmp host
--    V_COMMUNITY - the name of the SNMP community the host is part of
--    V_DESCRIPTION - the description of the device
PROCEDURE ADD_SNMP_DEVICE(v_device_name  IN VARCHAR2,
                          v_profile_name IN VARCHAR2,
                          v_snmp_host    IN VARCHAR2,
                          v_snmp_port    IN NUMBER DEFAULT 162,
                          v_community    IN VARCHAR2 DEFAULT 'public',
                          v_description  IN VARCHAR2 DEFAULT '');

-- PURPOSE
--   Procedure to change the snmp host of an existing SNMP notification
--   device that was added via ADD_SNMP_DEVICE
-- PARAMETERS
--    V_DEVICE_NAME - the name of the device
--    V_PROFILE_NAME - the name of the admin profile
--    V_SNMP_HOST - the new name of the snmp host
--    V_SNMP_PORT - the port of the snmp host
--    V_COMMUNITY - the name of the SNMP community the host is part of
--    V_DESCRIPTION - the description of the device
PROCEDURE EDIT_SNMP_DEVICE(v_device_name  IN VARCHAR2,
                           v_profile_name IN VARCHAR2,
                           v_snmp_host    IN VARCHAR2,
                           v_snmp_port    IN NUMBER DEFAULT 162,
                           v_community    IN VARCHAR2 DEFAULT 'public',
                           v_description  IN VARCHAR2 DEFAULT '');
                           
-- PURPOSE
--   Get the device owner for device - v_device_name of type v_type
-- PARAMETERS
--    V_DEVICE_NAME - the name of the device
--    V_TYPE - the type of the device
FUNCTION GET_DEVICE_OWNER(v_device_name  IN VARCHAR2,
                          v_type         IN INTEGER)
RETURN VARCHAR2;                           

-- PURPOSE
--   Procedure to set the start date of a user's notification schedule 
-- PARAMETERS
--    V_SCHEDULE_OWNER - the owner of the schedule
--    V_SCHEDULE_NAME - the name of the schedule
--    V_START_DATE - the date on which the schedule starts. Note that the hour
--                   and minutes are ignored since the schedule starts at 12:00
--    V_NUM_WEEKS - the number of weeks after which the schedule repeats
--    V_TIMEZONE_REGION - the timezone region of the schedule owner
--
PROCEDURE SET_SCHEDULE(v_schedule_owner IN VARCHAR2,
                       v_schedule_name  IN VARCHAR2 DEFAULT ' ',
                       v_start_date     IN DATE,
                       v_num_weeks      IN NUMBER DEFAULT 1,
                       v_timezone_region IN VARCHAR2 DEFAULT 
                                         TO_CHAR(SYSTIMESTAMP,'TZR'));

-- PURPOSE
--   Procedure to get the user who last updated a notification schedule
-- PARAMETERS
--    V_SCHEDULE_OWNER - the owner of the schedule
--    V_SCHEDULE_NAME - the name of the schedule
--    V_UPDATED_BY - the user who updated the schedule
--    V_UPDATED - when the update took place
--
PROCEDURE WHO_UPDATED_SCHEDULE(v_schedule_owner IN VARCHAR2,
                               v_schedule_name  IN VARCHAR2 DEFAULT ' ',
                               v_updated_by     OUT VARCHAR2, 
                               v_updated        OUT DATE);

-- PURPOSE
--   Procedure to disable a user's notification schedule for a period of time
-- PARAMETERS
--    V_SCHEDULE_OWNER - the owner of the schedule
--    V_SCHEDULE_NAME - the name of the schedule
--    V_DISABLE_START - the date on which the disabling period starts
--    V_DISABLE_END - the date on which the disabling period ends
--
-- NOTES
--   To clear a disabling period pass NULL in for both v_disable_start and
--   v_disable_end
--
PROCEDURE DISABLE_SCHEDULE(v_schedule_owner IN VARCHAR2,
                           v_schedule_name  IN VARCHAR2 DEFAULT ' ',
                           v_disable_start  IN DATE,
                           v_disable_end    IN DATE);

-- PURPOSE
--   Procedure to set the notification schedule of a notification device
-- PARAMETERS
--    V_SCHEDULE_OWNER - the owner of the schedule
--    V_SCHEDULE_NAME - the name of the schedule
--    V_DEVICE_NAME - the name of the device
--    V_DEVICE_OWNER - the name of the admin profile
--    V_SCHEDULE - the hours in this schedule when the device is available.
--                 This must be an array of 1344 characters
PROCEDURE SET_DEVICE_SCHEDULE(v_schedule_owner IN VARCHAR2,
                              v_schedule_name  IN VARCHAR2 DEFAULT ' ',
                              v_device_name    IN VARCHAR2,
                              v_device_owner   IN VARCHAR2,
                              v_schedule       IN VARCHAR2);

-- PURPOSE
--   Procedure to clear a user's schedule
-- PARAMETERS
--    V_SCHEDULE_OWNER - the owner of the schedule
--    V_SCHEDULE_NAME - the name of the schedule
PROCEDURE CLEAR_SCHEDULE(v_schedule_owner IN VARCHAR2, 
                         v_schedule_name IN VARCHAR2 DEFAULT ' ');

-- PURPOSE
--   Function to check if a user has a schedule set
-- PARAMETERS
--    V_SCHEDULE_OWNER - the owner of the schedule
--    V_SCHEDULE_NAME - the name of the schedule
FUNCTION USER_SCHEDULE_SET(v_schedule_owner IN VARCHAR2, 
                           v_schedule_name IN VARCHAR2 DEFAULT ' ')
RETURN NUMBER;

-- PURPOSE
--   Procedure to add a notification rule
-- PARAMETERS
--   V_RULE_NAME - name of the notification rule
--   V_OWNER - the profile name of the admin who created the rule
--   V_DESCRIPTION - a description of the notification rule
--   V_PUBLIC_RULE - whether the rule can be accessed by other profiles
PROCEDURE ADD_NOTIFICATION_RULE(v_rule_name   IN VARCHAR2,
                                v_owner       IN VARCHAR2,
                                v_description IN VARCHAR2,
                                v_public_rule IN NUMBER DEFAULT 0,
                                v_repeat_enabled IN NUMBER DEFAULT 0);

-- PURPOSE
--  Procedure to add a configuration to a rule. This allows the specification of
--  the severity properties to which the rule applies.
--
-- PARAMETERS
--   V_RULE_NAME - name of the notification rule
--   V_OWNER - the owner of the notification rule
--   V_TARGET_TYPE - the types of targets that this rule applies to
--   V_TARGET_NAME - the name of a target that this rule applies to
--   V_GROUP_GUID - the GUID of a group target that this rule applies to
--   V_METRIC_NAME - the name of a metric that this rule applies to
--   V_METRIC_COLUMN - the name of a column in a table metric that this rule
--                     applies to
--   V_KEY_VALUE - the value of the key for which this rule applies:
--                 % means no key specified so match all values
--                 %% means all composite keys
--                 string value specifies single key
--                 GUID value specifies composite key
--  V_IS_COMPOSITE_KEY - flag to indicate if this is a composite key
--                       1 - is composite key
--                       0 - is single key
--  V_TARGET_GUID - the target GUID of the target associated with the composite
--                  key
--  V_KEY_PART1 - the first part of the composite key
--  V_KEY_PART2 - the second part of the composite key
--  V_KEY_PART3 - the third part of the composite key
--  V_KEY_PART4 - the fourth part of the composite key
--  V_KEY_PART5 - the fifth part of the composite key
--  V_WANT_CLEARS - whether this rule applies to clears
--  V_WANT_WARNINGS - whether this rule applies to warnings
--  V_WANT_CRITICAL_ALERTS - whether this rule applies to criticals
--  V_WANT_TARGET_UP - whether this rule applies to target state up severities
--  V_WANT_TARGET_DOWN - whether this rule applies to target state down
--                       severities
--  V_WANT_TARGET_UNREACHABLE_ST - whether this rule applies to severities
--                                 indicating the start of a target being
--                                 unreachable
--  V_WANT_TARGET_UNREACHABLE_END - whether this rule applies to severities
--                                  indicating the end of a target being
--                                  unreachable and thus reachable again
--                                   target metric
--  V_WANT_TARGET_METRIC_ERR_END - whether this rule applies to severities
--                                 indicating the end of an error in a
--                                 target metric
--  V_WANT_TARGET_BLACKOUT_START - whether this rule applies to target
--                                 blackout start severities
--  V_WANT_POLICY_CLEARS - whether this rule applies to policy cleared
--  V_WANT_POLICY_VIOLATIONS - whether this rule applies to policy violations
--  V_WANT_WARNING_JOB_SUCCEEDED - whether this rule applies to succeeded
--                                state changes for a correcive action that
--                                was run as a result of a warning severity
--  V_WANT_WARNING_JOB_PROBLEMS - whether this rule applies to problem
--                                state changes for a correcive action that
--                                was run as a result of a warning severity
--  V_WANT_CRITICAL_JOB_SUCCEEDED - whether this rule applies to succeeded
--                                  state changes for a correcive action that
--                                  was run as a result of a critical severity
--  V_WANT_CRITICAL_JOB_PROBLEMS - whether this rule applies to problem
--                                 state changes for a correcive action that
--                                 was run as a result of a critical severity
--  V_WANT_POLICY_JOB_SUCCEEDED - whether this rule applies to succeeded
--                               state changes for a correcive action that
--                                was run as a result of a policy violation
--  V_WANT_POLICY_JOB_PROBLEMS - whether this rule applies to problem
--                               state changes for a correcive action that
--                               was run as a result of a policy violation
--
-- NOTES
--  For composite keys v_is_composite_key must be set to 1, the v_key_value
--  parameter must contain the composite GUID, the associated key parts
--  must be specified in v_key_part1..v_key_part5 and the target type must
--  be specified in v_target_guid
-- 
--  All the v_want_* columns can have the following values:
--    0 - DO NOT WANT THEM
--    1 - WANT THEM
--
--  except for metric errors. The following applies to the value of both
--  v_want_target_metric_start and v_want_target_metric_err_end
--  0 -> does not want to be notified for metric errors for
--       response/status or other metrics
--  1 -> wants to be notified for metric errors for response/status metric only
--  2 -> wants to be notified for metric errors for non-response/status metric
--       only
--  3 -> wants to be notified for metric errors for both response/status and
--       non-response/status metric
--
PROCEDURE ADD_CONFIGURATION_TO_RULE
(v_rule_name            IN VARCHAR2,
 v_owner                IN VARCHAR2,
 v_target_type          IN VARCHAR2 DEFAULT '%',
 v_target_name          IN VARCHAR2 DEFAULT '%',
 v_group_guid           IN VARCHAR2 DEFAULT NULL,
 v_metric_name          IN VARCHAR2 DEFAULT '%',
 v_metric_column        IN VARCHAR2 DEFAULT '%',
 v_key_value            IN VARCHAR2 DEFAULT '%',
 v_is_composite_key     IN NUMBER   DEFAULT 0,
 v_target_guid          IN RAW      DEFAULT NULL,
 v_key_part1            IN VARCHAR2 DEFAULT '',
 v_key_part2            IN VARCHAR2 DEFAULT '',
 v_key_part3            IN VARCHAR2 DEFAULT '',
 v_key_part4            IN VARCHAR2 DEFAULT '',
 v_key_part5            IN VARCHAR2 DEFAULT '',
 v_want_clears                  IN NUMBER DEFAULT 0,
 v_want_warnings                IN NUMBER DEFAULT 0,
 v_want_critical_alerts         IN NUMBER DEFAULT 0,
 v_want_target_up               IN NUMBER DEFAULT 0,
 v_want_target_down             IN NUMBER DEFAULT 0,
 v_want_target_unreachable_st   IN NUMBER DEFAULT 0,
 v_want_target_unreachable_end  IN NUMBER DEFAULT 0,
 v_want_target_metric_err_start IN NUMBER DEFAULT 0,
 v_want_target_metric_err_end   IN NUMBER DEFAULT 0,
 v_want_target_blackout_start   IN NUMBER DEFAULT 0,
 v_want_target_blackout_end     IN NUMBER DEFAULT 0,
 v_want_policy_clears           IN NUMBER DEFAULT 0, 
 v_want_policy_violations       IN NUMBER DEFAULT 0,
 v_want_warning_job_succeeded   IN NUMBER DEFAULT 0,
 v_want_warning_job_problems    IN NUMBER DEFAULT 0,
 v_want_critical_job_succeeded  IN NUMBER DEFAULT 0,
 v_want_critical_job_problems   IN NUMBER DEFAULT 0,
 v_want_policy_job_succeeded    IN NUMBER DEFAULT 0,
 v_want_policy_job_problems     IN NUMBER DEFAULT 0,
 v_ignore_rca                   IN NUMBER DEFAULT 0
);

-- PURPOSE
--  Procedure to add a configuration to a job rule. This allows the 
--  specification of the job properties to which the rule applies. This is also
--  used for corrective actions in which case the metric details are also
--  supplied
--
-- PARAMETERS
--   V_RULE_NAME - name of the notification rule
--   V_OWNER - the owner of the notification rule
--   V_TARGET_TYPE - the types of targets that this rule applies to
--   V_TARGET_NAME - the name of a target that this rule applies to
--   V_GROUP_GUID - the GUID of a group target that this rule applies to
--   V_JOB_NAME - the name of the job that this rule applies to
--   V_JOB_OWNER - the owner of the job that this rule applies to
--   V_JOB_TYPE - the type of the job that this rule applies to
--   V_WANT_JOB_SCHEDULED - whether this rule applies to scheduled state changes
--   V_WANT_JOB_RUNNING - whether this rule applies to running state changes
--   V_WANT_JOB_SUCCEEDED - whether this rule applies to succeeded state changes
--   V_WANT_JOB_SUSPENDED - whether this rule applies to suspended state changes
--   V_WANT_JOB_PROBLEMS - whether this rule applies to problem state changes
--
--  All the v_want_* columns can have the following values:
--    0 - DO NOT WANT THEM
--    1 - WANT THEM
--
PROCEDURE ADD_CONFIGURATION_TO_JOB_RULE
(v_rule_name            IN VARCHAR2,
 v_owner                IN VARCHAR2,
 v_target_type          IN VARCHAR2 DEFAULT '%',
 v_target_name          IN VARCHAR2 DEFAULT '%',
 v_group_guid           IN VARCHAR2 DEFAULT NULL,
 v_job_name             IN VARCHAR2 DEFAULT '%',
 v_job_owner            IN VARCHAR2 DEFAULT '%',
 v_job_type             IN VARCHAR2 DEFAULT '%',
 v_want_job_scheduled   IN NUMBER DEFAULT 0, 
 v_want_job_running     IN NUMBER DEFAULT 0, 
 v_want_job_succeeded   IN NUMBER DEFAULT 0, 
 v_want_job_suspended   IN NUMBER DEFAULT 0, 
 v_want_job_problems    IN NUMBER DEFAULT 0
);

-- PURPOSE
--   Procedure to set the properties of a rule. Properties include whether this
--   rule applies to criticals or warnings, and also allow the rule to match
--   collection dates and messages
-- PARAMETERS
--   V_RULE_NAME - name of the notification rule
--   V_OWNER - the owner of the notification rule
--   V_DESCRIPTION - a description of the notification rule
--   V_PUBLIC_RULE - whether the rule can be accessed by other profiles
--
--
PROCEDURE SET_RULE_PROPERTIES
(v_rule_name                    IN VARCHAR2,
 v_owner                        IN VARCHAR2,
 v_description                  IN VARCHAR2,
 v_public_rule                  IN NUMBER DEFAULT 0,
 v_repeat_enabled               IN NUMBER DEFAULT 0);

-- PURPOSE
--   Procedure to make a rule public
-- PARAMETERS
--   V_RULE_NAME - name of the notification rule
--   V_OWNER - the owner of the notification rule
--   V_MAKE_PUBLIC - whether the rule is public or private
PROCEDURE MAKE_RULE_PUBLIC(v_rule_name   IN VARCHAR2,
                           v_owner       IN VARCHAR2,
                           v_make_public IN NUMBER);

-- PURPOSE
--   Procedure to enable/disable repeat setting for a rule
-- PARAMETERS
--   P_RULE_NAME - name of the notification rule
--   P_OWNER - the owner of the notification rule
--   P_ENABLE_REPEAT - whether to enable or disable the reepat.
PROCEDURE SET_RULE_REPEAT(p_rule_name   IN VARCHAR2,
                          p_owner       IN VARCHAR2,
                          p_enable_repeat IN NUMBER);


-- PURPOSE
--   Procedure to assign notification devices to a notification rule
-- PARAMETERS
--   V_RULE_NAME - name of the notification rule
--   V_OWNER - the owner of the notification rule
--   V_DEVICE_NAME - the name of the device
--   V_PROFILE_NAME - the name of the admin profile
PROCEDURE ADD_DEVICE_TO_RULE(v_rule_name    IN VARCHAR2,
                             v_owner        IN VARCHAR2,
                             v_device_name  IN VARCHAR2,
                             v_profile_name IN VARCHAR2);

-- PURPOSE
--   Function to check if a notification rule exists
-- PARAMETERS
--   V_RULE_NAME - name of the notification rule
--   V_OWNER - the owner of the notification rule
FUNCTION RULE_EXISTS(v_rule_name IN VARCHAR2,
                     v_owner     IN VARCHAR2)
RETURN INTEGER;

-- PURPOSE
--   Function to find out if the email gateway has been setup
--   by the super user
FUNCTION IS_MAIL_GATEWAY_SETUP
RETURN NUMBER;

-- PURPOSE
--   Procedure to delete the configurations of a  notification rule
-- PARAMETERS
--   V_RULE_NAME - name of the notification rule
--   V_OWNER - the owner of the notification rule
PROCEDURE DELETE_RULE_CONFIGURATIONS(v_rule_name IN VARCHAR2,
                                     v_owner     IN VARCHAR2);

-- PURPOSE
--   Procedure to delete a specific configuration of a notification rule
-- PARAMETERS
--   V_RULE_NAME - name of the notification rule
--   V_OWNER - the owner of the notification rule
--   V_TARGET_NAME - the name of the target
--   V_TARGET_TYPE - the type of the target
--   V_TARGET_NAME - the name of the metric
--   V_TARGET_NAME - the name of the metric column
--   V_TARGET_NAME - the key value
--
-- NOTES
--   This is used to delete rule configurations when a UDM or transposed
--   metric is deleted. Only configurations that exactly match all input
--   parameters will be deleted
PROCEDURE DELETE_RULE_CONFIGURATION(v_rule_name IN VARCHAR2,
                                    v_owner     IN VARCHAR2,
                                    v_target_name IN VARCHAR2,
                                    v_target_type IN VARCHAR2,
                                    v_metric_name IN VARCHAR2,
                                    v_metric_column IN VARCHAR2,
                                    v_key_value IN VARCHAR2);


-- PURPOSE
--   Procedure to remove notification devices from a notification rule
-- PARAMETERS
--   V_RULE_NAME - name of the notification rule
--   V_OWNER - the owner of the notification rule
--   V_DEVICE_NAME - the name of the device
--   V_PROFILE_NAME - the name of the admin profile
PROCEDURE DELETE_DEVICE_FROM_RULE(v_rule_name       IN VARCHAR2,
                                  v_owner           IN VARCHAR2,
                                  v_device_name     IN VARCHAR2,
                                  v_profile_name    IN VARCHAR2);

-- PURPOSE
--   Procedure to remove notification devices from a notification rule
--   Note that this procedure does not take in a profile_name. This is needed
--   for deleting snmp device or program command device
-- PARAMETERS
--   V_RULE_NAME - name of the notification rule
--   V_OWNER - the owner of the notification rule
--   V_DEVICE_NAME - the name of the device
PROCEDURE DELETE_DEVICE_FROM_RULE(v_rule_name       IN VARCHAR2,
                                  v_owner           IN VARCHAR2,
                                  v_device_name     IN VARCHAR2);

-- PURPOSE
--   Procedure to delete an administrator profile
-- PARAMETERS
--   V_PROFILE_NAME - name of the administrator profile
-- NOTES
--   Only the super administrator can delete an administrator profile
PROCEDURE DELETE_ADMIN_PROFILE(v_profile_name IN VARCHAR2);

-- PURPOSE
--   Procedure to delete a notification device associated with an administrator
--   profile
-- PARAMETERS
--   V_PROFILE_NAME - name of the notification rule
--   V_DEVICE_NAME - name of the device
PROCEDURE DELETE_DEVICE(v_profile_name    IN VARCHAR2,
                        v_device_name     IN VARCHAR2);

-- PURPOSE
--   Procedure to delete a notification program device associated with an 
--   administrator profile
--   NOTE: This is a UI specific API and could change depending on the
--         UI design.
-- PARAMETERS
--   V_PROFILE_NAME - name of the notification rule
--   V_DEVICE_NAME - name of the device
PROCEDURE DELETE_PROGRAM_DEVICE(v_profile_name    IN VARCHAR2,
                                v_device_name     IN VARCHAR2);

-- PURPOSE
--  Used for testing purposes
FUNCTION IS_DEVICE_AVAILABLE(v_device_name   IN VARCHAR2,
                             v_profile_name  IN VARCHAR2,
                             v_time_to_check IN DATE)
RETURN BOOLEAN;


-- PURPOSE
--   Procedure to delete a notification rule
PROCEDURE DELETE_NOTIFICATION_RULE(v_rule_name  IN VARCHAR2,
                                   v_owner      IN VARCHAR2,
                                   v_super_user IN VARCHAR2);

-- PURPOSE
--   Get the details of a violation for sending in a notification
PROCEDURE GET_VIOLATION_DETAILS(violation_guid_in IN RAW,
                                language_in IN VARCHAR2,
                                country_in IN VARCHAR2,
                                details_out OUT MGMT_MAX_STRING_ARRAY,
                                collection_timestamp_out OUT DATE,
                                logged_timestamp_out OUT DATE,
                                keys_out OUT MGMT_NOTIFY_COLUMNS,
                                ctxt_list_out OUT MGMT_NOTIFY_COLUMNS,
                                is_plsql_notification IN BOOLEAN DEFAULT FALSE);

-- PURPOSE
--  To send a metric severity notification to a PL/SQL procedure
PROCEDURE METRIC_SEVERITY_FOR_PLSQL(violation_guid_in IN RAW,
                                    language_in IN VARCHAR2,
                                    country_in IN VARCHAR2,
                                    device_owner_in IN VARCHAR2,
                                    device_name_in IN VARCHAR2,
                                    rule_owner_in IN VARCHAR2,
                                    rule_name_in IN VARCHAR2,
                                    procedure_name_out OUT VARCHAR2,
                                    error_code_out OUT NUMBER,
                                    error_out OUT VARCHAR2);

-- PURPOSE
--   Get the details of a job state change for sending in a notification
PROCEDURE GET_JOB_STATE_CHANGE_DETAILS(state_change_guid_in IN RAW,
                                       language_in IN VARCHAR2,
                                       country_in IN VARCHAR2,
                                       details_out OUT MGMT_NOTIFY_JOB_SC);

-- PURPOSE
--  To send a job/ca state change notification to a PL/SQL procedure
PROCEDURE STATE_CHANGE_FOR_PLSQL(state_change_guid_in IN RAW,
                                 language_in IN VARCHAR2,
                                 country_in IN VARCHAR2,
                                 device_owner_in IN VARCHAR2,
                                 device_name_in IN VARCHAR2,
                                 rule_owner_in IN VARCHAR2,
                                 rule_name_in IN VARCHAR2,
                                 procedure_name_out OUT VARCHAR2,
                                 error_code_out OUT NUMBER,
                                 error_out OUT VARCHAR2);

-- PURPOSE
--  To set a notification up to be retried
PROCEDURE SETUP_FOR_RETRY(notification_in IN MGMT_NOTIFY_NOTIFICATION,
                          error_in IN VARCHAR2);

-- PURPOSE
--   Procedure to find the devices that need to be notified for a severity
--   occurrence and queue the notifications 
-- 
PROCEDURE QUEUE_METRIC_NOTIFICATIONS
(
 v_violation IN MGMT_VIOLATIONS%ROWTYPE,
 v_ca_state_change IN MGMT_JOB_STATE_CHANGES%ROWTYPE DEFAULT NULL
);

-- PURPOSE
--   Procedure to find the devices that need to be notified for a job state
--   change and queue the notifications
--
PROCEDURE QUEUE_JOB_NOTIFICATIONS (v_state_change IN MGMT_JOB_STATE_CHANGES%ROWTYPE);

--
-- PURPOSE
--  This procedure gets a list of all devices that are currently in the
--  down state and have pending notifications. 
PROCEDURE PING_DEVICES;

--
-- PURPOSE
--   To requeue all notifications for a device because the device is now
--   available again
PROCEDURE REQUEUE(v_device_name IN VARCHAR2,
                  v_device_owner IN VARCHAR2);

-- PURPOSE
--  Called when a device has been successfully pinged
PROCEDURE PING_SUCCEEDED(notification_in IN MGMT_NOTIFY_NOTIFICATION);

-- PURPOSE
--  Log the delivery time for a severity;
PROCEDURE LOG_DELIVERY_TIME(collection_time_in IN DATE,
                            load_time_in IN DATE,
                            tz_region_in IN VARCHAR2,
                            device_type_in IN NUMBER,
                            oms_name_in IN VARCHAR2 DEFAULT NULL);

-- PROCEDURE
--  To log a record of a delivery
PROCEDURE LOG_HISTORY(source_obj_type_in IN NUMBER,
                      source_obj_guid_in IN RAW,
                      delivered_in IN VARCHAR2,
                      message_in IN VARCHAR2,
                      tz_region_in IN VARCHAR2 DEFAULT NULL);

--
-- PURPOSE
--
--   To remove deleted targets from notification rules
--
--  PARAMETERS
--
--   TARGET_NAME_IN - the target name of the deleted target
--   TARGET_TYPE_IN - the type of the deleted target
--
PROCEDURE HANDLE_TARGET_DELETED(target_name_in IN VARCHAR2,
                                target_type_in IN VARCHAR2,
                                target_guid_in IN RAW);

-- PURPOSE
--
--  To cleanup when a key value has been deleted
--
-- PARAMETERS
--
--  P_TARGET_GUID - the target GUID
--  P_METRIC_GUID - the metric GUID
--  P_KEY_VALUE - the key value
--
PROCEDURE HANDLE_METRIC_KEYVAL_DELETION(
 p_target_guid  mgmt_targets.target_guid%TYPE,
 p_metric_guid  mgmt_metrics.metric_guid%TYPE,
 p_key_value    mgmt_metrics_raw.key_value%TYPE);


-- PURPOSE
--
--  To queue a severity into MGMT_NOTIFY_INPUT_Q 
--
-- PARAMETERS
--
--  p_msg - violation GUID or state change guid + guid type
--  p_guid_type - JOB_STATE_CHANGE or SEVERITY
PROCEDURE QUEUE_NOTIF_INPUT(p_msg_in RAW,
                            p_guid_type NUMBER);


-- PURPOSE
--
--  To queue a repeat signal into MGMT_NOTIFY_INPUT_Q 
--
-- PARAMETERS
--
--  p_msg - violation GUID or state change guid + guid type
--  p_rule_guid - GUID of the rule for which repeat is enabled
--  p_repeat_count - Number of times the repeat is sent for this severity
--  p_delay - duration in minutes after which repeat notification should be send
PROCEDURE QUEUE_REPEAT_NOTIF_INPUT(p_msg_in RAW,
                                   p_rule_guid RAW, 
                                   p_repeat_count NUMBER,
                                   p_delay NUMBER);

-- PURPOSE
--
--  To de-queue a severity from MGMT_NOTIFY_INPUT_Q 
--
-- PARAMETERS
-- 
--  p_notif_src_guid - SEVERITY OR JOB_STATE_CHANGE_GUID
--  p_guid_type - JOB_STATE_CHANGE or SEVERITY
--  p_rule_guid - RULE GUID in case of repeat signal, null otherwise
--  p_repeat_count - Repeat count in case of repeat notification, null otherwise 
PROCEDURE DEQUEUE_NOTIF_INPUT(p_notif_src_guid OUT RAW,
                              p_guid_type OUT NUMBER,
                              p_rule_guid OUT RAW,
                              p_repeat_count OUT NUMBER);

-- PURPOSE
--
--  To update and en-queue RCA completed severity in MGMT_NOTIFY_INPUT_Q 
--
-- PARAMETERS
--
--  violation_guid_in - violation guid 
PROCEDURE UPDATE_AND_ENQUEUE_RCA_SEV(violation_guid_in IN RAW,
                                     notif_status_in IN NUMBER);



FUNCTION GET_QUEUE(device_type_in IN NUMBER, device_owner_in IN VARCHAR2, 
                   source_guid_in IN RAW) RETURN VARCHAR2;


-- PURPOSE
--  To retrieve global repeat settings
--
-- PARAMETERS
--  p_enabled - 1 if enabled, 0 is disabled
--  p_frequency - frequency in minutes,
--  p_count -- maximum repeat count
--
PROCEDURE GET_GLOBAL_REPEAT_SETTINGS(
            p_enabled OUT NUMBER,
            p_frequency OUT NUMBER,
            p_count OUT NUMBER);

-- PURPOSE
--  To set global repeat settings
--
-- PARAMETERS
--  p_enabled - 1 if enabled, 0 is disabled
--  p_frequency - frequency in minutes,
--  p_count -- maximum repeat count
--
PROCEDURE SET_GLOBAL_REPEAT_SETTINGS(
            p_enabled IN NUMBER DEFAULT REPEAT_ENABLED_DEFAULT,
            p_frequency IN NUMBER DEFAULT REPEAT_FREQUENCY_DEFAULT,
            p_count IN NUMBER DEFAULT REPEAT_COUNT_DEFAULT);

-- 
-- PROCEDURE: ACKNOWLEDGE_ALERT
--
-- PURPOSE
--  To acknowledge a particular alert record.
--
-- PARAMETERS
--  p_violation_guid - GUID of the violation record that needs to be acknowledged.
--  p_acknowledged_by - User name who aknowledged this alert.
--  p_annotation_type - Annotation type for severity acknowledged annotation
--  p_message - Annotation message
PROCEDURE ACKNOWLEDGE_ALERT(p_violation_guid  IN RAW,
                            p_acknowledged_by IN VARCHAR2,
                            p_annotation_type IN VARCHAR2,
                            p_message         IN VARCHAR2);
END EMD_NOTIFICATION;
/
show errors
