Rem
Rem $Header: sdk_policy_pkgbody.sql 03-aug-2006.22:49:41 sthiruna Exp $
Rem
Rem sdk_policy_pkgbody.sql
Rem
Rem Copyright (c) 2004, 2006, Oracle. All rights reserved.  
Rem
Rem    NAME
Rem      sdk_policy_pkgbody.sql - <one-line expansion of the name>
Rem
Rem    DESCRIPTION
Rem      <short description of component this file declares/defines>
Rem
Rem    NOTES
Rem      <other useful comments, qualifications, etc.>
Rem
Rem    MODIFIED   (MM/DD/YY)
Rem    sthiruna    07/31/06 - Copying 10.2.0.3 Config Standard changes from
Rem                           EMCORE_MAIN_LINUX
Rem    niramach    03/01/06 - Change owner to author in 
Rem                           generate_cs_policy_name. 
Rem    niramach    01/25/06 - Add generate_cs_policy_name function. 
Rem    rpinnama    09/01/05 - Add delete_policy_metadata API 
Rem    rpinnama    08/30/05 - Remove violations while deleting the policy 
Rem    jsadras     08/07/05 - call lock_cas_for_object with target_type 
Rem    jsadras     07/29/05 - bug:4517606
Rem    rpinnama    07/27/05 - Fix 4495847 : Decrement CA counter on remove 
Rem                           polic/metric association 
Rem    rpinnama    06/08/05 - Auto enable the policy only for applicable 
Rem                           targets 
Rem    rpinnama    06/06/05 - Fix 4012083 : Use <SYSTEM> owner for ORACLE 
Rem                           authored policies 
Rem    rpinnama    06/15/05 - Fix 4340418: Re-associate policies on detecting 
Rem                           version change 
Rem    rpinnama    05/24/05 - Use <SYSTEM> owner for 'ORACLE' authored 
Rem                           policies 
Rem    rpinnama    05/19/05 - Fix 4297228 : Auto enable policy only for 
Rem                           applicable targets 
Rem    rpinnama    03/14/05 - Bug 4012110 : API to autoenable on existing 
Rem                           targets 
Rem    rpinnama    02/23/05 - Use the new copy_policy_cfg API 
Rem    jsadras     02/09/05 - check condition_operator 
Rem    rpinnama    01/07/05 - Add support for start/end type_meta_vers 
Rem    rpinnama    12/02/04 - Support for clear_message and 
Rem                           clear_message_nlsid 
Rem    rpinnama    11/22/04 - Support transposed metrics 
Rem    rpinnama    11/18/04 - Validate PL/SQL signature 
Rem    rpinnama    10/06/04 - Remove violations while removing target-policy 
Rem                           association 
Rem    rpinnama    10/06/04 - Implement modify_policy 
Rem    rpinnama    09/30/04 - Change the implementation of copy_default 
Rem    pshishir    09/14/04 - 
Rem    rpinnama    09/02/04 - Add get_config_key_details procedure 
Rem    rpinnama    08/30/04 - Throw policy_does_not_exist error 
Rem    rpinnama    08/26/04 - Fix the parameters to is_valid_condition 
Rem    groyal      08/20/04 - CREATE_POLICY should use G_ALL_ZERO_GUID 
Rem    rpinnama    08/04/04 - Add corrective action procedures 
Rem    rpinnama    08/02/04 - Add the is_valid_condition check 
Rem    rpinnama    07/26/04 - rpinnama_add_policy_api
Rem    rpinnama    07/14/04 - Created
Rem

CREATE OR REPLACE PACKAGE BODY mgmt_policy
AS


  -- generates the policy guid
  FUNCTION generate_policy_guid(p_target_type   IN VARCHAR2,
                                p_policy_name   IN VARCHAR2)
    RETURN RAW
  IS
    l_policy_guid MGMT_POLICIES.policy_guid%TYPE;
  BEGIN

    l_policy_guid := DBMS_OBFUSCATION_TOOLKIT.md5(
                         input => UTL_RAW.cast_to_raw('ora$policy' || ';' ||
                                                      p_target_type || ';'|| 
                                                      p_policy_name));

    RETURN l_policy_guid;
  END generate_policy_guid;

  -- Looks up the policy guid from the policy table
  FUNCTION get_policy_guid(p_target_type   IN VARCHAR2,
                           p_policy_name   IN VARCHAR2)
    RETURN RAW
  IS
    l_policy_guid MGMT_POLICIES.policy_guid%TYPE;
  BEGIN
    BEGIN
      SELECT policy_guid INTO l_policy_guid
        FROM mgmt_policies
       WHERE target_type = p_target_type
         AND policy_name = p_policy_name;
    EXCEPTION
      WHEN NO_DATA_FOUND THEN
        raise_application_error(MGMT_GLOBAL.POLICY_DOES_NOT_EXIST_ERR,
          'Policy not found. target_type = ' || p_target_type ||
          ' policy_name = ' || p_policy_name);
    END;
    RETURN l_policy_guid;
  END get_policy_guid;

   --
  -- FUNCTION: generate_cs_policy_name
  --
  -- PURPOSE:
  --   Given a original policy name,configuration standard name,rule name, 
  --   author and version this function computes the cs policy name for the rule.
  --   MD5 hashing of config standard name,rule name,author and version is done 
  --   to get a unique id.
  --
  --   Then original policy name is prepended up to a length of 96 characters to
  --   the unique id.
  --
  -- IN Parameters:
  --   p_orig_policy_name: The original 10.2 policy name
  --   p_config_std_name: The configuration standard name
  --   p_rule_name : The rule name to which the cs policy belongs to.  
  --   p_author : The author of config std
  --   p_version : The version of config std.
  --
  -- RETURN:
  --   The generated configuration standard policy name.
  --
  FUNCTION generate_cs_policy_name(p_orig_policy_name IN VARCHAR2,
                                   p_config_std_name  IN VARCHAR2,
                                   p_rule_name IN VARCHAR2,
                                   p_author IN VARCHAR2,
                                   p_version IN NUMBER)
    RETURN VARCHAR2
  IS
    l_cs_policy_name MGMT_POLICIES.policy_name%TYPE;    
  BEGIN

    l_cs_policy_name := DBMS_OBFUSCATION_TOOLKIT.md5(
             input => UTL_RAW.cast_to_raw('ora$cs_policy' || ';' ||
                           p_config_std_name || ';'||
                           p_rule_name || ';' ||
                           p_author || ';' ||
                           p_version));
     
    l_cs_policy_name := substr(p_orig_policy_name,1,96)||l_cs_policy_name;
    RETURN l_cs_policy_name;
  END generate_cs_policy_name;


  -- Creates a new policy
  PROCEDURE create_policy(
      p_target_type              IN VARCHAR2,
      p_policy_name              IN VARCHAR2,
      p_metric_name              IN VARCHAR2,
      p_author                   IN VARCHAR2 DEFAULT 'ORACLE',
      p_policy_label_nlsid       IN VARCHAR2 DEFAULT NULL,
      p_description              IN VARCHAR2 DEFAULT ' ',
      p_description_nlsid        IN VARCHAR2 DEFAULT NULL,
      p_impact                   IN VARCHAR2 DEFAULT ' ',
      p_impact_nlsid             IN VARCHAR2 DEFAULT NULL,
      p_recommendation           IN VARCHAR2 DEFAULT ' ',
      p_recommendation_nlsid     IN VARCHAR2 DEFAULT NULL,
      p_violation_level          IN NUMBER   DEFAULT NULL,
      p_condition_type           IN NUMBER   DEFAULT 1,
      p_condition                IN VARCHAR2 DEFAULT NULL,
      p_condition_operator       IN NUMBER   DEFAULT 0,
      p_detailed_url_link        IN VARCHAR2 DEFAULT NULL,
      p_message                  IN VARCHAR2 DEFAULT NULL,
      p_message_nlsid            IN VARCHAR2 DEFAULT NULL,
      p_clear_message            IN VARCHAR2 DEFAULT NULL,
      p_clear_message_nlsid      IN VARCHAR2 DEFAULT NULL,
      p_auto_enable              IN NUMBER   DEFAULT 0,
      p_cs_consider_percentage   IN NUMBER   DEFAULT 0,
      p_start_type_meta_ver      IN VARCHAR2 DEFAULT '1.0',
      p_end_type_meta_ver        IN VARCHAR2 DEFAULT NULL,
      p_category_list            IN MGMT_CATEGORY_ARRAY DEFAULT NULL,
      p_policy_param_list        IN MGMT_POLICY_PARAM_DEF_ARRAY DEFAULT NULL,
      p_dflt_param_val_list      IN MGMT_POLICY_KEY_VAL_ARRAY DEFAULT NULL,
      p_viol_ctxt_list           IN MGMT_VIOL_CTXT_DEF_ARRAY DEFAULT NULL)
  IS
    l_policy_guid mgmt_policies.policy_guid%TYPE;
    l_metric_guid mgmt_metrics.metric_guid%TYPE;
    l_column_guid mgmt_metrics.metric_guid%TYPE;
    l_default_guid mgmt_policies.policy_guid%TYPE;
    l_current_user      mgmt_created_users.user_name%TYPE;
    l_owner             mgmt_policies.owner%TYPE;
    l_category          MGMT_CATEGORY_OBJ;
    l_policy_param      MGMT_POLICY_PARAM_DEF;
    l_viol_ctxt         MGMT_VIOL_CTXT_DEF;
    l_key_val           MGMT_POLICY_KEY_VAL;
    l_param_val         MGMT_POLICY_PARAM_VAL;
    l_idx               INTEGER;
    l_metric_col_list   MGMT_SHORT_STRING_ARRAY;
    l_policy_param_list MGMT_SHORT_STRING_ARRAY;
    l_bind_var_list     MGMT_NAMEVALUE_ARRAY;
    l_cond_stat         BOOLEAN := FALSE;
    l_err_msg           VARCHAR2(4096);
    l_arg_type_list     MGMT_SHORT_STRING_ARRAY;
    l_meta_ver_list     MGMT_SHORT_STRING_ARRAY;
    l_proc_name         VARCHAR2(32) := 'create_policy';
  BEGIN

    l_default_guid := EM_TARGET.generate_target_type_guid(p_target_type);

    l_current_user := MGMT_USER.GET_CURRENT_EM_USER;
    -- Check super user privilege..
    EM_CHECK.check_super_user_priv;

    -- Check for NULLs
    EM_CHECK.check_not_null(p_target_type, 'p_target_type');
    EM_CHECK.check_not_null(p_policy_name, 'p_policy_name');
    EM_CHECK.check_not_null(p_metric_name, 'p_metric_name');

    -- Check condition_type
    EM_CHECK.check_not_null(p_condition_type, 'p_condition_type');
    EM_CHECK.check_range(p_condition_type, MGMT_GLOBAL.G_CONDITION_MIN, 
      MGMT_GLOBAL.G_CONDITION_MAX, 'p_condition_type');

    IF EMDW_LOG.P_IS_DEBUG_SET THEN
      EMDW_LOG.DEBUG(l_proc_name || 
          ' Target type = [' || p_target_type || ']' ||
          ' Policy Name = [' || p_policy_name || ']' ||
          ' Metric Name = [' || p_metric_name || ']',
          G_MODULE_NAME);
    END IF;

    -- Check and see if the metric_name provided is valid
    l_metric_guid := MGMT_METRIC.get_metric_guid(p_target_type, p_metric_name);

    l_idx := 1;
    l_metric_col_list   := MGMT_SHORT_STRING_ARRAY();
    l_policy_param_list := MGMT_SHORT_STRING_ARRAY();
    l_bind_var_list     := MGMT_NAMEVALUE_ARRAY();

    -- Check condition operator
    IF p_condition_type = MGMT_GLOBAL.G_CONDITION_THRESHOLD THEN

       EM_CHECK.check_not_null(p_condition_operator, 'p_condition_operator');
       EM_CHECK.check_range(p_condition_operator, 
                            MGMT_GLOBAL.G_THRESHOLD_MIN,
                            MGMT_GLOBAL.G_THRESHOLD_MAX,
                            'p_condition_operator');

    ELSIF (p_condition_type = MGMT_GLOBAL.G_CONDITION_SQL) THEN

      IF EMDW_LOG.P_IS_DEBUG_SET THEN 
          EMDW_LOG.DEBUG('create_policy'||'Fetching metric column for ' || p_metric_name, 'POLICY') ; 
      END IF ;
      -- Get all metric columns
      FOR crec IN (SELECT DISTINCT(metric_column) FROM MGMT_METRICS
                    WHERE target_type = p_target_type
                      AND metric_name = p_metric_name
                      AND metric_column <> ' ')
      LOOP
        l_metric_col_list.extend();
        l_metric_col_list(l_idx) := crec.metric_column;
        l_idx := l_idx + 1;
      END LOOP;
              

      -- Get all policy parameters
      IF  ( (p_policy_param_list IS NOT NULL) AND (p_policy_param_list.COUNT > 0) )THEN
        l_policy_param_list := MGMT_SHORT_STRING_ARRAY();
        FOR param_ctr IN p_policy_param_list.FIRST..p_policy_param_list.LAST
        LOOP
          l_policy_param_list.extend();
          l_policy_param_list(param_ctr) := p_policy_param_list(param_ctr).param_name;
        END LOOP;
      END IF;

      l_cond_stat := EM_POLICY_EVAL.is_valid_condition(
                       p_condition_text => p_condition, 
                       p_param_columns => l_policy_param_list,
                       p_metric_columns => l_metric_col_list, 
                       p_bind_columns => l_bind_var_list,
                       p_error_message => l_err_msg);

      IF (l_cond_stat = FALSE) THEN
        raise_application_error(MGMT_GLOBAL.INVALID_PARAMS_ERR,
           'Invalid SQL condition. Error = ' || l_err_msg);
      END IF;

    ELSIF (p_condition_type = MGMT_GLOBAL.G_CONDITION_PLSQL) THEN
      
      l_arg_type_list := MGMT_SHORT_STRING_ARRAY('MGMT_METRIC_VALUE_OBJ', 
                             'MGMT_POLICY_PARAM_VAL_ARRAY', 'NUMBER', 'NUMBER', 'NUMBER');

      IF (EM_CHECK.is_valid_signature(p_condition, l_arg_type_list, l_err_msg) = FALSE) 
      THEN
          raise_application_error(MGMT_GLOBAL.INVALID_PARAMS_ERR,
             'Invalid PLSQL condition. Error = ' || l_err_msg);
      END IF;

    END IF;

    -- Generate policy guid
    l_policy_guid := generate_policy_guid(p_target_type, p_policy_name);

    l_owner := l_current_user;
    IF (p_author = MGMT_GLOBAL.G_AUTHOR_ORACLE) THEN
      l_owner := '<SYSTEM>';
    END IF;


    EM_POLICY.add_policy(
        p_policy_guid => l_policy_guid,
        p_target_type => p_target_type,
        p_policy_name => p_policy_name,
        p_metric_guid => l_metric_guid,
        p_start_type_meta_ver  => p_start_type_meta_ver,
        p_end_type_meta_ver    => p_end_type_meta_ver,
        p_policy_type => MGMT_GLOBAL.G_TYPE_POLICY,
        p_policy_label_nlsid => p_policy_label_nlsid,
        p_description => p_description,
        p_description_nlsid => p_description_nlsid,
        p_author => p_author,
        p_auto_enable => p_auto_enable,
        p_cs_consider_percentage => p_cs_consider_percentage,
        p_impact => p_impact,
        p_impact_nlsid => p_impact_nlsid,
        p_recommendation => p_recommendation,
        p_recommendation_nlsid => p_recommendation_nlsid,
        p_violation_level => p_violation_level,
        p_condition_type => p_condition_type,
        p_condition => p_condition,
        p_condition_operator => p_condition_operator,
        p_detailed_url_link => p_detailed_url_link,
        p_message  => p_message,
        p_message_nlsid  => p_message_nlsid,
        p_clear_message  => p_clear_message,
        p_clear_message_nlsid  => p_clear_message_nlsid,
        p_owner => l_owner,
        p_last_updated_by => l_current_user);

    -- Insert bind vars
    IF  ( (l_bind_var_list IS NOT NULL) AND (l_bind_var_list.COUNT > 0) )THEN
      FOR bind_ctr IN l_bind_var_list.FIRST..l_bind_var_list.LAST
      LOOP

        EM_POLICY.add_policy_bind_var(l_policy_guid, 
           l_bind_var_list(bind_ctr).name, l_bind_var_list(bind_ctr).value);

      END LOOP;
    END IF;

    -- Insert into categories
    EM_METRIC.add_object_categories(l_policy_guid, MGMT_GLOBAL.G_TYPE_POLICY,
                                    p_target_type, '1.0', p_category_list);

    -- Insert into policy parameters
    IF  ( (p_policy_param_list IS NOT NULL) AND (p_policy_param_list.COUNT > 0) )THEN
      FOR param_ctr IN p_policy_param_list.FIRST..p_policy_param_list.LAST
      LOOP

        l_policy_param := p_policy_param_list(param_ctr);

        EM_POLICY.add_policy_parameter(l_policy_guid, l_policy_param.param_name,
            l_policy_param.param_name_nlsid, l_policy_param.param_type);

      END LOOP;
    END IF;

    -- Insert into violation definition
    IF  ( (p_viol_ctxt_list IS NOT NULL) AND (p_viol_ctxt_list.COUNT > 0) )THEN
      FOR viol_ctr IN p_viol_ctxt_list.FIRST..p_viol_ctxt_list.LAST
      LOOP

        l_viol_ctxt := p_viol_ctxt_list(viol_ctr);

        l_column_guid := mgmt_metric.get_metric_guid(p_target_type, 
                                                     p_metric_name, 
                                                     l_viol_ctxt.metric_column);

        EM_POLICY.add_viol_ctxt_def(l_policy_guid, l_viol_ctxt.metric_column, 
                          l_column_guid, viol_ctr, l_viol_ctxt.is_hidden,
                          l_viol_ctxt.url_link_type, l_viol_ctxt.url_link_template);

      END LOOP;
    END IF;

    -- Populate target type versions
    -- Get the list of type meta versions for which this policy has to be applied
    SELECT type_meta_ver
           BULK COLLECT INTO l_meta_ver_list
      FROM mgmt_target_type_versions
     WHERE target_type = p_target_type
       AND (em_target.compare_type_meta_vers(p_start_type_meta_ver, type_meta_ver) <= 0)
       AND ( (p_end_type_meta_ver IS NULL) OR
             (em_target.compare_type_meta_vers(type_meta_ver, p_end_type_meta_ver) <= 0) )
     UNION
     SELECT p_start_type_meta_ver
       FROM DUAL ;
    IF ( (l_meta_ver_list IS NOT NULL) AND (l_meta_ver_list.COUNT > 0) ) THEN

      FORALL mv_ctr IN l_meta_ver_list.FIRST..l_meta_ver_list.LAST
        INSERT INTO mgmt_policy_type_versions (policy_guid, type_meta_ver)
         VALUES (l_policy_guid, l_meta_ver_list(mv_ctr));

    ELSE

      -- This policy cannot be applied to any existint type meta versions.
      -- Do nothing..
      NULL;

    END IF;

    -- Store default parameter configuration
    EM_POLICY.add_object_policy_assoc(
        p_object_guid => l_default_guid, 
        p_policy_guid => l_policy_guid,
        p_coll_name   => p_metric_name,
        p_object_type => MGMT_GLOBAL.G_OBJECT_TYPE_DEFAULT,
        p_policy_type => MGMT_GLOBAL.G_TYPE_POLICY,
        p_is_enabled  => MGMT_GLOBAL.G_TRUE,
        p_policy_val_list => p_dflt_param_val_list); 

    IF EMDW_LOG.P_IS_DEBUG_SET THEN
      EMDW_LOG.DEBUG(l_proc_name || ' Exit.', G_MODULE_NAME);
    END IF;

  END create_policy;


  -- Modifies the given policy
  PROCEDURE modify_policy(
      p_target_type              IN VARCHAR2,
      p_policy_name              IN VARCHAR2,
      p_metric_name              IN VARCHAR2 DEFAULT NULL,
      p_author                   IN VARCHAR2 DEFAULT NULL,
      p_policy_label_nlsid       IN VARCHAR2 DEFAULT NULL,
      p_description              IN VARCHAR2 DEFAULT NULL,
      p_description_nlsid        IN VARCHAR2 DEFAULT NULL,
      p_impact                   IN VARCHAR2 DEFAULT NULL,
      p_impact_nlsid             IN VARCHAR2 DEFAULT NULL,
      p_recommendation           IN VARCHAR2 DEFAULT NULL,
      p_recommendation_nlsid     IN VARCHAR2 DEFAULT NULL,
      p_violation_level          IN NUMBER   DEFAULT NULL,
      p_condition_type           IN NUMBER   DEFAULT 1,
      p_condition                IN VARCHAR2 DEFAULT NULL,
      p_condition_operator       IN NUMBER   DEFAULT 0,
      p_detailed_url_link        IN VARCHAR2 DEFAULT NULL,
      p_message                  IN VARCHAR2 DEFAULT NULL,
      p_message_nlsid            IN VARCHAR2 DEFAULT NULL,
      p_clear_message            IN VARCHAR2 DEFAULT NULL,
      p_clear_message_nlsid      IN VARCHAR2 DEFAULT NULL,
      p_auto_enable              IN NUMBER   DEFAULT 0,
      p_cs_consider_percentage   IN NUMBER   DEFAULT 0,
      p_start_type_meta_ver      IN VARCHAR2 DEFAULT '1.0',
      p_end_type_meta_ver        IN VARCHAR2 DEFAULT NULL,
      p_category_list            IN MGMT_CATEGORY_ARRAY DEFAULT NULL,
      p_policy_param_list        IN MGMT_POLICY_PARAM_DEF_ARRAY DEFAULT NULL,
      p_dflt_param_val_list      IN MGMT_POLICY_KEY_VAL_ARRAY DEFAULT NULL,
      p_viol_ctxt_list           IN MGMT_VIOL_CTXT_DEF_ARRAY DEFAULT NULL)
  IS
    l_policy_guid mgmt_policies.policy_guid%TYPE;
    l_metric_guid mgmt_metrics.metric_guid%TYPE;
    l_column_guid mgmt_metrics.metric_guid%TYPE;
    l_default_guid mgmt_policies.policy_guid%TYPE;
    l_current_user      mgmt_created_users.user_name%TYPE;
    l_owner             mgmt_policies.owner%TYPE;
    l_category          MGMT_CATEGORY_OBJ;
    l_policy_param      MGMT_POLICY_PARAM_DEF;
    l_viol_ctxt         MGMT_VIOL_CTXT_DEF;
    l_key_val           MGMT_POLICY_KEY_VAL;
    l_param_val         MGMT_POLICY_PARAM_VAL;
    l_idx               INTEGER;
    l_metric_col_list   MGMT_SHORT_STRING_ARRAY;
    l_policy_param_list MGMT_SHORT_STRING_ARRAY;
    l_bind_var_list     MGMT_NAMEVALUE_ARRAY;
    l_cond_stat         BOOLEAN := FALSE;
    l_err_msg           VARCHAR2(4096);
    l_meta_ver_list     MGMT_SHORT_STRING_ARRAY;
    l_ca_id_list        MGMT_TARGET_GUID_ARRAY;
    l_proc_name         VARCHAR2(32) := 'modify_policy';
  BEGIN
    l_default_guid := EM_TARGET.generate_target_type_guid(p_target_type);

    l_current_user := MGMT_USER.GET_CURRENT_EM_USER;
    -- Check super user privilege..
    EM_CHECK.check_super_user_priv;

    -- Check for NULLs
    EM_CHECK.check_not_null(p_target_type, 'p_target_type');
    EM_CHECK.check_not_null(p_policy_name, 'p_policy_name');
    EM_CHECK.check_not_null(p_metric_name, 'p_metric_name');
  
    -- Check condition operator 
    IF p_condition_type = MGMT_GLOBAL.G_CONDITION_THRESHOLD THEN
       EM_CHECK.check_not_null(p_condition_operator, 'p_condition_operator');
       EM_CHECK.check_range(p_condition_operator, 
                            MGMT_GLOBAL.G_THRESHOLD_MIN,
                            MGMT_GLOBAL.G_THRESHOLD_MAX,
                            'p_condition_operator');
    END IF;

    IF EMDW_LOG.P_IS_DEBUG_SET THEN
      EMDW_LOG.DEBUG(l_proc_name || 
          ' Target type = [' || p_target_type || ']' ||
          ' Policy Name = [' || p_policy_name || ']' ||
          ' Metric Name = [' || p_metric_name || ']',
          G_MODULE_NAME);
    END IF;

    -- Get policy guid
    l_policy_guid := get_policy_guid(p_target_type, p_policy_name);

    -- generate metric guid
    l_metric_guid := MGMT_METRIC.get_metric_guid(p_target_type, p_metric_name);

    l_idx := 1;
    l_metric_col_list   := MGMT_SHORT_STRING_ARRAY();
    l_policy_param_list := MGMT_SHORT_STRING_ARRAY();
    l_bind_var_list     := MGMT_NAMEVALUE_ARRAY();

    IF ( (p_condition_type IS NOT NULL) AND (p_condition_type = MGMT_GLOBAL.G_CONDITION_SQL)) 
    THEN

      IF EMDW_LOG.P_IS_DEBUG_SET THEN 
          EMDW_LOG.DEBUG('modify_policy'||'Fetching metric column for ' || p_metric_name, 
             'POLICY') ; 
      END IF ;
      -- Get all metric columns
      FOR crec IN (SELECT DISTINCT(metric_column) FROM MGMT_METRICS
                    WHERE target_type = p_target_type
                      AND metric_name = p_metric_name
                      AND metric_column <> ' ')
      LOOP
        l_metric_col_list.extend();
        l_metric_col_list(l_idx) := crec.metric_column;
        l_idx := l_idx + 1;
      END LOOP;
              

      -- Get all policy parameters
      IF  ( (p_policy_param_list IS NOT NULL) AND (p_policy_param_list.COUNT > 0) )THEN
        l_policy_param_list := MGMT_SHORT_STRING_ARRAY();
        FOR param_ctr IN p_policy_param_list.FIRST..p_policy_param_list.LAST
        LOOP
          l_policy_param_list.extend();
          l_policy_param_list(param_ctr) := p_policy_param_list(param_ctr).param_name;
        END LOOP;
      END IF;

      l_cond_stat := EM_POLICY_EVAL.is_valid_condition(
                       p_condition_text => p_condition, 
                       p_param_columns => l_policy_param_list,
                       p_metric_columns => l_metric_col_list, 
                       p_bind_columns => l_bind_var_list,
                       p_error_message => l_err_msg);

      IF (l_cond_stat = FALSE) THEN
        raise_application_error(MGMT_GLOBAL.INVALID_PARAMS_ERR,
           'Invalid SQL condition. Error = ' || l_err_msg);
      END IF;

    END IF;

    
    l_owner := l_current_user;
    IF (p_author = MGMT_GLOBAL.G_AUTHOR_ORACLE) THEN
      l_owner := '<SYSTEM>';
    END IF;

    EM_POLICY.update_policy(
        p_policy_guid => l_policy_guid,
        p_target_type => p_target_type,
        p_policy_name => p_policy_name,
        p_metric_guid => l_metric_guid,
        p_start_type_meta_ver  => p_start_type_meta_ver,
        p_end_type_meta_ver    => p_end_type_meta_ver,
        p_policy_type => MGMT_GLOBAL.G_TYPE_POLICY,
        p_policy_label_nlsid => p_policy_label_nlsid,
        p_description => p_description,
        p_description_nlsid => p_description_nlsid,
        p_author => p_author,
        p_auto_enable => p_auto_enable,
        p_cs_consider_percentage => p_cs_consider_percentage,
        p_impact => p_impact,
        p_impact_nlsid => p_impact_nlsid,
        p_recommendation => p_recommendation,
        p_recommendation_nlsid => p_recommendation_nlsid,
        p_violation_level => p_violation_level,
        p_condition_type => p_condition_type,
        p_condition => p_condition,
        p_condition_operator => p_condition_operator,
        p_detailed_url_link => p_detailed_url_link,
        p_message  => p_message,
        p_message_nlsid  => p_message_nlsid,
        p_clear_message  => p_clear_message,
        p_clear_message_nlsid  => p_clear_message_nlsid,
        p_owner => l_owner,
        p_last_updated_by => l_current_user);

    -- Delete old bind vars and add new bind vars
    EM_POLICY.remove_policy_bind_vars(l_policy_guid);
    IF  ( (l_bind_var_list IS NOT NULL) AND (l_bind_var_list.COUNT > 0) )THEN
      FOR bind_ctr IN l_bind_var_list.FIRST..l_bind_var_list.LAST
      LOOP

        EM_POLICY.add_policy_bind_var(l_policy_guid, 
           l_bind_var_list(bind_ctr).name, l_bind_var_list(bind_ctr).value);

      END LOOP;
    END IF;

    -- Delete old category map and add category map
    IF  ( (p_category_list IS NOT NULL) AND (p_category_list.COUNT > 0) )THEN

      EM_METRIC.remove_object_categories(l_policy_guid, 
                                 MGMT_GLOBAL.G_TYPE_POLICY, p_target_type);
      EM_METRIC.add_object_categories(l_policy_guid, MGMT_GLOBAL.G_TYPE_POLICY,
                                    p_target_type, '1.0', p_category_list);
    END IF;

    -- Delete old policy parameters and add policy parameters
    IF  ( (p_policy_param_list IS NOT NULL) AND (p_policy_param_list.COUNT > 0) )THEN

      EM_POLICY.remove_policy_parameters(l_policy_guid);

      FOR param_ctr IN p_policy_param_list.FIRST..p_policy_param_list.LAST
      LOOP

        l_policy_param := p_policy_param_list(param_ctr);

        EM_POLICY.add_policy_parameter(l_policy_guid, l_policy_param.param_name,
            l_policy_param.param_name_nlsid, l_policy_param.param_type);

      END LOOP;
    END IF;

    -- Delete old definition and add new violation definition
    IF  ( (p_viol_ctxt_list IS NOT NULL) AND (p_viol_ctxt_list.COUNT > 0) )THEN
      EM_POLICY.remove_viol_ctxt_defs(l_policy_guid);
      FOR viol_ctr IN p_viol_ctxt_list.FIRST..p_viol_ctxt_list.LAST
      LOOP

        l_viol_ctxt := p_viol_ctxt_list(viol_ctr);

        l_column_guid := mgmt_metric.get_metric_guid(p_target_type, 
                                                     p_metric_name, 
                                                     l_viol_ctxt.metric_column);

        EM_POLICY.add_viol_ctxt_def(l_policy_guid, l_viol_ctxt.metric_column, 
                          l_column_guid, viol_ctr, l_viol_ctxt.is_hidden,
                          l_viol_ctxt.url_link_type, l_viol_ctxt.url_link_template);

      END LOOP;
    END IF;


    IF (p_start_type_meta_ver IS NOT NULL) THEN

      -- Delete the old set of policy versions.
      EM_POLICY.remove_policy_type_version(l_policy_guid);

      -- Populate the new list of versions for which this policy is applicable
      SELECT type_meta_ver
             BULK COLLECT INTO l_meta_ver_list
        FROM mgmt_target_type_versions
       WHERE target_type = p_target_type
         AND (em_target.compare_type_meta_vers(p_start_type_meta_ver, type_meta_ver) <= 0)
         AND ( (p_end_type_meta_ver IS NULL) OR
               (em_target.compare_type_meta_vers(type_meta_ver, p_end_type_meta_ver) <= 0) );

      IF ( (l_meta_ver_list IS NOT NULL) AND (l_meta_ver_list.COUNT > 0) ) THEN

        FORALL mv_ctr IN l_meta_ver_list.FIRST..l_meta_ver_list.LAST
          INSERT INTO mgmt_policy_type_versions (policy_guid, type_meta_ver)
           VALUES (l_policy_guid, l_meta_ver_list(mv_ctr));

      ELSE

        INSERT INTO mgmt_policy_type_versions (policy_guid, type_meta_ver)
         VALUES (l_policy_guid, p_start_type_meta_ver);

      END IF;

    END IF;

    -- Delete old default parameter configuration and add the new one

    -- Lock all CAs associated with this target type
    MGMT_JOB_ENGINE.lock_cas_for_object(l_default_guid, 
                                        MGMT_CA.CA_SCOPE_TARGET_TYPE,
                                        p_target_type);

    l_ca_id_list := EM_POLICY.get_ca_id_list(
        p_object_guid     => l_default_guid,
        p_policy_guid     => l_policy_guid );

    EM_POLICY.remove_object_policy_assoc(
        p_object_guid     => l_default_guid,
        p_policy_guid     => l_policy_guid,
        p_coll_name       => NULL,
        p_remove_ca_assoc => MGMT_GLOBAL.G_FALSE);

    EM_POLICY.add_object_policy_assoc(
        p_object_guid => l_default_guid, 
        p_policy_guid => l_policy_guid,
        p_coll_name   => p_metric_name,
        p_object_type => MGMT_GLOBAL.G_OBJECT_TYPE_DEFAULT,
        p_policy_type => MGMT_GLOBAL.G_TYPE_POLICY,
        p_is_enabled  => MGMT_GLOBAL.G_TRUE,
        p_policy_val_list => p_dflt_param_val_list); 

    -- Decrement the CA ids ref counter. Zero counters are automatically deleted
    EM_POLICY.decrement_ca_ids(l_ca_id_list);

    IF EMDW_LOG.P_IS_DEBUG_SET THEN
      EMDW_LOG.DEBUG(l_proc_name || ' Exit.', G_MODULE_NAME);
    END IF;

  END modify_policy;


  -- NOTE: This procedure does intermittent commits. Use it with care
  --   If only metadata has to be deleted, use delete_policy_metadata
  PROCEDURE delete_policy(
      p_target_type              IN VARCHAR2,
      p_policy_name              IN VARCHAR2)
  IS
    l_policy_guid mgmt_policies.policy_guid%TYPE;
    l_proc_name     VARCHAR2(32) := 'delete_policy';
  BEGIN
    -- Check super user privilege..
    EM_CHECK.check_super_user_priv;

    -- Check for NULLs
    EM_CHECK.check_not_null(p_target_type, 'p_target_type');
    EM_CHECK.check_not_null(p_policy_name, 'p_policy_name');

    IF EMDW_LOG.P_IS_DEBUG_SET THEN
      EMDW_LOG.DEBUG(l_proc_name || 
          ' Target type = [' || p_target_type || ']' ||
          ' Policy Name = [' || p_policy_name || ']',
          G_MODULE_NAME);
    END IF;

    l_policy_guid := get_policy_guid(p_target_type, p_policy_name);

    -- Delete policy data
    EM_POLICY.delete_policy_data(p_target_type, p_policy_name);

    -- Delete the policy metadata
    delete_policy_metadata(p_target_type, p_policy_name);

    IF EMDW_LOG.P_IS_DEBUG_SET THEN
      EMDW_LOG.DEBUG(l_proc_name || ' Exit.', G_MODULE_NAME);
    END IF;

  END delete_policy;


  -- Deletes policy metadata
  -- This can be used along with create_policy to recreate the policy
  -- without affecting the policy associations
  PROCEDURE delete_policy_metadata(
      p_target_type              IN VARCHAR2,
      p_policy_name              IN VARCHAR2)
  IS
    l_policy_guid mgmt_policies.policy_guid%TYPE;
    l_default_guid mgmt_policy_assoc.object_guid%TYPE;
    l_proc_name     VARCHAR2(32) := 'delete_policy_metadata ';
  BEGIN
    -- Check super user privilege..
    EM_CHECK.check_super_user_priv;

    -- Check for NULLs
    EM_CHECK.check_not_null(p_target_type, 'p_target_type');
    EM_CHECK.check_not_null(p_policy_name, 'p_policy_name');

    IF EMDW_LOG.P_IS_DEBUG_SET THEN
      EMDW_LOG.DEBUG(l_proc_name || 
          ' Target type = [' || p_target_type || ']' ||
          ' Policy Name = [' || p_policy_name || ']',
          G_MODULE_NAME);
    END IF;

    l_default_guid := EM_TARGET.generate_target_type_guid(p_target_type);
    l_policy_guid := get_policy_guid(p_target_type, p_policy_name);

    -- Delete default policy configuration, and any CA assocations
    EM_POLICY.remove_object_policy_assoc(
        p_object_guid     => l_default_guid,
        p_policy_guid     => l_policy_guid,
        p_coll_name       => NULL,
        p_remove_ca_assoc => MGMT_GLOBAL.G_TRUE);

    -- Delete policy type versions..
    EM_POLICY.remove_policy_type_version(l_policy_guid);

    -- Delete violation context
    EM_POLICY.remove_viol_ctxt_defs(l_policy_guid);

    -- Delete policy parameters
    EM_POLICY.remove_policy_parameters(l_policy_guid);

    -- Delete bind parameters
    EM_POLICY.remove_policy_bind_vars(l_policy_guid);

    -- Delete object categorization
    EM_METRIC.remove_object_categories(l_policy_guid, 
        MGMT_GLOBAL.G_TYPE_POLICY, p_target_type);

    -- Delete policy entry
    EM_POLICY.remove_policy(l_policy_guid);

    IF EMDW_LOG.P_IS_DEBUG_SET THEN
      EMDW_LOG.DEBUG(l_proc_name || ' Exit.', G_MODULE_NAME);
    END IF;

  END delete_policy_metadata;


  -- Sets the end type metaver for a given policy
  PROCEDURE set_end_type_meta_ver(
      p_target_type              IN VARCHAR2,
      p_policy_name              IN VARCHAR2,
      p_end_type_meta_ver        IN VARCHAR2)
  IS
    l_old_start_tmv mgmt_metrics.type_meta_ver%TYPE;
    l_old_end_tmv   mgmt_metrics.type_meta_ver%TYPE;
    l_policy_guid   mgmt_policies.policy_guid%TYPE;
    l_meta_ver_list MGMT_SHORT_STRING_ARRAY;
    l_auto_enable   mgmt_policies.auto_enable%TYPE;
    l_metric_guid   mgmt_metrics.metric_guid%TYPE;
    l_metric_name   mgmt_metrics.metric_name%TYPE;
    l_default_guid  RAW(16);
    l_ca_id_list    MGMT_TARGET_GUID_ARRAY;
    l_comp          NUMBER := 0;
  BEGIN
    -- Check super user privilege..
    EM_CHECK.check_super_user_priv;

    -- Check for NULLs
    EM_CHECK.check_not_null(p_target_type, 'p_target_type');
    EM_CHECK.check_not_null(p_policy_name, 'p_policy_name');

    l_policy_guid := get_policy_guid(p_target_type, p_policy_name);

    SELECT start_type_meta_ver, end_type_meta_ver, auto_enable, metric_guid
      INTO l_old_start_tmv, l_old_end_tmv, l_auto_enable, l_metric_guid
      FROM mgmt_policies
     WHERE policy_guid = l_policy_guid;

    SELECT DISTINCT metric_name
      INTO l_metric_name
      FROM mgmt_metrics
     WHERE metric_guid = l_metric_guid;

    l_comp := EM_TARGET.compare_type_meta_vers(
                 p_type_meta_ver_1 => NVL(l_old_end_tmv, '99999.9999'),
                 p_type_meta_ver_2 => NVL(p_end_type_meta_ver, '99999.9999') );

    IF (l_comp > 0) THEN
      -- old_tmv > new_tmv
      -- Delete any outscoped policy versions

      SELECT DISTINCT type_meta_ver
             BULK COLLECT INTO l_meta_ver_list
        FROM mgmt_policy_type_versions
       WHERE policy_guid = l_policy_guid
         AND (em_target.compare_type_meta_vers(type_meta_ver, 
                        NVL(p_end_type_meta_ver, '99999.99999')) > 0);

      FOR mv_ctr IN l_meta_ver_list.FIRST..l_meta_ver_List.LAST
      LOOP
        DELETE FROM mgmt_policy_type_versions
         WHERE policy_guid = l_policy_guid
           AND type_meta_ver = l_meta_ver_list(mv_ctr);

        -- TODO: Need to remove the policy assoc and viol data
      END LOOP;

    ELSIF (l_comp < 0) THEN
      -- old_tmv < new_tmv
      -- Copy the metric to new versions

      SELECT type_meta_ver
             BULK COLLECT INTO l_meta_ver_list
        FROM mgmt_target_type_versions
       WHERE target_type = p_target_type
         AND (em_target.compare_type_meta_vers(type_meta_ver, 
                        NVL(l_old_end_tmv, '99999.99999')) >= 0);

      FOR mv_ctr IN l_meta_ver_list.FIRST..l_meta_ver_List.LAST
      LOOP
        INSERT INTO mgmt_policy_type_versions (policy_guid, type_meta_ver)
         VALUES (l_policy_guid, l_meta_ver_list(mv_ctr));
      END LOOP;

      IF (l_auto_enable = MGMT_GLOBAL.G_TRUE) THEN

        l_default_guid := EM_TARGET.generate_target_type_guid(p_target_type);

        FOR tgt_rec IN (SELECT target_guid FROM mgmt_targets
                         WHERE target_type = p_target_type
                           AND type_meta_ver IN 
                                (SELECT * FROM TABLE(CAST(l_meta_ver_list 
                                    AS MGMT_SHORT_STRING_ARRAY))) )
        LOOP
          -- Lock all CAs associated with this target 
          MGMT_JOB_ENGINE.lock_cas_for_object(tgt_rec.target_guid, MGMT_CA.CA_SCOPE_TARGET);

          l_ca_id_list := EM_POLICY.get_ca_id_list(
            p_object_guid     => tgt_rec.target_guid,
            p_policy_guid     => l_policy_guid,
            p_coll_name       => l_metric_name);

          -- Remove existing policy association for this target
          EM_POLICY.remove_object_policy_assoc(      
              p_object_guid     => tgt_rec.target_guid,
              p_policy_guid     => l_policy_guid,
              p_coll_name       => l_metric_name,
              p_remove_ca_assoc => MGMT_GLOBAL.G_FALSE);

          -- Copy the default configuration.
          -- Use metric name as the collection name
          EM_POLICY.copy_object_policy_assoc(      
            p_src_object_guid     => l_default_guid,
            p_src_object_type     => MGMT_GLOBAL.G_OBJECT_TYPE_DEFAULT,
            p_dest_object_guid    => tgt_rec.target_guid,
            p_dest_object_type    => MGMT_GLOBAL.G_OBJECT_TYPE_TARGET,
            p_policy_guid         => l_policy_guid,
            p_policy_type         => MGMT_GLOBAL.G_TYPE_POLICY,
            p_src_coll_name       => l_metric_name,
            p_dest_coll_name      => l_metric_name);

          -- Decrement the CA ids ref counter. Zero counters are automatically deleted
          EM_POLICY.decrement_ca_ids(l_ca_id_list);

        END LOOP;
      END IF;

    END IF;

    UPDATE mgmt_policies
       SET end_type_meta_ver = p_end_type_meta_ver
     WHERE policy_guid = l_policy_guid;

  END set_end_type_meta_ver;


  --
  -- PROCEDURE: auto_enable_existing_targets
  --
  -- PURPOSE: 
  --   Associates the given policy to all existing targets of the same type.
  --   This procedure removes the existing configuration for the policy and
  --   copies the default configuration of the policy to all the targets.
  -- 
  PROCEDURE auto_enable_existing_targets(
      p_target_type              IN VARCHAR2,
      p_policy_name              IN VARCHAR2)
  IS
    l_policy_guid   mgmt_policies.policy_guid%TYPE;
    l_default_guid  mgmt_policy_assoc.object_guid%TYPE;
    l_ca_id_list    MGMT_TARGET_GUID_ARRAY;
    l_proc_name     VARCHAR2(32) := 'auto_enable_existing_targets';
  BEGIN

    -- Check super user privilege..
    -- Auto associating to existing targets require super user privileges, as
    -- this is always done immediately after policy creation.
    EM_CHECK.check_super_user_priv;

    -- Check for NOT NULL.
    EM_CHECK.check_not_null(p_target_type, 'p_target_type');
    EM_CHECK.check_not_null(p_policy_name, 'p_policy_name');

    IF EMDW_LOG.P_IS_DEBUG_SET THEN
      EMDW_LOG.DEBUG(l_proc_name || 
          ' Target type = [' || p_target_type || ']' ||
          ' Policy Name = [' || p_policy_name || ']',
          G_MODULE_NAME);
    END IF;

    l_default_guid := EM_TARGET.generate_target_type_guid(p_target_type);
    l_policy_guid := get_policy_guid(p_target_type, p_policy_name);

    -- Pick policies that are applicable
    --   policy should be applicable to target type meta ver and
    --   metric on which the policy is defined is applicable to target.
    FOR tgt_rec IN (SELECT t.target_guid, m.metric_name
                      FROM mgmt_targets t, mgmt_metrics m,
                           mgmt_policies p, mgmt_policy_type_versions ptv
                     WHERE p.target_type =  m.target_type
                       AND p.metric_guid = m.metric_guid
                       AND p.policy_guid = ptv.policy_guid
                       AND ptv.type_meta_ver = t.type_meta_ver
                       AND m.target_type = t.target_type
                       AND m.type_meta_ver = t.type_meta_ver
                       AND ( (m.category_prop_1 = t.category_prop_1) OR
                             (m.category_prop_1 = ' '))
                       AND ( (m.category_prop_2 = t.category_prop_2) OR
                             (m.category_prop_2 = ' '))
                       AND ( (m.category_prop_3 = t.category_prop_3) OR
                             (m.category_prop_3 = ' '))
                       AND ( (m.category_prop_4 = t.category_prop_4) OR
                             (m.category_prop_4 = ' '))
                       AND ( (m.category_prop_5 = t.category_prop_5) OR
                             (m.category_prop_5 = ' '))
                       AND p.auto_enable = MGMT_GLOBAL.G_TRUE
                       AND t.target_type = p_target_type
                       AND p.policy_guid = l_policy_guid
                     ORDER BY t.target_guid)
    LOOP

      -- Lock all CAs associated with this target type
      MGMT_JOB_ENGINE.lock_cas_for_object(tgt_rec.target_guid, MGMT_CA.CA_SCOPE_TARGET);

      l_ca_id_list := EM_POLICY.get_ca_id_list(
        p_object_guid     => tgt_rec.target_guid,
        p_policy_guid     => l_policy_guid,
        p_coll_name       => tgt_rec.metric_name);

      -- Remove existing policy association for this target
      EM_POLICY.remove_object_policy_assoc(      
          p_object_guid     => tgt_rec.target_guid,
          p_policy_guid     => l_policy_guid,
          p_coll_name       => tgt_rec.metric_name,
          p_remove_ca_assoc => MGMT_GLOBAL.G_FALSE);

      -- Copy the default configuration.
      -- Use metric name as the collection name
      EM_POLICY.copy_object_policy_assoc(      
        p_src_object_guid     => l_default_guid,
        p_src_object_type     => MGMT_GLOBAL.G_OBJECT_TYPE_DEFAULT,
        p_dest_object_guid    => tgt_rec.target_guid,
        p_dest_object_type    => MGMT_GLOBAL.G_OBJECT_TYPE_TARGET,
        p_policy_guid         => l_policy_guid,
        p_policy_type         => MGMT_GLOBAL.G_TYPE_POLICY,
        p_src_coll_name       => tgt_rec.metric_name,
        p_dest_coll_name      => tgt_rec.metric_name);

      -- Decrement the CA ids ref counter. Zero counters are automatically deleted
      EM_POLICY.decrement_ca_ids(l_ca_id_list);

      COMMIT;

    END LOOP;

    IF EMDW_LOG.P_IS_DEBUG_SET THEN
      EMDW_LOG.DEBUG(l_proc_name || ' Exit.', G_MODULE_NAME);
    END IF;

  END auto_enable_existing_targets;
  

  -- Adds the target policy association
  PROCEDURE add_target_policy_assoc(
      p_target_type        IN VARCHAR2,
      p_target_name        IN VARCHAR2,
      p_policy_name        IN VARCHAR2,
      p_coll_name          IN VARCHAR2 DEFAULT NULL,
      p_is_enabled         IN NUMBER DEFAULT 1,
      p_policy_val_list    IN MGMT_POLICY_KEY_VAL_ARRAY DEFAULT NULL)
  IS
    l_target_guid  mgmt_targets.target_guid%TYPE;
    l_default_guid mgmt_policy_assoc.object_guid%TYPE;
    l_policy_guid  mgmt_policies.policy_guid%TYPE;
    l_policy_info  EM_POLICY.POLICY_INFO_REC;
    l_metric_name  mgmt_metrics.metric_name%TYPE;
    l_coll_name    mgmt_policy_assoc.coll_name%TYPE;
    l_proc_name    VARCHAR2(32) := 'add_target_policy_assoc';
  BEGIN
    EM_CHECK.check_not_null(p_target_type, 'p_target_type');
    EM_CHECK.check_not_null(p_target_name, 'p_target_name');
    EM_CHECK.check_not_null(p_policy_name, 'p_policy_name');

    IF EMDW_LOG.P_IS_DEBUG_SET THEN
      EMDW_LOG.DEBUG(l_proc_name || 
          ' Target type = [' || p_target_type || ']' ||
          ' Target name = [' || p_target_name || ']' ||
          ' Policy Name = [' || p_policy_name || ']' ||
          ' Enabled = [' || p_is_enabled || ']',
          G_MODULE_NAME);
      EMDW_LOG.DEBUG(l_proc_name || ' BEGIN DUMP ------------', G_MODULE_NAME);
      EM_POLICY.dump_key_val_list(p_policy_val_list, l_proc_name, G_MODULE_NAME);
      EMDW_LOG.DEBUG(l_proc_name || ' END DUMP ------------', G_MODULE_NAME);
    END IF;

    l_target_guid := MGMT_TARGET.get_target_guid(p_target_name, p_target_type);
    l_policy_guid := MGMT_POLICY.get_policy_guid(p_target_type, p_policy_name);

    -- Verify if the policy is applicable to the target
    EM_POLICY.get_policy_info_for_target(l_policy_guid, l_target_guid, l_policy_info);

    IF (l_policy_info.policy_guid IS NOT NULL) THEN

      -- Use the metric name as coll name if no coll_name is specified
      l_coll_name := NVL(p_coll_name, l_policy_info.metric_name);

    ELSE
      -- raise exception with error saying this policy is not applicable to
      -- the target
      raise_application_error(MGMT_GLOBAL.INVALID_PARAMS_ERR,
          'Policy policy_name = ' || p_policy_name ||
          ' is not applicable to target ' || p_target_name);
    END IF;
    
    IF ( (p_policy_val_list IS NOT NULL)  AND (p_policy_val_list.COUNT > 0) ) THEN

      EM_POLICY.add_object_policy_assoc(
        p_object_guid => l_target_guid, 
        p_policy_guid => l_policy_guid,
        p_coll_name   => l_coll_name,
        p_object_type => MGMT_GLOBAL.G_OBJECT_TYPE_TARGET,
        p_policy_type => MGMT_GLOBAL.G_TYPE_POLICY,
        p_is_enabled  => p_is_enabled,
        p_policy_val_list => p_policy_val_list);

    ELSE

      l_default_guid := EM_TARGET.generate_target_type_guid(p_target_type);

      -- Here the default settings are copied, so the
      -- the collection name picked up is the default coll name
      EM_POLICY.copy_object_policy_assoc(
        p_src_object_guid     => l_default_guid,
        p_src_object_type     => MGMT_GLOBAL.G_OBJECT_TYPE_DEFAULT,
        p_dest_object_guid    => l_target_guid,
        p_dest_object_type    => MGMT_GLOBAL.G_OBJECT_TYPE_TARGET,
        p_policy_guid         => l_policy_guid,
        p_policy_type         => MGMT_GLOBAL.G_TYPE_POLICY,
        p_src_coll_name       => l_coll_name);
      
    END IF;

    IF EMDW_LOG.P_IS_DEBUG_SET THEN
      EMDW_LOG.DEBUG(l_proc_name || ' Exit.', G_MODULE_NAME);
    END IF;

  END add_target_policy_assoc;


  -- Enables target policy association
  PROCEDURE enable_target_policy_assoc(
      p_target_type        IN VARCHAR2,
      p_target_name        IN VARCHAR2,
      p_policy_name        IN VARCHAR2,
      p_coll_name          IN VARCHAR2 DEFAULT NULL)
  IS
    l_target_guid mgmt_targets.target_guid%TYPE;
    l_policy_guid mgmt_policies.policy_guid%TYPE;
    l_coll_name    mgmt_policy_assoc.coll_name%TYPE;
    l_policy_info  EM_POLICY.POLICY_INFO_REC;
    l_proc_name     VARCHAR2(32) := 'enable_target_policy_assoc';
  BEGIN
    EM_CHECK.check_not_null(p_target_type, 'p_target_type');
    EM_CHECK.check_not_null(p_target_name, 'p_target_name');
    EM_CHECK.check_not_null(p_policy_name, 'p_policy_name');

    IF EMDW_LOG.P_IS_DEBUG_SET THEN
      EMDW_LOG.DEBUG(l_proc_name || 
          ' Target type = [' || p_target_type || ']' ||
          ' Target name = [' || p_target_name || ']' ||
          ' Policy Name = [' || p_policy_name || ']',
          G_MODULE_NAME);
    END IF;

    l_target_guid := MGMT_TARGET.get_target_guid(p_target_name, p_target_type);
    l_policy_guid := MGMT_POLICY.get_policy_guid(p_target_type, p_policy_name);

    -- Verify if the policy is applicable to the target
    EM_POLICY.get_policy_info_for_target(l_policy_guid, l_target_guid, l_policy_info);

    IF (l_policy_info.policy_guid IS NOT NULL) THEN

      -- Use the metric name as coll name if no coll_name is specified
      l_coll_name := NVL(p_coll_name, l_policy_info.metric_name);

    ELSE
      -- raise exception with error saying this policy is not applicable to
      -- the target
      raise_application_error(MGMT_GLOBAL.INVALID_PARAMS_ERR,
          'Policy policy_name = ' || p_policy_name ||
          ' is not applicable to target ' || p_target_name);
    END IF;
    
    -- TODO : Verify that the association and configuration exists..
    EM_POLICY.update_policy_assoc(
        p_object_guid => l_target_guid,
        p_policy_guid => l_policy_guid,
        p_coll_name   => l_coll_name,
        p_is_enabled  => MGMT_GLOBAL.G_TRUE);

    IF EMDW_LOG.P_IS_DEBUG_SET THEN
      EMDW_LOG.DEBUG(l_proc_name || ' Exit.', G_MODULE_NAME);
    END IF;

  END enable_target_policy_assoc;
     
  -- Disables target policy association
  PROCEDURE disable_target_policy_assoc(
      p_target_type        IN VARCHAR2,
      p_target_name        IN VARCHAR2,
      p_policy_name        IN VARCHAR2,
      p_coll_name          IN VARCHAR2 DEFAULT NULL)
  IS
    l_target_guid mgmt_targets.target_guid%TYPE;
    l_policy_guid mgmt_policies.policy_guid%TYPE;
    l_coll_name    mgmt_policy_assoc.coll_name%TYPE;
    l_policy_info  EM_POLICY.POLICY_INFO_REC;
    l_proc_name     VARCHAR2(32) := 'disable_target_policy_assoc';
  BEGIN
    EM_CHECK.check_not_null(p_target_type, 'p_target_type');
    EM_CHECK.check_not_null(p_target_name, 'p_target_name');
    EM_CHECK.check_not_null(p_policy_name, 'p_policy_name');

    IF EMDW_LOG.P_IS_DEBUG_SET THEN
      EMDW_LOG.DEBUG(l_proc_name || 
          ' Target type = [' || p_target_type || ']' ||
          ' Target name = [' || p_target_name || ']' ||
          ' Policy Name = [' || p_policy_name || ']',
          G_MODULE_NAME);
    END IF;

    l_target_guid := MGMT_TARGET.get_target_guid(p_target_name, p_target_type);
    l_policy_guid := MGMT_POLICY.get_policy_guid(p_target_type, p_policy_name);

    -- Verify if the policy is applicable to the target
    EM_POLICY.get_policy_info_for_target(l_policy_guid, l_target_guid, l_policy_info);

    IF (l_policy_info.policy_guid IS NOT NULL) THEN

      -- Use the metric name as coll name if no coll_name is specified
      l_coll_name := NVL(p_coll_name, l_policy_info.metric_name);

    ELSE
      -- raise exception with error saying this policy is not applicable to
      -- the target
      raise_application_error(MGMT_GLOBAL.INVALID_PARAMS_ERR,
          'Policy policy_name = ' || p_policy_name ||
          ' is not applicable to target ' || p_target_name);
    END IF;

    -- TODO : Verify that the association and configuration exists..
    EM_POLICY.update_policy_assoc(
        p_object_guid => l_target_guid,
        p_policy_guid => l_policy_guid,
        p_coll_name   => l_coll_name,
        p_is_enabled  => MGMT_GLOBAL.G_FALSE);

    IF EMDW_LOG.P_IS_DEBUG_SET THEN
      EMDW_LOG.DEBUG(l_proc_name || ' Exit.', G_MODULE_NAME);
    END IF;

  END disable_target_policy_assoc;

  -- Removes target policy association
  -- NOTE : If p_coll_name passed is NULL, all policy assocs are removed
  PROCEDURE remove_target_policy_assoc(
      p_target_type        IN VARCHAR2,
      p_target_name        IN VARCHAR2,
      p_policy_name        IN VARCHAR2,
      p_coll_name          IN VARCHAR2 DEFAULT NULL)
  IS
    l_target_guid mgmt_targets.target_guid%TYPE;
    l_policy_guid mgmt_policies.policy_guid%TYPE;
    l_proc_name     VARCHAR2(32) := 'remove_target_policy_assoc';
  BEGIN
    EM_CHECK.check_not_null(p_target_type, 'p_target_type');
    EM_CHECK.check_not_null(p_target_name, 'p_target_name');
    EM_CHECK.check_not_null(p_policy_name, 'p_policy_name');

    IF EMDW_LOG.P_IS_DEBUG_SET THEN
      EMDW_LOG.DEBUG(l_proc_name || 
          ' Target type = [' || p_target_type || ']' ||
          ' Target name = [' || p_target_name || ']' ||
          ' Policy Name = [' || p_policy_name || ']',
          G_MODULE_NAME);
    END IF;

    l_target_guid := MGMT_TARGET.get_target_guid(p_target_name, p_target_type);
    l_policy_guid := MGMT_POLICY.get_policy_guid(p_target_type, p_policy_name);

    -- No need to verify that the association and configuration exists..
    -- Remove policy assoc and any CAs associated with it.
    EM_POLICY.remove_object_policy_assoc(
        p_object_guid     => l_target_guid,
        p_policy_guid     => l_policy_guid,
        p_coll_name       => p_coll_name,
        p_remove_ca_assoc => MGMT_GLOBAL.G_TRUE);

    -- Remove policy evaluation state
    EM_POLICY.remove_policy_eval_details(l_target_guid, l_policy_guid, p_coll_name);
    EM_POLICY.remove_policy_eval_summ(l_target_guid, l_policy_guid, p_coll_name);

    -- Remove violations
    DELETE FROM mgmt_violations
     WHERE target_guid = l_target_guid
       AND policy_guid = l_policy_guid;

    IF EMDW_LOG.P_IS_DEBUG_SET THEN
      EMDW_LOG.DEBUG(l_proc_name || ' Exit.', G_MODULE_NAME);
    END IF;

  END remove_target_policy_assoc;

  PROCEDURE add_template_policy_assoc(
      p_target_type        IN VARCHAR2,
      p_template_name      IN VARCHAR2,
      p_policy_name        IN VARCHAR2,
      p_coll_name          IN VARCHAR2 DEFAULT NULL,
      p_is_enabled         IN NUMBER   DEFAULT 1,
      p_policy_val_list    IN MGMT_POLICY_KEY_VAL_ARRAY DEFAULT NULL)
  IS
    l_template_guid mgmt_templates.template_guid%TYPE;
    l_policy_guid mgmt_policies.policy_guid%TYPE;
    l_policy_info  EM_POLICY.POLICY_INFO_REC;
    l_coll_name    mgmt_policy_assoc.coll_name%TYPE;
    l_proc_name     VARCHAR2(32) := 'add_template_policy_assoc';
  BEGIN
    EM_CHECK.check_not_null(p_target_type, 'p_target_type');
    EM_CHECK.check_not_null(p_template_name, 'p_template_name');
    EM_CHECK.check_not_null(p_policy_name, 'p_policy_name');

    l_template_guid := MGMT_TEMPLATE.get_template_guid(p_target_type, p_template_name);
    l_policy_guid := MGMT_POLICY.get_policy_guid(p_target_type, p_policy_name);

    -- Verify if the policy is applicable to the target
    EM_POLICY.get_policy_info(l_policy_guid, l_policy_info);

    IF (l_policy_info.policy_guid IS NOT NULL) THEN

      -- Use the metric name as coll name if no coll_name is specified
      l_coll_name := NVL(p_coll_name, l_policy_info.metric_name);

    ELSE
      -- raise exception with error saying this policy is not applicable to
      -- the target
      raise_application_error(MGMT_GLOBAL.INVALID_PARAMS_ERR,
          'Policy policy_name = ' || p_policy_name ||
          ' is not applicable to target type ' || p_target_type);
    END IF;

    EM_POLICY.add_object_policy_assoc(
        p_object_guid => l_template_guid, 
        p_policy_guid => l_policy_guid,
        p_coll_name => ' ',
        p_object_type => MGMT_GLOBAL.G_OBJECT_TYPE_TEMPLATE,
        p_policy_type => MGMT_GLOBAL.G_TYPE_POLICY,
        p_is_enabled => p_is_enabled,
        p_policy_val_list => p_policy_val_list);

  END add_template_policy_assoc;

  PROCEDURE enable_template_policy_assoc(
      p_target_type        IN VARCHAR2,
      p_template_name      IN VARCHAR2,
      p_policy_name        IN VARCHAR2,
      p_coll_name          IN VARCHAR2 DEFAULT NULL)
  IS
    l_template_guid mgmt_templates.template_guid%TYPE;
    l_policy_guid mgmt_policies.policy_guid%TYPE;
    l_coll_name    mgmt_policy_assoc.coll_name%TYPE;
    l_policy_info  EM_POLICY.POLICY_INFO_REC;
    l_proc_name     VARCHAR2(32) := 'enable_template_policy_assoc';
  BEGIN
    EM_CHECK.check_not_null(p_target_type, 'p_target_type');
    EM_CHECK.check_not_null(p_template_name, 'p_template_name');
    EM_CHECK.check_not_null(p_policy_name, 'p_policy_name');

    l_template_guid := MGMT_TEMPLATE.get_template_guid(p_target_type, p_template_name);
    l_policy_guid := MGMT_POLICY.get_policy_guid(p_target_type, p_policy_name);

    -- Verify if the policy is applicable to the target
    EM_POLICY.get_policy_info(l_policy_guid, l_policy_info);

    IF (l_policy_info.policy_guid IS NOT NULL) THEN

      -- Use the metric name as coll name if no coll_name is specified
      l_coll_name := NVL(p_coll_name, l_policy_info.metric_name);

    ELSE
      -- raise exception with error saying this policy is not applicable to
      -- the target
      raise_application_error(MGMT_GLOBAL.INVALID_PARAMS_ERR,
          'Policy policy_name = ' || p_policy_name ||
          ' is not applicable to target type ' || p_target_type);
    END IF;

    EM_POLICY.update_policy_assoc(
        p_object_guid => l_template_guid,
        p_policy_guid => l_policy_guid,
        p_coll_name => ' ',
        p_is_enabled => MGMT_GLOBAL.G_TRUE);

  END enable_template_policy_assoc;
     
  PROCEDURE disable_template_policy_assoc(
      p_target_type        IN VARCHAR2,
      p_template_name      IN VARCHAR2,
      p_policy_name        IN VARCHAR2,
      p_coll_name          IN VARCHAR2 DEFAULT NULL)
  IS
    l_template_guid mgmt_templates.template_guid%TYPE;
    l_policy_guid mgmt_policies.policy_guid%TYPE;
    l_coll_name    mgmt_policy_assoc.coll_name%TYPE;
    l_policy_info  EM_POLICY.POLICY_INFO_REC;
    l_proc_name     VARCHAR2(32) := 'disable_template_policy_assoc';
  BEGIN
    EM_CHECK.check_not_null(p_target_type, 'p_target_type');
    EM_CHECK.check_not_null(p_template_name, 'p_template_name');
    EM_CHECK.check_not_null(p_policy_name, 'p_policy_name');

    l_template_guid := MGMT_TEMPLATE.get_template_guid(p_target_type, p_template_name);
    l_policy_guid := MGMT_POLICY.get_policy_guid(p_target_type, p_policy_name);

    -- Verify if the policy is applicable to the target
    EM_POLICY.get_policy_info(l_policy_guid, l_policy_info);

    IF (l_policy_info.policy_guid IS NOT NULL) THEN

      -- Use the metric name as coll name if no coll_name is specified
      l_coll_name := NVL(p_coll_name, l_policy_info.metric_name);

    ELSE
      -- raise exception with error saying this policy is not applicable to
      -- the target
      raise_application_error(MGMT_GLOBAL.INVALID_PARAMS_ERR,
          'Policy policy_name = ' || p_policy_name ||
          ' is not applicable to target type ' || p_target_type);
    END IF;

    EM_POLICY.update_policy_assoc(
        p_object_guid => l_template_guid,
        p_policy_guid => l_policy_guid,
        p_coll_name => ' ',
        p_is_enabled => MGMT_GLOBAL.G_FALSE);

  END disable_template_policy_assoc;

  PROCEDURE remove_template_policy_assoc(
      p_target_type        IN VARCHAR2,
      p_template_name      IN VARCHAR2,
      p_policy_name        IN VARCHAR2)
  IS
    l_template_guid mgmt_templates.template_guid%TYPE;
    l_policy_guid mgmt_policies.policy_guid%TYPE;
    l_proc_name     VARCHAR2(32) := 'remove_template_policy_assoc';
  BEGIN
    EM_CHECK.check_not_null(p_target_type, 'p_target_type');
    EM_CHECK.check_not_null(p_template_name, 'p_template_name');
    EM_CHECK.check_not_null(p_policy_name, 'p_policy_name');

    l_template_guid := MGMT_TEMPLATE.get_template_guid(p_target_type, p_template_name);
    l_policy_guid := MGMT_POLICY.get_policy_guid(p_target_type, p_policy_name);

    -- Remove template-policy assoc and any CAs associated with that.
    EM_POLICY.remove_object_policy_assoc(
        p_object_guid     => l_template_guid,
        p_policy_guid     => l_policy_guid,
        p_coll_name       => NULL,
        p_remove_ca_assoc => MGMT_GLOBAL.G_TRUE);

  END remove_template_policy_assoc;

  PROCEDURE set_target_policy_ca(
      p_target_type        IN VARCHAR2,
      p_target_name        IN VARCHAR2,
      p_policy_name        IN VARCHAR2,
      p_coll_name          IN VARCHAR2 DEFAULT ' ',
      p_key_value          IN VARCHAR2 DEFAULT ' ',
      p_key_operator       IN NUMBER   DEFAULT 0,
      p_violation_level    IN NUMBER   DEFAULT MGMT_GLOBAL.G_SEVERITY_CRITICAL,
      p_job_id             IN RAW)
  IS
    l_target_guid mgmt_targets.target_guid%TYPE;
    l_policy_guid mgmt_policies.policy_guid%TYPE;
    l_crit_ca     mgmt_policy_assoc_cfg.crit_action_job_id%TYPE;
    l_warn_ca     mgmt_policy_assoc_cfg.warn_action_job_id%TYPE;
    l_info_ca     mgmt_policy_assoc_cfg.info_action_job_id%TYPE;
    l_ca          mgmt_policy_assoc_cfg.info_action_job_id%TYPE;
  BEGIN
    EM_CHECK.check_not_null(p_target_type, 'p_target_type');
    EM_CHECK.check_not_null(p_target_name, 'p_target_name');
    EM_CHECK.check_not_null(p_policy_name, 'p_policy_name');
    EM_CHECK.check_not_null(p_violation_level, 'p_violation_level');
    EM_CHECK.check_not_null(p_job_id, 'p_job_id');

    -- TODO: validate severity level is one of CRIT/WARN/INFO

    l_target_guid := MGMT_TARGET.get_target_guid(p_target_name, p_target_type);
    l_policy_guid := MGMT_POLICY.get_policy_guid(p_target_type, p_policy_name);

    -- TODO : Verify that the association and configuration exists..
    l_ca := p_job_id;

    l_crit_ca := NULL; l_warn_ca := NULL; l_info_ca := NULL;

    IF (p_violation_level = MGMT_GLOBAL.G_SEVERITY_CRITICAL) THEN
      l_crit_ca := l_ca;
    ELSIF (p_violation_level = MGMT_GLOBAL.G_SEVERITY_WARNING) THEN
      l_warn_ca := l_ca;
    ELSIF (p_violation_level = MGMT_GLOBAL.G_SEVERITY_INFORMATIONAL) THEN
      l_info_ca := l_ca;
    ELSE
      -- Raise exception
      NULL;
    END IF;

    EM_POLICY.update_policy_assoc_cfg(
        p_object_guid => l_target_guid,
        p_policy_guid => l_policy_guid,
        p_coll_name => p_coll_name, 
        p_key_value => p_key_value,
        p_key_operator => p_key_operator,
        p_crit_action_job_id => l_crit_ca,
        p_warn_action_job_id => l_warn_ca,
        p_info_action_job_id => l_info_ca);

  END set_target_policy_ca;

  PROCEDURE set_target_metric_ca(
      p_target_type        IN VARCHAR2,
      p_target_name        IN VARCHAR2,
      p_metric_name        IN VARCHAR2,
      p_metric_column      IN VARCHAR2 DEFAULT NULL,
      p_coll_name          IN VARCHAR2 DEFAULT ' ',
      p_key_value          IN VARCHAR2 DEFAULT ' ',
      p_key_operator       IN NUMBER   DEFAULT 0,
      p_violation_level    IN NUMBER   DEFAULT MGMT_GLOBAL.G_SEVERITY_CRITICAL,
      p_job_id             IN RAW)
      
  IS
    l_target_guid mgmt_targets.target_guid%TYPE;
    l_metric_guid mgmt_policies.policy_guid%TYPE;
    l_crit_ca     mgmt_policy_assoc_cfg.crit_action_job_id%TYPE;
    l_warn_ca     mgmt_policy_assoc_cfg.warn_action_job_id%TYPE;
    l_info_ca     mgmt_policy_assoc_cfg.info_action_job_id%TYPE;
    l_ca          mgmt_policy_assoc_cfg.info_action_job_id%TYPE;
  BEGIN
    EM_CHECK.check_not_null(p_target_type, 'p_target_type');
    EM_CHECK.check_not_null(p_target_name, 'p_target_name');
    EM_CHECK.check_not_null(p_metric_name, 'p_metric_name');
    EM_CHECK.check_not_null(p_metric_column, 'p_metric_column');
    EM_CHECK.check_not_null(p_violation_level, 'p_violation_level');
    EM_CHECK.check_not_null(p_job_id, 'p_job_id');
    -- TODO: validate severity level is one of CRIT/WARN/INFO

    l_target_guid := MGMT_TARGET.get_target_guid(p_target_name, p_target_type);
    l_metric_guid := MGMT_METRIC.get_metric_guid(p_target_type, p_metric_name, p_metric_column);

    -- TODO : Verify that the association and configuration exists..

    l_ca := p_job_id;

    l_crit_ca := NULL; l_warn_ca := NULL; l_info_ca := NULL;

    IF (p_violation_level = MGMT_GLOBAL.G_SEVERITY_CRITICAL) THEN
      l_crit_ca := l_ca;
    ELSIF (p_violation_level = MGMT_GLOBAL.G_SEVERITY_WARNING) THEN
      l_warn_ca := l_ca;
    ELSIF (p_violation_level = MGMT_GLOBAL.G_SEVERITY_INFORMATIONAL) THEN
      l_info_ca := l_ca;
    ELSE
      -- Raise exception
      NULL;
    END IF;
    EM_POLICY.update_policy_assoc_cfg(
        p_object_guid => l_target_guid,
        p_policy_guid => l_metric_guid,
        p_coll_name => p_coll_name, 
        p_key_value => p_key_value,
        p_key_operator => p_key_operator,
        p_crit_action_job_id => l_crit_ca,
        p_warn_action_job_id => l_warn_ca,
        p_info_action_job_id => l_info_ca);
  END set_target_metric_ca;

  PROCEDURE set_target_metric_fixit_job(
      p_target_type        IN VARCHAR2,
      p_target_name        IN VARCHAR2,
      p_metric_name        IN VARCHAR2,
      p_metric_column      IN VARCHAR2 DEFAULT NULL,
      p_coll_name          IN VARCHAR2 DEFAULT ' ',
      p_key_value          IN VARCHAR2 DEFAULT ' ',
      p_key_operator       IN NUMBER   DEFAULT 0,
      p_agent_fixit        IN VARCHAR2 DEFAULT NULL)
  IS
    l_target_guid mgmt_targets.target_guid%TYPE;
    l_metric_guid mgmt_policies.policy_guid%TYPE;
  BEGIN
    EM_CHECK.check_not_null(p_target_type, 'p_target_type');
    EM_CHECK.check_not_null(p_target_name, 'p_target_name');
    EM_CHECK.check_not_null(p_metric_name, 'p_metric_name');

    l_target_guid := MGMT_TARGET.get_target_guid(p_target_name, p_target_type);
    l_metric_guid := MGMT_METRIC.get_metric_guid(p_target_type, p_metric_name, p_metric_column);

    EM_POLICY.update_policy_assoc_cfg(
        p_object_guid => l_target_guid,
        p_policy_guid => l_metric_guid,
        p_coll_name => p_coll_name, 
        p_key_value => p_key_value,
        p_key_operator => p_key_operator,
        p_fixit_job => p_agent_fixit);

  END set_target_metric_fixit_job;

  PROCEDURE set_template_policy_ca(
      p_target_type        IN VARCHAR2,
      p_template_name      IN VARCHAR2,
      p_policy_name        IN VARCHAR2,
      p_coll_name          IN VARCHAR2 DEFAULT ' ',
      p_key_value          IN VARCHAR2 DEFAULT ' ',
      p_key_operator       IN NUMBER   DEFAULT 0,
      p_violation_level    IN NUMBER   DEFAULT MGMT_GLOBAL.G_SEVERITY_CRITICAL,
      p_job_id             IN RAW)
  IS
    l_template_guid mgmt_templates.template_guid%TYPE;
    l_policy_guid mgmt_policies.policy_guid%TYPE;
    l_crit_ca     mgmt_policy_assoc_cfg.crit_action_job_id%TYPE;
    l_warn_ca     mgmt_policy_assoc_cfg.warn_action_job_id%TYPE;
    l_info_ca     mgmt_policy_assoc_cfg.info_action_job_id%TYPE;
    l_ca          mgmt_policy_assoc_cfg.info_action_job_id%TYPE;
  BEGIN
    EM_CHECK.check_not_null(p_target_type, 'p_target_type');
    EM_CHECK.check_not_null(p_template_name, 'p_template_name');
    EM_CHECK.check_not_null(p_policy_name, 'p_policy_name');
    EM_CHECK.check_not_null(p_violation_level, 'p_violation_level');
    EM_CHECK.check_not_null(p_job_id, 'p_job_id');

    -- TODO: validate severity level is one of CRIT/WARN/INFO

    l_template_guid := MGMT_TEMPLATE.get_template_guid(p_target_type, p_template_name);
    l_policy_guid := MGMT_POLICY.get_policy_guid(p_target_type, p_policy_name);

    -- TODO : Verify that the association and configuration exists..

    l_ca := p_job_id;

    l_crit_ca := NULL; l_warn_ca := NULL; l_info_ca := NULL;

    IF (p_violation_level = MGMT_GLOBAL.G_SEVERITY_CRITICAL) THEN
      l_crit_ca := l_ca;
    ELSIF (p_violation_level = MGMT_GLOBAL.G_SEVERITY_WARNING) THEN
      l_warn_ca := l_ca;
    ELSIF (p_violation_level = MGMT_GLOBAL.G_SEVERITY_WARNING) THEN
      l_warn_ca := l_ca;
    ELSE
      -- Raise exception
      NULL;
    END IF;

    EM_POLICY.update_policy_assoc_cfg(
        p_object_guid => l_template_guid,
        p_policy_guid => l_policy_guid,
        p_coll_name => p_coll_name, 
        p_key_value => p_key_value,
        p_key_operator => p_key_operator,
        p_crit_action_job_id => l_crit_ca,
        p_warn_action_job_id => l_warn_ca,
        p_info_action_job_id => l_info_ca);

  END set_template_policy_ca;

  PROCEDURE set_template_metric_ca(
      p_target_type        IN VARCHAR2,
      p_template_name      IN VARCHAR2,
      p_metric_name        IN VARCHAR2,
      p_metric_column      IN VARCHAR2 DEFAULT NULL,
      p_coll_name          IN VARCHAR2 DEFAULT ' ',
      p_key_value          IN VARCHAR2 DEFAULT ' ',
      p_key_operator       IN NUMBER   DEFAULT 0,
      p_violation_level    IN NUMBER   DEFAULT MGMT_GLOBAL.G_SEVERITY_CRITICAL,
      p_job_id             IN RAW)  
  IS
    l_template_guid mgmt_templates.template_guid%TYPE;
    l_metric_guid mgmt_policies.policy_guid%TYPE;
    l_crit_ca     mgmt_policy_assoc_cfg.crit_action_job_id%TYPE;
    l_warn_ca     mgmt_policy_assoc_cfg.warn_action_job_id%TYPE;
    l_info_ca     mgmt_policy_assoc_cfg.info_action_job_id%TYPE;
    l_ca          mgmt_policy_assoc_cfg.info_action_job_id%TYPE;
  BEGIN
    EM_CHECK.check_not_null(p_target_type, 'p_target_type');
    EM_CHECK.check_not_null(p_template_name, 'p_template_name');
    EM_CHECK.check_not_null(p_metric_name, 'p_metric_name');
    EM_CHECK.check_not_null(p_metric_column, 'p_metric_column');
    EM_CHECK.check_not_null(p_violation_level, 'p_violation_level');
    EM_CHECK.check_not_null(p_job_id, 'p_job_id');

    -- TODO: validate severity level is one of CRIT/WARN/INFO

    l_template_guid := MGMT_TEMPLATE.get_template_guid(p_target_type, p_template_name);
    l_metric_guid := MGMT_METRIC.get_metric_guid(p_target_type, p_metric_name, p_metric_column);

    -- TODO : Verify that the association and configuration exists..

    l_ca := p_job_id;

    l_crit_ca := NULL; l_warn_ca := NULL; l_info_ca := NULL;

    IF (p_violation_level = MGMT_GLOBAL.G_SEVERITY_CRITICAL) THEN
      l_crit_ca := l_ca;
    ELSIF (p_violation_level = MGMT_GLOBAL.G_SEVERITY_WARNING) THEN
      l_warn_ca := l_ca;
    ELSIF (p_violation_level = MGMT_GLOBAL.G_SEVERITY_WARNING) THEN
      l_warn_ca := l_ca;
    ELSE
      -- Raise exception
      NULL;
    END IF;

    EM_POLICY.update_policy_assoc_cfg(
        p_object_guid => l_template_guid,
        p_policy_guid => l_metric_guid,
        p_coll_name => p_coll_name, 
        p_key_value => p_key_value,
        p_key_operator => p_key_operator,
        p_crit_action_job_id => l_crit_ca,
        p_warn_action_job_id => l_warn_ca,
        p_info_action_job_id => l_info_ca);

  END set_template_metric_ca;

  PROCEDURE set_template_metric_fixit_job(
      p_target_type        IN VARCHAR2,
      p_template_name      IN VARCHAR2,
      p_metric_name        IN VARCHAR2,
      p_metric_column      IN VARCHAR2 DEFAULT NULL,
      p_coll_name          IN VARCHAR2 DEFAULT ' ',
      p_key_value          IN VARCHAR2 DEFAULT ' ',
      p_key_operator       IN NUMBER   DEFAULT 0,
      p_agent_fixit        IN VARCHAR2 DEFAULT NULL)
  IS
    l_template_guid mgmt_templates.template_guid%TYPE;
    l_metric_guid mgmt_policies.policy_guid%TYPE;
  BEGIN
    EM_CHECK.check_not_null(p_target_type, 'p_target_type');
    EM_CHECK.check_not_null(p_template_name, 'p_template_name');
    EM_CHECK.check_not_null(p_metric_name, 'p_metric_name');
    EM_CHECK.check_not_null(p_metric_column, 'p_metric_column');

    l_template_guid := MGMT_TEMPLATE.get_template_guid(p_target_type, p_template_name);
    l_metric_guid := MGMT_METRIC.get_metric_guid(p_target_type, p_metric_name, p_metric_column);

    EM_POLICY.update_policy_assoc_cfg(
        p_object_guid => l_template_guid,
        p_policy_guid => l_metric_guid,
        p_coll_name => p_coll_name, 
        p_key_value => p_key_value,
        p_key_operator => p_key_operator,
        p_fixit_job => p_agent_fixit);

  END set_template_metric_fixit_job;



END mgmt_policy;
/


show errors



