Rem
Rem $Header: sdk_severity_helper_pkgbody.sql 13-jun-2005.15:59:06 gan Exp $
Rem
Rem sdk_severity_helper_pkgbody.sql
Rem
Rem Copyright (c) 2003, 2005, Oracle. All rights reserved.  
Rem
Rem    NAME
Rem      sdk_severity_helper_pkgbody.sql - <one-line expansion of the name>
Rem
Rem    DESCRIPTION
Rem      <short description of component this file declares/defines>
Rem
Rem    NOTES
Rem      <other useful comments, qualifications, etc.>
Rem
Rem    MODIFIED   (MM/DD/YY)
Rem    gan         05/16/05 - nls availability message 
Rem    pmodi       01/21/05 - NLS string for repository side metrics 
Rem    streddy     09/08/04 - Fix error messages 
Rem    streddy     12/01/03 - Return status as UNKNOWN when nothing to 
Rem    streddy     04/09/03 - streddy_composite_target_availability
Rem    streddy     03/06/03 - Created
Rem

CREATE OR REPLACE PACKAGE BODY mgmt_severity_helper AS
 
-- NLS IDs for availability
SVC_CLEAR constant VARCHAR2(64) := mgmt_global.G_EM_SYS_PREFIX||'SVC_CLEAR';
SVC_CRITICAL constant VARCHAR2(64) := mgmt_global.G_EM_SYS_PREFIX||'SVC_CRITICAL';
SVC_UNREACHABLE constant VARCHAR2(64) := mgmt_global.G_EM_SYS_PREFIX||'SVC_UNREACHABLE';
SVC_METRIC_ERROR constant VARCHAR2(64) := mgmt_global.G_EM_SYS_PREFIX||'SVC_METRIC_ERROR';
SVC_UNKNOWN constant VARCHAR2(64) := mgmt_global.G_EM_SYS_PREFIX||'SVC_UNKNOWN';

ASVC_CLEAR constant VARCHAR2(64) := mgmt_global.G_EM_SYS_PREFIX||'ASVC_CLEAR';
ASVC_CRITICAL constant VARCHAR2(64) := mgmt_global.G_EM_SYS_PREFIX||'ASVC_CRITICAL';
ASVC_UNREACHABLE constant VARCHAR2(64) := mgmt_global.G_EM_SYS_PREFIX||'ASVC_UNREACHABLE';
ASVC_METRIC_ERROR constant VARCHAR2(64) := mgmt_global.G_EM_SYS_PREFIX||'ASVC_METRIC_ERROR';
ASVC_UNKNOWN constant VARCHAR2(64) := mgmt_global.G_EM_SYS_PREFIX||'ASVC_UNKNOWN';

OTHER_CLEAR constant VARCHAR2(64) := mgmt_global.G_EM_SYS_PREFIX||'OTHER_CLEAR';
OTHER_CRITICAL constant VARCHAR2(64) := mgmt_global.G_EM_SYS_PREFIX||'OTHER_CRITICAL';
OTHER_UNREACHABLE constant VARCHAR2(64) := mgmt_global.G_EM_SYS_PREFIX||'OTHER_UNREACHABLE';
OTHER_METRIC_ERROR constant VARCHAR2(64) := mgmt_global.G_EM_SYS_PREFIX||'OTHER_METRIC_ERROR';
OTHER_UNKNOWN constant VARCHAR2(64) := mgmt_global.G_EM_SYS_PREFIX||'OTHER_UNKNOWN';



-- 
-- PURPOSE
--   Internal function to return a message from a severity
-- 
FUNCTION GET_MSG_FROM_SEVERITY(v_count   IN NUMBER,
                               v_status  IN VARCHAR2)
RETURN VARCHAR2 IS
BEGIN
   IF (v_count = 0) THEN
     return '';
   ELSIF (v_count = 1) THEN
     return '( 1 member target, ' || v_status || ') ';
   ELSE 
     return '(' || v_count || ' member targets, ' || v_status || ') ';
   END IF;
END GET_MSG_FROM_SEVERITY;

  

--
-- PURPOSE 
--   Internal method to construct severity message
--

PROCEDURE CONSTRUCT_SEVERITY_MESSAGE(v_clear_count     IN NUMBER,
                                     v_warning_count   IN NUMBER,
                                     v_critical_count  IN NUMBER,
                                     v_error_count     IN NUMBER,
                                     v_message        OUT VARCHAR2)
IS
BEGIN

  v_message := get_msg_from_severity(v_clear_count, 'Clear') ||
               get_msg_from_severity(v_warning_count, 'Warning') ||
               get_msg_from_severity(v_critical_count, 'Critical') ||
               get_msg_from_severity(v_error_count, 'Metric Error');

END CONSTRUCT_SEVERITY_MESSAGE;



--
-- PURPOSE 
--   Internal method to construct status message
--

PROCEDURE CONSTRUCT_STATUS_MESSAGE(v_up_count          IN NUMBER,
                                   v_down_count        IN NUMBER,
                                   v_agent_down_count  IN NUMBER,
                                   v_unreachable_count IN NUMBER,
                                   v_blackout_count    IN NUMBER,
                                   v_unknown_count     IN NUMBER,
                                   v_error_count       IN NUMBER,
                                   v_message          OUT VARCHAR2)
IS
BEGIN

  v_message := 'Member target details --> ' ||
                get_msg_from_severity(v_up_count, 'Up') ||
                get_msg_from_severity(v_down_count, 'Down') ||
                get_msg_from_severity(v_unreachable_count, ' Unreachable') ||
                get_msg_from_severity(v_blackout_count, 'Blackout') ||
                get_msg_from_severity(v_agent_down_count, 'Agent Down') ||
                get_msg_from_severity(v_error_count, 'Metric Error') ||
                get_msg_from_severity(v_unknown_count, 'Status Pending');


END CONSTRUCT_STATUS_MESSAGE;

PROCEDURE construct_asvc_status_message(v_down_dependents IN MGMT_TARGET_ARRAY,
                                       v_sev_code    IN NUMBER,
                                       v_message     OUT VARCHAR2,
                                       v_message_nls_id OUT VARCHAR2,
                                       v_message_params OUT VARCHAR2)
AS
BEGIN
  IF   v_sev_code = mgmt_global.G_SEVERITY_CLEAR THEN
      v_message_nls_id := ASVC_CLEAR;
      v_message := 'Aggregate service is up';
  ELSIF   v_sev_code = mgmt_global.G_SEVERITY_UNREACHABLE_START THEN
      v_message_nls_id := ASVC_UNREACHABLE;
      v_message := 'Aggregate service is in agent unreachable state';
  ELSIF   v_sev_code = mgmt_global.G_SEVERITY_UNKNOWN THEN
      v_message_nls_id := ASVC_UNKNOWN;
      v_message := 'Aggregate service status pending';
  ELSIF   v_sev_code = mgmt_global.G_SEVERITY_ERROR_START THEN
      v_message_nls_id := ASVC_METRIC_ERROR;
      v_message := 'Aggregate service encountered metric erros; at least one sub-service in in metric error';
  ELSE
      v_message_nls_id := ASVC_CRITICAL ;
      v_message := 'Aggregate service is down; {0} sub-services are down: {1}';
      FOR indx IN 1..v_down_dependents.COUNT
      LOOP
         v_message_params:= v_message_params || v_down_dependents(indx).target_name||',';
      END LOOP;
      v_message_params := v_down_dependents.COUNT||'&'||v_message_params;
  END IF;
END construct_asvc_status_message;

PROCEDURE construct_svc_status_message(v_down_dependents IN MGMT_TARGET_ARRAY,
                                       v_sev_code    IN NUMBER,
                                       v_message     OUT VARCHAR2,
                                       v_message_nls_id OUT VARCHAR2,
                                       v_message_params OUT VARCHAR2)
AS
BEGIN
  IF   v_sev_code = mgmt_global.G_SEVERITY_CLEAR THEN
      v_message_nls_id := SVC_CLEAR;
      v_message := 'Service is up';
  ELSIF   v_sev_code = mgmt_global.G_SEVERITY_UNREACHABLE_START THEN
      v_message_nls_id := SVC_UNREACHABLE;
      v_message := 'Service is in agent unreachable state';
  ELSIF   v_sev_code = mgmt_global.G_SEVERITY_UNKNOWN THEN
      v_message_nls_id := SVC_UNKNOWN;
      v_message := 'Service status pending';
  ELSIF   v_sev_code = mgmt_global.G_SEVERITY_ERROR_START THEN
      v_message_nls_id := SVC_METRIC_ERROR;
      v_message := 'Service encountered metric erros; at least one key component in in metric error';
  ELSE
      v_message_nls_id := SVC_CRITICAL ;
      v_message := 'Service is down; {0} key components are down: {1}';
      FOR indx IN 1..v_down_dependents.COUNT
      LOOP
         v_message_params:= v_message_params || v_down_dependents(indx).target_name||',';
      END LOOP;
      v_message_params := v_down_dependents.COUNT||'&'||v_message_params;
  END IF;
END construct_svc_status_message;

PROCEDURE construct_other_status_message(v_down_dependents IN MGMT_TARGET_ARRAY,
                                       v_sev_code    IN NUMBER,
                                       v_message     OUT VARCHAR2,
                                       v_message_nls_id OUT VARCHAR2,
                                       v_message_params OUT VARCHAR2)
AS
BEGIN
  IF   v_sev_code = mgmt_global.G_SEVERITY_CLEAR THEN
      v_message_nls_id := OTHER_CLEAR;
      v_message := 'Target is up';
  ELSIF   v_sev_code = mgmt_global.G_SEVERITY_UNREACHABLE_START THEN
      v_message_nls_id := OTHER_UNREACHABLE;
      v_message := 'Target is in agent unreachable state';
  ELSIF   v_sev_code = mgmt_global.G_SEVERITY_UNKNOWN THEN
      v_message_nls_id := OTHER_UNKNOWN;
      v_message := 'Target status pending';
  ELSIF   v_sev_code = mgmt_global.G_SEVERITY_ERROR_START THEN
      v_message_nls_id := OTHER_METRIC_ERROR;
      v_message := 'Target encountered metric erros; at least one member in in metric error';
  ELSE
      v_message_nls_id := OTHER_CRITICAL ;
      v_message := 'Target is down; {0} members are down: {1}';
      FOR indx IN 1..v_down_dependents.COUNT
      LOOP
         v_message_params:= v_message_params || v_down_dependents(indx).target_name||',';
      END LOOP;
      v_message_params := v_down_dependents.COUNT||'&'||v_message_params;
  END IF;
END construct_other_status_message;


PROCEDURE CONSTRUCT_NLS_STATUS_MESSAGE(v_target_guid IN RAW, 
                                       v_down_dependents IN MGMT_TARGET_ARRAY,
                                       v_sev_code    IN NUMBER,
                                       v_message     OUT VARCHAR2,
                                       v_message_nls_id OUT VARCHAR2,
                                       v_message_params OUT VARCHAR2)
IS
l_target_type mgmt_targets.target_type%TYPE;
l_target_name mgmt_targets.target_name%TYPE;
l_is_aggregate_service NUMBER(1);
l_is_service           NUMBER(1);
BEGIN
  SELECT target_name , target_type
    INTO l_target_name, l_target_type
    FROM mgmt_targets
   WHERE target_guid = v_target_guid;

  l_is_aggregate_service := mgmt_service.is_aggregate_service(l_target_type);
  l_is_service := mgmt_service.implements_service_interface(l_target_type);

  IF l_is_aggregate_service = 1 THEN
     construct_asvc_status_message(v_down_dependents, v_sev_code, v_message,
                                      v_message_nls_id, v_message_params);
  ELSIF l_is_service  = 1 THEN
     construct_svc_status_message(v_down_dependents, v_sev_code, v_message,
                                      v_message_nls_id, v_message_params);
  ELSE
     construct_other_status_message(v_down_dependents, v_sev_code, v_message,
                                      v_message_nls_id, v_message_params);
  END IF;
END CONSTRUCT_NLS_STATUS_MESSAGE; 
                                       
--
-- PURPOSE 
--   Method to analyze the member availabilities
--
PROCEDURE ANALYZE_STATUS_DETAILS(v_dependency_list IN MGMT_METRIC_DETAILS_ARRAY,
                                 v_up_count          OUT NUMBER,
                                 v_down_count        OUT NUMBER,
                                 v_agent_down_count  OUT NUMBER,
                                 v_unreachable_count OUT NUMBER,
                                 v_blackout_count    OUT NUMBER,
                                 v_unknown_count     OUT NUMBER,
                                 v_error_count       OUT NUMBER)
IS
  l_avail_status  NUMBER;
BEGIN
  v_up_count := 0;
  v_down_count := 0;
  v_agent_down_count := 0;
  v_unreachable_count:= 0;
  v_blackout_count   := 0;
  v_unknown_count    := 0;
  v_error_count      := 0;

   -- Get the count of individual statuses
   FOR i IN 1..v_dependency_list.COUNT
   LOOP
  
     l_avail_status := v_dependency_list(i).avail_status;

     IF (l_avail_status = mgmt_global.G_STATUS_UP) THEN
       v_up_count := v_up_count + 1;
     ELSIF (l_avail_status = mgmt_global.G_STATUS_DOWN) THEN
       v_down_count := v_down_count + 1;
     ELSIF (l_avail_status = mgmt_global.G_STATUS_ERROR) THEN
       v_error_count := v_error_count + 1;
     ELSIF (l_avail_status = mgmt_global.G_STATUS_AGENT_DOWN) THEN
       v_agent_down_count := v_agent_down_count + 1;
     ELSIF (l_avail_status = mgmt_global.G_STATUS_UNREACHABLE) THEN
       v_unreachable_count := v_unreachable_count + 1;
     ELSIF (l_avail_status = mgmt_global.G_STATUS_BLACKOUT) THEN
       v_blackout_count := v_blackout_count + 1;
     ELSIF (l_avail_status = mgmt_global.G_STATUS_UNKNOWN) THEN
       v_unknown_count := v_unknown_count + 1;
     END IF;
   END LOOP;


END ANALYZE_STATUS_DETAILS;

--
-- PURPOSE 
--   Method to analyze the member availabilities
--
PROCEDURE ANALYZE_STATUS_DETAILS(v_dependency_list IN MGMT_METRIC_DETAILS_ARRAY,
                                 v_up_count          OUT NUMBER,
                                 v_down_count        OUT NUMBER,
                                 v_agent_down_count  OUT NUMBER,
                                 v_unreachable_count OUT NUMBER,
                                 v_blackout_count    OUT NUMBER,
                                 v_unknown_count     OUT NUMBER,
                                 v_error_count       OUT NUMBER,
                                 v_down_dependents   OUT MGMT_TARGET_ARRAY)
IS
  l_avail_status  NUMBER;
BEGIN
  v_up_count := 0;
  v_down_count := 0;
  v_agent_down_count := 0;
  v_unreachable_count:= 0;
  v_blackout_count   := 0;
  v_unknown_count    := 0;
  v_error_count      := 0;
  v_down_dependents  := MGMT_TARGET_ARRAY();

   -- Get the count of individual statuses
   FOR i IN 1..v_dependency_list.COUNT
   LOOP
  
     l_avail_status := v_dependency_list(i).avail_status;

     IF (l_avail_status = mgmt_global.G_STATUS_UP) THEN
       v_up_count := v_up_count + 1;
     ELSIF (l_avail_status = mgmt_global.G_STATUS_DOWN) THEN
       v_down_count := v_down_count + 1;
       v_down_dependents.EXTEND;
       SELECT mgmt_target_obj.NEW(target_name, target_type)
         INTO v_down_dependents(v_down_count)
         FROM mgmt_targets
        WHERE target_guid = v_dependency_list(i).target_guid;
     ELSIF (l_avail_status = mgmt_global.G_STATUS_ERROR) THEN
       v_error_count := v_error_count + 1;
     ELSIF (l_avail_status = mgmt_global.G_STATUS_AGENT_DOWN) THEN
       v_agent_down_count := v_agent_down_count + 1;
     ELSIF (l_avail_status = mgmt_global.G_STATUS_UNREACHABLE) THEN
       v_unreachable_count := v_unreachable_count + 1;
     ELSIF (l_avail_status = mgmt_global.G_STATUS_BLACKOUT) THEN
       v_blackout_count := v_blackout_count + 1;
     ELSIF (l_avail_status = mgmt_global.G_STATUS_UNKNOWN) THEN
       v_unknown_count := v_unknown_count + 1;
     END IF;
   END LOOP;


END ANALYZE_STATUS_DETAILS;


--
-- PURPOSE 
--   Method to analyze the member severities

PROCEDURE ANALYZE_SEVERITY_DETAILS(v_dependency_list IN MGMT_METRIC_DETAILS_ARRAY,
                                   v_clear_count          OUT NUMBER,
                                   v_warning_count        OUT NUMBER,
                                   v_critical_count       OUT NUMBER,
                                   v_error_count          OUT NUMBER)
IS
  l_sev_code  NUMBER;
BEGIN
   -- Initialize all return counts to zero
   v_clear_count := 0;
   v_warning_count := 0;
   v_critical_count := 0;
   v_error_count      := 0;

   -- Get the count of individual statuses
   FOR i IN 1..v_dependency_list.COUNT
   LOOP
     -- only consider the UP targets for severity calculations
     IF (v_dependency_list(i).avail_status <> mgmt_global.G_STATUS_UP) THEN
       GOTO next_dependency;
     END IF;

     l_sev_code := v_dependency_list(i).severity_code;

     IF (l_sev_code = mgmt_global.G_SEVERITY_CLEAR) THEN
       v_clear_count := v_clear_count + 1;
     ELSIF (l_sev_code = mgmt_global.G_SEVERITY_CRITICAL) THEN
      v_critical_count := v_critical_count + 1;
     ELSIF (l_sev_code = mgmt_global.G_SEVERITY_WARNING) THEN
       v_warning_count := v_warning_count + 1;
     ELSIF (l_sev_code = mgmt_global.G_SEVERITY_ERROR_START) THEN
       v_error_count := v_error_count + 1;
     END IF;
     <<next_dependency>>
     NULL;
   END LOOP;

END ANALYZE_SEVERITY_DETAILS;


--
-- EXTERNALLY CALLABLE PROCEDURES
--


-- PURPOSE
--    Calculate the new Response/Status severity state of a target based
--    on the dependency metric severities using OR predicate logic. This
--    proc assumes that at least one of the member severities have to in CLEAR
--    state for the overall target to be in CLEAR state. The possible severity
--    codes it returns are CLEAR, CRITICAL, UNREACHABLE, ERROR, UNKNOWN, and
--    BLACKOUT.
-- PARAMETERS
--     v_target_guid   Target guid
--     v_metric_guid   Metric guid
--     v_dependency_list List of metric dependencies and their severity
--                       values
--     v_sev_code        New severity code
--     v_message         Message for the severity
--     v_message_nlsid   Optional NLS ID for the message
--     v_message_params  Message parameters for NLS message
--
--    Message param would be in following order, separated by "&"
--     up_count, down_count, agent_down_count, unreachable_count, blackout_count, unknown_count and error_count 
--
PROCEDURE OR_AVAIL_EVAL_PROC(v_target_guid       IN RAW, 
                             v_metric_guid       IN RAW,
                             v_dependency_list   IN MGMT_METRIC_DETAILS_ARRAY,
                             v_sev_code          OUT NUMBER,
                             v_message           OUT VARCHAR2,
                             v_message_nlsid     OUT VARCHAR2,
                             v_message_params    OUT VARCHAR2)
IS
  l_up_count          NUMBER := 0;
  l_down_count        NUMBER := 0;
  l_agent_down_count  NUMBER := 0;
  l_unreachable_count NUMBER := 0;
  l_blackout_count    NUMBER := 0;
  l_unknown_count     NUMBER := 0;
  l_error_count       NUMBER := 0;
  l_down_dependents   MGMT_TARGET_ARRAY;
BEGIN
  -- Return unknown if we are called with empty list
  IF (v_dependency_list.COUNT = 0) THEN
    v_sev_code := mgmt_global.G_SEVERITY_UNKNOWN;
    CONSTRUCT_NLS_STATUS_MESSAGE(v_target_guid , 
                              NULL ,
                              v_sev_code   ,
                              v_message ,
                              v_message_nlsid ,
                              v_message_params);

    return;
  END IF;

  analyze_status_details(v_dependency_list, l_up_count, l_down_count,
                         l_agent_down_count, l_unreachable_count,
                         l_blackout_count, l_unknown_count, l_error_count,
                         l_down_dependents);


              

  -- Evaluate status in this order
  --   o If at least one of the members is UP, then we are UP
  --   o If any of the members is UNREACHABLE, then we are too
  --   o If any of the members is in BLACKOUT, AGENT_DOWN, or
  --     UNKNOWN, then we are in UNKNOWN
  --   o If any of the members is in ERROR, then we are in ERROR
  --   o If none of the above is TRUE, then we are in DOWN state
  IF (l_up_count > 0) THEN
     v_sev_code := mgmt_global.G_SEVERITY_CLEAR;
  ELSIF (l_unreachable_count > 0) THEN
     v_sev_code := mgmt_global.G_SEVERITY_UNREACHABLE_START;
  ELSIF (l_blackout_count > 0 OR l_agent_down_count > 0 OR
         l_unknown_count > 0) THEN
     v_sev_code := mgmt_global.G_SEVERITY_UNKNOWN;
  ELSIF (l_error_count > 0) THEN
     v_sev_code := mgmt_global.G_SEVERITY_ERROR_START;
  ELSE
     v_sev_code := mgmt_global.G_SEVERITY_CRITICAL;
  END IF;
 
 CONSTRUCT_NLS_STATUS_MESSAGE(v_target_guid , 
                              l_down_dependents ,
                              v_sev_code   ,
                              v_message ,
                              v_message_nlsid ,
                              v_message_params);

END OR_AVAIL_EVAL_PROC;


--
-- PURPOSE
--    Calculate the new Response/Status severity state of a target based
--    on the dependency metric severities using AND predicate logic. This
--    proc assumes that all the member severities have to in CLEAR
--    state for the overall target to be in CLEAR state. The possible severity
--    codes it returns are CLEAR, CRITICAL, UNREACHABLE, ERROR, UNKNOWN, and
--    BLACKOUT.
-- PARAMETERS
--    See OR_AVAIL_EVAL_PROC description.
--
--    Message param would be in following order, separated by "&"
--     up_count, down_count, agent_down_count, unreachable_count, blackout_count, unknown_count and error_count 
-- 
PROCEDURE AND_AVAIL_EVAL_PROC(v_target_guid      IN RAW, 
                             v_metric_guid       IN RAW,
                             v_dependency_list   IN MGMT_METRIC_DETAILS_ARRAY,
                             v_sev_code          OUT NUMBER,
                             v_message           OUT VARCHAR2,
                             v_message_nlsid     OUT VARCHAR2,
                             v_message_params    OUT VARCHAR2)
IS
  l_up_count          NUMBER := 0;
  l_down_count        NUMBER := 0;
  l_agent_down_count  NUMBER := 0;
  l_unreachable_count NUMBER := 0;
  l_blackout_count    NUMBER := 0;
  l_unknown_count     NUMBER := 0;
  l_error_count       NUMBER := 0;
  l_down_dependents   MGMT_TARGET_ARRAY;
BEGIN
  -- Return unknown if we are called with empty list
  IF (v_dependency_list.COUNT = 0) THEN
    v_sev_code := mgmt_global.G_SEVERITY_UNKNOWN;
    CONSTRUCT_NLS_STATUS_MESSAGE(v_target_guid , 
                              NULL,
                              v_sev_code   ,
                              v_message ,
                              v_message_nlsid ,
                              v_message_params);

    return;
  END IF;

  analyze_status_details(v_dependency_list, l_up_count, l_down_count,
                         l_agent_down_count, l_unreachable_count,
                         l_blackout_count, l_unknown_count, l_error_count,
                         l_down_dependents);


  -- Evaluate status in this order
  --   o If any of the members is DOWN, then we are DOWN
  --   o If any of the members is UNREACHABLE, then we are too
  --   o If any of the members is in BLACKOUT, AGENT_DOWN, or
  --     UNKNOWN, then we are in UNKNOWN
  --   o If any of the members is in ERROR, then we are in ERROR
  --   o If none of the above is TRUE, then we are in UP state
  IF (l_down_count > 0) THEN
     v_sev_code := mgmt_global.G_SEVERITY_CRITICAL;
  ELSIF (l_unreachable_count > 0) THEN
     v_sev_code := mgmt_global.G_SEVERITY_UNREACHABLE_START;
  ELSIF (l_blackout_count > 0 OR l_agent_down_count > 0 OR
         l_unknown_count > 0) THEN
     v_sev_code := mgmt_global.G_SEVERITY_UNKNOWN;
  ELSIF (l_error_count > 0) THEN
     v_sev_code := mgmt_global.G_SEVERITY_ERROR_START;
  ELSE
     v_sev_code := mgmt_global.G_SEVERITY_CLEAR;
  END IF;
 CONSTRUCT_NLS_STATUS_MESSAGE(v_target_guid , 
                              l_down_dependents ,
                              v_sev_code   ,
                              v_message ,
                              v_message_nlsid ,
                              v_message_params);

END AND_AVAIL_EVAL_PROC;



--
-- PURPOSE
--   Calculate the new severity state of a metric based on the dependency
--   metric severities using OR predicate logic. This proc assumes that
--   at least one of the member severities have to in CLEAR state for the
--   overall metric to be in CLEAR state. The possible severity codes it
--   returns are CLEAR, WARNING, CRITICAL, UNREACHABLE, ERROR, and UNKNOWN.
--
--- PARAMETERS
--    See OR_AVAIL_EVAL_PROC description.
--
--    Message param would be in following order, separated by "&"
--     clear_count, warning_count, critical_count and error_count 
-- 
PROCEDURE OR_SEVERITY_EVAL_PROC(v_target_guid    IN RAW, 
                             v_metric_guid       IN RAW,
                             v_dependency_list   IN MGMT_METRIC_DETAILS_ARRAY,
                             v_sev_code          OUT NUMBER,
                             v_message           OUT VARCHAR2,
                             v_message_nlsid     OUT VARCHAR2,
                             v_message_params    OUT VARCHAR2)
IS
  l_clear_count      NUMBER := 0;
  l_critical_count   NUMBER := 0;
  l_warning_count    NUMBER := 0;
  l_error_count      NUMBER := 0;
BEGIN
  analyze_severity_details(v_dependency_list, l_clear_count,
                           l_warning_count, l_critical_count,
                           l_error_count); 

  construct_severity_message(v_clear_count     => l_clear_count,  
                             v_warning_count   => l_warning_count,  
                             v_critical_count  => l_critical_count,
                             v_error_count     => l_error_count,
                             v_message         => v_message );

  v_message_params := l_clear_count||'&'||l_warning_count||'&'||l_critical_count||'&'||l_error_count ;

  -- Evaluate status in this order
  --   o If at least one of the members is CLEAR, then we are in CLEAR
  --   o If at least one of the members is WARNING, then we are in WARNING
  --   o If at least one of the members is ERROR, then we are in ERROR
  --   o Otherwise, we are CRITICAL
  IF (l_clear_count  > 0) THEN
     v_sev_code := mgmt_global.G_SEVERITY_CLEAR;
  ELSIF (l_warning_count > 0) THEN
     v_sev_code := mgmt_global.G_SEVERITY_WARNING;
  ELSIF (l_error_count > 0) THEN
     v_sev_code := mgmt_global.G_SEVERITY_ERROR_START;
  ELSE
     v_sev_code := mgmt_global.G_SEVERITY_CRITICAL;
  END IF;

END OR_SEVERITY_EVAL_PROC;


--
-- PURPOSE
--   Calculate the new severity state of a metric based on the dependency
--   metric severities using AND predicate logic. This proc assumes that
--   at least one of the member severities have to in CLEAR state for the
--   overall metric to be in CLEAR state. The possible severity codes it
--   returns are CLEAR, WARNING, CRITICAL, AND ERROR.
--
--- PARAMETERS
--    See OR_AVAIL_EVAL_PROC description.
--
--    Message param would be in following order, separated by "&"
--     clear_count, warning_count, critical_count and error_count 
-- 
PROCEDURE AND_SEVERITY_EVAL_PROC(v_target_guid    IN RAW, 
                                 v_metric_guid       IN RAW,
                                 v_dependency_list   IN MGMT_METRIC_DETAILS_ARRAY,
                                 v_sev_code          OUT NUMBER,
                                 v_message           OUT VARCHAR2,
                                 v_message_nlsid     OUT VARCHAR2,
                                 v_message_params    OUT VARCHAR2)
IS
  l_clear_count      NUMBER := 0;
  l_critical_count   NUMBER := 0;
  l_warning_count    NUMBER := 0;
  l_error_count      NUMBER := 0;
BEGIN
  analyze_severity_details(v_dependency_list, l_clear_count,
                           l_warning_count, l_critical_count,
                           l_error_count); 

  construct_severity_message(v_clear_count     => l_clear_count,  
                             v_warning_count   => l_warning_count,  
                             v_critical_count  => l_critical_count,
                             v_error_count     => l_error_count,
                             v_message         => v_message );

  v_message_params := l_clear_count||'&'||l_warning_count||'&'||l_critical_count||'&'||l_error_count ;

  -- Evaluate status in this order
  --   o If at least one of the members is CRITICAL, then we are in CRITICAL
  --   o If at least one of the members is WARNING, then we are in WARNING
  --   o If at least one of the members is ERROR, then we are in ERROR
  --   o Otherwise, we are CLEAR
  IF (l_critical_count  > 0) THEN
     v_sev_code := mgmt_global.G_SEVERITY_CRITICAL;
  ELSIF (l_warning_count > 0) THEN
     v_sev_code := mgmt_global.G_SEVERITY_WARNING;
  ELSIF (l_error_count > 0) THEN
     v_sev_code := mgmt_global.G_SEVERITY_ERROR_START;
  ELSE
     v_sev_code := mgmt_global.G_SEVERITY_CLEAR;
  END IF;
END AND_SEVERITY_EVAL_PROC;

END mgmt_severity_helper;

/
show errors


