Rem
Rem $Header: security_pkgbody.sql 31-jul-2007.05:39:58 pchebrol Exp $
Rem
Rem security_pkgbody.sql
Rem
Rem Copyright (c) 2002, 2007, Oracle. All rights reserved.  
Rem
Rem    NAME
Rem      security_pkgbody.sql - APIs to store/retreive EM Security info
Rem
Rem    DESCRIPTION
Rem      APIs to store/retreive EM Security info from tables defined in
Rem      security_tables.sql
Rem
Rem    MODIFIED   (MM/DD/YY)
Rem    pchebrol    06/22/07 - Backport shianand_core_5121288 from main
Rem    pchebrol    06/21/07 - 
Rem    shianand    07/21/06 - Backport shianand_bug-5346292 from main 
Rem    shianand    07/12/06 - fix bug 5346292, fix encrypt, decrypt from 
Rem                           rootkit attacks 
Rem    shianand    07/06/06 - fix bug 5346292, fix encrypt, decrypt from 
Rem                           rootkit attacks 
Rem    shianand    07/29/05 - fix bug 4523084 
Rem    shianand    07/23/05 - fix bug 3335221 
Rem    shianand    06/22/05 - fix bug 4242977 
Rem    neearora    06/10/05 -  added delete_agent_key 
Rem    rpinnama    11/23/04 - Fix 3879698 : Fix key match errors with similar 
Rem                           named hosts 
Rem    rpinnama    12/10/03 - Fix 3310479 : Use encryptBytes/decryptBytes for 
Rem                           byte arrays 
Rem    rpinnama    09/26/03 - Fix get agentguid to lookup properly
Rem    izhang      01/16/03 - 
Rem    izhang      01/06/03 - multiple ipw management
Rem    rpinnama    12/23/02 - Return NULL if the agent is not secured
Rem    izhang      11/13/02 - changing pki from varchar2 to BLOB
Rem    dmshah      10/25/02 - Fixing update agent key
Rem    dmshah      10/18/02 - Set agent_key inserts data into the agent_key if target guid not found
Rem    izhang      10/11/02 - Adding api to take host,port
Rem    dmshah      10/03/02 - dmshah_security_repository_apis
Rem    dmshah      10/02/02 - 
Rem    dmshah      10/02/02 - Code review comments
Rem    dmshah      09/30/02 - Fixing GET_AGENT_KEY
Rem    dmshah      09/26/02 - Created
Rem

CREATE OR REPLACE PACKAGE BODY EM_SEC AS

--
-- This is a private procedure which returns an agent_guid
-- if the agent_guid is in targets table, otherwise calculates
-- the guid and returns it
--
PROCEDURE GET_AGENT_GUID( host IN VARCHAR2,
                          port IN VARCHAR2,
                          agent_guid_out OUT RAW) IS
  url_in varchar2(512);

  CURSOR get_agent_guid (url_in VARCHAR2) IS
    SELECT target_guid FROM MGMT_TARGETS
    WHERE target_type=MGMT_GLOBAL.G_AGENT_TARGET_TYPE AND
    emd_url like url_in;

BEGIN
  url_in := '%://'|| host || ':' || port || '/%';

  OPEN get_agent_guid( url_in );
  FETCH get_agent_guid INTO agent_guid_out;

  IF get_agent_guid%NOTFOUND THEN
    agent_guid_out := mgmt_target.generate_target_guid(
                                        host||':'||port,
                                        MGMT_GLOBAL.G_AGENT_TARGET_TYPE);
  END IF;

  CLOSE get_agent_guid;

END;

--
-- SET_AGENT_IPW
--
-- PURPOSE
--
--   Set the Agent IPW
--
-- PARAMETERS
--
--   VERIFIER_IN - The initial Agent Installation Password that generates
--                 the actual agent key.
--
--
PROCEDURE SET_AGENT_IPW( verifier_in  IN VARCHAR2) IS
BEGIN
    UPDATE MGMT_SEC_INFO SET ipw =  encrypt(verifier_in) ;
END SET_AGENT_IPW;

--
-- GET_AGENT_IPW
--
-- PURPOSE
--
--   Get the Agent IPW
--
-- PARAMETERS
--
--   VERIFIER_OUT - The initial Agent Installation Password that generates
--                  the actual agent key.
--
--
PROCEDURE GET_AGENT_IPW( verifier_out  OUT VARCHAR2) IS
BEGIN
    SELECT decrypt(ipw) INTO verifier_out FROM mgmt_sec_info;
END GET_AGENT_IPW;

--
-- PURPOSE
--
--   Set the Certificate Authority
--
-- PARAMETERS
--
--   PKI - The PKI material in the CA
--   CA_PWD - The PWD in the CA
--   CA     - The actual CA
--   RESET  - Whether or not to reset (if previously set). If passed in as 0
--   (do not reset) and if 1 is passed reset the CA. 
--   
--
PROCEDURE SET_CA( pki_in    IN RAW,
                  ca_pwd_in IN varchar2,
                  ca_in     IN RAW,
                  reset_in  IN NUMBER DEFAULT 0 ) IS

l_ca_pwd_in mgmt_sec_info.ca_pwd%TYPE := '';

BEGIN
    LOCK TABLE mgmt_sec_info IN EXCLUSIVE MODE NOWAIT;

    IF(reset_in = 0) THEN
      SELECT ca_pwd INTO l_ca_pwd_in
        FROM mgmt_sec_info;
      IF(l_ca_pwd_in IS NULL) THEN
         UPDATE MGMT_SEC_INFO SET ca_pki = pki_in,
                                  ca_pwd = encrypt(ca_pwd_in),
                                  ca     = encryptBytes(ca_in);
      END IF;
    ELSIF (reset_in = 1) THEN
      UPDATE MGMT_SEC_INFO SET ca_pki = pki_in,
                               ca_pwd = encrypt(ca_pwd_in),
                               ca     = encryptBytes(ca_in);
    END IF;

END SET_CA;

--
-- PURPOSE
--
--   Get the Certificate Authority
--
-- PARAMETERS
--
--   PKI - The PKI material in the CA
--   CA_PWD - The PWD in the CA
--   CA     - The actual CA
--
PROCEDURE GET_CA( pki_out    OUT RAW,
                  ca_pwd_out OUT varchar2,
                  ca_out     OUT RAW ) IS
BEGIN

   -- SELECT SYS.UTL_RAW.cast_to_raw(decrypt(SYS.UTL_RAW.cast_to_varchar2(ca)))
   SELECT decryptBytes(ca)
          INTO ca_out FROM mgmt_sec_info;

   SELECT ca_pki INTO pki_out FROM mgmt_sec_info;
   SELECT decrypt(ca_pwd) INTO ca_pwd_out FROM mgmt_sec_info;

END GET_CA;

--
-- PURPOSE
--
--   Set Agent Key
--
-- PARAMETERS
--
--   HOST   - The host name
--   PORT   - The port
--   KEY    - The actual CA
--
PROCEDURE SET_AGENT_KEY( host     IN varchar2,
                         port     IN varchar2,
                         key      IN varchar2) IS
  agent_guid mgmt_targets.target_guid%type;
  agent_guid_temp mgmt_targets.target_guid%type;

  CURSOR get_agent (agent_guid_in RAW) IS
    SELECT target_guid FROM MGMT_AGENT_SEC_INFO
    WHERE target_guid=agent_guid_in;

BEGIN
  
  EM_SEC.GET_AGENT_GUID( host, port, agent_guid);

  OPEN get_agent( agent_guid );
  FETCH get_agent INTO agent_guid_temp;

  IF get_agent%NOTFOUND THEN
    INSERT INTO MGMT_AGENT_SEC_INFO( target_guid, agent_key)
       VALUES ( agent_guid, encrypt(key));
  ELSE
    UPDATE MGMT_AGENT_SEC_INFO
           SET agent_key=encrypt(key) where target_guid=agent_guid;
  END IF;

  CLOSE get_agent;

END SET_AGENT_KEY;

-- PURPOSE
--
--   Delete Agent Key
--
-- PARAMETERS
--
--   HOST   - The host name
--   PORT   - The port
PROCEDURE DELETE_AGENT_KEY( host     IN varchar2,
                            port     IN varchar2)
IS
  agent_guid mgmt_targets.target_guid%type;
BEGIN
  
  EM_SEC.GET_AGENT_GUID( host, port, agent_guid);
  
  DELETE 
    FROM mgmt_agent_sec_info
   WHERE target_guid = agent_guid;
   
END;
-- PURPOSE
--
--   Get Agent Key
--
-- PARAMETERS
--
--   HOST   - The host name
--   PORT   - The port
PROCEDURE GET_AGENT_KEY( host         IN varchar2,
                         port         IN varchar2,
                         key_out     OUT varchar2 ) IS
  agent_guid mgmt_targets.target_guid%type;
BEGIN
  BEGIN
    key_out := NULL;

    EM_SEC.GET_AGENT_GUID( host, port, agent_guid);

    SELECT decrypt(agent_key) INTO key_out 
      FROM MGMT_AGENT_SEC_INFO 
     WHERE target_guid = agent_guid;

  EXCEPTION
    WHEN NO_DATA_FOUND THEN
      -- When Agent is not secured, return NULL
      key_out := NULL;      
  END;

END GET_AGENT_KEY;

--
-- PURPOSE
--
--   Set One IPW information
--
--
--
PROCEDURE SET_IPW( pwd_idx_in   IN VARCHAR2,
                   pwd_in       IN VARCHAR2,
                   desc_str_in  IN VARCHAR2,
                   type_in      IN VARCHAR2,
                   expire_in    IN DATE) IS
BEGIN
    INSERT INTO em_ipw_info (pwd_idx, ipw, desc_str, pwd_type, expire_date)
    VALUES(pwd_idx_in, encrypt(pwd_in), desc_str_in, type_in, expire_in);

END SET_IPW;

--
-- PURPOSE
--
--   Get ONE IPW information with SPECIFIED PASSWORD INDEX
--
-- PARAMETERS
--
--
PROCEDURE GET_IPW( pwd_idx_in    IN  VARCHAR2,
                   pwd_out       OUT VARCHAR2,
                   desc_str_out  OUT VARCHAR2,
                   type_out      OUT VARCHAR2,
                   expire_out    OUT DATE) IS
BEGIN
    SELECT  decrypt(ipw), desc_str, pwd_type, expire_date
    INTO    pwd_out, desc_str_out, type_out, expire_out
    FROM    em_ipw_info
    WHERE   pwd_idx = pwd_idx_in;

END GET_IPW;



--
-- PURPOSE
--
--   Get ALL Agent IPWs information
--
-- PARAMETERS
--
--
PROCEDURE GET_IPW_INFO(pwd_info_out   OUT cursorType) IS
                  
BEGIN
  OPEN pwd_info_out FOR
  SELECT distinct pwd_idx, decrypt(ipw), desc_str, pwd_type, expire_date
  FROM em_ipw_info
  ORDER BY pwd_idx;

END GET_IPW_INFO;



--
-- PURPOSE
--
-- Validates the given verifier with the verifier stored in the Repository  
--
-- Function returns the following:
-- 0 - If the verifier passed in the same as the one in the repository
-- 1 - If the verifier passed is not the same as the one in the repository
-- 2 - If no verifier is stored in the repository. 


FUNCTION VERIFY_IPW(verifier_in IN VARCHAR2) RETURN NUMBER IS

l_ret         NUMBER        := 2;
l_count       NUMBER        := 0;
l_verifier_in VARCHAR2(256) := null;

BEGIN
  l_verifier_in := SYS.DBMS_OBFUSCATION_TOOLKIT.md5(input => SYS.UTL_RAW.cast_to_raw(verifier_in));
 
  l_verifier_in := SYS.DBMS_OBFUSCATION_TOOLKIT.md5(input => SYS.UTL_RAW.cast_to_raw(l_verifier_in));
 
  SELECT count(*) into l_count
  FROM em_ipw_info;
  
  IF (l_count = 0) THEN
     l_ret := 2;
  ELSE
    SELECT count(*) into l_count
    FROM em_ipw_info
    WHERE pwd_idx = l_verifier_in;

    IF (l_count > 0) THEN
       l_ret := 0;
    ELSE
       l_ret := 1;
    END IF;
  END IF;

  RETURN l_ret;
END VERIFY_IPW;


--
-- PURPOSE
-- 
-- This procedure updates the row with the b64 local certificate while securing oms.
-- 
-- 
PROCEDURE UPDATE_B64_LOCAL_CA(p_b64_local_ca_out OUT CLOB) IS

BEGIN

  SELECT b64_local_ca INTO p_b64_local_ca_out FROM mgmt_sec_info
  FOR UPDATE;

  EXCEPTION
    WHEN OTHERS THEN
      raise;

END UPDATE_B64_LOCAL_CA;


--
-- PURPOSE
-- 
-- This procedure updates the row with the b64 internet certificate while securing oms.
-- 
-- 
PROCEDURE UPDATE_B64_INTERNET_CA(p_b64_internet_ca_out OUT CLOB) IS

BEGIN

  SELECT b64_internet_ca INTO p_b64_internet_ca_out FROM mgmt_sec_info
    FOR UPDATE;

  EXCEPTION
    WHEN OTHERS THEN
      raise;

END UPDATE_B64_INTERNET_CA;


--
-- PURPOSE
-- 
-- This procedure updates the row with the agent-wallet generated during auth request.
-- 
-- 
PROCEDURE UPDATE_AGENT_EWALLET(p_agent_host_in IN VARCHAR2,
                               p_agent_port_in IN VARCHAR2,
                               p_request_status_in IN VARCHAR2,
                               p_agent_ewallet_out OUT BLOB) IS

agent_guid mgmt_targets.target_guid%type;

BEGIN

  EM_SEC.GET_AGENT_GUID(p_agent_host_in, p_agent_port_in, agent_guid);

  SELECT agent_ewallet INTO p_agent_ewallet_out
  FROM mgmt_agent_sec_info
  WHERE target_guid = agent_guid
  FOR UPDATE;

  EXCEPTION
    WHEN OTHERS THEN
      raise;

END UPDATE_AGENT_EWALLET;


--
-- PURPOSE
--
-- This routines initializes the for a particular agent-guid
--
--
--
PROCEDURE INITIALIZE (p_agent_guid_in VARCHAR2) IS

BEGIN
  INSERT INTO mgmt_agent_sec_info 
  (target_guid, agent_host, agent_seed, oms_seed, request_status, agent_version)
  VALUES 
  (p_agent_guid_in, null, null, null, STATUS_UNINIT, null);

  EXCEPTION
    WHEN DUP_VAL_ON_INDEX THEN
      UPDATE mgmt_agent_sec_info
      SET agent_host = null,
          agent_seed = null,
          request_status = STATUS_UNINIT,
          agent_version  = null
      WHERE target_guid = p_agent_guid_in;

END INITIALIZE;


--
-- PURPOSE
--
-- This routines checks whether the IPW stored in the repository is expired or valid.
-- it also checks the IPW type and if its one-time then it deletes from the repository.
--
--
FUNCTION VERIFY_PWD_IDX(p_pwd_idx_in VARCHAR2,
                        p_update_ipw BOOLEAN DEFAULT FALSE) RETURN NUMBER IS

l_ret NUMBER := IPW_NOT_FOUND;
l_type_out em_ipw_info.ipw%type           := '';
l_expire_out em_ipw_info.expire_date%type := null;
l_curr_date DATE := SYSDATE();

BEGIN

  SELECT pwd_type, expire_date
  INTO l_type_out, l_expire_out
  FROM em_ipw_info
  WHERE pwd_idx = p_pwd_idx_in;

  IF(l_expire_out IS NULL OR l_expire_out >= l_curr_date ) THEN
     l_ret := IPW_VALID;

     IF(p_update_ipw) THEN

       IF(l_type_out = ONE_TIME_IPW) THEN
         DELETE FROM em_ipw_info
         WHERE pwd_idx = p_pwd_idx_in
         AND pwd_type = ONE_TIME_IPW;

         l_ret := IPW_ONE_TIME_EXP_SUCCESS;

       ELSIF(l_type_out = NORMAL_IPW) THEN
         l_ret := IPW_PERSISTENT;
       END IF;

     END IF;

  ELSE
     l_ret := IPW_EXPIRED;
  END IF;

  RETURN l_ret;

  EXCEPTION
    WHEN NO_DATA_FOUND THEN
      RETURN IPW_NOT_FOUND;
    WHEN OTHERS THEN
      RETURN IPW_ERROR;

END VERIFY_PWD_IDX;


--
-- PURPOSE
--
-- This routines returns true if init is already completed.
--
--
FUNCTION IS_INIT_COMPLETE(p_agent_guid_in VARCHAR2) RETURN BOOLEAN IS

l_request_status_out mgmt_agent_sec_info.request_status%type := '';

BEGIN

  SELECT request_status INTO l_request_status_out
  FROM mgmt_agent_sec_info
  WHERE target_guid = p_agent_guid_in
  AND request_status = STATUS_INIT_COMPLETED;

  RETURN true;

  EXCEPTION
    WHEN NO_DATA_FOUND THEN
    RETURN false;

END IS_INIT_COMPLETE;


--
-- PURPOSE
--
-- This routines returns true if init is already completed.
--
--
FUNCTION IS_WALLET_DOWNLOAD_COMPLETE(p_agent_host_in VARCHAR2,
                                     p_agent_port_in IN VARCHAR2 DEFAULT UNDEFINED_PORT,
                                     p_agent_version_in IN VARCHAR2) RETURN BOOLEAN IS

l_request_status_out mgmt_agent_sec_info.request_status%type := '';
agent_guid mgmt_targets.target_guid%type;

BEGIN

  -- this check is done for backward compatibility where agent doesnt send port and pwd_idx during download
  IF (p_agent_port_in = UNDEFINED_PORT) THEN

    SELECT request_status INTO l_request_status_out
    FROM mgmt_agent_sec_info
    WHERE agent_host = p_agent_host_in
    AND agent_version = p_agent_version_in
    AND request_status = STATUS_DOWNLOAD_WALLET;

  ELSE
    EM_SEC.GET_AGENT_GUID(p_agent_host_in, p_agent_port_in, agent_guid);

    SELECT request_status INTO l_request_status_out
    FROM mgmt_agent_sec_info
    WHERE target_guid = agent_guid
    AND request_status = STATUS_DOWNLOAD_WALLET;

  END IF;

  RETURN true;

  EXCEPTION
    WHEN OTHERS THEN
      RETURN false;

END IS_WALLET_DOWNLOAD_COMPLETE;


--
-- PURPOSE
--
-- This routines returns true if init is already completed.
--
--
FUNCTION IS_B64LOC_DOWNLOAD_COMPLETE(p_agent_host_in VARCHAR2,
                                     p_agent_port_in IN VARCHAR2 DEFAULT UNDEFINED_PORT,
                                     p_agent_version_in IN VARCHAR2) RETURN BOOLEAN IS

l_request_status_out mgmt_agent_sec_info.request_status%type := '';
agent_guid mgmt_targets.target_guid%type;

BEGIN

  -- this check is done for backward compatibility where agent doesnt send port and pwd_idx during download
  IF (p_agent_port_in = UNDEFINED_PORT) THEN

    SELECT request_status INTO l_request_status_out
    FROM mgmt_agent_sec_info
    WHERE agent_host = p_agent_host_in
    AND agent_version = p_agent_version_in
    AND request_status = STATUS_DOWNLOAD_LOCAL_CA;

  ELSE
    EM_SEC.GET_AGENT_GUID(p_agent_host_in, p_agent_port_in, agent_guid);

    SELECT request_status INTO l_request_status_out
    FROM mgmt_agent_sec_info
    WHERE target_guid = agent_guid
    AND request_status = STATUS_DOWNLOAD_LOCAL_CA;

  END IF;

  RETURN true;

  EXCEPTION
    WHEN OTHERS THEN
      RETURN false;

END IS_B64LOC_DOWNLOAD_COMPLETE;


--
-- PURPOSE
--
-- The procedure will set the request_status against the agent-guid depending on the AUTH,
-- DOWNLOAD request in the mgmt_agent_sec_info table.
-- 
-- 
FUNCTION UPDATE_STATUS(p_agent_host_in VARCHAR2,
                       p_agent_port_in IN VARCHAR2 DEFAULT UNDEFINED_PORT,
                       p_agent_version_in IN VARCHAR2,
                       p_request_status_in VARCHAR2) RETURN NUMBER IS

l_update_count NUMBER := 0;
l_ret          NUMBER := UPDATE_STATUS_ERROR;
agent_guid mgmt_targets.target_guid%type;

BEGIN

  -- this check is done for backward compatibility where agent doesnt send port and pwd_idx during download
  IF (p_agent_port_in = UNDEFINED_PORT) THEN

        UPDATE mgmt_agent_sec_info
        SET request_status = p_request_status_in
        WHERE agent_host = p_agent_host_in
        AND agent_version = p_agent_version_in;

        l_update_count := SQL%ROWCOUNT;

  ELSE
        EM_SEC.GET_AGENT_GUID(p_agent_host_in, p_agent_port_in, agent_guid);

        UPDATE mgmt_agent_sec_info
        SET request_status = p_request_status_in
        WHERE target_guid = agent_guid;

        l_update_count := SQL%ROWCOUNT;
  END IF;

  IF (l_update_count = 1) THEN
    l_ret := UPDATE_STATUS_SUCCESS;
  ELSIF (l_update_count > 1) THEN
    l_ret := UPDATE_STATUS_MANY_ROWS;
  ELSE
    l_ret := UPDATE_STATUS_ERROR;
  END IF;

  RETURN l_ret;

  EXCEPTION
    WHEN OTHERS THEN
      RETURN UPDATE_STATUS_ERROR;

END UPDATE_STATUS;



--
-- PURPOSE
--
-- The procedure will set the agent random seed along with the agent-guid during
-- INIT request in the mgmt_agent_sec_info table.The procedure will also set the
-- oms random seed along with the agent-guid in the mgmt_agent_sec_info table.
-- 
-- 
PROCEDURE PROCESS_INIT(p_agent_host_in IN VARCHAR2,
                       p_agent_port_in IN VARCHAR2,
                       p_agent_seed_in IN VARCHAR2,
                       p_oms_seed_in IN VARCHAR2,
                       p_pwd_idx_in IN VARCHAR2,
                       p_agent_version_in IN VARCHAR2) IS

agent_guid mgmt_targets.target_guid%type;
l_ret NUMBER              := 1;

BEGIN

  EM_SEC.GET_AGENT_GUID(p_agent_host_in, p_agent_port_in, agent_guid);

  -- initiliaze the row corresponding to agent-guid
  INITIALIZE(agent_guid);

  -- verifies the pwd_idx
  l_ret := VERIFY_PWD_IDX(p_pwd_idx_in);

  IF(l_ret = IPW_VALID) THEN

  -- updates the row with the agent-seed, oms-seed and updates the status to init completed.
    UPDATE mgmt_agent_sec_info
    SET agent_host = p_agent_host_in,
        agent_seed = encrypt(p_agent_seed_in),
        oms_seed = encrypt(p_oms_seed_in),
        agent_version = p_agent_version_in,
        request_status = STATUS_INIT_COMPLETED
    WHERE target_guid = agent_guid;

  ELSIF (l_ret = IPW_EXPIRED) THEN
    raise_application_error(MGMT_GLOBAL.INVALID_REG_PWD_ERR,
                            MGMT_GLOBAL.INVALID_REG_PWD_ERR_M || ' Registration password provided has expired.');
  ELSIF (l_ret = IPW_NOT_FOUND) THEN
    raise_application_error(MGMT_GLOBAL.INVALID_REG_PWD_ERR,
                            MGMT_GLOBAL.INVALID_REG_PWD_ERR_M );
  ELSE
    raise_application_error(MGMT_GLOBAL.PROCESS_INIT_EX_ERR,
                            VERIFY_IPW_ERR_MSG);
  END IF;

END PROCESS_INIT;


--
-- PURPOSE
-- 
-- This procedure carries out the auth, it checks the agent-seed and auth-str and pwd-idx then generates 
-- agent-key and stores it in the repository.
-- 
-- 
PROCEDURE PROCESS_AUTH(p_agent_host_in IN VARCHAR2,
                       p_agent_port_in IN VARCHAR2,
                       p_agent_seed_in IN VARCHAR2,
                       p_auth_str_in IN VARCHAR2,
                       p_pwd_idx_in IN VARCHAR2,
                       p_agent_version_in IN VARCHAR2,
                       p_agent_key_out OUT VARCHAR2) IS

l_ret NUMBER                                         := IPW_NOT_FOUND;
l_auth_str VARCHAR2(256)                             := '';
l_pwd_out em_ipw_info.pwd_type%type                  := '';
l_oms_seed_out mgmt_agent_sec_info.oms_seed%type     := '';
l_agent_seed_out mgmt_agent_sec_info.agent_seed%type := '';

agent_guid mgmt_targets.target_guid%type;

BEGIN

  EM_SEC.GET_AGENT_GUID(p_agent_host_in, p_agent_port_in, agent_guid);

-- verifies the pwd_idx
  l_ret := VERIFY_PWD_IDX(p_pwd_idx_in);

  IF(l_ret = IPW_VALID) THEN

    -- checks if init is already done or not.
    IF(IS_INIT_COMPLETE(agent_guid)) THEN

      BEGIN

        SELECT decrypt(ipw) INTO l_pwd_out
        FROM em_ipw_info
        WHERE pwd_idx = p_pwd_idx_in;

        SELECT decrypt(agent_seed), decrypt(oms_seed)
        INTO l_agent_seed_out, l_oms_seed_out
        FROM mgmt_agent_sec_info
        WHERE target_guid = agent_guid;

        -- calculates the auth string
        l_auth_str := SYS.DBMS_OBFUSCATION_TOOLKIT.md5(input => SYS.UTL_RAW.cast_to_raw(l_pwd_out || l_oms_seed_out));

        -- checks if the auth string and agent-seed matches with the one sent through agent-request.
        IF(l_agent_seed_out = p_agent_seed_in AND l_auth_str = p_auth_str_in) THEN

          -- calculates the agent-key.
          p_agent_key_out := SYS.DBMS_OBFUSCATION_TOOLKIT.md5(input => SYS.UTL_RAW.cast_to_raw(l_agent_seed_out || l_pwd_out || l_oms_seed_out));

          -- stores the agent-key.
          SET_AGENT_KEY(p_agent_host_in, p_agent_port_in, p_agent_key_out);

          IF (p_agent_version_in = MIN_COMPAT_AGENT_VER) THEN
            -- this is updating the verifier in the repository to expired if its one-time password for backward compatibility.
            l_ret := VERIFY_PWD_IDX(p_pwd_idx_in, true);

            IF (l_ret != IPW_PERSISTENT and l_ret != IPW_ONE_TIME_EXP_SUCCESS) THEN
              raise_application_error(MGMT_GLOBAL.PROCESS_DOWNLOAD_EX_ERR,
                                      ONE_TIME_EXP_IPW_FAILED_MSG);
            END IF;

          END IF;

          -- updates the status to auth completed
          IF (UPDATE_STATUS(p_agent_host_in, p_agent_port_in, p_agent_version_in, STATUS_AUTH_COMPLETED) != UPDATE_STATUS_SUCCESS) THEN
             raise_application_error(MGMT_GLOBAL.PROCESS_AUTH_EX_ERR,
                                    PROCESS_AUTH_FAILED_MSG);
          END IF;
        ELSE
          p_agent_key_out := null;
          raise_application_error(MGMT_GLOBAL.PROCESS_AUTH_EX_ERR,
                                  INVALID_AUTH_STR);
        END IF;

      END;

    ELSE
      raise_application_error(MGMT_GLOBAL.PROCESS_AUTH_EX_ERR,
                              INIT_NOT_DONE);

    END IF;

  ELSIF (l_ret = IPW_EXPIRED) THEN
    raise_application_error(MGMT_GLOBAL.INVALID_REG_PWD_ERR,
                            MGMT_GLOBAL.INVALID_REG_PWD_ERR_M || ' Registration password provided has expired.');
  ELSIF (l_ret = IPW_NOT_FOUND) THEN
    raise_application_error(MGMT_GLOBAL.INVALID_REG_PWD_ERR,
                            MGMT_GLOBAL.INVALID_REG_PWD_ERR_M );
  ELSE
    raise_application_error(MGMT_GLOBAL.PROCESS_AUTH_EX_ERR,
                              VERIFY_IPW_ERR_MSG);
  END IF;

END PROCESS_AUTH;


--
-- PURPOSE
--
-- The procedure will delete the agent random seed along with oms random seed,
-- agent-wallet and the request_status stored against agent-guid from the 
-- mgmt_agent_sec_info table.
-- 
-- 
PROCEDURE CLEAN_UP(p_agent_host_in IN VARCHAR2,
                   p_agent_port_in IN VARCHAR2 DEFAULT UNDEFINED_PORT,
                   p_agent_version_in IN VARCHAR2) IS

agent_guid mgmt_targets.target_guid%type;

BEGIN

  IF (p_agent_port_in = UNDEFINED_PORT) THEN
        UPDATE mgmt_agent_sec_info
          SET agent_host = null,
              agent_seed = null,
              oms_seed = null,
              request_status = STATUS_UNINIT,
              agent_version = null,
              agent_ewallet = empty_blob()
          WHERE agent_host = p_agent_host_in
          AND agent_version = p_agent_version_in;
  ELSE

        EM_SEC.GET_AGENT_GUID(p_agent_host_in, p_agent_port_in, agent_guid);

        UPDATE mgmt_agent_sec_info
          SET agent_host = null,
              agent_seed = null,
              oms_seed = null,
              request_status = STATUS_UNINIT,
              agent_version = null,
              agent_ewallet = empty_blob()
          WHERE target_guid = agent_guid;
  END IF;

  EXCEPTION
    WHEN OTHERS THEN
      raise;

END CLEAN_UP;


--
-- PURPOSE
--
-- This procedure validates the pwd_idx and then open a outputstream of b64-Local-Certificate.
-- For older agents it also checks the agent_version to refine the search and avoid collisions.
-- Port undefined is the older agent case.
-- 
-- 
PROCEDURE GET_B64_LOCAL_CA(p_agent_host_in IN VARCHAR2,
                           p_agent_port_in IN VARCHAR2 DEFAULT UNDEFINED_PORT,
                           p_b64_local_ca_out OUT CLOB,
                           p_agent_version_in IN VARCHAR2,
                           p_pwd_idx_in IN VARCHAR2) IS

l_ret NUMBER := IPW_NOT_FOUND;
is_old BOOLEAN := false;

BEGIN

  -- this check is done for backward compatibility where agent doesnt send port and pwd_idx during download
  IF (p_agent_port_in != UNDEFINED_PORT) THEN
    l_ret := VERIFY_PWD_IDX(p_pwd_idx_in);
  ELSE
    l_ret := IPW_VALID;
  END IF;

  IF (l_ret = IPW_VALID) THEN

  -- this checks that whether wallet download is already done or not.
    IF (IS_WALLET_DOWNLOAD_COMPLETE(p_agent_host_in, p_agent_port_in, p_agent_version_in))THEN

      SELECT b64_local_ca INTO p_b64_local_ca_out
      FROM mgmt_sec_info;

      -- this updating the status to download b64-Local-Certificate completed.
      IF (UPDATE_STATUS(p_agent_host_in, p_agent_port_in, p_agent_version_in, STATUS_DOWNLOAD_LOCAL_CA) != UPDATE_STATUS_SUCCESS) THEN
        raise_application_error(MGMT_GLOBAL.PROCESS_DOWNLOAD_EX_ERR,
                                PROCESS_AUTH_FAILED_MSG);
      END IF;

    ELSE
      raise_application_error(MGMT_GLOBAL.PROCESS_DOWNLOAD_EX_ERR,
                              WALLET_NOT_DONE);
    END IF;

  ELSIF (l_ret = IPW_EXPIRED) THEN
    raise_application_error(MGMT_GLOBAL.PROCESS_DOWNLOAD_EX_ERR,
                            IPW_EXPIRED_MSG);
  ELSIF (l_ret = IPW_NOT_FOUND) THEN
    raise_application_error(MGMT_GLOBAL.PROCESS_DOWNLOAD_EX_ERR,
                            IPW_NOT_FOUND_MSG);
  ELSE
    raise_application_error(MGMT_GLOBAL.PROCESS_DOWNLOAD_EX_ERR,
                            VERIFY_IPW_ERR_MSG);
  END IF;

END GET_B64_LOCAL_CA;


--
-- PURPOSE
--
-- This procedure validates the pwd_idx and then open a outputstream of b64-Internet-Certificate.
-- For older agents it also checks the agent_version to refine the search and avoid collisions.
-- Port undefined is the older agent case.
-- 
-- 
PROCEDURE GET_B64_INTERNET_CA(p_agent_host_in IN VARCHAR2,
                              p_agent_port_in IN VARCHAR2 DEFAULT UNDEFINED_PORT,
                              p_b64_internet_ca_out OUT CLOB,
                              p_agent_version_in IN VARCHAR2,
                              p_pwd_idx_in IN VARCHAR2) IS

l_ret NUMBER := IPW_NOT_FOUND;
is_old BOOLEAN := false;

BEGIN

  -- this check is done for backward compatibility where agent doesnt send port and pwd_idx during download
  IF (p_agent_port_in != UNDEFINED_PORT) THEN
    -- this is deleting the verifier from the repository if its one-time password.
    l_ret := VERIFY_PWD_IDX(p_pwd_idx_in, true);
  ELSE
    l_ret := IPW_ONE_TIME_EXP_SUCCESS;
  END IF;

  IF (l_ret = IPW_PERSISTENT or l_ret = IPW_ONE_TIME_EXP_SUCCESS) THEN

  -- this checks that whether download of b64-Local-Certificate is completed or not.
    IF(IS_B64LOC_DOWNLOAD_COMPLETE(p_agent_host_in, p_agent_port_in, p_agent_version_in)) THEN

      SELECT b64_internet_ca INTO p_b64_internet_ca_out
      FROM mgmt_sec_info ;

      -- this updating the status to download b64-Internet-Certificate completed.
      IF (UPDATE_STATUS(p_agent_host_in, p_agent_port_in, p_agent_version_in, STATUS_DOWNLOAD_INTERNTET_CA) != UPDATE_STATUS_SUCCESS) THEN
        raise_application_error(MGMT_GLOBAL.PROCESS_DOWNLOAD_EX_ERR,
                                PROCESS_AUTH_FAILED_MSG);
      END IF;

      CLEAN_UP(p_agent_host_in, p_agent_port_in, p_agent_version_in);

    ELSE
      raise_application_error(MGMT_GLOBAL.PROCESS_DOWNLOAD_EX_ERR,
                              B64_LOC_NOT_DONE);
    END IF;

  ELSIF (l_ret = IPW_EXPIRED) THEN
    raise_application_error(MGMT_GLOBAL.PROCESS_DOWNLOAD_EX_ERR,
                            IPW_EXPIRED_MSG);
  ELSIF (l_ret = IPW_NOT_FOUND) THEN
    raise_application_error(MGMT_GLOBAL.PROCESS_DOWNLOAD_EX_ERR,
                            IPW_NOT_FOUND_MSG);
  ELSE
    raise_application_error(MGMT_GLOBAL.PROCESS_DOWNLOAD_EX_ERR,
                            ONE_TIME_EXP_IPW_FAILED_MSG);
  END IF;

END GET_B64_INTERNET_CA;


--
-- PURPOSE
-- 
-- This procedure validates the pwd_idx and then open a outputstream of agent wallet corresponding 
-- agent-guid. For older agents it also checks the agent_version to refine the search and avoid collisions.
-- Port undefined is the older agent case
-- 
-- 
PROCEDURE GET_AGENT_EWALLET(p_agent_host_in IN VARCHAR2,
                            p_agent_port_in IN VARCHAR2 DEFAULT UNDEFINED_PORT,
                            p_agent_ewallet_out OUT BLOB,
                            p_agent_version_in IN VARCHAR2,
                            p_pwd_idx_in IN VARCHAR2 DEFAULT NULL) IS

l_ret NUMBER := 1;
agent_guid mgmt_targets.target_guid%type;

BEGIN

  EM_SEC.GET_AGENT_GUID(p_agent_host_in, p_agent_port_in, agent_guid);

  -- this check is done for backward compatibility where agent doesnt send port and pwd_idx during download
  IF (p_agent_port_in = UNDEFINED_PORT) THEN

    SELECT agent_ewallet INTO p_agent_ewallet_out
    FROM mgmt_agent_sec_info
    WHERE agent_host = p_agent_host_in
    AND agent_version = p_agent_version_in
    AND request_status = STATUS_AUTH_COMPLETED;

  ELSE

    -- this check validates the pwd_idx 
    l_ret := VERIFY_PWD_IDX(p_pwd_idx_in);

    IF (l_ret = IPW_VALID) THEN

      --checks whether Auth is already done or not.
      SELECT agent_ewallet INTO p_agent_ewallet_out
      FROM mgmt_agent_sec_info
      WHERE target_guid = agent_guid
      AND request_status = STATUS_AUTH_COMPLETED;


    ELSIF (l_ret = IPW_EXPIRED) THEN
      raise_application_error(MGMT_GLOBAL.PROCESS_DOWNLOAD_EX_ERR,
                              IPW_EXPIRED_MSG);
    ELSIF (l_ret = IPW_NOT_FOUND) THEN
      raise_application_error(MGMT_GLOBAL.PROCESS_DOWNLOAD_EX_ERR,
                              IPW_NOT_FOUND_MSG);
    ELSE
      raise_application_error(MGMT_GLOBAL.PROCESS_DOWNLOAD_EX_ERR,
                              VERIFY_IPW_ERR_MSG);
    END IF;

  END IF;

  -- this updating the status to download agent-wallet completed.
  IF (UPDATE_STATUS(p_agent_host_in, p_agent_port_in, p_agent_version_in, STATUS_DOWNLOAD_WALLET) != UPDATE_STATUS_SUCCESS) THEN
     raise_application_error(MGMT_GLOBAL.PROCESS_DOWNLOAD_EX_ERR,
                             PROCESS_AUTH_FAILED_MSG);
  END IF;

  EXCEPTION
    WHEN NO_DATA_FOUND THEN
      raise_application_error(MGMT_GLOBAL.PROCESS_DOWNLOAD_EX_ERR,
                              AUTH_NOT_DONE);
END GET_AGENT_EWALLET;


--
-- PURPOSE
--
-- The procedure will get the secure slb port from mgmt_oms_parameters table.
-- 
-- 
PROCEDURE GET_SEC_SLB_PORT(p_oms_instance_in  IN VARCHAR2,
                           p_slb_sec_port_out OUT VARCHAR2) IS

BEGIN

  SELECT value INTO p_slb_sec_port_out
  FROM mgmt_oms_parameters
  WHERE host_url = p_oms_instance_in
  AND name = SLB_SECURE_UPLOAD_PORT;

  EXCEPTION
    WHEN NO_DATA_FOUND THEN
      p_slb_sec_port_out := UNDEFINED_PORT;
      raise;

END GET_SEC_SLB_PORT;

END EM_SEC;
/

show errors;
