Rem 
Rem common_pkgbodys.sql 
Rem 
Rem Copyright (c) 2004, 2007, Oracle. All rights reserved.  
Rem 
Rem    NAME 
Rem      common_pkgbody.sql - <one-line expansion of the name>
Rem
Rem    DESCRIPTION
Rem      PL/SQL package storage_ecm_pkg for post processing of the storage report ecm snapshot
Rem
Rem    NOTES
Rem      <other useful comments, qualifications, etc.>
Rem
Rem    MODIFIED   (MM/DD/YY)
Rem    mnihalan    05/08/07 - Backport mnihalan_bug-6026731 from main
Rem    ajdsouza    09/08/05 - fix bug 4589706, check for array before runing updates for guid
Rem    sreddy      07/23/05 - Filter for current snapshots in
Rem                           mgmt_v_storage_report_data view
Rem    mgoodric    07/20/05 - add additional parameter to after_load proc 
Rem    ajdsouza    04/25/05 - convert queries to do blk processing instead of loops
Rem                            use temporary tables
Rem    ajdsouza    02/09/05 - Log a single message if diffrent aliases present 
Rem                           for NFS server
Rem    ajdsouza    02/03/05 - remove the virtual flag from entities that are used by the db
Rem                            added function STORAGE_ECM_PKG.DIGEST_GLOBAL_UNIQUE_ID
Rem    ajdsouza    01/26/05 - Removed field message from mgmt_storage_report_issues, added field message_counter
Rem    ajdsouza    01/11/05 - free space in unallocated disk partitions shown in disk
Rem    ajdsouza    12/03/04 - 
Rem    ajdsouza    11/18/04 - Enhance nls messages for issues
Rem    ajdsouza    08/27/04 - 
Rem    ajdsouza    08/10/04 - ajdsouza_ecm_processing
Rem    ajdsouza    08/02/04 - add procedure for db and nfs post processing
Rem    ajdsouza    07/30/04 - ajdsouza_fix_storage_metrics
Rem    ajdsouza    07/28/04 - Moved to new storage component under functional unit common
Rem    ajdsouza    06/25/04 - storage reporting sources 
Rem    ajdsouza    05/12/04 - Created
Rem

CREATE OR REPLACE PACKAGE BODY STORAGE_ECM_PKG AS

MESSAGE_TYPE_ERROR          CONSTANT mgmt_storage_report_issues.type%TYPE := MGMT_GLOBAL.G_ERROR;
MESSAGE_TYPE_WARNING        CONSTANT mgmt_storage_report_issues.type%TYPE := MGMT_GLOBAL.G_WARN;

DISK_STORAGE_LAYER          CONSTANT mgmt_storage_report_data.storage_layer%TYPE := 'OS_DISK';
LOCALFS_STORAGE_LAYER       CONSTANT mgmt_storage_report_data.storage_layer%TYPE := 'LOCAL_FILESYSTEM';
NFS_STORAGE_LAYER           CONSTANT mgmt_storage_report_data.storage_layer%TYPE := 'NFS';
DISK_ENTITY_TYPE            CONSTANT mgmt_storage_report_data.entity_type%TYPE := 'DISK';
DISK_PARTITION_ENTITY_TYPE  CONSTANT mgmt_storage_report_data.entity_type%TYPE := 'DISK PARTITION';
MOUNTPOINT_ENTITY_TYPE      CONSTANT mgmt_storage_report_data.entity_type%TYPE := 'Mountpoint';
FILESYSTEM_ENTITY_TYPE      CONSTANT mgmt_storage_report_data.entity_type%TYPE := 'Filesystem';

-- maximum length of the message param and messgae field in mgmt_storage_report_issues
MESSAGE_PARAM_LENGTH        CONSTANT INTEGER := 512;
MESSAGE_LENGTH              CONSTANT INTEGER := 1000;
e_abort_processing          EXCEPTION;
e_skip_to_next_db           EXCEPTION;
bulk_errors                 EXCEPTION;
PRAGMA                      EXCEPTION_INIT(bulk_errors, -24381);

TYPE rowid_list is TABLE OF ROWID INDEX BY PLS_INTEGER;
TYPE raw_list is TABLE OF RAW(20) INDEX BY PLS_INTEGER;
TYPE num_list IS TABLE OF NUMBER INDEX BY PLS_INTEGER;

-- Repository Post Processing NLS Message  IDs
ERROR_REP_FETCH_NFS_DATA                   CONSTANT   INTEGER(2) := 1;
ERROR_REP_PROC_NFS_FS                      CONSTANT   INTEGER(2) := 2;
ERROR_REP_PROC_NFS                         CONSTANT   INTEGER(2) := 3;
ERROR_REP_PROC_DB_ENTITY                   CONSTANT   INTEGER(2) := 4;
ERROR_REP_PROC_DB                          CONSTANT   INTEGER(2) := 5;
ERROR_REP_GET_DB_TARGETS                   CONSTANT   INTEGER(2) := 6;
ERROR_REP_GET_DB_ENTITY                    CONSTANT   INTEGER(2) := 7;
ERROR_REP_TGT_NOT_MNTRD                    CONSTANT   INTEGER(2) := 8;
ERROR_REP_INST_MAPPING                     CONSTANT   INTEGER(2) := 9;
ERROR_REP_INST_MULT_MATCH                  CONSTANT   INTEGER(2) := 10;
ERROR_REP_DATA_MULT_MATCH                  CONSTANT   INTEGER(2) := 11;
ERROR_REP_DATA_NOT_FOUND                   CONSTANT   INTEGER(2) := 12;
ERROR_REP_DATA_MISSING                     CONSTANT   INTEGER(2) := 13;
ERROR_REP_COLL_NO_METRICS                  CONSTANT   INTEGER(2) := 14;
ERROR_REP_COLL_METR_COL_NULL               CONSTANT   INTEGER(2) := 15;
ERROR_REP_NFS_NO_FILESYSTEM                CONSTANT   INTEGER(2) := 16;
ERROR_REP_UPD_NFS_GID                      CONSTANT   INTEGER(2) := 17;
ERROR_REP_UPD_DB_QFLG                      CONSTANT   INTEGER(2) := 18;
ERROR_REP_NFS_TGT_UNSUP                    CONSTANT   INTEGER(2) := 19;
ERROR_REP_FETCH_FILESYSTEMS                CONSTANT   INTEGER(2) := 20;
ACTION_REP_ORACLE_SUPPORT                  CONSTANT   INTEGER(2) := 21;
ACTION_REP_CONF_VALUE_UNIQUE               CONSTANT   INTEGER(2) := 22;
ACTION_REP_COLL_NFS_HOST                   CONSTANT   INTEGER(2) := 23;
ACTION_REP_COLL_NETAP                      CONSTANT   INTEGER(2) := 24;
ACTION_REP_COLL_DATABASE                   CONSTANT   INTEGER(2) := 25;
ACTION_REP_NFS_SVR_TGT_NMNTRD              CONSTANT   INTEGER(2) := 26;

c_message_id_array             CONSTANT   MGMT_MAX_STRING_ARRAY := MGMT_MAX_STRING_ARRAY
  (
   'ERROR_REP_FETCH_NFS_DATA',
   'ERROR_REP_PROC_NFS_FS',
   'ERROR_REP_PROC_NFS',
   'ERROR_REP_PROC_DB_ENTITY',
   'ERROR_REP_PROC_DB',
   'ERROR_REP_GET_DB_TARGETS',
   'ERROR_REP_GET_DB_ENTITY',
   'ERROR_REP_TGT_NOT_MNTRD',
   'ERROR_REP_INST_MAPPING',
   'ERROR_REP_INST_MULT_MATCH',
   'ERROR_REP_DATA_MULT_MATCH',
   'ERROR_REP_DATA_NOT_FOUND',
   'ERROR_REP_DATA_MISSING',
   'ERROR_REP_COLL_NO_METRICS',
   'ERROR_REP_COLL_METR_COL_NULL',
   'ERROR_REP_NFS_NO_FILESYSTEM',
   'ERROR_REP_UPD_NFS_GID',
   'ERROR_REP_UPD_DB_QFLG',
   'ERROR_REP_NFS_TGT_UNSUP',
   'ERROR_REP_FETCH_FILESYSTEMS',
   'ACTION_REP_ORACLE_SUPPORT',
   'ACTION_REP_CONF_VALUE_UNIQUE',
   'ACTION_REP_COLL_NFS_HOST',
   'ACTION_REP_COLL_NETAP',
   'ACTION_REP_COLL_DATABASE',
   'ACTION_REP_NFS_SVR_TGT_NMNTRD'
  );


  c_metric_netapp_filesystems CONSTANT mgmt_metrics.metric_name%TYPE   := 'Volumes';
  c_metric_netapp_netinterf   CONSTANT mgmt_metrics.metric_name%TYPE   := 'NetworkInterfaces';
  c_metric_netapp_qtree       CONSTANT mgmt_metrics.metric_name%TYPE   := 'Qtrees';
  
  c_netapp_mac_address       CONSTANT mgmt_metrics.metric_column%TYPE := 'PhysAddress';
  c_netapp_ip_address        CONSTANT mgmt_metrics.metric_column%TYPE := 'IpAddress';
  c_netapp_qtree_path        CONSTANT mgmt_metrics.metric_column%TYPE := 'PathName';
  -- metric_column volume is the key_value, choose VolumeTotalGb to get the key_value
  c_netapp_volume            CONSTANT mgmt_metrics.metric_column%TYPE := 'VolumeTotalGb';
    
  l_netapp_mac_guid          MGMT_METRICS.METRIC_GUID%TYPE := NULL;
  l_netapp_ip_guid           MGMT_METRICS.METRIC_GUID%TYPE := NULL;
  l_netapp_qtree_path_guid   MGMT_METRICS.METRIC_GUID%TYPE := NULL;
  l_netapp_volume_guid       MGMT_METRICS.METRIC_GUID%TYPE := NULL;

------------------------------------------------------------
-- PROCEDURE LGDB
------------------------------------------------------------
PROCEDURE LGDB ( v_gid  RAW, v_msg  IN VARCHAR2 ) 
IS
BEGIN

  DBMS_OUTPUT.PUT_LINE(v_msg);
  -- tmp_write_file(v_gid,v_msg);

END LGDB;

------------------------------------------------------------
-- PROCEDURE LOG_ISSUE
------------------------------------------------------------
PROCEDURE LOG_ISSUE 
( 
  v_snapshot_guid mgmt_ecm_gen_snapshot.snapshot_guid%TYPE,
  v_message_id    INTEGER,
  v_msg_params    MGMT_MAX_STRING_ARRAY,
  v_action_id     INTEGER,
  v_action_params MGMT_MAX_STRING_ARRAY
) IS

l_type              mgmt_storage_report_issues.type%TYPE := STORAGE_ECM_PKG.MESSAGE_TYPE_ERROR;
l_count             mgmt_storage_report_issues.message_counter%TYPE;
l_message_id        mgmt_storage_report_issues.message_id%TYPE;
l_message_params    mgmt_storage_report_issues.message_params%TYPE;
l_action_id         mgmt_storage_report_issues.action_id%TYPE;
l_action_params     mgmt_storage_report_issues.action_params%TYPE;
l_dummy             INTEGER(1);
l_errmsg            mgmt_system_error_log.ERROR_MSG%TYPE;
l_params            mgmt_storage_report_issues.action_params%TYPE;
l_no_of_params      INTEGER;

BEGIN

  -- Initialize message based on message_id
  -- Log an error is nls ids are not in the list of messages
  BEGIN

    l_message_id := c_message_id_array(v_message_id);

  EXCEPTION

   WHEN OTHERS THEN

    l_message_id := TO_CHAR(v_message_id);

    l_action_id := TO_CHAR(v_action_id);

    RAISE;

  END;

  BEGIN

    l_action_id  := c_message_id_array(v_action_id);

  EXCEPTION

   WHEN OTHERS THEN

    l_action_id := TO_CHAR(v_action_id);

    RAISE;

  END;

  -- Add buffer overflow checks to these lines after demo
  IF v_msg_params IS NOT NULL AND v_msg_params.EXISTS(1) THEN

    l_no_of_params := v_msg_params.COUNT;

    FOR i IN v_msg_params.FIRST..v_msg_params.LAST LOOP 

      -- replace , with - as , is a param seperator
      l_params := SUBSTR(REPLACE(v_msg_params(i),',','-'),1,(STORAGE_ECM_PKG.MESSAGE_PARAM_LENGTH/l_no_of_params));

      IF i = 1 THEN

        l_message_params := l_params;

      ELSE

        l_message_params := SUBSTR(l_message_params||','||l_params,1,STORAGE_ECM_PKG.MESSAGE_PARAM_LENGTH);

      END IF;

    END LOOP;

  END IF;

  IF v_action_params IS NOT NULL AND v_action_params.EXISTS(1) THEN

    l_no_of_params := v_action_params.COUNT;

    FOR i IN v_action_params.FIRST..v_action_params.LAST LOOP 

      l_params := SUBSTR(REPLACE(v_action_params(i),',','-'),1,(STORAGE_ECM_PKG.MESSAGE_PARAM_LENGTH/l_no_of_params));

      IF i = 1 THEN

        l_action_params := l_params;

      ELSE

        l_action_params := SUBSTR(l_action_params||','||l_params,1,STORAGE_ECM_PKG.MESSAGE_PARAM_LENGTH);

      END IF;

    END LOOP;

  END IF;


  -- Check if the message already exists , then dont log it
  BEGIN

    SELECT 1
    INTO   l_dummy
    FROM   dual
    WHERE  EXISTS 
    (
       SELECT 1
       FROM   mgmt_storage_report_issues
       WHERE  ecm_snapshot_id = v_snapshot_guid
       AND    message_id = l_message_id
       AND    message_params = NVL(l_message_params,message_params)
    );

  EXCEPTION

    WHEN NO_DATA_FOUND THEN

     SELECT COUNT(*)
     INTO   l_count
     FROM   mgmt_storage_report_issues
     WHERE  ecm_snapshot_id = v_snapshot_guid;
     
     l_count := l_count + 1;

     INSERT INTO mgmt_storage_report_issues
     (
       ecm_snapshot_id,
       type,
       message_counter,
       message_id,
       message_params,
       action_id,
       action_params
     )
     VALUES
     (
       v_snapshot_guid,
       l_type,
       l_count,
       l_message_id,
       l_message_params,
       l_action_id,
       l_action_params
     );

  END;

  LGDB(NULL,'Logging Issue '||l_message_id||' '||l_message_params);
 
EXCEPTION

 WHEN OTHERS THEN

  -- Parse the error message
  l_errmsg := SUBSTR('Error logging message '||l_message_id||','||l_action_id
    ||' to issue table , for snapshot '||v_snapshot_guid 
      ||SUBSTR(SQLERRM,NVL(INSTRB(SQLERRM,':'),0)+1),
         1,2048);

  -- Log an error to the em error tables
  MGMT_LOG.LOG_ERROR
  (
   'STORAGE_REPORTING',
   -20001,
   l_errmsg
  );

END LOG_ISSUE;

------------------------------------------------------------
-- FUNCTION DIGEST_GLOBAL_UNIQUE_ID
------------------------------------------------------------
FUNCTION DIGEST_GLOBAL_UNIQUE_ID( l_string VARCHAR2 ) 
  RETURN mgmt_storage_report_data.global_unique_id%TYPE
IS

l_errmsg            mgmt_system_error_log.ERROR_MSG%TYPE;
l_digest            mgmt_long_text.digest%TYPE;

BEGIN

  IF l_string IS NULL THEN

   RETURN NULL;

  END IF;

  -- if the string is already digested , return the saved digest
  BEGIN

    SELECT  a.digest
    INTO    l_digest
    FROM    mgmt_long_text a
    WHERE   a.prefix = SUBSTR(l_string,1,4000) 
    AND     ROWNUM = 1;

  EXCEPTION
   -- create a digest and save it to the mgmt_long_text table
   WHEN NO_DATA_FOUND THEN

      l_digest := DBMS_OBFUSCATION_TOOLKIT.MD5(
             input => UTL_RAW.CAST_TO_RAW(l_string));

     -- if in this window a digest gets created, ignore
     -- insert error from primary key and return digest
     BEGIN

      INSERT INTO mgmt_long_text ( digest, prefix, entire )
       VALUES 
       (
        l_digest,
        SUBSTR(l_string,1,4000),
        SUBSTR(l_string,4001)
       );

     EXCEPTION
      -- value got inserted by another session in this window
      WHEN DUP_VAL_ON_INDEX  THEN
        NULL;

     END;

     LGDB(NULL,'DEBUG:GENERATING GUID FOR '||l_string||'-'||l_digest);

  END;

  RETURN l_digest;

EXCEPTION

 WHEN OTHERS THEN
  -- Parse the error message

  l_errmsg := SUBSTR('Failed to generate Global Unique Id from '||l_string 
      ||SUBSTR(SQLERRM,NVL(INSTRB(SQLERRM,':'),0)+1),
         1,2048);

  RAISE_APPLICATION_ERROR(-20101,l_errmsg);

END DIGEST_GLOBAL_UNIQUE_ID;

------------------------------------------------------------
-- PROCEDURE NFS_POST_PROCESSING 
------------------------------------------------------------
PROCEDURE NFS_POST_PROCESSING 
( 
  v_snapshot_guid mgmt_ecm_gen_snapshot.snapshot_guid%TYPE 
) IS

l_dummy                INTEGER(1);
l_rowid_list           rowid_list;
l_gid_list             raw_list;
l_fs_list              MGMT_STORAGE_STRING_TABLE;
l_host_name            mgmt_targets.target_name%TYPE;
l_errmsg               mgmt_system_error_log.ERROR_MSG%TYPE;
l_target_guid          mgmt_targets.target_guid%TYPE;


BEGIN

  LGDB(v_snapshot_guid,'NFS - In NFS Post Processing '); 
  BEGIN

    SELECT a.target_name,
           a.target_guid
    INTO   l_host_name,
           l_target_guid
    FROM   mgmt_ecm_gen_snapshot a
    WHERE  a.snapshot_guid = v_snapshot_guid;

  EXCEPTION

   WHEN OTHERS THEN

      -- Parse the error message
      l_errmsg := SUBSTR(SQLERRM,NVL(INSTRB(SQLERRM,':'),0)+1,2048);

      STORAGE_ECM_PKG.LOG_ISSUE
       (
         v_snapshot_guid,
         ERROR_REP_DATA_NOT_FOUND,
         MGMT_MAX_STRING_ARRAY
         (
          'Target Name',
          'Snapshot '||v_snapshot_guid
         ),
         ACTION_REP_ORACLE_SUPPORT,NULL
       );

      RETURN;

  END;

  BEGIN
    -- select the same server name if the ip address is identical
    -- this will ensure that duplicate error messages will not be logged
    INSERT INTO mgmt_storage_tmp_nfs_data   
     (
       key_value,
       server_name,
       server_mac_address,
       server_ip_address,
       filesystem
     )
    SELECT            
       key_value,
       FIRST_VALUE(a2) OVER ( PARTITION BY NVL(a5,a2) ORDER BY LENGTH(a2) DESC ),             
       a7,
       a5,
       a8
    FROM     mgmt_storage_report_data
    WHERE    ecm_snapshot_id = v_snapshot_guid
    AND      storage_layer = STORAGE_ECM_PKG.NFS_STORAGE_LAYER
    AND      entity_type = STORAGE_ECM_PKG.FILESYSTEM_ENTITY_TYPE;

  EXCEPTION

    WHEN NO_DATA_FOUND THEN
      -- NO nfs mountpoints for this host
      RETURN;

    WHEN OTHERS THEN

     -- Parse the error message
     l_errmsg := SUBSTR(SQLERRM,NVL(INSTRB(SQLERRM,':'),0)+1,2048);

      STORAGE_ECM_PKG.LOG_ISSUE
      (
        v_snapshot_guid,
        ERROR_REP_FETCH_NFS_DATA,
        MGMT_MAX_STRING_ARRAY
        (
          l_errmsg
        ),
        ACTION_REP_ORACLE_SUPPORT,NULL
      );

      RAISE e_abort_processing;

  END;

  LGDB(v_snapshot_guid,'NFS - Fetched NFS Data '||SQL%ROWCOUNT); 

   -----------------------------------------------------
   --  VALIDATTION OF NFS DATA
   -----------------------------------------------------   
   FOR rec IN 
    (
       -- NFS server name is NULL in the nfs record
       SELECT    a.filesystem
       FROM      mgmt_storage_tmp_nfs_data a
       WHERE     a.server_name IS NULL
    
    )
   LOOP
    
        STORAGE_ECM_PKG.LOG_ISSUE
        (
          v_snapshot_guid,
          ERROR_REP_COLL_NO_METRICS,
          MGMT_MAX_STRING_ARRAY
          (
           'NFS Server Name ',
           'NFS Filesystem '||' '||rec.filesystem
          ),
          ACTION_REP_ORACLE_SUPPORT,NULL
        );

   END LOOP;



   FOR rec IN 
    (
        -- NFS filesystem is NULL in the nfs record
       SELECT    a.server_name
       FROM      mgmt_storage_tmp_nfs_data a
       WHERE     a.filesystem IS NULL
    
    )
   LOOP
    
        STORAGE_ECM_PKG.LOG_ISSUE
        (
          v_snapshot_guid,
          ERROR_REP_COLL_NO_METRICS,
          MGMT_MAX_STRING_ARRAY
          (
           'NFS Filesystem ',
           'NFS Filesystem '||' '||rec.server_name
          ),
          ACTION_REP_ORACLE_SUPPORT,NULL
        );

   END LOOP;


   FOR rec IN 
    (
       -- NFS server mac address, and ip address are NULL in the nfs record
       SELECT    a.server_name,
                 a.filesystem
       FROM      mgmt_storage_tmp_nfs_data a
       WHERE     a.server_mac_address IS NULL
       AND       a.server_ip_address IS NULL
    
    )
   LOOP
    
        STORAGE_ECM_PKG.LOG_ISSUE
        (
          v_snapshot_guid,
          ERROR_REP_COLL_METR_COL_NULL,
          MGMT_MAX_STRING_ARRAY
          (
           'MAC Address , IP Address',
           'NFS server '||rec.server_name,
           'NFS Filesystem '||' '||rec.filesystem
          ),
          ACTION_REP_ORACLE_SUPPORT,NULL
        );

   END LOOP;

  LGDB(v_snapshot_guid,'NFS - Validated the NFS Data '); 

   ----------------------------------------------------------
   -- DO NOT PROCESS NFS FILESYSTEMS WHICH DO NOT HAVE ALL
   -- REQUIRED DATA
   ----------------------------------------------------------
   DELETE FROM mgmt_storage_tmp_nfs_data
   WHERE      filesystem IS NULL
   OR         server_name IS NULL 
   OR     
   (
             server_ip_address IS NULL
             AND  server_mac_address IS NULL             
   );

     ----------------------------------------------------------
     -- IDENTIFY THE NFS SERVER TARGET IN EM
     ----------------------------------------------------------

     ----------------------------------------------------------
     -- Host NFS SERVER based on MAC address
     ----------------------------------------------------------
     BEGIN

     -- update the target_guid from mac address of nf  server host       
      UPDATE mgmt_storage_tmp_nfs_data a
      SET    (
              target_guid,
              target_name, 
              target_type 
             ) = 
             (
              SELECT c.target_guid,
                     c.target_name,
                     c.target_type  
              FROM   mgmt$hw_nic b,
                     mgmt$target c
              WHERE  b.mac_address = a.server_mac_address
              AND    b.host_name = c.target_name
              AND    c.target_type = STORAGE_ECM_PKG.c_host_target_type
              AND    ROWNUM = 1
             )
      WHERE a.server_mac_address IS NOT NULL;

     EXCEPTION

      WHEN OTHERS THEN
       -- Parse the error message
       l_errmsg := SUBSTR(SQLERRM,NVL(INSTRB(SQLERRM,':'),0)+1,2048);

       STORAGE_ECM_PKG.LOG_ISSUE
       ( 
          v_snapshot_guid,
          ERROR_REP_PROC_NFS_FS,
          MGMT_MAX_STRING_ARRAY
          (
            'Host NFS Server on MAC Address '||l_errmsg
          ),
          ACTION_REP_ORACLE_SUPPORT,NULL
       );
 
     END;
  LGDB(v_snapshot_guid,'NFS - Identified the nfs server with match to host mac address '||SQL%ROWCOUNT); 


     -- Found target_guids for all nfs filesystems mounted on the
     -- host
     BEGIN

      SELECT 1
      INTO   l_dummy
      FROM   mgmt_storage_tmp_nfs_data a
      WHERE  a.target_guid IS NULL
      AND    ROWNUM = 1;

     EXCEPTION

      WHEN NO_DATA_FOUND THEN

       GOTO fetch_nfs_exports;

     END;

     ----------------------------------------------------------
     -- NTAP NFS server on MAC address    
     ----------------------------------------------------------
     BEGIN

     -- update the target_guid from mac address of netapp nfs serverr host       
      UPDATE mgmt_storage_tmp_nfs_data a
      SET    (
              target_guid,
              target_name, 
              target_type 
             ) = 
            (
             SELECT c.target_guid,
                    c.target_name,
                    c.target_type  
             FROM   mgmt_current_metrics b,
                    mgmt$target c         
             WHERE  b.metric_guid  = STORAGE_ECM_PKG.l_netapp_mac_guid
             AND    b.string_value = a.server_mac_address
             AND    b.target_guid  = c.target_guid
             AND    c.target_type  = STORAGE_ECM_PKG.c_netapp_filer_target_type
             AND    ROWNUM = 1
            )
      WHERE a.server_mac_address IS NOT NULL
      AND   a.target_guid IS NULL;   

     EXCEPTION

      WHEN OTHERS THEN

      -- Parse the error message
       l_errmsg := SUBSTR(SQLERRM,NVL(INSTRB(SQLERRM,':'),0)+1,2048);

       STORAGE_ECM_PKG.LOG_ISSUE
       ( 
          v_snapshot_guid,
          ERROR_REP_PROC_NFS_FS,
          MGMT_MAX_STRING_ARRAY
          (
            'Netapp NFS Server on MAC Address '||l_errmsg
          ),
          ACTION_REP_ORACLE_SUPPORT,NULL
       );

     END;
  LGDB(v_snapshot_guid,'NFS - Identified the netapp nfs server with match for ntap mac address '||SQL%ROWCOUNT); 

     -- Found target_guids for all nfs filesystems mounted on the
     -- host
     BEGIN

      SELECT 1
      INTO   l_dummy
      FROM   mgmt_storage_tmp_nfs_data a
      WHERE  a.target_guid IS NULL
      AND    ROWNUM = 1;

     EXCEPTION

      WHEN NO_DATA_FOUND THEN

       GOTO fetch_nfs_exports;

     END;
          

     ----------------------------------------------------------       
     -- Host NFS Server on IP Address  
     ----------------------------------------------------------     
     BEGIN

      -- update the target_guid from ip address of nf  server host    
      -- using the data from config metrics   
      UPDATE mgmt_storage_tmp_nfs_data a
      SET    (
              target_guid,
              target_name, 
              target_type 
             ) = 
             (
              SELECT c.target_guid,
                     c.target_name,
                     c.target_type  
              FROM   mgmt$hw_nic b,
                     mgmt$target c
              WHERE  b.inet_address = a.server_ip_address
              AND    b.host_name = c.target_name
              AND    c.target_type = STORAGE_ECM_PKG.c_host_target_type
              AND    ROWNUM = 1
             )
      WHERE a.server_ip_address IS NOT NULL
      AND   a.target_guid IS NULL;   

    EXCEPTION

      WHEN OTHERS THEN

       -- Parse the error message
       l_errmsg := SUBSTR(SQLERRM,NVL(INSTRB(SQLERRM,':'),0)+1,2048);

       STORAGE_ECM_PKG.LOG_ISSUE
       ( 
          v_snapshot_guid,
          ERROR_REP_PROC_NFS_FS,
          MGMT_MAX_STRING_ARRAY
          (
            'Host NFS Server on IP Address using nic table '||l_errmsg
          ),
          ACTION_REP_ORACLE_SUPPORT,NULL
       );      
       
     END;
  LGDB(v_snapshot_guid,'NFS - Identified the host nfs server with match for host ip address - from hw_nic table '||SQL%ROWCOUNT); 

     -- Found target_guids for all nfs filesystems mounted on the
     -- host
     BEGIN

      SELECT 1
      INTO   l_dummy
      FROM   mgmt_storage_tmp_nfs_data a
      WHERE  a.target_guid IS NULL
      AND    ROWNUM = 1;

     EXCEPTION

      WHEN NO_DATA_FOUND THEN

       GOTO fetch_nfs_exports;

     END;
 
     ----------------------------------------------------------            
     -- NTAP NFS Server on IP Address
     ----------------------------------------------------------      
     BEGIN

     -- update the target_guid from mac address of netapp nfs server host       
      UPDATE mgmt_storage_tmp_nfs_data a
      SET    (
              target_guid,
              target_name, 
              target_type 
             ) = 
            (
             SELECT c.target_guid,
                    c.target_name,
                    c.target_type  
             FROM   mgmt_current_metrics b,
                    mgmt$target c         
             WHERE  b.metric_guid  = STORAGE_ECM_PKG.l_netapp_ip_guid
             AND    b.string_value = a.server_ip_address
             AND    b.target_guid  = c.target_guid
             AND    c.target_type  = STORAGE_ECM_PKG.c_netapp_filer_target_type
             AND    ROWNUM = 1
            )
      WHERE a.server_ip_address IS NOT NULL
      AND   a.target_guid IS NULL;   

     EXCEPTION

      WHEN OTHERS THEN

       -- Parse the error message
       l_errmsg := SUBSTR(SQLERRM,NVL(INSTRB(SQLERRM,':'),0)+1,2048);

       STORAGE_ECM_PKG.LOG_ISSUE
       ( 
          v_snapshot_guid,
          ERROR_REP_PROC_NFS_FS,
          MGMT_MAX_STRING_ARRAY
          (
            'Netapp NFS Server on IP Address '||l_errmsg
          ),
          ACTION_REP_ORACLE_SUPPORT,NULL
       );
                 
     END;
  LGDB(v_snapshot_guid,'NFS - Identified the netapp nfs server with match for netapp ip address '||SQL%ROWCOUNT); 


     -------------------------------------------------------------
     -- VALIDATIONS FOR THE TARGE_GUID OF THE NFS SERVER TARGET
     -------------------------------------------------------------

     -- Log Issues for NFS servers with no EM targets
     FOR rec IN 
     (
       -- guid, name or type cannot be null for the target
       SELECT server_name,
              target_guid,
              target_name,
              target_type
       FROM   mgmt_storage_tmp_nfs_data a
       WHERE  
       (
              a.target_guid IS NULL
         OR   a.target_name IS NULL
         OR   a.target_type IS NULL
       )
     )
     LOOP
     
        -- NFS Server is not a target in EM
        IF  rec.target_guid IS NULL  
             AND rec.target_name IS NULL 
              AND rec.target_type IS NULL THEN

          -- Give an different error message if grid control or otherwise
          STORAGE_ECM_PKG.LOG_ISSUE
          ( 
            v_snapshot_guid,
            ERROR_REP_TGT_NOT_MNTRD,
            MGMT_MAX_STRING_ARRAY
            (
             rec.server_name
            ),
            ACTION_REP_NFS_SVR_TGT_NMNTRD,
            MGMT_MAX_STRING_ARRAY
            (
             rec.server_name,
             l_host_name
            )
          );  

        ELSE
        -- DATA inconsistency

         -- Give an different error message if grid control or otherwise
         STORAGE_ECM_PKG.LOG_ISSUE
         ( 
           v_snapshot_guid,
           ERROR_REP_DATA_MISSING,
           MGMT_MAX_STRING_ARRAY
           (
            'Target Properties',
            rec.server_name
           ),
           ACTION_REP_ORACLE_SUPPORT,NULL
         );

        END IF;      
    
   END LOOP;
  LGDB(v_snapshot_guid,'NFS - Completed validation of identified nfs servers '||SQL%ROWCOUNT); 

   -------------------------------------------------------------
   -- FTECH THE NFS EXPORTS from the identified NFS SERVERS
   --  that match the filesystem
   -------------------------------------------------------------
   <<fetch_nfs_exports>>

   ------------------------------------------------
   -- EXACT match from exported filesystems from
   -- host based NFS servers
   ------------------------------------------------
   BEGIN

     UPDATE  mgmt_storage_tmp_nfs_data a
     SET     (
              exported_fs,
              global_unique_id
             ) =
      (
        SELECT    b.local_fs_mountpoint,
                  b.global_unique_id
        FROM      mgmt_v_storage_report_data b,
                  mgmt$ecm_current_snapshots snap
        WHERE     b.ecm_snapshot_id = snap.ecm_snapshot_id
        AND       b.target_name = a.target_name
        AND       b.target_type = a.target_type
        AND       b.local_fs_mountpoint = a.filesystem
        AND       b.storage_layer = STORAGE_ECM_PKG.LOCALFS_STORAGE_LAYER
        AND       b.entity_type =  STORAGE_ECM_PKG.MOUNTPOINT_ENTITY_TYPE
        AND       ROWNUM = 1      
      )
      WHERE   target_guid IS NOT NULL
      AND    target_type = STORAGE_ECM_PKG.c_host_target_type;

   EXCEPTION

     WHEN OTHERS THEN

       -- Parse the error message
       l_errmsg := SUBSTR(SQLERRM,NVL(INSTRB(SQLERRM,':'),0)+1,2048);

       STORAGE_ECM_PKG.LOG_ISSUE
       ( 
          v_snapshot_guid,
          ERROR_REP_PROC_NFS_FS,
          MGMT_MAX_STRING_ARRAY
          (
            'Fetching exact NFS Export from NFS Servers of type host '||l_errmsg
          ),
          ACTION_REP_ORACLE_SUPPORT,NULL
       );

   END;
  LGDB(v_snapshot_guid,'NFS - Found the exact matching filesystem exported by host nfs server '||SQL%ROWCOUNT); 



     -- Found exported filesystems for all nfs filesystems mounted on the
     -- host
     BEGIN

      SELECT 1
      INTO   l_dummy
      FROM   mgmt_storage_tmp_nfs_data a
      WHERE  a.target_guid IS NOT NULL
      AND    a.exported_fs IS NULL
      AND    ROWNUM = 1;

     EXCEPTION

      WHEN NO_DATA_FOUND THEN

       GOTO validate_exported_fs;

     END;


   ------------------------------------------------
   -- EXACT match from vols /qtrees from
   -- NETAPP based NFS servers
   ------------------------------------------------
   BEGIN

     -- QTREES Exported by NETAPP
     UPDATE  mgmt_storage_tmp_nfs_data a
     SET     exported_fs =
      (
        SELECT    b.string_value
        FROM      mgmt_current_metrics b   
        WHERE     b.target_guid = a.target_guid
        AND       b.metric_guid = STORAGE_ECM_PKG.l_netapp_qtree_path_guid 
        AND       a.filesystem = 
          -- For netapp
          -- Trim the dir seperator '/' if its added at the end of the exported filesystem
            DECODE
            (
             LENGTH(b.string_value),1,b.string_value,
             DECODE
             (
              SUBSTR(b.string_value,LENGTH(b.string_value)),'/',
              SUBSTR(b.string_value,1,LENGTH(b.string_value)-1),b.string_value
             )
            )   
         AND       ROWNUM = 1   
       )
      WHERE  a.target_type = STORAGE_ECM_PKG.c_netapp_filer_target_type 
      AND    a.target_guid IS NOT NULL
      AND    a.exported_fs IS NULL;

   EXCEPTION

     WHEN OTHERS THEN

       -- Parse the error message
       l_errmsg := SUBSTR(SQLERRM,NVL(INSTRB(SQLERRM,':'),0)+1,2048);

       STORAGE_ECM_PKG.LOG_ISSUE
       ( 
          v_snapshot_guid,
          ERROR_REP_PROC_NFS_FS,
          MGMT_MAX_STRING_ARRAY
          (
            'Fetching exact exported QTREE from Netap NFS servers '||l_errmsg
          ),
          ACTION_REP_ORACLE_SUPPORT,NULL
       );

   END;
  LGDB(v_snapshot_guid,'NFS - Found the exact matching qtree exported by netapp nfs server '||SQL%ROWCOUNT); 


     -- Found exported filesystems for all nfs filesystems mounted on the
     -- host
     BEGIN

      SELECT 1
      INTO   l_dummy
      FROM   mgmt_storage_tmp_nfs_data a
      WHERE  a.target_guid IS NOT NULL
      AND    a.exported_fs IS NULL
      AND    ROWNUM = 1;

     EXCEPTION

      WHEN NO_DATA_FOUND THEN

       GOTO validate_exported_fs;

     END;



   BEGIN

     -- VOLUMES Exported by NETAPP
     UPDATE  mgmt_storage_tmp_nfs_data a
     SET     exported_fs =
      (
        SELECT    b.key_value
        FROM      mgmt_current_metrics b   
        WHERE     b.target_guid = a.target_guid
        AND       b.metric_guid = STORAGE_ECM_PKG.l_netapp_volume_guid
        AND       a.filesystem = 
          -- For netapp
          -- Trim the dir seperator '/' if its added at the end of the exported filesystem
            DECODE
            (
             LENGTH(b.key_value),1,b.key_value,
             DECODE
             (
              SUBSTR(b.key_value,LENGTH(b.key_value)),'/',
              SUBSTR(b.key_value,1,LENGTH(b.key_value)-1),b.key_value
             )
            )     
         AND       ROWNUM = 1
       )
      WHERE  a.target_type = STORAGE_ECM_PKG.c_netapp_filer_target_type 
      AND    a.target_guid IS NOT NULL
      AND    a.exported_fs IS NULL;

   EXCEPTION

    WHEN OTHERS THEN

       -- Parse the error message
       l_errmsg := SUBSTR(SQLERRM,NVL(INSTRB(SQLERRM,':'),0)+1,2048);

       STORAGE_ECM_PKG.LOG_ISSUE
       ( 
          v_snapshot_guid,
          ERROR_REP_PROC_NFS_FS,
          MGMT_MAX_STRING_ARRAY
          (
            'Fetching exact exported Volume from Netapp NFS servers'||l_errmsg
          ),
          ACTION_REP_ORACLE_SUPPORT,NULL
       );

   END;
  LGDB(v_snapshot_guid,'NFS - Found the exact matching volume exported by netapp nfs server '||SQL%ROWCOUNT); 


     -- Found exported filesystems for all nfs filesystems mounted on the
     -- host
     BEGIN

      SELECT 1
      INTO   l_dummy
      FROM   mgmt_storage_tmp_nfs_data a
      WHERE  a.target_guid IS NOT NULL
      AND    a.exported_fs IS NULL
      AND    ROWNUM = 1;

     EXCEPTION

      WHEN NO_DATA_FOUND THEN

       GOTO validate_exported_fs;

     END;


   ------------------------------------------------
   -- CLOSEST match to exported filesystems from
   -- host based NFS servers
   ------------------------------------------------
   -- Fetch the filesystem with MAX LENGTH
   BEGIN

     UPDATE  mgmt_storage_tmp_nfs_data a
     SET     ( 
               exported_fs, 
               global_unique_id 
             ) =
      (
        SELECT    DISTINCT
                  FIRST_VALUE(b.local_fs_mountpoint) OVER (ORDER BY LENGTH(b.local_fs_mountpoint) DESC ) AS local_fs_mountpoint,
                  FIRST_VALUE(b.global_unique_id) OVER (ORDER BY LENGTH(b.local_fs_mountpoint) DESC ) AS global_unique_id
        FROM      mgmt_v_storage_report_data b,
                  mgmt$ecm_current_snapshots snap
        WHERE     b.ecm_snapshot_id = snap.ecm_snapshot_id
        AND       b.target_name = a.target_name
        AND       b.target_type = a.target_type
        AND       a.filesystem LIKE b.local_fs_mountpoint||'%'
        AND       b.storage_layer = STORAGE_ECM_PKG.LOCALFS_STORAGE_LAYER
        AND       b.entity_type =  STORAGE_ECM_PKG.MOUNTPOINT_ENTITY_TYPE    
      )
      WHERE  target_guid IS NOT NULL
      AND    exported_fs IS NULL
      AND    target_type = STORAGE_ECM_PKG.c_host_target_type;

   EXCEPTION

    WHEN OTHERS THEN

       -- Parse the error message
       l_errmsg := SUBSTR(SQLERRM,NVL(INSTRB(SQLERRM,':'),0)+1,2048);

       STORAGE_ECM_PKG.LOG_ISSUE
       ( 
          v_snapshot_guid,
          ERROR_REP_PROC_NFS_FS,
          MGMT_MAX_STRING_ARRAY
          (
            'Fetching closest matching NFS Export from NFS Server of type host '||l_errmsg
          ),
          ACTION_REP_ORACLE_SUPPORT,NULL
       );

   END;
  LGDB(v_snapshot_guid,'NFS - Found the closest matching filesystem exported by host nfs server '||SQL%ROWCOUNT); 



     -- Found exported filesystems for all nfs filesystems mounted on the
     -- host
     BEGIN

      SELECT 1
      INTO   l_dummy
      FROM   mgmt_storage_tmp_nfs_data a
      WHERE  a.target_guid IS NOT NULL
      AND    a.exported_fs IS NULL
      AND    ROWNUM = 1;

     EXCEPTION

      WHEN NO_DATA_FOUND THEN

       GOTO validate_exported_fs;

     END;



   ------------------------------------------------
   -- CLOSEST match from vols /qtrees from
   -- NETAPP based NFS servers
   ------------------------------------------------
   -- TBD , Fetch the Qtree with MAX LENGTH
   BEGIN

     -- QTREES Exported by NETAPP
     UPDATE  mgmt_storage_tmp_nfs_data a
     SET     exported_fs =
      (
        SELECT    b.string_value
        FROM      mgmt_current_metrics b   
        WHERE     b.target_guid = a.target_guid
        AND       b.metric_guid = STORAGE_ECM_PKG.l_netapp_qtree_path_guid 
        -- this is a like match, the trailing / will not matter here
        AND       a.filesystem LIKE b.string_value||'%'
        AND      ROWNUM = 1
       )
      WHERE  a.target_type = STORAGE_ECM_PKG.c_netapp_filer_target_type 
      AND    a.target_guid IS NOT NULL
      AND    a.exported_fs IS NULL;

   EXCEPTION

    WHEN OTHERS THEN

       -- Parse the error message
       l_errmsg := SUBSTR(SQLERRM,NVL(INSTRB(SQLERRM,':'),0)+1,2048);

       STORAGE_ECM_PKG.LOG_ISSUE
       ( 
          v_snapshot_guid,
          ERROR_REP_PROC_NFS_FS,
          MGMT_MAX_STRING_ARRAY
          (
            'Fetching closest matching QTREE from Netapp NFS Servers '||l_errmsg
          ),
          ACTION_REP_ORACLE_SUPPORT,NULL
       );

   END;
  LGDB(v_snapshot_guid,'NFS - Found the closest matching qtree exported by netapp nfs server '||SQL%ROWCOUNT); 



     -- Found exported filesystems for all nfs filesystems mounted on the
     -- host
     BEGIN

      SELECT 1
      INTO   l_dummy
      FROM   mgmt_storage_tmp_nfs_data a
      WHERE  a.target_guid IS NOT NULL
      AND    a.exported_fs IS NULL
      AND    ROWNUM = 1;

     EXCEPTION

      WHEN NO_DATA_FOUND THEN

       GOTO validate_exported_fs;

     END;


  BEGIN
     --  TBD Fetch the volume with the max LENGTH
     -- VOLUMES Exported by NETAPP
     UPDATE  mgmt_storage_tmp_nfs_data a
     SET     exported_fs =
      (
        SELECT    b.key_value
        FROM      mgmt_current_metrics b   
        WHERE     b.target_guid = a.target_guid
        AND       b.metric_guid = STORAGE_ECM_PKG.l_netapp_volume_guid
        AND       a.filesystem LIKE
          -- For netapp
          -- Trim the dir seperator '/' if its added at the end of the exported filesystem
            DECODE
            (
             LENGTH(b.key_value),1,b.key_value,
             DECODE
             (
              SUBSTR(b.key_value,LENGTH(b.key_value)),'/',
              SUBSTR(b.key_value,1,LENGTH(b.key_value)-1),b.key_value
             )
            )||'%'              
        AND      ROWNUM = 1
       )
      WHERE  a.target_type = STORAGE_ECM_PKG.c_netapp_filer_target_type 
      AND    a.target_guid IS NOT NULL
      AND    a.exported_fs IS NULL;

   EXCEPTION

    WHEN OTHERS THEN

       -- Parse the error message
       l_errmsg := SUBSTR(SQLERRM,NVL(INSTRB(SQLERRM,':'),0)+1,2048);

       STORAGE_ECM_PKG.LOG_ISSUE
       ( 
          v_snapshot_guid,
          ERROR_REP_PROC_NFS_FS,
          MGMT_MAX_STRING_ARRAY
          (
            'Fetching closest matching Volume from Netapp NFS Servers '||l_errmsg
          ),
          ACTION_REP_ORACLE_SUPPORT,NULL
       );


   END;


   <<validate_exported_fs>>
      ---------------------------------------------------------
      -- VALIDATIONS FOR IDENTIFIED EXPORTED FILESYSTEMs
      ---------------------------------------------------------

      -- Found the NFS server but not the exported filesystem
      -- Log issues , could not find a exported filesystem
      -- from the nfs server  
      FOR rec IN 
      (
          SELECT   target_type,
                   target_name,
                   filesystem
          FROM     mgmt_storage_tmp_nfs_data a
          WHERE    a.target_guid IS NOT NULL
          AND      a.exported_fs IS NULL
      )
      LOOP

         -- Give an different error message if grid control or otherwise
         IF rec.target_type = STORAGE_ECM_PKG.c_netapp_filer_target_type THEN

          -- For a netapp type nfs server
          STORAGE_ECM_PKG.LOG_ISSUE
          ( 
            v_snapshot_guid,
            ERROR_REP_NFS_NO_FILESYSTEM,
            MGMT_MAX_STRING_ARRAY
            (
             rec.filesystem,
             rec.target_type||' '||rec.target_name
           ),
           ACTION_REP_COLL_NETAP,
           MGMT_MAX_STRING_ARRAY
           (
            rec.target_name
           )
          );

         ELSE

          -- For a host type nfs server
          STORAGE_ECM_PKG.LOG_ISSUE
          ( 
            v_snapshot_guid,
            ERROR_REP_NFS_NO_FILESYSTEM,
            MGMT_MAX_STRING_ARRAY
            (
              rec.filesystem,
              rec.target_type||' '||rec.target_name
            ),
            ACTION_REP_COLL_NFS_HOST,
            MGMT_MAX_STRING_ARRAY
            (
             rec.target_name,
             l_host_name            
            )
          );

         END IF;

      END LOOP;
  LGDB(v_snapshot_guid,'NFS - Comlpeted validatons for matchign filesystems '||SQL%ROWCOUNT); 


     ---------------------------------------------------------
     -- GENERATE GUID
     ---------------------------------------------------------
 
     -- If the NFS server target type is type netap, NFS server global unique id is 
     -- netapp target_id:exported filesystem 
     BEGIN

       UPDATE   mgmt_storage_tmp_nfs_data a
       SET      a.global_unique_id = 
                 STORAGE_ECM_PKG.DIGEST_GLOBAL_UNIQUE_ID
                 ( 'EM_'||a.target_name||'_'||a.target_type||':'||a.exported_fs )
       WHERE    a.target_type = STORAGE_ECM_PKG.c_netapp_filer_target_type
       AND      a.exported_fs IS NOT NULL
       AND      a.target_guid IS NOT NULL;
                     

    EXCEPTION
 
     WHEN OTHERS THEN

       -- Parse the error message
       l_errmsg := SUBSTR(SQLERRM,NVL(INSTRB(SQLERRM,':'),0)+1,2048);

       STORAGE_ECM_PKG.LOG_ISSUE
       ( 
          v_snapshot_guid,
          ERROR_REP_PROC_NFS_FS,
          MGMT_MAX_STRING_ARRAY
          (
            'Error updating Global_unique_id for Netapp Servers '||l_errmsg
          ),
          ACTION_REP_ORACLE_SUPPORT,NULL
       );

    END;

      ---------------------------------------------------------------------
      -- NOTE : GUID for filesystems exported from host type nfs servers
      --  is updated when getting the exported filesystem
      ---------------------------------------------------------------------

      -- create an improved target_guid over the one instrumented by the script
      -- , with target_guid of the
      -- nfs server and the nfs filesystem as instrumented in the metrics
      BEGIN

          UPDATE   mgmt_storage_tmp_nfs_data a
          SET      global_unique_id =
                    STORAGE_ECM_PKG.DIGEST_GLOBAL_UNIQUE_ID
                    (
                      'EM_'||a.target_name||'-'||a.target_type||':'||
                       NVL(a.exported_fs,a.filesystem)
                    )
          WHERE    a.target_guid IS NOT NULL
          AND      a.global_unique_id IS NULL;

      EXCEPTION
       WHEN OTHERS THEN

       -- Parse the error message
       l_errmsg := SUBSTR(SQLERRM,NVL(INSTRB(SQLERRM,':'),0)+1,2048);

       STORAGE_ECM_PKG.LOG_ISSUE
       ( 
          v_snapshot_guid,
          ERROR_REP_PROC_NFS_FS,
          MGMT_MAX_STRING_ARRAY
          (
            'Error updating enhanced global_unique_id for nfs filesystems '||l_errmsg
          ),
          ACTION_REP_ORACLE_SUPPORT,NULL
       );

      END;      


      -- Failed to create a global_unique_id  , inspite of 
      -- having the nfs server target in em
      -- This is a processing error
      -- Log issue
      -- retain the guid instrumented by the agent scripts
      FOR rec IN 
      (
          SELECT   target_type,
                   target_name,
                   filesystem
          FROM     mgmt_storage_tmp_nfs_data a
          WHERE    a.global_unique_id IS NULL
          AND      a.target_guid IS NOT NULL
      )
      LOOP  

         -- Log an issue and continue
         STORAGE_ECM_PKG.LOG_ISSUE
         (
           v_snapshot_guid,
           ERROR_REP_UPD_NFS_GID,
           MGMT_MAX_STRING_ARRAY
           (
            rec.target_name||' '||rec.target_type||' '||rec.filesystem
           ),
           ACTION_REP_ORACLE_SUPPORT,NULL
         );

      END LOOP;


      -- For NFS filesystems for which a nfs server target cannot be identified
      -- in EM, retain the default global_unique_id instrumented by the agent scripts

  LGDB(v_snapshot_guid,'NFS - Generated global_unique_id for matchign filesystems '||SQL%ROWCOUNT); 
  LGDB(v_snapshot_guid,'NFS - Found the closest matching volume exported by netapp nfs server '||SQL%ROWCOUNT); 
  FOR rec IN ( SELECT server_name, filesystem, target_guid, exported_fs , global_unique_id FROM mgmt_storage_tmp_nfs_data a WHERE a.target_guid IS NOT NULL )
  LOOP
   LGDB(v_snapshot_guid,' NFS - Matches '||rec.server_name||' '||rec.filesystem||' '||rec.target_guid||' '||rec.exported_fs||' '||rec.global_unique_id);
  END LOOP;
      -------------------------------------------------------------
        -- UPDATE GUID
      -------------------------------------------------------------
      BEGIN

        -- Fetch the rowids to be updated
        -- Update the NFS filesystem and the storage entities which are parents 
        -- ( higher in topology on it )                
        SELECT  a.ROWID,                    
                b.global_unique_id,
                b.server_name||':'||b.filesystem
        BULK COLLECT INTO 
                l_rowid_list,
                l_gid_list,
                l_fs_list
        FROM    mgmt_storage_report_data a,
                mgmt_storage_tmp_nfs_data b
        WHERE   a.ecm_snapshot_id = v_snapshot_guid  
        AND     b.global_unique_id IS NOT NULL
        AND     a.key_value IN 
        (
              SELECT key_value
              FROM   mgmt_storage_report_keys
              CONNECT BY
              ecm_snapshot_id = PRIOR ecm_snapshot_id
              AND key_value = PRIOR parent_key_value
              AND key_value != PRIOR key_value              
              START WITH  
              ecm_snapshot_id = v_snapshot_guid    
              AND key_value = b.key_value
        );
      EXCEPTION

       WHEN OTHERS THEN
 
        -- Parse the error message
        l_errmsg := SUBSTR(SQLERRM,NVL(INSTRB(SQLERRM,':'),0)+1,2048);

        STORAGE_ECM_PKG.LOG_ISSUE
        ( 
          v_snapshot_guid,
          ERROR_REP_PROC_NFS_FS,
          MGMT_MAX_STRING_ARRAY
          (
            ' - identifying the rows to update with global_unique_id '||l_errmsg
          ),
          ACTION_REP_ORACLE_SUPPORT,NULL
        );

     END;       
  LGDB(v_snapshot_guid,'NFS - Fetched rows to update for global_unique_id in storage data table '||SQL%ROWCOUNT); 
  
     -- If thre are no global unique ids to update , end nfs post processing
     IF l_gid_list IS NULL OR NOT l_gid_list.EXISTS(1) THEN

      LGDB(v_snapshot_guid,'NFS -There are no global_unique_ids to update for NFS filesystems'); 

      GOTO end_nfs_processing;

     END IF; 


     -- Update NFS global unique ids here
     BEGIN

       FORALL i IN l_gid_list.FIRST..l_gid_list.LAST
         SAVE EXCEPTIONS
         UPDATE mgmt_storage_report_data
         SET    global_unique_id = l_gid_list(i)
         WHERE  ROWID = l_rowid_list(i);

     EXCEPTION 
      WHEN bulk_errors THEN
 
        FOR j in 1..SQL%BULK_EXCEPTIONS.COUNT
        LOOP

         -- Parse the error message         
         l_errmsg := SUBSTR(SQLERRM(-SQL%BULK_EXCEPTIONS(j).error_code),
                      NVL(INSTRB(
                       SQLERRM(-SQL%BULK_EXCEPTIONS(j).error_code),':'),0)+1,2048);

         -- Log an issue and continue
         STORAGE_ECM_PKG.LOG_ISSUE
         (
           v_snapshot_guid,
           ERROR_REP_UPD_NFS_GID,
           MGMT_MAX_STRING_ARRAY
           (
            l_fs_list(SQL%BULK_EXCEPTIONS(j).error_index)||' '||l_errmsg 
           ),
           ACTION_REP_ORACLE_SUPPORT,NULL
         );

        END LOOP;

     END;
  LGDB(v_snapshot_guid,'NFS - Updated rows for global_unique_id in storage data table '||SQL%ROWCOUNT); 

<<end_nfs_processing>>
NULL;

EXCEPTION

  WHEN e_abort_processing THEN
   NULL;

  WHEN OTHERS THEN
    -- Parse the error message
    l_errmsg := SUBSTR(SUBSTR(SQLERRM,NVL(INSTRB(SQLERRM,':'),0)+1), 1,2048);

    -- Log an issue
    STORAGE_ECM_PKG.LOG_ISSUE
    (
      v_snapshot_guid,
      ERROR_REP_PROC_NFS,
      MGMT_MAX_STRING_ARRAY
      (
        l_errmsg 
      ),
      ACTION_REP_ORACLE_SUPPORT,NULL
    );

END NFS_POST_PROCESSING;


------------------------------------------------------------
-- PROCEDURE DB_POST_PROCESSING 
------------------------------------------------------------
PROCEDURE DB_POST_PROCESSING 
( 
  v_snapshot_guid mgmt_ecm_gen_snapshot.snapshot_guid%TYPE 
) IS

l_host_name                 mgmt_targets.target_name%TYPE;

l_dbtg_list                 MGMT_METRIC_INSTANCE_ARRAY;
l_ose_list                  MGMT_STORAGE_STRING_TABLE;
l_osew_list                 MGMT_STORAGE_STRING_TABLE;
l_dbent_list                MGMT_STORAGE_DB_OS_ENT_ARRAY;
l_dbentf_list               MGMT_STORAGE_DB_OS_ENT_ARRAY;
l_dbentmp_list              MGMT_STORAGE_DB_OS_ENT_ARRAY;
l_rowid_list                rowid_list;
l_strgdata_list             MGMT_STORAGE_STRING_TABLE;
l_intdata_list              num_list;
l_idx_list                  num_list;

l_storage_layer             mgmt_storage_report_data.storage_layer%TYPE;
l_entity_type               mgmt_storage_report_data.entity_type%TYPE;
l_sizeb                     mgmt_storage_report_data.sizeb%TYPE;
l_global_unique_id          mgmt_storage_report_data.global_unique_id%TYPE;
l_key_value                 mgmt_storage_report_data.key_value%TYPE;
l_em_query_flag             mgmt_storage_report_data.em_query_flag%TYPE;
l_target_name               mgmt_targets.target_name%TYPE;
l_target_type               mgmt_targets.target_type%TYPE;
l_target_guid               mgmt_targets.target_guid%TYPE;
l_errmsg                    mgmt_system_error_log.ERROR_MSG%TYPE;

BEGIN
  
  -- Get the list of db targets on the host
  -- Get the list of asm targets on the host
  
  -- For dbs get os storage entity for
  -- datafiles
  -- controlfiles
  -- redologs
  
  -- For asm get disk path for disks
  
  -- Get key_value from the mgmt_storage_report_alias table 
  -- where os_storage_entity or ( disk path ) = value
  
  -- If the key_value if for a mountpoint then do nothing
  -- Mark it as _ALLOCATED_TO_DB_ if its marked as _UNALLOCATED_
  
  -- If cannot find a key_value in mgmt_storage_report_alias, log issue 
  -- If the os_storage_entity is null, log an issue
  -- If the database has no metrics log an issue

  -- Get the host name
  LGDB(v_snapshot_guid,'In DB post processing'); 

  BEGIN

    SELECT a.target_name,
           a.target_guid
    INTO   l_host_name, 
           l_target_guid
    FROM   mgmt_ecm_gen_snapshot a
    WHERE  a.snapshot_guid = v_snapshot_guid;

  EXCEPTION

   WHEN OTHERS THEN

      -- Parse the error message
      l_errmsg := SUBSTR(SQLERRM,NVL(INSTRB(SQLERRM,':'),0)+1,2048);

      STORAGE_ECM_PKG.LOG_ISSUE
       (
         v_snapshot_guid,
         ERROR_REP_DATA_NOT_FOUND,
         MGMT_MAX_STRING_ARRAY
         (
          'Target Name',
          'Snapshot '||v_snapshot_guid
         ),
         ACTION_REP_ORACLE_SUPPORT,NULL
       );

      RETURN;

  END;


  -- Get all the db, osm and rac targets for the given host target
  BEGIN 

    l_dbtg_list := STORAGE_COMMON_DB_UTILS.GET_ORACLE_DB_TARGETS (l_host_name);

  EXCEPTION

    WHEN OTHERS THEN
      
      -- Parse the error message
      l_errmsg := SUBSTR(SQLERRM,NVL(INSTRB(SQLERRM,':'),0)+1,2048);

      STORAGE_ECM_PKG.LOG_ISSUE
       (
         v_snapshot_guid,
         ERROR_REP_GET_DB_TARGETS,
         MGMT_MAX_STRING_ARRAY
         (
          l_host_name||' '||l_errmsg
         ),
         ACTION_REP_ORACLE_SUPPORT,NULL
       );

      RETURN;

  END;


  -- No db, rac or osm target
  IF l_dbtg_list IS NULL OR l_dbtg_list.COUNT <= 0 THEN

    RETURN;

  END IF;
    
  LGDB(v_snapshot_guid,'DB - Fetched list of DBS '||l_dbtg_list.COUNT); 
  -- Process each db , rac or osm target
  FOR i IN l_dbtg_list.FIRST..l_dbtg_list.LAST LOOP


   BEGIN
   
    IF l_dbtg_list(i).target_type IS NULL OR l_dbtg_list(i).target_name IS NULL THEN

       STORAGE_ECM_PKG.LOG_ISSUE
       (
         v_snapshot_guid,
         ERROR_REP_DATA_MISSING,
         MGMT_MAX_STRING_ARRAY
         (
          'Database target properties',
          l_dbtg_list(i).target_type||' '||l_dbtg_list(i).target_name
         ),
         ACTION_REP_ORACLE_SUPPORT,NULL
       );

       RAISE e_skip_to_next_db;

    END IF;
  LGDB(v_snapshot_guid,'DB - Processing for db  '||l_dbtg_list(i).target_type||' '||l_dbtg_list(i).target_name); 


    -- Get the list of os entities for the db target
    BEGIN

      l_ose_list := STORAGE_COMMON_DB_UTILS.GET_OS_ENTITIES_FOR_ORCL_TGTS
                    (
                      l_dbtg_list(i).target_name,
                      l_dbtg_list(i).target_type
                    );
    EXCEPTION

     WHEN OTHERS THEN

       -- Parse the error message
       l_errmsg := SUBSTR(SQLERRM,NVL(INSTRB(SQLERRM,':'),0)+1,2048);

       -- Error to get the List of metrics for DB or ASM
       STORAGE_ECM_PKG.LOG_ISSUE
       (
         v_snapshot_guid,
         ERROR_REP_GET_DB_ENTITY,
         MGMT_MAX_STRING_ARRAY
         (
           l_dbtg_list(i).target_type||' '||l_dbtg_list(i).target_name||' '||l_errmsg
         ),
         ACTION_REP_ORACLE_SUPPORT,NULL
       );
       LGDB(v_snapshot_guid,'DB - Error fetching os entities for db '||l_dbtg_list(i).target_type||' '||l_dbtg_list(i).target_name); 

       RAISE e_skip_to_next_db;

    END;
    
    -- NO OS entities log an issue
    IF l_ose_list IS NULL OR l_ose_list.COUNT <= 0 THEN

     STORAGE_ECM_PKG.LOG_ISSUE
     (
       v_snapshot_guid,
       ERROR_REP_COLL_NO_METRICS,
       MGMT_MAX_STRING_ARRAY
       ( 
        'Storage Entities',
        l_dbtg_list(i).target_type||' '||l_dbtg_list(i).target_name
       ),
       ACTION_REP_COLL_DATABASE,
       MGMT_MAX_STRING_ARRAY
       (
         l_dbtg_list(i).target_type||' '||l_dbtg_list(i).target_name,
         l_host_name            
       )
      );
      LGDB(v_snapshot_guid,'DB - Fetched zilch os entities for db '||l_dbtg_list(i).target_type||' '||l_dbtg_list(i).target_name); 
  
     RAISE e_skip_to_next_db;

    END IF;
    LGDB(v_snapshot_guid,'DB - Fetched list of Entities for DB '||l_dbtg_list(i).target_type||' '||l_dbtg_list(i).target_name||' '||l_ose_list.COUNT); 
      

    -- OS entity value is null , log an issue
    FOR j IN l_ose_list.FIRST..l_ose_list.LAST LOOP
  
      -- if the OS entity does not have the os_storage_entity field instrumented 
      IF l_ose_list(j) IS NULL THEN

        STORAGE_ECM_PKG.LOG_ISSUE
        (
          v_snapshot_guid,
          ERROR_REP_COLL_METR_COL_NULL,
          MGMT_MAX_STRING_ARRAY
          (
           'Storage Entity',
           l_dbtg_list(i).target_type||' '||l_dbtg_list(i).target_name,
           NULL
          ),
          ACTION_REP_COLL_DATABASE,
          MGMT_MAX_STRING_ARRAY
          (
            l_dbtg_list(i).target_type||' '||l_dbtg_list(i).target_name,
            l_host_name            
          )
        );

      END IF;
    
    END LOOP;


    -- Storage entities that match the os entities in the db 
      SELECT MGMT_STORAGE_DB_OS_ENTITY_OBJ
             ( 
               ROWNUM,
               ROWIDTOCHAR(b.ROWID),   
               b.key_value,
               b.storage_layer,
               b.entity_type,
               b.global_unique_id,
               b.em_query_flag,
               b.sizeb,
               a.value             
             )
      BULK COLLECT INTO l_dbent_list
      FROM   mgmt_storage_report_alias a,
             mgmt_storage_report_data b,
             TABLE ( CAST ( l_ose_list AS MGMT_STORAGE_STRING_TABLE) ) c
      WHERE  b.ecm_snapshot_id = v_snapshot_guid
      AND    a.ecm_snapshot_id = b.ecm_snapshot_id
      AND    a.key_value = b.key_value
      AND    a.value = VALUE(c)
      AND    VALUE(c) IS NOT NULL;

    LGDB(v_snapshot_guid,'DB - Matched OS entities to OS storage data for '||l_dbtg_list(i).target_type||' '||l_dbtg_list(i).target_name); 

    -- Check for mapping errors, log issue
      SELECT   VALUE(a)
      BULK COLLECT INTO l_osew_list
      FROM     TABLE ( CAST ( l_ose_list AS MGMT_STORAGE_STRING_TABLE) ) a
      WHERE    VALUE(a) IS NOT NULL
      MINUS
      SELECT   b.os_entity
      FROM     TABLE ( CAST ( l_dbent_list AS MGMT_STORAGE_DB_OS_ENT_ARRAY ) ) b;
    
      IF l_osew_list IS NOT NULL AND l_osew_list.EXISTS(1) THEN

       FOR j IN l_osew_list.FIRST..l_osew_list.LAST LOOP

        -- Mapping error
        STORAGE_ECM_PKG.LOG_ISSUE
        (
          v_snapshot_guid,
          ERROR_REP_INST_MAPPING,
          MGMT_MAX_STRING_ARRAY
          (
           l_dbtg_list(i).target_type||' '||l_dbtg_list(i).target_name,
           l_osew_list(j)
          ),
          ACTION_REP_COLL_DATABASE,
          MGMT_MAX_STRING_ARRAY
          (
            l_dbtg_list(i).target_type||' '||l_dbtg_list(i).target_name,
            l_host_name            
          )
        );

       END LOOP;

     END IF;

   -- l_dbent_list is the list of os entities for the db which have a matcing entity
   --  in the storage data for the host
      
    -- Leave out list of os entities which are based on filesystem
    SELECT   VALUE(b)
    BULK COLLECT INTO l_dbentf_list 
    FROM     TABLE ( CAST ( l_dbent_list AS MGMT_STORAGE_DB_OS_ENT_ARRAY ) ) b 
    WHERE    UPPER(b.storage_layer) 
     NOT IN (STORAGE_ECM_PKG.LOCALFS_STORAGE_LAYER,STORAGE_ECM_PKG.NFS_STORAGE_LAYER);

    -- All entities are filesystem based
    IF l_dbentf_list IS NULL OR NOT l_dbentf_list.EXISTS(1) THEN  

      RAISE e_skip_to_next_db;

    END IF;
    LGDB(v_snapshot_guid,'DB - Prepared the OS data for UPDATE '||l_dbtg_list(i).target_type||' '||l_dbtg_list(i).target_name); 


    -- UPDATEs
    BEGIN
         
      -- UPDATE THE CHILD ( lower in topology ) OF THE ENTITY IN USE BY THE DATABASE
      --
      -- if the os entity providing raw storage to the db entity has a em query flag as virtual
      -- update the used space in the entities child as less size of the parent
      -- Virtual entity - ( its a entity thats configured but not in use and its size is not 
      --  factored to its child , it is not used in storage calculations )
      SELECT idx
      BULK COLLECT INTO l_idx_list
      FROM    TABLE ( CAST ( l_dbentf_list AS MGMT_STORAGE_DB_OS_ENT_ARRAY ) ) a 
      WHERE   a.em_query_flag LIKE '%_VIRTUAL_%';

      IF l_idx_list IS NOT NULL AND l_idx_list.EXISTS(1) THEN

       LGDB(v_snapshot_guid,
        'DB - DB uses virtual parents '||l_idx_list.COUNT);

       -- List of entity objects for virtual entity updates
       SELECT    p.idx,
                 c.ROWID,
                 p.sizeb
       BULK COLLECT INTO l_idx_list,
                         l_rowid_list,
                         l_intdata_list
       FROM    mgmt_storage_report_data c,
               TABLE ( CAST ( l_dbentf_list AS MGMT_STORAGE_DB_OS_ENT_ARRAY ) ) p
       WHERE  c.storage_layer = p.storage_layer
       AND    c.ecm_snapshot_id = v_snapshot_guid
       AND    p.em_query_flag LIKE '%_VIRTUAL_%'
       AND    EXISTS 
        -- exists a virtual parent for the child with the global_unique_id
        -- of the entity allocated to the oracle database
              (
                SELECT   1
                FROM     mgmt_storage_report_keys ck,
                         mgmt_storage_report_data pk
                WHERE    ck.ecm_snapshot_id = c.ecm_snapshot_id
                AND      ck.ecm_snapshot_id = pk.ecm_snapshot_id
                AND      ck.key_value = c.key_value
                AND      ck.parent_key_value = pk.key_value
                AND      pk.global_unique_id = p.global_unique_id
                AND      pk.storage_layer = p.storage_layer
                AND      pk.entity_type = p.entity_type 
                AND      ck.key_value != ck.parent_key_value
              );        

       IF l_rowid_list IS NOT NULL AND l_rowid_list.EXISTS(1) THEN

        LGDB(v_snapshot_guid,
         'DB - Size for children of virtual parents used by db being updated');

        -- get the rowids into binary table to avoid
        -- PLS-00436: implementation restriction: cannot reference fields 
        -- of BULK In-BIND table of records."
        -- reduce size of partition from free size of disk
        -- add size of partition to used size of disk
        BEGIN

         FOR i IN l_intdata_list.FIRST..l_intdata_list.LAST
         LOOP

          LGDB(v_snapshot_guid,'DB - Updating child for virtual '||l_rowid_list(i)
           ||' '||l_intdata_list(i));

         END LOOP;

         FORALL j IN l_intdata_list.FIRST..l_intdata_list.LAST
          SAVE EXCEPTIONS
          UPDATE mgmt_storage_report_data a
          SET    usedb = LEAST( (usedb+l_intdata_list(j)), sizeb ),
                 freeb = GREATEST( (freeb-l_intdata_list(j)) , 0)
          WHERE  ROWID = l_rowid_list(j); 

        EXCEPTION
         WHEN bulk_errors THEN
 
          FOR j in 1..SQL%BULK_EXCEPTIONS.COUNT
          LOOP

           -- Parse the error message         
           l_errmsg := SUBSTR(SQLERRM(-SQL%BULK_EXCEPTIONS(j).error_code),
                        NVL(INSTRB(
                         SQLERRM(-SQL%BULK_EXCEPTIONS(j).error_code),':'),0)+1,2048);

           -- Log an issue and continue
           STORAGE_ECM_PKG.LOG_ISSUE
           (
             v_snapshot_guid,
             ERROR_REP_UPD_DB_QFLG,
             MGMT_MAX_STRING_ARRAY 
             ( 
              NULL,
               l_dbtg_list(i).target_type||' '||
                 l_dbtg_list(i).target_name||' '||
                  l_dbentf_list(l_idx_list(j)).os_entity||' '||   
                   l_errmsg
             ),
             ACTION_REP_ORACLE_SUPPORT,NULL
           );

         END LOOP;

         LGDB(v_snapshot_guid,'DB -  Updated Virtual entities that are used  by DB '
          ||l_dbtg_list(i).target_type||' '||l_dbtg_list(i).target_name); 

        END;

       END IF;

      END IF;


      -- UPDATE THE ENTITY IN USE BY THE DATABASE
      --
      -- ALl entities with the same global_unique_id as the eneity used by he database
      -- in the same storage layer and of the same entity type should be updated
      -- the query flag should indicate they are allocated and top
      -- the usedb and free should be updated
      BEGIN

        SELECT    b.idx,
	          a.ROWID, 
                  -- replace '_TOP_ALLOCATED_TO_DB_' before appending it to ensure
                  --  that we have only one instance of it in em_query_flag
                  REPLACE(REPLACE(REPLACE(REPLACE(b.em_query_flag,'_UNALLOCATED',
                   ''),'_VIRTUAL',''),'_INTERMEDIATE',''),
                    '_TOP_ALLOCATED_TO_DB_','')
                     ||'_TOP_ALLOCATED_TO_DB_'
       BULK COLLECT INTO l_idx_list,
                         l_rowid_list,
                         l_strgdata_list
       FROM   mgmt_storage_report_data a,
              TABLE ( CAST ( l_dbentf_list AS MGMT_STORAGE_DB_OS_ENT_ARRAY ) ) b
       WHERE  a.global_unique_id = b.global_unique_id
       AND    a.storage_layer = b.storage_layer
       AND    a.entity_type = b.entity_type
       AND    a.ecm_snapshot_id = v_snapshot_guid;

      IF l_strgdata_list IS NOT NULL AND l_strgdata_list.EXISTS(1) THEN

       FORALL j IN l_strgdata_list.FIRST..l_strgdata_list.LAST
        SAVE EXCEPTIONS
        UPDATE mgmt_storage_report_data
        SET    em_query_flag = l_strgdata_list(j),
               usedb = sizeb,
               freeb = 0
        WHERE  ROWID = l_rowid_list(j);

      END IF;

     -- execption handler for second update
     EXCEPTION

         WHEN bulk_errors THEN
 
          FOR j in 1..SQL%BULK_EXCEPTIONS.COUNT
          LOOP

           -- Parse the error message         
           l_errmsg := SUBSTR(SQLERRM(-SQL%BULK_EXCEPTIONS(j).error_code),
                        NVL(INSTRB(
                         SQLERRM(-SQL%BULK_EXCEPTIONS(j).error_code),':'),0)+1,2048);

           -- Log an issue and continue
           STORAGE_ECM_PKG.LOG_ISSUE
           (
             v_snapshot_guid,
             ERROR_REP_UPD_DB_QFLG,
             MGMT_MAX_STRING_ARRAY 
             ( 
              NULL,
                l_dbtg_list(i).target_type||' '||
                 l_dbtg_list(i).target_name||' '||
                  l_dbentf_list(l_idx_list(j)).os_entity||' '||
                   l_errmsg
             ),
             ACTION_REP_ORACLE_SUPPORT,NULL
           );

          END LOOP;

     END;
    LGDB(v_snapshot_guid,'DB - Updated size, qflag for os entities used by db '||l_dbtg_list(i).target_type||' '||l_dbtg_list(i).target_name); 

    -- Exception handler for all updates
    EXCEPTION

      WHEN OTHERS THEN
 
         -- Parse the error message
         l_errmsg := SUBSTR(SQLERRM,NVL(INSTRB(SQLERRM,':'),0)+1,2048);

         -- Log an issue and continue
         STORAGE_ECM_PKG.LOG_ISSUE
         (
           v_snapshot_guid,
           ERROR_REP_UPD_DB_QFLG,
           MGMT_MAX_STRING_ARRAY 
           ( 
            NULL,
            l_dbtg_list(i).target_type||' '||l_dbtg_list(i).target_name||' '||l_errmsg
           ),
           ACTION_REP_ORACLE_SUPPORT,NULL
         );

         RAISE e_skip_to_next_db;

    END;


   -- Exception handler for the loop 
   EXCEPTION

     WHEN e_skip_to_next_db THEN
       NULL;
     -- For this error code stop post processing of nfs filesystems

     WHEN e_abort_processing THEN
       RAISE;

     WHEN OTHERS THEN 

        -- Parse the error message
        l_errmsg := SUBSTR(SUBSTR(SQLERRM,NVL(INSTRB(SQLERRM,':'),0)+1), 1,2048);

        -- Log an issue and continue
        STORAGE_ECM_PKG.LOG_ISSUE
        (
          v_snapshot_guid,
          ERROR_REP_PROC_DB,
          MGMT_MAX_STRING_ARRAY
          (
           l_dbtg_list(i).target_type||' '||l_dbtg_list(i).target_name||' '||l_errmsg 
          ),
          ACTION_REP_ORACLE_SUPPORT,
          NULL
        );

   END;

  END LOOP;
  
EXCEPTION

  WHEN OTHERS THEN

    -- Parse the error message
    l_errmsg := SUBSTR(SUBSTR(SQLERRM,NVL(INSTRB(SQLERRM,':'),0)+1), 1,2048);

    -- Log an issue
    STORAGE_ECM_PKG.LOG_ISSUE
    (
      v_snapshot_guid,
      ERROR_REP_PROC_DB,
      MGMT_MAX_STRING_ARRAY
      (
        l_errmsg
      ),
      ACTION_REP_ORACLE_SUPPORT,
      NULL
    );

END DB_POST_PROCESSING;


------------------------------------------------------------
-- PROCEDURE POST_PROCESSING 
------------------------------------------------------------
PROCEDURE POST_PROCESSING 
( 
  v_snapshot_guid IN mgmt_ecm_gen_snapshot.snapshot_guid%TYPE,
  v_afterLoadInput IN SMP_EMD_NVPAIR_ARRAY
) IS

l_errmsg          mgmt_system_error_log.error_msg%TYPE;
l_compMode        VARCHAR2(10);

BEGIN

  IF v_snapshot_guid IS NULL THEN
   RETURN;
  END IF;

  LGDB(v_snapshot_guid,'Begin Post processing');
  l_compMode := 'CENTRAL';
  SELECT COMPONENT_MODE INTO l_compMode
  FROM mgmt_versions
  WHERE COMPONENT_NAME = 'CORE';

  IF (UPPER(TRIM(l_compMode)) = 'CENTRAL') THEN
    STORAGE_ECM_PKG.NFS_POST_PROCESSING(v_snapshot_guid);
  END IF;

  LGDB(v_snapshot_guid,'Completed NFS processing - Begin DB processing ');
  STORAGE_ECM_PKG.DB_POST_PROCESSING(v_snapshot_guid);

  LGDB(v_snapshot_guid,'END Post Processing');

EXCEPTION

  WHEN OTHERS THEN

   -- Parse the error message
   l_errmsg := SUBSTR('Error Post Processing storage data ,'
     ||SUBSTR(SQLERRM,NVL(INSTRB(SQLERRM,':'),0)+1),
        1,2048);

   -- Log error
  MGMT_LOG.LOG_ERROR
  (
   'STORAGE_REPORTING',
   -20001,
    l_errmsg
  );

  LGDB(v_snapshot_guid,'END Post Processing with ERROR '||l_errmsg);

END POST_PROCESSING;

BEGIN
------------------------------------------------
-- INITIALIZATION
------------------------------------------------

    LGDB(NULL,'Begin Initialization');

    IF STORAGE_ECM_PKG.l_netapp_mac_guid IS NULL THEN
      BEGIN
        STORAGE_ECM_PKG.l_netapp_mac_guid        := 
          MGMT_METRIC.GET_METRIC_GUID
          (
            STORAGE_ECM_PKG.c_netapp_filer_target_type, 
            STORAGE_ECM_PKG.c_metric_netapp_netinterf, 
            STORAGE_ECM_PKG.c_netapp_mac_address
          );
      EXCEPTION
        WHEN MGMT_GLOBAL.no_such_metric_found THEN
          RAISE_APPLICATION_ERROR(-20101,'Metric '
            ||STORAGE_ECM_PKG.c_netapp_mac_address
            ||' for target type '
            ||STORAGE_ECM_PKG.c_netapp_filer_target_type
            ||' not configured');
        WHEN OTHERS THEN
          RAISE_APPLICATION_ERROR(-20101,'Failed finding Metric '
            ||STORAGE_ECM_PKG.c_netapp_mac_address
            ||' for target type '
            ||STORAGE_ECM_PKG.c_netapp_filer_target_type);
      END;
    END IF;


     IF STORAGE_ECM_PKG.l_netapp_ip_guid  IS NULL THEN 
       BEGIN
         STORAGE_ECM_PKG.l_netapp_ip_guid         := 
           MGMT_METRIC.GET_METRIC_GUID
           (
             STORAGE_ECM_PKG.c_netapp_filer_target_type, 
             STORAGE_ECM_PKG.c_metric_netapp_netinterf, 
             STORAGE_ECM_PKG.c_netapp_ip_address
           );
       EXCEPTION
         WHEN MGMT_GLOBAL.no_such_metric_found THEN
           RAISE_APPLICATION_ERROR(-20101,'Metric '
            ||STORAGE_ECM_PKG.c_netapp_ip_address
            ||' for target type '
            ||STORAGE_ECM_PKG.c_netapp_filer_target_type
            ||' not configured');
         WHEN OTHERS THEN
           RAISE_APPLICATION_ERROR(-20101,'Failed finding Metric '
            ||STORAGE_ECM_PKG.c_netapp_ip_address
            ||' for target type '
            ||STORAGE_ECM_PKG.c_netapp_filer_target_type);
       END;
     END IF;


    IF STORAGE_ECM_PKG.l_netapp_qtree_path_guid IS NULL THEN
      BEGIN
        STORAGE_ECM_PKG.l_netapp_qtree_path_guid := 
         MGMT_METRIC.GET_METRIC_GUID
         (
           STORAGE_ECM_PKG.c_netapp_filer_target_type, 
           STORAGE_ECM_PKG.c_metric_netapp_qtree, 
           STORAGE_ECM_PKG.c_netapp_qtree_path
         );
      EXCEPTION
        WHEN MGMT_GLOBAL.no_such_metric_found THEN
          RAISE_APPLICATION_ERROR(-20101,'Metric '
           ||STORAGE_ECM_PKG.c_netapp_qtree_path||' for target type '
           ||STORAGE_ECM_PKG.c_netapp_filer_target_type||' not configured');
        WHEN OTHERS THEN
          RAISE_APPLICATION_ERROR(-20101,'Failed finding Metric '
           ||STORAGE_ECM_PKG.c_netapp_qtree_path
           ||' for target type '
           ||STORAGE_ECM_PKG.c_netapp_filer_target_type);
      END;
    END IF;
  
    IF STORAGE_ECM_PKG.l_netapp_volume_guid IS NULL THEN
      BEGIN
        STORAGE_ECM_PKG.l_netapp_volume_guid     := 
          MGMT_METRIC.GET_METRIC_GUID
          (
             STORAGE_ECM_PKG.c_netapp_filer_target_type, 
             STORAGE_ECM_PKG.c_metric_netapp_filesystems, 
             STORAGE_ECM_PKG.c_netapp_volume
          );
      EXCEPTION
        WHEN MGMT_GLOBAL.no_such_metric_found THEN
          RAISE_APPLICATION_ERROR(-20101,'Metric '
            ||STORAGE_ECM_PKG.c_netapp_volume
            ||' for target type '
            ||STORAGE_ECM_PKG.c_netapp_filer_target_type
            ||' not configured');
        WHEN OTHERS THEN
          RAISE_APPLICATION_ERROR(-20101,'Failed finding Metric '
            ||STORAGE_ECM_PKG.c_netapp_volume
            ||' for target type '
            ||STORAGE_ECM_PKG.c_netapp_filer_target_type);
      END;

    END IF;

    LGDB(NULL,'Initialization successful');

END STORAGE_ECM_PKG;
/
