#!/usr/bin/perl
#
# Copyright (c) 2001, 2005, Oracle. All rights reserved.  
#
#    NAME
#    post_crs_create.pl
#
#    SYNOPSIS
#    <perlbin>/perl post_crs_create.pl ORACLE_HOME=<ORACLE_HOME>
#
#    DESCRIPTION
#    This script needs to be run as post-script while running create 
#      new cluster operation
#
#    MODIFIED   (MM/DD/YY)
#       audupi     09/16/05 - audupi_bug-4579173
#       makarapu   08/25/05 - Intial creation
#
############################################################################

#############################################################################
#              POST SCRIPT FOR CREATE CRS CLONING                           #
#############################################################################

# Check that the version of perl used is 5.6.1 or above
require v5.6.1;

use English qw(-no_match_vars);  
use File::Basename;
use File::Spec();
use File::Spec::Functions;
use File::Path;

#Initialize global Variables
$ORACLE_HOME = "Not yet set!!";
$REMOTE_PORT = "Not yet set!!";
$BIN_DIR = "Not yet set!!"; 
@NODE_LIST;

parse_arguments(@ARGV);
set_remote_port();
set_node_list();
run_racgons();
run_oifcfg();
########## END ##########



#############################################################################
# NAME   : parse_arguments
# PURPOSE: Checks the command line arguments.
# INPUTS : The command line arguments
# OUTPUTS: NONE
# NOTES  : Updates global variables with command line arguments.
#
#############################################################################
sub parse_arguments 
{
	(@ARGV)=@_;
	my %ARGS;
	$no_arguments = scalar ( @ARGV );
	for ( my $i = 0; $i < $no_arguments; $i++ )
	{
		if ($ARGV[$i] =~ /=/)
		{
			($name, $value) = split (/=/, $ARGV[$i]);
			$ARGS{$name} = $value;
		}
	}

	if (exists $ARGS{"ORACLE_HOME"})
	{
		$ORACLE_HOME = $ARGS{"ORACLE_HOME"};
	}
	else
	{
		stop_on_err("Mandatory parameter ORACLE_HOME has not been passed.", "true");
	}

	if (!-e File::Spec->catfile($ORACLE_HOME, "opmn", "conf", "ons.config"))
	{
		stop_on_err("Invalid Oracle Home specified.", "true");
		exit -1;
	}

	$BIN_DIR = File::Spec->catdir($ORACLE_HOME, "bin");
}



#############################################################################
# NAME   : set_remote_port
# PURPOSE: Sets the global variable REMOTE_PORT
# INPUTS : NONE
# OUTPUTS: NONE
# NOTES  : This value is retrieved from the $ORACLE_HOME/opmn/conf/ons.config file
#############################################################################
sub set_remote_port
{
	$ons_config_file=File::Spec->catfile($ORACLE_HOME, "opmn", "conf", "ons.config");
	open (ONS_FILE, $ons_config_file);
	@ons_config=<ONS_FILE>;
	close (ONS_FILE);
	foreach $line (@ons_config)
	{
		if ($line =~ /remoteport/)
		{
			($tmp, $REMOTE_PORT) = split("=", $line);
			chomp($REMOTE_PORT);
			return;
		}
	}
}



#############################################################################
# NAME   : set_node_list
# PURPOSE: Sets the global variable NODE_LIST
# INPUTS : NONE
# OUTPUTS: NONE
# NOTES  :  This value is retrieved from the 
#              $ORACLE_HOME/inventory/ContentsXML/oraclehomeproperties.xml file
#              in the <NODE_LIST> tag
###############################################################################
sub set_node_list
{
	my $ohprops_file=File::Spec->catfile($ORACLE_HOME, "inventory", "ContentsXML", "oraclehomeproperties.xml");
	open (OHPROPS_FILE, $ohprops_file);
	my @array=<OHPROPS_FILE>;
	close (OHPROPS_FILE);
	my $node_list_flag = 0;
	my $node_list_tag ;
	

	foreach $line (@array)
	{
		if ( $line =~ /<(.*)NODE_LIST(.*)>/ ) 
        {
            $node_list_flag = 1 ;
        }
		if ( $node_list_flag == 1 )
		{
			chomp($line);
			$node_list_tag = $node_list_tag. " ". $line;
		}
		if ( $line =~ /<\/(.*)NODE_LIST(.*)>/ ) 
        {
            last;
        }
	}
	my (@NODES)= split ("NAME=",$node_list_tag);
	my $no_nodes = scalar ( @NODES );
	for ( my $i = 1; $i < $no_nodes; $i++ )
	{
		($tmp2, $NODE_LIST[$i-1], $tmp2)=split("\"",$NODES[$i]);
	}

}


#############################################################################
# NAME   : run_racgons
# PURPOSE: run the command
#              ./racgons add_config node2:<remote_port> node3:<remote_port>
# INPUTS : NONE
# OUTPUTS: NONE
# NOTES  : 
#############################################################################
sub run_racgons
{
	my $RACGONS_CMD = File::Spec->catfile($ORACLE_HOME, "bin", "racgons");
	$cmd = "$RACGONS_CMD add_config ";
	foreach $node (@NODE_LIST)
	{
		$cmd = $cmd." $node:$REMOTE_PORT";
	}
	print "\nExecuting -- $cmd";
	system("$cmd");
	my $status= $?;
	print "\nExit Status: $status\n";
	if ($status != 0)
	{
		print " \nWARNING: Non-zero exit code returned \n";
	}
}


#############################################################################
# NAME   : set_remote_port
# PURPOSE: Sets the global variable REMOTE_PORT
# INPUTS : NONE
# OUTPUTS: NONE
# NOTES  : 
# '$ORACLE_HOME/bin/oifcfg iflist -p'  needs to be executed to get the required information
#	Then we need to run the oifcfg as follows:
#   oifcfg setif -global <interface_name>/<subnet>:public 
#            <inteface_name>/<subnet>:cluster_interconnect 
#            [<interface_name>/<subnet>:public 
#            <inteface_name>/<subnet>:cluster_interconnect .......]
##############################################################################
sub run_oifcfg
{
	my @PUBLIC_ARR;
	my @CLUSTER_INTERCONNECT_ARR;
	my $PUBLIC_SET = "false";
	my $CLUSTER_INTERCONNECT_SET = "false";
	my $UNKNOWN = "UNKNOWN";
	my $PRIVATE = "PRIVATE";

	my $tmp_file_out = File::Spec->catfile($ORACLE_HOME, "clone", "tmpfile_out");
	my $tmp_file_err = File::Spec->catfile($ORACLE_HOME, "clone", "tmpfile_err");
	my $OIFCFG_CMD = File::Spec->catfile($ORACLE_HOME, "bin", "oifcfg");
	$cmd = "$OIFCFG_CMD iflist -p ";
	system ("$cmd 1> $tmp_file_out 2> $tmp_file_err");
	$status = $?;
	if ($status != 0) 
	{
		write_file ($tmp_file_err);
		stop_on_err("Failed executing command $cmd. Exit status $status was returned.", "false");
	}

	$cmd = "$OIFCFG_CMD setif -global ";

	# Parse the output to retrieve required values
	#If you get multiple "interface_name" of type UNKNOWN,
	#then you need to choose 1 interface and mark it as "public".
	#If you get multiple "interface_name" of type PRIVATE,
	#then you need to choose 1 interface and mark it as "cluster_interconnect"

	open (OUT_FILE, $tmp_file_out);
	@array=<OUT_FILE>;
	close (OUT_FILE);
	foreach $line (@array)
	{
		if ($PUBLIC_SET eq "false") 
		{
			if ($line =~ /$UNKNOWN/) 
			{
				(@PUBLIC) = split(" ", $line, 3);
				$cmd = $cmd.$PUBLIC[0].'/'.$PUBLIC[1].":public ";
				$PUBLIC_SET = "true";
			}
		}
		if ($CLUSTER_INTERCONNECT_SET eq "false") 
		{
			if ($line =~ /$PRIVATE/) 
			{
				(@CLUSTER_INTERCONNECT) = split(" ", $line, 3);
				$cmd = $cmd.$CLUSTER_INTERCONNECT[0].'/'.$CLUSTER_INTERCONNECT[1].":cluster_interconnect ";
				$CLUSTER_INTERCONNECT_SET = "true";
			}
		}
	}
	print "\nExecuting -- $cmd";
	system("$cmd");
	$status= $?;
	print "\nExit Status: $status \n";
	if ($status != 0)
	{
		print " WARNING: Non-zero exit code returned";
	}
}


#############################################################################
# NAME   : stop_on_err
# PURPOSE: Exit the program
# INPUTS : Message, true/false
# OUTPUTS: NONE
# NOTES  : 
#############################################################################
sub stop_on_err
{
	($msg, $exit)=@_;
	print "$msg\n";
	if ($exit eq "true") 
	{
		print "Aborting the post script operation.";
		exit -1;
	}
}

#############################################################################
# NAME   : write_file
# PURPOSE: Prints the contents of the file
# INPUTS : File to be printed
# OUTPUTS: NONE
# NOTES  : 
#############################################################################
sub write_file
{
	my $msg;
	($file_name)=@_;
	open (FILE_TO_PRINT, $file_name);
	@array=<FILE_TO_PRINT>;
	close (FILE_TO_PRINT);
	foreach $line (@array)
	{
		$msg = $msg.$line;
	}
	print "\n$msg\n";
}

