#!/usr/local/bin/perl
#
# $Header: emdb/sysman/webapps/em/WEB-INF/perl/db/rac/rac_services_TNS_o.pl /st_emdbsa_11.2/4 2009/04/06 09:16:28 shasingh Exp $
#
# rac_services_TNS_o.pl
#
# Copyright (c) 2007, 2009, Oracle and/or its affiliates. All rights reserved. 
#
#    NAME
#      rac_services_TNS_o.pl - Used for TNS entry update for Cluster Managed Database Service
#
#    DESCRIPTION
#      This module takes care of updating service alias in tnsnames.ora when invoked from
#	   cluster managed database services page
#
#    NOTES
#      <other useful comments, qualifications, etc.>
#
#    MODIFIED   (MM/DD/YY)
#    shasingh    03/31/09 - bug: 8322939
#    rsamaved    03/20/09 - avoid duplicate scan vip entries in tnsnames
#    shasingh    01/28/09 - revrese the rcp/scp order excetion
#    shasingh    03/11/07 - Creation
#

use strict;
use File::Spec::Functions qw(catfile);
use IPC::Open2;

require "$ENV{EMDROOT}/sysman/admin/scripts/db/db_common.pl";
require "$ENV{EMDROOT}/sysman/admin/scripts/semd_common.pl";

# SEP: platform directory seperator, WINOS: is window OS
# TNS_FILE_NAME; TNS file name
my ( $SEP, $WINOS, $TNS_FILE_NAME ) = ( "/", 0, "tnsnames.ora" );
if ( $^O =~ /MSWin/i ) {
	$WINOS = 1;
	$SEP   = "\\";
}

# processing info message sent back to OMS for display on service pge
my $tnsOpInfomsg = "";
1;
# Entry point for service add/modify operation
sub addServiceTNSEntry {
	my ( $serviceName, $lsnrHostsInfoRef, $taf ) = @_;
	doAddServiceEntry( $serviceName, $lsnrHostsInfoRef, $taf );
	print $tnsOpInfomsg;
}

sub doAddServiceEntry {
	my ( $serviceName, $lsnrHostsInfoRef, $taf ) = @_;
	my $tnsServiceString  = "";
	my $backupServiceName = "";
	my $tnsBackServiceString;
	my $localAddSuc;
	EMD_PERL_DEBUG("rac_services_TNS_o:addServiceTNSEntry serviceName $serviceName\n taf $taf\n addressStrs @$lsnrHostsInfoRef");

	# for taf 'preconnect' construct backup service alias name
	if ( uc($taf) eq "PRECONNECT" ) {
		$backupServiceName = $serviceName . "_" . $taf;
		$tnsBackServiceString .= constructTNSServiceString( $lsnrHostsInfoRef, $backupServiceName, "BASIC", "" );
	}

	# construct tns tring to be added in tnsnames.ora file
	$tnsServiceString = constructTNSServiceString( $lsnrHostsInfoRef, $serviceName, $taf, $backupServiceName );
	$tnsServiceString .= $tnsBackServiceString if defined $tnsBackServiceString;

	my $oracleHome   = $ENV{ORACLE_HOME};
	my $tnsnamesFile = getTNSFile($oracleHome);
	updateResultMsg( "TNS_FILE", "$tnsnamesFile" );

	EMD_PERL_DEBUG("rac_services_TNS_o:addServiceTNSEntry ORACLE_HOME : $oracleHome");
	EMD_PERL_DEBUG( "rac_services_TNS_o:addServiceTNSEntry tnsnames.ora file loc : $tnsnamesFile" );

	# filter out local host and remote hosts (in case of remote agent local host would be undefined)
	my ( $remoteHostsRef, $localHost ) = getRemoteHosts($lsnrHostsInfoRef);
	if ( defined($localHost) ) {
		EMD_PERL_DEBUG( "rac_services_TNS_o:addServiceTNSEntry local (to agent) host found" );
		$localAddSuc = add_Local_TNS_entry( $tnsnamesFile, $tnsServiceString, $serviceName, );
		updateResultMsg( "SUC_HOST", "$localHost" ) if $localAddSuc;
	}

	# Check if tns file location is shared, call to subroutine in db_common
	my $tnsDir = getDirname($tnsnamesFile);
	my $isSharedTNSDir = isSharedPath( $tnsDir, $remoteHostsRef, 1 );
	updateResultMsg( "TNS_DIR_SHARED", "TRUE" ) if $isSharedTNSDir;
	EMD_PERL_DEBUG( "rac_services_TNS_o:addServiceTNSEntry isSharedTNSDir = $isSharedTNSDir" );

	# return if tns directory shared and entry once added (same file from all node)
	return if ( $localAddSuc && $isSharedTNSDir );

	# Do add tns alias add on remote hosts
	processRemoteTNSEntry( $remoteHostsRef, $tnsnamesFile, $serviceName, "add", $isSharedTNSDir, $tnsServiceString );

	return;
}

sub processRemoteTNSEntry {
	my ( $remoteHostsRef, $tnsFile, $serviceName, $mode, $isShared, $tnsServiceString ) = @_;
	EMD_PERL_DEBUG(
		"rac_services_TNS_o:processRemoteTNSEntry 
    				 remoteHosts = @$remoteHostsRef
    				 tnsFile = $tnsFile
    				 serviceName = $serviceName
    				 mode = $mode
    				 isShared = $isShared"
	);
	my @remoteHosts = @{$remoteHostsRef};
	foreach my $host (@remoteHosts) {
		my $remoteTnsFile = getRemoteFileName( $tnsFile, $host );
		my $tempfile      = getTempFile();

		# copy remote file to local as temp file
		my ( $tnsFileExist, $err ) = fileExistOnHost( $host, $tnsFile );
        # in case of copy error move to next host
		next if $err;
		
		if ($tnsFileExist) {
			next if !executeRemoteCopy( $remoteTnsFile, $tempfile );
		}
		else {
			next if $mode eq "remove";
		}

		my $sucCode = 0;
		if ( $mode eq "add" ) {
			$sucCode = add_Local_TNS_entry( $tempfile, $tnsServiceString, $serviceName );
		}
		elsif ( $mode eq "remove" ) {
			$sucCode = remove_Local_TNS_entry( $tempfile, $serviceName );
		}

		# skip remote copy if sucCode  is 0
		next if !$sucCode;

		#copy back modified temp file to remote node
		my $opCode = executeRemoteCopy( $tempfile, $remoteTnsFile );
		EMD_PERL_DEBUG("rac_services_TNS_o:processRemoteTNSEntry opCode = $opCode");
		updateResultMsg( "SUC_HOST", "$host" ) if $opCode;
		last if $isShared and $opCode;
	}
}

sub getRemoteHosts {
	my ($lsnrHostsInfoRef) = @_;
	my @lsnrHostsInfo      = @{$lsnrHostsInfoRef};
	my %seen               = ();
	my $localHostName;
	foreach my $entry (@lsnrHostsInfo) {
		my ( $hostStatus, $host ) = split( /:/, $entry, 4 );
	
		# temp fix bug: 8322939 for beta2 to get node name by chopping off domain name, 
		#must use more generic approach for node name, get it either from olsnodes or node name property
		my $host = get_node_name_from_host($host); 
		
		if ( $hostStatus eq 'R' && !exists( $seen{$host} ) ) {
			$seen{$host} = $host;
		}
		elsif ( $hostStatus eq 'L' ) {
			$localHostName = $host;
		}
	}
	my @remoteHosts = keys %seen;
	return ( \@remoteHosts, $localHostName );
}

sub constructTNSServiceString {
	my ( $listenerInfoRef, $servName, $tafPolicy, $backSerName ) = @_;
	my @listenerInfo = @{$listenerInfoRef};
	my ( $eqChar, $nl, $sp, $sp2 ) = ( " = ", "\n", " ", "  " );

	# According to configuration file format rule parameter should the leftmost column
	# so remove leading whitespace char
	$servName = trim($servName);
	my $tnsStr = "";
    my $prevHostVIP = "";
	foreach my $entry (@listenerInfo) {
		my ( $hostStatus, $host, $hostVIP, $port ) = split( /:/, $entry, 4 );
        # check if hostVIP:Port was already added and continue
        if ($prevHostVIP =~ ($hostVIP . ":" . $port))
        {
            next;
        }

		$tnsStr .= $sp2 . "(ADDRESS" . $eqChar . "(PROTOCOL" . $eqChar . "TCP)(HOST" . $eqChar . $hostVIP . ")(PORT" . $eqChar . $port . "))" . $nl;
        $prevHostVIP .= $hostVIP . ":" . $port . ",";
	}
	$tnsStr .= $sp2 . "(LOAD_BALANCE" . $eqChar . "YES)" . $nl;
	$tnsStr .= $sp2 . "(";
	$tnsStr .= "CONNECT_DATA" . $eqChar . $nl;
	$tnsStr .= $sp . $sp2 . "(SERVER" . $eqChar . "DEDICATED)" . $nl;
	$tnsStr .= $sp . $sp2 . "(SERVICE_NAME" . $eqChar . $servName . ")" . $nl;
	if ( (uc($tafPolicy) eq "BASIC" ) || (uc($tafPolicy) eq "PRECONNECT" ) ) {
		$tnsStr .= $sp . $sp2 . "(FAILOVER_MODE" . $eqChar . $nl;
		if ( $backSerName ne "" ) {
			$tnsStr .= $sp . $sp2 . "(BACKUP" . $eqChar . $backSerName . ")" . $nl;
		}
		$tnsStr .= $sp . $sp2 . "(TYPE" . $eqChar . "SELECT)(METHOD" . $eqChar . $tafPolicy . ")(RETRIES" . $eqChar . "180)(DELAY" . $eqChar . "5)" . $nl . $sp . $sp2 . ")" . $nl;
	}
	$tnsStr .= $sp2 . ")";
	$tnsStr = $sp . "(DESCRIPTION" . $eqChar . $nl . $tnsStr . $nl . $sp . ")";
	$tnsStr = $servName . $eqChar . $nl . $tnsStr;

	#append and prepend new line char
	$tnsStr = $nl . $tnsStr . $nl;
}

# Platform specific implementation
sub executeRemoteCopy {
	my ( $src, $dest ) = @_;
	EMD_PERL_DEBUG("rac_services_TNS_o:executeRemoteCopy src $src dest $dest  ");

	if ($WINOS) {
		EMD_PERL_DEBUG("rac_services_TNS_o:executeRemoteCopy doing window remote copy");
		return executeWindowRemoteCopy( $src, $dest );
	}

	my $cmd = "";
	my ( $scp_cmd, $rcp_cmd ) = ( "/usr/bin/scp", "/usr/bin/rcp" );

	# retVal -1 signify failure in copy
	my $retVal = undef;

	

	if ( -e $scp_cmd ) {
		# assume success op
		$retVal = 1;
		# -o option introduced for scenario where command might require password as input so avoid hang.
		$cmd = "$scp_cmd -p -o RSAAuthentication=no -o PasswordAuthentication=no -o NumberOfPasswordPrompts=0 $src $dest";
		my ( $err, $output ) = executeCommand($cmd);
		$retVal = -1 if $err ne "" or ( $output =~ /\s*Permission denied\s*/i );
		EMD_PERL_DEBUG( "rac_services_TNS_o:executeRemoteCopy cmd $cmd err $err output $output" );
	}
	
	if ( (!defined($retVal) || $retVal == -1) && -e $rcp_cmd ) {
        # assume success op
        $retVal = 1;
        $cmd    = "$rcp_cmd -p $src $dest";
        my ( $err, $output ) = executeCommand($cmd);
        $retVal = -1 if $err ne "" or ( $output =~ /\s*Permission denied\s*/i );
        EMD_PERL_DEBUG( "rac_services_TNS_o:executeRemoteCopy cmd $cmd err $err output $output" );
    }

	my $opSucCode = 1;
	if (  $retVal == -1 || !defined($retVal) ) {
		$opSucCode = 0;
	}

	return $opSucCode;
}

sub executeWindowRemoteCopy {
	my ( $src, $dest ) = @_;
	my $WINDIR_VAL = $ENV{WINDIR};

	# WINDIR env not found so command could not be located
	return 0 if !defined $WINDIR_VAL;

	my ( $xcopy_exe, $xcopy_commd_path ) = ( "xcopy.exe", "" );
	$xcopy_commd_path = $WINDIR_VAL . "\\" . $xcopy_exe;

	if ( !( -e $xcopy_commd_path ) ) {
		$xcopy_commd_path = $WINDIR_VAL . "\\system32\\" . $xcopy_exe;
	}

	# could not locate xcopy.exe executable
	return 0 if !-e $xcopy_commd_path;

	my $cmd = "$xcopy_commd_path $src $dest /Q /Y";

	# passing "F" as input, which is required input in case file does not exist on destination
	my ( $err, $output ) = executeCommand( $cmd, "F" );
	EMD_PERL_DEBUG("rac_services_TNS_o:executeWindowRemoteCopy  err $err output $output");

	return 0 if $err ne "" or ( $output =~ /\s*File Not Found\s*|\s*Invalid drive specification\s*/i );
	# successful operation
	return 1;
}

sub executeCommand {
	my ( $cmd, $input )  = @_;
	my ( $err, $output ) = ( "", "" );

	if ( open2( *README, *WRITEME, $cmd ) ) {
		print WRITEME $input if defined $input;
		$output = <README>;
		if ( !close(README) ) {
			if ( $output ne "" ) {
				$err    = "\"${cmd}\" returned: \"" . $output . "\"";
				$output = "";
			}
			else {
				$err = "bad \"$cmd\": $! $?";
			}
		}
		close(WRITEME);
	}
	else {
		# open error
		$err = "cannot execute \"$cmd\": $!";
	}
	return ( $err, $output );
}

# Platform specific implementation
sub fileExistOnHost {
	my ( $host, $file ) = @_;
	EMD_PERL_DEBUG("rac_services_TNS_o:fileExistOnHost host $host file $file");
	return fileExistOnWindowHost( $host, $file ) if $WINOS;

	my ( $visible, $notvisible ) = ( "filevisible",  "notfilevisible" );
	my ( $err,     $output )     = ( "","" );
	my ( $ssh_cmd, $rsh_cmd )    = ( "/usr/bin/ssh", "/usr/bin/rsh" );
	my $commArg = " $host -n /bin/sh  -c \'\" if [ -f $file ] ; then echo $visible; else echo $notvisible;  fi \"\' ";

	my $retVal = undef;
	my $cmd;

	
	if (  -e $ssh_cmd ) {
		# assume success op
		$retVal =1 ;
		$cmd = "$ssh_cmd -o FallBackToRsh=no -o PasswordAuthentication=no -o NumberOfPasswordPrompts=0  -o StrictHostKeyChecking=yes ";
		$cmd .= $commArg;
		( $err, $output ) = executeCommand($cmd);
		$retVal = -1 if $err ne "" or ( $output ne "" && !( $output =~ /\s*($visible|$notvisible)\s*/ ) );
		EMD_PERL_DEBUG( "rac_services_TNS_o:fileExistOnHost retVal = $retVal  output = $output err=$err " );
	}
	
	if ( (!defined($retVal) || $retVal == -1) && -e $rsh_cmd ) {
        # assume success op
        $retVal =1 ;
        $cmd = "$rsh_cmd ";
        $cmd .= $commArg;
        ( $err, $output ) = executeCommand($cmd);
        $retVal = -1   if $err ne "" or ( $output ne "" && !( $output =~ /\s*($visible|$notvisible)\s*/ ) );
        EMD_PERL_DEBUG( "rac_services_TNS_o:fileExistOnHost retVal = $retVal  output = $output err=$err " );
    }

	$output = trim($output);
	my ( $fexist, $errOcc ) = ( 0, 0 );

	#if retVal still -1 means error occured during command execution .
	if ( $retVal == -1 || !defined($retVal)) {
		$fexist = 0;
		$errOcc = 1;
	}
	elsif ( $output =~ /^$visible$/ ) {
		$fexist = 1;
	}
	elsif ( $output =~ /^$notvisible$/ ) {
		$fexist = 0;
	}
	EMD_PERL_DEBUG( "rac_services_TNS_o:fileExistOnHost fexist = $fexist  errOcc = $errOcc  " );

	return ( $fexist, $errOcc );
}

sub fileExistOnWindowHost {
	my ( $host, $file ) = @_;
	EMD_PERL_DEBUG("rac_services_TNS_o:fileExistOnWindowHost host $host file $file");
	my $netFileName = getRemoteFileName( $file, $host );
	my $fexist = ( -e $netFileName );
	$fexist = 0 if !defined($fexist);
	EMD_PERL_DEBUG( "rac_services_TNS_o:fileExistOnWindowHost file=$file on host=$host exists =$fexist " );
	return $fexist;
}

sub getRemoteFileName {
	my ( $file, $host ) = @_;
	my $netAwareFileName = "";

	if ($WINOS) {
		#replace ":" with "$"
		$file =~ s/:/\$/g;
		$netAwareFileName = "\\\\" . $host . "\\" . $file;
	}
	else {
		$netAwareFileName = $host . ":" . $file;
	}

	return $netAwareFileName;
}

sub getTempFile {
	my ($suffix) = @_;
	my $dir = tempdir( CLEANUP => 1 );
	my ( $fh, $filename );

	if ( defined($suffix) ) {
		( $fh, $filename ) = tempfile( DIR => $dir, SUFFIX => $suffix );
	}
	else {
		( $fh, $filename ) = tempfile( DIR => $dir );
	}
	# close open file handle
	close $fh;
	return ($filename);
}

sub add_Local_TNS_entry {
	my ( $tnsnamesFile, $tnsServiceString, $serviceName ) = @_;
	EMD_PERL_DEBUG( "rac_services_TNS_o:add_TNS_entry tnsnamesFile $tnsnamesFile tnsServiceString $tnsServiceString serviceName $serviceName" );

	my $appendServiceStr = 0;
	if ( !-e $tnsnamesFile ) {
		$appendServiceStr = 1;
	}
	else {
		# for existing file open file in update mode
		open( TNS_ADD_FILE, "+< $tnsnamesFile" ) || return 0;
		my @fileContent = <TNS_ADD_FILE>;
		# don't process emty file for remove op
		my ( $modifiedContent, $retVal ) = removeServiceStr( \@fileContent, $serviceName )  if ( -s $tnsnamesFile );
		if ($retVal) {
			$modifiedContent .= "\n" . $tnsServiceString;
			seek( TNS_ADD_FILE, 0, 0 );
			print TNS_ADD_FILE $modifiedContent;
			truncate( TNS_ADD_FILE, tell(TNSFILE) );
		}
		else {
			$appendServiceStr = 1;
		}
		close(TNS_ADD_FILE) || return 0;
	}

	if ($appendServiceStr) {
		open( TNS_ADD_FILE, ">> $tnsnamesFile" ) || return 0;
		print TNS_ADD_FILE $tnsServiceString;
		close(TNS_ADD_FILE) || return 0;
	}

	return (1);
}

# Entry point for service remove
sub removeServiceTNSEntry {
	my ( $serviceName, $hostsRef ) = @_;
	doRemoveServiceEntry( $serviceName, $hostsRef );
	print $tnsOpInfomsg;

}

sub doRemoveServiceEntry {
	my ( $serviceName, $hostsRef ) = @_;
	EMD_PERL_DEBUG( "rac_services_TNS_o:removeServiceTNSEntry  serviceName $serviceName HostsRef @$hostsRef" );
	my $localRemSuc;

	my $oracleHome = $ENV{ORACLE_HOME};
	my $tnsnamesFile = getTNSFile($oracleHome);
	# call to subroutine in db_common
	my $tnsDir = getDirname($tnsnamesFile);
	updateResultMsg( "TNS_FILE", "$tnsnamesFile" );
    
    EMD_PERL_DEBUG( "rac_services_TNS_o:removeServiceTNSEntry ENV ORACLE_HOME $oracleHome " );
    EMD_PERL_DEBUG( "rac_services_TNS_o:removeServiceTNSEntry tnsnames.ora File loc $tnsnamesFile " );
    
    my ( $remoteHostsRef, $localHost ) = getRemoteHosts($hostsRef);
    if ( defined($localHost) ) {
		EMD_PERL_DEBUG( "rac_services_TNS_o:addServiceTNSEntry local (to agent) host found" );
		$localRemSuc = remove_Local_TNS_entry( $tnsnamesFile, $serviceName );
		updateResultMsg( "SUC_HOST", "$localHost" ) if $localRemSuc;
	}

	# Check if tns file location is shared 
	my $isSharedTNSDir = isSharedPath( $tnsDir, $remoteHostsRef, 1 );
	updateResultMsg( "TNS_DIR_SHARED", "TRUE" ) if $isSharedTNSDir;

	return if ( $localRemSuc && $isSharedTNSDir );

	processRemoteTNSEntry( $remoteHostsRef, $tnsnamesFile, $serviceName, "remove", $isSharedTNSDir );

	return;
}

sub remove_Local_TNS_entry {
	my ( $file, $serviceName ) = @_;
	if ( !( -e $file ) || !( -s $file ) ) {
		EMD_PERL_DEBUG( "rac_services_TNS_o:remove_Local_TNS_entry  File  $file does not exist or have no content" );
		return 0;
	}
	else {
		#open file in update mode
		open( TNSFILE, "+< $file" ) || return 0;
		my @content = <TNSFILE>;
		my ( $modContent, $retVal ) = removeServiceStr( \@content, $serviceName );
		if ($retVal) {
			seek( TNSFILE, 0, 0 );
			print TNSFILE $modContent;
			truncate( TNSFILE, tell(TNSFILE) );
		}
		close(TNSFILE) || return 0;
		return $retVal;
	}

}

sub removeServiceStr {
	my ( $contentRef, $serviceName ) = @_;
	my @content = @$contentRef;
	EMD_PERL_DEBUG("rac_services_TNS_o:removeServiceStr serviceName $serviceName");
	my ( $modContent, $matchFound ) = removeEntryFromBuffer( \@content, $serviceName );

	# for taf policy 'preconnect' remove backup service also
	my $bMatchFound   = 0;
	my $backupService = $serviceName . "_PRECONNECT";
	@content = split( /\n/, $modContent );
	( $modContent, $bMatchFound ) = removeEntryFromBuffer( \@content, $backupService );
	return ( $modContent, ( $bMatchFound | $matchFound ) );
}

sub removeEntryFromBuffer {
	my ( $contentRef, $serviceName ) = @_;
	my @content    = @$contentRef;
	my $inProcess  = 0;
	my $out        = "";
	my $bcount     = undef;
	my $comment    = "#";
	my $matchFound = 0;
	$serviceName = trim($serviceName);
	my $origContent = "";

	foreach my $line (@content) {
		$origContent .= $line . "\n";
		chomp($line);
		if ( !( $line =~ /^[#]/ ) )    # do not process commented line
		{
			if ( (!$inProcess ) 
			       && ( ( $line =~ m/^\s*$serviceName\s*=\s*/i )
			            || ( $line =~ m/(\))(\s*$serviceName\s*=\s*)/i ) 
			          )
			    )
			{
				EMD_PERL_DEBUG( "rac_services_TNS_o:removeEntryFromBuffer found match $& in line $line " );
				$inProcess = 1;
				
				#Below logic take care of scenario whare servie entry like "...) servicename = (....)"
				# Although above format not valid from systax rule
				my ($beforeTxt,$afterTxt)  = ($`.$1, $2.$'); 
				# put new char at end so that service str start fron new line
				$out .= $beforeTxt."\n" if length(trim($beforeTxt)) !=0 ; 
				$line  = $afterTxt if length(trim($afterTxt)) !=0;
			}

			if ($inProcess) {
				my $processStr = "";
				( $bcount, $processStr ) = updateBracketMatchCounter( $bcount, $line );
				if ( defined($bcount) && $bcount == 0 ) {
					$line       = $processStr;
					$inProcess  = 0;
					$matchFound = 1;
					$bcount     = undef;         # for multiple match
				}
				$line = $comment . $line;
			}
		}
		$out .= $line . "\n";
	}

	$out = $origContent if !$matchFound;
	return ( $out, $matchFound );
}

sub updateBracketMatchCounter {
	my ( $count, $str ) = @_;
	my @chars          = split( //, $str );
	my $charIndex      = 0;
	my $returnStr      = "";
	my $additionalData = "";
	foreach my $ch (@chars) {
		$returnStr .= $ch;
		$charIndex++;
		if ( $ch eq "\(" ) { $count++; }    # Increment open bracket counter
		elsif ( $ch eq "\)"	&& --$count == 0 )              # Decrease open bracket counter and check for maching
		{
			splice( @chars, 0, $charIndex );
			$"              = "";
			$additionalData = "@chars";
			$additionalData = trim($additionalData);
			# add additional character after bracket match in new line
			$returnStr      = $returnStr . "\n" . $additionalData   if length($additionalData) != 0;
			last;
		}
	}
	return ( $count, $returnStr );
}

sub getTNSFile {
	my ($oracleHome) = @_;
	my $defualtTNSDir = "$oracleHome" . $SEP . "network" . $SEP . "admin";
	my ( $tnsDir, $tnsFileLoc );

	if ($WINOS) {
		# call sub routine from db_common.pl file
		$tnsDir = getTNS_ADMINOnNT($oracleHome);
	}
	else {
		$tnsDir = $ENV{TNS_ADMIN} if defined( $ENV{TNS_ADMIN} );
	}
	$tnsDir = $defualtTNSDir if !defined $tnsDir or $tnsDir eq "";
	$tnsFileLoc = catfile( ($tnsDir), $TNS_FILE_NAME );
	return $tnsFileLoc;
}

sub updateResultMsg {
	my ( $name, $value ) = @_;
	my ( $valSep, $NVSep ) = ( ":val_sep:", ":nv_sep:" );
	if ( !defined($tnsOpInfomsg) ) {
		$tnsOpInfomsg = $name . $NVSep . $value;
		return;
	}
	$tnsOpInfomsg .= $valSep . $name . $NVSep . $value;
	return;
}

sub isSharedPath {
	my $shared = 0;
	my ( $dirPath, $hostsRef, $shortCheck ) = @_;
	EMD_PERL_DEBUG( "rac_services_TNS_o:isSharedPath dirPath = $dirPath hosts = @$hostsRef short= $shortCheck  " );

	# return if invalid arguments passed
	return $shared
	  if ( ( !defined($dirPath) || $dirPath eq "" )
		|| ( !defined($hostsRef) || $hostsRef eq "" ) );

	# return if path not directory
	return $shared if !( -d $dirPath && -w $dirPath );
	my $suffix   = ".tmp";
	my $template = "shPathCheckXXXX";

	my ( $fh, $tmpfile ) = tempfile( $template, DIR => $dirPath, SUFFIX => $suffix, UNLINK => 1 );
	close $fh;

	my %seen  = ();
	my @hosts = @{$hostsRef};

	#check hosts length return 0 for length <=1
	return $shared if @hosts <= 0;
	my $checkPassedCount = 0;
	foreach my $host (@hosts) {
		next if exists( $seen{$host} );
		my ( $status, $errCode ) = fileExistOnHost( $host, $tmpfile );
		EMD_PERL_DEBUG( "rac_services_TNS_o:isSharedPath status = $status  errCode=$errCode " );
		$seen{$host} = $host;

		#if error occured check for next host
		next if $errCode != 0;
		$shared = $status;

		#break if check count passed or file not visible from any one host
		last if ( $shortCheck && ++$checkPassedCount == 2 ) or $status == 0;
	}

	return $shared;
}

sub get_node_name_from_host()
{
	my $host_name = trim(@_[0]);
	my ($node_name) =  split /\./, $host_name,2;
	$node_name = $host_name if $node_name eq undef or $node_name eq '';  
	
	return $node_name;
}
