# Copyright (c) 2004, 2009, Oracle and/or its affiliates. All rights reserved. 
#
#    NAME
#      asmcmdglobal - ASM CoMmanD line interface (Global Memory Module)
#
#    DESCRIPTION
#      ASMCMD is a Perl utility that provides easy nagivation of files within
#      ASM diskgroups.
#
#      This module contains the global configuration variables for the
#      modular setup of ASMCMD.  Each ASMCMD module defines a number of
#      callback functions that are used by the driver module, asmcmdcore.
#      References to these callback functions are stored in the global
#      variables of this module.
#
#    NOTES
#      usage: asmcmdcore [-p] [command]
#
#      Whenever a new module of ASMCMD commands is added to ASMCMD,
#      global arrays in this module needs to be updated with 
#      callback definitions for these functions:
#        1) Entrance function to call all module-specific commands
#        2) Function to list help messages for module-specific commands
#        3) Function to list all available commands for a module
#        4) Function to determine if a given command is one supported
#           by the module
#        5) Function to determine if a given command supports the use
#           of wildcard characters in its path argument(s).
#        6) Function to display the correct syntax for a command within
#           the module
#        7) Function to display error messages for a module
#        8) Function to display messages for exceptions signalled
#        9) Function to determine if a given command can run without
#           an ASM instance available.
#       10) Function to determine if a given command can run without
#           an ASM instance.
#
#      These arrays should be updated by the means of a module
#      initialization function, which is defined within each module and
#      referenced in the asmcmdcore:asmcmdcore_init_modules array.
#
#      Here is the modular layout of ASMCMD:
#
#      asmcmd
#        |
#        v
#      asmcmdcore
#        |
#        |___________________________________________________________
#        |         |            |            |             |        |
#        |         v            v            v             v        v
#        |     asmcmdbase   asmcmddisk  asmcmdambr   asmcmdtemplate*|
#        |         |            |            |             |        |
#        |         |____________|____________|_____________|        |
#        |                      |            |                      |
#        |                      |            |______________________|
#        |                      |            |
#        |                      |            v
#        |                      |       asmcmdshare
#        |                      |            |
#        v                      v            v
#        |______________________|____________|
#                      |
#                      v
#                 asmcmdglobal
#
#      N.B. asmcmdtemplate* represents any new module to be added.  Based
#      on this modular model, we have these restrictions:
#        1) asmcmdglobal should never use any of the other modules.
#        2) asmcmdshare can use only asmcmdglobal.
#        3) No module can use asmcmdcore.
#        4) The asmcmdtempate* modules must not use each other, e.g.
#           asmcmddisk cannot use asmcmdbase.
#
#
#    MODIFIED  (MM/DD/YY)
#    sanselva   04/06/09 - add asmcmdglobal_options,asmcmdglobal_deprecated_options
#    heyuen     10/14/08 - move CONN_STR to global
#    heyuen     09/19/08 - add infinite command indicator in global hash
#    heyuen     08/02/08 - add verbose
#    heyuen     07/28/08 - use command property array
#    heyuen     07/14/08 - 
#    soye       03/20/08 - update version to 11.2
#    heyuen     05/25/07 - add return codes for errors
#    hqian      05/24/07 - Update asmcmd version to 11.1.0.6.0
#    hqian      03/02/07 - update comments; update ASM version initial value
#    jilim      12/11/06 - Add comment on error message for cp
#    hqian      07/20/06 - #5397026: new @asmcmdglobal_no_instance_callbacks 
#    hqian      06/19/06 - Add comments on error message ranges for ambr and 
#                          disk 
#    hqian      01/20/06 - #4939032: creation of ASMCMD global memory module
#############################################################################

package asmcmdglobal;
require Exporter;
our @ISA    = qw(Exporter);
our @EXPORT = qw(%asmcmdglobal_hash
                 %asmcmdglobal_cmds
                 %asmcmdglobal_options
                 %asmcmdglobal_deprecated_options
                 @asmcmdglobal_command_callbacks
                 @asmcmdglobal_help_callbacks
                 @asmcmdglobal_command_list_callbacks
                 @asmcmdglobal_is_command_callbacks
                 @asmcmdglobal_is_wildcard_callbacks
                 @asmcmdglobal_syntax_error_callbacks
                 @asmcmdglobal_no_instance_callbacks
                 @asmcmdglobal_error_message_callbacks
                 @asmcmdglobal_signal_exception_callbacks
                 $ASMCMDGLOBAL_WCARD_CHARS
                 $ASMCMDGLOBAL_VER_10gR1
                 $ASMCMDGLOBAL_VER_10gR2
                 $ASMCMDGLOBAL_VER_11gR1
                 $ASMCMDGLOBAL_VER_11gR2
                 $ASMCMDGLOBAL_USE_CONN_STR
                 );

use strict;

######################### ASMCMD Global Constants ############################
our ($ASMCMDGLOBAL_WCARD_CHARS) = '[%*]';      # Set of wildcard characters. #
our ($ASMCMDGLOBAL_VER_10gR1) = '10.1.0.2.0';  # ASM version for 10gR1.      #
our ($ASMCMDGLOBAL_VER_10gR2) = '10.2.0.1.0';  # ASM version for 10gR2.      #
our ($ASMCMDGLOBAL_VER_11gR1) = '11.1.0.3.0';  # ASM version for 11gR1.      #
our ($ASMCMDGLOBAL_VER_11gR2) = '11.2.0.0.0';  # ASM version for 11gR2.      #

######################### ASMCMD Global Variables ############################
#
our %asmcmdglobal_hash = (cwdnm  => '',  # Current directory name            #
                          cwdref => '',  # Current directory reference id    #
                          gname  => '',  # Group name                        #
                          gnum   => '',  # Group number                      #

                          cmd    => '',  # The current internal command      #
                          lprmt  => 'n', # Use long prompt (with cwd)?       #
                          usr    => '',  # Username for authentication       #
                          pswd   => '',  # Password for authentication       #
                          ident  => '',  # Connect identifier                #
                          mode   => 'i', # i=interactive,n=non-interactive   #
                          nfnd   => 0,   # 1 if $level==0 and file not found #
                          ver    => '0.0.0.0.0', # Version of ASM instance,  #
                                         # in form                           #
                                         # xxyz, from format xx.y.0.z;       #
                                         # 0 => ASM version known            #
                          endn   => '',  # Endianness of the system          #
                          acver  => '11.2.0.0.0', # ASMCMD version number    #
                                         # Update acver for every ASMCMD     #
                                         # release!!!                        #
                          contyp => '',  # connection type: sysasm or sysdba #
                          verbose=> '',  # verbose mode                      #
                          sth    => undef, # statement handler used only by  #
                                           # signal handler to cancel        #
                                           # statement                       #
                          e      => 0,                    # output the error #
                          running=> 0,   #is an infinite command running?    #
                          consistchk => 'n'  #run options consistancy check ?#
                          );

our ($ASMCMDGLOBAL_USE_CONN_STR) = 0;   # 1 allows the use of -c option to  #
                                        # specify a connect string; 0 dis-  #
                                        # allows it. 

# The following arrays hold lists of functions.  Each ASMCMD module must
# have exactly one function in each of these arrays.
#

# These functions process ASMCMD commands within their respective modules.
our (@asmcmdglobal_command_callbacks);

# These functions process the help command within their respective modules.
our (@asmcmdglobal_help_callbacks);

# These functions list available ASMCMD commands supported by their
# respective modules.
our (@asmcmdglobal_command_list_callbacks);

# These functions determine if a command is supported by their respective
# modules.
our (@asmcmdglobal_is_command_callbacks);

# These functions determine if a command supports wildcards.
our(@asmcmdglobal_is_wildcard_callbacks);

# These functions display the correct syntax of an ASMCMD command supported
# by their respective modules.
our (@asmcmdglobal_syntax_error_callbacks);

# These functions determine if a command can run even without an ASM 
# instance.
our (@asmcmdglobal_no_instance_callbacks);

our (%asmcmdglobal_cmds);

# Global Hash table to store all options for consistency check
our (%asmcmdglobal_options);

# List of deprecated options for 11.2
# Pls update this table with all options that are deprecated
# in the next release so that WARNING can be shown when used
# Delete the entries from the table in the succesive release
# after the deprecation.
# #########################################################
# Format
# ( COMMAND1 => { Option1 => [ GetoptSyntax , NewOption ]
#                 Option2 => [ GetoptSyntax , NewOption ]
#   COMMAND2 => { Option 1 .......
#  
#  if NewOption is NULL then, option no more supported
#
#  Algo for handling deprecated options
#--------------------------------------
#  1. push the current options in @string for GetOptions
#  2. push deprecated options if any 
#  3. process args using GetOptions
#  4. if any depr options were used, set the value of the 
#     corresponding options
#  5. print WARNING for each depcr option used.
#
# Please see init() for each perl module and asmcmdshare_handle_deprecation 
#
# Note: There are corner cases where both new option and the
# corresponding deprecated options is used together in a
# command the value that appears later in the order takes
# precedence.
# for eg : ASMCMD> md_backup -G DG1 -g DG2
#          ASMCMD> md_backup -G DG1 -G DG2
#          ASMCMD> md_backup -g DG1 -G DG2
# In the above examples md_backup will backup DG2 not DG1
###########################################################
our (%asmcmdglobal_deprecated_options) = ( find      => { 't'=>
                                                        ['t=s', 'type']},
                                           ls        => { 'r'=> 
                                                        ['r','reverse'],
                                                          'c'=> ['c','NULL']},
                                           lsdg      => { 'c'=> 
                                                        ['c','discovery']},
                                           lsdsk     => { 'c'=> 
                                                        ['c','discovery'],
                                                          'd'=> ['d=s','G'],
                                                          'm'=> 
                                                   ['m=s','member|candidate']},
                                           md_backup => { 'b'=> ['b','NULL'],
                                                          'g'=> ['g=s','G']},
                                           md_restore=> { 'b'=> ['b=s','NULL'],
                                                          'i'=> ['i','ignore'],
                                                          't'=> 
                                                     ['t=s','full|nodg|newdg'],
                                                          'f'=> ['f=s','S'],
                                                          'g'=> ['g=s','G']}
                                         );

# These functions display error messages for module-specific ASMCMD commands.
# ASMCMD Error Message Numbers:
#   8000-8099 - General ASMCMD errors.
#     8001    - bad diskgroup.
#     8002    - directory/file not found.
#     8003    - connected to non-ASM instance.
#     8004    - invalid alias name.
#     8005    - ambiguous wildcard match for CD.
#     8006    - entry '%s' does not refer to a valid directory.
#     8007    - unclosed single-quote.
#     8008    - local instance not presented for cp cmd 
#     8009    - no multiple source files can copy from a remote instance
#     8010    - no usr name is specified in connect string
#     8011    - no instance identifier is specified in connect string
#     8012    - can not determine file type for file
#     8013    - can not determine logical block size for file
#     8014    - can not open file
#     8015    - can not create file
#     8016    - copy source file to  target file failed
#     8017    - error while reading data from file
#     8018    - error while writing data to file
#     8019    - failed to commit file
#
#   8100-8099 - Non-exception SQL errors.
#     8100    - select statement failed.
#
#   8200-8299 - Exceptions.
#     8200    - connection to Oracle lost.
#     8201    - cannot connect to Oracle.
#     8202    - asmcmd internal error.
#
#   8300-8399 - Volume Device Driver
#     8300    - ?
#
#   8400-8499 - File Access Control
#     8400    - ?
#
#   9345-9370 - Metadata Backup
#     9345    - Cannot open Intermediate file: '$arg'
#     9346    - Cannot close Intermediate file: '$arg'!,
#     9347    - Cannot re-create data structures from Intermediate file:
#               '$arg'
#     9348    - Override option can only be specified with -t newdg,
#               default -t full
#     9349    - Diskgroup '$arg' not discovered by ASM instance, Skipping...
#     9350    - Diskgroup '$arg' not mounted by ASM instance, Skipping...
#     9351    - ASM instance has no diskgroups mounted
#     9352    - Create Diskgroup failed '$arg'
#     9353    - Create/Alter Template failed '$arg'
#     9354    - Create alias directory failed '$arg'
#     9355    - Couldnot find information for diskgroup '$arg' in backup file
#     9356    - Backup file '$arg' is either empty or cannot be interpreted
#     9357    - A file with name '$arg' already exists
#     9358    - Option -t newdg specified without any override options
#     9359    - Invalid diskgroup name '$arg' specified in Override options
#
#   9371-9390 - Disk Operations
#     9371    - disk '$arg1' does not exist in disk group '$arg2'
#     9372    - physical blocks $arg1-$arg2 do not map to a valid ASM file
#     9373    - not all physical blocks submitted for remap
#     9374    - cannot open disk header format file '$arg1' \n$arg2
#     9375    - cannot close disk header format file '$arg1' \n$arg2
#     9376    - cannot re-create data structures from disk header
#               format file: '$arg'
#     9377    - disk header format file '$arg' is either empty or 
#               cannot be interpreted
#     9378    - the scanning of disk headers is supported only on UNIX platforms
#     9379    - character devices are not supported
#     9380    - cannot find 'syscall.ph'; please make sure that
#               the Perl utility 'h2ph'  \nhas been run to 
#               convert system .h files to Perl .ph files.
#               \n\n$arg
#     9381    - remap command requires ASM software version
#               $ASMCMDGLOBAL_VER_11gR1 or later.
#
# Functions for recording errors; does not terminate ASMCMD session.
our (@asmcmdglobal_error_message_callbacks);
# Functions for signaling errors; terminates ASMCMD session.
our (@asmcmdglobal_signal_exception_callbacks);

return 1;
