create or replace view sys_uncompressed_segs_base
    (OWNER, SEGMENT_NAME,
     PARTITION_NAME,
     SEGMENT_TYPE, SEGMENT_TYPE_ID,
     TABLESPACE_ID, TABLESPACE_NAME, BLOCKSIZE,
     BYTES, BLOCKS, EXTENTS,
     RELATIVE_FNO, SEGMENT_FLAGS, SEGMENT_OBJD, segment_objn) 
as
select NVL(u.name, 'SYS'), o.name, o.subname,
       so.object_type, s.type#,
       ts.ts#, ts.name, ts.blocksize,
       s.blocks * ts.blocksize, s.blocks, s.extents,
       s.file#, NVL(s.spare1,0), o.dataobj#, o.obj#
from sys.user$ u, sys.obj$ o, sys.ts$ ts, sys.sys_objects so, sys.seg$ s
where s.file# = so.header_file
  and s.block# = so.header_block
  and s.ts# = so.ts_number
  and s.ts# = ts.ts#
  and ts.bitmapped <> 0
  and ts.name not in ('SYSTEM', 'SYSAUX') -- exclude admin tablespaces
  and o.obj# = so.object_id
  and bitand(o.flags, 128) != 128 -- exclude bin objects
  and o.owner# = u.user# (+)
  and u.name <> 'SYS' -- exclude sys owned objects
  and s.type# = so.segment_type_id
  and o.type# = so.object_type_id
  and so.object_type in ('TABLE', 'TABLE PARTITION')
  and bitand(s.spare1, 2048) != 2048 -- exclude compressed segments
/
grant select on sys_uncompressed_segs_base to select_catalog_role
/
create or replace view sys_uncompressed_segs (tabowner, tabname, partname, tsname, tbsid,
segsize, segobjd, segobjn, segment_type, total_indexes) as 
select 
  uncmp.owner, uncmp.segment_name, uncmp.partition_name, 
  uncmp.tablespace_name, uncmp.tablespace_id, uncmp.bytes,  
  uncmp.segment_objd, uncmp.segment_objn, 
  uncmp.segment_type,
  count(i.obj#)
from obj$ o, sys_uncompressed_segs_base uncmp, ind$ i
where 
  uncmp.segment_name = o.name and
  o.obj# = i.bo# 
group by 
  uncmp.owner, uncmp.segment_name, uncmp.partition_name, 
  uncmp.tablespacE_name, uncmp.tablespace_id, uncmp.bytes, 
  uncmp.segment_objd, uncmp.segment_objn, uncmp.segment_type
/
grant select on sys_uncompressed_segs to select_catalog_role
/
create or replace package body dbms_space wrapped 
a000000
1
abcd
abcd
abcd
abcd
abcd
abcd
abcd
abcd
abcd
abcd
abcd
abcd
abcd
abcd
abcd
b
223fd 6f01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/
show errors;
