# $Header: emagent/sysman/admin/scripts/Ptdpm0.pm /st_emagent_10.2.0.4.2db11.2/1 2008/11/07 01:47:21 nparaddi Exp $
#
# Package : Ptdpm0.pm
#
# Copyright (c) 2002, 2008, Oracle and/or its affiliates.All rights reserved. 
#
#   NAME
#       Ptdpm0.pm
#
#   DESCRIPTION
#       Collects platform-specific Hardware and Software data
#
#   NOTES
#
#   MODIFIED     (MM/DD/YY)
#      nparaddi   03/13/08 - Backport nparaddi_bug1_6487348 from
#                            st_emcore_10.2.0.1.0
#      mgoodric   08/16/05 - fix ifconfig processing 
#      mgoodric   05/03/05 - add DMESG 
#      mgoodric   09/27/04 - code cleanup 
#      mgoodric   04/13/04 - Exclude -x none for df command 
#      mgoodric   04/01/04 - Linux cleanup
#      mgoodric   03/17/04 - Remove LC_ALL=C
#      rlal       03/11/04 - Cpu vendor name change for ia64 linux
#      rlal       01/26/04 - Fix for OS name and cut
#      nsharma    12/22/03 - Fix nslookup path for linux
#      mgoodric   06/16/03 - fix converting Solaris timezone to JAVA timezones
#      mgoodric   05/11/03 - added -debug tracing
#      mgoodric   04/15/03 - fix using uname for arch
#      mgoodric   04/07/03 - fix finding MAC address
#      rlal       02/25/03 - EM 4.0 changed for Linux
#      goramamo   01/21/03 - Make scripts more portable
#      goramamo   12/12/02 - Changes for SPARC III+ machines
#      goramamo   11/21/02 - Changes to include tvmlet8x tests
#      goramamo   11/18/02 - tcp & udp parms collection bug 2673413
#      goramamo   08/16/02 - Error Msg and getent changes
#      goramamo   08/16/02 - 904 Changes
#      goramamo   06/26/02 - Added getent call
#      xxu        06/25/02 - remove /usr/local/bin/perl
#      goramamo   06/20/02 - Added GIT5 collection
#      goramamo   04/12/02 - Pass errorstream to calling Java code
#      goramamo   03/28/02 - Make NLS compliant;
#                          - Added code to generte XML from files
#      goramamo   02/28/02 - Fix domainname collection bug 2226505
#      mgoodric   02/15/02 - Make scripts more portable
#      mgoodric   02/13/02 - Creation
#
##*************************************************************

use strict;

package Ptdpm0;

use Carp;
require 5.005;

#require 5.6.1;
require Exporter;

use ecmErrorMsgs;
use Ptdpm3;

#******************************************
#   Export Subroutines
#******************************************

@Ptdpm0::ISA    = ('Exporter');
@Ptdpm0::EXPORT = qw(
  &setupOSD
  &getOSname
  &getOSName
  &getHostName
  &getDomainName
  &getCCinfo
  &getHardwareProvider
  &getHardwareName
  &getProcessorType
  &getArchType
  &getDiskInfo
  &getOSrelease
  &getOSversion
  &getSwapSpaceInfo
  &getEtcSystemLocation
  &getLoadedKernelModules
  &getNICDetails
  &getMACAddress
  &getInstructionSetList
  &getCurrSysDef
  &getSysConf
  &getKernelDriverTunables
  &getKernelDriverTunableValues
  &getEtcMnttabLocation
  &getPkginfoFilesLocation
  &getAdminEntries
  &getIPCSStatus
  &getGlobalVariables
  &getDefaultRouter
  &getResolvConf
  &getGetConfig
  );

#******************************************
#     Global Variables
#******************************************

use constant NIL => '';

$Ptdpm0::DATALOC    = NIL;
$Ptdpm0::DEVNULL    = '/dev/null';
$Ptdpm0::ARP        = '/usr/sbin/arp';
$Ptdpm0::BIN        = '/bin';
$Ptdpm0::CAT        = '/bin/cat';
$Ptdpm0::CUT        = '/usr/bin/cut';
$Ptdpm0::DF         = '/bin/df';
$Ptdpm0::EGREP      = '/bin/egrep';
$Ptdpm0::GETCONF    = '/usr/bin/getconf';
$Ptdpm0::GETENT     = '/usr/bin/getent';
$Ptdpm0::HEAD       = '/usr/bin/head';
$Ptdpm0::HOSTNAME   = '/bin/hostname';
$Ptdpm0::IFCONFIG   = '/usr/sbin/ifconfig';
$Ptdpm0::IPCS       = '/usr/bin/ipcs';
$Ptdpm0::ISALIST    = '/bin/isalist';
$Ptdpm0::MACH       = '/bin/mach';
$Ptdpm0::MODINFO    = '/usr/sbin/modinfo';
$Ptdpm0::NDD        = '/usr/sbin/ndd';
$Ptdpm0::NSLOOKUP   = '/usr/sbin/nslookup';
$Ptdpm0::SED        = '/bin/sed';
$Ptdpm0::SHOWREV    = '/bin/showrev';
$Ptdpm0::SORT       = '/bin/sort';
$Ptdpm0::SWAP       = '/usr/sbin/swap';
$Ptdpm0::SYSDEF     = '/usr/sbin/sysdef';
$Ptdpm0::TAIL       = '/usr/bin/tail';
$Ptdpm0::UNAME      = '/bin/uname';
$Ptdpm0::PRTDIAG    = '/sbin/prtdiag';
$Ptdpm0::KVMPRTDIAG = '/usr/kvm/prtdiag';

$Ptdpm0::PLATFORMLOC          = '/usr/platform/';
$Ptdpm0::ETCDEFAULTINITFILE   = '/etc/default/init';
$Ptdpm0::ETCSYSTEMFILE        = '/etc/system';
$Ptdpm0::ETCMNTTABFILE        = '/etc/mnttab';
$Ptdpm0::ETCDEFAULTROUTERFILE = '/etc/defaultrouter';
$Ptdpm0::ETCRESOLVCONFFILE    = '/etc/resolv.conf';
$Ptdpm0::TIMEZONEFILE         = '/etc/TIMEZONE';
$Ptdpm0::PKGLOC               = '/var/sadm/pkg';
$Ptdpm0::VOSRELEASEFILE       = '/etc/opt/SUNWvosr/release';

#******************************************
#     Exported Subroutines
#******************************************

sub setupOSD($)
{
    $Ptdpm0::DATALOC = $_[0];
    if (equalsIgnoreCase($main::OS, 'linux'))
    {
        $Ptdpm0::ARCH          = '/bin/arch';
        $Ptdpm0::ARP           = '/sbin/arp';
        $Ptdpm0::DMESG         = '/bin/dmesg';
        $Ptdpm0::IFCONFIG      = '/sbin/ifconfig';
        $Ptdpm0::LSPCI         = '/sbin/lspci';
        $Ptdpm0::MODINFO       = '/sbin/modinfo';
        $Ptdpm0::NSLOOKUP      = '/usr/bin/nslookup';
        $Ptdpm0::RPM           = '/bin/rpm';
        $Ptdpm0::SYSCTL        = '/sbin/sysctl';
        $Ptdpm0::ETCMNTTABFILE = '/etc/mtab';
        $Ptdpm0::ETCSYSTEMFILE = '/etc/sysctl.conf';
    }
}

sub getOSname
{
    debugTrace('Calling... getOSname()');

    # Purpose: To get the OS name

    my $datadir = $Ptdpm0::DATALOC;
    my $cmd     = NIL;
    my $value   = NIL;
    if ($datadir eq NIL)
    {
        my $rc = checkCmd($Ptdpm0::UNAME);
        if ($rc == 1)
        {
            $cmd = "$Ptdpm0::UNAME -s";
        }
        else
        {
            return $value;
        }
    }
    else
    {
        $cmd = catCmd('unames.out');
    }
    $value = execCmd($cmd);

    return $value;
}

sub getOSName
{
    return $main::OSNAME;
}

sub getOSrelease
{
    debugTrace('Calling... getOSrelease()');

    # Purpose: To get OS Release

    my $datadir = $Ptdpm0::DATALOC;
    my $cmd     = NIL;
    my $value   = NIL;
    if ($datadir eq NIL)
    {
        my $rc = checkCmd($Ptdpm0::UNAME);
        if ($rc == 1)
        {
            $cmd = "$Ptdpm0::UNAME -r";
        }
        else
        {
            return $value;
        }
    }
    else
    {
        $cmd = catCmd('unamer.out');
    }
    $value = execCmd($cmd);

    return $value;
}

sub getOSversion
{
    debugTrace('Calling... getOSversion()');

    # Purpose: To get OS Version

    my $datadir = $Ptdpm0::DATALOC;
    my $cmd     = NIL;
    my $value   = NIL;
    if ($datadir eq NIL)
    {
        my $rc = checkCmd($Ptdpm0::UNAME);
        if ($rc == 1)
        {
            $cmd = "$Ptdpm0::UNAME -v";
        }
        else
        {
            return $value;
        }
    }
    else
    {
        $cmd = catCmd('unamev.out');
    }
    $value = execCmd($cmd);

    return $value;
}

sub getHardwareName
{
    debugTrace('Calling... getHardwareName()');

    # Purpose: To get hardware platform

    my $datadir = $Ptdpm0::DATALOC;
    my $cmd     = NIL;
    my $value   = NIL;
    if ($datadir eq NIL)
    {
        my $rc = checkCmd($Ptdpm0::UNAME);
        if ($rc == 1)
        {
            $cmd = "$Ptdpm0::UNAME -m";
        }
        else
        {
            return $value;
        }
    }
    else
    {
        $cmd = catCmd('unamemi.out');
    }
    $value = execCmd($cmd);

    return $value;
}

sub getProcessorType
{
    debugTrace('Calling... getProcessorType()');

    # Purpose: To get processor type

    my $datadir = $Ptdpm0::DATALOC;
    my $cmd     = NIL;
    my $value   = NIL;
    if ($datadir eq NIL)
    {
        my $rc = checkCmd($Ptdpm0::EGREP) & checkCmd($Ptdpm0::TAIL) & checkCmd($Ptdpm0::CUT);
        if ($rc == 1)
        {
            $cmd =
              "$Ptdpm0::EGREP '^vendor_id[ 	]*:' /proc/cpuinfo | $Ptdpm0::TAIL -1 | $Ptdpm0::CUT -f2 -d:";
        }
        else
        {
            return $value;
        }
    }
    else
    {
        $cmd = catCmd('mach.out');
    }
    $value = execCmd($cmd);

    return $value;
}

sub getArchType
{
    debugTrace('Calling... getArchType()');

    # Purpose: To get  architecture type

    my $datadir = $Ptdpm0::DATALOC;
    my $cmd     = NIL;
    my $value   = NIL;
    if ($datadir eq NIL)
    {
        my $rc = checkCmd($Ptdpm0::ARCH);
        if ($rc == 1)
        {
            $cmd = "$Ptdpm0::ARCH -k";
        }
        else
        {
            return $value;
        }
    }
    else
    {
        $cmd = catCmd('unamem.out');
    }
    $value = execCmd($cmd);

    return $value;
}

sub getHostName
{
    debugTrace('Calling... getHostName()');

    # Purpose: To get hostname

    my $datadir = $Ptdpm0::DATALOC;
    my $cmd     = NIL;
    my $value   = NIL;
    if ($datadir eq NIL)
    {
        my $rc = checkCmd($Ptdpm0::HOSTNAME);
        if ($rc == 1)
        {
            $cmd = "$Ptdpm0::HOSTNAME";
        }
        else
        {
            return $value;
        }
    }
    else
    {
        $cmd = catCmd('hostname.out');

    }
    $value            = execCmd($cmd);
    $Ptdpm0::hostname = $value;

    return $value;
}

sub getDomainName
{
    debugTrace('Calling... getDomainName()');

    # Purpose: To get domain name

    my $datadir = $Ptdpm0::DATALOC;
    my $cmd     = NIL;
    my $value   = NIL;
    my @domarr  = ();
    if ($datadir eq NIL)
    {

        #Changed to fix the bug 2226505
        my $rc = checkCmd($Ptdpm0::NSLOOKUP);
        if ($rc == 1)
        {
            $cmd = "$Ptdpm0::NSLOOKUP $Ptdpm0::hostname 2>&1";
        }
        else
        {
            return @domarr;
        }
    }
    else
    {
        $cmd = catCmd('nslookuphost.out');
    }

    #my $dom =`$cmd 2>$Ptdpm0::DEVNULL`;
    $value = evalCmd($cmd);
    my @dom = split ("\n", $value);
    @domarr = grep /$Ptdpm0::hostname/, @dom;
    chomp(@domarr);

    return @domarr;
}

sub getHardwareProvider
{
    debugTrace('Calling... getHardwareProvider()');

    # Purpose: To get hardware vendor

    my $datadir = $Ptdpm0::DATALOC;
    my $cmd     = NIL;
    my $value   = NIL;
    my @info    = ();
    if ($datadir eq NIL)
    {
        my $rc = checkCmd($Ptdpm0::EGREP) & checkCmd($Ptdpm0::TAIL) & checkCmd($Ptdpm0::CUT);
        if ($rc == 1)
        {
            $cmd =
                "$Ptdpm0::EGREP '^vendor_id[ 	]*:' /proc/cpuinfo | $Ptdpm0::TAIL -1 | $Ptdpm0::CUT -f2 -d:";
        }
        else
        {
            return @info;
        }
    }
    else
    {
        $cmd = catCmd('showrev.out');
    }
    $value = execCmd($cmd);
    @info = ("Hardware provider: Intel Based Hardware") if ($value =~ m/intel/i);

    return @info;
}

sub getDiskInfo
{
    debugTrace('Calling... getDiskInfo()');

    # Purpose: To get disk space/filesystem space

    my $datadir = $Ptdpm0::DATALOC;
    my $cmd     = NIL;
    my $value   = NIL;
    my @diskinfo = ();
    if ($datadir eq NIL)
    {
        my $rc = checkCmd($Ptdpm0::DF) & checkCmd($Ptdpm0::TAIL) & checkCmd($Ptdpm0::SORT);
        if ($rc == 1)
        {
            $cmd = "$Ptdpm0::DF -lk -x tmpfs -x none | $Ptdpm0::TAIL -n +2 | $Ptdpm0::SORT -u";
        }
        else
        {
            return @diskinfo;
        }
    }
    else
    {
        $cmd = catCmd('dfFufsk.out');
    }
    $value = execCmd($cmd);
    @diskinfo = split ("\n", $value);
    chomp(@diskinfo);

    return @diskinfo;
}

sub getEtcSystemLocation
{
    debugTrace('Calling... getEtcSystemLocation()');

    # Purpose: To confirm the presence of /etc/system

    my $datadir = $Ptdpm0::DATALOC;
    my $value   = $Ptdpm0::ETCSYSTEMFILE;
    if ($datadir ne NIL)
    {
        $value = $datadir . $value;
    }
    if (!(-e "$value"))
    {
        throwECMError('FileNotFound', $value);
        $value = NIL;
    }

    return $value;
}

sub getCCinfo
{
    debugTrace('Calling... getCCinfo()');

    # Purpose: To get certified config info (obsolete)

    my $datadir = $Ptdpm0::DATALOC;

    # For VOS configurations, the release file is at /etc/opt/SUNWvosr/release
    my $releasefile = $Ptdpm0::VOSRELEASEFILE;

    if ($datadir ne NIL)
    {
        $releasefile = $datadir . 'CCrelease.out';
    }
    my @ccdet = ();

    if (-e "$releasefile")
    {
        $ccdet[0] = 'Sun Microsystems, Veritas' . "\n";    ## 11/08/2001
        debugTrace('Opening... ' . $releasefile);
        open(IN5, $releasefile);
        my $i = 1;
        while (<IN5>)
        {
            if (index($_, "Copyright") == -1) { $ccdet[$i] = $_; $i++; }
        }
        close(IN5);
    }
    else
    {

        #print HWInfo "$star The VOS Image release file $releasefile NOT found $star";
    }

    return @ccdet;
}

sub getLoadedKernelModules
{
    debugTrace('Calling... getLoadedKernelModules()');

    # Purpose: To get loaded kernel modules

    my $datadir = $Ptdpm0::DATALOC;
    my $cmd     = NIL;
    my $value   = NIL;
    my @modinfo = ();
    if ($datadir eq NIL)
    {
        my $rc = checkCmd($Ptdpm0::MODINFO);
        if ($rc == 1)
        {
            $cmd = "$Ptdpm0::MODINFO";
        }
        else
        {
            return @modinfo;
        }
    }
    else
    {
        $cmd = catCmd('modinfo.out');
    }
    $value = execCmd($cmd);
    @modinfo = split ("\n", $value);
    chomp(@modinfo);

    return @modinfo;
}

sub getSwapSpaceInfo
{
    debugTrace('Calling... getSwapSpaceInfo()');

    # Purpose: To get swap space

    my $datadir = $Ptdpm0::DATALOC;
    my $cmd     = NIL;
    my $value   = NIL;
    my @swap    = ();
    if ($datadir eq NIL)
    {
        my $rc = checkCmd($Ptdpm0::EGREP) & checkCmd($Ptdpm0::CUT);
        if ($rc == 1)
        {
            $cmd =
                "$Ptdpm0::EGREP '^SwapTotal:' /proc/meminfo | $Ptdpm0::CUT -f2 -d: | $Ptdpm0::CUT -f1 -dk";
        }
        else
        {
            return @swap;
        }
    }
    else
    {
        $cmd = catCmd('swapl.out');
    }
    $value = execCmd($cmd);
    @swap = split ("\n", $value);
    chomp(@swap);

    return @swap;
}

# GIT5 Additions

sub getNICDetails
{
    debugTrace('Calling... getNICDetails()');

    # Purpose: To get network interface card names

    my $datadir = $Ptdpm0::DATALOC;
    my $cmd     = NIL;
    my $value   = NIL;
    my $i       = 0;
    my @nic     = ();
    if ($datadir eq NIL)
    {
        my $rc = checkCmd($Ptdpm0::IFCONFIG);
        if ($rc == 1)
        {
            $cmd = "$Ptdpm0::IFCONFIG -a";
        }
        else
        {
            return @nic;
        }
    }
    else
    {
        $cmd = catCmd('ifconfiga.out');
    }

    # ifconfig -a -s format truncates NIC names to 5 characters:
    #
    # Iface   MTU Met    RX-OK RX-ERR RX-DRP RX-OVR    TX-OK TX-ERR TX-DRP TX-OVR Flg
    # eth0       1500   0 395560808      0      0      0 341080081      0      0      0 BMRU

    # ifconfig -a format doesn't trunctate NIC names, but extraneous lines need pruning
    #
    # eth0      Link encap:Ethernet  HWaddr 00:04:23:A5:EF:4E
    #           inet addr:140.87.0.249  Bcast:140.87.3.255  Mask:255.255.252.0
    #           UP BROADCAST RUNNING MULTICAST  MTU:1500  Metric:1
    #           RX packets:395565003 errors:0 dropped:0 overruns:0 frame:0
    #           TX packets:341083682 errors:0 dropped:0 overruns:0 carrier:0
    #           collisions:0 txqueuelen:1000
    #           RX bytes:825575343 (787.3 Mb)  TX bytes:2608718729 (2487.8 Mb)
    #           Base address:0xdcc0 Memory:fcf60000-fcf80000

    $value = execCmd($cmd);
    my @temp_nic = split ("\n", $value);
    chomp(@temp_nic);
    foreach $value (@temp_nic)
    {
       if ($value =~ m/^\w/)
       {
           $nic[$i++] = $value;
       }
    }

    return @nic;
}

sub getMACAddress($)
{
    debugTrace('Calling... getMACAddress()');

    # Purpose: To get MAC address for a given inet address

    my ($inet)  = @_;
    my $datadir = $Ptdpm0::DATALOC;
    my $cmd     = NIL;
    my $value   = NIL;
    if ($datadir eq NIL)
    {
        my $rc = checkCmd($Ptdpm0::ARP);
        if ($rc == 1)
        {
            $cmd = "$Ptdpm0::ARP $inet 2>&1";
        }
        else
        {
            return $value;
        }
    }
    else
    {
        $cmd = catCmd("arp/$inet.out");
    }
    $value = evalCmd($cmd);

    return $value;
}

sub getInstructionSetList
{
    debugTrace('Calling... getInstructionSetList()');

    # Purpose: To get the set of instruction lists that can be run Sun SPARC machines

    my $datadir = $Ptdpm0::DATALOC;
    my $cmd     = NIL;
    my $value   = NIL;
    if ($datadir eq NIL)
    {
        my $rc = checkCmd($Ptdpm0::UNAME);
        if ($rc == 1)
        {
            $cmd = "$Ptdpm0::UNAME -m";
        }
        else
        {
            return $value;
        }
    }
    else
    {
        $cmd = catCmd('isalist.out');
    }
    $value = execCmd($cmd);

    return $value;
}

sub getIPCSStatus
{
    debugTrace('Calling... getIPCSStatus()');

    # Purpose: To report inter-process communication facilities status

    my $cmd   = NIL;
    my $value = NIL;
    if (-x "$Ptdpm0::IPCS")
    {
        my $rc = checkCmd($Ptdpm0::IPCS);
        if ($rc == 1)
        {
            $cmd = "$Ptdpm0::IPCS -a";
        }
        else
        {
            return $value;
        }
    }
    $value = execCmd($cmd);

    return $value;
}

sub getCurrSysDef
{
    debugTrace('Calling... getCurrSysDef()');

    # Purpose: To get current system definition

    my $datadir = $Ptdpm0::DATALOC;
    my $cmd     = NIL;
    my $value   = NIL;
    my @sdef    = ();
    if ($datadir eq NIL)
    {
        my $rc = checkCmd($Ptdpm0::SYSCTL);
        if ($rc == 1)
        {
            $cmd = "$Ptdpm0::SYSCTL -a 2>$Ptdpm0::DEVNULL";
        }
        else
        {
            return @sdef;
        }
    }
    else
    {
        $cmd = catCmd('sysdefi.out');
    }
    $value = execCmd($cmd);
    @sdef = split ("\n", $value);
    chomp(@sdef);

    return @sdef;
}

sub getSysConf
{
    debugTrace('Calling... getSysConf()');

    # Purpose: To get system configuration values

    my $datadir = $Ptdpm0::DATALOC;
    my $cmd     = NIL;
    my $value   = NIL;
    my @sdef    = ();
    if ($datadir eq NIL)
    {
        my $rc = checkCmd($Ptdpm0::GETCONF);
        if ($rc == 1)
        {
            $cmd = "$Ptdpm0::GETCONF";
        }
        else
        {
            return @sdef;
        }
    }
    else
    {
        $cmd = catCmd('getconfa.out');
    }

    $cmd   = "$Ptdpm0::GETCONF OPEN_MAX";
    $value = execCmd($cmd);
    $sdef[0] = "OPEN_MAX:$value";
    $cmd   = "$Ptdpm0::GETCONF CHILD_MAX";
    $value = execCmd($cmd);
    $sdef[1] = "CHILD_MAX:$value";
    chomp(@sdef);

    return @sdef;
}

sub getGetConfig($)
{
    debugTrace('Calling... getGetConfig()');

    # Purpose: To get value for given system configuration setting

    my ($parm)  = @_;
    my $datadir = $Ptdpm0::DATALOC;
    my $cmd     = NIL;
    my $value   = NIL;
    if ($datadir eq NIL)
    {
        my $rc = checkCmd($Ptdpm0::GETCONF);
        if ($rc == 1)
        {
            $cmd = "$Ptdpm0::GETCONF $parm";
        }
        else
        {
            return $value;
        }
    }
    else
    {
        $cmd = catCmd('getconfa.out');
    }
    $value = execCmd($cmd);

    return $value;
}

sub getKernelDriverTunables($)
{
    debugTrace('Calling... getKernelDriverTunables()');

    # Purpose: To get kernel driver tunables, only parameters alone for a given protocol

    my ($protocol) = @_;
    my $datadir  = $Ptdpm0::DATALOC;
    my $cmd      = NIL;
    my $value    = NIL;
    my @parm     = ();
    if ($datadir eq NIL)
    {
        my $rc = checkCmd($Ptdpm0::NDD);
        if ($rc == 1)
        {
            $cmd = "$Ptdpm0::NDD $protocol \\?";
        }
        else
        {
            return @parm;
        }
    }
    else
    {
        my @temp = split ("\/", $protocol);
        $cmd = catCmd("ndddev/$temp[2].out");
    }
    $value = execCmd($cmd);
    @parm = split ("\n", $value);
    chomp(@parm);

    return @parm;
}

sub getKernelDriverTunableValues($)
{
    debugTrace('Calling... getKernelDriverTunableValues()');

    # Purpose: To get values for the given set of kernel driver tunable parameters for a given protocol

    my ($parmlist) = @_;
    my $datadir  = $Ptdpm0::DATALOC;
    my $cmd      = NIL;
    my $value    = NIL;
    my @parm     = ();
    if ($datadir eq NIL)
    {
        my $rc = checkCmd($Ptdpm0::NDD);
        if ($rc == 1)
        {
            $cmd = "$Ptdpm0::NDD $parmlist";
        }
        else
        {
            return @parm;
        }
    }
    else
    {
        my @temp1 = split (' ', $parmlist, 2);
        my @temp2 = split ('_', $temp1[1], 2);
        my $temp = $temp2[0] . 'values';
        $cmd = catCmd("ndddev/$temp.out");
    }
    $value = execCmd($cmd);
    @parm = split ("\n", $value);
    chomp(@parm);

    return @parm;
}

sub getEtcMnttabLocation
{
    debugTrace('Calling... getEtcMnttabLocation()');

    # Purpose: To confirm the presence of /etc/mnttab

    my $datadir = $Ptdpm0::DATALOC;
    my $value   = $Ptdpm0::ETCMNTTABFILE;
    if ($datadir ne NIL)
    {
        $value = $datadir . $value;
    }
    if (!(-e "$value"))
    {
        throwECMError('FileNotFound', $value);
        $value = NIL;
    }

    return $value;
}

sub getGlobalVariables
{
    debugTrace('Calling... getGlobalVariables()');

    # Purpose: To get global variables

    my $datadir = $Ptdpm0::DATALOC;
    my $value   = $Ptdpm0::ETCDEFAULTINITFILE;
    if ($datadir ne NIL)
    {
        $value = $datadir . $value;
    }
    if (!(-e "$value"))
    {
        $value = NIL;
    }

    return $value;
}

sub getDefaultRouter
{
    debugTrace('Calling... getDefaultRouter()');

    # Purpose: To get default router

    my $datadir = $Ptdpm0::DATALOC;
    my $value   = $Ptdpm0::ETCDEFAULTROUTERFILE;
    if ($datadir ne NIL)
    {
        $value = $datadir . $value;
    }
    if (!(-e "$value"))
    {
        $value = NIL;
    }

    return $value;
}

sub getResolvConf
{
    debugTrace('Calling... getResolvConf()');

    # Purpose: To get name servers

    my $datadir = $Ptdpm0::DATALOC;
    my $value   = $Ptdpm0::ETCRESOLVCONFFILE;
    if ($datadir ne NIL)
    {
        $value = $datadir . $value;
    }
    if (!(-e "$value"))
    {
        $value = NIL;
    }

    return $value;
}

sub getPkginfoFilesLocation
{
    debugTrace('Calling... getPkginfoFilesLocation()');

    # Purpose: To confirm the presence of /var/sadm/pkg

    my $datadir = $Ptdpm0::DATALOC;
    my $value   = $Ptdpm0::PKGLOC;
    if ($datadir ne NIL)
    {
        $value = $datadir . 'pkg';
    }
    if (!(-d "$value"))
    {

        # When /var/sadm/pkg dir is NOT present
        # pkginfo: neither returns output nor error
        # showrev: opendir
        throwECMError('PackNotFound', $Ptdpm0::PKGLOC);
        $value = NIL;
    }

    return $value;
}

sub getAdminEntries($)
{
    debugTrace('Calling... getAdminEntries()');

    # Purpose: To get the entries from administrative database for given database and value

    my ($parm)  = @_;
    my $datadir = $Ptdpm0::DATALOC;
    my $cmd     = NIL;
    my $value   = NIL;
    if ($datadir eq NIL)
    {
        if (-x "$Ptdpm0::GETENT")
        {
            $cmd = "$Ptdpm0::GETENT $parm";
        }
        else
        {
            return $value;
        }
    }
    else
    {
        my $temp1 = NIL;
        my $temp2 = NIL;
        ($temp1, $temp2) = split (' ', $parm, 2);
        $cmd = catCmd("getent/$temp2.out");
    }
    $value = evalCmd($cmd);

    return $value;
}

#******************************************
#   Non-Exported Subroutines
#******************************************

sub catCmd($)
{
    return join ($Ptdpm0::DATALOC, $Ptdpm0::CAT . ' ', $_[0]);
}

sub checkCmd($)
{
    # Purpose: To check whether the given command is present and executable

    my ($cmd) = @_;
    my $rc  = 1;
    if (!(-x "$cmd"))
    {
        $rc = 0;
        throwECMError('CmdNotFound', $cmd);
    }

    return $rc;
}

sub execCmd($)
{
    # Purpose: To execute the given command checking for errors

    my ($cmd) = @_;
    my $value = NIL;
    debugTrace('Executing... ' . $cmd);
    chomp($value = `$cmd` || throwECMError('Failed', $cmd));

    return $value;
}

sub evalCmd($)
{
    # Purpose: To execute the given command w/o checking for errors

    my ($cmd) = @_;
    my $value = NIL;
    debugTrace('Executing... ' . $cmd);
    chomp($value = `$cmd`);

    return $value;
}

1;

# End of the Program
