# $Header: emagent/sysman/admin/scripts/Ptdpm1.pm /st_emagent_10.2.0.4.2db11.2/1 2008/11/07 01:47:21 nparaddi Exp $
#
# Package : Ptdpm1.pm
#
# Copyright (c) 2002, 2008, Oracle and/or its affiliates.All rights reserved. 
#
#   NAME
#       Ptdpm1.pm
#
#   DESCRIPTION
#       Collects hardware data
#
#   NOTES
#
#   MODIFIED     (MM/DD/YY)
#      nparaddi   03/13/08 - Backport nparaddi_bug1_6487348 from
#                            st_emcore_10.2.0.1.0
#      rrawat     07/11/06 - Backport rrawat_bug-4771471 from main 
#      rrawat     01/05/06 - Bug-4771471
#      mgoodric   05/10/05 - increase buffer read for dmesg 
#      mgoodric   05/03/05 - fix roundup on FREQUENCY_IN_MHZ 
#      mgoodric   11/03/04 - fix HyperThreading CPU count 
#      mgoodric   09/27/04 - code cleanup 
#      mgoodric   04/01/04 - Linux cleanup
#      mgoodric   03/19/04 - Fix rounding to 2 decimals
#      mgoodric   03/18/04 - Fix MEMORY_SIZE_IN_MB calculation
#      rlal       03/05/04 - Cpu vendor name change for ia64 linux
#      rlal       01/26/04 - Fix for OS name and cut
#      rlal       01/13/04 - Fix for 3267886
#      mgoodric   09/25/03 - Fix error calculating <MEMORY_SIZE_IN_MB> for 1GB
#      mgoodric   09/19/03 - Fix error collecting IO Cards
#      mgoodric   05/22/03 - Bug 2971407
#      mgoodric   05/11/03 - added -debug tracing
#      mgoodric   04/07/03 - fix finding MAC address
#      rlal       02/25/03 - EM 4.0 changed for Linux
#      goramamo   12/12/02 - Changes for SPARC III+ machines
#      goramamo   10/10/02 - Verifying subcomponents to avoid total failure
#      goramamo   09/03/02 - NIC Details collction moved to Ptdpm14
#      goramamo   08/21/02 - Enable hostname alias collection again.
#      goramamo   08/02/02 - Collect all hostnames and aliases
#      goramamo   07/26/02 - To fix MAC Display
#      goramamo   07/12/02 - To handle NIC log. intf
#      goramamo   06/26/02 - Read sys admin tables for domain
#      xxu        06/25/02 - remove /usr/local/bin/perl
#      goramamo   06/02/02 - GIT5: NIC, MAC, OSBit, Mnttab additions
#      goramamo   05/02/02 - OS Subroutines moved to Ptdpm7.pm
#      goramamo   04/14/02 - Added code to check pkginfo files
#      goramamo   04/12/02 - Added code to handle Sun Fire
#                            and Blade series
#      goramamo   03/28/02 - Added code to generte XML from files
#      goramamo   03/05/02 - Added OS_MAX_SWAP_SPACE_IN_MB;
#                            Dropped 12 fields (8 tags);
#      goramamo   02/28/02 - Bug 2226505 fix
#      jmansur    02/15/02 - only keep last if duplicate kernel param assignments
#      mgoodric   02/14/02 - Make scripts more portable
#      jmansur    01/23/02 - collect OS kernel parameters, a la kernel.pl
#      goramamo   12/14/01 - Added TOTAL_LOCAL_DISK_SPACE_IN_GB,
#                            OS_BASE_VERSION and OS_UPDATE_LEVEL
#      goramamo   11/08/01 - Modified code to display hardware provider uniformly
#                            everywhere for Sun Microsystems
#      goramamo   10/30/01 - Modified date format to YYYY-MM-DD
#      goramamo   10/15/01 - Creation
#
##*************************************************************

use strict;

package Ptdpm1;

use Carp;
require 5.005;

#require 5.6.1;
require Exporter;

use ecmErrorMsgs;
use Ptdpm0;
use Ptdpm2;
use Ptdpm3;
use Ptdpm14;
use Ptdpm15;
use Ptdpm16;

#******************************************
#   Export Subroutines
#******************************************

@Ptdpm1::ISA    = ('Exporter');
@Ptdpm1::EXPORT = qw(
  &getSysSummary
  &getHWDetails
  &getCPUDetails
  &getIODDetails
  );

#******************************************
#     Global Variables
#******************************************

use constant NOT_FOUND      => -1;
use constant DETAIL_SP      => '  ';
use constant COLON_CHR      => ':';
use constant DOT_CHR        => '.';
use constant UNDERLINE_CHR  => '_';
use constant NIL            => '';

$Ptdpm1::arch    = NIL;
@Ptdpm1::CPUdata = ();
@Ptdpm1::IODdata = ();

#******************************************
#   Exported Subroutines
#******************************************

sub getSysSummary
{
    debugTrace('Calling... getSysSummary()');

    #Arguments: none
    #Outputs  : array containing system summary
    #Function : returns array containing system summary

    my @syssum = ();
    my $domain = NIL;
    my $value  = NIL;
    my $j      = 0;

    my $hostname = getHostName();
    my $host     = left(DOT_CHR, $hostname);

    my @admnt = split (' ', getAdminEntries("hosts $host"));
    foreach $value (@admnt)
    {
        if ($value =~ m/^$host[.]/)
        {
            $domain = right(DOT_CHR, $value);
            last;
        }
    }
    if ($domain eq NIL)
    {
        my @admnt = getDomainName();
        foreach $value (@admnt)
        {
            if ($value =~ m/^Name:\s*$host[.]/)
            {
                $domain = right(DOT_CHR, $value);
                last;
            }
        }
    }

    my @cc = getCCDetails();

    $j = 0;
    $syssum[$j++] = joinValue($Ptdpm2::HOSTNAME_TAG, $hostname);
    $syssum[$j++] = joinValue($Ptdpm2::DOMAIN_TAG,   $domain);
    $syssum[$j++] = $cc[0];
    $syssum[$j++] = $cc[1];
    $syssum[$j++] = $cc[2];

    return @syssum;
}

sub getHWDetails
{
    debugTrace('Calling... getHWDetails()');

    #Arguments: none
    #Outputs  : array containing hardware details
    #Function : get hardware details, filters it and returns required data

    my @hwdata  = getHardwareDetails();

    return @hwdata;
}

sub getCPUDetails
{
    debugTrace('Calling... getCPUDetails()');

    return @Ptdpm1::CPUdata;
}

sub getIODDetails
{
    debugTrace('Calling... getIODDetails()');

    return @Ptdpm1::IODdata;
}

#******************************************
#   Non-exported Subroutines
#******************************************

sub getCCDetails
{
    debugTrace('Calling... getCCDetails()');

    #Arguments: none
    #Outputs  : array containing CC details
    #Function : reads CC related files and return the version and build date

    my @ccdet  = getCCinfo();
    my @ccdet2 = ();
    my $j      = 0;

    for ($j = 0 ; $j < 3 ; $j++)
    {
        $ccdet[$j] = NIL if (!defined $ccdet[$j]);
    }
    $ccdet[0] = joinValue($Ptdpm2::CC_PARTNER_INFO_TAG,     $ccdet[0]);
    $ccdet[1] = joinValue($Ptdpm2::CC_SOFTWARE_VERSION_TAG, $ccdet[1]);
    @ccdet2 = split (' ', $ccdet[2]);
    ### Date conversion to be done
    $ccdet2[2] = NIL;
    $ccdet[2]  = joinValue($Ptdpm2::CC_SOFTWARE_BUILD_DATE_TAG, $ccdet2[2]);

    return @ccdet;
}

sub getFSB
{
    # ..... host bus clock speed is 133.3374 MHz.

    my $clock_MHz = NIL;

    my @dmesg = `$Ptdpm0::DMESG -s131072 | $Ptdpm0::EGREP ' host bus clock speed is '`;
    chomp(@dmesg);

    if ($dmesg[0] ne NIL)
    {
        my $bus_MHz = $dmesg[0];
        $bus_MHz =~ s/[^\d]+(\d+[.]*\d*)\s*MHz.*/$1/i;
        if ($bus_MHz ne $dmesg[0])
        {
            # convert CLOCK_FREQUENCY_IN_MHZ to integer
            $clock_MHz = int($bus_MHz);
        }
    }

    return $clock_MHz;
}

sub getHardwareDetails
{
    debugTrace('Calling... getHardwareDetails()');

    #Arguments: none
    #Outputs  : array containing formatted hardware details
    #Function : more text processing

    ## This subroutine is not exported

    my @hwdata    = ();
    my @cpudata   = ();
    my @ioddata   = ();
    my @nicdata   = ();

    my $NoOfBrd   = 0;
    my $NoOfCPU   = 0;
    my $NoOfCards = 0;
    my $NoOfFan   = NIL;
    my $NoOfPower = NIL;

    my $id        = NIL;
    my $name      = NIL;
    my $value     = NIL;
    my $numbr     = 0;
    my $size      = 0;

    my $i = 0;
    my $j = 0;
    my $n = 0;
    my $u = 0;
    my $w = 0;

    my $model        = NIL;
    my $memory       = NIL;
    my $clock_MHz    = NIL;
    my $vendor_id    = NIL;
    my $cpu_family   = NIL;
    my $model_name   = NIL;
    my $cpu_MHz      = 0;
    my $cache_size   = 0;
    my $physical_id  = 0;
    my $siblings     = 0;
    my $freq_MHz     = 0;
    my $revision     = NIL;
    my $type         = NIL;
    my $key          = NIL;
    my %hyperThreads = (); 

    my @vendor1 = getHardwareProvider();
    my @hwpro   = split ($Ptdpm2::SEP, $vendor1[0]);

    my $mach = getInstructionSetList();    # Added on 01/07/2002
    if (index($mach, 'sparcv9') > NOT_FOUND)
    {
        $mach = '64-bit sparcv9';
    }
    else
    {
        $mach = getProcessorType();
    }

    $Ptdpm1::arch = getArchType();
    my $arch = join (' ', $mach, $Ptdpm1::arch);    # Added on 01/07/2002

    my $file = '/proc/cpuinfo';

    debugTrace('Opening... ' . $file);
    open(CPUINFO, $file);
    LOOP: while (<CPUINFO>)
    {
        # processor       : 0
        # vendor_id       : GenuineIntel
        # cpu family      : 15
        # model           : 2
        # model name      : Intel(R) Pentium(R) 4 CPU 3.06GHz
        # stepping        : 9
        # cpu MHz         : 3066.889
        # cache size      : 512 KB
        # physical id     : 0
        # siblings        : 2
        # runqueue        : 0
        # fdiv_bug        : no
        # hlt_bug         : no
        # f00f_bug        : no
        # coma_bug        : no
        # fpu             : yes
        # fpu_exception   : yes
        # cpuid level     : 2
        # wp              : yes
        # flags           : fpu vme de pse tsc msr pae mce cx8 apic sep mtrr pge mca cmov 
        # bogomips        : 6121.06

        next if (!m/^processor\s*:/);

        $siblings = 0;  # just in case we are not hyperThreading

        while (<CPUINFO>)
        {
            chomp($_);
            if ($_ eq NIL)
            {
                last;
            }
            ($id, $value) = split($Ptdpm2::SEP, $_);
            $id          = trim($id);
            $value       = trim($value);
            $vendor_id   = $value if ($id eq 'vendor_id');
            $cpu_family  = $value if ($id eq 'cpu family');
            $model       = $value if ($id eq 'model');
            $model_name  = $value if ($id eq 'model name');
            $cpu_MHz     = $value if ($id eq 'cpu MHz');
            $cache_size  = $value if ($id eq 'cache size');
            $physical_id = $value if ($id eq 'physical id');
            $siblings    = $value if ($id eq 'siblings');
        }

        if ($siblings != 0)
        {
            $key = $physical_id . $Ptdpm2::SEP . $siblings;
            next if ($hyperThreads{$key}); 
            $hyperThreads{$key} = 1;
        }

        $NoOfCPU++;

        # convert CPU_SPEED_IN_IN_MHZ to integer
        if ($cpu_MHz =~ m/^\d+[.]*\d*$/)
        {
            $cpu_MHz = int($cpu_MHz);
        }
        else
        {
            $cpu_MHz = NIL;
        }
        # convert ECACHE_IN_MB to MB
        if ($cache_size =~ m/^\d+\s*KB$/i)
        {
            $cache_size =~ s/\s*KB//i;
            $cache_size = ($cache_size / 1024);
        }
        elsif ($cache_size =~ m/^\d+\s*MB$/i)
        {
            $cache_size =~ s/\s*MB//i;
        }
        $cpudata[$u++] = joinValue($Ptdpm2::CPU_VENDOR_TAG, $vendor_id);
        $cpudata[$u++] = joinValue($Ptdpm2::CPU_SPEED_IN_MHZ_TAG, $cpu_MHz);
        $cpudata[$u++] = joinValue($Ptdpm2::ECACHE_IN_MB_TAG, $cache_size);
        $cpudata[$u++] = joinValue($Ptdpm2::CPU_IMPLEMENTATION_TAG, $model_name);
        $cpudata[$u++] = joinValue($Ptdpm2::REVISION_TAG, $model);
        $cpudata[$u++] = joinValue($Ptdpm2::MASK_TAG, $cpu_family);
    }

    close(CPUINFO);

    my @iobuf  = ();
    my @iodat1 = ();

    chomp(@iobuf = `$Ptdpm0::LSPCI -vv 2>$Ptdpm0::DEVNULL`);
 
    for ($n = 0 ; $n < scalar(@iobuf) ; $n++)
    {
        # 00:03.1 Class ff00: Dell Remote Access Card III (rev 02)
        #        Subsystem: Dell Remote Access Card III
        #        Control: I/O+ Mem+ BusMaster- SpecCycle- MemWINV- VGASnoop- ParErr- Stepping- SERR+ FastB2B-
        #        Status: Cap+ 66Mhz- UDF- FastB2B+ ParErr- DEVSEL=medium >TAbort- <TAbort- <MAbort- >SERR- 
        #        Interrupt: pin B routed to IRQ 18
        #        Region 0: Memory at fe102000 (32-bit, non-prefetchable) [size=4K]
        #        Region 1: I/O ports at ec80 [size=64]
        #        Region 2: Memory at feb00000 (32-bit, prefetchable) [size=512K]
        #        Capabilities: <available only to root>

        next if (!($iobuf[$n] =~ m/^\s*Subsystem:/i));

        $NoOfCards++;

        @iodat1    = split ($Ptdpm2::SEP, $iobuf[$n]);
        $vendor_id = $iodat1[1];
        @iodat1    = split ($Ptdpm2::SEP, $iobuf[$n - 1]);
        $name      = $iodat1[0] . $Ptdpm2::SEP . $iodat1[1];
        $revision  = NIL;
        if ($iobuf[$n - 1] =~ m/\(rev\s/)
        {
            $revision = left(")", right("(rev ", $iobuf[$n - 1]));
        }
        $freq_MHz  = (split (' ', $iobuf[$n + 2]))[2];
        $freq_MHz  =~ s/^[^0-9]*([0-9]+).*/$1/; 

        $ioddata[$w++] = joinValue($Ptdpm2::VENDOR_TAG, $vendor_id);
        $ioddata[$w++] = joinValue($Ptdpm2::NAME_TAG, $name);
        $ioddata[$w++] = joinValue($Ptdpm2::FREQUENCY_IN_MHZ_TAG, $freq_MHz);
        $ioddata[$w++] = joinValue($Ptdpm2::BUS_TYPE_TAG, "PCI");
        $ioddata[$w++] = joinValue($Ptdpm2::REVISION_TAG, $revision);
    }

    $name = getHardwareName();

    chomp($memory = `$Ptdpm0::EGREP "^MemTotal:" /proc/meminfo`);
    $memory = left("kB", right("MemTotal:", $memory), 1);
    
    # convert MEMORY_SIZE_IN_MB to MB
    $memory = int($memory / 1024);

    $NoOfBrd = 1 if ($NoOfCPU > 0);

    my $diskCap      = 0;
    my $totalDiskCap = 0;
    my $line         = NIL;

    my @diskinfo     = getDiskInfo();
    foreach $line (@diskinfo)
    {
        $_ = $line;
        ($diskCap) = /\w+\s+(\w+)/i;    # get at the disk capacity column
        $totalDiskCap += $diskCap;
    }

    $totalDiskCap = sprintf("%.2f", $totalDiskCap / (1024 * 1024));

    $clock_MHz = getFSB();

    @nicdata = getNetworkInterfaceDetails();

    $j = 0;
    $hwdata[$j++] = joinValue($Ptdpm2::HARDWARE_PROVIDER_TAG, $hwpro[1]);
    $hwdata[$j++] = joinValue($Ptdpm2::SYSTEM_CONFIGURATION_TAG, $name);
    $hwdata[$j++] = joinValue($Ptdpm2::MACHINE_ARCHITECTURE_TAG,  $arch);
    $hwdata[$j++] = joinValue($Ptdpm2::CLOCK_FREQUENCY_IN_MHZ_TAG, $clock_MHz);
    $hwdata[$j++] = joinValue($Ptdpm2::MEMORY_SIZE_IN_MB_TAG, $memory);
    $hwdata[$j++] = joinValue($Ptdpm2::TOTAL_LOCAL_DISK_SPACE_IN_GB_TAG, $totalDiskCap);
    $hwdata[$j++] = joinValue($Ptdpm2::NUMBER_OF_CPUS_TAG, $NoOfCPU);
    $hwdata[$j++] = joinValue($Ptdpm2::NUMBER_OF_CPU_BOARDS_TAG, $NoOfBrd);
    $hwdata[$j++] = joinValue($Ptdpm2::NUMBER_OF_IO_DEVICES_TAG, $NoOfCards);
    $hwdata[$j++] = joinValue($Ptdpm2::NUMBER_OF_FANS_TAG, $NoOfFan);
    $hwdata[$j++] = joinValue($Ptdpm2::NUMBER_OF_POWER_SUPPLIES_TAG, $NoOfPower);

    @hwdata  = trim1DimArrayAfterSplit(@hwdata);

    @hwdata = verifyNotNullFields($Ptdpm2::MGMT_HC_HARDWARE_MASTER, \@hwdata);
    $size   = scalar(@hwdata);
    if ($size > 0)
    {
        @hwdata = verifyNumericFields($Ptdpm2::MGMT_HC_HARDWARE_MASTER, \@hwdata);
        $size = scalar(@hwdata);
    }

    @cpudata = trim1DimArrayAfterSplit(@cpudata);
    @cpudata = verifyNotNullFields($Ptdpm2::MGMT_HC_CPU_DETAILS, \@cpudata);
    $size = scalar(@cpudata);
    if ($size > 0)
    {
        @cpudata = verifyNumericFields($Ptdpm2::MGMT_HC_CPU_DETAILS, \@cpudata);
        $size = scalar(@cpudata);
        for ($i = 0 ; $i < $size ; $i++)
        {
            $hwdata[$j++] = $cpudata[$i];
        }
    }

    @ioddata = trim1DimArrayAfterSplit(@ioddata);
    @ioddata = verifyNotNullFields($Ptdpm2::MGMT_HC_IOCARD_DETAILS, \@ioddata);
    $size = scalar(@ioddata);
    if ($size > 0)
    {
        @ioddata = verifyNumericFields($Ptdpm2::MGMT_HC_IOCARD_DETAILS, \@ioddata);
        $size = scalar(@ioddata);
        for ($i = 0 ; $i < $size ; $i++)
        {
            $hwdata[$j++] = $ioddata[$i];
        }
    }

    @nicdata = trim1DimArrayAfterSplit(@nicdata);
    @nicdata = verifyNotNullFields($Ptdpm2::MGMT_HC_NIC_DETAILS, \@nicdata);
    $size    = scalar(@nicdata);
    if ($size > 0)
    {
        @nicdata = verifyNumericFields($Ptdpm2::MGMT_HC_NIC_DETAILS, \@nicdata);
        $size = scalar(@nicdata);
        for ($i = 0 ; $i < $size ; $i++)
        {
            $hwdata[$j++] = $nicdata[$i];
        }
    }

    @Ptdpm1::CPUdata = @cpudata;
    @Ptdpm1::IODdata = @ioddata;

    return @hwdata;
}

1;

# End of the Program
