#!/usr/bin/perl
# 
# $Header: has/install/crsconfig/rootcrs.pl /st_has_11.2.0.1.0/9 2010/03/23 14:06:12 dpham Exp $
#
# rootcrs.pl
# 
# Copyright (c) 2008, 2010, Oracle and/or its affiliates. All rights reserved. 
#
#    NAME
#      rootcrs.pl - root configuration script for Oracle Clusterware home
#
#    DESCRIPTION
#      rootcrs.pl - root configuration script for Oracle Clusterware home
#
#    NOTES
#      This is run once during configuration or patching of Oracle
#      clusterware home.
#      This script does the following:
#      1) Setup permissions of binaries and DSOs in Oracle Clusterware home
#      2) Setup OLR for storing Oracle local registry data
#      3) Setup GPnP wallet and profile
#      4) Setup and copy files for CLSINIT daemon
#      5) Start CLSINIT daemon
#      6) Copy required CRS resources for CLSINIT to start
#
#      NOTE: This is run during Oracle Clusterware home configurations only.
#            It should not be used for any other homes such as SI HA home,
#            or Oracle RAC homes.
#
#    MODIFIED   (MM/DD/YY)
#    dpham       03/18/10 - Fix bug 9282344
#    dpham       01/18/10 - Call s_getAuthorizedOwner()
#    dpham       12/07/09 - Increase wait time to 30 mins for ohasd & crsd
#    dpham       11/17/09 - Increase wait time for ohasd to start
#    dpham       11/05/09 - Increase wait time for crsd and evmd to start
#    			    from 24 to 48 (9062113)
#    dpham       10/15/09 - Add '-remotenode' option
#    dpham       09/25/09 - Check return code from configureAllRemoteNodes()
#    dpham       07/13/09 - Add 'no_auto_abbrev' to Getopt() function
#    ksviswan    06/26/09 - change default perl path
#    dpham       06/17/09 - Add '-deconfig' option
#    hchau       05/28/09 - Rewrite Step 9 to clean up the code for stack
#                           startup
#    dpham       05/25/09 - Call 'clscfg -lastnode' if it's last node to upgrade
#    dpham       05/21/09 - Add '-lastnode' option in "clscfg -upgrade"
#    hchau       05/12/09 - Start ctssd resource with env var CTSS_REBOOT=TRUE
#                           when starting the stack
#    mimili      05/10/09 - Bug 8496223 and 8488395: remove 'clscfg -upgradeevm', which 
#                           is only used for OCR permission problem in evmd upgrade
#    ksviswan    05/07/09 - Implement downgrade
#    dpham       05/12/09 - Call getUpgradeConfig, configNode, backupOLR
#    dpham       05/05/09 - update_ons_config() function should be called after 
#			    get_OldVipInfo or get_OldVipInfoFromOCRDump
#    dpham       05/05/09 - Backup OLR (8484172)
#    hchau       04/29/09 - Swap startup sequence to 'ctssd' then 'asm'
#    dpham       04/29/09 - Add start_resource(ora.asm) and stop_resource(ora.asm) (8309620).
#			  - Bypass isInterfaceValid() if it's new node (8438342).
#    dpham       04/27/09 - Call s_houseCleaning after successfully upgrade
#    ksviswan    04/25/09 - Add Patching support
#    dpham       04/22/09 - Call isLastNodeToUpgrade for upgrede
#    spavan      03/23/09 - install cvuqdisk rpm as part of root
#    dpham       03/17/09 - Check return code from configLastNode and configNewNode
#    dpham       03/10/09 - Add ORACLE_BASE
#    dpham       03/09/09 - isInterfaceValid() should be called if not UPGRADE
#    jleys       03/07/09 - Remove merge lines
#    jleys       03/07/09 - Remove merge lines
#    agraves     03/06/09 - Add -g option to pass ASMGRP for SYSTEM.ACFS to
#                           clscfg.
#    dpham       02/22/09 - Call isCRSAlreadyConfigured to check if CRS is
#                           already configured
#    jleys       02/06/09 - Remove expansion of VF discovery string
#    dpham       02/26/09 - Remove add_acfs_registry
#    dpham       02/23/09 - Call isInterfaceValid to check if interface is valid
#    dpham       02/22/09 - Call isCRSAlreadyConfigured to check if CRS is already configured
#    dpham       02/11/09 - Call add_acfs_registry
#    sravindh    02/11/09 - Bugfix 7714358
#    dpham       02/04/09 - Add isRAC_appropriate to check for rac_on/rac_off
#    dpham       01/27/09 - Add -unlock option
#    dpham       01/20/09 - Remove configure_ACFS
#    rsreekum    01/16/09 - fix bug 7678799
#    dpham       12/29/08 - Fix syntax error on isLastNodeToStart function
#    ksviswan    12/26/08 - Fix bug 7561694
#    jleys       11/29/08 - Set delete flag in $cfg
#    jleys       11/26/08 - Correct case of method
#    jleys       11/14/08 - Packagize perl scripts
#    dpham       12/11/08 - Remove -firstNode option
#                         - Configure HASD as "NT AUTHORITY\SYSTEM"
#                         - Call "crssetup install" to configure remote nodes
#    dpham       11/24/08 - Add 'clscfg -upgradeevm'
#    dpham       11/13/08 - remove ORA_CRS_HOME param from validate_9iGSD
#    dpham       10/27/08 - add 'srvctl config nodeapps' in get_OldVipInfo
#    psankara    10/23/08 - Use a new crsctl command to obtain VF Disc. string
#    ysharoni    10/20/08 - save results of make_vf_discovery_string in %param
#    rdasari     10/15/08 - remove the temporary changes to copy onsctl
#    dpham       10/09/08 - Trap error output from usm_root so that it gets written
#                           to the CRS log.
#    dpham       10/08/08 - The rootcrs.log is overwritten if ORACLE_HOME is shared.
#    jtellez     10/08/08 - fix asm start w/ storage_option==2
#    jleys       09/24/08 - Do not complete if initial config not successful
#    ppallapo    09/22/08 - Get OCRID and CLUSTER_GUID for GPnP Profile.
#    rdasari     09/12/08 - mv CH/bin/onsctl CH/opmn/bin/onsctli
#    dpham       08/12/08 - OCRCONFIGBIN is undefined.
#    jleys       07/11/08 - Put this program path into perl search path
#    jleys       07/07/08 - Correct typo
#    jleys       04/09/08 - Change css_start to CSS_start
#    jleys       04/05/08 - Change CSS package name
#    jleys       04/01/08 - Consolidate gpnp setup
#    jleys       03/29/08 - Make it easier to get debugging output
#    jleys       02/29/08 - Break out CSS functionality
#    dpham       06/27/08 - pass trace parameters to hasdconfig.pl (bug 6903750)
#    dpham       06/17/08 - fix usmca (bug 7159411)
#    dpham       06/16/08 - Add new node logic
#    averhuls    06/04/08 - Change externally visible USM text to ACFS.
#    srisanka    05/13/08 - replace ORA_CRS_HOME with ORACLE_HOME
#    averhuls    05/06/08 - Add usmfs.Add usmfs
#    samjo       05/02/08 - Use hostname in olr file name
#    hkanchar    05/05/08 - Remove CRS_HOME_ENV for upgrade
#    samjo       05/02/08 - Use hostname in olr file name
#    rxkumar     04/30/08 - remove srvctl add asm
#    srisanka    04/30/08 - Bug 7010466: fix typo
#    srisanka    04/30/08 - pass ASM_DISCOVERY_STRINGto USMCA only if non-NULL
#    dpham       04/28/08 - Add '-delete' for root deconfig  
#    jleys       04/21/08 - Add function call to determine if this is the last
#                           node
#    priagraw    04/16/08 - add gipcd agent start 
#    srisanka    04/09/08 - use USMCA path
#    samjo       04/03/08 - Pass owner and group during OLR format
#    ysharoni    03/31/08 - bug 6895319
#    skakarla    03/27/08 - fixing srvctl path
#    hkanchar    03/30/08 - Refactor upgrade scripts
#    srisanka    03/18/08 - handle stdout/stderr
#    samjo       02/25/08 - Add CTSSD start
#    jtellez     02/20/08 - trace files
#    jtellez     02/18/08 - srvctl clobbers rootcrs.pl output
#    srisanka    02/12/08 - do OSD actions
#    yizhang     02/18/08 - Fix bug 6822550
#    skakarla    01/18/08 - adding upgrade specific changes
#    srisanka    01/09/08 - separate generic and OSD code
#    jleys       01/22/08 - Tolerate failure to star in exclusive mode
#    jachang     01/13/08 - Modify the order of clscfg in script
#    jleys       01/07/08 - Remove cssvfupgd
#    yizhang     01/08/08 - Add scan listener as oracle owner
#    khsingh     01/08/08 - Add asm resource
#    yizhang     01/02/08 - Start scan listener after adding scan listener
#    ysharoni    12/27/07 - Static CSS_LEASEDURATION and ASM_SPFILE to paramf
#    jleys       12/21/07 - Workaround the lack of VF discovery string
#    yizhang     12/10/07 - Add scan and scan_listener
#    minzhu      12/19/07 - update css mode
#    ysharoni    12/14/07 - gpnp cont-d
#    jachang     12/07/07 - Add exclusive mode processing
#    samjo       12/07/07 - Making OCR and OLR names consistent
#    srisanka    11/30/07 - bug 6661111: fix CRS_NODEVIPS parsing
#    jachang     11/29/07 - Adding exclusive mode steps
#    ysharoni    11/28/07 - Add gpnp
#    srisanka    11/19/07 - add use Net::Ping
#    srisanka    08/23/07 - Creation
# 

################ Documentation ################

# The SYNOPSIS section is printed out as usage when incorrect parameters
# are passed

=head1 NAME

  rootcrs.pl - Configure Oracle Clusterware

=head1 SYNOPSIS

  rootcrs.pl [-verbose] [-upgrade | -patch] [-hahome <directory>]
             [-paramfile <parameter-file>] 
             [-deconfig | -downgrade] [-force] [-lastnode]
             [-downgrade] [-oldcrshome <old crshome path>] [-version <old crs version>]  
             [-unlock [-crshome <path to crs home>]]

  Options:
   -verbose    Run this script in verbose mode
   -upgrade    Oracle HA is being upgraded from previous version
   -patch      Oracle HA is being upgraded to a patch version
   -hahome     Complete path of Oracle Clusterware home
   -paramfile  Complete path of file specifying HA parameter values
   -lastnode   Force the node this is executing on to be considered the
               last node of the install and perform actions associated
               with configuring the last node
   -downgrade  Downgrade the clusterware
   -version    For use with downgrade; special handling is required if
               downgrading to 9i. This is the old crs version in the format
               A.B.C.D.E (e.g 11.1.0.6.0).
   -deconfig   Remove Oracle Clusterware to allow it to be uninstalled or reinstalled.
   -force      Force the execution of steps in delete that cannot be verified 
	       to be safe
   -unlock     Unlock CRS home 
   -crshome    Complete path of crs home. Use with unlock option.
   -oldcrshome For use with downgrade. Complete path of the old crs home.


  If neither -upgrade nor -patch is supplied, a new install is performed

  To see the full manpage for this program, execute:
    perldoc rootcrs.pl

=head1 DESCRIPTION

  This script performs the operations necessary to install the Oracle
  Clusterware stack on a node of a cluster.  It must be run once on
  each node and it must be run by an authorized user, e.g. root on UNIX
  platforms, Administrator on Windows.

  Upon successful completion on each node, the Oracle Clusterware will
  be executing on that node.

=cut

################ End Documentation ################

use strict;
use English;
use File::Basename;
use File::Spec::Functions;

BEGIN {
  # Add the directory of this file to the search path
  push @INC, dirname($PROGRAM_NAME);
}

use Net::Ping;
use Getopt::Long qw(:config no_auto_abbrev);
use Pod::Usage;
use crsconfig_lib;
require crsdelete;
require crspatch;

use oracss;

my $OLD_CRS_HOME;

# Global variables
our $g_force = 0;
our $g_delete = 0;
our $g_lastnode = 0;
our $g_downgrade = 0;
our $g_unlock = 0;
our $g_version;
our $g_help = 0;
our $g_patch = 0;

our $DEBUG;

# pull all parameters defined in crsconfig_params and s_crsconfig_defs (if
# it exists) as variables in Perl
my $paramfile_default = catfile (dirname ($0), "crsconfig_params");

# pull all parameters defined in crsconfig_addparams
my $addparams = catfile (dirname ($0), "crsconfig_addparams");

# pull all definitions in s_crsconfig_defs (if it exists) as variables in Perl
# this file might not exist for all platforms
my $defsfile = catfile (dirname ($0), "s_crsconfig_defs");

my $UPGRADE;
my $firstNode;
my $hahome;
my $PARAM_FILE_PATH = $paramfile_default;
my $unlock_crshome;
my $DOWNGRADE;
my $oldcrshome;
my $oldcrsver;
my $REMOTENODE;

### Parse command line args
# If an incorrect option is specified, the perl POD at the beginning
# of the file is parsed into a man page

=head2 FIXME: Supported parameters

The following paramters do not appear to be acutally supported:
  hahome
  paramfile
  patch

=cut

# the return code to give when the incorrect parameters are passed
my $usage_rc = 1;

GetOptions('verbose!'    => \$DEBUG,
           'upgrade!'    => \$UPGRADE,
           'patch!'      => \$g_patch,
           'hahome=s'    => \$hahome,
           'paramfile=s' => \$PARAM_FILE_PATH,
           'delete'      => \$g_delete,
           'deconfig'    => \$g_delete,
           'force'       => \$g_force,
           'lastnode'    => \$g_lastnode,
           'downgrade'   => \$DOWNGRADE,
           'version=s'   => \$oldcrsver,
           'unlock'      => \$g_unlock,
           'crshome=s'   => \$unlock_crshome,
           'oldcrshome=s'=> \$oldcrshome,
           'remotenode'  => \$REMOTENODE,
           'help!'       => \$g_help) or pod2usage($usage_rc);

# Check validity of args
pod2usage(-msg => "Invalid extra options passed: @ARGV",
          -exitval => $usage_rc) if (@ARGV);

#
# MAIN SCRIPT BODY
#
local $SIG{'__DIE__'} = sub { dietrap(@_); };

my @cmd;
if (!$DEBUG) { $DEBUG = $ENV{'ORA_INSTALL_DEBUG'}; }

### Set this host name (lower case and no domain name)
our $HOST = tolower_host();
die "$!" if ($HOST eq "");

# Set the following vars appropriately for cluster env
### check if run as super user
our $SUPERUSER = check_SuperUser ();
if (!$SUPERUSER) {
  error("Insufficient privileges to execute this script");
  exit 1;
}

# Read the config files and set up the configuration data for
# subsequent processing
my $cfg =
  crsconfig_lib->new(IS_SIHA             => FALSE,
                     paramfile           => $PARAM_FILE_PATH,
                     osdfile             => $defsfile,
                     addfile             => $addparams,
                     crscfg_trace        => TRUE,
                     CRSDelete           => $g_delete,
                     DEBUG               => $DEBUG,
                     HAS_USER            => $SUPERUSER,
                     HOST                => $HOST,
                     UPGRADE             => $UPGRADE,
                     unlock_crshome      => $unlock_crshome,
                     CRSPatch            => $g_patch,
                     DOWNGRADE           => $DOWNGRADE,
                     oldcrshome          => $oldcrshome,
                     oldcrsver           => $oldcrsver,
                     REMOTENODE          => $REMOTENODE,
                     );

# redirect stdout/stderr as appropriate for Windows only
s_redirect_souterr ($cfg->crscfg_trace_file . "_OUT");

my $ORACLE_HOME  = $cfg->params('ORACLE_HOME');
my $ORACLE_BASE  = $cfg->params('ORACLE_BASE');
my $ORA_CRS_HOME = $ORACLE_HOME;
my $ORA_HA_HOME  = $ORACLE_HOME;
my $asmgrp       = $CFG->params('ORA_ASM_GROUP');

$ENV{'ORA_CRS_HOME'} = $ORA_CRS_HOME;
$ENV{'ORACLE_HOME'}  = $ORACLE_HOME;
$ENV{'ORACLE_BASE'}  = $ORACLE_BASE;

# set some stuff we were not able to set until we had read params
my $CLSCFGBIN = catfile ($ORACLE_HOME, "bin", "clscfg");

if    ($g_help)   { pod2usage(0); }
elsif ($g_delete) { CRSDelete(); }
elsif ($g_unlock) { unlockCRSHome(); }
elsif ($g_patch)  { CRSPatch(); }
elsif ($DOWNGRADE){ CRSDelete(); }
else  {
 
  my $success = TRUE;

  # check if CRS is configured
  if (isCRSAlreadyConfigured()) {
     exit 1;
  }

  # validate RAC_ON/RAC_OFF
  if (! is_dev_env () && ! isRAC_appropriate()) {
     exit 1;
  }

  if (! $UPGRADE &&
     (! isAddNode($HOST, $cfg->params('NODE_NAME_LIST'))) &&
     (! isInterfaceValid())) {
     exit 1;
  }

  # perform platform-specific setup actions
  if (SUCCESS != s_osd_setup ()) {
    error ("Platform-specific setup failed");
    exit 1;
  }

  # If $firstNode is set, perform cluster-wide one-time actions and exit
  if (SUCCESS != first_node_tasks ()) {
    error ("Cluster-wide one-time actions failed");
    exit 1;
  }
  else {
     trace ("Cluster-wide one-time actions... Done!");
  }

  # run directory creation, script instantiation, files creation/permissions
  # modules
  run_env_setup_modules ();

  ## FIXME: This should not be required
  $ENV{'ORA_PCW_AGENTS'} = "true";

  ## Step3: Check if CRS is already configured

  # check_CRSConfig outers
  my $crsConfigured   = FALSE;
  my $gpnp_setup_type = GPNP_SETUP_BAD ;

  check_CRSConfig ($ORA_CRS_HOME, $cfg->HOST,
                   $cfg->params('ORACLE_OWNER'),
                   $cfg->params('ORA_DBA_GROUP'),
                   $cfg->params('GPNPGCONFIGDIR'),
                   $cfg->params('GPNPCONFIGDIR'),
                   $crsConfigured, $gpnp_setup_type)
    or die "check_CRSConfig failed";

  if ($UPGRADE) {
     getUpgradeConfig();
  }

  # In case of upgrade crs stack is always configured. But to upgrade to
  # 11.2 we still need to do same things as we do for fresh install.
  # TODO: going forward when we upgrade from 11.2 to higher versions
  # this step may not be required.
  if (!$crsConfigured || $UPGRADE) {
    ## Step4: Validate existing current configuration on the node
    initial_cluster_validation();

    ## Step5: Create and populate OLR and OCR keys
    olr_initial_config() or die("OLR configuration failed\n");

    # Initialize the SCR settings.
    s_init_scr ();

    # incase of upgrade  voting files would be upgraded from previous
    # version 
    ## create GPnP wallet/profile 
    initialize_local_gpnp($HOST, $gpnp_setup_type);

    ## Step8: Setup OHASD service/daemon
    trace ("Registering ohasd");
    register_service ("ohasd") or die "Can't register ohasd service";

    # Start OHASD service/daemon
    trace ("Starting ohasd");
    start_service ("ohasd") or die "ohasd failed to start";

    # Check if the service/daemon has started
    trace ("Checking ohasd");
    my $ohasd_running = check_service ("ohasd", 360);

    if ($ohasd_running) {
      trace ("ohasd started successfully");
    } else {
      trace ("Timed out waiting for ohasd to start.");
      exit 1;
    }

    ###FIXME: Change once we know types of resources to be created
    trace ("Creating CRS resources and dependencies");
    my $hasdconfig_superuser = $SUPERUSER;
    if ($OSNAME eq 'MSWin32' && is_dev_env ()) {
      $hasdconfig_superuser = $cfg->params('ORACLE_OWNER');
    }

    trace("Configuring HASD");
    my $status;
    if ($cfg->platform_family eq "windows") {
       my $NT_AUTHORITY = '';
       if (is_dev_env()) {
          $NT_AUTHORITY = $cfg->params('ORACLE_OWNER');
       }

       $status = configure_hasd('crs', $HOST, $NT_AUTHORITY,
                                $NT_AUTHORITY, 
                                $cfg->params('ORA_DBA_GROUP'));
    }
    else {
      $status = configure_hasd('crs', $HOST, 
                               $cfg->params('ORACLE_OWNER'),
                               $hasdconfig_superuser,
                               $cfg->params('ORA_DBA_GROUP'));
    }

    if ($status) {
      trace ("Successfully created CRS resources for cluster daemon and ASM");
    } else {
      error ("Failed to create CRS resources for cluster daemon and ASM");
      # TODO: this script has to be idempotent. In case of failure
      # if this script runs second time that time ohasd resource
      # would have been created already. So before creating the
      # resources check for the existence of the resources in ohasd.
      #exit 1;
    }

    # perform CSS config or upgrade, propagate cluster-wide config if needed
    if ($UPGRADE) {
      if (! perform_upgrade_config($gpnp_setup_type)) {
         my $trace_file = $CFG->crscfg_trace_file;
         error ("Cluster configuration upgrade failed.  " . 
		"See $trace_file for details");
         exit 1;
      }
    } elsif (! perform_initial_config()) {
       my $trace_file = $CFG->crscfg_trace_file;
       error ("Initial cluster configuration failed.  " . 
	      "See $trace_file for details");
       exit 1;
    }

    ## Step9: Start Oracle Clusterware stack
    # Start gpnpd
    if ( ! start_clusterware(START_STACK_GPNPD) )
    {
      error ("Failed to start GPnP");
      error ("Failed to start Oracle Clusterware stack");
      exit 1;
    }

    # Upgrade the CSS voting disks.
    if ( ($UPGRADE) && (!CSS_upgrade($cfg)) )
    {
      error ("Failed to upgrade the CSS voting disks ");
      error ("Failed to start Oracle Clusterware stack");
      exit 1;
    }

    # Start CSS in clustered mode
    if ( ! CSS_start_clustered() )
    {
      error ("Failed to start CSS in clustered mode");
      error ("Failed to start Oracle Clusterware stack");
      exit 1;
    }

    # Start CTSS with reboot option to signal step sync
    # Note: Before migrating stack startup to 'crsctl start crs',
    #       'CTSS_REBOOT=TRUE' is a workaround to signal step sync. 
    if ( ! start_resource("ora.ctssd", "-init",
                          "-env", "USR_ORA_ENV=CTSS_REBOOT=TRUE") )
    {
      error ("Failed to start CTSS");
      error ("Failed to start Oracle Clusterware stack");
      exit 1;
    }

    # Start ASM if needed
    if ( ($CRS_STORAGE_OPTION == 1) &&
         (! start_resource("ora.asm", "-init")) )
    {
      error ("Failed to start ASM");
      error ("Failed to start Oracle Clusterware stack");
      exit 1;
    }

    # Start CRS
    if ( ! start_resource("ora.crsd", "-init") )
    {
      error ("Failed to start CRS");
      error ("Failed to start Oracle Clusterware stack");
      exit 1;
    }

    # Start EVM
    if ( ! start_resource("ora.evmd", "-init") )
    {
      error ("Failed to start EVM");
      error ("Failed to start Oracle Clusterware stack");
      exit 1;
    }

    trace ("Successfully started Oracle clusterware stack");
  }

  if (!wait_for_stack_start(360)) { exit 1; }

  # Now execute clscfg -upgrade.
  if ($UPGRADE) {
    my $status;
    if (isLastNodeToUpgrade ($HOST, $CFG->params('NODE_NAME_LIST'))) {
       $status = run_crs_cmd('clscfg', '-upgrade', '-g', $asmgrp, '-lastnode');
    } else {
       $status = run_crs_cmd('clscfg', '-upgrade', '-g', $asmgrp);
    }

    if ($status == 0) {
      trace ("$CLSCFGBIN -upgrade completed successfully");
    } else {
      error ("$CLSCFGBIN -upgrade failed");
      exit 1;
    }
  } elsif (isAddNode($HOST, $cfg->params('NODE_NAME_LIST'))) {
    my $status = run_crs_cmd('clscfg', '-add');
    if ($status == 0) {
      trace ("$CLSCFGBIN -add completed successfully");
    } else {
      error ("$CLSCFGBIN -add failed");
      exit 1;
    }
  }

  # configure node
  $success = configNode();

  if ($success) {
     # for windows only, configure all remote nodes
     $success = configureAllRemoteNodes();
  }

  if ($success) {
     backupOLR();
  }

  configureCvuRpm ();

  if ($success) {
     print  "Configure Oracle Grid Infrastructure for a Cluster ... succeeded\n";
     trace ("Configure Oracle Grid Infrastructure for a Cluster ... succeeded");
  } else {
     print  "Configure Oracle Grid Infrastructure for a Cluster ... failed\n";
     trace ("Configure Oracle Grid Infrastructure for a Cluster ... failed");
     exit 1;
  }
}

# restore stdout/stderr as appropriate
s_restore_souterr ();

0;

