#!/usr/bin/perl
# 
# $Header: has/install/crsconfig/roothas.pl /stpl_db_11.2.0.1.0_win.x64/1 2010/03/26 22:58:21 dpham Exp $
#
# roothas.pl
# 
# Copyright (c) 2007, 2010, Oracle and/or its affiliates. All rights reserved. 
#
################ Documentation ################

# The SYNOPSIS section is printed out as usage when incorrect parameters
# are passed

=head1 NAME

  roothas.pl - root configuration script for SI-HAS

=head1 SYNOPSIS

  roothas.pl [-verbose] [-upgrade | -patch] [-hahome <directory>]
             [-paramfile <parameter-file>] [-delete] [-force]
             [-unlock [-hahome <path to Oracle Restart home>]]

  Options:
   -verbose   Run this script in verbose mode
   -upgrade   Oracle HA is being upgraded from previous version
   -patch     Oracle HA is being upgraded to a patch version
   -hahome    Complete path of Oracle HA home
   -paramfile Complete path of file specifying HA parameter values
   -delete    To deconfigure Oracle Restart
   -unlock    Unlock Oracle Restart home
   -force     To deconfigure Oracle Restart when configuration
              is in a good condition

  If neither -upgrade nor -patch is supplied, a new install is performed

  To see the full manpage for this program, execute:
    perldoc roothas.pl

=head1 DESCRIPTION

  This script performs the operations necessary to install the Oracle
  SingleInstance-HighAvailability stack.  This is run once during the
  configuration of SI-HAS home.

  This script does the following:
  1) Setup permissions of binaries and DSOs in SI-HAS home
  2) Setup OLR for storing SI-HAS configuration data

  Upon successful completion on each node, the Oracle SI-HA stack will
  be executing.

=cut

################ End Documentation ################

#    MODIFIED   (MM/DD/YY)
#    dpham       03/18/10 - Fix bug 9282344
#    dpham       03/18/10 - Call s_deltOldServ() during upgrade
#    dpham       01/18/10 - Call s_getAuthorizedOwner()
#    dpham       01/13/10 - Fix upgrade issue (9268527)
#    dpham       12/17/09 - Redirect stdout to trace file
#    dpham       12/08/09 - ohasd resource should be owned by user in test env
#    dpham       12/07/09 - Increase wait time to 30 mins for ohasd
#    jleys       09/29/09 - Clean CSS local endpts on upgrade
#    ksviswan    07/21/09 - add -unlock
#    ksviswan    06/26/09 - Change default perl path
#    dpham       05/05/09 - Backup OLR (8484172)
#    ksviswan    05/02/09 - Patching support
#    dpham       04/21/09 - Add CRSDelete option
#    dpham       04/16/09 - Fix logic error when checking during upgrade
#    dpham       04/15/09 - Call 'crsctl pin css' if local_config doesn't exist
#    dpham       04/07/09 - Install acfsroot kernel
#    dpham       03/31/09 - ACFS is not supported in SIHA in 11gR2
#    dpham       03/25/09 - Call createLocalOnlyOCR
#    ksviswan    03/20/09 - Stop CSSD first before ocr migration
#    dpham       03/11/09 - Call CSS_start if ($RESTART_CSS || 
#			    ($cfg->params('ASM_UPGRADE') =~ m/true/i)) 
#    dpham       03/10/09 - Call isCRSAlreadyConfigured to check if SIHA is
#                           already configured
#                         - Add ORACLE_BASE
#    dpham       02/26/09 - Call start_acfs_registry instead of add_acfs_registry
#    ksviswan    02/20/09 - Fix bug 8255312 and create SCR/NS dirs for upgrade
#    dpham       02/11/09 - Call add_acfs_registry
#    ksviswan    02/05/09 - Fix bug 8208293
#    dpham       02/04/09 - Add isRAC_appropriate to check for rac_on/rac_off
#    dpham       01/20/09 - Remove configure_ACFS 
#    jleys       11/30/08 - Remove setting of HAS_USER (done in
#                           crsconfig_lib->new
#    jleys       11/26/08 - Remove setusage call
#    jleys       11/14/08 - Packagize perl scripts
#    ksviswan    11/19/08 - Support SIHA deconfig
#    diguma      11/18/08 - adding use oracss
#    dpham       10/09/08 - Trap error output from usm_root so that it gets written
#                           to the CRS log.
#    jleys       09/24/08 - Add callback to get params hash
#    dpham       09/05/08 - Change all SIHA resources to own by "NT AUTHORITY\SYSTEM".
#    jleys       07/11/08 - Put this program path into perl search path
#    jleys       04/15/08 - Use parameters hash
#    dpham       06/23/08 - change PERLLIB583 to use OH/perl/5.8.3/lib
#    averhuls    06/03/08 - Call usm_root to install the USM sub-system.
#    srisanka    05/12/08 - Bug 5748551: copy earlier local-only OCR to SIHA
#                           home
#    srisanka    05/12/08 - validate SIHA params
#    srisanka    05/08/08 - fix local_only check
#    samjo       05/02/08 - Use hostname in olr file name
#    srisanka    05/01/08 - ASM_UPGRADE check for true/false
#    srisanka    04/09/08 - handle pre-existing CSS/ASM
#    jleys       04/06/08 - Use configure_hasd instead of invoking
#                           hasdconfig.pl
#    srisanka    02/26/08 - remove output to STDOUT
#    srisanka    02/25/08 - modify args to hasdconfig.pl
#    srisanka    01/08/08 - separate generic and OSD code
#    samjo       12/07/07 - Making OCR and OLR names consistent
#    srisanka    11/20/07 - make HOST a global var
#    srisanka    08/21/07 - Creation
# 

use strict;
use English;

use File::Spec::Functions;
use File::Basename;

BEGIN {
  # Add the directory of this file to the search path
  push @INC, dirname($PROGRAM_NAME);
}

use Getopt::Long;

use crsconfig_lib;
use oracss;
require crsdelete;
require crspatch;

my $USAGE;
my $hahome = "";
my $upgrade;
our $g_delete = 0;
our $g_force = 0;
our $g_patch = 0;
our $g_unlock = 0;

# the return code to give when the incorrect parameters are passed
my $usage_rc = 1;

#
# MAIN SCRIPT BODY
#

push @INC, dirname ($0);

our $DEBUG = FALSE;

my $PARAM_FILE_PATH = catfile (dirname ($0), "crsconfig_params");

GetOptions('verbose!'    => \$DEBUG,
           'upgrade!',   => \$upgrade,
           'deconfig!',  => \$g_delete,
           'delete!',    => \$g_delete,
           'force'       => \$g_force,
           'patch!'      => \$g_patch,
           'unlock'      => \$g_unlock,
           'hahome=s'    => \$hahome,
           'paramfile=s' => \$PARAM_FILE_PATH
          ) or pod2usage($usage_rc);

# Check validity of args
pod2usage(-msg => "Invalid extra options passed: @ARGV",
          -exitval => $usage_rc) if (@ARGV);


local $SIG{'__DIE__'} = sub { dietrap(@_); };

# pull all parameters defined in crsconfig_params and s_crsconfig_defs (if
# it exists) as variables in Perl
my $defsfile = catfile (dirname ($0), "s_crsconfig_defs");

### check if run as super user
my $SUPERUSER = check_SuperUser ();
if (!$SUPERUSER) {
  error("Insufficient privileges to execute this script");
  exit 1;
}

my $HOST = tolower_host ();
die "$!" if ($HOST eq "");

# Read the config files and set up the configuration data for
# subsequent processing
my $cfg =
  crsconfig_lib->new(IS_SIHA             => TRUE,
                     paramfile           => $PARAM_FILE_PATH,
                     osdfile             => $defsfile,
                     crscfg_trace        => TRUE,
                     CRSDelete           => $g_delete,
                     hahome              => $hahome,
                     HOST                => $HOST,
                     UPGRADE             => $upgrade,
                     HAPatch             => $g_patch,
                     );

# redirect stdout/stderr as appropriate
s_redirect_souterr ($cfg->crscfg_trace_file . "_OUT");

my $ORACLE_HOME  = $cfg->params('ORACLE_HOME');
my $ORACLE_BASE  = $cfg->params('ORACLE_BASE');
my $ORA_CRS_HOME = $ORACLE_HOME;
my $ORA_HA_HOME  = $ORACLE_HOME;

### Set this host name (lower case and no domain name)
# Set the following vars appropriately for SI-HA env
my $OCRCONFIGBIN = catfile ($ORACLE_HOME, "bin", "ocrconfig");

if    ($g_delete) { HADeconfigure(); }
elsif ($g_patch)  { HAPatch(); }
elsif ($g_unlock) { unlockHAHome(); }
else {

  # check if CRS is configured
  if (isCRSAlreadyConfigured()) {
     exit 1;
  }

  # validate RAC_ON/RAC_OFF
  if (! is_dev_env () && ! isRAC_appropriate()) {
     exit 1;
  }

 # run directory creation, script instantiation, files creation/permissions
 # modules
  run_env_setup_modules (); 

  $ENV{'ORA_CRS_HOME'} = $ORA_CRS_HOME;
  $ENV{'ORACLE_HOME'} = $ORACLE_HOME;
  $ENV{'ORACLE_BASE'} = $ORACLE_BASE;

  validate_olrconfig ($cfg->OLR_LOCATION, $ORA_CRS_HOME)
    or die "Error in validate_olrconfig: $!";

  my $local_config_exists = local_only_config_exists();

  if (! $local_config_exists) {
     createLocalOnlyOCR();
  }

  olr_initial_config() or die("OLR configuration failed\n");

  # check if older version SI CSS exists, by looking in OCRLOC
  my $status;
  my $RESTART_CSS = FALSE;
  if ($local_config_exists) {
    # If the stack is up, keep track of this and stop it now
    if (($RESTART_CSS = local_only_stack_active())) {
      if (!stop_local_only_stack()) {
        die("Unable to stop running CSS");
      }
      #Bug 8255312 - In single instance asm upgrade to siha
      #these directories gets removed when stopping the old css
      #by localconfig -delete
      my $SCRDIR = catfile ($SCRBASE, $HOST);
      my $NSDIR  = "/var/tmp/.oracle";
      my $ORA_DBA_GROUP = $CFG->params('ORA_DBA_GROUP');
      if (! -d $SCRBASE) {
         trace("creating dir $SCRDIR");
         create_dir($SCRDIR);
         s_set_ownergroup ($CFG->SUPERUSER, $ORA_DBA_GROUP, $SCRDIR);
      }
      if (! -d $NSDIR) { 
         trace("creating dir $NSDIR");
         create_dir($NSDIR);
         s_set_ownergroup ($CFG->SUPERUSER, $ORA_DBA_GROUP, $NSDIR);
         s_set_perms ("01777", $NSDIR);
      }    
    }

    # Clean up local endpts used by CSSD, bug 891745
    CSS_Clean_Local_Endpts($cfg);

    if (!migrate_dbhome_to_SIHA()) {
      die("Migration of older version local_only to SIHA failed");
    }
  } else {
    # Create necessary configuration with 'crsctl pin css ...'
    # This will create the same configuration as 'clscfg -local -install
    my $crsctl = catfile ($ORACLE_HOME, "bin", "crsctl");
    my $cmd    = "$crsctl pin css -n $HOST";
    my $rc     = system_cmd ($cmd);
    if ($rc != 0) {
       error("Creating local-only OCR ($cmd) ... failed");
       exit;
    } else {
       trace ("Creating local-only OCR ($cmd) ... succeeded");
    }
  }

  # Initialize the SCR settings.
  s_init_scr ();

  # register OHASD service/daemon with OS
  trace ("Registering ohasd");
  register_service ("ohasd") or die "Can't register ohasd service: $!";

  # Need to start OHASD as non-$SUPERUSER, i.e., as crsuser, which in this case
  # would be ORACLE_OWNER
  trace ("Starting ohasd");
  start_service ("ohasd", $cfg->HAS_USER) or
    die "ohasd failed to start: $!";

  # Check if the service/daemon has started
  trace ("Checking ohasd");
  my $ohasd_running = check_service ("ohasd", 360);

  if ($ohasd_running) {
    trace ("ohasd started successfully");
  } else {
    error ("Timed out waiting for ohasd to start.");
    exit 1;
  }

  trace ("Creating HA resources and dependencies");

  my $owner = $cfg->params('ORACLE_OWNER');
  if ((! is_dev_env()) && ($cfg->platform_family eq "windows")) {
     $owner = '';
  }
  
  $status = configure_hasd('has', $HOST, $owner, $owner,
                           $cfg->params('ORA_DBA_GROUP'));

  if ($status) {
    trace ("Successfully created HA resources for HAS daemon and ASM");
  } else {
    error ("Failed to create HA resources for HAS daemon and ASM: $!");
    exit 1;
  }

  # if old SI CSS was running when roothas.pl was invoked *or* ASM needs to be
  # upgraded, start SIHA CSS
  if ($RESTART_CSS || ($cfg->params('ASM_UPGRADE') =~ m/true/i)) {
    if (!CSS_start()) {
      error ("Failed to start CSS daemon");
      exit 1;
    }
  }

  # install acfsroot kernel
  isACFSSupported();

  # backup OLR
  my $ocrconfig = catfile($cfg->params('ORACLE_HOME'), 'bin', 'ocrconfig');
  my $rc = system ("$ocrconfig -local -manualbackup");
  if ($rc == 0) {
     trace ("$ocrconfig -local -manualbackup ... passed");
  } else {
     trace ("$ocrconfig -local -manualbackup ... failed");
  }

  print "Successfully configured Oracle Grid Infrastructure for a Standalone Server\n";
  trace ("Successfully configured Oracle Grid Infrastructure for a Standalone Server");
}

0;

