/*=======================================================================+
 |  Copyright (c) 2000 Oracle Corporation Redwood Shores, California, USA|
 |                            All rights reserved.                       |
 +=======================================================================+
 | FILENAME
 |   WFLOGPKB.pls
 |
 | DESCRIPTION
 |   PL/SQL body for package:  WF_LOG_PKG
 |
 | APIs
 |   procedure String
 |   function  Test
 |   procedure Init
 |
 *=======================================================================*/

set verify off;
whenever sqlerror exit failure rollback;
whenever oserror exit failure rollback;

create or replace package body WF_LOG_PKG as
/* $Header: WFLOGPKB.pls 26.7 2004/06/29 21:12:32 yohuang ship $ */

LEVEL_UNEXPECTED CONSTANT NUMBER  := 6;
LEVEL_ERROR      CONSTANT NUMBER  := 5;
LEVEL_EXCEPTION  CONSTANT NUMBER  := 4;
LEVEL_EVENT      CONSTANT NUMBER  := 3;
LEVEL_PROCEDURE  CONSTANT NUMBER  := 2;
LEVEL_STATEMENT  CONSTANT NUMBER  := 1;

/*
**  Global variables to hold logging information. For standalone we will support
**  only the existing logging feature, with changes such that the Apps specific
**  implementation works. Now, instead of setting WF_LOG_PKG.WF_DEBUG_FLAG alone,
**  following global variables need to be set in order to enable Logging.
**
**  Later on, when FND_LOG is available standalone, these values will be held in
**  AFLOG_ENABLED, AFLOG_LEVEL, AFLOG_MODULE and AFLOG_FILENAME respectively.
**
**  These values are available only for a particular session. We are still
**  retaining the existing behaviour of writing the log messages using
**  DBMS_OUTPUT.PUT_LINE, but using a different set-up.
*/
AFLOG_ENABLED   BOOLEAN DEFAULT FALSE; 
AFLOG_LEVEL     NUMBER := 999999;
AFLOG_MODULE    VARCHAR2(255);
AFLOG_FILENAME  VARCHAR2(255);

------------------------------------------------------------------------------
/*
** Init - Initialise the Logging global variables to do standalone testing.
**        This will do the same work as wf_log_pkg.wf_debug_flag.
**        (This API to be used by WF Dev only)
*/
procedure Init (
   LOG_ENABLED  in binary_integer,
   LOG_FILENAME in varchar2,
   LOG_LEVEL    in number,
   LOG_MODULE   in varchar2,
   FND_USER_ID  in number,
   FND_RESP_ID  in number,
   FND_APPL_ID  in number 
)
is
begin

   if (log_enabled = 1) then
     WF_LOG_PKG.AFLOG_ENABLED := true;
   else
     WF_LOG_PKG.AFLOG_ENABLED := false;
   end if;
   WF_LOG_PKG.AFLOG_LEVEL := log_level;
   WF_LOG_PKG.AFLOG_MODULE := log_module;
   WF_LOG_PKG.AFLOG_FILENAME := log_filename;

   if (WF_LOG_PKG.AFLOG_ENABLED) then
      -- This is a workaround in order for the changes to work in Standalone as
      -- well as Apps instance. This level value will be checked by the callers.
      FND_LOG.G_CURRENT_RUNTIME_LEVEL := log_level;
   end if;

exception
   when others then
      wf_core.context('WF_LOG_PKG', 'Init');
      raise;
end Init;
------------------------------------------------------------------------------
/*
** set_level - Described in Spec
**
*/
procedure SET_LEVEL(
  LOG_LEVEL in number
) 
is

begin
   if (log_level is NOT NULL) then
      if WF_LOG_PKG.AFLOG_ENABLED then
        FND_LOG.G_CURRENT_RUNTIME_LEVEL := log_level;
      end if;
   end if;
end SET_LEVEL;
/*
** String - Described in Spec
**
*/
procedure String(
  LOG_LEVEL	in number,
  MODULE	in varchar2,
  MESSAGE	in varchar2
) 
is
begin

   if (WF_LOG_PKG.Test(log_level, module)) then
      dbms_output.put_line('Level:'||log_level||' Module:'||module||' Time:'||
                          to_char(sysdate,'DD-MM-YYYY HH24:MI:SS'));
      dbms_output.put_line('>>> Message: '||message);
   end if;

exception
  when others then
    null;
end String;

------------------------------------------------------------------------------
/*
**  Test - Check if logging is enabled for the given level and module.
**         This involves only checking the values in the global variables. This
**         check should be moved to FND_LOG.Test once FND_LOG is available
**         Standalone.
*/
function Test(
   LOG_LEVEL in number,
   MODULE    in varchar2
)
return boolean
is
begin
  
  if (NOT WF_LOG_PKG.AFLOG_ENABLED) then
    return FALSE;
  end if;
 
  if (log_level >= WF_LOG_PKG.AFLOG_LEVEL and 
          upper(module) like upper(WF_LOG_PKG.AFLOG_MODULE)) then
     return TRUE;
  end if;
  return FALSE;

end Test;

end WF_LOG_PKG;
/
-- show errors package body WF_LOG_PKG
commit;
exit;
