REM dbdrv: sql ~PROD ~PATH ~FILE none none none package &phase=pls \
REM dbdrv: checkfile:~PROD:~PATH:~FILE
/*=======================================================================+
 |  Copyright (c) 1995 Oracle Corporation Redwood Shores, California, USA|
 |                            All rights reserved.                       |
 +=======================================================================+
 | FILENAME
 |   wfdirs.sql
 | DESCRIPTION
 |   WF_DIRECTORY package spec
 | NOTES
 |
 | MODIFIED
 |   01 Jan 1996  S Stratton    Created
 |   18 Mar 1996  G Buzsaki     Changed %type construction to explicit
 |   01 Jan 2002  J Smith       BUG 2001012 - Increased name, email_address,
 |                              roles to varchar2(320), display_name 360
 *=======================================================================*/

SET VERIFY OFF
WHENEVER SQLERROR EXIT FAILURE ROLLBACK;
WHENEVER OSERROR EXIT FAILURE ROLLBACK;

create or replace package Wf_Directory as
/* $Header: wfdirs.pls 26.33 2005/01/10 13:27:29 anachatt ship $ */
/*#
 * This public interface provides APIs that can be 
 * called by an application program or a workflow 
 * function in the runtime phase to retrieve information 
 * about existing users and roles, as well as to create 
 * and manage new ad hoc users and roles in the 
 * directory service.
 * @rep:scope public
 * @rep:product OWF
 * @rep:displayname Workflow Directory Services
 * @rep:lifecycle active
 * @rep:compatibility S
 * @rep:category BUSINESS_ENTITY WF_USER
 * @rep:ihelp FND/@a_ds See the related online help
 */

type UserTable is table of varchar2(320)
  index by binary_integer;

type RoleTable is table of varchar2(320)
  index by binary_integer;

type OrigSysTable is table of varchar2(30) index by binary_integer;

/*===========================================================================

  PL*SQL TABLE NAME:    wf_local_roles_tble_type

  DESCRIPTION:          Stores a list of attributes from
                        WF_LOCAL_ROLES

============================================================================*/

TYPE wf_local_roles_rec_type IS RECORD
(
 NAME                            VARCHAR2(320),
 DISPLAY_NAME                    VARCHAR2(360),
 DESCRIPTION                     VARCHAR2(1000),
 NOTIFICATION_PREFERENCE         VARCHAR2(8),
 LANGUAGE                        VARCHAR2(30),
 TERRITORY                       VARCHAR2(30),
 EMAIL_ADDRESS                   VARCHAR2(320),
 FAX                             VARCHAR2(240),
 STATUS                          VARCHAR2(8),
 EXPIRATION_DATE                 DATE,
 ORIG_SYSTEM                     VARCHAR2(240),
 ORIG_SYSTEM_ID                  NUMBER,
 PARENT_ORIG_SYSTEM              VARCHAR2(240),
 PARENT_ORIG_SYSTEM_ID           NUMBER,
 OWNER_TAG                       VARCHAR2(50),
 LAST_UPDATE_DATE                DATE,
 LAST_UPDATED_BY                 NUMBER(15),
 CREATION_DATE                   DATE,
 CREATED_BY                      NUMBER(15),
 LAST_UPDATE_LOGIN               NUMBER(15)

);

TYPE wf_local_roles_tbl_type IS TABLE OF
  wf_directory.wf_local_roles_rec_type
INDEX BY BINARY_INTEGER;

/*#
 * This API returns a table of users
 * for a given role.
 * @param role Role 
 * @param users Users 
 * @rep:scope public
 * @rep:lifecycle active
 * @rep:displayname Get Role Users
 * @rep:ihelp FND/@a_ds#a_getru See the related online help
 */

procedure GetRoleUsers(
  Role in varchar2,
  Users out nocopy Wf_Directory.UserTable);

procedure GetUserRelation(
  Base_User in varchar2,
  Relation in varchar2,
  Users out nocopy Wf_Directory.UserTable);

/*#
 * This API returns a table of roles that 
 * a given user is assigned to.
 * @param user User 
 * @param roles Roles 
 * @rep:scope public
 * @rep:lifecycle active
 * @rep:displayname Get User Roles
 * @rep:ihelp FND/@a_ds#a_getur See the related online help
 */

procedure GetUserRoles(
  User in varchar2,
  Roles out nocopy Wf_Directory.RoleTable);

/*#
 * This API returns the information about the display name, 
 * email address, notification preference, language and 
 * territory of a given role.
 * @param role Role 
 * @param display_name Display Name 
 * @param email_address Email Address 
 * @param notification_preference Notification Preference 
 * @param language Language 
 * @param territory Territory 
 * @rep:scope public
 * @rep:lifecycle active
 * @rep:displayname Get Role Info
 * @rep:ihelp FND/@a_ds#a_getrinf See the related online help
 */

procedure GetRoleInfo(
  Role in varchar2,
  Display_Name out nocopy varchar2,
  Email_Address out nocopy varchar2,
  Notification_Preference out nocopy varchar2,
  Language out nocopy varchar2,
  Territory out nocopy varchar2);

/*#
 * This API returns the information regarding display name , 
 * description, notification preference, language , 
 * territory, email address , fax , status , expiration 
 * date about a role in a SQL table.
 * @param role Role
 * @param Role_Info_Tbl Role Info Table 
 * @rep:scope public
 * @rep:lifecycle active
 * @rep:displayname Get Role Info Table
 * @rep:ihelp FND/@a_ds#a_getrinf2 See the related online help
 */


procedure GetRoleInfo2(
  Role in varchar2,
  Role_Info_Tbl out nocopy wf_directory.wf_local_roles_tbl_type);

procedure GetRoleInfoMail(
  role in varchar2,
  display_name out nocopy varchar2,
  email_address out nocopy varchar2,
  notification_preference out nocopy varchar2,
  language out nocopy varchar2,
  territory out nocopy varchar2,
  orig_system out nocopy varchar2,
  orig_system_id out nocopy number,
  installed_flag out nocopy varchar2);

--
-- GetRoleOrigSysInfo (PRIVATE)
-- Used by Workflow Internal Only
--
procedure GetRoleOrigSysInfo(
  Role in varchar2,
  Orig_System out nocopy varchar2,
  Orig_System_Id out nocopy number);

--
-- GetRoleNtfPref (PRIVATE) 
-- Used by Workflow Internal 
-- To obtain notification preference
-- for a given role 

function GetRoleNtfPref(
  Role in varchar2)
return varchar2;

/*#
 * This API returns true or false to identify 
 * whether a user is a performer of a role.
 * @param user User 
 * @param role Role 
 * @return IsPerformer 
 * @rep:scope public
 * @rep:lifecycle active
 * @rep:displayname Is Performer
 * @rep:ihelp FND/@a_ds#a_isperf See the related online help
 */

function IsPerformer(
    User in varchar2,
    Role in varchar2)
return boolean;

function CurrentUser
return varchar2;
--
-- Function determins if a user is currently active
-- and therefore available to participate in a workflow
--  In :
--      Username
-- Returns:
--      True  - If user is Active
--      False - If User is NOT Active
--
/*#
 * This API determines if a user currently has a 
 * status of 'ACTIVE' and is available to 
 * participate in a workflow. Returns TRUE if the 
 * user has a status of 'ACTIVE', otherwise it returns
 * FALSE.
 * @param username User Name 
 * @return IsActive
 * @rep:scope public
 * @rep:lifecycle active
 * @rep:displayname User Active
 * @rep:ihelp FND/@a_ds#a_usract See the related online help
 */

function UserActive(username in varchar2)
return boolean;
--
-- Procedure Gets the User_name given the orig system info
--   Get the username
-- IN:
--      orig_system     - Code identifying the original table
--      orig_system_id  - Id of the row in original table
-- Returns:
--      user_name       - Workflow user_name
--
/*#
 * This API returns a Workflow display name 
 * and username for a user given the system 
 * information from the original user and roles repository.
 * @param p_orig_system Originating System
 * @param p_orig_system_id Originating System ID 
 * @param p_name Name 
 * @param p_display_name Display Name 
 * @rep:scope public
 * @rep:lifecycle active
 * @rep:displayname Get User Name
 * @rep:ihelp FND/@a_ds#a_getun See the related online help
 */

procedure GetUserName ( p_orig_system           in varchar2,
                        p_orig_system_id        in varchar2,
                        p_name                  out nocopy varchar2,
                        p_display_name          out nocopy varchar2 );
--
--
--
-- Procedure Gets the Role_name given the orig system info
--   Get the Role name
-- IN:
--      orig_system     - Code identifying the original table
--      orig_system_id  - Id of the row in original table
-- Returns:
--      Role_name       - Workflow role_name
--
/*#
 * This API returns a Workflow display name 
 * and role name for a role given the
 * system information from the original user and roles 
 * repository.
 * @param p_orig_system Originating System
 * @param p_orig_system_id Originating System ID 
 * @param p_name Name 
 * @param p_display_name Display Name 
 * @rep:scope public
 * @rep:lifecycle active
 * @rep:displayname Get RoleName
 * @rep:ihelp FND/@a_ds#a_grname See the related online help
 */

procedure GetRoleName ( p_orig_system           in varchar2,
                        p_orig_system_id        in varchar2,
                        p_name                  out nocopy varchar2,
                        p_display_name          out nocopy varchar2 );
--
/*#
 * This API returns a Workflow role's display 
 * name given the role's internal name.
 * @param p_role_name Role Name 
 * @return Display Name
 * @rep:scope public
 * @rep:lifecycle active
 * @rep:displayname Get Role Display Name
 * @rep:ihelp FND/@a_ds#a_grdname See the related online help
 */

function GetRoleDisplayName(p_role_name in varchar2)
return varchar2;
pragma restrict_references(GetRoleDisplayName, WNDS, WNPS);
--
--

--
-- SetAdHocUserStatus
--   Update status for user
-- IN
--   user_name        -
--   status           - status could be 'ACTIVE' or 'INACTIVE'
-- OUT
--
/*#
 * This API sets status of an ad hoc user 
 * as 'ACTIVE' or 'INACTIVE'.
 * @param user_name User Name 
 * @param status Status 
 * @rep:scope public
 * @rep:lifecycle active
 * @rep:displayname Set AdhocUser  Status
 * @rep:ihelp FND/@a_ds#a_sahust See the related online help
 */

procedure SetAdHocUserStatus(user_name      in varchar2,
                        status         in varchar2 default 'ACTIVE');

--
-- SetAdHocRoleStatus
--   Update status for role
-- IN
--   role_name        -
--   status           - status could be 'ACTIVE' or 'INACTIVE'
-- OUT
--
/*#
 * This API Sets the status of an ad hoc role as 
 * 'ACTIVE' or 'INACTIVE'.
 * @param role_name User Name 
 * @param status Status
 * @rep:scope public
 * @rep:lifecycle active
 * @rep:displayname Set AdhocRole Status
 * @rep:ihelp FND/@a_ds#a_sahrs See the related online help
 */

procedure SetAdHocRoleStatus(role_name      in varchar2,
                        status         in varchar2 default 'ACTIVE');



--
-- CreateUser (PRIVATE)
--   Create a User
-- IN
--   name          - User name
--   display_name  - User display name
--   orig_system
--   orig_system_id
--   language      -
--   territory     -
--   description   -
--   notification_preference -
--   email_address -
--   fax           -
--   status        -
--   expiration_date - NULL expiration date means no expiration
--   start_date
--   parent_orig_system
--   parent_orig_system_id
--   owner_tag -
--   last_update_date -
--   last_updated_by - 
--   creation_date -
--   created_by -
--   last_update_login
-- OUT
--
procedure CreateUser( name                    in  varchar2,
                      display_name            in  varchar2,
                      orig_system             in  varchar2,
                      orig_system_id          in  number,
                      language                in  varchar2 default null,
                      territory               in  varchar2 default null,
                      description             in  varchar2 default null,
                      notification_preference in  varchar2 default 'MAILHTML',
                      email_address           in  varchar2 default null,
                      fax                     in  varchar2 default null,
                      status                  in  varchar2 default 'ACTIVE',
                      expiration_date         in  date     default NULL,
                      start_date              in  date     default sysdate,
                      parent_orig_system      in  varchar2 default null,
                      parent_orig_system_id   in  number   default null,
                      owner_tag               in  varchar2 default null,
                      last_update_date        in  date     default sysdate,
                      last_updated_by         in  number   default null,
                      creation_date           in  date     default sysdate,
                      created_by              in  number   default null,
                      last_update_login       in  number   default null);


--
-- CreateAdHocUser
--   Create an ad hoc user given a user name, display name, etc.
-- IN
--   name          - User name
--   display_name  - User display name
--   description   -
--   notification_preference -
--   language      -
--   territory     -
--   email_address -
--   fax           -
--   status        -
--   expiration_date - NULL expiration date means no expiration
-- OUT
--
/*#
 * This API creates a user at runtime by creating a value 
 * in the WF_LOCAL_USERS table.This is referred to as 
 * an ad hoc user.
 * @param name Name 
 * @param display_name Display Name    
 * @param language Language           
 * @param territory Territory       
 * @param description Description      
 * @param notification_preference Notification Preference 
 * @param email_address Email Address       
 * @param fax Fax             
 * @param status Status         
 * @param expiration_date Expiration Date      
 * @param parent_orig_system Parent Orignating System    
 * @param parent_orig_system_id Parent Originating System ID
 * @rep:scope public
 * @rep:lifecycle active
 * @rep:displayname Create AdHoc User
 * @rep:ihelp FND/@a_ds#a_crahu See the related online help
 */

procedure CreateAdHocUser(name                in out nocopy varchar2,
                      display_name            in out nocopy  varchar2,
                      language                in  varchar2 default null,
                      territory               in  varchar2 default null,
                      description             in  varchar2 default null,
                      notification_preference in varchar2 default 'MAILHTML',
                      email_address           in  varchar2 default null,
                      fax                     in  varchar2 default null,
                      status                  in  varchar2 default 'ACTIVE',
                      expiration_date         in  date default null,
                      parent_orig_system      in varchar2 default null,
                      parent_orig_system_id   in number   default null);


--
-- CreateRole (PRIVATE)
--   Create an ad hoc role given a specific name
-- IN
--   role_name          -
--   role_display_name  -
--   role_description   -
--   notification_preference -
--   language      -
--   territory     -
--   email_address -
--   fax           -
--   status        -
--   expiration_date   - Null means no expiration date
--   orig_system   -
--   orig_system_id
-- OUT
--
procedure CreateRole( role_name               in  varchar2,
                      role_display_name       in  varchar2,
                      orig_system             in  varchar2,
                      orig_system_id          in  number,
                      language                in  varchar2 default null,
                      territory               in  varchar2 default null,
                      role_description        in  varchar2 default null,
                      notification_preference in  varchar2 default 'MAILHTML',
                      email_address           in  varchar2 default null,
                      fax                     in  varchar2 default null,
                      status                  in  varchar2 default 'ACTIVE',
                      expiration_date         in  date default NULL,
                      start_date              in  date default sysdate,
                      parent_orig_system      in  varchar2 default null,
                      parent_orig_system_id   in  number   default null,
                      owner_tag               in  varchar2 default null,
                      last_update_date        in  date     default sysdate,
                      last_updated_by         in  number   default null,
                      creation_date           in  date     default sysdate,
                      created_by              in  number   default null,
                      last_update_login       in  number   default null);




--
-- CreateAdHocRole
--   Create an ad hoc role given a specific name
-- IN
--   role_name          -
--   role_display_name  -
--   role_description   -
--   notification_preference -
--   role_users         - Comma or space delimited list
--   language      -
--   territory     -
--   email_address -
--   fax           -
--   status        -
--   expiration_date   - Null means no expiration date
-- OUT
--
/*#
 * This API creates a role at runtime by creating a value 
 * WF_LOCAL_ROLES table. This is referred to as an 
 * ad hoc role.
 * @param role_name Role Name 
 * @param role_display_name Role Display Name         
 * @param language Language             
 * @param territory Territory         
 * @param role_description Role Description       
 * @param notification_preference Notification Preference
 * @param role_users Role Users 
 * @param email_address Email Address      
 * @param fax Fax            
 * @param status Status          
 * @param expiration_date Expiration Date 
 * @param parent_orig_system Parent Orignating System   
 * @param parent_orig_system_id Parent Originating System ID 
 * @param owner_tag Owner Tag 
 * @rep:scope public
 * @rep:lifecycle active
 * @rep:displayname Create AdHoc Role
 * @rep:ihelp FND/@a_ds#a_crahr See the related online help
 */

procedure CreateAdHocRole(role_name           in out nocopy varchar2,
                      role_display_name       in out nocopy  varchar2,
                      language                in  varchar2 default null,
                      territory               in  varchar2 default null,
                      role_description        in  varchar2 default null,
                      notification_preference in varchar2 default 'MAILHTML',
                      role_users              in  varchar2 default null,
                      email_address           in  varchar2 default null,
                      fax                     in  varchar2 default null,
                      status                  in  varchar2 default 'ACTIVE',
                      expiration_date         in  date default null,
                      parent_orig_system      in varchar2 default null,
                      parent_orig_system_id   in number default null,
                      owner_tag               in  varchar2 default null);



--
-- CreateAdHocRole2
--   Create an ad hoc role given a specific name
-- IN
--   role_name          -
--   role_display_name  -
--   role_description   -
--   notification_preference -
--   role_users         - Comma or space delimited list
--   language      -
--   territory     -
--   email_address -
--   fax           -
--   status        -
--   expiration_date   - Null means no expiration date
-- OUT
--
procedure CreateAdHocRole2(role_name          in out nocopy varchar2,
                      role_display_name       in out nocopy  varchar2,
                      language                in  varchar2 default null,
                      territory               in  varchar2 default null,
                      role_description        in  varchar2 default null,
                      notification_preference in  varchar2 default 'MAILHTML',
                      role_users              in  WF_DIRECTORY.UserTable,
                      email_address           in  varchar2 default null,
                      fax                     in  varchar2 default null,
                      status                  in  varchar2 default 'ACTIVE',
                      expiration_date         in  date default null,
                      parent_orig_system      in  varchar2 default null,
                      parent_orig_system_id   in  number default null,
                      owner_tag               in  varchar2 default null);
                      
                      
--
-- CreateUserRole (PRIVATE)
--   Create a user to role relationship.
-- IN
--   user_name -
--   role_name -
--   user_orig_system -
--   user_orig_system_id -
--   role_orig_system -
--   role_orig_system_id -
--
procedure CreateUserRole(user_name             in varchar2,
                         role_name             in varchar2,
                         user_orig_system      in varchar2 default NULL,
                         user_orig_system_id   in number default NULL,
                         role_orig_system      in varchar2 default NULL,
                         role_orig_system_id   in number default NULL,
                         validateUserRole      in boolean default TRUE,
                         start_date            in date default sysdate,
                         end_date              in date default NULL,
                         created_by            in number default -1,
                         creation_date         in date default sysdate,
                         last_updated_by       in number default -1,
                         last_update_date      in date default sysdate,
                         last_update_login     in number default -1,
                         assignment_type       in varchar2 default 'D',
                         parent_orig_system    in varchar2 default NULL,
                         parent_orig_system_id in number default null,
                         owner_tag             in  varchar2 default null);


--
-- SetUserRoleAttr (PRIVATE)
--   Update a user to role relationship.
-- IN
--   user_name -
--   role_name -
--   start_date -
--   expiration_date -
--   user_orig_system -
--   user_orig_system_id -
--   role_orig_system -
--   role_orig_system_id -
--   assignment_type -
--   parent_orig_system -
--   parent_orig_system_id
--   owner_tag -
--
procedure SetUserRoleAttr ( user_name             in  varchar2,
                            role_name             in  varchar2,
                            start_date            in  date     default NULL,
                            end_date              in  date     default NULL,
                            user_orig_system      in  varchar2,
                            user_orig_system_id   in  number,
                            role_orig_system      in  varchar2,
                            role_orig_system_id   in  number,
                            OverWrite             in  boolean  default FALSE,
                            last_updated_by       in  number   default -1,
                            last_update_date      in  date     default sysdate,
                            last_update_login     in  number   default -1,
                            assignment_type       in  varchar2 default 'D',
                            parent_orig_system    in  varchar2 default null,
                            parent_orig_system_id in  number   default null,
                            owner_tag             in  varchar2 default null);


--
-- RemoveUserRole (PRIVATE)
--   Remove a user from a role.
-- IN
--   user_name -
--   role_name -
--   user_orig_system -
--   user_orig_system_id -
--   role_orig_system -
--   role_orig_system_id -
--
procedure RemoveUserRole(user_name           in varchar2,
                         role_name           in varchar2,
                         user_orig_system    in varchar2,
                         user_orig_system_id in number,
                         role_orig_system    in varchar2,
                         role_orig_system_id in number);

--
-- AddUsersToAdHocRole
--   Add users to an existing ad hoc role
-- IN
--   role_name     - AdHoc role name
--   role_users    - Space or comma delimited list of apps-based users
--                      or adhoc users
-- OUT
--
/*#
 * This API adds users to a existing ad hoc role.
 * @param role_name Role Name
 * @param role_users Role Users       
 * @rep:scope public
 * @rep:lifecycle active
 * @rep:displayname Add Users To AdHocRole
 * @rep:ihelp FND/@a_ds#a_autahr See the related online help
 */
procedure AddUsersToAdHocRole(role_name         in varchar2,
                              role_users       in  varchar2);

--
-- AddUsersToAdHocRole2
--   Add users to an existing ad hoc role
-- IN
--   role_name     - AdHoc role name
--   role_users    - Table of user names.
--                     
-- OUT
--
procedure AddUsersToAdHocRole2(role_name         in varchar2,
                               role_users        in WF_DIRECTORY.UserTable);

--
-- SetUserAttr (PRIVATE)
--   Update additional attributes for users
-- IN
--   user_name        - user name
--   orig_system      -
--   orig_system_id   -
--   display_name  -
--   description -
--   notification_preference -
--   language      -
--   territory     -
--   email_address -
--   fax           -
--   expiration_date  - New expiration date
--   status           - status could be 'ACTIVE' or 'INACTIVE'
--   start_date
--   OverWrite  - Turns off existing data protection.
--   parent_orig_system -
--   parent_orig_system_id -
--   owner_tag
-- OUT
--
procedure SetUserAttr(user_name               in  varchar2,
                      orig_system             in  varchar2,
                      orig_system_id          in  number,
                      display_name            in  varchar2 default null,
                      description             in  varchar2 default null,
                      notification_preference in varchar2  default null,
                      language                in  varchar2 default null,
                      territory               in  varchar2 default null,
                      email_address           in  varchar2 default null,
                      fax                     in  varchar2 default null,
                      expiration_date         in  date     default null,
                      status                  in  varchar2 default null,
                      start_date              in  date     default null,
                      OverWrite               in  boolean  default FALSE,
                      Parent_Orig_System      in  varchar2 default null,
                      Parent_Orig_System_ID   in  number   default null,
                      owner_tag               in  varchar2 default null,
                      last_updated_by         in  number   default null,
                      last_update_date        in  date     default sysdate,
                      last_update_login       in  number   default null);

--
-- SetRoleAttr (PRIVATE)
--   Update additional attributes for roles
-- IN
--   role_name        - role name
--   orig_system      -
--   orig_system_id   -
--   display_name  -
--   notification_preference -
--   language      -
--   territory     -
--   email_address -
--   fax           -
--   expiration_date  - New expiration date
--   status           - status could be 'ACTIVE' or 'INACTIVE'
--   start_date
--   OverWrite - Turns off existing data protection.
-- OUT
--
procedure SetRoleAttr(role_name               in  varchar2,
                      orig_system             in  varchar2,
                      orig_system_id          in  number,
                      display_name            in  varchar2 default null,
                      description             in  varchar2 default null,
                      notification_preference in  varchar2 default null,
                      language                in  varchar2 default null,
                      territory               in  varchar2 default null,
                      email_address           in  varchar2 default null,
                      fax                     in  varchar2 default null,
                      expiration_date         in  date     default null,
                      status                  in  varchar2 default null,
                      start_date              in  date     default null,
                      OverWrite               in  boolean  default FALSE,
                      Parent_Orig_System      in  varchar2 default null,
                      Parent_Orig_System_ID   in  number   default null,
                      owner_tag               in  varchar2 default null,
                      last_updated_by         in  number   default null,
                      last_update_date        in  date     default sysdate,
                      last_update_login       in  number   default null);

--
-- SetAdHocUserExpiration
--   Update expiration date for ad hoc users
-- IN
--   user_name        - Ad hoc user name
--   expiration_date  - New expiration date
-- OUT
--
/*#
 * This API updates the expiration date for 
 * an ad hoc user.
 * @param user_name User Name
 * @param expiration_date Expiration Date
 * @rep:scope public
 * @rep:lifecycle active
 * @rep:displayname Set AdHocUser Expiration
 * @rep:ihelp FND/@a_ds#a_sahue See the related online help
 */

procedure SetAdHocUserExpiration(user_name      in varchar2,
                      expiration_date           in date default sysdate);

--
-- SetAdHocRoleExpiration
--   Update expiration date for ad hoc roles, user roles
-- IN
--   role_name        - Ad hoc role name
--   expiration_date  - New expiration date
-- OUT
--
/*#
 * This API updates the expiration date for 
 * an ad hoc user.
 * @param role_name Role Name 
 * @param expiration_date Expiration Date
 * @rep:scope public
 * @rep:lifecycle active
 * @rep:displayname Set AdHocRole Expiration
 * @rep:ihelp FND/@a_ds#a_sahre See the related online help
 */

procedure SetAdHocRoleExpiration(role_name      in varchar2,
                      expiration_date           in date default sysdate);

--
-- SetAdHocUserAttr
--   Update additional attributes for ad hoc users
-- IN
--   user_name        - Ad hoc user name
--   display_name  -
--   notification_preference -
--   language      -
--   territory     -
--   email_address -
--   fax           -
-- OUT
--
/*#
 * This API updates the attributes for an ad hoc user.
 * @param user_name User Name  
 * @param display_name Display Name 
 * @param notification_preference Notification Preference 
 * @param language Language
 * @param territory Territory 
 * @param email_address Email address 
 * @param fax Fax 
 * @param parent_orig_system Parent Originating System
 * @param parent_orig_system_id Parent Originating System ID 
 * @param owner_tag Owner Tag 
 * @rep:scope public
 * @rep:lifecycle active
 * @rep:displayname Set AdHocUser Attributes
 * @rep:ihelp FND/@a_ds#a_sahua See the related online help
 */

procedure SetAdHocUserAttr(user_name               in  varchar2,
                           display_name            in  varchar2 default null,
                           notification_preference in  varchar2 default null,
                           language                in  varchar2 default null,
                           territory               in  varchar2 default null,
                           email_address           in  varchar2 default null,
                           fax                     in  varchar2 default null,
                           parent_orig_system      in  varchar2 default null,
                           parent_orig_system_id   in  number   default null,
                           owner_tag               in  varchar2 default null);

--
-- SetAdHocRoleAttr
--   Update additional attributes for ad hoc roles, user roles
-- IN
--   role_name        - Ad hoc role name
--   display_name  -
--   notification_preference -
--   language      -
--   territory     -
--   email_address -
--   fax           -
-- OUT
--
/*#
 * This API updates the attributes for an ad hoc role.
 * @param role_name UserName 
 * @param display_name Display Name
 * @param notification_preference Notification Preference
 * @param language Language
 * @param territory Territory 
 * @param email_address Email address 
 * @param fax Fax 
 * @param parent_orig_system Parent Originating System 
 * @param parent_orig_system_id Parent Originating System ID
 * @param owner_tag Owner Tag  
 * @rep:scope public
 * @rep:lifecycle active
 * @rep:displayname Set AdHocRole Attributes
 * @rep:ihelp FND/@a_ds#a_sahra See the related online help
 */
procedure SetAdHocRoleAttr(role_name          in  varchar2,
                      display_name            in  varchar2 default null,
                      notification_preference in  varchar2 default null,
                      language                in  varchar2 default null,
                      territory               in  varchar2 default null,
                      email_address           in  varchar2 default null,
                      fax                     in  varchar2 default null,
                      parent_orig_system      in  varchar2 default null,
                      parent_orig_system_id   in  number   default null,
                      owner_tag               in  varchar2 default null);

--
-- RemoveUsersFromAdHocRole
--   Remove users from an existing ad hoc role
-- IN
--   role_name     -
--   role_users    -
-- OUT
--
/*#
 * This API removes users from an existing 
 * ad hoc role.
 * @param role_name Role Name
 * @param role_users Role Users       
 * @rep:scope public
 * @rep:lifecycle active
 * @rep:displayname Remove Users From AdHocRole
 * @rep:ihelp FND/@a_ds#a_rufahr See the related online help
 */

procedure RemoveUsersFromAdHocRole(role_name in varchar2,
                      role_users             in varchar2 default null);

--
-- ChangeLocalUserName
--  Change a User's Name in the WF_LOCAL_USERS table.
-- IN
--  OldName
--  NewName
--  Propagate - call WF_MAINTENANCE.PropagateChangedName.
-- OUT
--
function ChangeLocalUserName(OldName in varchar2,
                             NewName in varchar2,
                             Propagate in boolean default true)
return BOOLEAN;

--
-- ReassignUserRoles
--   Reassigns user/roles when the user information changes.
-- IN
--   p_user_name
--   p_old_user_origSystem
--   p_old_user_origSystemID
--   p_new_user_origSystem
--   p_new_user_origSystemID
--
-- OUT
--
procedure ReassignUserRoles (p_user_name             in VARCHAR2,
                             p_old_user_origSystem   in VARCHAR2,
                             p_old_user_origSystemID in VARCHAR2,
                             p_new_user_origSystem   in VARCHAR2,
                             p_new_user_origSystemID in VARCHAR2);


--
-- AssignPartition (PRIVATE)
--
-- IN
--  p_orig_system (VARCHAR2)
--
-- RETURNS
--  Partition ID (NUMBER)
--
-- COMMENTS
--  This api will check to see the partition for the p_orig_system exists.
--  if it does not exist, it will be added to p_table_name.  In either case
--  the Partition_ID will be returned for the calling api to properly populate
--  that column on insert/update.
--
procedure  AssignPartition (p_orig_system   in  varchar2,
                            p_partitionID   out nocopy number,
                            p_partitionName out nocopy varchar2);

--Bug 3090738
--This API queries wf_roles for information of the
--user when the e-mail address is give.
procedure getInfoFromMail(mailid   in varchar2,
                          role out nocopy  varchar2,
                          display_name out NOCOPY varchar2,
                          description out NOCOPY varchar2,
                          notification_preference out NOCOPY varchar2,
                          language out NOCOPY varchar2,
                          territory out NOCOPY varchar2,
                          fax       out NOCOPY varchar2,
                          expiration_date out NOCOPY date,
                          status  out NOCOPY varchar2,
                          orig_system  out NOCOPY varchar2,
                          orig_system_id out NOCOPY number);

function IsMLSEnabled(p_orig_system  in   varchar2)
return boolean;

--
-- Change_Name_References_RF (PRIVATE)
--
-- IN
--  p_sub_guid (RAW)
--  p_event    (WF_EVENT_T)
--
-- RETURNS
--  varchar2
--
-- COMMENTS
--  This api is a rule function to be called by BES.  It is primarily used for
--  a user name change to update all the fk references.  The subscription is 
--  set as deferred to offline the updates to return control back to the user
--  more quickly.
--
function Change_Name_References_RF( p_sub_guid  in            RAW,
                                    p_event     in out NOCOPY WF_EVENT_T )
return varchar2;

end Wf_Directory;
/
--show errors package wf_directory;
commit;
exit;
