--
-- $Header: owb_lock.pks 14-sep-2006.16:28:22 rvelisar Exp $
--
-- owb_lock.pks
--
-- Copyright (c) 2006, Oracle. All rights reserved.  
--
--    NAME
--      owb_lock.pks - This package provides locking routines for OWB product.
--
--    DESCRIPTION
--      This package provides locking routines for OWB product.
-- 
--      ALLOCATE_UNIQUE is a wrapper around DBMS_LOCK.ALLOCATE_UNIQUE, 
--      but it executes it in an autonomous transaction.
-- 
--      LOCK_WORKSPACE and UNLOCK_WORKSPACE can be used to lock/unlock the 
--      access to a workspace. 
--
--  	  LOCK_USER and UNLOCK_USER can be used to lock/unlock the 
--      access to a user.
--
--      In general, any object can be locked in one of the folowing three modes:
--        EXCLUSIVE_MODE - this lock mode conflicts with any other mode.
--        WRITE_MODE - this mode conflicts with EXCLUSIVE_MODE and WRITE_MODE.
--        SHARED_MODE - this mode conflicts only with EXCLUSIVE_MODE.
--      The full lock compatibility table is shown below.
--        When another process holds "held", an attempt to get "get" does
--        the following:
--
--        held           get->  SHARED_MODE  WRITE_MODE  EXCLUSIVE_MODE
--        SHARED_MODE            success      success     failure
--        WRITE_MODE             success      failure     failure
--        EXCLUSIVE_MODE         failure      failure     failure
--
--
--    MODIFIED   (MM/DD/YY)
--    rvelisar    08/21/06 - Created
--

CREATE OR REPLACE PACKAGE OWB_LOCK AS

-----------------------------------------------------------------------------
-- Public constants
-----------------------------------------------------------------------------

NONE_MODE constant integer := 0;
SHARED_MODE constant integer := 4;  -- same as DBMS_LOCK.s_mode
WRITE_MODE constant integer := 5;
EXCLUSIVE_MODE constant integer := 6;  -- same as DBMS_LOCK.x_mode

-----------------------------------------------------------------------------
-- Procedures and functions
-----------------------------------------------------------------------------

-- Procedure ALLOCATE_UNIQUE is a wrapper around DBMS_LOCK.ALLOCATE_UNIQUE, 
-- but it executes it in an autonomous transaction. The parameters are the 
-- same as for DBMS_LOCK.ALLOCATE_UNIQUE.

PROCEDURE ALLOCATE_UNIQUE (
lockname           IN  VARCHAR2,
lockhandle         OUT VARCHAR2,
expiration_secs    IN  INTEGER DEFAULT 864000);


-- Function LOCK_WORKSPACE can be used to request a lock for a workspace, 
-- with a given mode. 
-- Input parameters:
--   workspace_id - the id (elementid) of the workspace to be locked
--   lock_mode    - the mode the lock is requested. Valid modes are: 
--                  SHARED_MODE, WRITE_MODE and EXCLUSIVE_MODE.
-- Return value:
--   true         - if the lock was obtained
--   false        - otherwise

FUNCTION LOCK_WORKSPACE (
workspace_id       IN  INTEGER,
lock_mode          IN  INTEGER) 
RETURN BOOLEAN;


-- Procedure UNLOCK_WORKSPACE can be used to release a lock for a workspace, 
-- you already hold with a given mode. If you don't hold that lock, the 
-- procedure will raise an exception.
-- Input parameters:
--   workspace_id - the id (elementid) of the workspace to be unlocked
--   lock_mode    - the mode the lock was obtained. Valid modes are: 
--                  SHARED_MODE, WRITE_MODE and EXCLUSIVE_MODE.

PROCEDURE UNLOCK_WORKSPACE (
workspace_id       IN  INTEGER,
lock_mode          IN  INTEGER);


-- Function LOCK_USER can be used to request a lock for a user, 
-- with a given mode. 
-- Input parameters:
--   workspace_id - the id (elementid) of the workspace containing the user.
--   user_id      - the id (elementid) of the user to be locked.
--   lock_mode    - the mode the lock is requested. Valid modes are: 
--                  WRITE_MODE and EXCLUSIVE_MODE, and they both behave the 
--                  same.
-- Return value:
--   true         - if the lock was obtained
--   false        - otherwise

FUNCTION LOCK_USER (
workspace_id       IN  INTEGER,
user_id            IN  INTEGER,
lock_mode          IN  INTEGER) 
RETURN BOOLEAN;


-- Procedure UNLOCK_USER can be used to release a lock for a user, 
-- you already hold with a given mode. If you don't hold that lock, the 
-- procedure will raise an exception.
-- Input parameters:
--   workspace_id - the id (elementid) of the workspace containing the user.
--   user_id      - the id (elementid) of the user to be unlocked.
--   lock_mode    - the mode the lock was obtained. Valid modes are: 
--                  WRITE_MODE and EXCLUSIVE_MODE.

PROCEDURE UNLOCK_USER (
workspace_id       IN  INTEGER,
user_id            IN  INTEGER,
lock_mode          IN  INTEGER);


END OWB_LOCK;
