# ------------------------------------------------------------------------
# Define connection Setup for OWB 10gR2
# IMPORTANT:  Make sure you are executing this script in a OWB 10gR2 environment.
# Export objects from OWB 10gR2 repository
# ------------------------------------------------------------------------
set repOwnerUserName owbrep
set repOwnerPassword owbrep
set host repos.us.oracle.com
set port 1521
set service ora1020.us.oracle.com

# ------------------------------------------------------------------------
# Define directory for creating the MDL data files and export log files
# ------------------------------------------------------------------------
set exportdir /data/upgrade



# ------------------------------------------------------------------------
# Define directory for creating the generated Import tcl script
# ------------------------------------------------------------------------
set importdir /data/upgrade


# ------------------------------------------------------------------------
# Do NOT make any changes below this line
# ------------------------------------------------------------------------

proc connectToOWB {} {
  global repOwnerUserName
  global repOwnerPassword
  global host
  global port
  global service

  log "\nConnecting to OWB... in single user mode\n"
  OMBCONNECT $repOwnerUserName/$repOwnerPassword@$host:$port:$service USE SINGLE_USER_MODE
}


proc disconnectFromOWB {} {
  OMBDISCONNECT
  log "Disconnecting from OWB..."
}


proc checkParisRepositoryVersion {} {

  set reposVer [OMBSHOW REPOSITORY_VERSION]

  if {$reposVer == "10.2.0.1.0" ||
      $reposVer == "10.2.0.2.0" ||
      $reposVer == "10.2.0.3.0" } {
     log "\nOWB repository version: $reposVer\n"
   } else {
     log "\n\n"
     log "\nError, repository version $reposVer is not an OWB 10g R2 repository"
     exit(1)
   }
     
}

proc exportProjects {} {
  global exportdir
  global importdir
  global debug

  if {[catch {
     set ProjectList [OMBLIST PROJECTS]
  } result]} {
     log "    $result \n"
     return 0;
  }

  # Public Project MUST be at the beginning of the list
  # If not, then move it to the start of the list

  set ndx [lsearch $ProjectList PUBLIC_PROJECT]
  if {$ndx > 0} {
     set ProjectList [ldelete $ProjectList PUBLIC_PROJECT]
     set Projects [linsert $ProjectList 0 PUBLIC_PROJECT]
  } else {
     set Projects $ProjectList
  }

  foreach p $Projects {

     set filename $exportdir/$p\_prj

     set mdlfilename [getMDLFileName $filename exp]
     set logfilename [getMDLLogName $filename exp]
     set mdlimpname  [getMDLFileName $p\_prj exp]

       if {$p == "PUBLIC_PROJECT"} {
          exportPublicProject $mdlfilename $logfilename $mdlimpname
       } else {

          log "    Exporting Project:  $p"

          if {$debug == 1} {
             log "\n    OMBEXPORT MDL_FILE '$mdlfilename' FROM PROJECT '$p' INCLUDE_GRANTS SUPPORTED_LANGUAGES ('*') OUTPUT LOG '$logfilename' \n\n"          
          }

          if {[catch {
               OMBEXPORT MDL_FILE '$mdlfilename' FROM PROJECT '$p' INCLUDE_GRANTS SUPPORTED_LANGUAGES ('*') OUTPUT LOG '$logfilename'
             } result]} {
                log "    $result"
                log "    Error occurred exporting project:  $p"
                log "       Check log file for details: $logfilename \n"
          } else {             
                # Normal completion           
                writeImportCreateUseUOIDCommand $mdlimpname $p\_prj

                log "       Created mdl file $mdlfilename"
                log "       Created log file $logfilename \n"
          }
       }
  }
}


proc exportPublicProject {mdlfilename logfilename filename} {
  global exportdir
  global importdir
  global adminFileName
  global adminImpName
  global debug
   

  log "PROJECTS"

  log "    Exporting Project:  PUBLIC_PROJECT"
  #
  
  if {$debug == 1} {
     log "\n    OMBEXPORT MDL_FILE '$mdlfilename' FROM PROJECT 'PUBLIC_PROJECT' INCLUDE_GRANTS SUPPORTED_LANGUAGES ('*') OUTPUT LOG '$logfilename' \n\n"          
  }
  
  if {[catch {
     OMBEXPORT MDL_FILE '$mdlfilename' FROM PROJECT 'PUBLIC_PROJECT' INCLUDE_GRANTS SUPPORTED_LANGUAGES ('*') OUTPUT LOG '$logfilename'
  } result]} {
     log "    $result"
     log "    Error occurred exporting project:  PUBLIC_PROJECT"
     log "       Check log file for details: $logfilename \n"
  } else {
     writeImportCreateUseUOIDCommand $filename PUBLIC_PROJECT_prj

     log "       Created mdl file $mdlfilename"
     log "       Created log file $logfilename \n"
  }

}


proc exportSnapshots {} {
  global exportdir
  global importdir
  global debug

  log "SNAPSHOTS"

  if {[catch {
     set Snapshots [OMBLIST SNAPSHOTS]
  } result]} {
     log "    $result \n"
     return 0
  }     
  

  foreach s $Snapshots {

     set filename $exportdir/$s\_snp

     set mdlfilename [getMDLFileName $filename exp]
     set logfilename [getMDLLogName $filename exp]
     set mdlsnapname [getMDLFileName $s\_snp exp]

     log "    Exporting Snapshot:  $s"

     if {$debug == 1} {
        log "\n    OMBEXPORT MDL_FILE '$mdlfilename' COMPONENTS (SNAPSHOT '$s') INCLUDE_USER_DEFINITIONS SUPPORTED_LANGUAGES ('*') OUTPUT LOG '$logfilename' \n\n"          
     }

     if {[catch {
       OMBEXPORT MDL_FILE '$mdlfilename' COMPONENTS (SNAPSHOT '$s') INCLUDE_USER_DEFINITIONS SUPPORTED_LANGUAGES ('*') OUTPUT LOG '$logfilename'
             } result]} {
                log "    $result"
                log "    Error occurred exporting snapshot:  $s"
                log "       Check log file for details: $logfilename \n"
     } else {             
       # Normal completion           
       writeImportCreateUseUOIDCommand $mdlsnapname $s\_snp

       log "       Created mdl file $mdlfilename"
       log "       Created log file $logfilename \n"
     }     
  }
}


proc exportAdminObjects {} {
  global exportdir
  global importdir
  global adminFileName
  global adminImpName
  global repOwnerUserName
  global debug

  set filename $exportdir/ADMINOBJS
  set logfilename [getMDLLogName $filename exp]

  if {[catch {
     set Users [OMBLIST USERS]
  } result]} {
     log "    $result \n"
     set UserComponent " "
  } else {
     # 4/27/07 Repository Owner can be imported into Tokyo repository
     set UserComponent [getComponentNamesCommaDelimited $Users USER]     
  }

  if {[catch {
     set Roles [OMBLIST ROLES]
  } result]} {
     log "    $result \n"
     set RoleComponent " "
  } else {
     set RoleComponent [getComponentNamesCommaDelimited $Roles ROLE]
  }


  if {[catch {
     set tempCmiDef [OMBLIST CMI_DEFINITIONS]
  } result]} {
     log "    $result \n"
     set CmiDefComponent " "
  } else {
     # remove any seeded CMI Definitions from the list
     set CmiDef [ldelete $tempCmiDef ORACLE_EBUSINESS_SUITE]
     set CmiDefComponent [getComponentNamesCommaDelimited $CmiDef CMI_DEFINITION]
  }

# log "\nUsers: <$UserComponent>"
# log "Roles: <$RoleComponent>"
# log "CMIs : <$CmiDefComponent> \n"
 
  set roleComma " "
  set userComma " "
  
  if {$RoleComponent != " "} {
     if {$UserComponent != " " || $CmiDefComponent != " "} {
        set roleComma ","
     }
  }
  
  if {$UserComponent != " "} {
     if {$CmiDefComponent != " "} {
        set userComma ","
     }  
  }

    log "ADMINISTRATION"
    log "    Exporting Users:  $Users"
    log "    Exporting Roles:  $Roles"
    log "    Exporting CMI definitions:  $CmiDef"

    
  if {$UserComponent == " " && $RoleComponent == " " && $CmiDefComponent == " "} {
    log "    Administration objects are NOT exported since they do not exist in the repository.\n"  
    return;
  }
    
  if {[catch {

    if {$debug == 1} {
        log "\n    OMBEXPORT MDL_FILE '$adminFileName' COMPONENTS ($RoleComponent $roleComma $UserComponent $userComma $CmiDefComponent) ALL_CLASS_DEFINITIONS INCLUDE_GRANTS SUPPORTED_LANGUAGES ('*') OUTPUT LOG '$logfilename' \n\n"          
     }

    # Export all admin objects and user-defined definitions 
    OMBEXPORT MDL_FILE '$adminFileName' COMPONENTS ($RoleComponent $roleComma $UserComponent $userComma $CmiDefComponent) ALL_CLASS_DEFINITIONS INCLUDE_GRANTS SUPPORTED_LANGUAGES ('*') OUTPUT LOG '$logfilename'

  } result]} {
    log "\n    OMBEXPORT MDL_FILE '$adminFileName' COMPONENTS ($RoleComponent $roleComma $UserComponent $userComma $CmiDefComponent) ALL_CLASS_DEFINITIONS INCLUDE_GRANTS SUPPORTED_LANGUAGES ('*') OUTPUT LOG '$logfilename'"
    log "\n    $result"
    log "    Error occurred exporting Administration objects"
    log "       Check log file for details: $logfilename \n"
  } else {             
    writeImportUpdateUseUOIDCommand $adminImpName ADMINOBJS

    log "       Created mdl file $adminFileName"
    log "       Created log file $logfilename \n"
  }
}


proc getComponentNamesCommaDelimited {names fco} {
  set componentnames " "
  set i 0
  foreach u $names {
    if {$i == 0} {
       append componentnames $fco " " '$u'
    } else {
       append componentnames ", " $fco " " '$u'
    } 
    incr i
  }

#  log "ComponentsNames: $componentnames"

  return $componentnames
}


proc getCurrentTime {} {
  return [clock format [clock seconds] -format {%a %b %d %H:%M:%S %Z %Y}]
}


proc getMDLFileName {filename type} {
  set mdlfilename $filename\_10_0.mdl
  return $mdlfilename
}


proc getMDLLogName {filename type} {
  set logfilename $filename\_10_0_$type.log
  return $logfilename
}


proc createMDLImportFile {} {
  global importdir
  set ImportScriptName $importdir/OWB_10_Repos_MDLImportFiles.tcl 

#  log "Creating MDL Import Script File:  $ImportScriptName"

  if { [catch {set fileHandle [open $ImportScriptName w]} FileError] } {
     log "Could not open $ImportScriptName for writing\n$FileError"
     exit(1)
  }

  return $fileHandle
}


proc closeMDLImportFile {} {
  global importFile
  close $importFile
}


proc writeImportCreateUseUOIDCommand {mdlfilename filename} {
  global importFile
  
  set logfilename $filename\_10_0_imp.log
  puts $importFile "OMBIMPORT MDL_FILE '\$exportdir/$mdlfilename' INCLUDE_USER_DEFINITIONS USE CREATE_MODE INCLUDE_GRANTS INCLUDE_CMI_DEFINITIONS SUPPORTED_LANGUAGES ('*') OUTPUT LOG TO '\$importdir/$logfilename'"
  # Want to flush each time an import command is written. 

  flush $importFile
}

proc writeImportUpdateUseUOIDCommand {mdlfilename filename} {
  global importFile
  
  set logfilename $filename\_10_0_imp.log
  puts $importFile "OMBIMPORT MDL_FILE '\$exportdir/$mdlfilename' INCLUDE_USER_DEFINITIONS USE UPDATE_MODE INCLUDE_GRANTS INCLUDE_CMI_DEFINITIONS SUPPORTED_LANGUAGES ('*') OUTPUT LOG TO '\$importdir/$logfilename'"
  # Want to flush each time an import command is written. 

  flush $importFile
}

proc writeImportUpdateUseNameCommand {mdlfilename filename} {
  global importFile
  
  set logfilename $filename\_10_0_imp.log
  puts $importFile "OMBIMPORT MDL_FILE '\$exportdir/$mdlfilename' INCLUDE_USER_DEFINITIONS USE UPDATE_MODE MATCH_BY NAMES INCLUDE_GRANTS INCLUDE_CMI_DEFINITIONS SUPPORTED_LANGUAGES ('*') OUTPUT LOG TO '\$importdir/$logfilename'"

  flush $importFile
}


proc createLogFile {} {
  global exportdir
  set LogFileName $exportdir/ExportEntireRepos.log

#  puts "Creating Log File:  $LogFileName"

  if { [catch {set fileHandle [open $LogFileName w]} FileError] } {
     puts "Could not open $LogFileName for writing\n$FileError"
     exit(1)
  }
  
  return $fileHandle
}



proc log {info} {
  global logFile
  
  puts "$info"
  puts $logFile "$info"
  flush $logFile
}


proc closeLogFile {} {
  global logFile
  close $logFile
}


proc ldelete {list value} {
  set ndx [lsearch -exact $list $value]
  if {$ndx >=0} {
     return [lreplace $list $ndx $ndx]
  } else {
     return $list
  }
}


# ------------------------------------------------------------------------
# Main processing begins here
# ------------------------------------------------------------------------
set importFile [createMDLImportFile]
set logFile [createLogFile]
set adminFileName [getMDLFileName $exportdir/ADMINOBJS exp]
set adminImpName [getMDLFileName ADMINOBJS exp]

set debug 0

connectToOWB

log "Export started at [getCurrentTime]\n"

checkParisRepositoryVersion

exportAdminObjects
exportProjects
exportSnapshots

log "Export ended at [getCurrentTime]\n"

closeMDLImportFile

disconnectFromOWB

closeLogFile

