#
# This file is the OMB+ initialization file. It lists some 
# predefined procedures, loads the OMB+ Tcl extension
# class, and sets the prompt.
#


####################################################################
#
# Procedure OMBToSettableString
# Input: A Tcl String
# Output: A Tcl String with all single-quotes stuffed
# Since: Calais
# Description: Useful when setting string values which contain
#              single quotes that need to be escaped.
#
####################################################################

proc OMBToSettableString { str } {
   set ans $str
   regsub -all {'} $ans {''} ans
   return $ans
} 

####################################################################
#
# Procedure OMBToTypeObjListString
# Input: A 2D list of the form {{<object_type> <name>} ... }
# Output: A Tcl String of the form "<object_type> <name>,..."
# Since: Calais
# Description: convert the input 2D list to comma-separated string
#              e.g. output from OMBRETRIEVE COLLECTION could use
#              this procedure to pipe into OMBEXPORT MDL_FILE
#
####################################################################

proc OMBToTypeObjListString {inputlist} {
    set s {}
    foreach x $inputlist {
       append result $s [join $x " "]
       set s ","
    }
    return $result
} 

####################################################################
#
# Procedure OMB2DListToArrayList 
# Input: A 2D list of the form {{<object_type> <name>} ... }
# Output: A Tcl list of the form "<object_type> <name>..."
# Since: Paris 
# Description: convert the input 2D list to list that can be set
#              as value to a Tcl Array.
#
####################################################################

proc OMB2DListToArrayList {inputlist} {
    set ans {}
    foreach elem $inputlist {
       lappend ans [lindex $elem 0] [lindex $elem 1]
    }
    return $ans
} 

####################################################################
#
# Procedure OMBPageBreak
# Input: nLines The number of lines in a page
#        str The string to be split into pages
# Return: None
# Since: Le Mans
# Description: Display the input string as a sequence of pages, 
#              pausing after each page. The height of each page is 
#              determined be the number of lines passes as the first
#              argument. An example when this procedure can be 
#              useful is when OMBHELP returns a description much 
#              longer than the page height, and scrolling up to see 
#              the whole text is not possible or difficult (like
#              for Accessibility)
#
####################################################################

proc OMBPageBreak {nLines str} {
  if { $nLines<=0 } {
    puts "The number of lines in a page must be greater than 0!"
    return
  }
  set l [split $str \n]
  set ll [llength $l]
  set i 0
  
  while {$i < $ll} {
    puts [lindex $l $i]
    incr i  
    if { [expr $i%$nLines]==0 && $i!=$ll} {
      puts -nonewline "Press <Enter> to continue..."
      flush stdout
      gets stdin
    }
  }
}


package require java
java::load oracle.owb.scripting.OMBTclExtension
catch {java::load oracle.owb.scripting.OMUTclExtension}
set tcl_prompt1 { puts -nonewline "OMB+> " } 


########################################################################
#
# Tcl procedure defining the error reporting logic for task flow
# execution.
#
# @author Justin Ho 
# @since Paris
#
########################################################################

proc setErrorTrace { } {
   global errorInfo
   set debugger [java::call -noconvert "oracle.wh.service.sdk.taskflow.execution.TclDebugger" "getInstance"]
   $debugger -noconvert "setStackTrace" $errorInfo
}


########################################################################
#
# Tcl procedure defining the trigger logic for global variables. 
#
# @param name1 the name of the variable to trace in the trace command.
# @param name2 unused
# @param op the operation performed on the trace variable.
#
# @author Justin Ho
# @since Paris
#
########################################################################

proc _OMBGlobalVariableTrigger { varName name2 op } {
   upvar $varName gvar
   global $varName
   if { $op == "r" } {
      set globalVarValueProvider [java::call -noconvert "oracle.owb.scripting.environment.GlobalVarValueProvider" "getInstance"]
      set $varName [ [$globalVarValueProvider "getValue" [java::getinterp] $varName] "toString"]
   } elseif { $op == "w" } {
      error "Global variable $varName is read-only"
   } elseif { $op == "u" } {
      trace variable $varName rwu _OMBGlobalVariableTrigger
      error "Global variable $varName cannot be unset"
   }
} 

trace variable OMB_CURRENT_PROJECT rwu _OMBGlobalVariableTrigger
trace variable OMB_CURRENT_SESSION rwu _OMBGlobalVariableTrigger
trace variable OMB_CURRENT_USER    rwu _OMBGlobalVariableTrigger

# TCLLIBPATH fix: the handling of TCLLIBPATH in JACL does not handle multiple
# path entries correctly
if {[lsearch [array name env] TCLLIBPATH] >= 0} {
  set libindex [lsearch $auto_path $env(TCLLIBPATH)]
  if {$libindex >= 0} {
    set auto_path [lreplace $auto_path $libindex $libindex]
    foreach dir $env(TCLLIBPATH) {
      if {[lsearch -exact $auto_path $dir] < 0} {
        lappend auto_path $dir
      }
    }
    if {[lsearch [info vars] dir] >= 0} {
      unset dir
    }
  }
  unset libindex
}
