Rem  part_owb.sql
Rem
Rem  Copyright (c) Oracle Corporation 2007. All Rights Reserved.
Rem
Rem   Based on the following Oracle Work Flow code:
Rem   [wfupart.sql 26.2 2001/11/07 21:26:53 mcraig noship]
Rem
Rem    NAME
Rem      part_owb.sql - This (Upgrade) script partitions
Rem      selected OWBSYS tables by Workspace ID.
Rem
Rem    DESCRIPTION
Rem      This SQL script migrates existing non-partitioned OWBSYS
Rem      tables to partitioned tables. You should make sure
Rem      your database is backed up. Also you should backup the
Rem      your OWBSYS schema, or at least all tables striped by workspace ID
Rem
Rem    NOTES
Rem      This is a sample script to partition OWBSYS tables by
Rem      Workspace ID (PARTITION BY LIST).
Rem      !!!For 9.2 Databases & Up, ONLY with Partition Option!!!
Rem      Make sure you run this script on a test instance first.
Rem
Rem
Rem
Rem    MODIFIED   (MM/DD/YYYY)
Rem
Rem    gggraham    03/29/2007 - Creation
Rem

-- add this to exit at error
-- at the connection testing stage, we cannot use exit failure rollback
whenever sqlerror exit failure;

set serveroutput on size 1000000

prompt Please review this script properly and ensure you have enough space
prompt in your tablespace, as well as enqueue_resources in your init.ora.
prompt Usage:
prompt   sqlplus <owbsys>/<owbsys_passwd> @part_owb <owbsys> <owbsys_passwd>

def u_app = &1
def p_app = &2

-- Testing connection to OWBSYS Account
prompt Test connection to &u_app
connect &u_app/&p_app;

-- All connected properly, now we can proceed with package creation.
whenever sqlerror exit failure rollback;

-- Create Packages
prompt Creating OWB_PARTITION package ...

create or replace package Owb_Partition
authid current_user as
g_partition_ct  number := 0;
Procedure CreateTable (
  tblname    in varchar2,
  tblspcname in varchar2 default null
);

Procedure CreateIndex (
  idxname    in varchar2,
  tblname    in varchar2,
  collist    in varchar2,
  primary    in boolean default false,
  tblspcname in varchar2 default null
);

-- objtype are like 'VIEW' and 'PACKAGE BODY'
-- objname can have wild card like 'WF%'
Procedure CompileObject (
  objtype    in varchar2,
  objname    in varchar2 default '%'
);

end Owb_Partition;
/

create or replace package body Owb_Partition as
Procedure CreateTable (
  tblname    in varchar2,
  tblspcname in varchar2 default null
)
is
  v_tablespace_name  varchar2(30);
  v_initial          number;
  v_next             number;
  v_pctinc           number;
  v_partitioned      varchar2(3);
  v_new_table        varchar2(30);
  v_old_table        varchar2(30);
  v_sql              varchar2(32000);
  l_sql              varchar2(32000);
  v_dummy            varchar2(1);
  i                  pls_integer := 0;

  cursor c is
  select workspaceID
  from owbsys.cmpworkspace_v
  order by workspaceID asc;

begin
  begin

    for c_rec in c loop
      i := i+1;
      l_sql := l_sql||
         ' partition owb_ws'||to_char(i)||' values('||to_char(c_rec.workspaceID)||'),';
    end loop;

    -- Set the Global Partition Count
    g_partition_ct := i;

    dbms_output.put_line('Number of Partitions to be created:'||to_char(i+1));

    select TABLESPACE_NAME, INITIAL_EXTENT, nvl(NEXT_EXTENT,10), nvl(PCT_INCREASE,0),
           PARTITIONED
    into   v_tablespace_name, v_initial, v_next, v_pctinc, v_partitioned
    from   USER_TABLES
    where  TABLE_NAME = tblname;
    dbms_output.put_line('tablespace name:'||v_tablespace_name);
    dbms_output.put_line('initial:'||v_initial);
    dbms_output.put_line('next:'||v_next);
    dbms_output.put_line('pctinc:'||v_pctinc);
    dbms_output.put_line('partitioned:'||v_partitioned);
  exception
    when NO_DATA_FOUND then
      dbms_output.put_line('--Error while querying storage parameters.');
      dbms_output.put_line('-- Table '||tblname||' does not exist.');
      dbms_output.put_line('-- Please check if you login to the correct user, database,');
      dbms_output.put_line('-- and that application installation has been completed successfully.');
      raise_application_error(-20001,'Missing table '||tblname);

    when OTHERS then
      dbms_output.put_line('--Error while querying storage parameters.');
      raise;
  end;

  if (v_partitioned = 'YES') then
    -- already partitioned.  Just exit.
    dbms_output.put_line('Table '||tblname||' already partitioned ... skipped.');
    return;
  end if;

  if (tblspcname is not null) then
    v_tablespace_name := tblspcname;
  end if;

  v_new_table := substr('WBN_'||tblname, 1, 30);

  -- check if new table name exists, if so, may ask to drop it.
  begin
    select null into v_dummy from sys.dual where exists (
      select 1
      from   USER_OBJECTS
      where  OBJECT_NAME = v_new_table
    );
    dbms_output.put_line('Name conflict.  Please first drop '||v_new_table);
    raise_application_error(-20002, v_new_table||' already exists.');
  exception
    when NO_DATA_FOUND then
      null;
    when OTHERS then
      dbms_output.put_line('--Error while checking the new table name.');
      raise;
  end;

  if (v_initial is null) then
    v_initial := 16384;  -- force it to 16k
  end if;

  dbms_output.put_line('Creating new table '||v_new_table);
  v_sql :='create table '||v_new_table||
          ' pctfree 40'||
          ' pctused 60'||
          ' initrans 10'||
          ' tablespace '||v_tablespace_name||
          ' storage (initial '||to_char(v_initial)||' next '||to_char(v_next) ||
          'M'||
          ' freelists 32 freelist groups 10'||
          ' pctincrease '||to_char(v_pctinc)||')' ||
          ' parallel (degree 1)'||
          ' nologging'||
          ' partition by list (workspaceID) (' ||l_sql||
          ' partition owb_ws'||to_char(i+1)||' values (DEFAULT))' ||
          ' as select * from '||tblname;

  begin
    execute immediate v_sql;
  exception
    when OTHERS then
      dbms_output.put_line('Error in SQL: '||substr(v_sql,1,200));
      dbms_output.put_line('Error in SQL: '||substr(v_sql,201,200));
      dbms_output.put_line('Error in SQL: '||substr(v_sql,401,200));
      raise;
  end;

  -- rename orig table to old table
  v_old_table := substr('WBO_'||tblname, 1, 30);
  dbms_output.put_line('Rename '||tblname||' to '||v_old_table);
  v_sql := 'alter table '||tblname||' rename to '||v_old_table;
  begin
    execute immediate v_sql;
  exception
    when OTHERS then
      dbms_output.put_line('Error in: '||substr(v_sql,1,220));
      raise;
  end;

  -- rename new table to orig table
  dbms_output.put_line('Rename '||v_new_table||' to '||tblname);
  v_sql := 'alter table '||v_new_table||' rename to '||tblname;
  begin
    execute immediate v_sql;
  exception
    when OTHERS then
      raise_application_error(-20005,'Error in SQL: '||substr(v_sql,1,3000));
  end;

exception
  when OTHERS then
    dbms_output.put_line('--Error in CreateTable '||tblname);
    raise;
end CreateTable;

Procedure CreateIndex (
  idxname    in varchar2,
  tblname    in varchar2,
  collist    in varchar2,
  primary    in boolean default false,
  tblspcname in varchar2 default null
)
is
  v_tablespace_name  varchar2(30);
  v_initial          number;
  v_next             number;
  v_pctinc           number;
  v_tblpartitioned   varchar2(3);
  v_idxpartitioned   varchar2(3);
  v_prefix           varchar2(8);
  v_new_index        varchar2(30);
  v_old_index        varchar2(30);
  v_sql              varchar2(32000);
  l_sql              varchar2(32000);
  v_dummy            varchar2(1);
  partition_index    boolean := false;
  drop_index         boolean := false;
  i                  pls_integer := 0;

begin
  begin

    --for c_rec in c loop
    for x in 1..g_partition_ct loop
      i := i+1;
      if (l_sql is null) then
        l_sql :=
         ' partition owb_ws'||to_char(i);
      else
        l_sql := l_sql||
         ' ,partition owb_ws'||to_char(i);
      end if;
    end loop;

    l_sql := l_sql||' ,partition owb_ws'||to_char(i+1);

    select I.TABLESPACE_NAME, I.INITIAL_EXTENT, nvl(I.NEXT_EXTENT,1), nvl(I.PCT_INCREASE,0),
           I.PARTITIONED, T.PARTITIONED
    into   v_tablespace_name, v_initial, v_next, v_pctinc,
           v_idxpartitioned, v_tblpartitioned
    from   USER_INDEXES I, USER_TABLES T
    where  I.INDEX_NAME = idxname
    and    I.TABLE_NAME = T.TABLE_NAME;

  exception
    when NO_DATA_FOUND then
      dbms_output.put_line('--Error while querying storage parameters.');
      dbms_output.put_line('-- Index '||idxname||' does not exist.');
      dbms_output.put_line('-- Please check if you login to the correct user, database,');
      dbms_output.put_line('-- and that application installation has been completed successfully.');
      raise_application_error(-20011,'Missing index '||idxname);

    when OTHERS then
      dbms_output.put_line('--Error while querying storage parameters.');
      raise;
  end;

  if (tblspcname is not null) then
    v_tablespace_name := tblspcname;
  end if;

  v_new_index := substr('WBN_'||idxname, 1, 30);

  -- check if new index name exists, if so, may ask to drop it.
  begin
    select null into v_dummy from sys.dual where exists (
      select 1
      from   USER_OBJECTS
      where  OBJECT_NAME = v_new_index
    );
    dbms_output.put_line('Name conflict.  Please first drop '||v_new_index);
    raise_application_error(-20012, v_new_index||' already exists.');
  exception
    when NO_DATA_FOUND then
      null;
    when OTHERS then
      dbms_output.put_line('--Error while checking the new index name.');
      raise;
  end;

  dbms_output.put_line('Recreating index '||idxname);

  v_prefix := 'WBO_';
  if (v_initial is null) then
    v_initial := 16384; -- force it to 16k
  end if;
  partition_index := true;
  if (primary) then
    v_sql := 'create unique index '||v_new_index;
  else
    v_sql := 'create index '||v_new_index;
  end if;
  v_sql := v_sql ||
           ' on ' || tblname || ' (WORKSPACEID,' || collist || ')' ||
           ' pctfree 40'||
           ' initrans 10'||
           ' tablespace '||v_tablespace_name||
           ' storage (initial '||to_char(v_initial)||' next '||to_char(v_next)||'M'||
           ' freelists 32 freelist groups 10'||
           ' pctincrease '||to_char(v_pctinc)||')' ||
           ' nologging compute statistics'||
           ' local ('||l_sql||')';

  -- now check if we need to proceed
  if (partition_index) then
    if (v_idxpartitioned = 'YES') then
      -- already partitioned.  Just exit.
      dbms_output.put_line('Index '||idxname||' already partitioned ... skipped.');
      return;
    elsif (v_tblpartitioned = 'YES') then
      -- first drop the index
      drop_index := true;
      begin
        dbms_output.put_line('Drop index '||idxname);
        execute immediate 'drop index '||idxname;
      exception
        when OTHERS then
          dbms_output.put_line('Error in SQL: '||substr('drop index '||
                               idxname,1,200));
          raise;
      end;

    end if;
  else
    if (v_tblpartitioned = 'YES') then
      -- Detected table partitioned meant this script had already been run.
      -- Just exit.
      dbms_output.put_line('Table for index '||idxname||' already partitioned ... skipped.');
      return;
    end if;
  end if;

  begin
    dbms_output.put_line('Create index '||v_new_index);
    execute immediate v_sql;
  exception
    when OTHERS then
      dbms_output.put_line('Error in SQL: '||substr(v_sql,1,220));
      raise;
  end;

  -- Rename only if we did not drop index before
  if (not drop_index) then
    -- rename orig index to old index
    v_old_index := substr(v_prefix||idxname, 1, 30);
    dbms_output.put_line('Rename '||idxname||' to '||v_old_index);
    v_sql := 'alter index '||idxname||' rename to '||v_old_index;
    begin
      execute immediate v_sql;
    exception
      when OTHERS then
        dbms_output.put_line('Error in SQL: '||substr(v_sql,1,220));
        raise;
    end;
  end if;

  -- rename new index to orig index
  dbms_output.put_line('Rename '||v_new_index||' to '||idxname);
  v_sql := 'alter index '||v_new_index||' rename to '||idxname;
  begin
    execute immediate v_sql;
  exception
    when OTHERS then
      dbms_output.put_line('Error in SQL: '||substr(v_sql,1,220));
      raise;
  end;

exception
  when OTHERS then
    dbms_output.put_line('--Error in CreateIndex '||idxname);
    raise;
end CreateIndex;

Procedure CompileObject (
  objtype    in varchar2,
  objname    in varchar2 default '%'
)
is
  cursor objcurs(objt varchar2, objn varchar2) is
    select OBJECT_NAME
    from   ALL_OBJECTS
    where  OBJECT_TYPE = objt
    and    OBJECT_NAME like objn
    and    STATUS = 'INVALID';

  v_sql varchar2(4000);

begin
  for obj in objcurs(objtype, objname) loop
    if (objtype = 'PACKAGE BODY') then
      v_sql := 'alter package '||obj.OBJECT_NAME||' compile body';
    else
      v_sql := 'alter '||objtype||' '||obj.OBJECT_NAME||' compile';
    end if;

    dbms_output.put_line('Running: '||v_sql);

    begin
      execute immediate v_sql;
    exception
      when OTHERS then
        -- continues when error is caught.
        dbms_output.put_line('--Error in SQL: '||v_sql);
    end;
  end loop;
exception
  when OTHERS then
    dbms_output.put_line('--Error in CompileObject '||objname);
    raise;
end CompileObject;

end Owb_Partition;
/

-- show errors package body OWB_PARTITION;

whenever sqlerror exit failure rollback;
set serveroutput on

prompt Recreating tables.  May take a long time...

prompt Recreating table ALLASSOCTABLE
begin
  dbms_output.enable(1000000);

  -- create table. Must contain workspaceID as a column.
  Owb_Partition.CreateTable('ALLASSOCTABLE');

  -- create index. Assumes workspaceID appears as first column in each index.
  Owb_Partition.CreateIndex('IDX_ASSOCTABLE_NEWID','ALLASSOCTABLE','NEWID',false);
  Owb_Partition.CreateIndex('IDX_ASSOCTABLE_OLDID','ALLASSOCTABLE','OLDID',false);
end;
/

-- drop old table
whenever sqlerror continue;
prompt drop table WBO_ALLASSOCTABLE
drop table WBO_ALLASSOCTABLE;
whenever sqlerror exit failure rollback;

--prompt Recreating table ALLCHANGELOG
--begin
--  dbms_output.enable(1000000);

  -- note: This is an indexed organized table.  Need an IOT specific procedure
  -- create table. Must contain workspaceID as a column.
  -- Owb_Partition.CreateTable('ALLCHANGELOG',true);

  -- create index. Assumes workspaceID appears as first column in each index.
  --Owb_Partition.CreateIndex('CHANGELOG_PK','ALLCHANGELOG','ELEMENTID',true);
--end;
--/

-- drop old table
-- note: This is an indexed organized table.  Need an IOT specific procedure
--whenever sqlerror continue;
--prompt drop table WBO_ALLCHANGELOG
--drop table WBO_ALLCHANGELOG;
--whenever sqlerror exit failure rollback;

prompt Recreating table ALLCOMPARESTORE
begin
  dbms_output.enable(1000000);

  -- create table. Must contain workspaceID as a column.
  Owb_Partition.CreateTable('ALLCOMPARESTORE');

  -- create index. Assumes workspaceID appears as first column in each index.
  Owb_Partition.CreateIndex('IDX_COMPARESTORE_ASSOCCOL','ALLCOMPARESTORE','ASSOCCOL',false);
  Owb_Partition.CreateIndex('IDX_COMPARESTORE_ASSOCTYPE','ALLCOMPARESTORE','ASSOCTYPE',false);
  Owb_Partition.CreateIndex('IDX_COMPARESTORE_COMPAREID','ALLCOMPARESTORE','COMPAREID',false);
  Owb_Partition.CreateIndex('IDX_COMPARESTORE_DIFFCOL','ALLCOMPARESTORE','DIFFCOL',false);
  Owb_Partition.CreateIndex('IDX_COMPARESTORE_UOID','ALLCOMPARESTORE','UOID',false);
end;
/

-- drop old table
whenever sqlerror continue;
prompt drop table WBO_ALLCOMPARESTORE
drop table WBO_ALLCOMPARESTORE;
whenever sqlerror exit failure rollback;

prompt Recreating table ALLFCOFOLDERNAMESPACESTORAGE
begin
  dbms_output.enable(1000000);

  -- create table. Must contain workspaceID as a column.
  Owb_Partition.CreateTable('ALLFCOFOLDERNAMESPACESTORAGE');

  -- create index. Assumes workspaceID appears as first column in each index.
  Owb_Partition.CreateIndex('FCOFOLDERNSSTORAGE_PK','ALLFCOFOLDERNAMESPACESTORAGE','SNAPSHOTID,PARENTID,DOMAIN,NAME,ISPHYS,LANGUAGEISOID',false);
  Owb_Partition.CreateIndex('IDX_FCONSSTORAGE_AGGPARENT','ALLFCOFOLDERNAMESPACESTORAGE','SNAPSHOTID,AGGREGATEPARENT,ELEMENTID,ISPHYS',true);
  Owb_Partition.CreateIndex('IDX_FCONSSTORAGE_ELEMID','ALLFCOFOLDERNAMESPACESTORAGE','SNAPSHOTID,ELEMENTID,ISPHYS',true);
end;
/

-- drop old table
whenever sqlerror continue;
prompt drop table WBO_ALLFCOFOLDERNAMESPACESTORA
drop table WBO_ALLFCOFOLDERNAMESPACESTORA;
whenever sqlerror exit failure rollback;

prompt Recreating table ALLINTERLINK
begin
  dbms_output.enable(1000000);

  -- create table. Must contain workspaceID as a column.
  Owb_Partition.CreateTable('ALLINTERLINK');

  -- create index. Assumes workspaceID appears as first column in each index.
  Owb_Partition.CreateIndex('IDX_INTERLINK_LINKFROM','ALLINTERLINK','LINKFROM',false);
  Owb_Partition.CreateIndex('IDX_INTERLINK_LINKTO','ALLINTERLINK','LINKTO',false);
end;
/

-- drop old table
whenever sqlerror continue;
prompt drop table WBO_ALLINTERLINK
drop table WBO_ALLINTERLINK;
whenever sqlerror exit failure rollback;

prompt Recreating table ALLINTERLINKSTORAGE
begin
  dbms_output.enable(1000000);

  -- create table. Must contain workspaceID as a column.
  Owb_Partition.CreateTable('ALLINTERLINKSTORAGE');

  -- create index. Assumes workspaceID appears as first column in each index.
  Owb_Partition.CreateIndex('IDX_INTERLINK_S_COMPUOID','ALLINTERLINKSTORAGE','COMPUOID,SNAPSHOTID',false);
  Owb_Partition.CreateIndex('IDX_INTERLINK_S_LINKFROM','ALLINTERLINKSTORAGE','LINKFROM,SNAPSHOTID',false);
  Owb_Partition.CreateIndex('IDX_INTERLINK_S_LINKTO','ALLINTERLINKSTORAGE','LINKTO,SNAPSHOTID',false);
  Owb_Partition.CreateIndex('IDX_INTERLINK_S_SNAPID','ALLINTERLINKSTORAGE','SNAPSHOTID',false);
end;
/

-- drop old table
whenever sqlerror continue;
prompt drop table WBO_ALLINTERLINKSTORAGE
drop table WBO_ALLINTERLINKSTORAGE;
whenever sqlerror exit failure rollback;

prompt Recreating table ALLINTRALINK
begin
  dbms_output.enable(1000000);

  -- create table. Must contain workspaceID as a column.
  Owb_Partition.CreateTable('ALLINTRALINK');

  -- create index. Assumes workspaceID appears as first column in each index.
  Owb_Partition.CreateIndex('IDX_INTRALINK_LINKFROM','ALLINTRALINK','LINKFROM',false);
  Owb_Partition.CreateIndex('IDX_INTRALINK_LINKTO','ALLINTRALINK','LINKTO',false);
end;
/

-- drop old table
whenever sqlerror continue;
prompt drop table WBO_ALLINTRALINK
drop table WBO_ALLINTRALINK;
whenever sqlerror exit failure rollback;

prompt Recreating table ALLINTRALINKSTORAGE
begin
  dbms_output.enable(1000000);

  -- create table. Must contain workspaceID as a column.
  Owb_Partition.CreateTable('ALLINTRALINKSTORAGE');

  -- create index. Assumes workspaceID appears as first column in each index.
  Owb_Partition.CreateIndex('IDX_INTRALINK_S_COMPUOID','ALLINTRALINKSTORAGE','COMPUOID,SNAPSHOTID',false);
  Owb_Partition.CreateIndex('IDX_INTRALINK_S_LINKFROM','ALLINTRALINKSTORAGE','LINKFROM,SNAPSHOTID',false);
  Owb_Partition.CreateIndex('IDX_INTRALINK_S_LINKTO','ALLINTRALINKSTORAGE','LINKTO,SNAPSHOTID',false);
  Owb_Partition.CreateIndex('IDX_INTRALINK_S_SNAPID','ALLINTRALINKSTORAGE','SNAPSHOTID',false);
end;
/

-- drop old table
whenever sqlerror continue;
prompt drop table WBO_ALLINTRALINKSTORAGE
drop table WBO_ALLINTRALINKSTORAGE;
whenever sqlerror exit failure rollback;

prompt Recreating table ALLLWCONTEXT
begin
  dbms_output.enable(1000000);

  -- create table. Must contain workspaceID as a column.
  Owb_Partition.CreateTable('ALLLWCONTEXT');

  -- create index. Assumes workspaceID appears as first column in each index.
  Owb_Partition.CreateIndex('IDX_LWCONTEXT_ASSOCCOL','ALLLWCONTEXT','ASSOCCOL,SNAPSHOTID',false);
  Owb_Partition.CreateIndex('IDX_LWCONTEXT_ASSOCID','ALLLWCONTEXT','ASSOCID,SNAPSHOTID',false);
  Owb_Partition.CreateIndex('IDX_LWCONTEXT_ASSOCTYPE','ALLLWCONTEXT','ASSOCTYPE,SNAPSHOTID',false);
  Owb_Partition.CreateIndex('IDX_LWCONTEXT_DIFFCOL','ALLLWCONTEXT','DIFFCOL,SNAPSHOTID',false);
  Owb_Partition.CreateIndex('IDX_LWCONTEXT_ELEMENTID','ALLLWCONTEXT','ELEMENTID,SNAPSHOTID',false);
  Owb_Partition.CreateIndex('IDX_LWCONTEXT_SNAPID','ALLLWCONTEXT','SNAPSHOTID',false);
  Owb_Partition.CreateIndex('IDX_LWCONTEXT_UOID_COMPUOID','ALLLWCONTEXT','UOID,COMPUOID',false);
end;
/

-- drop old table
whenever sqlerror continue;
prompt drop table WBO_ALLLWCONTEXT
drop table WBO_ALLLWCONTEXT;
whenever sqlerror exit failure rollback;

prompt Recreating table ALLPCTREE
begin
  dbms_output.enable(1000000);

  -- create table. Must contain workspaceID as a column.
  Owb_Partition.CreateTable('ALLPCTREE');

  -- create index. Assumes workspaceID appears as first column in each index.
  Owb_Partition.CreateIndex('IDX_PCTREE_CHILDID','ALLPCTREE','CHILDID',true);
  Owb_Partition.CreateIndex('IDX_PCTREE_CHILDID_PARENTID','ALLPCTREE','CHILDID,PARENTID',true);
  Owb_Partition.CreateIndex('IDX_PCTREE_PARENTID','ALLPCTREE','PARENTID',false);
  Owb_Partition.CreateIndex('IDX_PCTREE_PARENTID_CHILDID','ALLPCTREE','PARENTID,CHILDID',true);
end;
/

-- drop old table
whenever sqlerror continue;
prompt drop table WBO_ALLPCTREE
drop table WBO_ALLPCTREE;
whenever sqlerror exit failure rollback;

prompt Recreating table ALLPCTREESTORAGE
begin
  dbms_output.enable(1000000);

  -- create table. Must contain workspaceID as a column.
  Owb_Partition.CreateTable('ALLPCTREESTORAGE');

  -- create index. Assumes workspaceID appears as first column in each index.
  Owb_Partition.CreateIndex('IDX_PCTREESTORAGE_CID','ALLPCTREESTORAGE','CHILDID',false);
  Owb_Partition.CreateIndex('IDX_PCTREESTORAGE_COMPUOID','ALLPCTREESTORAGE','COMPUOID',false);
  Owb_Partition.CreateIndex('IDX_PCTREESTORAGE_SNAPID','ALLPCTREESTORAGE','SNAPSHOTID',false);
end;
/

-- drop old table
whenever sqlerror continue;
prompt drop table WBO_ALLPCTREESTORAGE
drop table WBO_ALLPCTREESTORAGE;
whenever sqlerror exit failure rollback;

prompt Recreating table ALLSNAPSHOTLOOKUP
begin
  dbms_output.enable(1000000);

  -- create table. Must contain workspaceID as a column.
  Owb_Partition.CreateTable('ALLSNAPSHOTLOOKUP');

  -- create index. Assumes workspaceID appears as first column in each index.
  Owb_Partition.CreateIndex('IDX_LOOKUPSNAPID','ALLSNAPSHOTLOOKUP','SNAPSHOTID',false);
  Owb_Partition.CreateIndex('IDX_LOOKUPSNAPNAME','ALLSNAPSHOTLOOKUP','SNAPSHOTNAME',false);
end;
/

-- drop old table
whenever sqlerror continue;
prompt drop table WBO_ALLSNAPSHOTLOOKUP
drop table WBO_ALLSNAPSHOTLOOKUP;
whenever sqlerror exit failure rollback;

prompt Recreating table ALLSNAPSHOTSTORETABLE
begin
  dbms_output.enable(1000000);

  -- create table. Must contain workspaceID as a column.
  Owb_Partition.CreateTable('ALLSNAPSHOTSTORETABLE');

  -- create index. Assumes workspaceID appears as first column in each index.
  Owb_Partition.CreateIndex('ALLSNAPSHOTSTORETABLE_PK','ALLSNAPSHOTSTORETABLE','SNAPSHOTID,UOID',true);
  Owb_Partition.CreateIndex('IDX_SNAPSTORESNAPID','ALLSNAPSHOTSTORETABLE','SNAPSHOTID',false);
  Owb_Partition.CreateIndex('IDX_SST_IS_ROOT','ALLSNAPSHOTSTORETABLE','ISROOT',false);
  Owb_Partition.CreateIndex('IDX_SST_UOID','ALLSNAPSHOTSTORETABLE','UOID',false);
  Owb_Partition.CreateIndex('IDX_SST_USERNAME','ALLSNAPSHOTSTORETABLE','USERNAME',false);
end;
/

-- drop old table
whenever sqlerror continue;
prompt drop table WBO_ALLSNAPSHOTSTORETABLE
drop table WBO_ALLSNAPSHOTSTORETABLE;
whenever sqlerror exit failure rollback;

prompt Recreating table CMPFCOCLASSES
begin
  dbms_output.enable(1000000);

  -- create table. Must contain workspaceID as a column.
  Owb_Partition.CreateTable('CMPFCOCLASSES');

  -- create index. Assumes workspaceID appears as first column in each index.
  Owb_Partition.CreateIndex('IDX_FCOCLASSNAMEELEMID','CMPFCOCLASSES','S2_1,I_1',false);
  Owb_Partition.CreateIndex('IDX_FCOELEMENTID','CMPFCOCLASSES','I_1',true);
  Owb_Partition.CreateIndex('IDX_FCONAME','CMPFCOCLASSES','S2_3,S2_1',false);
  Owb_Partition.CreateIndex('IDX_FCOOWNINGFOLDER','CMPFCOCLASSES','R_10',false);
  Owb_Partition.CreateIndex('IDX_FCOUOID','CMPFCOCLASSES','S2_5',false);
end;
/

-- drop old table
whenever sqlerror continue;
prompt drop table WBO_CMPFCOCLASSES
drop table WBO_CMPFCOCLASSES;
whenever sqlerror exit failure rollback;

prompt Recreating table CMPFCOSTORAGE
begin
  dbms_output.enable(1000000);

  -- create table. Must contain workspaceID as a column.
  Owb_Partition.CreateTable('CMPFCOSTORAGE');

  -- create index. Assumes workspaceID appears as first column in each index.
  Owb_Partition.CreateIndex('IDX_FCOACTIVECLASSNAME_VER','CMPFCOSTORAGE','ACTIVE,S2_1',false);
  Owb_Partition.CreateIndex('IDX_FCOCOMPUOID_VER','CMPFCOSTORAGE','COMPUOID',false);
  Owb_Partition.CreateIndex('IDX_FCOELEMENTID_VER','CMPFCOSTORAGE','I_1,SNAPSHOTID',true);
  Owb_Partition.CreateIndex('IDX_FCOFCOCLASS_VER','CMPFCOSTORAGE','R_11,S2_1',false);
  Owb_Partition.CreateIndex('IDX_FCOLOADINGPARENT_VER','CMPFCOSTORAGE','LOADINGPARENT',false);
  Owb_Partition.CreateIndex('IDX_FCOOWNINGFOLDER_VER','CMPFCOSTORAGE','R_10',false);
  Owb_Partition.CreateIndex('IDX_FCOSNAPID_VER','CMPFCOSTORAGE','SNAPSHOTID',false);
  Owb_Partition.CreateIndex('IDX_FCOUOID_VER','CMPFCOSTORAGE','S2_5',false);
end;
/

-- drop old table
whenever sqlerror continue;
prompt drop table WBO_CMPFCOSTORAGE
drop table WBO_CMPFCOSTORAGE;
whenever sqlerror exit failure rollback;

prompt Recreating table CMPMMMCLASSES
begin
  dbms_output.enable(1000000);

  -- create table. Must contain workspaceID as a column.
  Owb_Partition.CreateTable('CMPMMMCLASSES');

  -- create index. Assumes workspaceID appears as first column in each index.
  Owb_Partition.CreateIndex('IDX_MMMCLASSNAMEELEMID','CMPMMMCLASSES','S2_2,I_1',false);
  Owb_Partition.CreateIndex('IDX_MMMELEMENTID','CMPMMMCLASSES','I_1',true);
  Owb_Partition.CreateIndex('IDX_MMMLOADINGPARENT_ELEMID','CMPMMMCLASSES','LOADINGPARENT,I_1',false);
  Owb_Partition.CreateIndex('IDX_MMMNAME','CMPMMMCLASSES','S2_4,S2_2',false);
  Owb_Partition.CreateIndex('IDX_MMMUOID','CMPMMMCLASSES','S2_5',false);
end;
/

-- drop old table
whenever sqlerror continue;
prompt drop table WBO_CMPMMMCLASSES
drop table WBO_CMPMMMCLASSES;
whenever sqlerror exit failure rollback;

prompt Recreating table CMPMMMSTORAGE
begin
  dbms_output.enable(1000000);

  -- create table. Must contain workspaceID as a column.
  Owb_Partition.CreateTable('CMPMMMSTORAGE');

  -- create index. Assumes workspaceID appears as first column in each index.
  Owb_Partition.CreateIndex('IDX_MMMACTIVECLASSNAME_VER','CMPMMMSTORAGE','ACTIVE,S2_1',false);
  Owb_Partition.CreateIndex('IDX_MMMCOMPUOID_VER','CMPMMMSTORAGE','COMPUOID',false);
  Owb_Partition.CreateIndex('IDX_MMMELEMENTID_VER','CMPMMMSTORAGE','I_1,SNAPSHOTID',true);
  Owb_Partition.CreateIndex('IDX_MMMLOADINGPARENT_VER','CMPMMMSTORAGE','LOADINGPARENT',false);
  Owb_Partition.CreateIndex('IDX_MMMSNAPID_VER','CMPMMMSTORAGE','SNAPSHOTID',false);
  Owb_Partition.CreateIndex('IDX_MMMUOID_VER','CMPMMMSTORAGE','S2_5',false);
end;
/

-- drop old table
whenever sqlerror continue;
prompt drop table WBO_CMPMMMSTORAGE
drop table WBO_CMPMMMSTORAGE;
whenever sqlerror exit failure rollback;

prompt Recreating table CMPSCOCFGCLASSES
begin
  dbms_output.enable(1000000);

  -- create table. Must contain workspaceID as a column.
  Owb_Partition.CreateTable('CMPSCOCFGCLASSES');

  -- create index. Assumes workspaceID appears as first column in each index.
  Owb_Partition.CreateIndex('IDX_SCOCFGCLASSNAMEELEMID','CMPSCOCFGCLASSES','S2_1,I_1',false);
  Owb_Partition.CreateIndex('IDX_SCOCFGDIMENSION','CMPSCOCFGCLASSES','R_18,S2_1',false);
  Owb_Partition.CreateIndex('IDX_SCOCFGELEMENTID','CMPSCOCFGCLASSES','I_1',true);
  Owb_Partition.CreateIndex('IDX_SCOCFGFCOCLASS','CMPSCOCFGCLASSES','R_11,S2_1',false);
  Owb_Partition.CreateIndex('IDX_SCOCFGNAME','CMPSCOCFGCLASSES','S2_3,S2_1',false);
  Owb_Partition.CreateIndex('IDX_SCOCFGOWNINGRELATION','CMPSCOCFGCLASSES','R_20,S2_1',false);
  Owb_Partition.CreateIndex('IDX_SCOCFGUOID','CMPSCOCFGCLASSES','S2_5',false);
end;
/

-- drop old table
whenever sqlerror continue;
prompt drop table WBO_CMPSCOCFGCLASSES
drop table WBO_CMPSCOCFGCLASSES;
whenever sqlerror exit failure rollback;

prompt Recreating table CMPSCOCFGSTORAGE
begin
  dbms_output.enable(1000000);

  -- create table. Must contain workspaceID as a column.
  Owb_Partition.CreateTable('CMPSCOCFGSTORAGE');

  -- create index. Assumes workspaceID appears as first column in each index.
  Owb_Partition.CreateIndex('IDX_SCOCFGACTIVECLASSNAME_VER','CMPSCOCFGSTORAGE','ACTIVE,S2_1',false);
  Owb_Partition.CreateIndex('IDX_SCOCFGCOMPUOID_VER','CMPSCOCFGSTORAGE','COMPUOID',false);
  Owb_Partition.CreateIndex('IDX_SCOCFGELEMENTID_VER','CMPSCOCFGSTORAGE','I_1,SNAPSHOTID',true);
  Owb_Partition.CreateIndex('IDX_SCOCFGFCOCLASS_VER','CMPSCOCFGSTORAGE','R_11,S2_1',false);
  Owb_Partition.CreateIndex('IDX_SCOCFGLOADINGPARENT_VER','CMPSCOCFGSTORAGE','LOADINGPARENT',false);
  Owb_Partition.CreateIndex('IDX_SCOCFGOWNINGFOLDER_VER','CMPSCOCFGSTORAGE','R_10',false);
  Owb_Partition.CreateIndex('IDX_SCOCFGSNAPID_VER','CMPSCOCFGSTORAGE','SNAPSHOTID',false);
  Owb_Partition.CreateIndex('IDX_SCOCFGUOID_VER','CMPSCOCFGSTORAGE','S2_5',false);
end;
/

-- drop old table
whenever sqlerror continue;
prompt drop table WBO_CMPSCOCFGSTORAGE
drop table WBO_CMPSCOCFGSTORAGE;
whenever sqlerror exit failure rollback;

prompt Recreating table CMPSCOCLASSES
begin
  dbms_output.enable(1000000);

  -- create table. Must contain workspaceID as a column.
  Owb_Partition.CreateTable('CMPSCOCLASSES');

  -- create index. Assumes workspaceID appears as first column in each index.
  Owb_Partition.CreateIndex('IDX_SCOCLASSNAMEELEMID','CMPSCOCLASSES','S2_1,I_1',false);
  Owb_Partition.CreateIndex('IDX_SCOELEMENTID','CMPSCOCLASSES','I_1',true);
  Owb_Partition.CreateIndex('IDX_SCOFCOCLASS','CMPSCOCLASSES','R_11,S2_1',false);
  Owb_Partition.CreateIndex('IDX_SCOITEMSET','CMPSCOCLASSES','R_17,S2_1',false);
  Owb_Partition.CreateIndex('IDX_SCONAME','CMPSCOCLASSES','S2_3,S2_1',false);
  Owb_Partition.CreateIndex('IDX_SCOTRANSLATABLE','CMPSCOCLASSES','R_15,S2_1',false);
  Owb_Partition.CreateIndex('IDX_SCOUOID','CMPSCOCLASSES','S2_5',false);
end;
/

-- drop old table
whenever sqlerror continue;
prompt drop table WBO_CMPSCOCLASSES
drop table WBO_CMPSCOCLASSES;
whenever sqlerror exit failure rollback;

prompt Recreating table CMPSCOMAPCLASSES
begin
  dbms_output.enable(1000000);

  -- create table. Must contain workspaceID as a column.
  Owb_Partition.CreateTable('CMPSCOMAPCLASSES');

  -- create index. Assumes workspaceID appears as first column in each index.
  Owb_Partition.CreateIndex('IDX_SCOMAPCLASSNAMEELEMID','CMPSCOMAPCLASSES','S2_1,I_1',false);
  Owb_Partition.CreateIndex('IDX_SCOMAPELEMENTID','CMPSCOMAPCLASSES','I_1',true);
  Owb_Partition.CreateIndex('IDX_SCOMAPFCOCLASS','CMPSCOMAPCLASSES','R_11,S2_1',false);
  Owb_Partition.CreateIndex('IDX_SCOMAPMAPATTR','CMPSCOMAPCLASSES','R_20,S2_1',false);
  Owb_Partition.CreateIndex('IDX_SCOMAPMAPOPREF','CMPSCOMAPCLASSES','R_17,S2_1',false);
  Owb_Partition.CreateIndex('IDX_SCOMAPNAME','CMPSCOMAPCLASSES','S2_3,S2_1',false);
  Owb_Partition.CreateIndex('IDX_SCOMAPUOID','CMPSCOMAPCLASSES','S2_5',false);
end;
/

-- drop old table
whenever sqlerror continue;
prompt drop table WBO_CMPSCOMAPCLASSES
drop table WBO_CMPSCOMAPCLASSES;
whenever sqlerror exit failure rollback;

prompt Recreating table CMPSCOMAPSTORAGE
begin
  dbms_output.enable(1000000);

  -- create table. Must contain workspaceID as a column.
  Owb_Partition.CreateTable('CMPSCOMAPSTORAGE');

  -- create index. Assumes workspaceID appears as first column in each index.
  Owb_Partition.CreateIndex('IDX_SCOMAPACTIVECLASSNAME_VER','CMPSCOMAPSTORAGE','ACTIVE,S2_1',false);
  Owb_Partition.CreateIndex('IDX_SCOMAPCOMPUOID_VER','CMPSCOMAPSTORAGE','COMPUOID',false);
  Owb_Partition.CreateIndex('IDX_SCOMAPELEMENTID_VER','CMPSCOMAPSTORAGE','I_1,SNAPSHOTID',true);
  Owb_Partition.CreateIndex('IDX_SCOMAPFCOCLASS_VER','CMPSCOMAPSTORAGE','R_11,S2_1',false);
  Owb_Partition.CreateIndex('IDX_SCOMAPLOADINGPARENT_VER','CMPSCOMAPSTORAGE','LOADINGPARENT',false);
  Owb_Partition.CreateIndex('IDX_SCOMAPOWNINGFOLDER_VER','CMPSCOMAPSTORAGE','R_10',false);
  Owb_Partition.CreateIndex('IDX_SCOMAPSNAPID_VER','CMPSCOMAPSTORAGE','SNAPSHOTID',false);
  Owb_Partition.CreateIndex('IDX_SCOMAPUOID_VER','CMPSCOMAPSTORAGE','S2_5',false);
end;
/

-- drop old table
whenever sqlerror continue;
prompt drop table WBO_CMPSCOMAPSTORAGE
drop table WBO_CMPSCOMAPSTORAGE;
whenever sqlerror exit failure rollback;

prompt Recreating table CMPSCOPRPCLASSES
begin
  dbms_output.enable(1000000);

  -- create table. Must contain workspaceID as a column.
  Owb_Partition.CreateTable('CMPSCOPRPCLASSES');

  -- create index. Assumes workspaceID appears as first column in each index.
  Owb_Partition.CreateIndex('IDX_SCOPRPCLASSNAMEELEMID','CMPSCOPRPCLASSES','S2_1,I_1',false);
  Owb_Partition.CreateIndex('IDX_SCOPRPELEMENTID','CMPSCOPRPCLASSES','I_1',true);
  Owb_Partition.CreateIndex('IDX_SCOPRPFCOCLASS','CMPSCOPRPCLASSES','R_11,S2_1',false);
  Owb_Partition.CreateIndex('IDX_SCOPRPNAME','CMPSCOPRPCLASSES','S2_3,S2_1',false);
  Owb_Partition.CreateIndex('IDX_SCOPRPOWNERCLASSBUSNAME','CMPSCOPRPCLASSES','R_15,S2_1,S4_1',false);
  Owb_Partition.CreateIndex('IDX_SCOPRPUOID','CMPSCOPRPCLASSES','S2_5',false);
end;
/

-- drop old table
whenever sqlerror continue;
prompt drop table WBO_CMPSCOPRPCLASSES
drop table WBO_CMPSCOPRPCLASSES;
whenever sqlerror exit failure rollback;

prompt Recreating table CMPSCOPRPSTORAGE
begin
  dbms_output.enable(1000000);

  -- create table. Must contain workspaceID as a column.
  Owb_Partition.CreateTable('CMPSCOPRPSTORAGE');

  -- create index. Assumes workspaceID appears as first column in each index.
  Owb_Partition.CreateIndex('IDX_SCOPRPACTIVECLASSNAME_VER','CMPSCOPRPSTORAGE','ACTIVE,S2_1',false);
  Owb_Partition.CreateIndex('IDX_SCOPRPCOMPUOID_VER','CMPSCOPRPSTORAGE','COMPUOID',false);
  Owb_Partition.CreateIndex('IDX_SCOPRPELEMENTID_VER','CMPSCOPRPSTORAGE','I_1,SNAPSHOTID',true);
  Owb_Partition.CreateIndex('IDX_SCOPRPFCOCLASS_VER','CMPSCOPRPSTORAGE','R_11,S2_1',false);
  Owb_Partition.CreateIndex('IDX_SCOPRPLOADINGPARENT_VER','CMPSCOPRPSTORAGE','LOADINGPARENT',false);
  Owb_Partition.CreateIndex('IDX_SCOPRPOWNINGFOLDER_VER','CMPSCOPRPSTORAGE','R_10',false);
  Owb_Partition.CreateIndex('IDX_SCOPRPSNAPID_VER','CMPSCOPRPSTORAGE','SNAPSHOTID',false);
  Owb_Partition.CreateIndex('IDX_SCOPRPUOID_VER','CMPSCOPRPSTORAGE','S2_5',false);
end;
/

-- drop old table
whenever sqlerror continue;
prompt drop table WBO_CMPSCOPRPSTORAGE
drop table WBO_CMPSCOPRPSTORAGE;
whenever sqlerror exit failure rollback;

prompt Recreating table CMPSCOSTORAGE
begin
  dbms_output.enable(1000000);

  -- create table. Must contain workspaceID as a column.
  Owb_Partition.CreateTable('CMPSCOSTORAGE');

  -- create index. Assumes workspaceID appears as first column in each index.
  Owb_Partition.CreateIndex('IDX_SCOACTIVECLASSNAME_VER','CMPSCOSTORAGE','ACTIVE,S2_1',false);
  Owb_Partition.CreateIndex('IDX_SCOCOMPUOID_VER','CMPSCOSTORAGE','COMPUOID',false);
  Owb_Partition.CreateIndex('IDX_SCOELEMENTID_VER','CMPSCOSTORAGE','I_1,SNAPSHOTID',true);
  Owb_Partition.CreateIndex('IDX_SCOFCOCLASS_VER','CMPSCOSTORAGE','R_11,S2_1',false);
  Owb_Partition.CreateIndex('IDX_SCOLOADINGPARENT_VER','CMPSCOSTORAGE','LOADINGPARENT',false);
  Owb_Partition.CreateIndex('IDX_SCOOWNINGFOLDER_VER','CMPSCOSTORAGE','R_10',false);
  Owb_Partition.CreateIndex('IDX_SCOSNAPID_VER','CMPSCOSTORAGE','SNAPSHOTID',false);
  Owb_Partition.CreateIndex('IDX_SCOUOID_VER','CMPSCOSTORAGE','S2_5',false);
end;
/

-- drop old table
whenever sqlerror continue;
prompt drop table WBO_CMPSCOSTORAGE
drop table WBO_CMPSCOSTORAGE;
whenever sqlerror exit failure rollback;

prompt Recreating table CMPSYSCLASSES
begin
  dbms_output.enable(1000000);

  -- create table. Must contain workspaceID as a column.
  Owb_Partition.CreateTable('CMPSYSCLASSES');

  -- create index. Assumes workspaceID appears as first column in each index.
  Owb_Partition.CreateIndex('IDX_SYSCLASSNAMEELEMID','CMPSYSCLASSES','S2_1,I_1',false);
  Owb_Partition.CreateIndex('IDX_SYSELEMENTID','CMPSYSCLASSES','I_1',true);
  Owb_Partition.CreateIndex('IDX_SYSNAME','CMPSYSCLASSES','S2_3,S2_1',false);
  Owb_Partition.CreateIndex('IDX_SYSUOID','CMPSYSCLASSES','S2_5',false);
end;
/

-- drop old table
whenever sqlerror continue;
prompt drop table WBO_CMPSYSCLASSES
drop table WBO_CMPSYSCLASSES;
whenever sqlerror exit failure rollback;

prompt Recreating table CMPSYSSTORAGE
begin
  dbms_output.enable(1000000);

  -- create table. Must contain workspaceID as a column.
  Owb_Partition.CreateTable('CMPSYSSTORAGE');

  -- create index. Assumes workspaceID appears as first column in each index.
  Owb_Partition.CreateIndex('IDX_SYSACTIVECLASSNAME_VER','CMPSYSSTORAGE','ACTIVE,S2_1',false);
  Owb_Partition.CreateIndex('IDX_SYSCOMPUOID_VER','CMPSYSSTORAGE','COMPUOID',false);
  Owb_Partition.CreateIndex('IDX_SYSELEMENTID_VER','CMPSYSSTORAGE','I_1,SNAPSHOTID',true);
  Owb_Partition.CreateIndex('IDX_SYSFCOCLASS_VER','CMPSYSSTORAGE','R_11,S2_1',false);
  Owb_Partition.CreateIndex('IDX_SYSLOADINGPARENT_VER','CMPSYSSTORAGE','LOADINGPARENT',false);
  Owb_Partition.CreateIndex('IDX_SYSOWNINGFOLDER_VER','CMPSYSSTORAGE','R_10',false);
  Owb_Partition.CreateIndex('IDX_SYSSNAPID_VER','CMPSYSSTORAGE','SNAPSHOTID',false);
  Owb_Partition.CreateIndex('IDX_SYSUOID_VER','CMPSYSSTORAGE','S2_5',false);
end;
/

-- drop old table
whenever sqlerror continue;
prompt drop table WBO_CMPSYSSTORAGE
drop table WBO_CMPSYSSTORAGE;
whenever sqlerror exit failure rollback;

prompt Recreating table FCOFOLDERNAMESPACE
begin
  dbms_output.enable(1000000);

  -- create table. Must contain workspaceID as a column.
  Owb_Partition.CreateTable('FCOFOLDERNAMESPACE');

  -- create index. Assumes workspaceID appears as first column in each index.
  Owb_Partition.CreateIndex('FCOFOLDERNS_PK','FCOFOLDERNAMESPACE','PARENTID,DOMAIN,NAME,ISPHYS,LANGUAGEISOID',true);
  Owb_Partition.CreateIndex('OWBCMPNS_AGGREGATEPARENT_IDX','FCOFOLDERNAMESPACE','AGGREGATEPARENT',false);
  Owb_Partition.CreateIndex('OWBCMPNS_ELEMENTID_IDX','FCOFOLDERNAMESPACE','ELEMENTID',false);
  Owb_Partition.CreateIndex('OWBCMPNS_PARENTID_IDX','FCOFOLDERNAMESPACE','PARENTID',false);
end;
/

-- drop old table
whenever sqlerror continue;
prompt drop table WBO_FCOFOLDERNAMESPACE
drop table WBO_FCOFOLDERNAMESPACE;
whenever sqlerror exit failure rollback;

-- As OWBSYS Schema
set serveroutput on

prompt Recompiling views
-- recompile views
begin
  dbms_output.enable(1000000);
  Owb_Partition.CompileObject('VIEW');
  Owb_Partition.CompileObject('VIEW');
end;
/

set serveroutput on
prompt Recompiling package bodies
-- recompile packages
begin
  dbms_output.enable(1000000);
  Owb_Partition.CompileObject('PACKAGE BODY');
  Owb_Partition.CompileObject('PACKAGE BODY');
end;
/

-- clean up
prompt Drop package OWB_PARTITION in &u_app schema
drop package OWB_PARTITION;


prompt OWBSYS WorkspaceID partitioning script completed
prompt

commit;
exit;
