/* Copyright (c) 2006, Oracle. All rights reserved.  */
package javax.ide.extension;


import java.util.Iterator;

import javax.ide.extension.ExtensionHook;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Text;

/**
 * An extension hook that builds a DOM tree. The subclass must implement the
 * {@link #getRootElementName()} method and return an {@link ElementName} for
 * the root element. This should match the {@link ElementName} this hook is 
 * registered for.<p>
 * 
 * After extension hooks have been processed, call the {@link #getRootElement()}
 * method to get a DOM {@link Element}.
 *
 *  @since   2.0
 */
public abstract class DOMHook extends ExtensionHook
{
  private static final String KEY_PARENT_ELEMENT = "parentElement";

  private final ElementVisitor _genericVisitor = new GenericVisitor();
  private final ElementVisitorFactory _genericVisitorFactory = new ElementVisitorFactory()
  {
    public ElementVisitor getVisitor(ElementName name)
    {
      return _genericVisitor;
    }
  };
  
  private final DocumentBuilder _documentBuilder;
  private final Document _document;
  
  public DOMHook()
  {
    try
    {
      DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
      _documentBuilder = dbf.newDocumentBuilder();
      
      _document = _documentBuilder.newDocument();
      final Element rootElement = createElement( _document, getRootElementName() );
      _document.appendChild( rootElement );
    }
    catch ( ParserConfigurationException e )
    {
      throw new IllegalStateException( "Parser configuration error", e );
    }
  }
  
  /**
   * Returns the root element of the DOM tree built using this hook.
   * 
   * @return the root element of the DOM tree built using this hook.
   */
  public Element getRootElement()
  {
    return _document.getDocumentElement();
  }

  @Override public void start(ElementStartContext context)
  {
    context.registerVisitorFactory( _genericVisitorFactory );
    context.getScopeData().put( KEY_PARENT_ELEMENT, _document.getDocumentElement() );
  }

  @Override public void end(ElementEndContext context)
  {
  }

  private final class GenericVisitor extends ElementVisitor
  {
    @Override public void start(ElementStartContext context)
    {
      final Element newElement = createElement( _document, context.getElementName() );
      
      for ( Iterator i = context.getAttributeNames().iterator(); i.hasNext(); )
      {
        final String attributeName = (String) i.next();
        final String attributeValue = context.getAttributeValue( attributeName );        
        newElement.setAttribute( attributeName, attributeValue );
      }    
      final Element parentElement = getParentElement( context );
      parentElement.appendChild( newElement );
      
      context.getScopeData().put( KEY_PARENT_ELEMENT, newElement );
    }

    @Override public void end(ElementEndContext context)
    {
      final String text = context.getText();
      if ( text != null )
      {
        final Text textNode = _document.createTextNode( text );
        getParentElement( context ).appendChild( textNode );
      }
    }
  }
  
  private static Element getParentElement( ElementContext context )
  {
    return (Element) context.getScopeData().get( KEY_PARENT_ELEMENT );
  }

  private static Element createElement( Document doc, ElementName elementName )
  {
    return doc.createElementNS( elementName.getNamespaceURI(), 
      elementName.getLocalName() );
  }

  /**
   * Gets the root element name to use. This should match the <tt>ElementName</tt>
   * this extension hook is registered for.
   *
   * @return the root element name to use.
   */
  protected abstract ElementName getRootElementName();
  
}

