/*
 * Copyright 2005 by Oracle USA
 * 500 Oracle Parkway, Redwood Shores, California, 94065, U.S.A.
 * All rights reserved.
 */
package javax.ide.extension;

import java.util.Locale;
import java.util.MissingResourceException;
import java.util.ResourceBundle;
import java.util.logging.Level;

import javax.ide.extension.spi.ExtensionLogRecord;
import javax.ide.extension.spi.ExtensionVisitor;

/**
 * <tt>ElementVisitor</tt> is responsible for visiting elements in an XML file
 * processed by an XML parser used by a JSR-198 compatible IDE.
 *
 * Implementations of this class usually provide an implementation for the
 * start() and end() methods.
 */
public abstract class ElementVisitor 
{
  /**
   * A key that provides information about the position in the manifest file
   * currently being processed. In the scope data map, this key will resolve
   * to an instance of org.xml.sax.Locator, if the parser supports it.<p>
   * You should always check whether the locator is null, since it is not
   * required that every IDE provide a Locator in the parsing context.
   */
  public static final String KEY_LOCATOR = "xml.locator";

  /**
   * Log a message. This is the recommended way to log errors while processing
   * elements in the manifest file. It will ensure that positional information
   * (if available) is passed to the logger as the first parameter of the 
   * log message.
   * 
   * @param context the current context.
   * @param level the level to log at.
   * @param msg the message to log.
   */
  protected final void log( ElementContext context, Level level, String msg )
  {
    context.getLogger().log( new ExtensionLogRecord( context, level, msg ) );
  }

  /**
   * Log a message. This is the recommended way to log errors while processing
   * elements in the manifest file. It will ensure that positional information
   * (if available) is passed to the logger as the first parameter of the 
   * log message.
   * 
   * @param context the current context.
   * @param level the level to log at.
   * @param msg the message to log.
   * @param parameters one or more parameters for the message.
   */
  protected final void log( ElementContext context, Level level, String msg, Object... parameters )
  {
    ExtensionLogRecord record = new ExtensionLogRecord(context, level, msg);
    record.setParameters(parameters);
    context.getLogger().log(record);
  }

  /**
   * Visit the start tag of an xml element.<p>
   * 
   * This implementation does nothing.
   * 
   * @param context information about the xml start tag.
   */
  public void start( ElementStartContext context )
  {
    
  }
  
  /**
   * Visit the end tag of an xml element.<p>
   * 
   * This implementation does nothing.
   * 
   * @param context information about the xml end tag.
   */
  public void end( ElementEndContext context )
  {
    
  }
  
  /**
   * Gets the class loader to use in the specified context.
   * 
   * @param context the context.
   * @return the class loader to use in this context. Never returns null.
   * 
   * @since 2.0
   */
  public static final ClassLoader getClassLoader( ElementContext context )
  {
    ClassLoader classLoader = (ClassLoader) context.getScopeData().get(
      ExtensionVisitor.KEY_CLASSLOADER
    );
    if ( classLoader == null )
    {
      classLoader = Thread.currentThread().getContextClassLoader();
    }
    return classLoader;
  }  
  
  
  /**
   * Gets the resource bundle (if any) to be used for translatable resources
   * in the specified context.
   * 
   * @param context a processing context.
   * @return the resource bundle instance to use for this context, or null if
   *    none is available.
   * @since 2.0
   */
  public static final ResourceBundle getResourceBundle( ElementContext context )
  {
    final ClassLoader classLoader = getClassLoader( context );
    String resClass = (String) context.getScopeData().get(
      ExtensionHook.KEY_RSBUNDLE_CLASS );
    if ( resClass == null )
    {
      return null;
    }
    try
    {
        return s_resBundleProvider.getResourceBundle( resClass, Locale.getDefault(), classLoader);  
    }
    catch ( MissingResourceException mre )
    {
      return null;
    }
  }
  
  protected String getAttributeHelper(ElementStartContext context, 
                                      String attributeName,
                                      boolean trim,
                                      boolean logErrorIfMissing)
  {
    String value = context.getAttributeValue(attributeName);
    
    if (trim && (value != null))
    {
      value = value.trim();
      
      if (value.isEmpty())
      {
        value = null;
      }
    }
    
    if (logErrorIfMissing && (value == null))
    {
      log(context, Level.SEVERE, "<" + context.getElementName().getLocalName() + "> element is missing required attribute: " + attributeName);  //NOTRANS
    }
    
    return value;
  }
  
  protected String getTextHelper(ElementEndContext context, 
                                 boolean trim,
                                 boolean logErrorIfMissing)
  {
    String value = context.getText();
    
    if (trim && (value != null))
    {
      value = value.trim();
      
      if (value.isEmpty())
      {
        value = null;
      }
    }
    
    if (logErrorIfMissing && (value == null))
    {
      log(context, Level.SEVERE, "<" + context.getElementName().getLocalName() + "> element is missing required text content");  //NOTRANS
    }
    
    return value;
  }  

  /**
   * A class that creates the ResourceBundle instance.
   * 
   * @author svassile
   */
  public interface ResourceBundleProvider 
  {
    ResourceBundle getResourceBundle( String baseName, Locale locale, ClassLoader loader );
  }
    
  private static ResourceBundleProvider s_resBundleProvider = new ResourceBundleProvider() 
  {
    public ResourceBundle getResourceBundle( String baseName, Locale locale, ClassLoader loader ) 
    {
      return ResourceBundle.getBundle( baseName, locale, loader );
    }
  };
  
  public static void setResourceBundleProvider(ResourceBundleProvider resBundleProvider) 
  {
    if (resBundleProvider != null)
    {
      s_resBundleProvider = resBundleProvider;    
    }
  }
  
  public static ResourceBundleProvider getResourceBundleProvider() 
  {
    return s_resBundleProvider;    
  }
  
  protected ClassLoader getMetaClassLoader(ElementContext context)
  {
    ClassLoader classLoader =
      (ClassLoader) context.getScopeData().get(ExtensionVisitor.KEY_CLASSLOADER);
    
    if (classLoader == null)
    {
      Extension extension = (Extension) context.getScopeData().get(ExtensionHook.KEY_EXTENSION);
      if (extension != null)
      {
        ExtensionRegistry registry = ExtensionRegistry.getExtensionRegistry();
        classLoader = registry.getClassLoader(extension.getID());
      }
    }
    return classLoader;
  }
}
