/*
 * Copyright 2005 by Oracle USA
 * 500 Oracle Parkway, Redwood Shores, California, 94065, U.S.A.
 * All rights reserved.
 */
package javax.ide.extension.spi;

import java.net.URI;
import java.net.URL;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.MissingResourceException;
import java.util.ResourceBundle;
import java.util.logging.Level;
import java.util.logging.LogRecord;
import java.util.logging.Logger;

import javax.ide.extension.ElementContext;
import javax.ide.extension.ElementEndContext;
import javax.ide.extension.ElementName;
import javax.ide.extension.ElementStartContext;
import javax.ide.extension.ElementVisitor;
import javax.ide.extension.ElementVisitorFactory;
import javax.ide.extension.ElementVisitorFactory2;
import javax.ide.extension.Extension;
import javax.ide.extension.ExtensionHook;
import javax.ide.extension.UnrecognizedElementException;
import javax.ide.net.URIFactory;

import javax.xml.stream.events.Attribute;


/**
 * A default XML context implementation. This provides access to methods that
 * change the context and should only be used by ExtensibleSAXHandler.
 */
public class DefaultElementContext
  implements ElementContext, ElementStartContext, ElementEndContext
{
  private Attributes _attributes;
  private ElementName _name;
  private final Stack<ElementName> _elementPathStack = new Stack<ElementName>();
  private final ScopedMap _contextMap = new ScopedMap();
  private final Stack<StringBuffer> _elementText = new Stack<StringBuffer>();

  private Logger _logger;
  
  private final Stack<Handlers> _handlers = new Stack<Handlers>();
 
  private final MacroExpander _macroExpander = new MacroExpander() 
  {
    protected String getMacroValue(String macroText)
    {
      return DefaultElementContext.this.getMacroValue( macroText );
    }
  };
 
  public DefaultElementContext()
  {
    _handlers.push( new Handlers() );
  }
  
  /**
   * Resets the state of the context.
   * 
   * @since 2.0
   */
  void reset()
  {
    _contextMap.reset();
    _elementPathStack.clear();
    _elementText.clear();
    while ( _handlers.size() > 1 )
    {
      _handlers.pop();
    }
    _name = null;
    _attributes = null;
  }

  /**postEndElement
   * Get the local name of the current element.
   *
   * @return the local name of the current element.
   */
  public ElementName getElementName()
  {
    return _name;
  }
  
  
  /**
   * Gets the raw text contained in this element (without any macro processing).
   * 
   * @return the raw text contained in this element.
   * @since 2.0
   */
  final String getRawText()
  {
    StringBuffer buffer = _elementText.peek();

    if ( buffer == null )
    {
      return null;
    }
    
    return buffer.toString();
  }
  
  /**
   * Gets processed text (with macro processing applied).
   * 
   * @param rawText raw text, possibly containing macros.
   * @return processed text.
   * 
   * @since 2.0
   */
  final String getProcessedText( String rawText )
  {
    // Don't just make processText() public, this would cause a minor
    // source incompatibility with 1.0
    return processText( rawText );
  }
  
  /**
   * Sets the raw text of this element.
   * 
   * @param rawText the raw text of this element.
   */
  final void setText( String rawText )
  {
    // cheezyness to allow extension visitor to convert i18n string into
    // a real string lazily.
  
    if ( rawText == null )
    {
      // Throw immediately, avoid getting the stack in an inconsistent state.
      throw new NullPointerException();
    }
    _elementText.pop();
    _elementText.push( new StringBuffer( rawText ) );
  }

  /**
   * Get the text contained in this element. It's only valid to call this from
   * EDDElementHandler.handleEndElement.
   */
  public String getText()
  {
    StringBuffer buffer = _elementText.peek();

    if ( buffer == null )
    {
      return null;
    }

    return processText( buffer.toString() );
  }

  /**
   * Get the value of an attribute for this element. It's only valid to call
   * this from EDDElementHandler.handleStartElement.
   *
   * @param attributeName the local name of an XML attribute, must not be
   *    null.
   * @return the value of the specified attribute, or null if not present.
   */
  public String getAttributeValue( String attributeName )
  {
    if ( _attributes == null )
    {
      throw new IllegalStateException(
        "Cannot call from handleEndElement"
      );
    }

    if ( attributeName == null )
    {
      throw new IllegalArgumentException( "attributeName must not be null" );
    }

    return processText( _attributes.getValue( attributeName ) );
  }

  public Collection<String> getAttributeNames()
  {
    if (_attributes == null)
    {
      throw new IllegalStateException();
    }
    ArrayList<String> names = new ArrayList<String>();
    for ( Attribute attribute: _attributes )
    {
      names.add( attribute.getName().getLocalPart() );
    }

    return Collections.unmodifiableCollection( names );
  }
  
  final Attributes getRawAttributes()
  {
    return _attributes;
  }

  public Map getScopeData()
  {
    return _contextMap;
  }

  /**
   * Register a handler that will be used only for immediate children of the
   * current element.
   *
   * @param name the element name.
   * @param visitor the {@link ElementVisitor}.
   */
  public void registerChildVisitor( ElementName name, ElementVisitor visitor )
  {
    _handlers.peek().addChildVisitor( name, visitor );
  }


  public void registerVisitorFactory( ElementVisitorFactory factory )
  {
    _handlers.peek().addVisitorFactory( factory );
  }

  protected ElementVisitor getVisitorForEndElement()
  {
    return _handlers.peek().getStartElementVisitor();
  }

  protected ElementVisitor getVisitorForStartElement( ElementName name )
  {
    ElementVisitor visitor = null;
    
    try
    {
      visitor = getVisitorForStartElementImpl( name );
    }
    catch ( UnrecognizedElementException uee )
    {
      getLogger().log( new ExtensionLogRecord( this, Level.SEVERE, 
        uee.getMessage() ));
      visitor =  NULL_VISITOR;    // Prevent the factory from getting any child elements.
    }   
    
    Handlers handlers = _handlers.peek();
    handlers.setStartElementVisitor( visitor );    
      
    return visitor;
  }

  /**
   * Get a scoped handler for the current element (if any)
   */
  private ElementVisitor getVisitorForStartElementImpl( ElementName name )
    throws UnrecognizedElementException
  {
    ElementName qualified = name;
    ElementName unqualified = new ElementName( null, name.getLocalName() );
    ElementVisitor handler = null;
    
    

    // Look for child-only handlers first.
    final Handlers handlers = _handlers.peek();
    handler = handlers.getChildVisitor( qualified );
    if ( handler == null )
    {
      handler = handlers.getChildVisitor( unqualified );
    }

    if ( handler != null )
    {
      return handler;
    }

    // Visitor factories are scoped. We have to iterate the stack looking for
    // a factory that provides a visitor for the element.

    boolean firstLevel = true;
    for ( Handlers h : _handlers )
    {
      for ( ElementVisitorFactory factory : h.getVisitorFactories() )
      {
        if ( factory instanceof ElementVisitorFactory2 )
        {
          ElementVisitorFactory2 factory2 = (ElementVisitorFactory2) factory;
          
          if ( (!firstLevel && factory2.isDescending()) || firstLevel )
          {
            handler = factory2.getVisitor( this, qualified );
            if ( handler != null )
            {
              return handler;
            }
            handler = factory2.getVisitor( this, unqualified ); 
            if ( handler != null ) return handler;
          }
        }
        else
        {
          handler = factory.getVisitor( qualified );
          if ( handler != null )
          {
            return handler;
          }
          handler = factory.getVisitor( unqualified );
        }
      }
      
      firstLevel = false;
    }

    return handler;
  }

  private void setElement( String uri, String name )
  {
    _name = new ElementName( uri, name );
  }

  protected void beginElement( String uri, String name, Attributes attributes )
  {
    setElement( uri, name );
    _attributes = attributes;

    _contextMap.enterScope();
    
    _handlers.push( new Handlers() );
  }

  protected void appendCharacters( char[] characters, int start, int length )
  {
    if ( _elementText.isEmpty() )
    {
      throw new IllegalStateException();
    }

    StringBuffer buffer = _elementText.peek();
    if ( buffer == null )
    {
      buffer = new StringBuffer();
      _elementText.replace( buffer );
    }

    buffer.append( characters, start, length );
  }

  protected void postEndElement()
  {
    _contextMap.exitScope();
    _elementText.pop();
    _name = null;
    _attributes = null;
  }

  protected void endElement( String uri, String name )
  {
    setElement( uri, name );
    _handlers.pop();    
    _attributes = null;
    _elementPathStack.pop();
  }

  /**
   * Push the current element onto the element stack.
   */
  protected void postBeginElement()
  {
    _elementPathStack.push( _name );
    _elementText.push( null );    // StringBuffer is created lazily.
  }

  public Logger getLogger()
  {
    if ( _logger == null )
    {
      _logger = new NullLogger();
    }
    return _logger;
  }

  public void setMessageReporter( Logger logger )
  {
    _logger = logger;
  }

  /**
   * Get the extension currently being processed.
   *
   * @return the exetnsion currently being processed.
   */
  public Extension getExtension()
  {
    return (Extension) getScopeData().get( ExtensionHook.KEY_EXTENSION );
  }


  /**
   * Get the URI of the source of the extension. For extensions packaged in
   * JAR files, this is the URI of the JAR file.
   *
   * @return the URI of the exension source.
   */
  public URI getExtensionSourceURI()
  {
    ExtensionSource source = (ExtensionSource) getScopeData().get(
      ExtensionVisitor.KEY_EXTENSION_SOURCE
    );
    return source.getURI();
  }

  /**
   * Process the text of either an attribute value or a text node.
   * The default implementation calls substituteMacros(), then
   * handles the prefixes res: and url:.
   *
   * @param text the text to process.
   * @return the processed text.
   */
  protected String processText( String text )
  {
    if ( text == null ) return null;

    String newText = substituteMacros( text );

    if ( newText.startsWith( "res:" )  )
    {
      return resolveResource( newText );
    }
    else if ( newText.startsWith( "uri:" ) )
    {
      return resolveUri( newText );
    }

    return newText;
  }

  private String resolveUri( String text )
  {
    if ( text.length() > 4 )
    {
      String uriString = text.substring( 4 );
      Extension ext = (Extension) getScopeData().get(
        ExtensionHook.KEY_EXTENSION );
      ExtensionSource source = (ExtensionSource) getScopeData().get(
        ExtensionVisitor.KEY_EXTENSION_SOURCE );

      URI resolved = source.resolvePath( ext, uriString );
      if ( resolved == null )
      {
        getLogger().log( new ExtensionLogRecord( this, Level.WARNING,
          "Unresolved uri: '" + uriString + "'." ) );
        return text;
      }
      return "uri:" + resolved.toString();
    }
    getLogger().log( new ExtensionLogRecord( this, Level.SEVERE,
      "No uri specified."  ) );
    return text;
  }

  private String resolveResource(String text)
  {
    if ( text.length() > 4 )
    {
      String resString = text.substring( 4 );
      if ( resString.charAt( 0 ) == '/' && resString.length() > 1 )
      {
        // Load a resource relative to the context classloader.
        final ClassLoader cl = ElementVisitor.getClassLoader( this );
        final URL resUrl = cl.getResource( resString.substring( 1 ) );
        if ( resUrl != null )
        {
          URI uri = URIFactory.newURI( resUrl );
          return "uri:" + uri.toString();
        }
      }
      else
      {
        // Load resource relative to the resource bundle class.
        ResourceBundle bundle = ElementVisitor.getResourceBundle( this );
        if ( bundle != null )
        {
          final URL resUrl = bundle.getClass().getResource( resString );
          if ( resUrl != null )
          {
            URI uri = URIFactory.newURI( resUrl );
            return "uri:" + uri.toString();
          }
        }
      }
      getLogger().log( new ExtensionLogRecord( this, Level.SEVERE,
        "Resource not found: '" + resString + "'."));
      return text;
    }
    else
    {
      getLogger().log( new ExtensionLogRecord( this, Level.SEVERE,
        "No resource specified" ) );
      return text;
    }
  }

  /**
   * Perform macro substitutions on the specified string.
   *
   * @param s a string
   */
  protected final String substituteMacros( String s )
  {
    return _macroExpander.substituteMacros( s );
  }


  /**
   * Get the value of the specified macro.<p>
   *
   * This implementation looks up resource keys from the scope resource bundle.
   *
   * @param macroName the name of the macro
   * @return the value of the macro. If you return null, the macro is undefined
   *    and the original unsubstituted macro string will be used.
   */
  protected String getMacroValue( String macroName )
  {
    ResourceBundle bundle = ElementVisitor.getResourceBundle( this );
    if ( bundle != null )
    {
      try
      {
        return bundle.getString( macroName );
      }
      catch ( MissingResourceException mre )
      {
        getLogger().log( new ExtensionLogRecord( this, Level.WARNING,
          mre.getLocalizedMessage().replaceAll("'", "''") ) );
        return null;
      }
    }
    
    String resClass = (String) getScopeData().get(
      ExtensionHook.KEY_RSBUNDLE_CLASS );    
    if ( resClass == null )
    {
      getLogger().log( new ExtensionLogRecord( this, Level.SEVERE, 
        "Unrecognized macro '" + macroName + "' and no rsbundle-class attribute specified." ) );
    }
    else
    {
      getLogger().log( new ExtensionLogRecord( this, Level.SEVERE, 
        "Unable to find resource bundle class: " + resClass ));
    }
    return null;
  }

  public ClassLoader getClassLoader() {
    ClassLoader classLoader = (ClassLoader) this.getScopeData().get(
      ExtensionVisitor.KEY_CLASSLOADER
    );
    
    if ( classLoader == null )
    { 
      classLoader = Thread.currentThread().getContextClassLoader();
    }
    return classLoader;
  }


  /**
   * Data structure containing information about handlers for the current
   * context.
   */
  private class Handlers
  {
    private Map<ElementName,ElementVisitor> childVisitors;
    private List<ElementVisitorFactory> visitorFactories;
    private ElementVisitor startElementVisitor;
    
    public ElementVisitor getStartElementVisitor()
    {
      return startElementVisitor;
    }
    
    public void setStartElementVisitor( ElementVisitor visitor )
    {
      startElementVisitor = visitor;
    }
    
    public void addChildVisitor( ElementName name, ElementVisitor visitor )
    {
      if ( childVisitors == null ) childVisitors = new HashMap<ElementName,ElementVisitor>();
      childVisitors.put( name, visitor );
    }
    
    public ElementVisitor getChildVisitor( ElementName name )
    {
      if ( childVisitors == null ) return null;
      return childVisitors.get( name );
    }
    
    public void addVisitorFactory( ElementVisitorFactory factory )
    {
      if ( visitorFactories == null ) visitorFactories = new ArrayList<ElementVisitorFactory>();
      visitorFactories.add( factory );
    }
    
    public List<ElementVisitorFactory> getVisitorFactories()
    {
      return visitorFactories == null ? Collections.EMPTY_LIST : visitorFactories;
    }
    
    
  }


  private class NullLogger extends Logger
  {
    public NullLogger( )
    {
      super( null, null );
    }

    public void log( LogRecord logRecord )
    {
      // NO-OP
    }
  }
  
  public Iterable<ElementName> getElementPath() {
    return _elementPathStack;  
  }

  public static final ElementVisitor NULL_VISITOR = new NullVisitor();

  private static final class NullVisitor extends ElementVisitor
  {
    
  }
  
  public static interface Attributes extends Iterable<Attribute> {
    String getValue(String key);
  }
  
  public static class AttributesImpl implements Attributes {
    private ArrayList<Attribute> m_list;
    public AttributesImpl() {
      m_list = new ArrayList<Attribute>(1);
    }
    public AttributesImpl(Attributes attributes) {
      setAttributes(attributes);  
    }

    @Override
    public String getValue(String qName)
    {
      for (Attribute attribute: m_list)
        if (qName.equals(attribute.getName().getLocalPart()))
          return attribute.getValue();
      return null;
    }
    
    public void setAttributes(Attributes attributes) {
      m_list.clear();
      for (Attribute attribute: attributes)
      {
        m_list.add(attribute);
      }
    }
    
    public void addAttribute (Attribute attribute) {
      m_list.add(attribute);  
    }

    @Override
    public Iterator<Attribute> iterator() {
      return m_list.iterator();
    }
  }
}
