/*
 * Copyright 2005 by Oracle USA
 * 500 Oracle Parkway, Redwood Shores, California, 94065, U.S.A.
 * All rights reserved.
 */
package javax.ide.view;

import java.beans.PropertyChangeListener;
import javax.ide.util.IconDescription;
import javax.ide.view.Viewable;
import javax.swing.event.SwingPropertyChangeSupport;


/**
 * <code>DefaultViewable</code>s are objects that map directly to visible
 * GUI elements that users can select such as a menu items, nodes in
 * the project explorer, etc..
 */
public class DefaultViewable implements Viewable
{

  private boolean _visible = true;
  private SwingPropertyChangeSupport _propertyChangeSupport;

  //----------------------------------------------------------------------------
  // Public methods.
  //----------------------------------------------------------------------------

  /**
   *  Get a short descriptive label that can be shown to the user.
   *  @return A short descriptive label.
   */
  public String getLabel()
  {
    return "nolabel";
  }

  /*-
   * Viewable method.
   */
  public String getToolTip()
  {
    return getLabel();
  }

  /*-
   * Viewable method.
   */
  public IconDescription getIcon()
  {
    return null;
  }

  /*-
   * Viewable interface method. Returns <code>true</code> by default.
   */
  public boolean isVisible()
  {
    return _visible;
  }
  
  /*-
   * Viewable interface method.
   */
  public void setVisible( boolean visible )
  {
    final Boolean oldVal = new Boolean( _visible );
    _visible = visible;
    firePropertyChange( PROP_VISIBLE, oldVal, new Boolean( _visible ) );
  }
  
  /*-
   * Viewable interface method.
   */
  public void addPropertyChangeListener( PropertyChangeListener listener )
  {
    if ( _propertyChangeSupport == null )
    {
      _propertyChangeSupport = new SwingPropertyChangeSupport( this );
    }
    _propertyChangeSupport.addPropertyChangeListener( listener );
  }

  /*-
   * Viewable interface method.
   */
  public void removePropertyChangeListener( PropertyChangeListener listener )
  {
    if ( _propertyChangeSupport != null )
    {
      _propertyChangeSupport.removePropertyChangeListener( listener );
    }
  }

  //---------------------------------------------------------------------------
  // Object methods.
  //---------------------------------------------------------------------------
  public String toString()
  {
    return getLabel();
  }

  //---------------------------------------------------------------------------
  // Protected methods.
  //---------------------------------------------------------------------------


  /**
   * Fire the property change event. Fires the event for property 
   * <code>name</code> only if the <code>oldVal</code> is different from the 
   * <code>newVal</code>.
   * @param name The property name.
   * @param oldVal The current value of the specified property.
   * @param newVal The new value of the specified property.
   */
  protected void firePropertyChange( String name, Object oldVal, Object newVal )
  {
    if ( _propertyChangeSupport != null )
    {
      if ( oldVal == null || newVal == null || !oldVal.equals( newVal ))
      {
        _propertyChangeSupport.firePropertyChange( name, oldVal, newVal );
      }
    }
  }
}
