/*
 * Copyright 2005 by Oracle USA
 * 500 Oracle Parkway, Redwood Shores, California, 94065, U.S.A.
 * All rights reserved.
 */
package javax.ide.view;


/**
 * <code>WaitCursor</code> interface. A wait cursor should be used by 
 * extension writers when an extension executes a time consuming task.
 * To show the wait cursor call <code>show</code>. Once the time consuming 
 * task concludes, call <code>hide</code> to remove the wait cursor. These 
 * calls must happen on the event thread.
 *
 * To show a wait cursor:
 * <pre>
 * import javax.ide.view.WaitCursor;
 *
 * // ...
 *
 * void doSomethingTimeConsuming()
 * {
 *   WaitCursor wc = Ide.getGUIUtilities().getWaitCursor();
 *   try
 *   {
 *      wc.show();
 *
 *      //...
 *   } 
 *   finally
 *   {
 *      wc.hide();
 *   }
 * }
 * </pre>
 */
public interface WaitCursor {

    /**
     * Schedules the wait cursor to be shown after the specified number of 
     * milliseconds has elapsed. If {@link WaitCursor#hide} is called before
     * the delay has elapsed, then the wait cursor is not shown. <p>
     *
     * Nested call to this method are acceptable. There must be a call to 
     * the <code>hide</code> method for every call to <code>show</code>.
     * This method must be called from the event thread.
     *
     * @param delay the number of milliseconds to dealy before showing the 
     * wait cursor.
     */
    void show( int delay );
    
    /**
     * Hide the wait cursor.<p>
     *
     * Nested call to this method are acceptable. There must be a call to 
     * the <code>hide</code> method for every call to <code>show</code>.
     * This method must be called from the event thread.
     */
    void hide();
}
