create or replace
PACKAGE BODY TERADATA_UTILITIES AS

TYPE VARCHAR2_ARRAY IS TABLE OF VARCHAR2(100);  -- Customer : Please Modify the List of Date/Timestamp Formats so that the most common literals are at the top.
DT_FORMATS          VARCHAR2_ARRAY;

/**
 * 
 * Gets NLS_DATE_FORMAT format value
 * 
 * @return VARCHAR2
 */
/**
 * 
 * @param 
 * @return 
 */        		   
FUNCTION getNLSDATEformat RETURN VARCHAR2
IS
 l_format VARCHAR2(50);
BEGIN
  SELECT property_value INTO l_format FROM database_properties WHERE property_name = 'NLS_DATE_FORMAT';
  RETURN l_format;
END getNLSDATEformat;

/**
 * 
 * Gets NLS_TIMESTAMP_TZ_FORMAT format value
 * 
 * @return VARCHAR2
 */
FUNCTION getNLSTSTZformat RETURN VARCHAR2
IS
 tstz_format VARCHAR2(50);
BEGIN
  SELECT property_value INTO tstz_format FROM database_properties WHERE property_name = 'NLS_TIMESTAMP_TZ_FORMAT';
  RETURN tstz_format;
END getNLSTSTZformat;

/**
 * 
 * Gets NLS_TIMESTAMP_FORMAT format value
 * 
 * @return VARCHAR2
 */
FUNCTION getNLSTSformat RETURN VARCHAR2
IS
 ts_format VARCHAR2(50);
BEGIN
  SELECT property_value INTO ts_format FROM database_properties WHERE property_name = 'NLS_TIMESTAMP_FORMAT';
  RETURN ts_format;
END getNLSTSformat;

/**
 * 
 * Converts Teradata Date format masks to Oracle Date format masks
 * 
 * @param fmtstr
 * @param precsn
 * @return VARCHAR2
 * 
 */
FUNCTION getOracleDateTimeFormat(fmtstr IN VARCHAR2, precsn IN PLS_INTEGER) RETURN VARCHAR2 IS
  orclfmtstr VARCHAR2(50);
  l_precision PLS_INTEGER;
  l_index PLS_INTEGER := 0;
BEGIN
   IF precsn IS NULL THEN
     l_precision := 1;
   ELSE
     l_precision := precsn;
   END IF;
   
  -- date formats
  orclfmtstr := fmtstr;
  orclfmtstr := REPLACE(orclfmtstr, 'MMMM', 'MONTH');
  orclfmtstr := REPLACE(orclfmtstr, 'mmmm', 'MONTH');
  orclfmtstr := REPLACE(orclfmtstr, 'M4', 'MONTH');
  orclfmtstr := REPLACE(orclfmtstr, 'm4', 'MONTH');
  orclfmtstr := REPLACE(orclfmtstr, 'MMM', 'MON');
  orclfmtstr := REPLACE(orclfmtstr, 'mmm', 'MON');
  orclfmtstr := REPLACE(orclfmtstr, 'M3', 'MON');
  orclfmtstr := REPLACE(orclfmtstr, 'm3', 'MON');
  orclfmtstr := REPLACE(orclfmtstr, 'DDD', 'DDD');
  orclfmtstr := REPLACE(orclfmtstr, 'ddd', 'ddd');
  orclfmtstr := REPLACE(orclfmtstr, 'D3', 'DDD');
  orclfmtstr := REPLACE(orclfmtstr, 'd3', 'DDD');
  orclfmtstr := REPLACE(orclfmtstr, 'DD', 'DD');
  orclfmtstr := REPLACE(orclfmtstr, 'dd', 'DD');
  orclfmtstr := REPLACE(orclfmtstr, 'YYYY', 'YYYY');
  orclfmtstr := REPLACE(orclfmtstr, 'yyyy', 'YYYY');
  orclfmtstr := REPLACE(orclfmtstr, 'Y4', 'YYYY' );
  orclfmtstr := REPLACE(orclfmtstr, 'y4', 'YYYY' );
  orclfmtstr := REPLACE(orclfmtstr, 'YY', 'YY');
  orclfmtstr := REPLACE(orclfmtstr, 'yy', 'YY');
  orclfmtstr := REPLACE(orclfmtstr, 'EEEE', 'DAY');
  orclfmtstr := REPLACE(orclfmtstr, 'eeee', 'DAY');
  orclfmtstr := REPLACE(orclfmtstr, 'E4', 'DAY');
  orclfmtstr := REPLACE(orclfmtstr, 'e4', 'DAY');
  orclfmtstr := REPLACE(orclfmtstr, 'EEE', 'DY');
  orclfmtstr := REPLACE(orclfmtstr, 'eee', 'DY');
  orclfmtstr := REPLACE(orclfmtstr, 'E3', 'DY');
  orclfmtstr := REPLACE(orclfmtstr, 'e3', 'DY');
  orclfmtstr := REPLACE(orclfmtstr, '/', '/');
  orclfmtstr := REPLACE(orclfmtstr, 'B', ' ');
  orclfmtstr := REPLACE(orclfmtstr, 'b', ' ');
  orclfmtstr := REPLACE(orclfmtstr, ',', ',');
  -- orclfmtstr := REPLACE(orclfmtstr, ''', '''');
  orclfmtstr := REPLACE(orclfmtstr, ':', ':');
  orclfmtstr := REPLACE(orclfmtstr, '.', '.');
  orclfmtstr := REPLACE(orclfmtstr, '-', '-');
  orclfmtstr := REPLACE(orclfmtstr, 'HH', 'HH');
  orclfmtstr := REPLACE(orclfmtstr, 'hh', 'HH');
  orclfmtstr := REPLACE(orclfmtstr, 'MI', 'MI');
  orclfmtstr := REPLACE(orclfmtstr, 'mi', 'MI');
  orclfmtstr := REPLACE(orclfmtstr, 'SS', 'SS');
  orclfmtstr := REPLACE(orclfmtstr, 'ss', 'SS');
  orclfmtstr := REPLACE(orclfmtstr, 'S(F)', l_precision);
  orclfmtstr := REPLACE(orclfmtstr, 's(f)', l_precision);
  orclfmtstr := REPLACE(orclfmtstr, 'S(0)', 'FF1');
  orclfmtstr := REPLACE(orclfmtstr, 's(0)', 'FF1');
  orclfmtstr := REPLACE(orclfmtstr, 'S(1)', 'FF1');
  orclfmtstr := REPLACE(orclfmtstr, 's(1)', 'FF1');
  orclfmtstr := REPLACE(orclfmtstr, 'S(2)', 'FF2');
  orclfmtstr := REPLACE(orclfmtstr, 's(2)', 'FF2');
  orclfmtstr := REPLACE(orclfmtstr, 'S(3)', 'FF3');
  orclfmtstr := REPLACE(orclfmtstr, 's(3)', 'FF3');
  orclfmtstr := REPLACE(orclfmtstr, 'S(4)', 'FF4');
  orclfmtstr := REPLACE(orclfmtstr, 's(4)', 'FF4');
  orclfmtstr := REPLACE(orclfmtstr, 'S(5)', 'FF5');
  orclfmtstr := REPLACE(orclfmtstr, 's(5)', 'FF5');
  orclfmtstr := REPLACE(orclfmtstr, 'S(6)', 'FF6');
  orclfmtstr := REPLACE(orclfmtstr, 's(6)', 'FF6');
  l_index := REGEXP_INSTR(orclfmtstr, 'ssd', 1, 1, 0, 'i'); -- replace radix symbol 'd' or 'D' with '.'
  IF l_index > 0 THEN
     orclfmtstr := REGEXP_REPLACE(orclfmtstr, 'd', '.', l_index+2, 1, 'i');
  END IF;
  orclfmtstr := REPLACE(orclfmtstr, 'Z', 'TZH:TZM');
  orclfmtstr := REPLACE(orclfmtstr, 'HHh', 'HH24"h"');
  orclfmtstr := REPLACE(orclfmtstr, 'HHH', 'HH24"H"');
  orclfmtstr := REPLACE(orclfmtstr, 'hhh', 'hh24"h"');
  orclfmtstr := REPLACE(orclfmtstr, 'MIm', 'MI"m"');
  orclfmtstr := REPLACE(orclfmtstr, 'MMM', 'MM"M"');
  orclfmtstr := REPLACE(orclfmtstr, 'mmm', 'mm"m"');
  orclfmtstr := REPLACE(orclfmtstr, 'SSs', 'SS"s"');
  orclfmtstr := REPLACE(orclfmtstr, 'SSS', 'SS"S"');
  orclfmtstr := REPLACE(orclfmtstr, 'sss', 'ss"s"');
  orclfmtstr := REPLACE(orclfmtstr, 'FF1s', 'FF1"s"');
  orclfmtstr := REPLACE(orclfmtstr, 'FF1S', 'FF1"S"');
  orclfmtstr := REPLACE(orclfmtstr, 'FF2s', 'FF2"s"');
  orclfmtstr := REPLACE(orclfmtstr, 'FF2S', 'FF2"S"');
  orclfmtstr := REPLACE(orclfmtstr, 'FF3s', 'FF3"s"');
  orclfmtstr := REPLACE(orclfmtstr, 'FF3S', 'FF3"S"');
  orclfmtstr := REPLACE(orclfmtstr, 'FF4s', 'FF4"s"');
  orclfmtstr := REPLACE(orclfmtstr, 'FF4S', 'FF4"S"');
  orclfmtstr := REPLACE(orclfmtstr, 'FF5s', 'FF5"s"');
  orclfmtstr := REPLACE(orclfmtstr, 'FF5S', 'FF5"S"');
  orclfmtstr := REPLACE(orclfmtstr, 'FF6s', 'FF6"s"');
  orclfmtstr := REPLACE(orclfmtstr, 'FF6S', 'FF6"S"');
  RETURN orclfmtstr;
  
END getOracleDateTimeFormat;

/**
 * Converts Teradata Number format masks to Oracle Number format masks.
 * 
 * @param p_expr
 * @param fmtstr
 * @param p_precision
 * @param p_scale
 * @return VARCHAR2
 * 
 */

FUNCTION getOracleNumberFormat(p_expr IN VARCHAR2, fmtstr IN VARCHAR2, p_precision IN BINARY_INTEGER, 
           p_scale IN BINARY_INTEGER) RETURN VARCHAR2 IS
orclfmtstr VARCHAR2(50);
l_idx PLS_INTEGER;
l_tmpstr VARCHAR2(50);
BEGIN
   orclfmtstr := UPPER(fmtstr);
   orclfmtstr := REPLACE(orclfmtstr, 'Z', '9');
   orclfmtstr := REPLACE(orclfmtstr, 'V', 'D');
   orclfmtstr := REPLACE(orclfmtstr, '£', 'C');
   orclfmtstr := REPLACE(orclfmtstr, '¥', 'C');
   orclfmtstr := REPLACE(orclfmtstr, '¤', 'C');
   orclfmtstr := REPLACE(orclfmtstr, 'N', 'C');
   orclfmtstr := REPLACE(orclfmtstr, 'O', 'U');
   orclfmtstr := REPLACE(orclfmtstr, 'U', 'U');
   orclfmtstr := REPLACE(orclfmtstr, '+', 'S');
   orclfmtstr := REPLACE(orclfmtstr, '-', 'S');
   orclfmtstr := REPLACE(orclfmtstr, 'B', NULL);
   orclfmtstr := REPLACE(orclfmtstr, 'G', NULL);
   
   l_idx := REGEXP_INSTR(orclfmtstr,  '9\(\d+\)');
   IF l_idx > 0 THEN
      orclfmtstr := REGEXP_REPLACE(orclfmtstr, '[()]', NULL);
      l_idx := INSTR(orclfmtstr, 'D', 1);
      IF l_idx > 0 THEN
        orclfmtstr := LPAD(substr(orclfmtstr, 1, 1), SUBSTR(orclfmtstr, 2, 1), substr(orclfmtstr, 1, 1)) || 'D' ||
                      LPAD(SUBSTR(orclfmtstr, l_idx+1, 1), SUBSTR(orclfmtstr, l_idx+2, LENGTH(orclfmtstr)), 
                           SUBSTR(orclfmtstr, l_idx+1, 1));              
      END IF;
   END IF;
   
   l_idx := REGEXP_INSTR(orclfmtstr, '\d+\(I\)C'); 
   IF l_idx > 0 THEN
     orclfmtstr := REGEXP_REPLACE(orclfmtstr, '[CI()]', NULL);
     orclfmtstr := LPAD(9, orclfmtstr, 9) || 'C';
   END IF;
   
   l_idx := REGEXP_INSTR(orclfmtstr, 'C\d+\(I\)'); 
   IF l_idx > 0 THEN
     orclfmtstr := REGEXP_REPLACE(orclfmtstr, '[CI()]', NULL);
     orclfmtstr := 'C' || LPAD(9, orclfmtstr, 9);
   END IF;
   
   -- Integer part
   IF INSTR(orclfmtstr, '9(I)') > 0 THEN 
     IF p_precision > 0 THEN
        l_tmpstr := LPAD(9, p_precision - p_scale, 9);
     ELSE
        l_tmpstr := LPAD(9, REGEXP_COUNT(p_expr, '\d'), 9);
     END IF;
     orclfmtstr := REPLACE(orclfmtstr, '9(I)', l_tmpstr);
   END IF;
   
   -- Fractional part
   IF INSTR(orclfmtstr, '9(F)') > 0 AND p_precision > 0 THEN
      l_tmpstr := LPAD(9, p_scale, 9);
      orclfmtstr := REPLACE(orclfmtstr, '9(F)', l_tmpstr);
   END IF;
   
   -- Grouping may be required.
   /*
   IF INSTR(orclfmtstr, 'G') > 0 THEN
      orclfmtstr := SUBSTR(orclfmtstr, 2, 1) || 'G' || SUBSTR(orclfmtstr, 3, LENGTH(orclfmtstr));
   END IF;
   */
   RETURN orclfmtstr;
END getOracleNumberFormat;

/**
 * 
 * Converts VARCHAR2 string to TIMESTAMP WITH TIME ZONE type
 *  
 * @param p_date_expr
 * @param p_dateformat
 * @return TIMESTAMP WITH TIME ZONE
 */

FUNCTION str_to_timestamptz(p_date_expr IN VARCHAR2, p_dateformat IN VARCHAR2)
RETURN TIMESTAMP WITH TIME ZONE
IS
      temp_exp VARCHAR2(50);
      format_str VARCHAR2(50) := NULL;
BEGIN
      IF p_date_expr IS NULL THEN
        RETURN NULL;
      END IF;
      
      IF p_dateformat IS NULL THEN
        -- use the default server format
        format_str := getNLSTSTZformat;
      ELSE
        format_str := p_dateformat;
      END IF;
     
      temp_exp := TRIM(p_date_expr);
      RETURN TO_TIMESTAMP_TZ(temp_exp, format_str);
EXCEPTION
      WHEN OTHERS THEN
         raise_application_error(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END str_to_timestamptz;

/**
 * Uses the Teradata formula to calculate the Numeric. 
 * 
 * @param p_date
 * @return NUMBER
 * 
*/
FUNCTION date_to_numeric(p_date IN DATE) RETURN NUMBER AS
l_year NUMBER;
l_month NUMBER;
l_day NUMBER;
BEGIN
   l_year := EXTRACT(YEAR FROM p_date);
   l_month := EXTRACT(MONTH FROM p_date);
   l_day := EXTRACT(DAY FROM p_date);
   RETURN (l_year - 1900) * 10000 + (l_month * 100) + l_day;
EXCEPTION
      WHEN OTHERS THEN
         RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END date_to_numeric;

/**
 * 
 * Converts from NUMBER type to DATE type
 * 
 * @param p_numeric
 * @param p_format
 * @return DATE
 * 
 */
FUNCTION numeric_to_date(p_numeric IN NUMBER, p_format IN VARCHAR2 := NULL) RETURN DATE AS
l_year NUMBER;
l_month NUMBER;
l_day NUMBER;
l_format VARCHAR2(50);
l_tmpformat VARCHAR2(50); 
l_punct VARCHAR2(2);
l_punct_occurs NUMBER := 1;
l_dtelemType VARCHAR2(10);
l_punct_index NUMBER := 0;
l_datestr VARCHAR2(50);

BEGIN
  IF p_format IS NULL THEN
    l_format :=  getNLSDATEformat;
  ELSE
    l_format := p_format;
  END IF;
  
  l_year := FLOOR(p_numeric/10000) + 1900;
  l_month := FLOOR(MOD(p_numeric, 10000)/100);
  l_day := MOD(MOD(p_numeric, 10000), 100);
  l_tmpformat := l_format;
  WHILE l_tmpformat IS NOT NULL AND REGEXP_INSTR(l_tmpformat, '[[:punct:]]', 1, l_punct_occurs) >= 0
  LOOP
    BEGIN
     l_punct_index := REGEXP_INSTR(l_tmpformat, '[[:punct:]]', 1, l_punct_occurs);
     IF l_punct_index > 0 THEN
       l_dtelemType := SUBSTR(l_tmpformat, 1, l_punct_index-1);
     ELSE
       l_dtelemType := l_tmpformat;
     END IF;  
     CASE
     WHEN UPPER(l_dtelemType) = 'MM' OR UPPER(l_dtelemType) = 'MON' OR UPPER(l_dtelemType) = 'MONTH' THEN
         IF UPPER(l_dtelemType) = 'MON' THEN
            l_datestr := CONCAT(l_datestr, TO_CHAR(TO_DATE(l_month, 'MM'), 'MON'));
         ELSIF UPPER(l_dtelemType) = 'MONTH' THEN
            l_datestr := CONCAT(l_datestr, TO_CHAR(TO_DATE(l_month, 'MM'), 'MONTH'));
         ELSE
            l_datestr := CONCAT(l_datestr, l_month);
         END IF;
         IF REGEXP_INSTR(l_tmpformat, '[[:punct:]]') > 0 THEN
            l_datestr := CONCAT(l_datestr, l_punct); 
            l_punct := SUBSTR(l_tmpformat, REGEXP_INSTR(l_tmpformat, '[[:punct:]]'), 1);
         END IF;   
     WHEN UPPER(l_dtelemType) = 'DD' THEN
         l_datestr := CONCAT(l_datestr, l_day);
         IF REGEXP_INSTR(l_tmpformat, '[[:punct:]]') > 0 THEN
           l_punct := SUBSTR(l_tmpformat, REGEXP_INSTR(l_tmpformat, '[[:punct:]]'), 1);
           l_datestr := CONCAT(l_datestr, l_punct);
         END IF;  
     WHEN UPPER(l_dtelemType) = 'YYYY' OR UPPER(l_dtelemType) = 'Y,YYY' OR UPPER(l_dtelemType) = 'YYY' OR 
          UPPER(l_dtelemType) = 'YY' OR UPPER(l_dtelemType) = 'Y' OR
          UPPER(l_dtelemType) = 'SYYYY' OR UPPER(l_dtelemType) = 'SYEAR' OR UPPER(l_dtelemType) = 'YEAR' OR 
          UPPER(l_dtelemType) = 'IYYY' OR UPPER(l_dtelemType) = 'IYY' OR UPPER(l_dtelemType) = 'IY' OR UPPER(l_dtelemType) = 'I' OR 
          UPPER(l_dtelemType) = 'RRRR' OR UPPER(l_dtelemType) = 'RR' THEN
         l_datestr := CONCAT(l_datestr, l_year);
         IF REGEXP_INSTR(l_tmpformat, '[[:punct:]]') > 0 THEN
           l_punct := SUBSTR(l_tmpformat, REGEXP_INSTR(l_tmpformat, '[[:punct:]]'), 1);
           l_datestr := CONCAT(l_datestr, l_punct);
         END IF;  
     END CASE;
     IF l_punct_index > 0 THEN
       l_tmpformat := REPLACE(l_tmpformat, SUBSTR(l_tmpformat, 1, l_punct_index), '');
     ELSE
       l_tmpformat := NULL;
     END IF;  
    END;
  END LOOP;
  RETURN TO_DATE(l_datestr, l_format);
  
  -- RETURN TO_DATE(TO_CHAR(DATE l_datestr, l_format), l_format);
  /*
  IF p_format IS NOT NULL THEN
    RETURN TO_DATE(l_datestr, l_format);
  ELSE
     l_tmpformat := l_format;
     l_datestr := '';
     WHILE l_tmpformat IS NOT NULL AND REGEXP_INSTR(l_tmpformat, '[[:punct:]]', 1, 1) > 0
     LOOP
       BEGIN
        IF UPPER(SUBSTR(l_tmpformat, 1, REGEXP_INSTR(l_tmpformat, '[[:punct:]]', 1, 1)-1)) = 'DD' THEN
            l_datestr := CONCAT(l_datestr, l_day);
            l_punct := SUBSTR(l_tmpformat, REGEXP_INSTR(l_tmpformat, '[[:punct:]]'), 1);
            l_datestr := CONCAT(l_datestr, l_punct);
            -- l_tmpformat := REPLACE(l_tmpformat, SUBSTR(l_tmpformat, 1, REGEXP_INSTR(l_tmpformat, '[[:punct:]]', 1, 1)), '');
        ELSIF UPPER(SUBSTR(l_tmpformat, 1, REGEXP_INSTR(l_tmpformat, '[[:punct:]]', 1, 1)-1)) = 'MM' OR
              UPPER(SUBSTR(l_tmpformat, 1, REGEXP_INSTR(l_tmpformat, '[[:punct:]]', 1, 1)-1)) = 'MON' OR
              UPPER(SUBSTR(l_tmpformat, 1, REGEXP_INSTR(l_tmpformat, '[[:punct:]]', 1, 1)-1)) = 'MONTH' THEN
            l_datestr := CONCAT(l_datestr, l_month);
            l_punct := SUBSTR(l_tmpformat, REGEXP_INSTR(l_tmpformat, '[[:punct:]]'), 1);
            l_datestr := CONCAT(l_datestr, l_punct);
        ELSIF UPPER(SUBSTR(l_tmpformat, 1, REGEXP_INSTR(l_tmpformat, '[[:punct:]]', 1, 1)-1)) = 'YYYY' OR UPPER(SUBSTR(l_tmpformat, 1, REGEXP_INSTR(l_tmpformat, '[[:punct:]]', 1, 1)-1)) = 'Y,YYY' OR UPPER(SUBSTR(l_tmpformat, 1, REGEXP_INSTR(l_tmpformat, '[[:punct:]]', 1, 1)-1)) = 'YYY' OR 
          UPPER(SUBSTR(l_tmpformat, 1, REGEXP_INSTR(l_tmpformat, '[[:punct:]]', 1, 1)-1)) = 'YY' OR UPPER(SUBSTR(l_tmpformat, 1, REGEXP_INSTR(l_tmpformat, '[[:punct:]]', 1, 1)-1)) = 'Y' OR
          UPPER(SUBSTR(l_tmpformat, 1, REGEXP_INSTR(l_tmpformat, '[[:punct:]]', 1, 1)-1)) = 'SYYYY' OR UPPER(SUBSTR(l_tmpformat, 1, REGEXP_INSTR(l_tmpformat, '[[:punct:]]', 1, 1)-1)) = 'SYEAR' OR UPPER(SUBSTR(l_tmpformat, 1, REGEXP_INSTR(l_tmpformat, '[[:punct:]]', 1, 1)-1)) = 'YEAR' OR 
          UPPER(SUBSTR(l_tmpformat, 1, REGEXP_INSTR(l_tmpformat, '[[:punct:]]', 1, 1)-1)) = 'IYYY' OR UPPER(SUBSTR(l_tmpformat, 1, REGEXP_INSTR(l_tmpformat, '[[:punct:]]', 1, 1)-1)) = 'IYY' OR UPPER(l_dtelemType) = 'IY' OR UPPER(SUBSTR(l_tmpformat, 1, REGEXP_INSTR(l_tmpformat, '[[:punct:]]', 1, 1)-1)) = 'I' OR 
          UPPER(SUBSTR(l_tmpformat, 1, REGEXP_INSTR(l_tmpformat, '[[:punct:]]', 1, 1)-1)) = 'RRRR' OR UPPER(SUBSTR(l_tmpformat, 1, REGEXP_INSTR(l_tmpformat, '[[:punct:]]', 1, 1)-1)) = 'RR' THEN
            l_datestr := CONCAT(l_datestr, l_year);
            l_punct := SUBSTR(l_tmpformat, REGEXP_INSTR(l_tmpformat, '[[:punct:]]'), 1);
            l_datestr := CONCAT(l_datestr, l_punct);
        END IF;
        l_tmpformat := REPLACE(l_tmpformat, SUBSTR(l_tmpformat, 1, REGEXP_INSTR(l_tmpformat, '[[:punct:]]', 1, 1)), '');
       END;
     END LOOP;
     RETURN TO_DATE(l_datestr, l_format);
  END IF;
  */
 -- RETURN TO_DATE(TO_CHAR(CAST(l_datestr AS DATE), l_format), l_format);
EXCEPTION
  WHEN OTHERS THEN
         RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END numeric_to_date;

/**
 * 
 * Returns the Column type in the form of VARCHAR2 string.
 *
 * This function will not work well if the Oracle Metadata has 
 * case-sensitive identifiers.
 * 
 * @param p_expr
 * @return VARCHAR2
 */
FUNCTION type(p_expr IN VARCHAR2) RETURN VARCHAR2 AS
l_type VARCHAR2(1000) := NULL;
l_code NUMBER := NULL;
l_table_name VARCHAR2(100);
l_column_name VARCHAR2(100);
l_invalid_expr EXCEPTION;
BEGIN
    IF p_expr IS NOT NULL THEN
      IF REGEXP_INSTR(p_expr, '\.', 1, 1, 0, 'i', 0) > 0 THEN
        IF REGEXP_LIKE(p_expr, '^\d+\.\d+$') THEN
           RAISE l_invalid_expr; 
        ELSIF REGEXP_INSTR(p_expr, '\.', 1, 2, 0, 'i', 0) > 0 THEN
          l_table_name := REGEXP_SUBSTR(p_expr, '(\w+)(\.)(\w+)(\.)(\w+)', 1, 1, 'i', 3);
          l_column_name := REGEXP_SUBSTR(p_expr, '(\w+)(\.)(\w+)(\.)(\w+)', 1, 1, 'i', 5);
        ELSIF REGEXP_INSTR(p_expr, '\.', 1, 1, 0, 'i', 0) > 0 THEN
          l_table_name := REGEXP_SUBSTR(p_expr, '(\w+)(\.)(\w+)', 1, 1, 'i', 1);
          l_column_name := REGEXP_SUBSTR(p_expr, '(\w+)(\.)(\w+)', 1, 1, 'i', 3); 
        ELSE
          RAISE l_invalid_expr;
        END IF;
        
        IF l_table_name IS NOT NULL AND l_column_name IS NOT NULL AND LENGTH(l_table_name) > 0 AND LENGTH(l_column_name) > 0 THEN
           select data_type || '(' || data_length || ')' into l_type from user_tab_columns where table_name = l_table_name and column_name = l_column_name and rownum = 1;
        ELSE
           RAISE l_invalid_expr;
        END IF;
      END IF; 
    ELSE
     RETURN NULL;
    END IF;
    RETURN l_type;
EXCEPTION
  WHEN l_invalid_expr THEN
       RAISE_APPLICATION_ERROR(-20001, 'Invalid expression. The expression should be in the format [<SCHEMA>.]<TABLE>.<COLUMN>');
  WHEN OTHERS THEN
       RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END type;

/**
 * 
 * Looks for a default value of a column <p_expr2> for
 * one or more comma separated tables <p_expr1>. 
 * 
 * This function will be
 * successful if only one of the tables <p_expr1> has this
 * column <p_expr2> otherwise throws exception.
 * 
 * If the default value of the column is null or empty than
 * this function will return NULL. 
 * 
 * This function will not work well if the Oracle Metadata has 
 * case-sensitive identifiers.
 * 
 * @param p_expr1
 * @param p_expr2
 * @return VARCHAR2
 */
FUNCTION ora_default(p_expr1 IN VARCHAR2, p_expr2 IN VARCHAR2) RETURN VARCHAR2 AS
l_default LONG := NULL;
l_expr1 long := p_expr1;
l_expr2 VARCHAR2(100) := p_expr2;
l_table_name VARCHAR2(100);
l_column_name VARCHAR2(100);
l_count BINARY_INTEGER := 0;
l_outval BINARY_INTEGER := 100;
l_handle NUMBER;
l_sqlstr VARCHAR2(10000);
l_dummy NUMBER;
l_column_not_found EXCEPTION;
l_same_column_name_found EXCEPTION;
l_column_in_tables VARCHAR2(10000) := NULL;
l_tables VARCHAR2(10000) := NULL;
l_invalid_table EXCEPTION;
l_invalid_column EXCEPTION;

BEGIN
   
    IF p_expr1 IS NULL OR LENGTH(p_expr1) = 0 THEN
       RAISE l_invalid_table;
    END IF;
    
    IF p_expr2 IS NULL OR LENGTH(p_expr2) = 0 THEN
       RAISE l_invalid_column;
    END IF;
    
    l_handle := DBMS_SQL.OPEN_CURSOR;
    -- This will be a problem if the tables and columns are named case-sensitive.
    l_expr1 := UPPER(REGEXP_REPLACE(l_expr1, '(\w+)(\,)*', '''\1''\2'));
    l_expr2 := UPPER(REGEXP_REPLACE(l_expr2, '(\w+)', '''\1'''));
    l_sqlstr := 'select count(column_name) from user_tab_columns where upper(table_name) in' || '(' || l_expr1 ||')' || ' and upper(column_name) = ' || l_expr2;
    DBMS_SQL.PARSE(l_handle, l_sqlstr, DBMS_SQL.NATIVE);
    DBMS_SQL.DEFINE_COLUMN(l_handle, 1, l_count);
    l_dummy := DBMS_SQL.EXECUTE(l_handle);
    LOOP
      IF DBMS_SQL.FETCH_ROWS(l_handle) > 0 THEN 
        DBMS_SQL.COLUMN_VALUE(l_handle, 1, l_count);
      ELSE
        EXIT;
      END IF;
    END LOOP;  
    DBMS_SQL.CLOSE_CURSOR(l_handle);
   
    IF l_count = 0 THEN
      RAISE l_column_not_found;
    ELSIF l_count = 1 THEN
       l_handle := DBMS_SQL.OPEN_CURSOR;
       l_sqlstr := 'select data_default from user_tab_columns where upper(table_name) in ' || '(' || l_expr1 || ')' || ' and upper(column_name) = ' ||  l_expr2;
       DBMS_SQL.PARSE(l_handle, l_sqlstr, DBMS_SQL.NATIVE);
       DBMS_SQL.DEFINE_COLUMN_LONG(l_handle, 1);
       l_dummy := DBMS_SQL.EXECUTE(l_handle);
       LOOP
        IF DBMS_SQL.FETCH_ROWS(l_handle) > 0 THEN 
          DBMS_SQL.COLUMN_VALUE_LONG(l_handle, 1, l_outval, 0, l_default, l_outval);
        ELSE
          EXIT;
        END IF;
      END LOOP;
      DBMS_SQL.CLOSE_CURSOR(l_handle);
      
      IF l_default IS NULL or LENGTH(l_default) = 0 or LENGTH(REGEXP_REPLACE(l_default, 'NULL', '', 1, 1, 'i')) IS NULL THEN
        RETURN NULL;
      END IF;
      
      RETURN l_default;  
    ELSE --  l_count > 1 
      l_handle := DBMS_SQL.OPEN_CURSOR;
      l_sqlstr :=  'select table_name from user_tab_columns where upper(table_name) in ' || '(' || l_expr1 || ')' ||' and upper(column_name) = ' || l_expr2; 
      DBMS_SQL.PARSE(l_handle, l_sqlstr, DBMS_SQL.NATIVE);
      DBMS_SQL.DEFINE_COLUMN(l_handle, 1, l_tables, 100);
      l_dummy := DBMS_SQL.EXECUTE(l_handle);
      LOOP
        IF DBMS_SQL.FETCH_ROWS(l_handle) > 0 THEN 
          DBMS_SQL.COLUMN_VALUE(l_handle, 1, l_tables);
          IF l_column_in_tables IS NULL THEN
            l_column_in_tables := l_tables;
          ELSE
            l_column_in_tables := l_column_in_tables || ', ' || l_tables;
          END IF;  
        ELSE
          EXIT;
        END IF;
      END LOOP;
      DBMS_SQL.CLOSE_CURSOR(l_handle);
      RAISE l_same_column_name_found;
    END IF;
EXCEPTION
  WHEN l_invalid_table THEN
      RAISE_APPLICATION_ERROR(-20004, 'Table param <p_expr1> cannot be NULL or empty');
  WHEN l_invalid_column THEN
      RAISE_APPLICATION_ERROR(-20003, 'Column param <p_expr2> cannot be NULL or empty');
  WHEN l_column_not_found THEN
      RAISE_APPLICATION_ERROR(-20002, 'Column ' || p_expr2 || ' is not found in Tables ' || p_expr1);
  WHEN l_same_column_name_found THEN
      RAISE_APPLICATION_ERROR(-20001, 'Column ' || p_expr2 || ' is found in Tables ' || l_column_in_tables || ' unable to get its default value');
  WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END ora_default;

/**
 * 
 *
 * Looks for a default value of a column id <p_expr2> for
 * one or more comma separated tables <p_expr1>. 
 * 
 * This function will be
 * successful if only one of the tables <p_expr1> has this
 * column id <p_expr2> otherwise throws exception.
 * 
 * If the default value of the column id found is null or empty then
 * this function will return null. 
 * 
 * This function will not work well if the Oracle Metadata has 
 * case-sensitive identifiers.
 * 
 * @param p_expr1
 * @param p_expr2
 * @return VARCHAR2
 */
FUNCTION ora_default(p_expr1 IN VARCHAR2, p_expr2 IN BINARY_INTEGER) RETURN VARCHAR2 AS
l_expr1 long := p_expr1;
l_column_name VARCHAR2(100);
l_count BINARY_INTEGER := 0;
l_handle NUMBER;
l_sqlstr VARCHAR2(10000);
l_dummy NUMBER;
l_column_id_not_found EXCEPTION;
l_same_column_id_found EXCEPTION;
l_invalid_table EXCEPTION;
l_invalid_column_id EXCEPTION;
l_error_code NUMBER;
BEGIN
  
  IF p_expr1 IS NULL OR LENGTH(p_expr1) = 0 THEN
       RAISE l_invalid_table;
  END IF;
  
  IF p_expr2 IS NULL OR p_expr2 = 0 THEN
     RAISE l_invalid_column_id;
  END IF;
  
  -- This will be a problem if the tables are named case-sensitive.
  l_expr1 := UPPER(REGEXP_REPLACE(l_expr1, '(\w+)(\,)*', '''\1''\2'));
  l_handle := DBMS_SQL.OPEN_CURSOR;
  l_sqlstr := 'select count(distinct column_name) from user_tab_columns where upper(table_name) in' || '(' || l_expr1 ||')' || ' and column_id = ' || p_expr2;
  DBMS_SQL.PARSE(l_handle, l_sqlstr, DBMS_SQL.NATIVE);
  DBMS_SQL.DEFINE_COLUMN(l_handle, 1, l_count);
  l_dummy := DBMS_SQL.EXECUTE(l_handle);
  LOOP
    IF DBMS_SQL.FETCH_ROWS(l_handle) > 0 THEN 
      DBMS_SQL.COLUMN_VALUE(l_handle, 1, l_count);
    ELSE
      EXIT;
    END IF;
  END LOOP;  
  DBMS_SQL.CLOSE_CURSOR(l_handle);
  
  IF l_count = 0 THEN
    RAISE l_column_id_not_found;
  ELSIF l_count = 1 THEN
    l_handle := DBMS_SQL.OPEN_CURSOR;
    l_sqlstr := 'select distinct column_name from user_tab_columns where upper(table_name) in' || '(' || l_expr1 ||')' || ' and column_id = ' || p_expr2;
    DBMS_SQL.PARSE(l_handle, l_sqlstr, DBMS_SQL.NATIVE);
    DBMS_SQL.DEFINE_COLUMN(l_handle, 1, l_column_name, 100);
    l_dummy := DBMS_SQL.EXECUTE(l_handle);
    LOOP
      IF DBMS_SQL.FETCH_ROWS(l_handle) > 0 THEN 
        DBMS_SQL.COLUMN_VALUE(l_handle, 1, l_column_name);
      ELSE
        EXIT;
      END IF;
    END LOOP;  
    DBMS_SQL.CLOSE_CURSOR(l_handle);
    BEGIN
      RETURN ora_default(p_expr1, l_column_name);
    EXCEPTION
      WHEN OTHERS THEN
        l_error_code := SQLCODE;
        IF l_error_code = -20001 THEN
           RAISE l_same_column_id_found;
        END IF;   
    END;   
  ELSE -- l_count > 1
    RAISE l_same_column_id_found;  
  END IF;
EXCEPTION
  WHEN l_invalid_table THEN
      RAISE_APPLICATION_ERROR(-20004, 'Table param <p_expr1> cannot be NULL or empty');
  WHEN l_invalid_column_id THEN
      RAISE_APPLICATION_ERROR(-20003, 'Invalid Column Id ' || p_expr2);
  WHEN l_column_id_not_found THEN
      RAISE_APPLICATION_ERROR(-20002, 'Column Id ' || p_expr2 || ' is not found in Tables ' || p_expr1);
  WHEN l_same_column_id_found THEN
      RAISE_APPLICATION_ERROR(-20001, 'Column Id ' || p_expr2 || ' is found in more than one of the Tables ' || p_expr1 || '. Unable to get the Default value');    
  WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END ora_default;


/**
 * Returns a pi value.
 * 
 * @return NUMBER
 * 
 */
FUNCTION pi
RETURN NUMBER
IS
    pi NUMBER := 3.141592653589793116;
BEGIN
    RETURN pi;
END pi;

/**
 * Converts radians to degreees.
 * 
 * @param p_angle_radians
 * @return NUMBER
 */
FUNCTION degrees(p_angle_radians IN NUMBER) 
RETURN NUMBER
IS
BEGIN
    IF p_angle_radians IS NULL THEN
      RETURN NULL;
    END IF;
    
    RETURN p_angle_radians / pi() * 180;
EXCEPTION 
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END degrees;

/**
 * Converts degrees to radians.
 * 
 * @param p_degree
 * @return NUMBER
 */
FUNCTION radians(p_degree IN NUMBER)
RETURN NUMBER
IS
    v_rad NUMBER;
BEGIN
    v_rad := p_degree / 180 * pi();
    RETURN TRUNC(v_rad, 2);
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END radians;

FUNCTION random(p_x BINARY_INTEGER, p_y BINARY_INTEGER) RETURN BINARY_INTEGER AS
BEGIN
    RETURN DBMS_RANDOM.VALUE(p_x, p_y);
END random;

/**
 *  The Expected return values as described in the Teradata Documentation for 
 *  POSITION function.
 *  
 * @param p_findstr
 * @param p_instr
 * @return NUMBER
 *  
 */
FUNCTION position(p_findstr IN VARCHAR2, p_instr IN VARCHAR2) RETURN NUMBER AS
BEGIN
  IF p_findstr IS NULL AND p_instr IS NULL THEN
     RETURN NULL;
  ELSIF LENGTH(p_findstr) = 0 THEN
     RETURN 1;
  ELSIF LENGTH(p_findstr) > 0 AND LENGTH(p_instr) > 0 THEN
     RETURN INSTR(p_instr, p_findstr);
  ELSE
     RETURN 0;
  END IF;
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END position;

/**
 *  The Expected return values as described in the Teradata Documentation for 
 *  POSITION function.
 *  
 * @param p_findstr
 * @param p_instr
 * @return NUMBER
 *  
 */
FUNCTION position(p_findstr IN RAW, p_instr IN BLOB) RETURN NUMBER AS
BEGIN
  IF p_findstr IS NULL AND p_instr IS NULL THEN
     RETURN NULL;
  ELSIF LENGTH(p_findstr) = 0 THEN
     RETURN 1;
  ELSE 
     RETURN DBMS_LOB.INSTR(p_instr, p_findstr);
  END IF;
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END position;

/**
 * Returns the starting index of the p_str where p_findstr is found. 
 * 
 * @param p_str
 * @param p_findstr
 * @return NUMBER
 */
FUNCTION ora_index(p_str IN VARCHAR2, p_findstr IN VARCHAR2) RETURN NUMBER AS
BEGIN
  IF p_str IS NULL AND p_findstr IS NULL THEN
     RETURN NULL;
  ELSIF LENGTH(p_findstr) = 0 THEN
     RETURN NULL;
  ELSE 
     RETURN INSTR(p_str, p_findstr);
  END IF;   
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END ora_index;

/**
 * Returns the starting index of the p_str where p_findstr is found. 
 * 
 * @param p_str
 * @param p_findstr
 * @return NUMBER
 */
FUNCTION ora_index(p_str IN BLOB, p_findstr IN RAW) RETURN NUMBER AS
BEGIN
  IF p_str IS NULL AND p_findstr IS NULL THEN
     RETURN NULL;
  ELSIF LENGTH(p_findstr) = 0 THEN
     RETURN NULL;
  ELSE 
     RETURN position(p_findstr, p_str);
  END IF;  
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END ora_index;

/**
 * Returns the result in Hexadecimal. 
 * 
 * @param p_str
 * @return VARCHAR2
 */
FUNCTION char2hexint(p_str IN VARCHAR2) RETURN VARCHAR2 AS
BEGIN
  IF p_str IS NULL OR LENGTH(p_str) = 0 THEN
     RETURN NULL;
  ELSE
     RETURN RAWTOHEX(UTL_RAW.CAST_TO_RAW(p_str));
  END IF;
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END char2hexint;

/**
 * Compares the length of params p_expr1 and p_expr2 and left pads to
 * 0 the smaller param so the two params are of equal length.
 * 
 * @param p_expr1
 * @param p_expr2
 * 
 */
PROCEDURE lpad_args(p_expr1 IN OUT VARCHAR2, p_expr2 IN OUT VARCHAR2) AS
BEGIN
  IF LENGTH(p_expr1) > LENGTH(p_expr2) THEN
    p_expr2 := LPAD(p_expr2, LENGTH(p_expr1), '0');
  ELSIF LENGTH(p_expr1) < LENGTH(p_expr2) THEN
    p_expr1 := LPAD(p_expr1, LENGTH(p_expr2), '0');
  END IF;
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END lpad_args;

/**
 * Returns a Oracle number format(modal)
 * 
 * @param p_expr
 * @return VARCHAR2
 */
FUNCTION numformat(p_expr NUMBER) RETURN VARCHAR2 AS
BEGIN
  IF LENGTH(p_expr) = 0 THEN
    RETURN NULL;
  ELSE
    RETURN RPAD('X', p_expr, 'X');
  END IF;  
END numformat;

/**
 * Returns a hexadecimal value converted to RAW type.
 * 
 * @param p_expr
 * @return VARCHAR2
 */
FUNCTION to_raw(p_expr NUMBER) RETURN RAW AS
l_hexformat VARCHAR2(10000);
BEGIN
  l_hexformat := 'FM' || numformat(LENGTH(p_expr));
  RETURN HEXTORAW(TO_CHAR(p_expr, l_hexformat));
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END to_raw;

/**
 * Performs a Bit COMPLEMENT/NOT operation. Uses UTL_RAW package.
 * 
 * @param p_expr1
 * @return NUMBER
 */
FUNCTION ora_bitcomplement(p_expr IN NUMBER) RETURN NUMBER AS
l_val RAW(2000);
BEGIN
  IF p_expr IS NULL OR LENGTH(p_expr) = 0 THEN
    RETURN NULL;
  ELSE  
    l_val := to_raw(p_expr);
    RETURN TO_NUMBER(UTL_RAW.BIT_COMPLEMENT(l_val), numformat(LENGTH(l_val)));
  END IF;  
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END ora_bitcomplement;

/**
 * Performs a Bit COMPLEMENT/NOT operation. Uses UTL_RAW package.
 * 
 * @param p_expr1
 * @return RAW
 */
FUNCTION ora_bitcomplement(p_expr IN RAW) RETURN RAW AS
BEGIN
  IF p_expr IS NULL OR LENGTH(p_expr) = 0 THEN
    RETURN NULL;
  ELSE
    RETURN UTL_RAW.BIT_COMPLEMENT(p_expr);
  END IF;
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);  
END ora_bitcomplement;

/**
 * Performs a Bit OR operation. Uses UTL_RAW package.
 * 
 * @param p_expr1
 * @param p_expr2
 * @return NUMBER
 */
FUNCTION ora_bitor(p_expr1 IN NUMBER, p_expr2 IN NUMBER) RETURN NUMBER AS
l_val RAW(2000);
BEGIN
  IF p_expr1 IS NULL OR LENGTH(p_expr1) = 0 OR p_expr2 IS NULL OR LENGTH(p_expr2) = 0 THEN
    RETURN NULL;
  /*
  ELSIF p_expr1 = 0 AND p_expr2 = 0 THEN
    RETURN 0;
  ELSIF p_expr1 = 1 OR p_expr2 = 1 THEN
    RETURN 1;
  */  
  ELSE
    l_val := UTL_RAW.BIT_OR(to_raw(p_expr1), to_raw(p_expr2));
    RETURN TO_NUMBER(l_val, numformat(LENGTH(l_val)));
  END IF;
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END ora_bitor;

/**
 * Performs a Bit OR operation. Uses UTL_RAW package.
 * 
 * @param p_expr1
 * @param p_expr2
 * @return RAW
 */
FUNCTION ora_bitor(p_expr1 IN RAW, p_expr2 IN RAW) RETURN RAW AS
l_expr1 RAW(2000);
l_expr2 RAW(2000);
BEGIN
  IF p_expr1 IS NULL OR LENGTH(p_expr1) = 0 OR p_expr2 IS NULL OR LENGTH(p_expr2) = 0 THEN
    RETURN NULL;
  ELSE
    l_expr1 := p_expr1;
    l_expr2 := p_expr2;
    lpad_args(l_expr1, l_expr2);
    RETURN UTL_RAW.BIT_OR(l_expr1, l_expr2);
  END IF;
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END ora_bitor;

/**
 * Performs a Bit OR operation. Uses UTL_RAW package.
 * 
 * @param p_expr1
 * @param p_expr2
 * @return RAW
 */
FUNCTION ora_bitor(p_expr1 IN RAW, p_expr2 IN NUMBER) RETURN RAW AS
l_expr1 RAW(2000);
l_expr2 RAW(2000);
BEGIN
  IF p_expr1 IS NULL OR LENGTH(p_expr1) = 0 OR p_expr2 IS NULL OR LENGTH(p_expr2) = 0 THEN
    RETURN NULL;
  ELSE
    l_expr1 := p_expr1;
    l_expr2 := to_raw(p_expr2);
    lpad_args(l_expr1, l_expr2);
    RETURN UTL_RAW.BIT_OR(l_expr1, l_expr2);
  END IF;
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END ora_bitor;

/**
 * Performs a Bit OR operation. Uses UTL_RAW package.
 * 
 * @param p_expr1
 * @param p_expr2
 * @return NUMBER
 */
FUNCTION ora_bitor(p_expr1 IN NUMBER, p_expr2 IN RAW) RETURN NUMBER AS
l_expr1 RAW(2000);
l_expr2 RAW(2000);
l_val RAW(2000);
BEGIN
  IF p_expr1 IS NULL OR LENGTH(p_expr1) = 0 OR p_expr2 IS NULL OR LENGTH(p_expr2) = 0 THEN
    RETURN NULL;
  ELSE
    l_expr1 := to_raw(p_expr1);
    l_expr2 := p_expr2;
    lpad_args(l_expr1, l_expr2);
    l_val := UTL_RAW.BIT_OR(l_expr1, l_expr2);
    RETURN TO_NUMBER(l_val, numformat(LENGTH(l_val)));
  END IF;
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END ora_bitor;


/**
 * Performs a Bit XOR operation. Uses UTL_RAW package.
 * 
 * @param p_expr1
 * @param p_expr2
 * @return NUMBER
 */
FUNCTION ora_bitxor(p_expr1 IN NUMBER, p_expr2 IN NUMBER) RETURN NUMBER AS
l_val RAW(2000);
BEGIN
  IF p_expr1 IS NULL OR LENGTH(p_expr1) = 0 OR p_expr2 IS NULL OR LENGTH(p_expr2) = 0 THEN
    RETURN NULL;
  ELSE
    l_val := UTL_RAW.BIT_XOR(to_raw(p_expr1), to_raw(p_expr2));
    RETURN TO_NUMBER(l_val, numformat(LENGTH(l_val)));
  END IF;  
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END ora_bitxor;

/**
 * Performs a Bit XOR operation. Uses UTL_RAW package.
 * 
 * @param p_expr1
 * @param p_expr2
 * @return RAW
 */
FUNCTION ora_bitxor(p_expr1 IN RAW, p_expr2 IN RAW) RETURN RAW AS
BEGIN
  IF p_expr1 IS NULL OR LENGTH(p_expr1) = 0 OR p_expr2 IS NULL OR LENGTH(p_expr2) = 0 THEN
    RETURN NULL;
  ELSE
    RETURN UTL_RAW.BIT_XOR(p_expr1, p_expr2);
  END IF;  
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END ora_bitxor;

/**
 * Performs a Bit XOR operation. Uses UTL_RAW package.
 * 
 * @param p_expr1
 * @param p_expr2
 * @return RAW
 */
FUNCTION ora_bitxor(p_expr1 IN RAW, p_expr2 IN NUMBER) RETURN RAW AS
l_expr1 RAW(2000);
l_expr2 RAW(2000);
l_val RAW(2000);
BEGIN
  IF p_expr1 IS NULL OR LENGTH(p_expr1) = 0 OR p_expr2 IS NULL OR LENGTH(p_expr2) = 0 THEN
    RETURN NULL;
  ELSE
    l_expr1 := p_expr1;
    l_expr2 := to_raw(p_expr2);
    lpad_args(l_expr1, l_expr2);
    l_val := UTL_RAW.BIT_XOR(l_expr1, l_expr2);
    RETURN l_val;
  END IF;  
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END ora_bitxor;

/**
 * Performs a Bit XOR operation. Uses UTL_RAW package.
 * 
 * @param p_expr1
 * @param p_expr2
 * @return NUMBER
 */
FUNCTION ora_bitxor(p_expr1 IN NUMBER, p_expr2 IN RAW) RETURN NUMBER AS
l_expr1 RAW(2000);
l_expr2 RAW(2000);
l_val RAW(2000);
BEGIN
  IF p_expr1 IS NULL OR LENGTH(p_expr1) = 0 OR p_expr2 IS NULL OR LENGTH(p_expr2) = 0 THEN
    RETURN NULL;
  ELSE
    l_expr1 := to_raw(p_expr1);
    l_expr2 := p_expr2;
    lpad_args(l_expr1, l_expr2);
    l_val := UTL_RAW.BIT_XOR(l_expr1, l_expr2);
    RETURN TO_NUMBER(l_val, numformat(LENGTH(l_val)));
  END IF;  
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END ora_bitxor;

/**
 * Just calls the oracle BITAND function
 * 
 * @param p_expr1
 * @param p_expr2
 * @return NUMBER
 */
FUNCTION ora_bitand(p_expr1 IN NUMBER, p_expr2 IN NUMBER) RETURN NUMBER AS
BEGIN
  RETURN BITAND(p_expr1, p_expr2);
END ora_bitand;


/**
 * Performs BIT AND operation. Uses UTL_RAW package.
 * 
 * @param p_expr1
 * @param p_expr2
 * @return RAW
 */
FUNCTION ora_bitand(p_expr1 IN RAW, p_expr2 IN RAW) RETURN RAW AS
BEGIN
  IF p_expr1 IS NULL OR LENGTH(p_expr1) = 0 OR p_expr2 IS NULL OR LENGTH(p_expr2) = 0 THEN
    RETURN NULL;
  ELSE
    RETURN UTL_RAW.BIT_AND(p_expr1, p_expr2);
  END IF;  
END ora_bitand;

/**
 * Performs BIT AND operation. Uses UTL_RAW package.
 * 
 * @param p_expr1
 * @param p_expr2
 * @return RAW
 */
FUNCTION ora_bitand(p_expr1 IN RAW, p_expr2 IN NUMBER) RETURN RAW AS
l_expr1 RAW(2000);
l_expr2 RAW(2000);
l_val RAW(2000);
BEGIN
  IF p_expr1 IS NULL OR LENGTH(p_expr1) = 0 OR p_expr2 IS NULL OR LENGTH(p_expr2) = 0 THEN
    RETURN NULL;
  ELSE
    l_expr1 := p_expr1;
    l_expr2 := to_raw(p_expr2);
    lpad_args(l_expr1, l_expr2);
    l_val := UTL_RAW.BIT_AND(l_expr1, l_expr2);
    RETURN l_val;
  END IF;  
END ora_bitand;

/**
 * Performs BIT AND operation. Uses UTL_RAW package.
 * 
 * @param p_expr1
 * @param p_expr2
 * @return NUMBER
 */
FUNCTION ora_bitand(p_expr1 IN NUMBER, p_expr2 IN RAW) RETURN NUMBER AS
l_expr1 RAW(2000);
l_expr2 RAW(2000);
l_val RAW(2000);
BEGIN
  IF p_expr1 IS NULL OR LENGTH(p_expr1) = 0 OR p_expr2 IS NULL OR LENGTH(p_expr2) = 0 THEN
    RETURN NULL;
  ELSE
    l_expr1 := to_raw(p_expr1);
    l_expr2 := p_expr2;
    lpad_args(l_expr1, l_expr2);
    l_val := UTL_RAW.BIT_AND(l_expr1, l_expr2);
    RETURN TO_NUMBER(l_val, numformat(LENGTH(l_val)));
  END IF;  
END ora_bitand;

/**
 * 
 * @param p_tstz
 * @return NUMBER
 */
FUNCTION day_of_week(p_tstz IN TIMESTAMP WITH TIME ZONE) RETURN NUMBER AS
BEGIN
   RETURN TO_CHAR(p_tstz, 'D');
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END day_of_week;

/**
 * @param p_tstz
 * @param p_format
 * @return NUMBER
 */
FUNCTION day_of_week(p_tstz IN VARCHAR2, p_format VARCHAR2) RETURN NUMBER AS
BEGIN
   RETURN day_of_week(str_to_timestamptz(p_tstz, p_format));
END day_of_week;

/**
 * 
 * @param p_tstz
 * @return NUMBER
 */
FUNCTION day_of_month(p_tstz IN TIMESTAMP WITH TIME ZONE) RETURN NUMBER AS
BEGIN
   RETURN TO_CHAR(p_tstz, 'DD');
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END day_of_month;

/**
 * 
 * @param p_tstz
 * @param p_format
 * @return NUMBER
 */
FUNCTION day_of_month(p_tstz IN VARCHAR2, p_format VARCHAR2) RETURN NUMBER AS
BEGIN
   RETURN day_of_month(str_to_timestamptz(p_tstz, p_format));
END day_of_month;

/**
 * 
 * @param p_tstz
 * @return NUMBER
 */
FUNCTION day_of_year(p_tstz IN TIMESTAMP WITH TIME ZONE) RETURN NUMBER AS
BEGIN
   RETURN TO_CHAR(p_tstz, 'DDD');
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END day_of_year;

/**
 * 
 * @param p_tstz
 * @param p_format
 * @return NUMBER
 */
FUNCTION day_of_year(p_tstz IN VARCHAR2, p_format VARCHAR2) RETURN NUMBER AS
BEGIN
   RETURN day_of_year(str_to_timestamptz(p_tstz, p_format));
END day_of_year;

/**
 * 
 * @param p_tstz
 * @return NUMBER
 */
FUNCTION day_of_calendar(p_tstz IN TIMESTAMP WITH TIME ZONE) RETURN NUMBER AS
BEGIN
   -- Added +1 because Teradata returns Number of days including January 1, 1900
   RETURN EXTRACT(DAY FROM (CAST(p_tstz AS TIMESTAMP) - TO_TIMESTAMP('1900-01-01', 'yyyy-mm-dd')))+1;
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END day_of_calendar;

/**
 * 
 * @param p_tstz
 * @param p_format
 * @return NUMBER
 */
FUNCTION day_of_calendar(p_tstz IN VARCHAR2, p_format VARCHAR2) RETURN NUMBER AS
BEGIN
   RETURN day_of_calendar(str_to_timestamptz(p_tstz, p_format));
END day_of_calendar;

/**
 * 
 * @param p_tstz
 * @return NUMBER
 */
FUNCTION weekday_of_month(p_tstz IN TIMESTAMP WITH TIME ZONE) RETURN NUMBER AS
BEGIN
   RETURN TO_CHAR(p_tstz, 'W');
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);   
END weekday_of_month;

/**
 * 
 * @param p_tstz
 * @param p_format
 * @return NUMBER
 */
FUNCTION weekday_of_month(p_tstz IN VARCHAR2, p_format VARCHAR2) RETURN NUMBER AS
BEGIN
   RETURN weekday_of_month(str_to_timestamptz(p_tstz, p_format));
END weekday_of_month;

/**
 * 
 * @param p_tstz
 * @return NUMBER
 */
FUNCTION week_of_month(p_tstz IN TIMESTAMP WITH TIME ZONE) RETURN NUMBER AS
BEGIN
   -- Teradata returns only full weeks from the beginning of the month, partial weeks is taken as 0
   -- so had to substract -1
   RETURN TO_CHAR(p_tstz, 'W')-1;
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END week_of_month;

/**
 * 
 * @param p_tstz
 * @param p_format
 * @return NUMBER
 */
FUNCTION week_of_month(p_tstz IN VARCHAR2, p_format VARCHAR2) RETURN NUMBER AS
BEGIN
   RETURN week_of_month(str_to_timestamptz(p_tstz, p_format));
END week_of_month;

/**
 * 
 * @param p_tstz
 * @return NUMBER
 */
FUNCTION week_of_year(p_tstz IN TIMESTAMP WITH TIME ZONE) RETURN NUMBER AS
BEGIN
   RETURN TO_CHAR(p_tstz, 'IW');
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END week_of_year;

/**
 * 
 * @param p_tstz
 * @param p_format
 * @return NUMBER
 */
FUNCTION week_of_year(p_tstz IN VARCHAR2, p_format VARCHAR2) RETURN NUMBER AS
BEGIN
   RETURN week_of_year(str_to_timestamptz(p_tstz, p_format));
END week_of_year;

/**
 * 
 * @param p_tstz
 * @return NUMBER
 */
FUNCTION week_of_calendar(p_tstz IN TIMESTAMP WITH TIME ZONE) RETURN NUMBER AS
BEGIN
   RETURN TRUNC(day_of_calendar(p_tstz)/7);
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END week_of_calendar;

/**
 * 
 * @param p_tstz
 * @param p_format
 * @return NUMBER
 */
FUNCTION week_of_calendar(p_tstz IN VARCHAR2, p_format VARCHAR2) RETURN NUMBER AS
BEGIN
   RETURN week_of_calendar(str_to_timestamptz(p_tstz, p_format));
END week_of_calendar;

/**
 * 
 * @param p_tstz 
 * @return NUMBER
 */
FUNCTION month_of_quarter(p_tstz IN TIMESTAMP WITH TIME ZONE) RETURN NUMBER AS
l_mm NUMBER;
BEGIN
   l_mm := month_of_year(p_tstz);
   IF l_mm = 1 OR l_mm = 4 OR l_mm = 7 OR l_mm = 10 THEN
     RETURN 1;
   ELSIF l_mm = 2 OR l_mm = 5 OR l_mm = 8 OR l_mm = 11 THEN
     RETURN 2;
   ELSE
     RETURN 3;
   END IF;
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END month_of_quarter;

/**
 * 
 * @param p_tstz
 * @param p_format 
 * @return NUMBER
 */
FUNCTION month_of_quarter(p_tstz IN VARCHAR2, p_format VARCHAR2) RETURN NUMBER AS
BEGIN
   RETURN month_of_quarter(str_to_timestamptz(p_tstz, p_format));
END month_of_quarter;

/**
 * 
 * @param p_tstz
 * @return NUMBER
 */
FUNCTION month_of_year(p_tstz IN TIMESTAMP WITH TIME ZONE) RETURN NUMBER AS
BEGIN
   RETURN TO_CHAR(p_tstz, 'MM');
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END month_of_year;

/**
 * 
 * @param p_tstz
 * @return NUMBER
 */
FUNCTION month_of_year(p_tstz IN VARCHAR2, p_format VARCHAR2) RETURN NUMBER AS
BEGIN
   RETURN month_of_year(str_to_timestamptz(p_tstz, p_format));
END month_of_year;

/**
 * 
 * @param p_tstz
 * @return NUMBER
 */
FUNCTION month_of_calendar(p_tstz IN TIMESTAMP WITH TIME ZONE) RETURN NUMBER AS
BEGIN
   -- Added +1 because Teradata returns Number of months including January 1900
   RETURN TRUNC(MONTHS_BETWEEN(CAST(p_tstz AS TIMESTAMP),  TO_TIMESTAMP('01-01-1900', 'mm-dd-yyyy')))+1; 
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END month_of_calendar;

/**
 * 
 * @param p_tstz
 * @param p_format
 * @return NUMBER
 */
FUNCTION month_of_calendar(p_tstz IN VARCHAR2, p_format VARCHAR2) RETURN NUMBER AS
BEGIN
   RETURN month_of_calendar(str_to_timestamptz(p_tstz, p_format));
END month_of_calendar;

/**
 * 
 * @param p_tstz 
 * @return NUMBER
 */
FUNCTION quarter_of_calendar(p_tstz IN TIMESTAMP WITH TIME ZONE) RETURN NUMBER AS
BEGIN
    -- Added +1 because Teradata returns number of quarters first quarter of 1900
   RETURN TRUNC(MONTHS_BETWEEN(p_tstz, TO_DATE('01-01-1900', 'mm-dd-yyyy'))/3)+1;
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END quarter_of_calendar;

/**
 * 
 * @param p_tstz
 * @param p_format
 * @return NUMBER
 */
FUNCTION quarter_of_calendar(p_tstz IN VARCHAR2, p_format VARCHAR2) RETURN NUMBER AS
BEGIN
   RETURN quarter_of_calendar(str_to_timestamptz(p_tstz, p_format));
END quarter_of_calendar;

/**
 * 
 * @param p_tstz
 * @return NUMBER
 */
FUNCTION quarter_of_year(p_tstz IN TIMESTAMP WITH TIME ZONE) RETURN NUMBER AS
BEGIN
   RETURN TO_CHAR(p_tstz, 'Q');
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END quarter_of_year;

/**
 * 
 * @param p_tstz
 * @param p_format
 * @return NUMBER
 */
FUNCTION quarter_of_year(p_tstz IN VARCHAR2, p_format VARCHAR2) RETURN NUMBER AS
BEGIN
   RETURN quarter_of_year(str_to_timestamptz(p_tstz, p_format));
END quarter_of_year;

/**
 * 
 * @param p_tstz
 * @return NUMBER
 */
FUNCTION year_of_calendar(p_tstz IN TIMESTAMP WITH TIME ZONE) RETURN NUMBER AS
BEGIN
   RETURN TO_CHAR(p_tstz, 'YYYY');
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END year_of_calendar;

/**
 * 
 * @param p_tstz
 * @return NUMBER
 */
FUNCTION year_of_calendar(p_tstz IN VARCHAR2, p_format VARCHAR2) RETURN NUMBER AS
BEGIN
   RETURN year_of_calendar(str_to_timestamptz(p_tstz, p_format));
END year_of_calendar;

/**
 * 
 * @param p_tstz
 * @return NUMBER
 */
FUNCTION bit_complement(p_expr IN NUMBER) RETURN NUMBER AS
BEGIN
  -- RETURN UTL_RAW.CAST_TO_NUMBER(UTL_RAW.BIT_COMPLEMENT(UTL_RAW.CAST_FROM_NUMBER(p_expr)));
    RETURN (0 - p_expr);
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);  
END bit_complement;

/**
 * 
 * @param p_tstz
 * @return NUMBER
 */
 /*
FUNCTION bit_complement(p_expr IN RAW) RETURN RAW AS
BEGIN
  RETURN NULL;
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END bit_complement;
*/

/*
FUNCTION current_time(p_fra_prsn IN VARCHAR2) RETURN TIMESTAMP WITH TIME ZONE AS
l_pre_fmt VARCHAR2(11) := 'HH:MI:SS.FF';
l_pst_fmt VARCHAR2(3) := ' AM';
l_format VARCHAR2(20);
BEGIN
  IF p_fra_prsn IS NULL OR p_fra_prsn = '0' THEN
     l_format := 'HH:MI:SS AM';
  ELSE 
     l_format := l_pre_fmt || p_fra_prsn || l_pst_fmt;
  END IF;
  RETURN CAST(TO_CHAR(SYSTIMESTAMP, getNLSTSTZformat) AS TIMESTAMP WITH TIME ZONE);
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);  
END current_time;
*/

/*
   TODO: The p_tzstr passed is Teradata Time zone string and needs to be mapped to 
   Oracle time zone string.
   
   p_fra_prsn    -- Fractional Precision
   p_tzstr       -- Time Zone String
   p_tdwzcontrol -- This is a TimeDateWZControl flag in Teradata database the valid values or 'Y' or 'N'.
                    If the flag is 'Y' returns session time and 'N' returns server time.
*/

/*
FUNCTION CURRENT_TIME(p_fra_prsn IN VARCHAR2, p_tzstr IN VARCHAR2, p_tdwzcontrol IN VARCHAR2) RETURN TIMESTAMP WITH TIME ZONE AS
l_pre_fmt VARCHAR2(40) := 'DD-MON-RR HH:MI:SS.FF';
l_pst_fmt VARCHAR2(3) := ' AM';
l_tzhhmi VARCHAR2(10) := ' TZD';
l_format VARCHAR2(35);
l_tstz TIMESTAMP WITH TIME ZONE;
l_tzoffset VARCHAR2(10);
l_retval VARCHAR2(50);
BEGIN
  IF p_fra_prsn IS NULL OR p_fra_prsn = '0' THEN
     l_format := l_pre_fmt || l_tzhhmi;
  ELSE 
     l_format := l_pre_fmt || p_fra_prsn || l_pst_fmt;
  END IF;
  
  l_format := l_pre_fmt || p_fra_prsn || l_pst_fmt || l_tzhhmi;
  
  IF UPPER(p_tzstr) = 'LOCAL' THEN
     IF UPPER(p_tdwzcontrol) = 'Y' THEN
        l_tzoffset := TZ_OFFSET(SESSIONTIMEZONE);
        l_tstz := TO_TIMESTAMP_TZ(TO_CHAR(LOCALTIMESTAMP, l_format), l_format) AT TIME ZONE l_tzoffset;
        RETURN l_tstz;
     ELSIF UPPER(p_tdwzcontrol) = 'N' THEN
        l_tzoffset := TZ_OFFSET(DBTIMEZONE);
        l_tstz := TO_TIMESTAMP_TZ(TO_CHAR(LOCALTIMESTAMP, l_format), l_format) AT TIME ZONE l_tzoffset;
        RETURN l_tstz;
     END IF;
  -- END IF;
  ELSE  
    IF p_tzstr IS NULL OR p_tzstr = '' THEN
      l_tzoffset := TZ_OFFSET(SESSIONTIMEZONE);
      l_tstz := TO_TIMESTAMP_TZ(TO_CHAR(LOCALTIMESTAMP, l_format), l_format) AT TIME ZONE l_tzoffset;
      RETURN l_tstz;
    ELSE
      l_tzoffset := TZ_OFFSET(p_tzstr);
      l_tstz := TO_TIMESTAMP_TZ(TO_CHAR(LOCALTIMESTAMP, l_format), l_format) AT TIME ZONE l_tzoffset;
      RETURN l_tstz;
    END IF;
  END IF;
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);   
END current_time;

*/

/**
 * TODO: The p_tzstr passed is Teradata Time zone string and needs to be mapped to 
 * Oracle time zone string.
 * 
 * @param p_fra_prsn
 * @param p_tzstr
 * @return TIMESTAMP WITH TIME ZONE
 */
FUNCTION CURRENT_TIME(p_fra_prsn IN VARCHAR2, p_tzstr IN VARCHAR2) RETURN TIMESTAMP WITH TIME ZONE AS
l_pre_fmt VARCHAR2(40) := 'DD-MON-RR HH:MI:SS.FF';
l_pst_fmt VARCHAR2(3) := ' AM';
l_tzhhmi VARCHAR2(10) := ' TZD';
l_format VARCHAR2(35);
l_tstz TIMESTAMP WITH TIME ZONE;
l_tzoffset VARCHAR2(10);
l_retval VARCHAR2(50);
BEGIN
  IF p_fra_prsn IS NULL AND p_tzstr IS NULL THEN
     RETURN CURRENT_TIMESTAMP(NULL);
  END IF;   
  
  IF p_fra_prsn IS NULL OR p_fra_prsn = '0' THEN
     l_format := l_pre_fmt || l_tzhhmi;
  ELSE 
     l_format := l_pre_fmt || p_fra_prsn || l_pst_fmt;
  END IF;
  
  IF p_fra_prsn > 0 THEN
     l_format := l_pre_fmt || p_fra_prsn || l_pst_fmt || l_tzhhmi;
  ELSE
     l_format := l_pre_fmt || l_pst_fmt || l_tzhhmi;
  END IF;
  
  IF UPPER(p_tzstr) = 'LOCAL' THEN
        l_tzoffset := TZ_OFFSET(SESSIONTIMEZONE);
        l_tstz := TO_TIMESTAMP_TZ(TO_CHAR(LOCALTIMESTAMP, l_format), l_format) AT TIME ZONE l_tzoffset;
        RETURN l_tstz;
  ELSE  
    IF p_tzstr IS NULL OR p_tzstr = '' THEN
      l_tzoffset := TZ_OFFSET(SESSIONTIMEZONE);
      l_tstz := TO_TIMESTAMP_TZ(TO_CHAR(LOCALTIMESTAMP, l_format), l_format) AT TIME ZONE l_tzoffset;
      RETURN l_tstz;
    ELSE
      l_tzoffset := TZ_OFFSET(p_tzstr);
      l_tstz := TO_TIMESTAMP_TZ(TO_CHAR(LOCALTIMESTAMP, l_format), l_format) AT TIME ZONE l_tzoffset;
      RETURN l_tstz;
    END IF;
  END IF;
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);   
END current_time;

/**
 * TODO: The p_tzstr passed is a Teradata Time zone string and needs to be mapped to 
 * Oracle time zone string.
 *  
 * @param p_fra_prsn    -- Fractional Precision
 * @param p_tzstr       -- Time Zone String
 * @param p_tdwzcontrol -- This is a TimeDateWZControl flag in Teradata database the valid values or 'Y' or 'N'.
 *                   If the flag is 'Y' returns session time and 'N' returns server time.
 * @return TIMESTAMP WITH TIME ZONE
 */
FUNCTION current_timestamp(p_fra_prsn IN VARCHAR2, p_tzstr IN VARCHAR2, p_tdwzcontrol IN VARCHAR2) RETURN TIMESTAMP WITH TIME ZONE AS
l_pre_fmt varchar2(30) := 'YYYY-MM-DD HH:MI:SS.FF';
l_pst_fmt varchar2(3) := ' AM';
l_format varchar2(35);
l_tstz TIMESTAMP WITH TIME ZONE;
l_tzoffset VARCHAR2(10);
BEGIN
   IF p_fra_prsn IS NULL OR p_fra_prsn = '0' THEN
     l_format := 'YYYY-MM-DD HH:MI:SS AM';
   ELSE
     l_format := l_pre_fmt || p_fra_prsn || l_pst_fmt;
   END IF;
   
   IF UPPER(p_tzstr) = 'LOCAL' THEN
     IF UPPER(p_tdwzcontrol) = 'Y' THEN
        l_tzoffset := TZ_OFFSET(SESSIONTIMEZONE);
        l_tstz := TO_TIMESTAMP_TZ(TO_CHAR(LOCALTIMESTAMP, l_format), l_format) AT TIME ZONE l_tzoffset;
        RETURN l_tstz;
     ELSIF UPPER(p_tdwzcontrol) = 'N' THEN
        l_tzoffset := TZ_OFFSET(DBTIMEZONE);
        l_tstz := TO_TIMESTAMP_TZ(TO_CHAR(LOCALTIMESTAMP, l_format), l_format) AT TIME ZONE l_tzoffset;
        RETURN l_tstz;
     END IF;
   END IF;
   
   l_tzoffset := TZ_OFFSET(p_tzstr);
   l_tstz := TO_TIMESTAMP_TZ(TO_CHAR(LOCALTIMESTAMP, l_format), l_format) AT TIME ZONE l_tzoffset;
   RETURN l_tstz;
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK); 
END current_timestamp;

/**
 * 
 * @param p_fra_prsn
 * @param p_tzstr
 * @return TIMESTAMP WITH TIME ZONE
 */
FUNCTION CURRENT_TIMESTAMP(p_fra_prsn IN VARCHAR2, p_tzstr IN VARCHAR2) RETURN TIMESTAMP WITH TIME ZONE AS
l_pre_fmt varchar2(30) := 'YYYY-MM-DD HH:MI:SS.FF';
l_pst_fmt varchar2(3) := ' AM';
l_format varchar2(35);
l_tstz TIMESTAMP WITH TIME ZONE;
l_tzoffset VARCHAR2(10);
BEGIN
   IF p_fra_prsn IS NULL OR p_fra_prsn = '0' THEN
     l_format := 'YYYY-MM-DD HH:MI:SS AM';
   ELSE
     l_format := l_pre_fmt || p_fra_prsn || l_pst_fmt;
   END IF;
   
   IF UPPER(p_tzstr) = 'LOCAL' THEN
      l_tzoffset := TZ_OFFSET(SESSIONTIMEZONE);
      l_tstz := TO_TIMESTAMP_TZ(TO_CHAR(LOCALTIMESTAMP, l_format), l_format) AT TIME ZONE l_tzoffset;
      RETURN l_tstz;
   ELSE
      l_tzoffset := TZ_OFFSET(DBTIMEZONE);
      l_tstz := TO_TIMESTAMP_TZ(TO_CHAR(LOCALTIMESTAMP, l_format), l_format) AT TIME ZONE l_tzoffset;
      RETURN l_tstz;
   END IF;
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK); 
END current_timestamp;

/**
 * 
 * @param p_timestamp
 * @return TIMESTAMP WITH LOCAL TIME ZONE
 */
FUNCTION current_timestamp(p_timestamp TIMESTAMP WITH TIME ZONE) RETURN TIMESTAMP WITH LOCAL TIME ZONE AS
BEGIN
  return CAST(p_timestamp AS TIMESTAMP WITH LOCAL TIME ZONE);
END current_timestamp;

/**
 * 
 * @param p_expr
 * @param p_months
 * @return TIMESTAMP
 */
FUNCTION ora_add_months(p_expr IN VARCHAR2, p_months IN PLS_INTEGER) RETURN TIMESTAMP AS
BEGIN
    IF p_expr IS NULL OR p_months IS NULL THEN
        RETURN NULL;
    ELSE
        BEGIN
          RETURN ADD_MONTHS(p_expr, p_months);
        EXCEPTION
         WHEN OTHERS THEN
           -- let us try with the different Oracle datetime formats and see whether we could come up with a result. 
           FOR i in DT_FORMATS.FIRST .. DT_FORMATS.LAST
           LOOP
              BEGIN
                RETURN ADD_MONTHS(TO_TIMESTAMP_TZ(p_expr, DT_FORMATS(i)), p_months);
              EXCEPTION
               WHEN OTHERS THEN
                NULL; -- Keep Trying
              END;
           END LOOP;
        END;
        -- Finally throw an error to the application after trying all the possible formats in the loop above
        RETURN ADD_MONTHS(p_expr, p_months);
    END IF;
EXCEPTION
    WHEN OTHERS THEN
       RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK || 'The Expression should always be in a valid Oracle Datetime format for e.g.: ''YYYY-MM-DD''.');
END ora_add_months;

/**
 * 
 * @param p_expr
 * @return BINARY_INTEGER
 */
FUNCTION hex_2_integer(p_expr IN RAW) RETURN BINARY_INTEGER AS
BEGIN
  RETURN UTL_RAW.CAST_TO_BINARY_INTEGER(p_expr);
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);  
END hex_2_integer;

/**
 * 
 * @param p_expr
 * @return BINARY_INTEGER
 */
FUNCTION hex_2_char(p_expr IN RAW) RETURN VARCHAR2 AS
l_value VARCHAR2(160);
BEGIN
   IF p_expr IS NOT NULL OR LENGTH(p_expr) > 0 THEN
     select value into l_value from nls_database_parameters where parameter = 'NLS_CHARACTERSET';
     IF l_value IS NOT NULL OR LENGTH(l_value) > 0 THEN
        RETURN UTL_I18N.RAW_TO_CHAR(p_expr, l_value);
     ELSE
        RETURN NULL;
     END IF;
   ELSE
     RETURN NULL;
   END IF;  
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);    
END hex_2_char;

/**
 * 
 * @param p_str
 * @param p_fmtstr
 * @return DATE
 */
FUNCTION cast_to_date(p_str IN VARCHAR2, p_fmtstr IN VARCHAR2) RETURN DATE AS
l_fmtstr VARCHAR2(30);
BEGIN
-- TODO: need a function to convert Teradata format string to Oracle format.
  l_fmtstr := p_fmtstr;
  IF l_fmtstr IS NULL THEN
   l_fmtstr := getNLSDATEformat;
  END IF;
  RETURN TO_DATE(p_str, l_fmtstr);
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END cast_to_date;

/**
 * 
 * @param p_ts
 * @param p_fmtstr
 * @return DATE
 */        		   
FUNCTION cast_to_date(p_ts IN TIMESTAMP, p_fmtstr IN VARCHAR2) RETURN DATE AS
l_fmtstr VARCHAR2(30);
l_ts TIMESTAMP;
BEGIN
-- TODO: need a function to convert Teradata date format string to Oracle date format.
  l_fmtstr := p_fmtstr;
  l_ts := TO_TIMESTAMP(TO_CHAR(p_ts, l_fmtstr), l_fmtstr);
  RETURN CAST(l_ts AS DATE);
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END cast_to_date;

/**
 * 
 * @param p_num
 * @param p_fmtstr
 * @return DATE
 */        	 
FUNCTION cast_to_date(p_num IN NUMBER, p_fmtstr IN VARCHAR2) RETURN DATE AS
l_fmtstr VARCHAR2(30);
BEGIN
-- TODO: need a function to convert Teradata date format string to Oracle date format.
  l_fmtstr := p_fmtstr;
  RETURN numeric_to_date(p_num, l_fmtstr);
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END cast_to_date;

/**
 * 
 * @param p_str
 * @param p_fmtstr
 * @param p_tzstr
 * @return TIMESTAMP WITH TIME ZONE
 */
FUNCTION cast_to_timestamp_tz(p_str IN VARCHAR2, p_fmtstr IN VARCHAR2, p_tzstr IN VARCHAR2) RETURN TIMESTAMP WITH TIME ZONE AS
l_fmtstr VARCHAR2(50);
l_tzstr VARCHAR2(50);
l_tzoffset VARCHAR2(10);
BEGIN
-- TODO: need a function to convert Teradata format string to Oracle format.
    l_fmtstr := p_fmtstr;
-- TODO: need a function to convert Teradata time zone to Oracle time zone.
    l_tzstr := p_tzstr;
    
    IF p_tzstr IS NULL OR LENGTH(p_tzstr) = 0 THEN
       l_tzstr := SESSIONTIMEZONE;
    END IF;
    
    l_tzoffset := TZ_OFFSET(l_tzstr);
    RETURN TO_TIMESTAMP_TZ(p_str, l_fmtstr) AT TIME ZONE l_tzoffset; 
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END cast_to_timestamp_tz;

FUNCTION cast_to_time_tz(p_expr IN VARCHAR2, p_precision IN BINARY_INTEGER, 
           p_fmtstr IN VARCHAR2) RETURN TIMESTAMP WITH TIME ZONE AS
BEGIN
  RETURN cast_to_timestamp_tz(p_expr, p_precision, p_fmtstr);
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END cast_to_time_tz;

/**
 * 
 * @param p_ts
 * @param p_fmtstr
 * @param p_tzstr
 * @return TIMESTAMP WITH TIME ZONE
 */
FUNCTION cast_to_timestamp_tz(p_ts IN TIMESTAMP, p_fmtstr IN VARCHAR2, p_tzstr IN VARCHAR2) RETURN TIMESTAMP WITH TIME ZONE AS
l_fmtstr VARCHAR2(50);
l_tzstr VARCHAR2(50);
l_tzoffset VARCHAR2(10);
BEGIN
-- TODO: need a function to convert Teradata format string to Oracle format.
    l_fmtstr := p_fmtstr;
-- TODO: need a function to convert Teradata time zone to Oracle time zone.
    l_tzstr := p_tzstr;
    
    RETURN cast_to_timestamp_tz(TO_CHAR(p_ts, l_fmtstr), p_fmtstr, p_tzstr);
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END cast_to_timestamp_tz;

/**
 * 
 * @param p_str
 * @return INTERVAL YEAR TO MONTH
 */
FUNCTION cast_to_interval_year_to_month(p_str IN VARCHAR2) RETURN INTERVAL YEAR TO MONTH AS
BEGIN
  RETURN TO_YMINTERVAL(p_str);
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END cast_to_interval_year_to_month;

/**
 * 
 * @param p_num
 * @param p_component
 * @return INTERVAL YEAR TO MONTH
 */        		   
FUNCTION cast_to_interval_year_to_month(p_num IN NUMBER, p_component IN VARCHAR2) RETURN INTERVAL YEAR TO MONTH AS
BEGIN
   RETURN NUMTOYMINTERVAL(p_num, p_component);
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);   
END cast_to_interval_year_to_month;

/**
 * 
 * @param p_str
 * @return INTERVAL DAY TO SECOND
 */        		   
FUNCTION cast_to_interval_day_to_second(p_str IN VARCHAR2) RETURN INTERVAL DAY TO SECOND AS
BEGIN
  RETURN TO_DSINTERVAL(p_str);
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);  
END cast_to_interval_day_to_second;

/**
 * 
 * @param p_num
 * @param p_component
 * @return INTERVAL DAY TO SECOND
 */        		   
FUNCTION cast_to_interval_day_to_second(p_num IN NUMBER, p_component IN VARCHAR2) 
            RETURN INTERVAL DAY TO SECOND AS
BEGIN
  RETURN NUMTODSINTERVAL(p_num, p_component);
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END cast_to_interval_day_to_second;


/**
 * 
 * @param p_expr
 * @param p_fmtstr
 * @return VARCHAR2
 */        		   
FUNCTION cast_to_char(p_expr IN TIMESTAMP, p_fmtstr IN VARCHAR2) RETURN VARCHAR2 AS
l_fmtstr VARCHAR2(50);
BEGIN
-- TODO: need a function to convert Teradata format string to Oracle format.
 l_fmtstr := p_fmtstr;
 RETURN TO_CHAR(p_expr, l_fmtstr);
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);  
END cast_to_char;

/**
 * 
 * @param p_num
 * @param p_fmtstr
 * @param p_chrset
 * @return VARCHAR2
 */        		   
FUNCTION cast_to_char(p_num IN NUMBER, p_fmtstr IN VARCHAR2, p_chrset IN VARCHAR2) RETURN VARCHAR2 AS
l_fmtstr VARCHAR2(50);
BEGIN
-- TODO: need a function to convert Teradata format string to Oracle format.
 l_fmtstr := p_fmtstr;
   RETURN CONVERT(TO_CHAR(p_num, l_fmtstr), p_chrset);
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);  
END cast_to_char;

/**
 * 
 * @param p_idtos
 * @param p_fmtstr
 * @return VARCHAR2
 */        		   
FUNCTION cast_to_char(p_idtos IN INTERVAL DAY TO SECOND, p_fmtstr IN VARCHAR2) RETURN VARCHAR2 AS
BEGIN
  RETURN TO_CHAR(p_idtos);
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END cast_to_char;

/**
 * 
 * @param p_iytom
 * @param p_fmtstr
 * @return VARCHAR2
 */        		   
FUNCTION cast_to_char(p_iytom IN INTERVAL YEAR TO MONTH, p_fmtstr IN VARCHAR2) RETURN VARCHAR2 AS
BEGIN
  RETURN TO_CHAR(p_iytom);
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END cast_to_char;

/**
 * 
 * @param p_num
 * @param p_fmtstr
 * @return NUMBER
 */        		   
FUNCTION cast_to_numeric(p_num IN NUMBER, p_fmtstr IN VARCHAR2) RETURN NUMBER AS
l_fmtstr VARCHAR2(50);
BEGIN
  -- TODO: need a function to convert Teradata format string to Oracle format.
  l_fmtstr := p_fmtstr;
  RETURN CAST(TO_CHAR(p_num, l_fmtstr) AS NUMBER);
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END cast_to_numeric;

/**
 * 
 * @param p_str
 * @param p_fmtstr
 * @return NUMBER
 */        		   
FUNCTION cast_to_numeric(p_str IN VARCHAR2, p_fmtstr IN VARCHAR2) return NUMBER AS
l_fmtstr VARCHAR2(50);
BEGIN
  -- TODO: need a function to convert Teradata format string to Oracle format.
  l_fmtstr := p_fmtstr;
  -- RETURN TO_NUMBER(p_str, l_fmtstr);
  RETURN NULL;
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END cast_to_numeric;

/**
 * 
 * @param p_date
 * @param p_fmtstr
 * @return NUMBER
 */        		   
FUNCTION cast_to_numeric(p_date IN DATE, p_fmtstr IN VARCHAR2) RETURN NUMBER AS
l_val NUMBER;
BEGIN
  l_val := date_to_numeric(p_date);
  RETURN cast_to_numeric(l_val, p_fmtstr);
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END cast_to_numeric;

/**
 * 
 * @param p_iytom
 * @param p_fmtstr
 * @param p_component
 * @return NUMBER
 */        		   
FUNCTION cast_to_numeric(p_iytom IN INTERVAL YEAR TO MONTH, p_fmtstr IN VARCHAR2, p_component IN VARCHAR2) RETURN NUMBER AS
l_retval NUMBER;
BEGIN
  IF UPPER(p_component) = 'YEAR' THEN
     l_retval := cast_to_numeric(EXTRACT(YEAR FROM p_iytom), p_fmtstr);
  ELSIF UPPER(p_component) = 'MONTH' THEN
     l_retval := cast_to_numeric(EXTRACT(MONTH FROM p_iytom), p_fmtstr);
  ELSE
     l_retval := 0;
  END IF;
  RETURN l_retval;
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END cast_to_numeric;

/**
 * 
 * @param p_idtos
 * @param p_fmtstr
 * @param p_component
 * @return NUMBER
 */        		   
FUNCTION cast_to_numeric(p_idtos IN INTERVAL DAY TO SECOND, p_fmtstr IN VARCHAR2, p_component IN VARCHAR2) RETURN NUMBER AS
l_retval NUMBER;
BEGIN
  IF UPPER(p_component) = 'DAY' THEN
     l_retval := cast_to_numeric(EXTRACT(DAY FROM p_idtos), p_fmtstr);
  ELSIF UPPER(p_component) = 'HOUR' THEN
     l_retval := cast_to_numeric(EXTRACT(HOUR FROM p_idtos), p_fmtstr);
  ELSIF UPPER(p_component) = 'MINUTE' THEN
     l_retval := cast_to_numeric(EXTRACT(MINUTE FROM p_idtos), p_fmtstr);
  ELSIF UPPER(p_component) = 'SECOND' THEN
     l_retval := cast_to_numeric(EXTRACT(SECOND FROM p_idtos), p_fmtstr);
  ELSE
     l_retval := 0;
  END IF;   
  RETURN l_retval;
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END cast_to_numeric;

/**
 * 
 * @param 
 * @return 
 */        		   
FUNCTION cast_to_number(p_expr IN VARCHAR2, p_precision IN BINARY_INTEGER, p_scale IN BINARY_INTEGER, 
           p_fmtstr IN VARCHAR2, p_chrset IN VARCHAR2) RETURN NUMBER AS
l_fmtstr VARCHAR2(50);
l_retval NUMBER;
BEGIN
  -- TODO: need a function to convert Teradata format string to Oracle format.
  l_fmtstr := p_fmtstr;
  IF l_fmtstr IS NULL OR l_fmtstr = '' THEN
     l_retval := TO_NUMBER(p_expr);
  ELSE   
     l_fmtstr := LPAD(9, LENGTH(p_expr), 9);
     l_retval := TO_NUMBER(p_expr, l_fmtstr);
  END IF;
  RETURN l_retval;
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END cast_to_number;

/**
 * 
 * @param 
 * @return 
 */        		   
FUNCTION cast_to_number(p_expr IN DATE, p_precision IN BINARY_INTEGER, p_scale IN BINARY_INTEGER, 
           p_fmtstr IN VARCHAR2, p_chrset IN VARCHAR2) RETURN NUMBER AS
l_val VARCHAR2(50);
BEGIN
  l_val := CAST(date_to_numeric(p_expr) AS VARCHAR2);
  RETURN cast_to_number(l_val, p_precision, p_scale, p_fmtstr, p_chrset);
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END cast_to_number;

/**
 * 
 * @param 
 * @return 
 */        		   
FUNCTION cast_to_number(p_expr IN INTERVAL YEAR TO MONTH, p_precision IN BINARY_INTEGER, p_scale IN BINARY_INTEGER, 
           p_fmtstr IN VARCHAR2, p_chrset IN VARCHAR2) RETURN NUMBER AS
l_retval NUMBER;
BEGIN
  l_retval := EXTRACT(YEAR FROM p_expr);
  IF l_retval = 0 THEN
    l_retval := EXTRACT(MONTH FROM p_expr);
  END IF;
  RETURN cast_to_number(CAST(l_retval AS VARCHAR2), p_precision, p_scale, p_fmtstr, p_chrset);
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END cast_to_number;

/**
 * 
 * @param 
 * @return 
 */        		   
FUNCTION cast_to_number(p_expr IN INTERVAL DAY TO SECOND, p_precision IN BINARY_INTEGER, p_scale IN BINARY_INTEGER, 
           p_fmtstr IN VARCHAR2, p_chrset IN VARCHAR2) RETURN NUMBER AS
l_retval NUMBER := 0;
BEGIN
   l_retval := EXTRACT(DAY FROM p_expr);
   IF l_retval = 0 THEN
      l_retval := EXTRACT(HOUR FROM p_expr);
   END IF;
   
   IF l_retval = 0 THEN
      l_retval := EXTRACT(MINUTE FROM p_expr);
   END IF;
   
   IF l_retval = 0 THEN
      l_retval := EXTRACT(SECOND FROM p_expr);
   END IF;
   
   RETURN cast_to_number(CAST(l_retval AS VARCHAR2), p_precision, p_scale, p_fmtstr, p_chrset);
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END cast_to_number;


/**
 * 
 * @param 
 * @return 
 */        		   
FUNCTION cast_to_date(p_expr IN VARCHAR2, p_precision IN BINARY_INTEGER, 
           p_fmtstr IN VARCHAR2) RETURN DATE AS
--l_dt_ptrn1 VARCHAR2(50) := '\d{4}[[:punct:]]\d{2}[[:punct:]]\d{2}';
--l_dt_ptrn2 VARCHAR2(50) := '\d{2}[[:punct:]]\d{4}[[:punct:]]\d{2}';
--l_dt_ptrn3 VARCHAR2(50) := '\d{2}[[:punct:]]\d{2}[[:punct:]]\d{4}';
-- l_idx NUMBER;
l_fmtstr VARCHAR2(50);
fmtstr_is_null EXCEPTION;

BEGIN
   -- WIP need to check on values on day and month.
   /*
   IF p_fmtstr IS NULL THEN
      IF REGEXP_INSTR(p_expr, l_dt_ptrn1, 1, 1) > 0 THEN
         RETURN TO_DATE(p_expr, 'YYYY-MM-DD');
      ELSIF REGEXP_INSTR(p_expr, l_dt_ptrn2, 1, 1) > 0 THEN
         RETURN TO_DATE(p_expr, 'MM-YYYY-DD');  
      ELSIF REGEXP_INSTR(p_expr, l_dt_ptrn3, 1, 1) > 0 THEN
         RETURN TO_DATE(p_expr, 'MM-DD-YYYY');  
      END IF;    
   END IF;
   */
   
   IF p_fmtstr IS NULL THEN
         RAISE fmtstr_is_null;
   ELSE
      l_fmtstr := getOracleDateTimeFormat(p_fmtstr, p_precision);
      RETURN TO_DATE(p_expr, l_fmtstr);
   END IF;
   
EXCEPTION
    WHEN fmtstr_is_null THEN
      RAISE_APPLICATION_ERROR(-20001, 'Datetime format string cannot be NULL.');
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_BACKTRACE);
END cast_to_date;

/**
 * 
 * @param 
 * @return 
 */        		   
FUNCTION cast_to_timestamp(p_expr IN VARCHAR2, p_precision IN BINARY_INTEGER, 
           p_fmtstr IN VARCHAR2) RETURN TIMESTAMP AS
l_format VARCHAR2(35);
l_utcval NUMBER;
l_hour NUMBER;
l_idx NUMBER;
l_time VARCHAR2(10);
l_pretztimeptrn VARCHAR2(100)  := '[+-]\d{2}[[:punct:]]\d{2}\d{2}[[:punct:]]\d{2}[[:punct:]]\d{2}'; -- for prefixed timezone string
l_posttztimeptrn VARCHAR2(100) := '\d{2}[[:punct:]]\d{2}[[:punct:]]\d{2}[+-]\d{2}[[:punct:]]\d{2}'; -- for suffixed timezone string 
l_tzstr VARCHAR2(6);
l_intvlstr VARCHAR2(30);
l_expr VARCHAR2(100) := NULL;
fmtstr_is_null EXCEPTION;
BEGIN
   IF p_fmtstr IS NOT NULL THEN
      -- l_format := p_fmtstr || p_precision;
      IF p_precision > 0 AND p_precision <= 9 THEN
        l_format := getOracleDateTimeFormat(p_fmtstr, p_precision);
        IF REGEXP_INSTR(l_format, 'FF', 1, 1, 0, 'i') = 0 THEN
           l_format := l_format || '.FF' || p_precision;
        END IF;
      ELSE
        l_format := getOracleDateTimeFormat(p_fmtstr, p_precision);
      END IF;
      
      l_idx := INSTR(l_format, 'TZH', 1, 1);
      IF l_idx > 0 THEN
        l_format := SUBSTR(l_format, 1, l_idx-1);
        l_expr := SUBSTR(p_expr, 1, l_idx-2); -- takes + or - sign into consideration
      ELSE
        l_idx := INSTR(l_format, 'TZM', 1, 1);
        IF l_idx > 0 THEN
          l_format := SUBSTR(l_format, 1, l_idx-1);
          l_expr := SUBSTR(p_expr, 1, l_idx-1);
        END IF;  
      END IF;
   END IF;
   
   IF p_fmtstr IS NULL OR l_format IS NULL THEN
         RAISE fmtstr_is_null;
   END IF;
   
   l_idx := REGEXP_INSTR(p_expr, l_pretztimeptrn, 1, 1);
   IF l_idx > 0 THEN
       l_tzstr := SUBSTR(p_expr, REGEXP_INSTR(p_expr, '[+-]', l_idx, 1), 6);
       l_intvlstr := '0 ' || SUBSTR(l_tzstr, 2, 6) || ':00';
       --l_expr := SUBSTR(p_expr, 1, l_idx-1); -- takes + or - sign into consideration
       IF INSTR(l_tzstr, '+', 1, 1) > 0 THEN
         IF p_fmtstr IS NULL THEN 
           RETURN TO_TIMESTAMP(SUBSTR(p_expr, LENGTH(l_tzstr)+1, LENGTH(p_expr)), 'HH24:MI:SS') - TO_DSINTERVAL(l_intvlstr);
         ELSE
           -- RETURN TO_TIMESTAMP(SUBSTR(p_expr, LENGTH(l_tzstr)+1, LENGTH(p_expr)), l_format) - TO_DSINTERVAL(l_intvlstr);
           IF l_expr IS NOT NULL THEN
             RETURN TO_TIMESTAMP(l_expr, l_format) - TO_DSINTERVAL(l_intvlstr);
           END IF;
         END IF;
       ELSE
         IF p_fmtstr IS NULL THEN 
           RETURN TO_TIMESTAMP(SUBSTR(p_expr, LENGTH(l_tzstr)+1, LENGTH(p_expr)), 'HH24:MI:SS') + TO_DSINTERVAL(l_intvlstr);
         ELSE
           -- RETURN TO_TIMESTAMP(SUBSTR(p_expr, LENGTH(l_tzstr)+1, LENGTH(p_expr)), l_format) + TO_DSINTERVAL(l_intvlstr);
           IF l_expr IS NOT NULL THEN
              RETURN TO_TIMESTAMP(l_expr, l_format) + TO_DSINTERVAL(l_intvlstr);
           END IF;
         END IF;
       END IF;   
   END IF;
   
   l_idx := REGEXP_INSTR(p_expr, l_posttztimeptrn, 1, 1);
   IF l_idx > 0 THEN
       l_tzstr := SUBSTR(p_expr, REGEXP_INSTR(p_expr, '[+-]', l_idx, 1), 6);
       l_intvlstr := '0 ' || SUBSTR(l_tzstr, 2, LENGTH(l_tzstr)) || ':00';
       l_expr := REPLACE(p_expr, l_tzstr, '');
       IF INSTR(l_tzstr, '+', 1, 1) > 0 THEN
           l_idx := INSTR(p_expr, '+', 1, 1);
           IF l_expr IS NOT NULL THEN 
             RETURN TO_TIMESTAMP(l_expr, l_format) - TO_DSINTERVAL(l_intvlstr);
           END IF;
       ELSE
           l_idx := INSTR(p_expr, '-', 1, 1);
           IF p_fmtstr IS NULL THEN
              RETURN TO_TIMESTAMP(SUBSTR(p_expr, 1, l_idx-1), 'HH24:MI:SS') + TO_DSINTERVAL(l_intvlstr);
           ELSE
              IF l_expr IS NOT NULL THEN
                 RETURN TO_TIMESTAMP(l_expr, l_format) + TO_DSINTERVAL(l_intvlstr);
              END IF;
           END IF;   
       END IF;   
   END IF;
   
   RETURN TO_TIMESTAMP(p_expr, l_format);
EXCEPTION
    WHEN fmtstr_is_null THEN
      RAISE_APPLICATION_ERROR(-20001, 'Datetime format string cannot be NULL.');
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_BACKTRACE);
END cast_to_timestamp;

/**
 * 
 * @param 
 * @return 
 */        		   
FUNCTION cast_to_time(p_expr IN VARCHAR2, p_precision IN BINARY_INTEGER, 
           p_fmtstr IN VARCHAR2) RETURN TIMESTAMP AS
BEGIN
 RETURN cast_to_timestamp(p_expr, p_precision, p_fmtstr);
END cast_to_time;

/**
 * 
 * @param 
 * @return 
 */        		   
FUNCTION cast_to_timestamp_tz(p_expr IN VARCHAR2, p_precision IN BINARY_INTEGER, 
           p_fmtstr IN VARCHAR2) RETURN TIMESTAMP WITH TIME ZONE AS
l_format VARCHAR2(35);
l_val VARCHAR2(100);
fmtstr_is_null EXCEPTION;
l_idx PLS_INTEGER;
l_hh PLS_INTEGER;
BEGIN
   -- TODO: need a function to convert Teradata format string to Oracle format.
   IF p_fmtstr IS NULL THEN
     RAISE fmtstr_is_null;
   ELSE
     l_format := getOracleDateTimeFormat(p_fmtstr, p_precision);
     l_idx := INSTR(l_format, 'HH', 1, 1);
     IF l_idx > 0 THEN
        l_hh := SUBSTR(p_expr, l_idx, 2);
        IF l_hh > 12 THEN
           l_format := REPLACE(l_format, 'HH', 'HH24');
        END IF;
     END IF;
   END IF;  
   RETURN TO_TIMESTAMP_TZ(p_expr, l_format);
EXCEPTION
    WHEN fmtstr_is_null THEN
      RAISE_APPLICATION_ERROR(-20001, 'Datetime format string cannot be NULL.');
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_BACKTRACE);
END cast_to_timestamp_tz;

/*
FUNCTION cast_to_timestamp(p_expr IN VARCHAR2, p_precision IN BINARY_INTEGER, 
           p_fmtstr IN VARCHAR2) RETURN TIMESTAMP AS
l_pre_fmt varchar2(30) := 'YYYY-MM-DD HH:MI:SS';
l_pst_fmt VARCHAR2(3) := ' AM';
l_format varchar2(35);
BEGIN
   IF p_precision IS NULL OR p_precision = '0' THEN
     l_format := p_fmtstr;
   ELSE
     l_format := l_pre_fmt || p_precision || l_pst_fmt;
   END IF;
   -- TO_CHAR(p_expr, l_format);
   
   IF l_format IS NULL OR LENGTH(l_format) = 0 THEN
      l_format := 'YYYY-MM-DD HH:MI:SS.FF';
   END IF;
   
   RETURN TO_TIMESTAMP(p_expr, l_format);
   
END cast_to_timestamp;
*/

/**
 * 
 * @param 
 * @return 
 */        		   
FUNCTION cast_to_interval_year_to_month(p_expr IN VARCHAR2, p_precision IN BINARY_INTEGER,
           p_component IN VARCHAR2) RETURN YMINTERVAL_UNCONSTRAINED AS
l_iytom YMINTERVAL_UNCONSTRAINED;
l_fmt VARCHAR2(20);
l_years NUMBER := 0;
l_months NUMBER := 0;
BEGIN
   IF REGEXP_LIKE(p_expr, '^(|-)\d+[[:punct:]]\d+$') THEN
     IF SUBSTR(p_expr, 1, 1) = '-' THEN
       l_years := SUBSTR(p_expr, 1, INSTR(p_expr, '-', 2, 1)-1);
       l_months := SUBSTR(p_expr, INSTR(p_expr, '-', 2, 1)+1, LENGTH(p_expr));
       l_months := 0 - l_months;
     ELSE
       l_years := SUBSTR(p_expr, 1, INSTR(p_expr, '-', 1, 1)-1);
       l_months := SUBSTR(p_expr, INSTR(p_expr, '-', 1, 1)+1, LENGTH(p_expr));
     END IF;
   ELSIF REGEXP_LIKE(p_expr, '^\d+$') THEN
       IF UPPER(p_component) = 'MONTH' THEN
         l_months := p_expr;
       ELSE
         l_years := p_expr;
       END IF;
   END IF;
   
   IF UPPER(p_component) = 'YEAR' THEN
       IF ABS(l_months) > 11 THEN
          l_iytom := NUMTOYMINTERVAL(l_years, p_component) + NUMTOYMINTERVAL(TRUNC(l_months/12), p_component);
       ELSE
          l_iytom := NUMTOYMINTERVAL(l_years, p_component);
       END IF;    
       
       IF p_precision IS NOT NULL THEN
         l_fmt := LPAD('Y', p_precision, 'Y');
         l_iytom := TO_YMINTERVAL(TO_CHAR(l_iytom, l_fmt));
         -- l_iytom := TO_YMINTERVAL(l_iytom);
       END IF;  
     ELSIF UPPER(p_component) = 'MONTH' THEN
       l_iytom := NUMTOYMINTERVAL(l_years * 12, p_component) + NUMTOYMINTERVAL(l_months, p_component);
       IF p_precision IS NOT NULL THEN
         l_iytom := TO_YMINTERVAL(TO_CHAR(l_iytom, 'MM'));
         -- l_iytom := TO_YMINTERVAL(l_iytom);
       END IF; 
     ELSE
       l_iytom := NUMTOYMINTERVAL(l_years, 'YEAR') + NUMTOYMINTERVAL(l_months, 'MONTH');
   END IF;
   
   RETURN l_iytom;
EXCEPTION
    WHEN OTHERS THEN
      raise_application_error(-20000, dbms_utility.format_error_stack);
END cast_to_interval_year_to_month;

/**
 * 
 * @param 
 * @return 
 */        		   
FUNCTION cast_to_interval_day_to_second(p_expr IN VARCHAR2, p_precision IN BINARY_INTEGER,
           p_component IN VARCHAR2) RETURN DSINTERVAL_UNCONSTRAINED AS
l_days NUMBER;
l_hours NUMBER;
l_minutes NUMBER;
l_seconds NUMBER;
BEGIN

   IF REGEXP_LIKE(p_expr, '^(|-)\d+\s+\d+[[:punct:]]\d+[[:punct:]]\d+[[:punct:]]\d+$') OR 
      REGEXP_LIKE(p_expr, '^(|-)\d+\s+\d+[[:punct:]]\d+[[:punct:]]\d+$') THEN
      -- DBMS_OUTPUT.PUT_LINE('P_EXPR ' || p_expr); 
      l_days := SUBSTR(p_expr, 1, REGEXP_INSTR(p_expr, '\s+', 1, 1)-1);
      -- DBMS_OUTPUT.PUT_LINE('DAYS ' || l_days); 
      l_hours := SUBSTR(p_expr, REGEXP_INSTR(p_expr, '\s+', 1, 1)+1, REGEXP_INSTR(p_expr, ':', 1, 1)-1 - REGEXP_INSTR(p_expr, '\s+', 1, 1)); 
      -- DBMS_OUTPUT.PUT_LINE('HOURS ' || l_hours); 
      l_minutes := SUBSTR(p_expr, REGEXP_INSTR(p_expr, ':', 1, 1)+1, REGEXP_INSTR(p_expr, ':', 1, 2)-1 - REGEXP_INSTR(p_expr, ':', 1, 1));
      -- DBMS_OUTPUT.PUT_LINE('MINUTES ' || l_minutes); 
      l_seconds := SUBSTR(p_expr, REGEXP_INSTR(p_expr, ':', 1, 2)+1, LENGTH(p_expr));
      -- DBMS_OUTPUT.PUT_LINE('SECONDS ' || l_seconds); 
      IF SUBSTR(p_expr, 1, 1) = '-' THEN
         -- DBMS_OUTPUT.PUT_LINE('STARTS WITH -VE VALUE'); 
         l_hours := 0 - l_hours;
         -- DBMS_OUTPUT.PUT_LINE('HOURS ' || l_hours);
         l_minutes := 0 - l_minutes;
         l_seconds := 0 - l_seconds;
      END IF;
      
      IF UPPER(p_component) = 'DAY' THEN
        IF ABS(l_hours) > 23 THEN
          RETURN NUMTODSINTERVAL(l_days, 'DAY') + NUMTODSINTERVAL(TRUNC(l_hours/24), 'DAY');
        ELSE
          RETURN NUMTODSINTERVAL(l_days, 'DAY');
        END IF;
      ELSIF UPPER(p_component) = 'HOUR' THEN
        IF ABS(l_minutes) > 59 THEN
          RETURN NUMTODSINTERVAL(l_days * 24, 'HOUR') + NUMTODSINTERVAL(l_hours, 'HOUR') + NUMTODSINTERVAL(TRUNC(l_minutes/60), 'HOUR');
        ELSE
          RETURN NUMTODSINTERVAL(l_days * 24, 'HOUR') + NUMTODSINTERVAL(l_hours, 'HOUR');
        END IF;
      ELSIF UPPER(p_component) = 'MINUTE' THEN
        IF ABS(l_seconds) > 59 THEN
          RETURN NUMTODSINTERVAL(l_days * 24 * 60, 'MINUTE') + NUMTODSINTERVAL(l_hours * 60, 'MINUTE') + NUMTODSINTERVAL(l_minutes, 'MINUTE') + NUMTODSINTERVAL(TRUNC(l_seconds/60), 'MINUTE'); 
        ELSE
          RETURN NUMTODSINTERVAL(l_days * 24 * 60, 'MINUTE') + NUMTODSINTERVAL(l_hours * 60, 'MINUTE') + NUMTODSINTERVAL(l_minutes, 'MINUTE');
        END IF;
      ELSIF UPPER(p_component) = 'SECOND' THEN
          RETURN NUMTODSINTERVAL(l_days * 24 * 60 * 60, 'SECOND') + NUMTODSINTERVAL(l_hours * 60 * 60, 'SECOND') + NUMTODSINTERVAL(l_minutes * 60, 'SECOND') + NUMTODSINTERVAL(l_seconds, 'SECOND');
      END IF;
   ELSIF REGEXP_LIKE(p_expr,'^(|-)\d+[[:punct:]]\d+[[:punct:]]\d+[[:punct:]]\d+$') OR 
         REGEXP_LIKE(p_expr,'^(|-)\d+[[:punct:]]\d+[[:punct:]]\d+$') THEN
      l_hours := SUBSTR(p_expr, 1, REGEXP_INSTR(p_expr, ':', 1, 1)-1); 
      -- DBMS_OUTPUT.PUT_LINE('HOURS ' || l_hours); 
      l_minutes := SUBSTR(p_expr, REGEXP_INSTR(p_expr, ':', 1, 1)+1, REGEXP_INSTR(p_expr, ':', 1, 2)-1 -  REGEXP_INSTR(p_expr, ':', 1, 1));
      -- DBMS_OUTPUT.PUT_LINE('MINUTES ' || l_minutes); 
      l_seconds := SUBSTR(p_expr, REGEXP_INSTR(p_expr, ':', 1, 2)+1, LENGTH(p_expr));
      -- DBMS_OUTPUT.PUT_LINE('SECONDS ' || l_seconds);
      IF SUBSTR(p_expr, 1, 1) = '-' THEN
         l_minutes := 0 - l_minutes;
         l_seconds := 0 - l_seconds;
      END IF;
      
      IF UPPER(p_component) = 'DAY' THEN
          RETURN NUMTODSINTERVAL(TRUNC(l_hours/24), 'DAY');
      ELSIF UPPER(p_component) = 'HOUR' THEN
        IF ABS(l_minutes) > 59 THEN
          RETURN NUMTODSINTERVAL(l_hours, 'HOUR') + NUMTODSINTERVAL(TRUNC(l_minutes/60), 'HOUR');
        ELSE
          RETURN NUMTODSINTERVAL(l_hours, 'HOUR');
        END IF;
      ELSIF UPPER(p_component) = 'MINUTE' THEN
        IF ABS(l_seconds) > 59 THEN
          RETURN NUMTODSINTERVAL(l_hours * 60, 'MINUTE') + NUMTODSINTERVAL(l_minutes, 'MINUTE') + NUMTODSINTERVAL(TRUNC(l_seconds/60), 'MINUTE'); 
        ELSE
          RETURN NUMTODSINTERVAL(l_hours * 60, 'MINUTE') + NUMTODSINTERVAL(l_minutes, 'MINUTE');
        END IF;
      ELSIF UPPER(p_component) = 'SECOND' THEN
          RETURN NUMTODSINTERVAL(l_hours * 60 * 60, 'SECOND') + NUMTODSINTERVAL(l_minutes * 60, 'SECOND') + NUMTODSINTERVAL(l_seconds, 'SECOND');
      END IF;
   ELSIF REGEXP_LIKE(p_expr,'^(|-)\d+[[:punct:]]\d+[[:punct:]]\d+$') OR 
         REGEXP_LIKE(p_expr,'^(|-)\d+[[:punct:]]\d+$') THEN
      l_minutes := SUBSTR(p_expr, 1, REGEXP_INSTR(p_expr, ':', 1, 1)-1);
      -- DBMS_OUTPUT.PUT_LINE('MINUTES ' || l_minutes); 
      l_seconds := SUBSTR(p_expr, REGEXP_INSTR(p_expr, ':', 1, 1)+1, LENGTH(p_expr));
      -- DBMS_OUTPUT.PUT_LINE('SECONDS ' || l_seconds);
      IF SUBSTR(p_expr, 1, 1) = '-' THEN
         l_seconds := 0 - l_seconds;
      END IF;
      
      IF UPPER(p_component) = 'DAY' THEN
         RETURN NUMTODSINTERVAL(TRUNC(l_minutes/(60*60)), 'DAY');
      ELSIF UPPER(p_component) = 'HOUR' THEN
         RETURN NUMTODSINTERVAL(TRUNC(l_minutes/60), 'HOUR');
      ELSIF UPPER(p_component) = 'MINUTE' THEN
        IF ABS(l_seconds) > 59 THEN
          RETURN NUMTODSINTERVAL(l_minutes, 'MINUTE') + NUMTODSINTERVAL(TRUNC(l_seconds/60), 'MINUTE'); 
        ELSE
          RETURN NUMTODSINTERVAL(l_minutes, 'MINUTE');
        END IF;
      ELSIF UPPER(p_component) = 'SECOND' THEN
          RETURN NUMTODSINTERVAL(l_minutes * 60, 'SECOND') + NUMTODSINTERVAL(l_seconds, 'SECOND');
      END IF;
   ELSIF REGEXP_LIKE(p_expr,'^(|-)\d+[[:punct:]]\d+$') OR 
         REGEXP_LIKE(p_expr,'^(|-)\d+$') THEN
      l_seconds := p_expr;
      -- DBMS_OUTPUT.PUT_LINE('SECONDS ' || l_seconds);
      IF UPPER(p_component) = 'DAY' THEN
         RETURN NUMTODSINTERVAL(TRUNC(l_seconds/(24*60*60)), 'DAY');
      ELSIF UPPER(p_component) = 'HOUR' THEN
         RETURN NUMTODSINTERVAL(TRUNC(l_seconds/(60*60)), 'HOUR');
      ELSIF UPPER(p_component) = 'MINUTE' THEN
         RETURN NUMTODSINTERVAL(TRUNC(l_seconds/60), 'MINUTE');
      ELSIF UPPER(p_component) = 'SECOND' THEN
          RETURN NUMTODSINTERVAL(l_seconds, 'SECOND');
      END IF;
   END IF;
EXCEPTION
    WHEN OTHERS THEN
      raise_application_error(-20000, dbms_utility.format_error_stack);
END cast_to_interval_day_to_second;

/**
 * 
 * @param 
 * @return 
 */        		   
FUNCTION cast_to_interval_year_to_month(p_expr IN NUMBER, p_precision IN BINARY_INTEGER, p_component IN VARCHAR2) RETURN INTERVAL YEAR TO MONTH AS
BEGIN
     RETURN CAST(TO_CHAR(NUMTOYMINTERVAL(p_expr, p_component), LPAD('Y', p_precision, 'Y')) AS INTERVAL YEAR TO MONTH);
EXCEPTION
    WHEN OTHERS THEN
      raise_application_error(-20000, dbms_utility.format_error_stack);
END cast_to_interval_year_to_month;
/**
 * 
 * @param 
 * @return 
 */        		   
FUNCTION cast_to_interval_day_to_second(p_expr IN NUMBER, p_precision IN BINARY_INTEGER, p_component IN VARCHAR2) RETURN INTERVAL DAY TO SECOND AS
BEGIN
    RETURN CAST(TO_CHAR(NUMTODSINTERVAL(p_expr, p_component), LPAD('Y', p_precision, 'Y')) AS INTERVAL DAY TO SECOND);
EXCEPTION
    WHEN OTHERS THEN
      raise_application_error(-20000, dbms_utility.format_error_stack);
END cast_to_interval_day_to_second;

/**
 * 
 * @param 
 * @return 
 */        		   
FUNCTION apply_format(p_expr IN DATE, p_fmtstr IN VARCHAR2) RETURN TIMESTAMP AS
l_fmtstr VARCHAR2(50);
BEGIN
   -- TODO: need a function to convert Teradata format string to Oracle format.
  l_fmtstr := getOracleDateTimeFormat(p_fmtstr, NULL);
  RETURN TO_TIMESTAMP(TO_CHAR(p_expr, l_fmtstr), l_fmtstr);
EXCEPTION
    WHEN OTHERS THEN
      raise_application_error(-20000, dbms_utility.format_error_stack);
END apply_format;

/*
FUNCTION apply_format(p_expr IN TIMESTAMP WITH TIME ZONE, p_fmtstr IN VARCHAR2) 
      RETURN TIMESTAMP WITH TIME ZONE AS
l_fmtstr VARCHAR2(50);
BEGIN
  -- TODO: need a function to convert Teradata format string to Oracle format.
  l_fmtstr := p_fmtstr;
  RETURN TO_TIMESTAMP_TZ(TO_CHAR(p_expr, l_fmtstr), l_fmtstr);
END apply_format;
*/

/**
 * 
 * @param 
 * @return 
 */        		   
FUNCTION apply_format(p_expr IN VARCHAR2, p_fmtstr IN VARCHAR2) 
     RETURN VARCHAR2 AS
l_len PLS_INTEGER := 0;     
BEGIN
   IF REGEXP_INSTR(p_fmtstr, '^[X]+$') > 0 THEN
     l_len := LENGTH(p_fmtstr);
   ELSIF REGEXP_INSTR(p_fmtstr, '^X\(\d+\)$') > 0 THEN
      l_len := SUBSTR(p_fmtstr, INSTR(p_fmtstr, '(') + 1, LENGTH(p_fmtstr) - 3);
   ELSE
      l_len := 0;
   END IF;
   
   IF l_len > 0 THEN
     IF LENGTH(p_expr) > l_len THEN
        RETURN SUBSTR(p_expr, 1, l_len);
     ELSIF LENGTH(p_expr) < l_len THEN 
        RETURN RPAD(p_expr, l_len);
     ELSE
        RETURN p_expr;
     END IF;
   ELSE
     RETURN p_expr;
   END IF;
EXCEPTION
  WHEN OTHERS THEN
    raise_application_error(-20000, dbms_utility.format_error_stack);
END apply_format;

/**
 * 
 * @param 
 * @return 
 */        		   
FUNCTION cast_to_char(p_expr IN TIMESTAMP, p_precision IN BINARY_INTEGER, p_scale IN BINARY_INTEGER, 
           p_fmtstr IN VARCHAR2, p_chrset IN VARCHAR2) RETURN VARCHAR2 AS
l_pre_fmt varchar2(30) := 'YYYY-MM-DD HH:MI:SS.FF';
l_pst_fmt varchar2(3) := ' AM';
l_format VARCHAR2(35);
l_retval VARCHAR2(50);
BEGIN
   IF p_fmtstr IS NOT NULL THEN
      l_format := getOracleDateTimeFormat(p_fmtstr, p_precision);
   ELSE 
     l_format := 'YYYY-MM-DD HH:MI:SS AM';
   END IF;
   
   l_retval := TO_CHAR(p_expr, l_format);
   
   IF p_precision IS NOT NULL THEN
     l_retval := SUBSTR(l_retval, 1, p_precision);
   END IF;
   
   RETURN l_retval;
EXCEPTION
    WHEN OTHERS THEN
      raise_application_error(-20000, dbms_utility.format_error_stack);
END cast_to_char;

/**
 * 
 * @param 
 * @return 
 */        		   
FUNCTION cast_to_timestamp(p_expr IN TIMESTAMP, p_precision IN BINARY_INTEGER) RETURN TIMESTAMP AS
l_pre_fmt varchar2(30) := 'YYYY-MM-DD HH:MI:SS.FF';
l_pst_fmt varchar2(3) := ' AM';
l_format varchar2(35);
BEGIN
   IF p_precision IS NULL OR p_precision = '0' THEN
     l_format := 'YYYY-MM-DD HH:MI:SS AM';
   ELSE
     l_format := l_pre_fmt || p_precision || l_pst_fmt;
   END IF;
   RETURN CAST(TO_CHAR(p_expr, l_format) AS TIMESTAMP);
EXCEPTION
    WHEN OTHERS THEN
      raise_application_error(-20000, dbms_utility.format_error_stack);
END cast_to_timestamp;

/**
 * 
 * @param 
 * @return 
 */        		   
FUNCTION cast_to_time(p_expr IN TIMESTAMP, p_precision IN BINARY_INTEGER) RETURN TIMESTAMP AS
BEGIN
   RETURN cast_to_timestamp(p_expr, p_precision);
END cast_to_time;

/**
 * 
 * @param 
 * @return 
 */        		   
FUNCTION cast_to_char(p_expr IN INTERVAL YEAR TO MONTH, p_precision IN BINARY_INTEGER, 
           p_scale IN BINARY_INTEGER, p_fmtstr IN VARCHAR2, p_chrset IN VARCHAR2) RETURN VARCHAR2 AS
BEGIN
   RETURN TO_CHAR(p_expr, p_fmtstr);
EXCEPTION
    WHEN OTHERS THEN
      raise_application_error(-20000, dbms_utility.format_error_stack);
END cast_to_char;

/**
 * 
 * @param 
 * @return 
 */        		   
FUNCTION cast_to_char(p_expr IN INTERVAL DAY TO SECOND, p_precision IN BINARY_INTEGER, 
           p_scale IN BINARY_INTEGER, p_fmtstr IN VARCHAR2, p_chrset IN VARCHAR2) RETURN VARCHAR2 AS
BEGIN
   RETURN TO_CHAR(p_expr, p_fmtstr);
EXCEPTION
    WHEN OTHERS THEN
      raise_application_error(-20000, dbms_utility.format_error_stack);
END cast_to_char;

/**
 * 
 * @param 
 * @return 
 */        		   
FUNCTION cast_to_interval_year_to_month(p_expr IN YMINTERVAL_UNCONSTRAINED, p_precision IN BINARY_INTEGER,
           p_component IN VARCHAR2) RETURN YMINTERVAL_UNCONSTRAINED AS
l_year NUMBER;
l_month NUMBER;
l_result NUMBER;
l_charval VARCHAR2(20);
l_val NUMBER;
BEGIN
   l_year := EXTRACT(YEAR FROM p_expr);
   l_month := EXTRACT(MONTH FROM p_expr);
   IF UPPER(p_component) = 'YEAR' THEN
      l_result := l_year + TRUNC(l_month/12);
      l_val := TO_NUMBER(l_result, LPAD('9', p_precision, '9'));
      RETURN NUMTOYMINTERVAL(l_val, p_component);
   ELSIF UPPER(p_component) = 'MONTH' THEN
      l_result := l_month + l_year * 12;
      l_val := TO_NUMBER(l_result, LPAD('9', p_precision, '9'));
      RETURN NUMTOYMINTERVAL(l_val, p_component);
   ELSE
      RETURN p_expr;
   END IF;
EXCEPTION
    WHEN OTHERS THEN
      raise_application_error(-20000, dbms_utility.format_error_stack);
END cast_to_interval_year_to_month;


/*
   The caller of this function need to pass in valid interval values.
*/
/**
 * 
 * @param 
 * @return 
 */        		   
FUNCTION cast_to_interval_day_to_second(p_expr IN DSINTERVAL_UNCONSTRAINED, p_precision IN BINARY_INTEGER,
           p_component IN VARCHAR2) RETURN DSINTERVAL_UNCONSTRAINED AS
l_day NUMBER;
l_hour NUMBER;
l_minute NUMBER;
l_second NUMBER;
l_result NUMBER;
l_val NUMBER;
BEGIN
   l_day := EXTRACT(DAY FROM p_expr);
   -- DBMS_OUTPUT.PUT_LINE('DAY: ' || l_day);
   l_hour := EXTRACT(HOUR FROM p_expr);
   -- DBMS_OUTPUT.PUT_LINE('HOUR: ' || l_hour);
   l_minute := EXTRACT(MINUTE FROM p_expr);
   -- DBMS_OUTPUT.PUT_LINE('MINUTE: ' || l_minute);
   l_second := EXTRACT(SECOND FROM p_expr);
   -- DBMS_OUTPUT.PUT_LINE('SECOND: ' || l_second);
   IF UPPER(p_component) = 'DAY' THEN
      l_result := l_day + TRUNC(l_hour/24) + TRUNC(l_minute/(60*24)) + TRUNC(l_second/(60*60*24));
   ELSIF UPPER(p_component) = 'HOUR' THEN
      l_result := (l_day * 24) + l_hour + TRUNC(l_minute/60) + TRUNC(l_second/(60*60*24));
   ELSIF UPPER(p_component) = 'MINUTE' THEN
      l_result := (l_day * (24*60)) + (l_hour*60) + l_minute + TRUNC(l_second/60);
   ELSIF UPPER(p_component) = 'SECOND' THEN
      l_result := l_day * (24*60*60) + l_hour * (60*60) + l_minute * 60 + l_second;
   END IF;
   
   l_val := TO_NUMBER(l_result, LPAD('9', p_precision, '9'));
   RETURN NUMTODSINTERVAL(l_val, p_component);
EXCEPTION
    WHEN OTHERS THEN
      raise_application_error(-20000, dbms_utility.format_error_stack);
END cast_to_interval_day_to_second;

/**
 * 
 * @param 
 * @return 
 */        		   
FUNCTION apply_format(p_expr IN NUMBER, p_fmtstr IN VARCHAR2) RETURN VARCHAR2 AS
l_fmtstr VARCHAR2(50);
l_numof9s NUMBER:= 0;
BEGIN
  l_fmtstr := p_fmtstr;
  IF REGEXP_LIKE(l_fmtstr, '^9\(\d+\)$') THEN
     l_numof9s := REGEXP_REPLACE(l_fmtstr, '^(9)(\()(\d+)(\))$', '\3');
     RETURN LTRIM(TO_CHAR(p_expr, LPAD('9', l_numof9s, '9'))); -- left trim empty spaces.
  ELSE
     RETURN p_expr;
  END IF;
EXCEPTION
    WHEN OTHERS THEN
      raise_application_error(-20000, dbms_utility.format_error_stack);
END apply_format;

/**
 * 
 * @param 
 * @return 
 */        		   
FUNCTION cast_to_char(p_expr IN NUMBER, p_precision IN BINARY_INTEGER, p_scale IN BINARY_INTEGER, 
           p_fmtstr IN VARCHAR2, p_chrset IN VARCHAR2) RETURN VARCHAR2 AS
BEGIN
  RETURN TO_CHAR(p_expr, p_fmtstr);
EXCEPTION
    WHEN OTHERS THEN
      raise_application_error(-20000, dbms_utility.format_error_stack);
END cast_to_char;

/**
 * 
 * @param 
 * @return 
 */        		   
FUNCTION cast_to_date(p_expr IN NUMBER, p_precision IN BINARY_INTEGER, p_scale IN BINARY_INTEGER, 
           p_fmtstr IN VARCHAR2, p_chrset IN VARCHAR2) RETURN date AS
BEGIN
  RETURN numeric_to_date(p_expr);
EXCEPTION
    WHEN OTHERS THEN
      raise_application_error(-20000, dbms_utility.format_error_stack);
END cast_to_date;

/**
 * 
 * @param 
 * @return 
 */        		   
FUNCTION buildNumberFormat(p_precision IN BINARY_INTEGER, p_scale IN BINARY_INTEGER) RETURN VARCHAR2 AS
l_varchar2 VARCHAR2(50);
BEGIN
  IF p_precision > p_scale THEN
     FOR i IN 1 .. p_precision LOOP
       l_varchar2 := l_varchar2 || 9;
     END LOOP;
     l_varchar2 := SUBSTR(l_varchar2, 1, p_precision - p_scale) || '.' || SUBSTR(l_varchar2, (p_precision - p_scale) + 1, p_precision);
     RETURN l_varchar2;
  ELSIF p_scale > p_precision THEN
      FOR i IN 1 .. p_scale LOOP
       l_varchar2 := l_varchar2 || 9;
      END LOOP;
      l_varchar2 := '9.' || l_varchar2;
      RETURN l_varchar2;
  END IF; 
  RETURN l_varchar2;
END buildNumberFormat;

/**
 * 
 * @param 
 * @return 
 */        		   
FUNCTION cast_to_number(p_expr IN NUMBER, p_precision IN BINARY_INTEGER, p_scale IN BINARY_INTEGER, 
           p_fmtstr IN VARCHAR2, p_chrset IN VARCHAR2) RETURN NUMBER AS
l_fmtstr VARCHAR2(50);
l_retval NUMBER;
BEGIN
  l_fmtstr := buildNumberFormat(p_precision, p_scale);
  IF l_fmtstr IS NOT NULL OR LENGTH(l_fmtstr) > 0 THEN
    l_retval := TO_NUMBER(p_expr, l_fmtstr);
    RETURN l_retval;
  END IF;
  RETURN p_expr;
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END cast_to_number;


/**
 * 
 * @param 
 * @return 
 */        		   
FUNCTION cast_to_decimal(p_expr IN VARCHAR2, p_precision IN BINARY_INTEGER, p_scale IN BINARY_INTEGER, 
          p_fmtstr IN VARCHAR2, p_chrset IN VARCHAR2) RETURN NUMBER AS
l_fmtstr VARCHAR2(50) := NULL;
l_fmtstr0 VARCHAR2(50);
l_precision BINARY_INTEGER;
l_scale BINARY_INTEGER;
l_idx BINARY_INTEGER;
l_expr VARCHAR2(1000);
BEGIN
      l_expr := REGEXP_REPLACE(p_expr, 'US Dollars', 'USD', 1, 1, 'i');
      l_expr := REGEXP_REPLACE(l_expr, ' *', NULL); -- trim spaces inside the string
      
      IF p_fmtstr IS NOT NULL THEN
        l_fmtstr := getOracleNumberFormat(p_expr, p_fmtstr, p_precision, p_scale);
        IF l_expr = p_expr THEN
          l_idx := REGEXP_INSTR(l_fmtstr, '[.Dd]');
          IF l_idx > 0 THEN
             l_expr := SUBSTR(p_expr, 1, l_idx-1) || '.' || SUBSTR(p_expr, l_idx, LENGTH(p_expr));
          END IF;
        END IF;
      ELSE
        IF REGEXP_LIKE(l_expr, '^(+-)\d+$') THEN
          l_fmtstr := 'S' || LPAD(9, LENGTH(l_expr)-1, 9);
          RETURN TO_NUMBER(l_expr, l_fmtstr);
        ELSIF REGEXP_LIKE(l_expr, '^\d+(+-)$') THEN
          l_fmtstr := LPAD(9, LENGTH(l_expr)-1, 9) || 'S';
          RETURN TO_NUMBER(l_expr, l_fmtstr);
        ELSIF REGEXP_COUNT(l_expr, '-') > 1 THEN
           l_expr := REGEXP_REPLACE(l_expr, '-', NULL);
           RETURN TO_NUMBER(l_expr);
        END IF;
      END IF;
      
      l_idx := REGEXP_INSTR(l_expr, '[$£¥¤]');
      IF p_precision IS NULL AND p_scale IS NULL OR 
         p_precision IS NULL OR p_precision = 0 THEN
         IF l_idx = 1 THEN
           IF l_fmtstr IS NULL THEN
             RETURN TO_NUMBER(SUBSTR(l_expr, 2, LENGTH(l_expr)));
           ELSE
             RETURN TO_NUMBER(SUBSTR(l_expr, 2, LENGTH(l_expr)), l_fmtstr);
           END IF;  
         ELSIF l_idx = 0 THEN 
           IF l_fmtstr IS NULL THEN
             RETURN TO_NUMBER(SUBSTR(l_expr, 1, LENGTH(l_expr)));
           ELSE
             IF REGEXP_LIKE(l_expr, '^(\d+\.{1}\d+)|(\d+)$') THEN
               RETURN TO_NUMBER(l_expr);
             ELSE
               RETURN TO_NUMBER(SUBSTR(l_expr, 1, LENGTH(l_expr)), l_fmtstr);
             END IF;
             
           END IF; 
         END IF;
      ELSE
        l_precision := p_precision;
      END IF;
      
      IF p_scale IS NULL THEN
         l_scale := 0;
      ELSE
         l_scale := p_scale;
      END IF;
      
      IF l_fmtstr IS NULL THEN
        IF l_idx = 1 THEN
            l_fmtstr := SUBSTR(l_expr, 1, 1);
        END IF;
        l_fmtstr0 := LPAD(9, (l_precision - p_scale), 9) || '.' || RPAD(9, p_scale, 9);
        l_fmtstr := l_fmtstr || l_fmtstr0;
      END IF;
      
      IF REGEXP_LIKE(l_expr, '^(\d+\.{1}\d+)|(\d+)$') THEN
          RETURN TO_NUMBER(l_expr);
      END IF;
      
      RETURN TO_NUMBER(l_expr, l_fmtstr);
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END cast_to_decimal;


/**
 * 
 * @param 
 * @return 
 */        		   
FUNCTION cast_to_decimal(p_expr IN NUMBER, p_precision IN BINARY_INTEGER, p_scale IN BINARY_INTEGER, 
          p_fmtstr IN VARCHAR2, p_chrset IN VARCHAR2) RETURN NUMBER AS
l_fmtstr VARCHAR2(50);
l_precision BINARY_INTEGER;
l_scale BINARY_INTEGER;
BEGIN
      -- TODO: need a function to convert Teradata format string to Oracle format.
      l_fmtstr := getOracleDateTimeFormat(p_fmtstr, p_precision);
      IF p_precision IS NULL AND p_scale IS NULL THEN
         RETURN ROUND(p_expr);
      END IF;
      
      IF p_precision IS NULL OR p_precision = 0 THEN
         RETURN ROUND(p_expr);
      ELSE
         l_precision := p_precision;
      END IF;
      
      IF p_scale IS NULL THEN
         l_scale := 0;
      ELSE
         l_scale := p_scale;
      END IF;
      
      IF (l_precision - l_scale) <= 0 THEN
        RETURN ROUND(p_expr);  
      END IF;
      
      IF l_fmtstr IS NULL THEN
        l_fmtstr := LPAD(9, (l_precision - p_scale), 9) || '.' || RPAD(9, p_scale, 9);
      END IF;
      RETURN TO_NUMBER(TO_CHAR(p_expr, l_fmtstr));
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END cast_to_decimal;


/**
 * 
 * @param 
 * @return 
 */        		   
FUNCTION cast_to_decimal(p_expr IN INTERVAL YEAR TO MONTH, p_precision IN BINARY_INTEGER, p_scale IN BINARY_INTEGER, 
          p_fmtstr IN VARCHAR2, p_chrset IN VARCHAR2) RETURN NUMBER AS
BEGIN
    RETURN cast_to_number(p_expr, p_precision, p_scale, p_fmtstr, p_chrset);
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END cast_to_decimal;
           
/**
 * 
 * @param 
 * @return 
 */        		   
FUNCTION cast_to_decimal(p_expr IN INTERVAL DAY TO SECOND,  p_precision IN BINARY_INTEGER, p_scale IN BINARY_INTEGER, 
          p_fmtstr IN VARCHAR2, p_chrset IN VARCHAR2) RETURN NUMBER AS
BEGIN
    RETURN cast_to_number(p_expr, p_precision, p_scale, p_fmtstr, p_chrset);
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END cast_to_decimal;

/**
 * 
 * @param 
 * @return 
 */        		   
FUNCTION cast_to_decimal(p_expr IN DATE, p_precision IN BINARY_INTEGER, p_scale IN BINARY_INTEGER, 
          p_fmtstr IN VARCHAR2, p_chrset IN VARCHAR2) RETURN NUMBER AS
BEGIN
    RETURN cast_to_number(p_expr, p_precision, p_scale, p_fmtstr, p_chrset);
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END cast_to_decimal;

/**
 * 
 * @param 
 * @return 
 */        		   
FUNCTION cast_to_float(p_expr IN VARCHAR2, p_precision IN BINARY_INTEGER, p_scale IN BINARY_INTEGER, 
         p_fmtstr IN VARCHAR2, p_chrset IN VARCHAR2) RETURN BINARY_DOUBLE AS
l_expr VARCHAR2(10000);
l_idx PLS_INTEGER;
BEGIN
    -- TODO: need a function to convert Teradata format string to Oracle format.
    l_idx := REGEXP_INSTR(p_expr, '[Ee]{1}');
    IF l_idx > 0 THEN
       IF SUBSTR(p_expr, 1, l_idx-1) IS NULL THEN
          l_expr := 0;
       ELSE
          IF SUBSTR(p_expr, l_idx+1, 1) = '-' THEN
            l_expr := SUBSTR(p_expr, 1, l_idx-1) * (1/POWER(10, SUBSTR(p_expr, l_idx+2, LENGTH(p_expr))));
          ELSE
            l_expr := SUBSTR(p_expr, 1, l_idx-1) * POWER(10, SUBSTR(p_expr, l_idx+1, LENGTH(p_expr)));
          END IF;
       END IF;   
    END IF;
    
    RETURN TO_BINARY_DOUBLE(l_expr);
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END cast_to_float;

/**
 * 
 * @param 
 * @return 
 */        		   
FUNCTION cast_to_float(p_expr IN NUMBER, p_precision IN BINARY_INTEGER, p_scale IN BINARY_INTEGER, 
         p_fmtstr IN VARCHAR2, p_chrset IN VARCHAR2) RETURN BINARY_DOUBLE AS
BEGIN
    RETURN TO_BINARY_DOUBLE(p_expr);
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END cast_to_float;

/**
 * 
 * @param 
 * @return 
 */        		   
FUNCTION cast_to_float(p_expr IN INTERVAL YEAR TO MONTH, p_precision IN BINARY_INTEGER, p_scale IN BINARY_INTEGER, 
         p_fmtstr IN VARCHAR2, p_chrset IN VARCHAR2) RETURN BINARY_DOUBLE AS
BEGIN
    -- TODO: need a function to convert Teradata format string to Oracle format.
    RETURN TO_BINARY_DOUBLE(cast_to_number(p_expr, p_precision, p_scale, p_fmtstr, p_chrset));
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END cast_to_float;

/**
 * 
 * @param 
 * @return 
 */        		   
FUNCTION cast_to_float(p_expr IN INTERVAL DAY TO SECOND, p_precision IN BINARY_INTEGER, p_scale IN BINARY_INTEGER, 
         p_fmtstr IN VARCHAR2, p_chrset IN VARCHAR2) RETURN BINARY_DOUBLE AS
BEGIN
    -- TODO: need a function to convert Teradata format string to Oracle format.
    RETURN TO_BINARY_DOUBLE(cast_to_number(p_expr, p_precision, p_scale, p_fmtstr, p_chrset));
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END cast_to_float;

/**
 * 
 * @param 
 * @return 
 */        		   
FUNCTION cast_to_float(p_expr IN DATE, p_precision IN BINARY_INTEGER, p_scale IN BINARY_INTEGER, 
          p_fmtstr IN VARCHAR2, p_chrset IN VARCHAR2) RETURN BINARY_DOUBLE AS
BEGIN
    RETURN TO_BINARY_DOUBLE(cast_to_number(p_expr, p_precision, p_scale, p_fmtstr, p_chrset));
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END cast_to_float;

/**
 * 
 * @param 
 * @return 
 */        		   
FUNCTION cast_to_real(p_expr IN VARCHAR2, p_precision IN BINARY_INTEGER, p_scale IN BINARY_INTEGER, 
          p_fmtstr IN VARCHAR2, p_chrset IN VARCHAR2) RETURN BINARY_DOUBLE AS
BEGIN
   RETURN cast_to_float(p_expr, p_precision, p_scale, p_fmtstr, p_chrset);
END cast_to_real;
           
/**
 * 
 * @param 
 * @return 
 */        		   
FUNCTION cast_to_double_precision(p_expr IN VARCHAR2, p_precision IN BINARY_INTEGER, p_scale IN BINARY_INTEGER, 
          p_fmtstr IN VARCHAR2, p_chrset IN VARCHAR2) RETURN BINARY_DOUBLE AS
BEGIN
   RETURN cast_to_float(p_expr, p_precision, p_scale, p_fmtstr, p_chrset);
END cast_to_double_precision;
           
/**
 * 
 * @param 
 * @return 
 */        		   
FUNCTION cast_to_real(p_expr IN DATE, p_precision IN BINARY_INTEGER, p_scale IN BINARY_INTEGER, 
          p_fmtstr IN VARCHAR2, p_chrset IN VARCHAR2) RETURN BINARY_DOUBLE AS
BEGIN
  RETURN cast_to_float(p_expr, p_precision, p_scale, p_fmtstr, p_chrset);
END cast_to_real;
           
/**
 * 
 * @param 
 * @return 
 */        		   
FUNCTION cast_to_double_precision(p_expr IN DATE, p_precision IN BINARY_INTEGER, p_scale IN BINARY_INTEGER, 
          p_fmtstr IN VARCHAR2, p_chrset IN VARCHAR2) RETURN BINARY_DOUBLE AS
BEGIN
  RETURN cast_to_float(p_expr, p_precision, p_scale, p_fmtstr, p_chrset);
END cast_to_double_precision;
           
/**
 * 
 * @param 
 * @return 
 */        		   
FUNCTION cast_to_real(p_expr IN INTERVAL YEAR TO MONTH, p_precision IN BINARY_INTEGER, p_scale IN BINARY_INTEGER, 
          p_fmtstr IN VARCHAR2, p_chrset IN VARCHAR2) RETURN BINARY_DOUBLE AS
BEGIN
  RETURN cast_to_float(p_expr, p_precision, p_scale, p_fmtstr, p_chrset); 
END cast_to_real;
           
/**
 * 
 * @param 
 * @return 
 */        		   
FUNCTION cast_to_double_precision(p_expr IN INTERVAL YEAR TO MONTH, p_precision IN BINARY_INTEGER, p_scale IN BINARY_INTEGER, 
          p_fmtstr IN VARCHAR2, p_chrset IN VARCHAR2) RETURN BINARY_DOUBLE AS
BEGIN
  RETURN cast_to_float(p_expr, p_precision, p_scale, p_fmtstr, p_chrset); 
END cast_to_double_precision;
           
/**
 * 
 * @param 
 * @return 
 */        		   
FUNCTION cast_to_real(p_expr IN INTERVAL DAY TO SECOND, p_precision IN BINARY_INTEGER, p_scale IN BINARY_INTEGER, 
          p_fmtstr IN VARCHAR2, p_chrset IN VARCHAR2) RETURN BINARY_DOUBLE AS
BEGIN
  RETURN cast_to_float(p_expr, p_precision, p_scale, p_fmtstr, p_chrset);
END cast_to_real;
           
/**
 * 
 * @param 
 * @return 
 */        		   
FUNCTION cast_to_double_precision(p_expr IN INTERVAL DAY TO SECOND, p_precision IN BINARY_INTEGER, p_scale IN BINARY_INTEGER, 
          p_fmtstr IN VARCHAR2, p_chrset IN VARCHAR2) RETURN BINARY_DOUBLE AS
BEGIN
  RETURN cast_to_float(p_expr, p_precision, p_scale, p_fmtstr, p_chrset);
END cast_to_double_precision;
           
/**
 * 
 * @param 
 * @return 
 */        		   
FUNCTION cast_to_real(p_expr IN NUMBER, p_precision IN BINARY_INTEGER, p_scale IN BINARY_INTEGER, 
          p_fmtstr IN VARCHAR2, p_chrset IN VARCHAR2) RETURN BINARY_DOUBLE AS
BEGIN
  RETURN cast_to_float(p_expr, p_precision, p_scale, p_fmtstr, p_chrset);
END cast_to_real;
           
/**
 * 
 * @param 
 * @return 
 */        		   
FUNCTION cast_to_double_precision(p_expr IN NUMBER, p_precision IN BINARY_INTEGER, p_scale IN BINARY_INTEGER, 
          p_fmtstr IN VARCHAR2, p_chrset IN VARCHAR2) RETURN BINARY_DOUBLE AS
BEGIN
  RETURN cast_to_float(p_expr, p_precision, p_scale, p_fmtstr, p_chrset);
END cast_to_double_precision;

/**
 * 
 * @param 
 * @return 
 */        		   
FUNCTION cast_to_integer(p_expr IN VARCHAR2, p_precision IN BINARY_INTEGER, p_scale IN BINARY_INTEGER, 
         p_fmtstr IN VARCHAR2, p_chrset IN VARCHAR2) RETURN NUMBER AS
BEGIN
   -- RETURN TRUNC(TO_NUMBER(cast_to_decimal(p_expr, p_precision, p_scale, p_fmtstr, p_chrset)));
   RETURN TRUNC(cast_to_decimal(p_expr, p_precision, p_scale, p_fmtstr, p_chrset));
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END cast_to_integer;

/**
 * 
 * @param 
 * @return 
 */        		   
FUNCTION cast_to_smallint(p_expr IN VARCHAR2, p_precision IN BINARY_INTEGER, p_scale IN BINARY_INTEGER, 
         p_fmtstr IN VARCHAR2, p_chrset IN VARCHAR2) RETURN NUMBER AS
BEGIN
   RETURN cast_to_integer(p_expr, p_precision, p_scale, p_fmtstr, p_chrset);
END cast_to_smallint;

/**
 * 
 * @param 
 * @return 
 */        		   
FUNCTION cast_to_integer(p_expr IN NUMBER, p_precision IN BINARY_INTEGER, p_scale IN BINARY_INTEGER, 
         p_fmtstr IN VARCHAR2, p_chrset IN VARCHAR2) RETURN NUMBER AS
BEGIN
   RETURN TRUNC(p_expr);
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END cast_to_integer;

/**
 * 
 * @param 
 * @return 
 */        		   
FUNCTION cast_to_smallint(p_expr IN NUMBER, p_precision IN BINARY_INTEGER, p_scale IN BINARY_INTEGER, 
         p_fmtstr IN VARCHAR2, p_chrset IN VARCHAR2) RETURN NUMBER AS
BEGIN
   RETURN cast_to_integer(p_expr, p_precision, p_scale, p_fmtstr, p_chrset);
END cast_to_smallint;

/**
 * 
 * @param 
 * @return 
 */        		   
FUNCTION cast_to_integer(p_expr IN INTERVAL YEAR TO MONTH, p_precision IN BINARY_INTEGER, p_scale IN BINARY_INTEGER, 
         p_fmtstr IN VARCHAR2, p_chrset IN VARCHAR2) RETURN NUMBER AS
BEGIN
   RETURN TRUNC(cast_to_number(p_expr, p_precision, p_scale, p_fmtstr, p_chrset));
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END cast_to_integer;

/**
 * 
 * @param 
 * @return 
 */        		   
FUNCTION cast_to_smallint(p_expr IN INTERVAL YEAR TO MONTH, p_precision IN BINARY_INTEGER, p_scale IN BINARY_INTEGER, 
         p_fmtstr IN VARCHAR2, p_chrset IN VARCHAR2) RETURN NUMBER AS
BEGIN
   RETURN cast_to_integer(p_expr, p_precision, p_scale, p_fmtstr, p_chrset);
END cast_to_smallint;

/**
 * 
 * @param 
 * @return 
 */        		   
FUNCTION cast_to_integer(p_expr IN INTERVAL DAY TO SECOND, p_precision IN BINARY_INTEGER, p_scale IN BINARY_INTEGER, 
         p_fmtstr IN VARCHAR2, p_chrset IN VARCHAR2) RETURN NUMBER AS
BEGIN
   RETURN TRUNC(cast_to_number(p_expr, p_precision, p_scale, p_fmtstr, p_chrset));
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END cast_to_integer;

/**
 * 
 * @param 
 * @return 
 */        		   
FUNCTION cast_to_smallint(p_expr IN INTERVAL DAY TO SECOND, p_precision IN BINARY_INTEGER, p_scale IN BINARY_INTEGER, 
         p_fmtstr IN VARCHAR2, p_chrset IN VARCHAR2) RETURN NUMBER AS
BEGIN
   RETURN cast_to_integer(p_expr, p_precision, p_scale, p_fmtstr, p_chrset);
END cast_to_smallint;

/**
 * 
 * @param 
 * @return 
 */        		   
FUNCTION cast_to_integer(p_expr IN DATE, p_precision IN BINARY_INTEGER, p_scale IN BINARY_INTEGER, 
          p_fmtstr IN VARCHAR2, p_chrset IN VARCHAR2) RETURN NUMBER AS
BEGIN
   RETURN TRUNC(cast_to_number(p_expr, p_precision, p_scale, p_fmtstr, p_chrset));
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END cast_to_integer;

/**
 * 
 * @param 
 * @return 
 */        		   
FUNCTION cast_to_smallint(p_expr IN DATE, p_precision IN BINARY_INTEGER, p_scale IN BINARY_INTEGER, 
          p_fmtstr IN VARCHAR2, p_chrset IN VARCHAR2) RETURN NUMBER AS
BEGIN
   RETURN cast_to_integer(p_expr, p_precision, p_scale, p_fmtstr, p_chrset);
END cast_to_smallint;

/**
 * 
 * @param 
 * @return 
 */        		   
FUNCTION cast_to_blob(p_expr IN BLOB, p_precision IN BINARY_INTEGER, p_scale IN BINARY_INTEGER, 
          p_fmtstr IN VARCHAR2, p_chrset IN VARCHAR2) RETURN BLOB AS
BEGIN
   RETURN p_expr;
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END cast_to_blob;

/**
 * 
 * @param 
 * @return 
 */        		   
FUNCTION cast_to_blob(p_expr IN VARCHAR2, p_precision IN BINARY_INTEGER, p_scale IN BINARY_INTEGER, 
          p_fmtstr IN VARCHAR2, p_chrset IN VARCHAR2) RETURN BLOB AS
BEGIN
   RETURN TO_BLOB(UTL_RAW.CAST_TO_RAW(p_expr));
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END cast_to_blob;

/**
 * 
 * @param 
 * @return 
 */        		   
FUNCTION cast_to_char(p_expr IN CHAR, p_precision IN BINARY_INTEGER, p_scale IN BINARY_INTEGER, 
          p_fmtstr IN VARCHAR2, p_chrset IN VARCHAR2) RETURN CHAR AS
BEGIN
  IF LENGTH(p_expr) = 0 OR p_precision IS NULL THEN
     RETURN p_expr;
  ELSE
     RETURN SUBSTR(p_expr, 1, p_precision);
  END IF;
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END cast_to_char;

/**
 * 
 * @param 
 * @return 
 */        		   
FUNCTION cast_to_nchar(p_expr IN VARCHAR2, p_precision IN BINARY_INTEGER, p_scale IN BINARY_INTEGER, 
          p_fmtstr IN VARCHAR2, p_chrset IN VARCHAR2) RETURN NCHAR AS
BEGIN
  RETURN TO_NCHAR(p_expr);
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END cast_to_nchar;

/**
 * 
 * @param 
 * @return 
 */        		   
FUNCTION cast_to_clob(p_expr IN VARCHAR2, p_precision IN BINARY_INTEGER, p_scale IN BINARY_INTEGER, 
          p_fmtstr IN VARCHAR2, p_chrset IN VARCHAR2) RETURN CLOB AS
BEGIN
  RETURN TO_CLOB(p_expr);
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END cast_to_clob;

/**
 * 
 * @param 
 * @return 
 */        		   
FUNCTION cast_to_nvarchar2(p_expr IN VARCHAR2, p_precision IN BINARY_INTEGER, p_scale IN BINARY_INTEGER, 
          p_fmtstr IN VARCHAR2, p_chrset IN VARCHAR2) RETURN NVARCHAR2 AS
BEGIN
  RETURN TO_NCHAR(p_expr);
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END cast_to_nvarchar2;

/**
 * 
 * @param 
 * @return 
 */        		   
FUNCTION cast_to_nvarchar2(p_expr IN CLOB, p_precision IN BINARY_INTEGER, p_scale IN BINARY_INTEGER, 
          p_fmtstr IN VARCHAR2, p_chrset IN VARCHAR2) RETURN NVARCHAR2 AS
BEGIN
  RETURN TO_NCHAR(p_expr);
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END cast_to_nvarchar2;

/**
 * 
 * @param 
 * @return 
 */        		   
FUNCTION cast_to_varchar2(p_expr IN VARCHAR2, p_precision IN BINARY_INTEGER, p_scale IN BINARY_INTEGER, 
          p_fmtstr IN VARCHAR2, p_chrset IN VARCHAR2) RETURN VARCHAR2 AS
BEGIN
  RETURN p_expr;
EXCEPTION
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END cast_to_varchar2;

FUNCTION cast_to_char(p_expr IN BLOB, p_precision IN BINARY_INTEGER, p_scale IN BINARY_INTEGER, 
         p_fmtstr IN VARCHAR2, p_chrset IN VARCHAR2) RETURN CHAR AS
l_toolarge EXCEPTION;
BEGIN
  IF LENGTHB(p_expr) > 2000 THEN
    RAISE l_toolarge;
  ELSE
    RETURN UTL_RAW.CAST_TO_VARCHAR2(p_expr);
  END IF;
EXCEPTION
    WHEN l_toolarge THEN
      RAISE_APPLICATION_ERROR(-20001, DBMS_UTILITY.FORMAT_ERROR_STACK || 'Data is too large to convert to CHAR type.');  
    WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);  
END cast_to_char;


-- UNSUPPORTED FUNCTIONS
FUNCTION vargraphic(p_expr IN VARCHAR2) RETURN NVARCHAR2 AS
l_notsupported EXCEPTION;
BEGIN
  RAISE l_notsupported;
EXCEPTION
   WHEN l_notsupported THEN
      RAISE_APPLICATION_ERROR(-20001,  'VARGRAPHIC is not supported.');
   WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END vargraphic;

FUNCTION mlinreg(p_val_expr NUMBER, p_width BINARY_INTEGER, p_sort_expression NUMBER) RETURN NUMBER AS
l_notsupported EXCEPTION;
BEGIN
  RAISE l_notsupported;
EXCEPTION
   WHEN l_notsupported THEN
      RAISE_APPLICATION_ERROR(-20001,  'MLINREG is not supported.');
   WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END mlinreg;

FUNCTION range_n(p_expr_list IN VARCHAR2) RETURN BINARY_INTEGER AS
l_notsupported EXCEPTION;
BEGIN
  RAISE l_notsupported;
EXCEPTION
   WHEN l_notsupported THEN
      RAISE_APPLICATION_ERROR(-20001,  'RANGE_N is not supported.');
   WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END range_n;

FUNCTION case_n(p_expr_list IN VARCHAR2) RETURN BINARY_INTEGER AS
l_notsupported EXCEPTION;
BEGIN
  RAISE l_notsupported;
EXCEPTION
   WHEN l_notsupported THEN
      RAISE_APPLICATION_ERROR(-20001,  'CASE_N is not supported.');
   WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END case_n;

FUNCTION skew(p_val_expr VARCHAR2, p_expr VARCHAR2 DEFAULT 'ALL') RETURN NUMBER AS
l_notsupported EXCEPTION;
BEGIN
  RAISE l_notsupported;
EXCEPTION
   WHEN l_notsupported THEN
      RAISE_APPLICATION_ERROR(-20001,  'SKEW is not supported.');
   WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END skew;

FUNCTION kurtosis(p_val_expr VARCHAR2, p_expr VARCHAR2 DEFAULT 'ALL') RETURN NUMBER AS
l_notsupported EXCEPTION;
BEGIN
  RAISE l_notsupported;
EXCEPTION
   WHEN l_notsupported THEN
      RAISE_APPLICATION_ERROR(-20001,  'KURTOSIS is not supported.');
   WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END kurtosis;

FUNCTION string_cs(p_expr IN VARCHAR2) RETURN BINARY_INTEGER AS
l_notsupported EXCEPTION;
BEGIN
  RAISE l_notsupported;
EXCEPTION
   WHEN l_notsupported THEN
      RAISE_APPLICATION_ERROR(-20001,  'STRING_CS is not supported.');
   WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END string_cs;

FUNCTION translate(p_expr IN VARCHAR2, p_src_to_tgt_cset IN VARCHAR2) RETURN VARCHAR2 AS
l_notsupported EXCEPTION;
BEGIN
  RAISE l_notsupported;
EXCEPTION
   WHEN l_notsupported THEN
      RAISE_APPLICATION_ERROR(-20001,  'TRANSLATE is not supported.');
   WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END translate;

FUNCTION translate(p_expr IN CLOB, p_src_to_tgt_cset IN VARCHAR2) RETURN CLOB AS
l_notsupported EXCEPTION;
BEGIN
  RAISE l_notsupported;
EXCEPTION
   WHEN l_notsupported THEN
      RAISE_APPLICATION_ERROR(-20001,  'TRANSLATE is not supported.');
   WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END translate;

FUNCTION translate_chk(p_expr IN VARCHAR2, p_src_to_tgt_cset IN VARCHAR2) RETURN BINARY_INTEGER AS
l_notsupported EXCEPTION;
BEGIN
  RAISE l_notsupported;
EXCEPTION
   WHEN l_notsupported THEN
      RAISE_APPLICATION_ERROR(-20001,  'TRANSLATE_CHK is not supported.');
   WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END translate_chk;

FUNCTION translate_chk(p_expr IN CLOB, p_src_to_tgt_cset IN VARCHAR2) RETURN BINARY_INTEGER AS
l_notsupported EXCEPTION;
BEGIN
  RAISE l_notsupported;
EXCEPTION
   WHEN l_notsupported THEN
      RAISE_APPLICATION_ERROR(-20001,  'TRANSLATE_CHK is not supported.');
   WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END translate_chk;

FUNCTION camset(p_expr IN VARCHAR2) RETURN BLOB AS 
l_notsupported EXCEPTION;
BEGIN
  RAISE l_notsupported;
EXCEPTION
   WHEN l_notsupported THEN
      RAISE_APPLICATION_ERROR(-20001,  'CAMSET is not supported.');
   WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END camset;

FUNCTION camset_l(p_expr IN VARCHAR2) RETURN BLOB AS
l_notsupported EXCEPTION;
BEGIN
  RAISE l_notsupported;
EXCEPTION
   WHEN l_notsupported THEN
      RAISE_APPLICATION_ERROR(-20001,  'CAMSET_L is not supported.');
   WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END camset_l;

FUNCTION decamset(p_expr IN BLOB) RETURN VARCHAR2 AS
l_notsupported EXCEPTION;
BEGIN
  RAISE l_notsupported;
EXCEPTION
   WHEN l_notsupported THEN
      RAISE_APPLICATION_ERROR(-20001,  'DECAMSET is not supported.');
   WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END decamset;

FUNCTION decamset_l(p_expr IN BLOB) RETURN VARCHAR2 AS
l_notsupported EXCEPTION;
BEGIN
  RAISE l_notsupported;
EXCEPTION
   WHEN l_notsupported THEN
      RAISE_APPLICATION_ERROR(-20001,  'DECAMSET_L is not supported.');
   WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END decamset_l;

FUNCTION lzcomp(p_expr IN VARCHAR2) RETURN BLOB AS
l_notsupported EXCEPTION;
BEGIN
  RAISE l_notsupported;
EXCEPTION
   WHEN l_notsupported THEN
      RAISE_APPLICATION_ERROR(-20001,  'LZCOMP is not supported.');
   WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END lzcomp;

FUNCTION lzcomp_l(p_expr IN VARCHAR2) RETURN BLOB AS
l_notsupported EXCEPTION;
BEGIN
  RAISE l_notsupported;
EXCEPTION
   WHEN l_notsupported THEN
      RAISE_APPLICATION_ERROR(-20001,  'LZCOMP_L is not supported.');
   WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END lzcomp_l;

FUNCTION lzdecomp(p_expr IN BLOB) RETURN VARCHAR2 AS
l_notsupported EXCEPTION;
BEGIN
  RAISE l_notsupported;
EXCEPTION
   WHEN l_notsupported THEN
      RAISE_APPLICATION_ERROR(-20001,  'LZDECOMP is not supported.');
   WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END lzdecomp;

FUNCTION lzdecomp_l(p_expr IN BLOB) RETURN VARCHAR2 AS
l_notsupported EXCEPTION;
BEGIN
  RAISE l_notsupported;
EXCEPTION
   WHEN l_notsupported THEN
      RAISE_APPLICATION_ERROR(-20001,  'LZDECOMP_L is not supported.');
   WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END lzdecomp_l;

FUNCTION transunicode_to_utf8(p_expr IN VARCHAR2) RETURN BLOB AS
l_notsupported EXCEPTION;
BEGIN
  RAISE l_notsupported;
EXCEPTION
   WHEN l_notsupported THEN
      RAISE_APPLICATION_ERROR(-20001,  'TransUnicodeToUTF8 is not supported.');
   WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END transunicode_to_utf8;

FUNCTION transutf8_to_unicode(p_expr IN BLOB) RETURN VARCHAR2 AS
l_notsupported EXCEPTION;
BEGIN
  RAISE l_notsupported;
EXCEPTION
   WHEN l_notsupported THEN
      RAISE_APPLICATION_ERROR(-20001,  'TransUTF8ToUnicode is not supported.');
   WHEN OTHERS THEN
      RAISE_APPLICATION_ERROR(-20000, DBMS_UTILITY.FORMAT_ERROR_STACK);
END transutf8_to_unicode;

BEGIN
-- some possible formats
DT_FORMATS := VARCHAR2_ARRAY(
'fxfmdd/mm/yy',                    --British/French
'fxfmdd/mm/yyyy',                  --British/French(with Century)
'fxfmdd.mm.yy',                    --German(without Century)
'fxfmdd.mm.yyyy',                  --German(with Century)
'fxfmdd-mm-yy',                    --Italian(without Century)
'fxfmdd-mm-yyyy',                  --Italian(with Century)
'fxfmdd mon yy',                   --(without Century)
'fxfmdd mon yyyy',                 --(with Century)
'dd mon yyyy hh24:mi:ssxff3',      --Europe default+milliseconds
'fxdd mon yyyy hh12:mi:ss:ff3AM',  --Hijri calendar system 
'fxdd/mm/yy hh12:mi:ss:ff3AM',     --Hijri calendar system 
'fmdd yyyy MONTH',
'fmdd Month',
'fmdd Month yy',
'fmdd Month yyyy',
'fxddmmyy',
'fxddmmyyyy',
'fxdd-Mon-yy',
'fxdd-MON-yy',
'fxdd-Mon-yyyy',
'fxdd-MON-yyyy',

'fmmon dd yyyy hh:miAM',           --Default   
'fxfmmm/dd/yy',                    --US(without Century)
'fxfmmm/dd/yyyy',                  --US(with Century)
'fxfmMon dd, yy',                  --(without Century)
'fxfmMon dd, yyyy',                --(with Century)
'mon dd yyyy hh12:mi:ssxff3am',    --Default+milliseconds
'fxfmmm-dd-yy',                    --US(without Century)
'fxfmmm-dd-yyyy',                  --US(with Century)
'fmMONTH, yyyy',
'MON yyyy',
'fmMONTH dd, yyyy',
'mm/yy',
'mm/yyyy',
'fmMonth dd, yyyy',             
'fmMonth dd',
'mm-yy',
'mm-yyyy',
'fxmmddyy',
'fxmmddyyyy',
'Mon-yy',
'Mon-yyyy',
'MON-yy',
'MON-yyyy',

'fxfmyy.mm.dd',                    --ANSI(without Century)
'fxfmyyyy.mm.dd',                  --ANSI(with Century)
'fxfmyy/mm/dd',                    --Japan(without Century)
'fxfmyyyy/mm/dd',                  --Japan(with Century)
'yymmdd',                          --ISO(without Century)
'yyyymmdd',                        --ISO(with Century)
'fxyyyy-mm-dd hh24:mi:ss',         --ODBC canonical
'fxyyyy-mm-dd hh24:mi:ssxff3',     --ODBC canonical (with milliseconds)
'fxyyyy-mm-dd"T"hh12:mi:ssxff3',   --ISO8601 (no spaces)
'fxyyyy-mm-dd hh12:mi:ssxff3',     --ISO8601 (with space)
'fxyyyy-mm-dd hh12:mi:ssxff3tzr',  --ISO8601 with time zone Z
'fxyyyymmdd hh24:mi:ss',
'fxyyyy-mm-dd',                    --ISO8601 Date only
'fxyyyy-mm-dd hh12:mi:ss',
'yyyy MON',
'yy/mm',
'yyyy/mm',
'yy-mm',
'yyyy-mm'
);

END TERADATA_UTILITIES;